<?php
/**
 * AJAX Endpoint: ajax_apply_selected_updates_for_contracts.php
 * Input (POST):
 *   - updates (JSON string): Selected changes { vendor: {}, products: [], pricebooks: [{_comparison_key: '...', ...}] }.
 *   - contract_auto_id (int): The auto_id from tdu_vendors_contracts.
 * Action:
 *   1. Applies DB updates using the original algorithm.
 *   2. Adds logging for applied changes. (BaseVendorId validation removed for broader compatibility)
 *   3. Resets job status on success/warning.
 * Output (JSON): Overall success/error status and details/messages.
 */

// --- Setup & Includes ---
// error_reporting(E_ALL); ini_set('display_errors', 1); // Enable for dev if needed
ini_set('display_errors', 0); // Keep errors off in production AJAX
ini_set('log_errors', 1);
// Optionally set a specific error log file: 
ini_set('error_log', '/contract_compare_error.log');
header('Content-Type: application/json; charset=utf-8');
// Adjust the path to dbconn.php based on your file structure
require_once __DIR__ . "/../dbconn.php"; // Provides $conn

// --- Response Structure ---
$response = [
    'status' => 'error',
    'message' => 'Update process failed.',
    'details' => ['vendor' => [], 'products' => [], 'pricebooks' => []],
    'errors' => [],
    'warnings' => []
];

// --- DB Connection Check ---
if (!isset($conn) || !($conn instanceof mysqli) || $conn->connect_error) {
    http_response_code(500); $response['message'] = 'Database connection failed.'; $response['errors'] = [$conn->connect_error ?? 'Unknown DB conn error']; error_log("Apply Updates: DB connection error - " . ($conn->connect_error ?? 'Unknown error')); echo json_encode($response); exit;
}
if (!$conn->set_charset('utf8mb4')) { http_response_code(500); $response['message'] = 'Database charset failed.'; $response['errors'] = [$conn->error]; error_log("Apply Updates: DB charset error - " . $conn->error); echo json_encode($response); exit; }

// --- LOGGING SETUP: Ensure Log Table Exists ---
// (Log table creation/check code remains the same)
$createLogTableSQL = "
CREATE TABLE IF NOT EXISTS `tdu_ai_comparison_applied_changes_log` (
  `log_id` INT AUTO_INCREMENT PRIMARY KEY,
  `vendorid` INT NOT NULL,
  `productid` INT NULL DEFAULT NULL,
  `pricebook_auto_id` INT NULL DEFAULT NULL,
  `contractid` INT NOT NULL,
  `operation` VARCHAR(10) NOT NULL,
  `fields` VARCHAR(100) NOT NULL,
  `old_value` TEXT NULL DEFAULT NULL,
  `new_value` TEXT NULL DEFAULT NULL,
  `applied_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  INDEX `idx_contractid` (`contractid`),
  INDEX `idx_vendorid` (`vendorid`),
  INDEX `idx_productid` (`productid`),
  INDEX `idx_pricebook_auto_id` (`pricebook_auto_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
";
$checkOperationColumnSQL = "SELECT 1 FROM information_schema.columns WHERE table_schema = DATABASE() AND table_name = 'tdu_ai_comparison_applied_changes_log' AND column_name = 'operation'";
$addOperationColumnSQL = "ALTER TABLE `tdu_ai_comparison_applied_changes_log` ADD COLUMN `operation` VARCHAR(10) NOT NULL AFTER `contractid`";
if (!$conn->query($createLogTableSQL)) { error_log("Apply Updates: Failed create/verify log table: " . $conn->error); $response['warnings'][] = "Could not create/verify log table structure."; }
else { $resultCheck = $conn->query($checkOperationColumnSQL); if ($resultCheck && $resultCheck->num_rows === 0) { if (!$conn->query($addOperationColumnSQL)) { error_log("Apply Updates: Failed add 'operation' column: " . $conn->error); $response['warnings'][] = "Could not add 'operation' column."; } } elseif (!$resultCheck) { error_log("Apply Updates: Failed check 'operation' column: " . $conn->error); } if($resultCheck) $resultCheck->free_result(); }
// --- END LOGGING SETUP ---

// --- Request Validation ---
if ($_SERVER['REQUEST_METHOD'] !== 'POST') { http_response_code(405); $response['message'] = 'Invalid request method.'; echo json_encode($response); $conn->close(); exit; }
if (!isset($_POST['updates'])) { http_response_code(400); $response['message'] = 'Missing "updates" data.'; echo json_encode($response); $conn->close(); exit; }
$selectedUpdates = json_decode($_POST['updates'], true); // Decode JSON payload
if (json_last_error() !== JSON_ERROR_NONE) { http_response_code(400); $response['message'] = 'Invalid JSON in "updates".'; $response['errors'] = [json_last_error_msg()]; echo json_encode($response); $conn->close(); exit; }
if (!isset($_POST['contract_auto_id']) || !is_numeric($_POST['contract_auto_id']) || intval($_POST['contract_auto_id']) <= 0) { http_response_code(400); $response['message'] = 'Missing or invalid contract_auto_id.'; echo json_encode($response); $conn->close(); exit; }
$contract_auto_id = intval($_POST['contract_auto_id']); // ID from tdu_vendors_contracts


/**
 * LOGGING FUNCTION: Logs a single field change.
 */
function logChange(mysqli $conn, ?int $vendorId, ?int $productId, ?int $pricebookAutoId, int $contractId, string $operationType, string $field, ?string $oldValue, ?string $newValue): void {
    // LOGGING: Add check - if vendorId is null, log warning and skip logging for this change
    if ($vendorId === null || $vendorId <= 0) {
        error_log("LogChange Skipped: Missing or invalid VendorID (Vendor: {$vendorId}, Product: {$productId}, PB: {$pricebookAutoId}, Field: {$field})");
        return; // Skip logging if vendor ID is invalid
    }

    if ($operationType === 'Update' && $oldValue === $newValue) { return; } // Skip logging if no change

    $sql = "INSERT INTO `tdu_ai_comparison_applied_changes_log` (vendorid, productid, pricebook_auto_id, contractid, operation, fields, old_value, new_value) VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
    $stmt = $conn->prepare($sql);
    if (!$stmt) throw new Exception("Log change prepare failed: " . $conn->error);
    $oldValStr = ($oldValue !== null) ? strval($oldValue) : null;
    $newValStr = ($newValue !== null) ? strval($newValue) : null;
    if (!$stmt->bind_param('iiiissss', $vendorId, $productId, $pricebookAutoId, $contractId, $operationType, $field, $oldValStr, $newValStr)) { throw new Exception("Log change bind failed: " . $stmt->error); }
    if (!$stmt->execute()) { $stmt->close(); throw new Exception("Log change execute failed: " . $stmt->error); } // Close before throw on execute fail
    $stmt->close();
}


/**
 * Applies updates, logs changes. Minimal changes to core logic.
 */
function applyDatabaseUpdates(mysqli $conn, array $selectedUpdates, int $contractId): array {
    $results = [ /* Initial results */ 'status' => 'success', 'message' => 'Updates processed.', 'details' => ['vendor' => [], 'products' => [], 'pricebooks' => []], 'errors' => [], 'warnings' => [] ];

    // LOGGING: Attempt to determine baseVendorId for context, but DON'T error out here if not found.
    $baseVendorId = $selectedUpdates['vendor']['vendorid'] ?? $selectedUpdates['products'][0]['vendorid'] ?? null;
    if(!$baseVendorId && !empty($selectedUpdates['pricebooks'])) { /* Try find vendorId from first pricebook's product */
        $firstPricebook = reset($selectedUpdates['pricebooks']); $firstPbProductId = $firstPricebook['productid'] ?? null;
        if($firstPbProductId && is_numeric($firstPbProductId)) {
            $stmt_v = $conn->prepare("SELECT vendorid FROM tdu_products WHERE productid = ?");
            if($stmt_v){ $pIdInt=intval($firstPbProductId); $stmt_v->bind_param('i',$pIdInt); if($stmt_v->execute()){$res_v=$stmt_v->get_result(); if($res_v && $row_v = $res_v->fetch_assoc()){$baseVendorId = $row_v['vendorid'];} if($res_v) $res_v->free_result();} $stmt_v->close(); }
        }
    }
    // Convert to integer if found, otherwise remains null. logChange function will handle null vendorId if necessary.
    $baseVendorId = ($baseVendorId !== null && is_numeric($baseVendorId)) ? intval($baseVendorId) : null;
    // --- REMOVED Strict Validation Block ---
    // if (!$baseVendorId) { /* Error handling removed */ }


    // Start transaction
    if (!$conn->begin_transaction()) { /* Handle transaction start error */
         $results['status'] = 'error'; $results['message'] = 'Failed to start DB transaction.'; $results['errors'][] = 'Transaction begin failed: ' . $conn->error; return $results;
    }

    try {
        // --- 1. Update Vendor ---
        if (!empty($selectedUpdates['vendor']) && is_array($selectedUpdates['vendor'])) {
            $vendorData = $selectedUpdates['vendor'];
            $vendorId = intval($vendorData['vendorid'] ?? $baseVendorId ?? 0); // Fallback to 0 if base not found
            $oldVendorData = null; // LOGGING: For old values

            // Original validation for vendor update
            if ($vendorId <= 0) {
                 if (count($vendorData) > 1 || (count($vendorData) == 1 && !isset($vendorData['vendorid']))) { throw new Exception("Invalid or missing Vendor ID"); }
                 $vendorData = [];
            } else {
                 unset($vendorData['vendorid']); // Don't update PK

                 // LOGGING: Fetch old vendor data
                 if (!empty($vendorData)) {
                    $stmt_old_v = $conn->prepare("SELECT * FROM `tdu_vendors` WHERE `vendorid` = ?");
                    if(!$stmt_old_v) { throw new Exception("Log Vendor: Prep old data fail: ".$conn->error); } $stmt_old_v->bind_param('i', $vendorId); if(!$stmt_old_v->execute()) { $stmt_old_v->close(); throw new Exception("Log Vendor: Exec old data fail: ".$stmt_old_v->error); } $result_old_v = $stmt_old_v->get_result(); if ($result_old_v && $result_old_v->num_rows > 0) { $oldVendorData = $result_old_v->fetch_assoc(); } else { $results['warnings'][] = "Log Vendor: Vendor {$vendorId} not found. Update skipped."; $vendorData = []; } if($result_old_v) $result_old_v->free_result(); $stmt_old_v->close();
                 }
                 // END LOGGING FETCH
            }

            if (!empty($vendorData)) {
                // --- Original Logic for building update query ---
                $setClauses = []; $bindValues = []; $bindTypes = '';
                $vendorColumnMap = [ /* Your map */ 'vendorName'=>'vendorName', 'email'=>'email', 'paymentType'=>'paymentType','bookingMethod'=>'bookingMethod', 'vendorCountry'=>'vendorCountry', 'voucherRequired'=>'voucherRequired','description'=>'description', 'contactName'=>'contactName', 'phone'=>'phone','contactEmail'=>'contactEmail', 'ratings'=>'ratings', 'openingHours'=>'openingHours','cancelHoursGroup'=>'cancelHoursGroup', 'cancelHoursFIT'=>'cancelHoursFIT', 'address'=>'address','locationPhone'=>'locationPhone', 'types'=>'types', 'website'=>'website', 'preferred'=>'preferred','vendorActive'=>'vendorActive'];
                $processedValuesForLog = []; // LOGGING

                foreach ($vendorData as $key => $value) {
                    if (isset($vendorColumnMap[$key])) {
                        $column = $vendorColumnMap[$key]; $currentNewValue = null;
                        switch ($key) { /* Original processing logic */
                            case 'preferred': $bindTypes .= 'i'; $currentNewValue = ($value !== null && is_numeric($value)) ? intval($value) : 0; $bindValues[] = $currentNewValue; break; case 'vendorActive': case 'voucherRequired': $bindTypes .= 's'; $currentNewValue = ($value === true || strtolower(strval($value)) === 'yes' || $value === 1 || $value === '1') ? 'Yes' : 'No'; $bindValues[] = $currentNewValue; break; case 'ratings': case 'openingHours': case 'address': case 'locationPhone': case 'types': case 'website': $bindTypes .= 's'; $currentNewValue = ($value !== null) ? strval($value) : null; $bindValues[] = $currentNewValue; break; default: $bindTypes .= 's'; $currentNewValue = ($value !== null) ? strval($value) : ''; $bindValues[] = $currentNewValue; break;
                        }
                        $processedValuesForLog[$column] = $currentNewValue; // LOGGING
                        $setClauses[] = "`".$conn->real_escape_string($column)."` = ?";
                    } else { $results['warnings'][] = "Skip unknown vendor field: $key"; }
                }
                // --- End Original Logic ---

                // Execute Vendor Update (Original Logic)
                if (!empty($setClauses)) {
                    $sql = "UPDATE `tdu_vendors` SET ".implode(', ', $setClauses)." WHERE `vendorid` = ?";
                    $updateBindTypes = $bindTypes . 'i'; $updateBindValues = $bindValues; $updateBindValues[] = $vendorId;
                    $stmt = $conn->prepare($sql);
                    if (!$stmt) throw new Exception("Vendor update prep fail: ".$conn->error); if (strlen($updateBindTypes) !== count($updateBindValues)) throw new Exception("Vendor update bind count mismatch."); if (!$stmt->bind_param($updateBindTypes, ...$updateBindValues)) throw new Exception("Vendor update bind fail: ".$stmt->error); if (!$stmt->execute()) throw new Exception("Vendor update exec fail: ".$stmt->error);
                    $results['details']['vendor'][] = "Vendor {$vendorId} update attempt ({$stmt->affected_rows} rows)."; $stmt->close();

                    // LOGGING: Log changes after successful execution
                    if ($oldVendorData !== null) { foreach ($processedValuesForLog as $column => $currentNewValue) { $currentOldValueStr = isset($oldVendorData[$column]) ? strval($oldVendorData[$column]) : null; $currentNewValueStr = ($currentNewValue !== null) ? strval($currentNewValue) : null; logChange($conn, $vendorId, null, null, $contractId, 'Update', $column, $currentOldValueStr, $currentNewValueStr); } } // Note: logChange handles value comparison now
                    // END LOGGING
                }
            }
        } // End vendor update

        // --- 2. Update/Insert Products ---
        if (!empty($selectedUpdates['products']) && is_array($selectedUpdates['products'])) {
             $productColumnMap = [ /* Your map */ 'productid'=>'productid', 'vendorid'=>'vendorid', 'productName'=>'productName', 'category'=>'category','subcategory'=>'subcategory', 'city'=>'city', 'country'=>'country', 'productActive'=>'productActive','duration'=>'duration', 'seats'=>'seats', 'unitPrice'=>'unitPrice', 'childPrice'=>'childPrice','infantPrice'=>'infantPrice', 'familyPrice'=>'familyPrice', 'sharingPrice'=>'sharingPrice','triplePrice'=>'triplePrice', 'childWithBedPrice'=>'childWithBedPrice', 'childNoBedPrice'=>'childNoBedPrice','description'=>'description', 'ai_description'=>'ai_description', 'openingHours'=>'openingHours','startTime'=>'startTime', 'endTime'=>'endTime', 'productImage'=>'productImage', 'featured'=>'featured','preferred'=>'preferred', 'keywords'=>'keywords', 'childAge'=>'childAge', 'infantAge'=>'infantAge','parentid'=>'parentid', 'childOnly'=>'childOnly', 'childOnlyAgeMin'=>'childOnlyAgeMin','childOnlyAgeMax'=>'childOnlyAgeMax'];

            foreach ($selectedUpdates['products'] as $productData) {
                $productId = $productData['productid'] ?? null;
                $isNewProduct = ($productId === null || $productId === 'new' || $productId === '');
                $currentVendorId = intval($productData['vendorid'] ?? $baseVendorId ?? 0); // Use determined/fallback vendorID, default 0
                $oldProductData = null; // LOGGING
                $pIdInt = null; // LOGGING
                $processedValuesForLog = []; // LOGGING

                 // LOGGING: Fetch old product data if update
                if (!$isNewProduct) {
                    if (!$productId || !is_numeric($productId) || intval($productId) <= 0) { $results['errors'][] = "Invalid productid ('{$productId}') for update."; $results['status']='warning'; continue; }
                    $pIdInt = intval($productId);
                    $stmt_old_p = $conn->prepare("SELECT * FROM `tdu_products` WHERE `productid` = ?");
                    if(!$stmt_old_p) { throw new Exception("Log Prod: Prep old data failed: ".$conn->error); } $stmt_old_p->bind_param('i', $pIdInt); if(!$stmt_old_p->execute()) { $stmt_old_p->close(); throw new Exception("Log Prod: Exec old data failed: ".$stmt_old_p->error); } $result_old_p = $stmt_old_p->get_result();
                    if ($result_old_p && $result_old_p->num_rows > 0) { $oldProductData = $result_old_p->fetch_assoc(); } else { $results['warnings'][] = "Log Prod: Prod ID {$pIdInt} not found. Update skipped."; $result_old_p->free_result(); $stmt_old_p->close(); continue; }
                    if($result_old_p) $result_old_p->free_result(); $stmt_old_p->close();
                    // Update currentVendorId from fetched data for consistency in logging updates
                    $currentVendorId = intval($oldProductData['vendorid']);
                }
                // END LOGGING FETCH

                // --- Original Logic for building query ---
                $sqlData = []; $bindTypes = ''; $bindValues = [];
                if ($isNewProduct) { if ($currentVendorId <= 0) { $results['errors'][] = "Missing vendorid new product."; $results['status']='warning'; continue; } $sqlData['vendorid'] = "?"; $bindTypes .= 'i'; $bindValues[] = $currentVendorId; $processedValuesForLog['vendorid'] = $currentVendorId; } // LOGGING
                foreach ($productData as $key => $value) { if (isset($productColumnMap[$key])) { $column = $productColumnMap[$key]; if (!$isNewProduct && ($key === 'productid' || $key === 'vendorid')) continue; if ($isNewProduct && $key === 'vendorid') continue; $currentNewValue = null; switch ($key) { /* Original Processing */ case 'ai_description': case 'featured': case 'preferred': case 'parentid': case 'childOnly': case 'childOnlyAgeMin': case 'childOnlyAgeMax': $bindTypes .= 'i'; $currentNewValue = ($value !== null && is_numeric($value)) ? intval($value) : 0; $bindValues[] = $currentNewValue; break; case 'productName': case 'category': case 'subcategory': case 'city': case 'country': case 'duration': case 'seats': case 'description': case 'openingHours': case 'keywords': case 'childAge': case 'infantAge': $bindTypes .= 's'; $currentNewValue = ($value !== null) ? strval($value) : ''; $bindValues[] = $currentNewValue; break; case 'unitPrice': case 'childPrice': case 'infantPrice': case 'familyPrice': case 'sharingPrice': case 'triplePrice': case 'childWithBedPrice': case 'childNoBedPrice': $bindTypes .= 's'; $currentNewValue = ($value !== null && is_numeric($value)) ? number_format(floatval($value), 2, '.', '') : '0.00'; $bindValues[] = $currentNewValue; break; case 'productActive': $bindTypes .= 's'; $currentNewValue = ($value === true || strtolower(strval($value)) === 'yes' || $value === 1 || $value === '1') ? 'Yes' : 'No'; $bindValues[] = $currentNewValue; break; case 'startTime': case 'endTime': $bindTypes .= 's'; $currentNewValue = ($value && preg_match('/^\d{4}-\d{2}-\d{2}$/', $value)) ? $value : null; $bindValues[] = $currentNewValue; break; case 'productImage': $bindTypes .= 's'; $currentNewValue = ($value !== null) ? strval($value) : null; $bindValues[] = $currentNewValue; break; default: $results['warnings'][] = "Warn: Fallback product key: $key"; $bindTypes .= 's'; $currentNewValue = ($value !== null) ? strval($value) : null; $bindValues[] = $currentNewValue; break;} $processedValuesForLog[$column] = $currentNewValue; $sqlData[$column] = "?"; } else { $results['warnings'][] = "Skip unknown product field: $key"; } }
                // --- End Original Logic ---

                // --- Execute Product SQL ---
                 if (empty($sqlData) || ($isNewProduct && count($sqlData) <= 1 && isset($sqlData['vendorid']))) { if(!$isNewProduct) $results['warnings'][] = "No valid fields Prod ID {$pIdInt} update."; continue; }

                 if ($isNewProduct) { // INSERT (Original Logic)
                     if (!isset($processedValuesForLog['productName'])) { $results['errors'][] = "Skip Prod insert: Missing productName."; $results['status'] = 'warning'; continue; }
                     $insertColumnsSql = "`".implode('`, `', array_map([$conn, 'real_escape_string'], array_keys($sqlData)))."`"; $placeholders = implode(', ', array_values($sqlData)); $sql = "INSERT INTO `tdu_products` ($insertColumnsSql) VALUES ($placeholders)"; $stmt = $conn->prepare($sql); if (!$stmt) throw new Exception("Prod insert prep fail: ".$conn->error); if (strlen($bindTypes) !== count($bindValues)) throw new Exception("Prod insert bind count mismatch"); if (!$stmt->bind_param($bindTypes, ...$bindValues)) throw new Exception("Prod insert bind fail: ".$stmt->error); if (!$stmt->execute()) throw new Exception("Prod insert exec fail: ".$stmt->error); $newProdId = $conn->insert_id; $results['details']['products'][] = "New Product inserted (ID: {$newProdId})."; $stmt->close();
                     // LOGGING: Log inserted values
                     foreach($processedValuesForLog as $insertedCol => $insertedVal) { logChange($conn, $currentVendorId, $newProdId, null, $contractId, 'New', $insertedCol, null, strval($insertedVal)); } // Use $currentVendorId determined earlier
                     // END LOGGING
                 } else { // UPDATE (Original Logic)
                     $setClauses = []; foreach ($sqlData as $col => $placeholder) { $setClauses[] = "`".$conn->real_escape_string($col)."` = ".$placeholder; }
                     $updateBindTypes = $bindTypes . 'i'; $updateBindValues = $bindValues; $updateBindValues[] = $pIdInt;
                     if (!empty($setClauses)) {
                         $sql = "UPDATE `tdu_products` SET ".implode(', ', $setClauses)." WHERE `productid` = ?"; $stmt = $conn->prepare($sql); if (!$stmt) throw new Exception("Prod update prep fail (ID {$pIdInt}): ".$conn->error); if (strlen($updateBindTypes) !== count($updateBindValues)) throw new Exception("Prod update bind count mismatch (ID {$pIdInt})"); if (!$stmt->bind_param($updateBindTypes, ...$updateBindValues)) throw new Exception("Prod update bind fail (ID {$pIdInt}): ".$stmt->error); if (!$stmt->execute()) throw new Exception("Prod update exec fail (ID {$pIdInt}): ".$stmt->error); $affected_rows = $stmt->affected_rows; $results['details']['products'][] = "Prod ID {$pIdInt} update attempt ({$affected_rows} rows)."; if ($affected_rows === 0) { $results['warnings'][] = "Prod ID {$pIdInt} has no changes."; } $stmt->close();
                         // LOGGING: Log changed values after successful execution
                         if ($oldProductData !== null) { foreach ($processedValuesForLog as $column => $currentNewValue) { $currentOldValueStr = isset($oldProductData[$column]) ? strval($oldProductData[$column]) : null; $currentNewValueStr = ($currentNewValue !== null) ? strval($currentNewValue) : null; logChange($conn, $currentVendorId, $pIdInt, null, $contractId, 'Update', $column, $currentOldValueStr, $currentNewValueStr); } } // Use $currentVendorId determined from old data
                         // END LOGGING
                     } else { $results['warnings'][] = "No valid fields Prod ID {$pIdInt} update."; }
                 }
            } // End foreach product
        } // End if products exist


        // --- 3. Update/Insert Pricebooks ---
        if (!empty($selectedUpdates['pricebooks']) && is_array($selectedUpdates['pricebooks'])) {
             $pbColumnMap = [ /* Your map */ 'productid' => 'productid', 'name' => 'name', 'start_date' => 'start_date', 'end_date' => 'end_date', 'unit_price' => 'unit_price', 'child_price' => 'child_price', 'infant_price' => 'infant_price', 'sharing_price' => 'sharing_price', 'triple_price' => 'triple_price', 'child_with_bed_price' => 'child_with_bed_price', 'child_no_bed_price' => 'child_no_bed_price', 'price' => 'price', 'currency' => 'currency', 'description' => 'description', 'active' => 'active', 'dayOfWeek' => 'dayOfWeek'];

            foreach ($selectedUpdates['pricebooks'] as $pbData) { // Original iteration
                // Original Logic: Use _comparison_key from INSIDE the data
                $comparisonKey = $pbData['_comparison_key'] ?? null;
                unset($pbData['_comparison_key']); // Remove internal key

                // Original Logic: Determine if new/update based on internal key
                $isNewPricebook = ($comparisonKey === null || strpos(strval($comparisonKey), 'new_') === 0);
                $autoIdToUpdate = !$isNewPricebook ? intval($comparisonKey) : null; // Get ID if not new
                $oldPricebookData = null; // LOGGING
                $pbIdInt = $autoIdToUpdate; // LOGGING
                $processedValuesForLog = []; // LOGGING

                // Original Validation
                if ($comparisonKey === null && !$isNewPricebook) { $results['errors'][] = "Missing _comparison_key for PB."; $results['status'] = 'warning'; continue; }
                if (!$isNewPricebook && $autoIdToUpdate <= 0) { $results['errors'][] = "Invalid auto_id ('{$comparisonKey}') from _comparison_key."; $results['status'] = 'warning'; continue; }

                 // LOGGING: Fetch old pricebook data if update
                 if (!$isNewPricebook && $autoIdToUpdate) {
                    $stmt_old_pb = $conn->prepare("SELECT * FROM `tdu_pricebook` WHERE `auto_id` = ?");
                    if(!$stmt_old_pb) { throw new Exception("Log PB: Prep old data failed: ".$conn->error); } $stmt_old_pb->bind_param('i', $autoIdToUpdate); if(!$stmt_old_pb->execute()) { $stmt_old_pb->close(); throw new Exception("Log PB: Exec old data failed: ".$stmt_old_pb->error); } $result_old_pb = $stmt_old_pb->get_result();
                    if ($result_old_pb && $result_old_pb->num_rows > 0) { $oldPricebookData = $result_old_pb->fetch_assoc(); } else { $results['warnings'][] = "Log PB: PB auto_id {$autoIdToUpdate} not found. Update skipped."; $result_old_pb->free_result(); $stmt_old_pb->close(); continue; }
                    if($result_old_pb) $result_old_pb->free_result(); $stmt_old_pb->close();
                 }
                // END LOGGING FETCH

                 // Determine productid - needed early for validation/logging consistency
                 $currentProductId = intval($pbData['productid'] ?? $oldPricebookData['productid'] ?? 0);
                 if ($currentProductId <= 0 && isset($pbData['productid'])) { $results['errors'][] = "Skipping PB (Key: {$comparisonKey}): Invalid productid."; $results['status'] = 'warning'; continue; }
                 elseif($currentProductId <= 0 && $isNewPricebook) { $results['errors'][] = "Skipping PB (Key: {$comparisonKey}): Missing productid new entry."; $results['status'] = 'warning'; continue; }


                // --- Original Logic for building query ---
                $sqlData = []; $bindTypes = ''; $bindValues = [];
                foreach ($pbData as $key => $value) {
                     if (isset($pbColumnMap[$key])) {
                         $column = $pbColumnMap[$key];
                         if (!$isNewPricebook && ($key === 'auto_id')) continue; // Skip PK update

                         $currentNewValue = null; // Processed value

                         // Determine bind type and process value (Original Logic)
                          switch ($key) {
                              case 'productid': $bindTypes .= 'i'; $currentNewValue = ($value !== null && is_numeric($value)) ? intval($value) : 0; $bindValues[] = $currentNewValue; break; // Process productid here
                              case 'unit_price': case 'child_price': case 'infant_price': case 'sharing_price': case 'triple_price': case 'child_with_bed_price': case 'child_no_bed_price': case 'price': $bindTypes .= 's'; $currentNewValue = ($value !== null && is_numeric($value)) ? number_format(floatval($value), 2, '.', '') : '0.00'; $bindValues[] = $currentNewValue; break;
                              case 'active': $bindTypes .= 's'; $currentNewValue = ($value === true || strtolower(strval($value)) === 'yes' || $value === 1 || $value === '1') ? 'Yes' : 'No'; $bindValues[] = $currentNewValue; break;
                              case 'start_date': case 'end_date': $bindTypes .= 's'; $currentNewValue = ($value && preg_match('/^\d{4}-\d{2}-\d{2}$/', $value)) ? $value : null; $bindValues[] = $currentNewValue; break;
                              case 'name': case 'description': $bindTypes .= 's'; $currentNewValue = ($value !== null) ? strval($value) : null; $bindValues[] = $currentNewValue; break;
                              case 'currency': case 'dayOfWeek': $bindTypes .= 's'; $currentNewValue = ($value !== null) ? strval($value) : ''; $bindValues[] = $currentNewValue; break;
                              default: $results['warnings'][] = "Warn: Fallback PB key: $key"; $bindTypes .= 's'; $currentNewValue = ($value !== null) ? strval($value) : null; $bindValues[] = $currentNewValue; break;
                          }
                          $processedValuesForLog[$column] = $currentNewValue; // LOGGING
                          $sqlData[$column] = "?"; // For building columns/set clauses
                     } else { $results['warnings'][] = "Skip unknown PB field: $key"; }
                }
                 // --- End Original Logic ---

                // --- Execute Pricebook SQL ---
                if (empty($sqlData)) { if(!$isNewPricebook) $results['warnings'][] = "No valid fields for PB ID {$autoIdToUpdate}."; continue; }

                if ($isNewPricebook) { // INSERT (Original Logic)
                    if (!isset($processedValuesForLog['productid']) || $processedValuesForLog['productid'] <= 0) { $results['errors'][] = "Skip PB insert (Key: {$comparisonKey}): Missing/invalid productid."; $results['status'] = 'warning'; continue; }
                    $insertColumnsSql = "`".implode('`, `', array_map([$conn, 'real_escape_string'], array_keys($sqlData)))."`"; $placeholders = implode(', ', array_values($sqlData)); $sql = "INSERT INTO `tdu_pricebook` ($insertColumnsSql) VALUES ($placeholders)"; $stmt = $conn->prepare($sql); if (!$stmt) throw new Exception("PB insert prep fail (Key: {$comparisonKey}): ".$conn->error); if (strlen($bindTypes) !== count($bindValues)) throw new Exception("PB insert bind count mismatch"); if (!$stmt->bind_param($bindTypes, ...$bindValues)) throw new Exception("PB insert bind fail: ".$stmt->error); if (!$stmt->execute()) throw new Exception("PB insert exec fail: ".$stmt->error); $newPbId = $conn->insert_id; $results['details']['pricebooks'][] = "New PB inserted (ID: {$newPbId}) from key '{$comparisonKey}'."; $stmt->close();
                    // LOGGING: Log inserted values
                    $logProductId = $processedValuesForLog['productid']; // Get the definite product ID inserted
                    foreach($processedValuesForLog as $insertedCol => $insertedVal) { logChange($conn, $baseVendorId, $logProductId, $newPbId, $contractId, 'New', $insertedCol, null, strval($insertedVal)); }
                    // END LOGGING
                } else { // UPDATE (Original Logic)
                    $setClauses = []; foreach ($sqlData as $col => $placeholder) { $setClauses[] = "`".$conn->real_escape_string($col)."` = ".$placeholder; }
                    $updateBindTypes = $bindTypes . 'i'; $updateBindValues = $bindValues; $updateBindValues[] = $autoIdToUpdate;
                    if (!empty($setClauses)) {
                        $sql = "UPDATE `tdu_pricebook` SET ".implode(', ', $setClauses)." WHERE `auto_id` = ?"; $stmt = $conn->prepare($sql); if (!$stmt) throw new Exception("PB update prep fail (ID {$autoIdToUpdate}): ".$conn->error); if (strlen($updateBindTypes) !== count($updateBindValues)) throw new Exception("PB update bind count mismatch (ID: {$autoIdToUpdate})"); if (!$stmt->bind_param($updateBindTypes, ...$updateBindValues)) throw new Exception("PB update bind fail (ID {$autoIdToUpdate}): ".$stmt->error); if (!$stmt->execute()) throw new Exception("PB update exec fail (ID {$autoIdToUpdate}): ".$stmt->error); $affected_rows = $stmt->affected_rows; $results['details']['pricebooks'][] = "PB ID {$autoIdToUpdate} update attempt ({$affected_rows} rows)."; if ($affected_rows === 0) { $results['warnings'][] = "PB ID {$autoIdToUpdate} has no changes."; } $stmt->close();
                        // LOGGING: Log changed values after successful execution
                        if ($oldPricebookData !== null) { $logProductId = intval($oldPricebookData['productid']); foreach ($processedValuesForLog as $column => $currentNewValue) { $currentOldValueStr = isset($oldPricebookData[$column]) ? strval($oldPricebookData[$column]) : null; $currentNewValueStr = ($currentNewValue !== null) ? strval($currentNewValue) : null; logChange($conn, $baseVendorId, $logProductId, $autoIdToUpdate, $contractId, 'Update', $column, $currentOldValueStr, $currentNewValueStr); } }
                        // END LOGGING
                    } else { $results['warnings'][] = "No valid fields for PB ID {$autoIdToUpdate} update."; }
                }
            } // End foreach pricebook
        } // End if pricebooks exist


        // --- Commit Transaction ---
        if (!$conn->commit()) {
            $conn->rollback(); throw new Exception("Transaction commit failed: " . $conn->error . " - Rollback attempted.");
        }

        // Determine final status (Original Logic - based on errors accumulated)
        if (empty($results['errors'])) { if (!empty($results['warnings'])) { $results['status'] = 'warning'; $results['message'] = 'Updates applied with warnings.'; } else { $results['status'] = 'success'; $results['message'] = 'Updates applied successfully.'; } }
        else { $results['status'] = 'error'; $results['message'] = 'Update process encountered errors. Changes might be rolled back.'; }

    } catch (Exception $e) {
        // Exception handling (Original Logic + Enhanced Logging)
        $conn->rollback(); $results['status'] = 'error'; $results['message'] = 'An error occurred during update. Transaction rolled back.'; $dbError = $conn->error; $errorMessage = $e->getMessage() . ($dbError && strpos($e->getMessage(), $dbError) === false ? " | DB Error: ".$dbError : ""); $results['errors'][] = $errorMessage; error_log("Apply Updates Exception Contract ID {$contractId}: ".$errorMessage); if (http_response_code() < 400) { http_response_code(500); }
    }

    return $results;
} // --- End of applyDatabaseUpdates Function Definition ---


// --- Main Execution Flow ---
$update_response = applyDatabaseUpdates($conn, $selectedUpdates, $contract_auto_id); // LOGGING: Pass contract_id

// --- Reset Job Status --- (Original Logic)
if ($update_response['status'] !== 'error') {
    error_log("DB Updates OK (Status: {$update_response['status']}). Reset job contract_id: {$contract_auto_id}");
    try { /* Find and reset job logic remains the same */
        $latest_job_id = null; $stmt_find_job = $conn->prepare("SELECT id FROM tdu_ai_contract_comparison_jobs WHERE contract_auto_id = ? ORDER BY id DESC LIMIT 1"); if (!$stmt_find_job) { throw new Exception("Prep find job fail: ".$conn->error); } $stmt_find_job->bind_param('i', $contract_auto_id); if (!$stmt_find_job->execute()) { throw new Exception("Exec find job fail: ".$stmt_find_job->error); } $result_job = $stmt_find_job->get_result(); if ($result_job && $result_job->num_rows > 0) { $latest_job_id = (int)$result_job->fetch_assoc()['id']; } if($stmt_find_job) $stmt_find_job->close(); if ($result_job) $result_job->free_result();
        if ($latest_job_id) { $stmt_reset = $conn->prepare("UPDATE tdu_ai_contract_comparison_jobs SET status = 'PENDING', gemini_result = NULL, error_message = NULL, processing_started_at = NULL, completed_at = NULL WHERE id = ?"); if (!$stmt_reset) { throw new Exception("Prep reset job fail: ".$conn->error); } $stmt_reset->bind_param('i', $latest_job_id); if ($stmt_reset->execute()) { $affected_rows_reset = $stmt_reset->affected_rows; if ($affected_rows_reset > 0) { if (strpos($update_response['message'], 'Job reset to PENDING.') === false) { $update_response['message'] .= ' Job reset to PENDING.'; } error_log("Reset job ID {$latest_job_id} OK."); } else { $reset_warning = "Job ID {$latest_job_id} reset 0 rows."; $update_response['warnings'][] = $reset_warning; error_log($reset_warning); } } else { throw new Exception("Exec reset job fail: ".$stmt_reset->error); } if($stmt_reset) $stmt_reset->close(); }
        else { $no_job_warning = "No job found to reset contract ID {$contract_auto_id}."; $update_response['warnings'][] = $no_job_warning; error_log($no_job_warning); }
    } catch (Exception $e) { /* Handle reset job error */ $reset_error_msg = "Non-fatal error during job reset: ".$e->getMessage(); error_log($reset_error_msg); $update_response['errors'][] = $reset_error_msg; if ($update_response['status'] === 'success') { $update_response['status'] = 'warning'; if (strpos($update_response['message'], 'error resetting job') === false) { $update_response['message'] = rtrim($update_response['message'], '.') . ' but encountered error resetting job status.'; } } }
} else { error_log("DB Updates status '{$update_response['status']}'. Skip job reset contract_id: {$contract_auto_id}"); }

// --- Send Final Response --- (Original Logic)
if ($update_response['status'] === 'error') { if (http_response_code() < 400) { http_response_code(500); } } else { http_response_code(200); }
echo json_encode($update_response, JSON_INVALID_UTF8_SUBSTITUTE | JSON_UNESCAPED_UNICODE);
if (isset($conn) && $conn instanceof mysqli && $conn->ping()) { $conn->close(); }
exit;

?>