<?php
/**
 * AJAX Endpoint: ajax_get_comparison_status_and_results.php
 * Input (POST): contract_id (auto_id from tdu_vendors_contracts)
 * Output (JSON): Status/results of the latest job associated with that specific contract_auto_id.
 * --- Updated with full error handling and corrected current data fetching ---
 */

// --- Basic Setup ---
// error_reporting(E_ALL); // Enable for dev if needed
ini_set('display_errors', 0); // Keep errors off in production AJAX
ini_set('log_errors', 1);
// Optionally set a specific error log file: 
ini_set('error_log', '/contract_compare_error.log');
header('Content-Type: application/json; charset=utf-8'); // Added charset

// --- Includes ---
// Use absolute paths for reliability
require_once __DIR__ . "/../dbconn.php";
// *** Make sure this includes the updated version with getProductsAndPricebooks ***
require_once __DIR__ . "/contract_comparison_functions.php";

// --- Response Structure ---
$response = ['status' => 'error', 'message' => 'An unknown error occurred.'];

// --- Input Validation ---
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405); $response['message'] = 'Invalid request method.'; echo json_encode($response); exit;
}
if (!isset($_POST['contract_id']) || !is_numeric($_POST['contract_id']) || intval($_POST['contract_id']) <= 0) {
    http_response_code(400); $response['message'] = 'Missing or invalid contract_id.'; echo json_encode($response); exit;
}
$contract_auto_id = intval($_POST['contract_id']);

// --- Database Connection Check ---
if (!isset($conn) || !($conn instanceof mysqli) || $conn->connect_error) { // Added type check
    http_response_code(500); $response['message'] = 'Database connection failed.'; error_log("AJAX Get Status: DB connection error - " . ($conn->connect_error ?? 'Unknown error')); echo json_encode($response); exit;
}
if (!$conn->set_charset('utf8mb4')) { http_response_code(500); $response['message'] = 'Database character set failed.'; error_log("AJAX Get Status: DB charset error - " . $conn->error); echo json_encode($response); exit; }


// --- Core Logic ---
try {
    // 1. Find the latest job DIRECTLY using contract_auto_id
    // This function already exists in the updated contract_comparison_functions.php
    $latest_job = getLatestJobByContractID($contract_auto_id, $conn);

    // 2. Determine response based on job status
    if (!$latest_job) {
        // No job exists specifically for this contract_auto_id.
        // This could mean it was never queued, or maybe deleted.
        // Respond with PENDING might be misleading if it was never queued.
        // Consider a 'NOT_FOUND' or specific message. Let's stick to PENDING for now as per original logic.
        http_response_code(200); // It's not an error, just no job found/processed yet
        $response = ['status' => 'PENDING', 'message' => 'No comparison job found or initiated yet for this specific contract entry. It might be queued soon.'];

    } else {
        $job_status = $latest_job['status'];
        $job_id = $latest_job['id'];
        $vendor_id = $latest_job['vendor_id']; // Get vendor ID from the job record

        // Check if vendor_id is valid before proceeding, especially for DONE status
        if (empty($vendor_id) || $vendor_id <= 0) {
             error_log("Job ID {$job_id} (Contract {$contract_auto_id}) has invalid vendor_id: {$vendor_id}. Status: {$job_status}");
             // If job is done but vendor invalid, it's an error state for fetching current data
             if ($job_status === 'DONE') {
                 $response = ['status' => 'ERROR', 'message' => 'Comparison job data is inconsistent (invalid vendor). Please regenerate.'];
             } else {
                 // For other statuses, return the status but log the inconsistency
                 $response['status'] = $job_status; // Keep original status
                 $response['message'] = ($job_status === 'ERROR') ? ($latest_job['error_message'] ?? 'Job error with inconsistent data.') : "Job status is {$job_status}, but associated data is inconsistent.";
             }
        } else {
            // Vendor ID is valid, proceed based on status
            //error_log("Processing Job ID {$job_id} for Contract {$contract_auto_id} with Status: {$job_status}");

            switch ($job_status) {
                case 'DONE':
                    $gemini_data = json_decode($latest_job['gemini_result'] ?? '', true);
                    if (json_last_error() !== JSON_ERROR_NONE || !is_array($gemini_data)) {
                        error_log("Job ID {$job_id} (Contract {$contract_auto_id}) status DONE but invalid gemini_result JSON.");
                         $response = ['status' => 'ERROR', 'message' => 'Comparison finished, but result data is corrupted. Please regenerate.'];
                         break; // Exit switch
                    }

                    // Fetch current DB data for the vendor associated with THIS job
                    // *** Use the updated functions from contract_comparison_functions.php ***
                    $currentVendorInfo = getVendor($vendor_id, $conn) ?? []; // Fetch current vendor
                    $currentProductsWithNestedPBs = getProductsAndPricebooks($vendor_id, $conn); // Fetch products AND nested pricebooks

                    // ** Prepare data for FINAL RESPONSE (separate products/pricebooks) **
                    $finalCurrentProductsInfo = []; // Products only
                    $currentPricebooksInfoOnly = []; // Pricebooks keyed by auto_id
                    foreach ($currentProductsWithNestedPBs as $product) {
                        if (!empty($product['pricebooks_nested']) && is_array($product['pricebooks_nested'])) {
                            foreach ($product['pricebooks_nested'] as $pb) {
                                if (isset($pb['auto_id'])) {
                                    $currentPricebooksInfoOnly[(string)$pb['auto_id']] = $pb; // Key by string auto_id
                                }
                            }
                        }
                        unset($product['pricebooks_nested']); // Remove nested data before adding to final product list
                        $finalCurrentProductsInfo[] = $product;
                    }
                    // Make sure pricebooks is an object even if empty, as JS might expect it
                    if (empty($currentPricebooksInfoOnly)) {
                         $currentPricebooksInfoOnly = (object)[];
                    }


                    // Construct the final response structure matching JS expectations
                    $response = [
                        'status' => 'DONE',
                        'reportHtml' => $gemini_data['reportHtml'] ?? '<p>Report HTML not found in processed data.</p>',
                        'currentDbData' => [
                            'vendor' => $currentVendorInfo,
                            'products' => $finalCurrentProductsInfo, // Use the cleaned list
                            'pricebooks' => $currentPricebooksInfoOnly // Use the separated list (keyed object)
                        ],
                        'suggestedAiData' => [
                            // Ensure fallback to empty object/array if keys missing in gemini_data
                            'vendor' => $gemini_data['correctedVendor'] ?? (object)[],
                            'products' => $gemini_data['correctedProducts'] ?? [],
                            'pricebooks' => $gemini_data['correctedPriceBooks'] ?? (object)[]
                        ]
                    ];
                    break; // Exit DONE case

                case 'PENDING':
                    $response = ['status' => 'PENDING', 'message' => 'Comparison is queued and waiting to be processed.'];
                    break;
                case 'PROCESSING':
                    $response = ['status' => 'PROCESSING', 'message' => 'Comparison is currently being processed by the background system.'];
                    break;
                case 'ERROR':
                     $response = ['status' => 'ERROR', 'message' => $latest_job['error_message'] ?? 'An unknown error occurred during processing.'];
                     break;
                default:
                     $response = ['status' => 'ERROR', 'message' => "Unknown job status found: {$job_status}"];
                     error_log("Unknown job status '{$job_status}' for job ID {$job_id} (Contract {$contract_auto_id})");
                     break;
            } // End switch
        } // End else (valid vendor_id)
    } // End else (job exists)

    http_response_code(200); // OK status for valid processing or found status

} catch (Exception $e) {
    http_response_code(500); // Internal Server Error on exceptions
    $response['status'] = 'error'; // Ensure status reflects error
    $response['message'] = 'An server exception occurred while fetching status.'; // Keep message generic for client
    // $response['debug_error'] = $e->getMessage(); // Optionally include for debugging (remove/secure in production)
    error_log("AJAX Get Status Exception for Contract {$contract_auto_id}: " . $e->getMessage() . "\nTrace: " . $e->getTraceAsString());
}

// --- Send Response & Close Connection ---
// Use flags for better JSON encoding, substitute invalid UTF8 chars
echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_INVALID_UTF8_SUBSTITUTE);

// Close connection if it's open and valid
if (isset($conn) && $conn instanceof mysqli && !$conn->connect_error) {
    $conn->close();
}
exit;
?>