<?php
/**
 * AJAX Endpoint: ajax_regenerate_comparison.php
 * Input (POST): contract_id (auto_id from tdu_vendors_contracts)
 * Action: Finds the latest job associated with this contract_auto_id and resets it to PENDING.
 * Output (JSON): Success or error status.
 * --- Updated with full error handling ---
 */

// --- Basic Setup ---
error_reporting(E_ALL); ini_set('display_errors', 0); ini_set('log_errors', 1);
ini_set('error_log', '/contract_compare_error.log');
header('Content-Type: application/json');

// --- Includes ---
require_once "../dbconn.php"; // Provides $conn

// --- Response Structure ---
$response = ['status' => 'error', 'message' => 'Failed to regenerate comparison.'];

// --- Input Validation ---
if ($_SERVER['REQUEST_METHOD'] !== 'POST') { http_response_code(405); $response['message'] = 'Invalid request method.'; echo json_encode($response); exit; }
if (!isset($_POST['contract_id']) || !is_numeric($_POST['contract_id']) || intval($_POST['contract_id']) <= 0) { http_response_code(400); $response['message'] = 'Missing or invalid contract_id.'; echo json_encode($response); exit; }
$contract_auto_id = intval($_POST['contract_id']);

// --- Database Connection Check ---
if (!isset($conn) || $conn->connect_error) { http_response_code(500); $response['message'] = 'Database connection failed.'; error_log("AJAX Regenerate: DB connection error - " . ($conn->connect_error ?? 'Unknown error')); echo json_encode($response); exit; }
if (!$conn->set_charset('utf8mb4')) { http_response_code(500); $response['message'] = 'Database character set failed.'; error_log("AJAX Regenerate: DB charset error - " . $conn->error); echo json_encode($response); exit; }

// --- Core Logic ---
try {
    // 1. Find the ID of the latest job for this specific contract_auto_id
    $latest_job_id = null;
    $stmt_find_job = $conn->prepare("SELECT id FROM tdu_ai_contract_comparison_jobs WHERE contract_auto_id = ? ORDER BY id DESC LIMIT 1");
    if (!$stmt_find_job) { throw new Exception("Prepare failed (find job id): " . $conn->error); }

    $stmt_find_job->bind_param('i', $contract_auto_id);
    if (!$stmt_find_job->execute()) { throw new Exception("Execute failed (find job id): " . $stmt_find_job->error); }

    $result_job = $stmt_find_job->get_result();
    if (!$result_job) { throw new Exception("Get result failed (find job id): " . $conn->error); }

    if ($result_job->num_rows > 0) { $latest_job_id = (int)$result_job->fetch_assoc()['id']; }
    $stmt_find_job->close();
    $result_job->free_result();

    if (!$latest_job_id) {
        // No job found for this contract, nothing to regenerate
        $response = ['status' => 'success', 'message' => 'No existing job found for this specific contract entry. Nothing to regenerate.'];
        http_response_code(200);
    } else {
        // 2. Update THIS specific job to reset it
        $stmt_reset = $conn->prepare("UPDATE tdu_ai_contract_comparison_jobs SET status = 'PENDING', gemini_result = NULL, error_message = NULL, processing_started_at = NULL, completed_at = NULL WHERE id = ?");
        if (!$stmt_reset) { throw new Exception("Prepare failed (reset job): " . $conn->error); }

        $stmt_reset->bind_param('i', $latest_job_id);
        if ($stmt_reset->execute()) {
            if ($stmt_reset->affected_rows > 0) {
                $response = ['status' => 'success', 'message' => "Comparison job ID {$latest_job_id} reset to PENDING."];
                //error_log("Regenerate successful for job ID {$latest_job_id}");
            } else {
                $response = ['status' => 'success', 'message' => "Job ID {$latest_job_id} was not updated (possibly already PENDING or ID invalid)."];
                error_log("Regenerate query OK for job ID {$latest_job_id}, but 0 rows affected.");
            }
            http_response_code(200);
        } else {
            throw new Exception("Execute failed (reset job): " . $stmt_reset->error);
        }
        $stmt_reset->close();
    }

} catch (Exception $e) {
    http_response_code(500); // Internal Server Error on exceptions
    $response['message'] = 'An server exception occurred during regeneration.'; // Keep message generic
    $response['errors'] = [$e->getMessage()]; // Optional: include detail for debugging
    error_log("AJAX Regenerate Exception: " . $e->getMessage());
}

// --- Send Response & Close Connection ---
echo json_encode($response, JSON_INVALID_UTF8_SUBSTITUTE);
if (isset($conn) && $conn instanceof mysqli) {
    $conn->close();
}
exit;
?>