<?php
/**
 * Functions File: contract_comparison_functions.php
 * Core logic & DB interactions for AI contract comparison jobs.
 * Includes table creation, job status checks, job creation, and processing helpers.
 * --- Updated with BASIC/UNRELIABLE text extraction for DOCX/XLSX ---
 * --- Updated to include NESTED PRICEBOOKS in data fetching and AI prompt ---
 */
require_once __DIR__."/../file_extract_functions.php"; // Assumed to contain text extraction functions
require_once __DIR__."/../ai_keys.php"; // Provides apiURL_thinking()
ini_set('error_log', '/contract_compare_error.log');
// --- Database Table Creation ---

/**
 * Creates necessary DB tables if they don't exist.
 * (Unchanged from your provided code)
 */
function createDatabaseTablesIfNeeded(mysqli $conn): bool {
    $all_success = true;

    // SQL for the main jobs table
    $jobs_table_sql = "CREATE TABLE IF NOT EXISTS `tdu_ai_contract_comparison_jobs` (
          `id` INT AUTO_INCREMENT PRIMARY KEY,
          `vendor_id` INT NOT NULL,
          `contract_auto_id` INT NOT NULL,
          `contract_file_path` VARCHAR(1024) NOT NULL,
          `status` ENUM('PENDING', 'PROCESSING', 'DONE', 'ERROR') NOT NULL DEFAULT 'PENDING',
          `gemini_result` JSON NULL,
          `error_message` TEXT NULL,
          `created_at` TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP,
          `processing_started_at` TIMESTAMP NULL DEFAULT NULL,
          `completed_at` TIMESTAMP NULL DEFAULT NULL,
          INDEX `idx_status_created` (`status`, `created_at`),
          INDEX `idx_vendor_id` (`vendor_id`),
          INDEX `idx_contract_auto_id` (`contract_auto_id`),
          FOREIGN KEY (`contract_auto_id`) REFERENCES `tdu_vendors_contracts` (`auto_id`) ON DELETE CASCADE ON UPDATE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;";

    if (!$conn->query($jobs_table_sql)) {
        error_log("Failed to create/verify tdu_ai_contract_comparison_jobs table: " . $conn->error);
        $all_success = false;
    }

    // SQL for the vendor contracts table
    $vendor_contracts_table_sql = "CREATE TABLE IF NOT EXISTS `tdu_vendors_contracts` (
          `auto_id` INT AUTO_INCREMENT PRIMARY KEY,
          `vendorid` INT NOT NULL,
          `contractname` VARCHAR(200) NOT NULL,
          `start_time` DATE NULL,
          `end_time` DATE NULL,
          `attachment_path` VARCHAR(300) NOT NULL,
          `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP
          -- Optional FK to tdu_vendors
          -- , FOREIGN KEY (`vendorid`) REFERENCES `tdu_vendors` (`vendorid`) ON DELETE CASCADE ON UPDATE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;";

     if (!$conn->query($vendor_contracts_table_sql)) {
        error_log("Failed to create/verify tdu_vendors_contracts table: " . $conn->error);
        $all_success = false;
    }

    return $all_success;
}


// --- Database Interaction Functions ---

/**
 * Inserts a new PENDING job.
 * (Unchanged from your provided code)
 */
function insertNewJob(int $vendorId, int $contractAutoId, string $contractPath, mysqli $conn){
    if ($vendorId <= 0 || $contractAutoId <= 0 || empty($contractPath) || strlen($contractPath) > 1024) {
        error_log("Invalid parameters for insertNewJob (Vendor: {$vendorId}, ContractAutoId: {$contractAutoId})");
        return false;
    }

    $sql = "INSERT INTO tdu_ai_contract_comparison_jobs (vendor_id, contract_auto_id, contract_file_path, status)
            VALUES (?, ?, ?, 'PENDING')";
    $stmt = $conn->prepare($sql);
    if (!$stmt) { error_log("DB Error preparing insert new job: " . $conn->error); return false; }
    $stmt->bind_param('iis', $vendorId, $contractAutoId, $contractPath);

    if ($stmt->execute()) {
        $newJobId = $conn->insert_id;
        error_log("Inserted new PENDING job ID {$newJobId} for vendor {$vendorId}, contract auto_id {$contractAutoId}.");
        $stmt->close();
        return (int)$newJobId;
    } else {
        error_log("DB Error executing insert new job for vendor {$vendorId}, contract {$contractAutoId}: " . $stmt->error . " (Code: {$stmt->errno})");
        $stmt->close();
        return false;
    }
}

/**
 * Checks the status of the most recent job for a specific vendor.
 * (Unchanged from your provided code)
 */
function getLatestVendorJobStatus(int $vendorId, mysqli $conn): ?string {
    if ($vendorId <= 0) { return null; }
    $sql = "SELECT status FROM tdu_ai_contract_comparison_jobs WHERE vendor_id = ? ORDER BY id DESC LIMIT 1";
    $stmt = $conn->prepare($sql);
    if (!$stmt) { error_log("DB Error preparing latest vendor status check: " . $conn->error); return null; }
    $stmt->bind_param('i', $vendorId);
    if (!$stmt->execute()) { error_log("DB Error executing latest vendor status check: " . $stmt->error); $stmt->close(); return null;}
    $result = $stmt->get_result();
    if (!$result) { error_log("DB Error getting result for latest vendor status check: " . $conn->error); $stmt->close(); return null; }
    $status = 'PENDING'; // Default assumption if no jobs found? Or maybe null? Adjust if needed.
    if ($result->num_rows > 0) { $status = $result->fetch_assoc()['status']; }
    $stmt->close(); $result->free_result();
    return $status;
}

/**
 * Gets the details of the latest job associated with a specific contract auto_id.
 * (Unchanged from your provided code)
 */
function getLatestJobByContractID(int $contractAutoId, mysqli $conn): ?array {
    if ($contractAutoId <= 0) { return null; }
    $sql = "SELECT id, vendor_id, status, gemini_result, error_message FROM tdu_ai_contract_comparison_jobs WHERE contract_auto_id = ? ORDER BY id DESC LIMIT 1";
    $stmt = $conn->prepare($sql);
    if (!$stmt) { error_log("DB Error preparing job status by contract: " . $conn->error); return null; }
    $stmt->bind_param('i', $contractAutoId);
    if (!$stmt->execute()) { error_log("DB Error executing job status by contract: " . $stmt->error); $stmt->close(); return null; }
    $result = $stmt->get_result(); $jobData = null;
    if ($result && $result->num_rows > 0) { $jobData = $result->fetch_assoc(); }
    $stmt->close(); if ($result) { $result->free_result(); }
    return $jobData;
}

/**
 * Fetches basic vendor data by ID.
 * (Unchanged from your provided code, using prepared statement)
 */
function getVendor(int $vendorid, mysqli $conn): ?array {
    $vendor = null; if ($vendorid <= 0) { return null; }
    $sql = "SELECT * FROM tdu_vendors v WHERE v.vendorid = ? LIMIT 1";
    $stmt = $conn->prepare($sql); if(!$stmt) { error_log("DB Error preparing getVendor: " . $conn->error); return null; }
    $stmt->bind_param('i', $vendorid); if (!$stmt->execute()) { error_log("DB Error executing getVendor: " . $stmt->error); $stmt->close(); return null; }
    $result = $stmt->get_result(); if ($result && $result->num_rows > 0) { $vendor = $result->fetch_assoc(); } elseif (!$result) { error_log("DB Error in getVendor (get_result failed): " . $conn->error); }
    $stmt->close(); if ($result) { $result->free_result(); }
    return $vendor;
}

/**
 * Fetches active products AND their nested pricebooks for a vendor.
 * (REPLACED simple getProducts with logic from original ajax endpoint)
 */
function getProductsAndPricebooks(int $vendorid, mysqli $conn): array {
    $logPrefix = "[JobProcessing_GetPnP]"; // Prefix for cron context
    error_log("{$logPrefix} Fetching products/pricebooks for Vendor ID: $vendorid");

    $products = [];
    if ($vendorid <= 0) {
        error_log("{$logPrefix} Invalid vendor ID received: $vendorid");
        return [];
    }

    // Fetch products using prepared statements
    $sql = "SELECT p.* FROM tdu_products p WHERE vendorid=? AND productActive = 'Yes' ORDER BY productName ASC";
    $stmt = $conn->prepare($sql);
    if (!$stmt) {
        error_log("{$logPrefix} DB Error preparing product fetch: " . $conn->error);
        return [];
    }
    $stmt->bind_param('i', $vendorid);
    if (!$stmt->execute()) {
        error_log("{$logPrefix} DB Error executing product fetch: " . $stmt->error);
        $stmt->close();
        return [];
    }

    $result = $stmt->get_result();
    if (!$result) {
        error_log("{$logPrefix} DB Error getting product result: " . $conn->error);
        $stmt->close();
        return [];
    }

    $product_rows = [];
    $fetched_product_count = 0;
    while ($row = $result->fetch_assoc()) {
        $product_rows[intval($row['productid'])] = $row;
        $fetched_product_count++;
    }
    $stmt->close();
    $result->free_result();
    //error_log("{$logPrefix} Fetched $fetched_product_count products.");

    if (empty($product_rows)) {
        error_log("{$logPrefix} No active products found for vendor.");
        return [];
    }

    $productIds = array_keys($product_rows);
    //error_log("{$logPrefix} Product IDs found: " . implode(', ', $productIds));

    // Fetch associated pricebooks using prepared statements
    $pricebooks = [];
    $fetched_pricebook_count = 0;
    if (!empty($productIds)) {
        // Create placeholders for IN clause
        $placeholders = implode(',', array_fill(0, count($productIds), '?'));
        $types = str_repeat('i', count($productIds)); // Assuming productid is integer

        $sqlb = "SELECT * FROM tdu_pricebook WHERE productid IN ($placeholders)";
        $stmtb = $conn->prepare($sqlb);

        if ($stmtb) {
            $stmtb->bind_param($types, ...$productIds); // Unpack product IDs

            if ($stmtb->execute()) {
                $resultb = $stmtb->get_result();
                if($resultb) {
                    while ($rowb = $resultb->fetch_assoc()) {
                        $productIdKey = intval($rowb['productid']);
                        // Initialize array for this product ID if it doesn't exist
                        if (!isset($pricebooks[$productIdKey])) {
                            $pricebooks[$productIdKey] = [];
                        }
                        $pricebooks[$productIdKey][] = $rowb; // Append pricebook
                        $fetched_pricebook_count++;
                        // error_log("{$logPrefix} Found Pricebook auto_id: " . ($rowb['auto_id'] ?? 'N/A') . " for Product ID: " . $productIdKey); // Optional debug
                    }
                    $resultb->free_result();
                    //error_log("{$logPrefix} Fetched $fetched_pricebook_count total pricebook entries.");
                } else {
                     error_log("{$logPrefix} DB Error getting pricebook result: " . $conn->error);
                }
            } else {
                error_log("{$logPrefix} DB Error executing pricebook fetch: " . $stmtb->error);
            }
            $stmtb->close();
        } else {
            error_log("{$logPrefix} DB Error preparing pricebook fetch: " . $conn->error);
        }
    } else {
         error_log("{$logPrefix} Product ID list was empty, skipping pricebook query.");
    }

    // Combine data
    foreach ($product_rows as $prod_id => $prod_info) {
        // Assign associated pricebooks, default to empty array if none found
        $prod_info['pricebooks_nested'] = $pricebooks[$prod_id] ?? [];
        // Optional debug log if needed
        // if (empty($prod_info['pricebooks_nested'])) { error_log("{$logPrefix} No pricebooks associated with Product ID $prod_id."); }
        $products[] = $prod_info; // Add the product (with nested pricebooks) to the final list
    }

    error_log("{$logPrefix} Finished getProductsAndPricebooks for Vendor ID: $vendorid. Returning " . count($products) . " products with nested data.");
    return $products;
}


/**
 * Fetches job details for a specific job ID.
 * (Unchanged from your provided code, includes path validation)
 */
function getJobDetails(int $job_id, mysqli $conn): ?array {
    $details = null;
    if ($job_id <= 0) { return null; }

    $sql = "SELECT vendor_id, contract_auto_id, contract_file_path
            FROM tdu_ai_contract_comparison_jobs WHERE id= ? LIMIT 1";
    $stmt = $conn->prepare($sql);
    if(!$stmt) { error_log("[Job {$job_id}] DB Error preparing getJobDetails: " . $conn->error); return null; }

    $stmt->bind_param('i', $job_id);
    if (!$stmt->execute()) { error_log("[Job {$job_id}] DB Error executing getJobDetails: " . $stmt->error); $stmt->close(); return null; }

    $result = $stmt->get_result();
    if ($result && $result->num_rows > 0) {
        $row = $result->fetch_assoc();
        $relativeContractPath = trim($row['contract_file_path'] ?? ''); // Get the relative path from DB

        if (!empty($relativeContractPath) && !empty($row['vendor_id']) && $row['vendor_id'] > 0 && !empty($row['contract_auto_id']) && $row['contract_auto_id'] > 0) {

            // --- Construct the Absolute Path FIRST ---
            // Combine the script's directory and the relative path from DB
            $constructedPath = __DIR__ . '/../' . $relativeContractPath; // Use '/' which works on Linux/Windows
            //$constructedPath = '../' . $relativeContractPath; // Use '/' which works on Linux/Windows
            // Use realpath() to resolve '..', '.' and check existence robustly
            $absoluteContractPath = realpath($constructedPath);

            // --- Perform the check using the ABSOLUTE path ---
             if ($absoluteContractPath && is_file($absoluteContractPath) && is_readable($absoluteContractPath)) {
                 // Path is valid, store it
                 $details = [
                     'vendor_id' => (int)$row['vendor_id'],
                     'contract_auto_id' => (int)$row['contract_auto_id'],
                     'relative_contract_file_path' => $relativeContractPath, // Keep original relative path if needed
                     'contract_file_path' => $absoluteContractPath // Use this validated absolute path
                 ];
             } else {
                  error_log("[Job {$job_id}] Contract file specified in job invalid/unreadable using calculated absolute path. Attempted: '" . $constructedPath . "'. Resolved realpath: '" . ($absoluteContractPath ?: 'false') . "'. Check file existence/permissions at this location.");
             }
        } else { error_log("[Job {$job_id}] Invalid job data retrieved (missing relative path, vendor_id, or contract_auto_id)."); }
    }else { error_log("[Job {$job_id}] No job found in DB with ID {$job_id}"); }

    $stmt->close();
    if ($result) {$result->free_result();}
    return $details;
}

// --- Helper Function for Basic Text Extraction (Highly Unreliable for DOCX/XLSX) ---
// (Unchanged from your provided code)
/**
 * Attempts to extract printable strings from a binary file.
 * WARNING: This is a VERY basic and unreliable method for complex formats like DOCX/XLSX.
 * It will likely miss content and include junk/tags. Use dedicated libraries if possible.
 *
 * @param string $filePath Path to the file.
 * @return string Attempted extraction of text.
 */
function extractBasicTextFromBinary(string $filePath): string {
    error_log("[Job processing] Attempting BASIC string extraction (unreliable method) for: " . basename($filePath));
    $content = @file_get_contents($filePath);
    if ($content === false) {
        error_log("[Job processing] Failed to read file for basic extraction: " . basename($filePath));
        return '';
    }
    preg_match_all('/[[:print:]\s]{10,}/', $content, $matches); // Crude heuristic
    if (!empty($matches[0])) {
        $extractedText = implode("\n", $matches[0]);
        error_log("[Job processing] Basic string extraction completed (may contain junk/tags): " . basename($filePath));
        return $extractedText;
    } else {
        error_log("[Job processing] Basic string extraction found no matching sequences: " . basename($filePath));
        return '';
    }
}


// --- AI API Interaction Functions ---

/**
 * Calls the external Gemini API via cURL.
 * (Unchanged from your provided code)
 */
function callGeminiAPI(array $payload): array {
    if (!function_exists('apiURL_thinking')) { return ['status' => 'CONFIG_ERROR', 'message' => 'apiURL_thinking() missing.']; }
    $apiUrl = apiURL_thinking(); if (empty($apiUrl)) { return ['status' => 'CONFIG_ERROR', 'message' => 'AI API URL is empty.']; }
    $headers = ['Content-Type: application/json']; $ch = curl_init();
    curl_setopt_array($ch, [ CURLOPT_URL => $apiUrl, CURLOPT_RETURNTRANSFER => true, CURLOPT_POST => true, CURLOPT_POSTFIELDS => json_encode($payload), CURLOPT_HTTPHEADER => $headers, CURLOPT_CONNECTTIMEOUT => 30, CURLOPT_TIMEOUT => 1200, CURLOPT_SSL_VERIFYPEER => true, CURLOPT_SSL_VERIFYHOST => 2, ]);
    $response_body = curl_exec($ch); $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE); $curlErrorNo = curl_errno($ch); $curlError = curl_error($ch); curl_close($ch);
    if ($curlErrorNo) { return ['status' => 'CURL_ERROR', 'message' => "cURL Error ($curlErrorNo): $curlError"]; }
    if ($httpCode == 429) { return ['status' => 'RATE_LIMIT_ERROR', 'message' => 'API rate limit exceeded (429).']; }
    if ($httpCode >= 500) { return ['status' => 'SERVER_ERROR', 'message' => "API server error ($httpCode).", 'details' => $response_body]; }
    if ($httpCode >= 400) { error_log("Gemini API Client Error ({$httpCode}): " . $response_body); $errorData = json_decode($response_body, true); $apiMessage = $errorData['error']['message'] ?? 'API client error (4xx).'; return ['status' => 'CLIENT_ERROR', 'message' => $apiMessage, 'details' => $response_body]; }
    if ($httpCode >= 200 && $httpCode < 300) { $responseData = json_decode($response_body, true); if (json_last_error() !== JSON_ERROR_NONE) { return ['status' => 'JSON_DECODE_ERROR', 'message' => 'Failed to decode API response: ' . json_last_error_msg(), 'details' => $response_body]; } if (isset($responseData['candidates'][0]['content']['parts'][0]['text'])) { return ['status' => 'SUCCESS', 'raw_response' => $responseData]; } if (isset($responseData['promptFeedback']['blockReason'])) { return ['status' => 'API_FILTERED', 'message' => "API request blocked: {$responseData['promptFeedback']['blockReason']}", 'details' => $responseData]; } return ['status' => 'UNEXPECTED_RESPONSE', 'message' => 'API response structure invalid.', 'details' => $responseData]; }
    return ['status' => 'UNEXPECTED_HTTP_CODE', 'message' => "Unexpected HTTP status code: $httpCode", 'details' => $response_body];
}

/**
 * Prepares prompt data (including nested pricebooks), calls AI, processes the structured JSON result.
 * Handles PDF/Image (inline data), TXT (text),
 * and attempts *basic* string extraction for DOCX/XLSX (unreliable).
 * --- UPDATED to include nested pricebook formatting in prompt ---
 */
function compareDataWithContract_JSON(array $vendorData, array $productsWithNestedPBs, string $contractFilePath, string $customTask): array {
    $logPrefix = "[JobProcessing_AICompare]";
    // --- Validate File --- (Unchanged)
    $fileName = basename($contractFilePath);
    if (!file_exists($contractFilePath) || !is_readable($contractFilePath)) {
        return ['status' => 'FILE_ERROR', 'message' => 'Contract file invalid/unreadable: ' . $fileName];
    }

    // --- Determine MIME Type & Process Content --- (Unchanged)
    $processedContent = null; $isInlineData = false; $payloadMimeType = null; $extractionMethod = "Direct Read / Inline";
    $extension = strtolower(pathinfo($contractFilePath, PATHINFO_EXTENSION));
    $mimeType = function_exists('mime_content_type') ? mime_content_type($contractFilePath) : null;
    $docxMimes = ['application/vnd.openxmlformats-officedocument.wordprocessingml.document'];
    $xlsxMimes = ['application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'];
    $pdfMimes = ['application/pdf', 'application/x-pdf'];
    $imageMimesPrefix = 'image/'; $textMimes = ['text/plain'];

    try {
        if ($extension === 'docx' || in_array($mimeType, $docxMimes)) {
            $extractionMethod = "DOCX (Basic String Extraction - Unreliable)";
            $content = @file_get_contents($contractFilePath);
            $processedContent = extractTextfromWordDoc($content); // Assumes this function exists
            $isInlineData = false;
        } elseif ($extension === 'xlsx' || in_array($mimeType, $xlsxMimes)) {
            $extractionMethod = "XLSX (Basic String Extraction - Unreliable)";
            $content = @file_get_contents($contractFilePath);
            $processedContent =  extractTextFromExcelSheet($content); // Assumes this function exists
            $isInlineData = false;
        } elseif (in_array($mimeType, $pdfMimes) || ($mimeType && strpos($mimeType, $imageMimesPrefix) === 0)) {
            $extractionMethod = "{$mimeType} (Inline Base64)";
            $fileContentBytes = file_get_contents($contractFilePath);
            if ($fileContentBytes === false) { throw new Exception('Failed to read file content for base64 encoding.'); }
            $payloadMimeType = $mimeType; $processedContent = base64_encode($fileContentBytes); $isInlineData = true; unset($fileContentBytes);
        } elseif (in_array($mimeType, $textMimes) || ($extension === 'txt' && empty($mimeType))) {
            $extractionMethod = "Plain Text (Direct Read)";
            $fileContentBytes = file_get_contents($contractFilePath);
            if ($fileContentBytes === false) { throw new Exception('Failed to read plain text file content.'); }
            $processedContent = $fileContentBytes; $isInlineData = false; unset($fileContentBytes);
        } else { throw new Exception('Unsupported file type: ' . ($mimeType ?: $extension)); }
        error_log("{$logPrefix} Processed '{$fileName}' using method: {$extractionMethod}.");
    } catch (\Throwable $e) { error_log("{$logPrefix} File processing error for {$fileName}: " . $e->getMessage()); return ['status' => 'FILE_ERROR', 'message' => 'Failed to process contract file (' . $fileName . '): ' . $e->getMessage()]; }

    // --- Format DB Data for Prompt ---

    // Vendor Data (Unchanged)
    $vendorString = "-- CURRENT VENDOR DATA --\n";
    if (is_array($vendorData) && !empty($vendorData)) {
        foreach ($vendorData as $key => $value) {
            $vendorString .= str_pad($key . ":", 20) . (($value === null || $value === '') ? 'N/A' : $value) . "\n";
        }
    } else { $vendorString .= "No current vendor data available.\n"; }

    // Products and NESTED Pricebooks Data (UPDATED Formatting Logic)
    $productString = "-- CURRENT PRODUCT & ASSOCIATED PRICEBOOK DATA (" . count($productsWithNestedPBs) . " products) --\n";
    if (is_array($productsWithNestedPBs) && !empty($productsWithNestedPBs)) {
        foreach ($productsWithNestedPBs as $index => $product) {
            $productString .= "\n[Product #" . ($index + 1) . " - ID: " . ($product['productid'] ?? 'N/A') . "]\n";
            if (is_array($product)) {
                // Print product fields
                foreach ($product as $key => $value) {
                    if ($key === 'pricebooks_nested') continue; // Skip the nested array itself here
                    $productString .= "  " . str_pad($key . ":", 20) . (($value === '' || $value === null) ? 'N/A' : $value) . "\n";
                }
                // Print nested pricebooks (This section is added/restored)
                if (!empty($product['pricebooks_nested']) && is_array($product['pricebooks_nested'])) {
                    $productString .= "  Associated Pricebooks:\n";
                    foreach ($product['pricebooks_nested'] as $pbIndex => $pbEntry) {
                        if (is_array($pbEntry)) {
                            $productString .= "   - Pricebook Entry (auto_id: " . ($pbEntry['auto_id'] ?? 'N/A') . "):\n"; // Use auto_id for reference
                            foreach ($pbEntry as $pbKey => $pbValue) {
                                // Indent pricebook details further
                                $productString .= "      " . str_pad($pbKey . ":", 22) . (($pbValue === '' || $pbValue === null) ? 'N/A' : $pbValue) . "\n";
                            }
                        } else { $productString .= "   - Invalid pricebook entry data structure.\n"; }
                    }
                } else {
                    $productString .= "  Associated Pricebooks: None\n"; // Indicate if none exist for this product
                }
            } else { $productString .= " Invalid product data format.\n"; }
        }
    } else { $productString .= "No current product data available.\n"; }

    // --- Define AI Prompt & JSON Structure (Unchanged Definition) ---
    $jsonStructureDefinition = <<<JSONDEF
REQUIRED JSON OUTPUT STRUCTURE AND KEYS:
You MUST output ONLY a single, valid JSON object containing top-level keys: "reportHtml", "correctedVendor", "correctedProducts", "correctedPriceBooks".
1. "reportHtml": (String) HTML analysis report (divs, h2, p, table). Use unique IDs like 'summary-ai-XYZ'.
2. "correctedVendor": (Object) Vendor corrections. Use ONLY these keys (lowercase/camelCase): `vendorid`, `vendorName`, `email`, `paymentType`, `bookingMethod`, `voucherRequired`, `description`, `phone`, `cancelHoursGroup`, `cancelHoursFIT`, `address`, `locationPhone`, `types`, `website`, `preferred`, `openingHours`, `vendorCountry`. Use `null` for missing values. Include original `vendorid`.
3. "correctedProducts": (Array of Objects) Product corrections/new products. Use ONLY these keys: `productid`, `productName`, `category`, `city`, `country`, `productActive`, `unitPrice`, `childPrice`, `infantPrice`, `childAge`, `familyPrice`, `openingHours`, `duration`. Use `null` for missing values. Include original `productid`.
4. "correctedPriceBooks": (Object) Price book updates/new entries.
    *   **Matching Existing Pricebooks:** For each price/rate found in the contract, you MUST compare it to the entries listed in the '-- CURRENT PRODUCT & ASSOCIATED PRICEBOOK DATA --' section provided earlier. Match based primarily on `productid` and secondarily on `name`. If a rate CLEARLY corresponds to an existing price book entry (match found), you MUST use the EXISTING `auto_id` (e.g., "32", "33") from that section as the key for that suggestion in this output object.
    *   **Creating New Pricebooks:** ONLY if a rate/price found in the contract CANNOT be reasonably matched to ANY existing price book entry, then generate a unique key starting with "new_" followed by the productid and an index (e.g., "new_102536720_1"). DO NOT create "new_" keys for price books that match existing ones.
    *   **Value Object Keys:** For the value object associated with each key (existing `auto_id` or "new_..."), use ONLY the following keys (matching database columns): `productid`, `name`, `start_date` (YYYY-MM-DD), `end_date` (YYYY-MM-DD), `unit_price` (numeric), `child_price` (numeric), `infant_price` (numeric), `currency`, `description`, `dayOfWeek`.
IMPORTANT: - Adhere STRICTLY to the specified keys (lowercase/camelCase). - Use `null` for missing values. - Ensure boolean values are `true` or `false`. - Output ONLY valid JSON. No markdown backticks (\`\`\`).

RULES FOR DATA:
GENERAL:
The data that are mentioned in the corrected segments are the important ones, the data that are not mentioned are to be ignored when searching for mismatches.
infantAge is a defunct data, ignore it
VENDORS:
voucherRequired: binary, 'Yes' or 'No'
paymentType: binary, 'Prepaid' or 'Credit'
bookingMethod: binary, 'Online' or 'Email'
cancelHoursGroup: in terms of hours, no need for unit, just the number
cancelHoursFIT: in terms of hours, no need for unit, just the number
openingHours: usual opening hours for a week, ignore any special non-operational days such as holidays and etc. Example output("Monday: Open 24 hours, Tuesday: Open 24 hours, Wednesday: Open 24 hours, Thursday: Open 24 hours, Friday: Open 24 hours, Saturday: Open 24 hours, Sunday: Open 24 hours", "Monday: 10:00 AM – 7:30 PM, Tuesday: 10:00 AM – 7:30 PM, Wednesday: 10:00 AM – 7:30 PM, Thursday: 10:00 AM – 7:30 PM, Friday: 10:00 AM – 7:30 PM, Saturday: 10:00 AM – 7:30 PM, Sunday: 10:00 AM – 7:30 PM")
description: A description of the service provided by the vendor
PRODUCTS:
unitPrice: for general pricing of the products, null is a last resort
childPrice: for children pricing of the products, null is a last resort
infantPrice: for infant pricing of the products, null is a last resort
familyPrice: for family pricing of the products, null is a last resort
duration: In hours so convert to match, there is no need to put a unit behind it just the number is enough, this should be a string
productActive: Its a 'Yes' or 'No'. keep it the same, no need to change
productid: auto increment id, when creating a new one just leave blank
childAge: the age range of the children price, format it in a range, it HAS to have BOTH a minimum and maximum, minimum being 2 and maximum being 14 (e.g., '2-15', '3-12'), if its not specified, default to '2-14'
PRICEBOOK:
unitPrice: for general pricing of the products, null is a last resort
childPrice: for children pricing of the products, null is a last resort
infantPrice: for infant pricing of the products, null is a last resort
dayOfWeek: Output options ('Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday', '') an empty string output means the whole week, only output 1 option, keep in mind that N/A in the database means empty string
JSONDEF;


    // Construct the final text prompt - includes extracted text if not inline data
    $contractContentForPrompt = ($processedContent !== null) ? $processedContent : "[Error during file processing or empty file]";
    $contractContentPromptPart = "-- CONTRACT CONTENT START --\n";
    if ($isInlineData) {
        $contractContentPromptPart .= "[Content provided separately as inline data for MIME type: {$payloadMimeType}]";
    } else {
        if (strpos($extractionMethod, 'Basic String Extraction') !== false) {
             $contractContentPromptPart .= "[WARNING: Content below derived via BASIC string extraction from {$extension} - may be incomplete/contain non-content strings.]\n";
        }
        $contractContentPromptPart .= $contractContentForPrompt; // Embed text content
    }
    $contractContentPromptPart .= "\n-- CONTRACT CONTENT END --\n\n";

    // Construct final prompt including the updated product/pricebook string
    $finalTextPrompt = "Analyze the attached contract content against the provided CURRENT VENDOR DATA and CURRENT PRODUCT & ASSOCIATED PRICEBOOK DATA based on the TASK below.\n\n"
        . $contractContentPromptPart // Contract content (inline placeholder or actual text)
        . $vendorString . "\n\n"     // Vendor data
        . $productString . "\n\n"    // Product data WITH nested pricebooks
        . "TASK:\n" . $customTask . "\n\n"
        . $jsonStructureDefinition;

    error_log($logPrefix . " Final AI Prompt (excluding contract data):\n" . $finalTextPrompt); // Log part of the prompt for debugging if needed

    // --- Construct API Payload --- (Unchanged Logic)
    $payloadParts = [];
    $payloadParts[] = ['text' => trim($finalTextPrompt)];
    if ($isInlineData && $payloadMimeType && $processedContent) {
        $payloadParts[] = ['inlineData' => ['mimeType' => $payloadMimeType, 'data' => $processedContent]];
        error_log("{$logPrefix} Sending API request with inline data (MIME: {$payloadMimeType}) and text prompt.");
    } else {
        error_log("{$logPrefix} Sending API request with contract content embedded in text prompt (Method: {$extractionMethod}).");
    }
    $payload = [ 'contents' => [['parts' => $payloadParts]], 'generationConfig' => ['responseMimeType' => 'application/json', 'temperature' => 0.1, 'maxOutputTokens' => 30000 ] ]; // Ensure token limit is adequate
    unset($processedContent); // Free memory

    // --- Call API ---
    error_log("{$logPrefix} Calling Gemini API...");
    $result = callGeminiAPI($payload);

    // --- Process API Result --- (Unchanged Logic)
    if ($result['status'] === 'SUCCESS' && isset($result['raw_response']['candidates'][0]['content']['parts'][0]['text'])) {
        $responseText = trim($result['raw_response']['candidates'][0]['content']['parts'][0]['text']);
        $extractedData = json_decode($responseText, true);
        // Validate structure more robustly
        if (json_last_error() === JSON_ERROR_NONE && is_array($extractedData)
            && isset($extractedData['reportHtml']) && is_string($extractedData['reportHtml'])
            && isset($extractedData['correctedVendor']) // Allow object or empty array initially
            && isset($extractedData['correctedProducts']) && is_array($extractedData['correctedProducts'])
            && isset($extractedData['correctedPriceBooks']) // Allow object or empty array initially
        ) {
            // Standardize empty results to objects as expected by some frontend/backend logic
            if (empty($extractedData['correctedVendor']) && is_array($extractedData['correctedVendor'])) {
                 $extractedData['correctedVendor'] = (object)[];
            } else if (!is_object($extractedData['correctedVendor'])) {
                // If it's not empty and not an object, log a warning maybe? Or try to cast? For now, assume structure is generally correct if not empty array.
                 if(!is_array($extractedData['correctedVendor'])) { // If not an array or object, force object
                     error_log("{$logPrefix} AI returned non-object/array for correctedVendor. Forcing to object.");
                     $extractedData['correctedVendor'] = (object)[];
                 }
            }

            if (empty($extractedData['correctedPriceBooks']) && is_array($extractedData['correctedPriceBooks'])) {
                $extractedData['correctedPriceBooks'] = (object)[];
            } else if (!is_object($extractedData['correctedPriceBooks'])) {
                 if(!is_array($extractedData['correctedPriceBooks'])) { // If not an array or object, force object
                     error_log("{$logPrefix} AI returned non-object/array for correctedPriceBooks. Forcing to object.");
                     $extractedData['correctedPriceBooks'] = (object)[];
                 }
            }

            error_log("{$logPrefix} AI call successful. Parsed structured JSON response.");
            return ['status' => 'SUCCESS', 'data' => $extractedData];
        } else {
            error_log("{$logPrefix} AI JSON decode error/structure mismatch: " . json_last_error_msg() . " | Raw Start: " . substr($responseText, 0, 500));
            return ['status' => 'JSON_DECODE_ERROR', 'message' => 'AI response invalid JSON or structure.', 'raw_data' => $responseText];
        }
    } else { // Handle API call failures or unexpected successful response format
        if ($result['status'] === 'SUCCESS') { // Success status but missing data
            error_log("{$logPrefix} AI success status but missing expected text part. Response: " . json_encode($result['raw_response'] ?? null));
            return ['status' => 'UNEXPECTED_RESPONSE', 'message' => 'API success response missing text part.', 'details' => $result['raw_response'] ?? null];
        } else { // Actual API call error
             error_log("{$logPrefix} AI API call failed - Status: {$result['status']}, Message: {$result['message']}");
             return $result; // Return the error status and message from callGeminiAPI
        }
    }
}


/**
 * The main logic for processing a single job.
 * --- UPDATED to use getProductsAndPricebooks ---
 */
function processAiComparisonJob(int $job_id, mysqli $conn): bool {
    error_log("[Job {$job_id}] Starting processing...");

    // 1. Get Job Details (including validated file path)
    $jobDetails = getJobDetails($job_id, $conn);
    if (!$jobDetails) {
        // Error logged in getJobDetails, update status here
        updateJobStatus($job_id, 'ERROR', $conn, null, "Failed to get job details or contract file inaccessible/invalid for job ID {$job_id}.");
        return false; // Indicate processing failed to start properly
    }
    $vendorId = $jobDetails['vendor_id'];
    $contractAutoId = $jobDetails['contract_auto_id'];
    $contractFilePath = $jobDetails['contract_file_path']; // Absolute, validated path
    error_log("[Job {$job_id}] Processing Vendor ID: {$vendorId}, Contract AutoID: {$contractAutoId}, File: " . basename($contractFilePath));

    // 2. Fetch Current DB Data (Vendor + Products WITH NESTED Pricebooks)
    $currentVendorInfo = getVendor($vendorId, $conn) ?? [];
    // *** Use the function that fetches nested data ***
    $currentProductsWithNestedPBs = getProductsAndPricebooks($vendorId, $conn);
    //error_log("[Job {$job_id}] Fetched current DB data for prompt. Products (with nested pricebooks) found: " . count($currentProductsWithNestedPBs));

    // 3. Define the AI Task (Unchanged task definition)
    $comparisonTask = <<<TASK
1. Analyze the contract for terms, rates, validity, policies, products/services.
2. Compare contract details against CURRENT VENDOR DATA and CURRENT PRODUCT & ASSOCIATED PRICEBOOK DATA.
3. Identify discrepancies (pricing, names, dates, cancellation, contacts, status, etc.).
4. Generate an HTML report ("reportHtml") summarizing contract, findings table, suggested changes table.
    a. Summary: Summarise the contract on the important details
    b. Findings Table: Show A table with the columns of Data Name, Database Value, Contract Value, Mismatch (color code them along with the words, light colored background with deep colored text for contrast: Green for Match, Red for Mismatch, Orange for Partial Mismatch(leaning more towards mismatch), Yellow for Partial Match(leaning more towards match (e.g., rounding error)), Reasoning
    c. Suggested Change Table: Show A table with the columns of Data Name, Current Value, Suggested Value, Reasoning
5. Generate structured JSON ("correctedVendor", "correctedProducts", "correctedPriceBooks") with corrections/new values based STRICTLY on the contract, adhering to the REQUIRED JSON OUTPUT STRUCTURE AND KEYS provided.
6. Make sure all the styles and html element has incredibly unique id to ensure that it does not interfere with existing html
TASK;

    // 4. Call AI Comparison Function
    error_log("[Job {$job_id}] Calling AI comparison function...");
    $ai_result = compareDataWithContract_JSON(
        $currentVendorInfo,
        $currentProductsWithNestedPBs, // *** Pass the data with nested pricebooks ***
        $contractFilePath,
        trim($comparisonTask)
    );

    // 5. Process AI Result and Update Job Status
    if ($ai_result['status'] === 'SUCCESS' && isset($ai_result['data'])) {
        error_log("[Job {$job_id}] AI analysis successful.");
        // Encode the structured data part for storage
        $resultJson = json_encode($ai_result['data'], JSON_UNESCAPED_UNICODE | JSON_INVALID_UTF8_SUBSTITUTE);
        if (json_last_error() !== JSON_ERROR_NONE) {
             error_log("[Job {$job_id}] Error encoding AI result JSON: " . json_last_error_msg());
             updateJobStatus($job_id, 'ERROR', $conn, null, "Internal error encoding AI result: " . json_last_error_msg());
        } else {
            // Update status to DONE with the JSON result
            updateJobStatus($job_id, 'DONE', $conn, $resultJson, null);
        }
    } else {
        // AI processing failed (includes FILE_ERROR, API errors, JSON decode errors etc.)
        $errorMessage = $ai_result['message'] ?? 'Unknown AI processing error.';
        $errorStatus = $ai_result['status'] ?? 'AI_ERROR';
        error_log("[Job {$job_id}] AI analysis failed. Status: {$errorStatus}, Message: {$errorMessage}");
        // Update status to ERROR with the specific message
        updateJobStatus($job_id, 'ERROR', $conn, null, "AI Error ({$errorStatus}): {$errorMessage}");
    }

    // Regardless of success/failure of AI step, the job processing attempt itself is done
    return true;
}


/**
 * Updates the final status (DONE or ERROR) of a job.
 * (Unchanged from your provided code)
 */
function updateJobStatus(int $job_id, string $status, mysqli $conn, $resultJson = null, $errorMessage = null): bool {
    if ($job_id <= 0 || !in_array($status, ['DONE', 'ERROR'])) { error_log("Invalid input to updateJobStatus: Job ID {$job_id}, Status {$status}"); return false; }

    $set_parts = []; $bind_types = ''; $bind_values = [];
    $set_parts[] = "`status` = ?"; $bind_types .= 's'; $bind_values[] = $status;
    $set_parts[] = "`completed_at` = NOW()";

    if ($status === 'DONE' && $resultJson !== null) {
        $set_parts[] = "`gemini_result` = ?"; $bind_types .= 's'; $bind_values[] = $resultJson; $set_parts[] = "`error_message` = NULL";
    } elseif ($status === 'ERROR') {
        $set_parts[] = "`gemini_result` = NULL"; $set_parts[] = "`error_message` = ?"; $bind_types .= 's'; $bind_values[] = $errorMessage ?? 'Unknown error occurred.';
    } else { // Should not happen based on input validation, but fallback
        $set_parts[] = "`gemini_result` = NULL"; $set_parts[] = "`error_message` = NULL";
    }

    $bind_types .= 'i'; $bind_values[] = $job_id; // For WHERE clause
    $sql = "UPDATE tdu_ai_contract_comparison_jobs SET " . implode(', ', $set_parts) . " WHERE `id` = ?";
    $stmt = $conn->prepare($sql); if (!$stmt) { error_log("[Job {$job_id}] DB Error preparing final status update to '{$status}': " . $conn->error); return false; }

    if (strlen($bind_types) !== count($bind_values)) { error_log("[Job {$job_id}] Mismatch bind types/values for status update."); $stmt->close(); return false; }
    // Use argument unpacking (...) for bind_param
    if (!empty($bind_types)) { $stmt->bind_param($bind_types, ...$bind_values); }

    if ($stmt->execute()) {
        $affected_rows = $stmt->affected_rows; $stmt->close();
        if ($affected_rows > 0) { error_log("[Job {$job_id}] Final status updated to '{$status}'."); return true; }
        else { error_log("[Job {$job_id}] Final status update query OK for '{$status}', but no rows affected (job ID not found or status already set?)."); return false; } // Added more context
    } else { error_log("[Job {$job_id}] DB Error executing final status update to '{$status}': " . $stmt->error); $stmt->close(); return false; }
}

?>