<?php
// qbo_pnl_graph_report.php
// FINAL, COMPLETE VERSION: Implements the Yearly Average graph, a CSS spinner, and the final Net Profit trend line. (With corrected filter logic)

// PHP part at the top is unchanged
use QuickBooksOnline\API\DataService\DataService;
require_once __DIR__ . '/../vendor/autoload.php';
require_once __DIR__ . '/qbo_functions.php';
require_once __DIR__ . '/config/qbo_config.php';
global $qboBaseConfig;
$qbo_is_connected = false;
$tokenStorageFile = __DIR__ . '/tokens/qbo_token.json';
if (file_exists($tokenStorageFile)) {
    $currentTokens = json_decode(file_get_contents($tokenStorageFile), true);
    if ($currentTokens && !empty($currentTokens['access_token'])) { $qbo_is_connected = true; }
}
?>

<!-- CSS is unchanged -->
<style>
    .tab-container { border-bottom: 2px solid #dee2e6; display: flex; margin-bottom: 20px; }
    .tab-btn { background-color: transparent; border: none; padding: 12px 18px; font-size: 1.1em; cursor: pointer; position: relative; color: #6c757d; }
    .tab-btn.active { font-weight: bold; color: #005ea2; }
    .tab-btn.active::after { content: ''; position: absolute; bottom: -2px; left: 0; right: 0; height: 2px; background-color: #005ea2; }
    .tab-content { display: none; }
    .tab-content.active { display: block; }
    .comparison-container { display: grid; grid-template-columns: repeat(auto-fit, minmax(500px, 1fr)); gap: 20px; }
    .graph-instance { background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); margin-top: 20px; position: relative; }
    .graph-instance h3 { margin-top: 0; color: #333; padding-right: 30px; }
    .add-comparison-btn { background-color: #007bff; color: white; border: none; padding: 8px 15px; border-radius: 4px; cursor: pointer; font-size: 0.9em; margin-top: 20px; }
    .remove-comparison-btn { background-color: #dc3545; color: white; border: none; font-size: 1em; line-height: 1.2; cursor: pointer; padding: 2px 8px; border-radius: 50%; position: absolute; top: 15px; right: 15px; }
    .graph-canvas-wrapper { position: relative; height: 400px; }
    
    .loading-overlay { position: absolute; top: 0; left: 0; right: 0; bottom: 0; background: rgba(255, 255, 255, 0.85); display: flex; align-items: center; justify-content: center; z-index: 10; border-radius: 5px; transition: opacity 0.3s, visibility 0.3s; opacity: 0; visibility: hidden; }
    .loading-overlay.visible { opacity: 1; visibility: visible; }
    .spinner { width: 50px; height: 50px; border: 5px solid #f3f3f3; border-top: 5px solid #005ea2; border-radius: 50%; animation: spin 1s linear infinite; }
    @keyframes spin { 0% { transform: rotate(0deg); } 100% { transform: rotate(360deg); } }
</style>

<div class="report-header">
    <h2>Profit & Loss Analysis</h2>
    <p>Visual representation of your financial performance.</p>
</div>

<!-- HTML is unchanged -->
<div class="tab-container">
    <button class="tab-btn active" data-tab="monthly">Monthly</button>
    <button class="tab-btn" data-tab="quarterly">Quarterly</button>
    <button class="tab-btn" data-tab="yearly">Yearly</button>
    <button class="tab-btn" data-tab="average">Yearly Average</button>
</div>

<div id="monthly" class="tab-content active"><div id="monthly-graphs" class="comparison-container"></div><button class="add-comparison-btn" data-view="monthly">+ Add Comparison</button></div>
<div id="quarterly" class="tab-content"><div id="quarterly-graphs" class="comparison-container"></div><button class="add-comparison-btn" data-view="quarterly">+ Add Comparison</button></div>
<div id="yearly" class="tab-content"><div id="yearly-graphs" class="comparison-container"></div></div>
<div id="average" class="tab-content"><div id="average-graphs" class="comparison-container"></div></div>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const qboIsConnected = <?php echo json_encode($qbo_is_connected); ?>;
    const chartInstances = {};

    // RESTORED: Original, simple year selector generator.
    const generateYearSelector = (graphId) => {
        const now = new Date();
        let optionsHtml = '';
        for (let i = 0; i <= 5; i++) {
            const year = now.getFullYear() - i;
            const selected = i === 0 ? 'selected' : '';
            optionsHtml += `<option value="${year}" ${selected}>${year}</option>`;
        }
        return `<label>For Year: <select class="date-selector" data-graph-id="${graphId}">${optionsHtml}</select></label>`;
    };

    async function fetchDataForGraph(startDate, endDate, columns) {
        const url = `qbo_ajax_pnl_graph_data.php?start_date=${startDate}&end_date=${endDate}&columns=${columns}`;
        const response = await fetch(url);
        const result = await response.json();
        return result.success ? result.data : null;
    }

    async function updateGraph(graphId, view) {
        const canvas = document.getElementById(graphId);
        const wrapper = canvas.parentElement;
        const loadingOverlay = wrapper.querySelector('.loading-overlay');
        loadingOverlay.classList.add('visible');

        const selector = document.querySelector(`.date-selector[data-graph-id="${graphId}"]`);
        let startDate, endDate, columns;

        if (view === 'monthly' || view === 'quarterly') {
            const year = selector.value;
            startDate = `${year}-01-01`; endDate = `${year}-12-31`; columns = 'Months';
        } else if (view === 'yearly') {
            const endYear = new Date().getFullYear(); const startYear = endYear - 6;
            startDate = `${startYear}-01-01`; endDate = `${endYear}-12-31`; columns = 'Years';
        } else if (view === 'average') {
            // Note: The selector value is not used here, matching original functionality.
            const endYear = new Date().getFullYear(); const startYear = endYear - 6;
            startDate = `${startYear}-01-01`; endDate = `${endYear}-12-31`; columns = 'Total_for_avg';
        }

        const data = await fetchDataForGraph(startDate, endDate, columns);
        loadingOverlay.classList.remove('visible');

        const ctx = canvas.getContext('2d');
        if (!data || !data.labels || data.labels.length === 0) {
            if(chartInstances[graphId]) { chartInstances[graphId].destroy(); delete chartInstances[graphId]; }
            ctx.clearRect(0, 0, canvas.width, canvas.height);
            ctx.font = "16px Arial"; ctx.textAlign = "center";
            ctx.fillText("No data found for this period.", canvas.width / 2, 50);
            return;
        }

        let chartLabels = data.labels;
        let chartIncome = data.income; let chartCogs = data.cogs;
        let chartExpenses = data.expenses; let chartNet = data.net;

        if (view === 'quarterly') {
            chartLabels = ['Q1', 'Q2', 'Q3', 'Q4'];
            const quarterlyIncome = [0, 0, 0, 0], quarterlyCogs = [0, 0, 0, 0], quarterlyExpenses = [0, 0, 0, 0], quarterlyNet = [0, 0, 0, 0];
            for (let i = 0; i < data.income.length; i++) {
                const quarterIndex = Math.floor(i / 3);
                if(quarterlyIncome[quarterIndex] !== undefined) {
                    quarterlyIncome[quarterIndex] += data.income[i]; quarterlyCogs[quarterIndex] += data.cogs[i];
                    quarterlyExpenses[quarterIndex] += data.expenses[i]; quarterlyNet[quarterIndex] += data.net[i];
                }
            }
            chartIncome = quarterlyIncome; chartCogs = quarterlyCogs; chartExpenses = quarterlyExpenses; chartNet = quarterlyNet;
        }
        
        const showTrendLine = (view === 'monthly' || view === 'quarterly' || view === 'yearly');
        const datasets = [
            { label: 'Total Income', data: chartIncome, backgroundColor: 'rgba(54, 162, 235, 0.7)', yAxisID: 'y' },
            { label: 'Cost of Sales', data: chartCogs, backgroundColor: 'rgba(255, 159, 64, 0.7)', yAxisID: 'y' },
            { label: 'Expenses', data: chartExpenses, backgroundColor: 'rgba(255, 99, 132, 0.7)', yAxisID: 'y' }
        ];

        if (showTrendLine) {
            const profitMarginData = chartIncome.map((income, index) => (income <= 0) ? 0 : (chartNet[index] / income) * 100);
            datasets.push({ label: 'Net Profit Margin', data: profitMarginData, type: 'line', borderColor: '#34A853', backgroundColor: '#34A853', yAxisID: 'y1', tension: 0.1 });
        }
        
        // THE ONLY ADDITION: Add the trend line for Average Net Profit on the 'average' tab.
        if (view === 'average') {
            datasets.push({
                label: 'Average Net Profit',
                data: chartNet, // This data is the monthly average net profit per year
                type: 'line',
                borderColor: 'rgba(75, 192, 192, 1)',
                backgroundColor: 'rgba(75, 192, 192, 1)',
                yAxisID: 'y', // Plot against the primary currency axis
                tension: 0.1,
                borderWidth: 2
            });
        }
        
        if (chartInstances[graphId]) {
            chartInstances[graphId].data.labels = chartLabels;
            chartInstances[graphId].data.datasets = datasets;
            chartInstances[graphId].options.scales.y1.display = showTrendLine;
            chartInstances[graphId].update();
        } else {
            const newChart = new Chart(ctx, {
                type: 'bar',
                data: { labels: chartLabels, datasets: datasets },
                options: {
                    responsive: true, maintainAspectRatio: false,
                    scales: {
                        y: { type: 'linear', position: 'left', beginAtZero: true, ticks: { callback: value => '$' + value.toLocaleString() } },
                        y1: { type: 'linear', position: 'right', grid: { drawOnChartArea: false }, ticks: { callback: value => value.toFixed(0) + '%' }, display: showTrendLine, suggestedMax: 100, suggestedMin: -50 }
                    },
                    plugins: { tooltip: { callbacks: { label: function(context) { let label = context.dataset.label || ''; if (label) { label += ': '; } const value = context.parsed.y; if (context.dataset.yAxisID === 'y1') { const index = context.dataIndex; const netForPeriod = chartNet[index]; const incomeForPeriod = chartIncome[index]; return [ `Margin: ${value.toFixed(2)}%`, `  Net: ${new Intl.NumberFormat('en-US', { style: 'currency', currency: 'USD' }).format(netForPeriod)}`, `  Income: ${new Intl.NumberFormat('en-US', { style: 'currency', currency: 'USD' }).format(incomeForPeriod)}` ]; } return label + new Intl.NumberFormat('en-US', { style: 'currency', currency: 'USD' }).format(value); } } } }
                }
            });
            chartInstances[graphId] = newChart;
        }
    }

    // RESTORED: Original graph creation logic.
    function createGraphInstance(view, isFirst = false) {
        const graphContainer = document.getElementById(`${view}-graphs`);
        const instanceId = `${view}-${Date.now()}`;
        const canvasId = `chart-${instanceId}`;
        const graphDiv = document.createElement('div');
        graphDiv.className = 'graph-instance';
        
        let titleText = isFirst ? 'Primary View' : 'Comparison View';
        let filterHtml = '';
        
        // The yearly average tab will have a year selector
        if (view === 'monthly' || view === 'quarterly' || view === 'average') {
            filterHtml = generateYearSelector(canvasId);
        }
        
        // Set titles based on view
        if(view === 'yearly') { titleText = isFirst ? '7-Year Trend' : '7-Year Trend Comparison'; }
        if(view === 'average') { titleText = 'Average Monthly Performance'; }

        graphDiv.innerHTML = `<h3>${titleText} <button class="remove-comparison-btn" style="display: ${isFirst && graphContainer.children.length === 0 ? 'none' : 'inline-block'};">×</button></h3><div class="filter-form">${filterHtml}</div><div class="graph-canvas-wrapper"><div class="loading-overlay"><div class="spinner"></div></div><canvas id="${canvasId}"></canvas></div>`;
        
        graphContainer.appendChild(graphDiv);
        updateGraph(canvasId, view);

        if (filterHtml) {
            graphDiv.querySelector('.date-selector').addEventListener('change', () => updateGraph(canvasId, view));
        }
    }
    
    // All event listeners and initial load logic remain the same.
    document.querySelectorAll('.comparison-container').forEach(container => { container.addEventListener('click', function(e) { if (e.target.classList.contains('remove-comparison-btn')) { const instanceDiv = e.target.closest('.graph-instance'); const canvasId = instanceDiv.querySelector('canvas').id; if (chartInstances[canvasId]) { chartInstances[canvasId].destroy(); delete chartInstances[canvasId]; } instanceDiv.remove(); } }); });
    const tabs = document.querySelectorAll('.tab-btn');
    tabs.forEach(tab => { tab.addEventListener('click', () => { tabs.forEach(t => t.classList.remove('active')); tab.classList.add('active'); const view = tab.dataset.tab; document.querySelectorAll('.tab-content').forEach(content => { content.classList.toggle('active', content.id === view); }); }); });
    document.querySelectorAll('.add-comparison-btn').forEach(btn => { btn.addEventListener('click', function() { const view = this.dataset.view; createGraphInstance(view, false); }); });
    
    if (!qboIsConnected) {
        document.body.innerHTML = '<h1>QuickBooks Online connection not found. Please connect to QBO first.</h1>';
        return;
    }
    createGraphInstance('monthly', true);
    createGraphInstance('quarterly', true);
    createGraphInstance('yearly', true);
    createGraphInstance('average', true);
});
</script>