<?php
// qbo_supplier_analytics_report.php
// The new, unified dashboard for all supplier analytics.
// VERSION 2: Visualizes the Top 10 Suppliers with a Horizontal Bar Chart.

use QuickBooksOnline\API\DataService\DataService;
use QuickBooksOnline\API\ReportService\ReportService;

require_once __DIR__ . '/../vendor/autoload.php';
require_once __DIR__ . '/qbo_functions.php';
require_once __DIR__ . '/config/qbo_config.php';
global $qboBaseConfig;

// --- Report Filters & Variables ---
$finYearStart = (date('m') < 7) ? date('Y-07-01', strtotime('-1 year')) : date('Y-07-01');
$startDate = isset($_GET['start_date']) && !empty($_GET['start_date']) ? $_GET['start_date'] : $finYearStart;
$endDate = isset($_GET['end_date']) && !empty($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-d');
$accountingMethod = 'Accrual';

$dataService = null;
$topSuppliers = [];

// --- QBO Connection & Token Refresh ---
$tokenStorageFile = __DIR__ . '/tokens/qbo_token.json';
$currentTokens = file_exists($tokenStorageFile) ? json_decode(file_get_contents($tokenStorageFile), true) : null;
if ($currentTokens && !empty($currentTokens['access_token'])) {
    $qboBaseConfig['accessTokenKey'] = $currentTokens['access_token'];
    try { $dataService = DataService::Configure($qboBaseConfig); } catch (Exception $e) { $dataService = null; }
} else { exit("QBO Authentication failed."); }

// --- DATA FETCHING & PROCESSING LOGIC (Unchanged) ---
if ($dataService) {
    try {
        $reportService = new ReportService($dataService->getServiceContext());
        $reportService->setStartDate($startDate);
        $reportService->setEndDate($endDate);
        $report = $reportService->executeReport("TransactionList");
        
        if ($report && isset($report->Rows->Row)) {
            $supplierTotals = [];
            $nameIndex = -1; $amountIndex = -1;
            foreach ($report->Columns->Column as $index => $column) {
                if ($column->ColTitle === 'Name') $nameIndex = $index;
                if ($column->ColTitle === 'Amount') $amountIndex = $index;
            }

            if ($nameIndex !== -1 && $amountIndex !== -1) {
                $rows = is_array($report->Rows->Row) ? $report->Rows->Row : [$report->Rows->Row];
                foreach ($rows as $row) {
                    if (isset($row->type) && $row->type === 'Data') {
                        $name = $row->ColData[$nameIndex]->value ?? null;
                        $amount = (float)($row->ColData[$amountIndex]->value ?? 0);
                        if ($name && $amount > 0) {
                            if (!isset($supplierTotals[$name])) { $supplierTotals[$name] = 0; }
                            $supplierTotals[$name] += $amount;
                        }
                    }
                }
                arsort($supplierTotals);
                $topSuppliers = array_slice($supplierTotals, 0, 10, true);
            }
        }
    } catch (Exception $e) { echo "<!-- QBO Error: " . $e->getMessage() . " -->"; }
}
?>
<div class="report-header">
    <h2>Supplier Analytics</h2>
</div>

<!-- Filter Form is unchanged -->
<form method="get" class="filter-form">
    <input type="hidden" name="report" value="supplier_analytics">
    <div><label>From:</label><input type="date" name="start_date" id="start_date" value="<?php echo htmlspecialchars($startDate); ?>"></div>
    <div><label>To:</label><input type="date" name="end_date" id="end_date" value="<?php echo htmlspecialchars($endDate); ?>"></div>
    <button type="submit">Run Report</button>
</form>

<!-- NEW: Graph container -->
<div class="graph-instance" style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); margin-top: 20px;">
    <h3>Top 10 Performing Suppliers by Expense</h3>
    <p>For period: <?php echo htmlspecialchars(date("F j, Y", strtotime($startDate))) . " - " . htmlspecialchars(date("F j, Y", strtotime($endDate))); ?></p>
    <div class="graph-canvas-wrapper" style="position: relative; height:450px;">
        <canvas id="topSuppliersChart"></canvas>
    </div>
</div>


<!-- The ranked table remains for detailed view -->
<div class="report-instance" style="margin-top: 20px;">
    <h3>Top 10 Suppliers Data</h3>
    <table>
        <thead><tr><th>Rank</th><th>Supplier</th><th class="currency">Total Expenses</th></tr></thead>
        <tbody>
            <?php if (!empty($topSuppliers)): ?>
                <?php $rank = 1; ?>
                <?php foreach ($topSuppliers as $name => $total): ?>
                    <tr>
                        <td><?php echo $rank++; ?></td>
                        <td><?php echo htmlspecialchars($name); ?></td>
                        <td class="currency"><?php echo number_format($total, 2); ?></td>
                    </tr>
                <?php endforeach; ?>
            <?php else: ?>
                <tr><td colspan="3" class="no-data-message">No supplier expense data found for the selected period.</td></tr>
            <?php endif; ?>
        </tbody>
    </table>
</div>

<!-- NEW: Chart.js library and the script to render the graph -->
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Get the data processed by PHP
    const topSuppliersData = <?php echo json_encode($topSuppliers); ?>;
    const ctx = document.getElementById('topSuppliersChart');

    if (topSuppliersData && Object.keys(topSuppliersData).length > 0) {
        // Prepare data for Chart.js
        const labels = Object.keys(topSuppliersData);
        const data = Object.values(topSuppliersData);

        new Chart(ctx, {
            // Using a horizontal bar chart for ranked data
            type: 'bar',
            data: {
                labels: labels,
                datasets: [{
                    label: 'Total Expenses',
                    data: data,
                    backgroundColor: 'rgba(54, 162, 235, 0.7)',
                    borderColor: 'rgba(54, 162, 235, 1)',
                    borderWidth: 1
                }]
            },
            options: {
                // Set the bars to be horizontal
                indexAxis: 'y',
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false // Hide legend as the chart is self-explanatory
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                let label = context.dataset.label || '';
                                if (label) {
                                    label += ': ';
                                }
                                // Format the value as currency
                                label += new Intl.NumberFormat('en-US', { style: 'currency', currency: 'AUD' }).format(context.parsed.x);
                                return label;
                            }
                        }
                    }
                },
                scales: {
                    x: {
                        beginAtZero: true,
                        ticks: {
                            // Format the x-axis (bottom) as currency
                            callback: function(value, index, values) {
                                return '$' + value.toLocaleString();
                            }
                        }
                    }
                }
            }
        });
    } else {
        // If no data, display a message on the canvas
        const context = ctx.getContext('2d');
        context.textAlign = 'center';
        context.font = '16px Arial';
        context.fillText('No data available to display graph.', ctx.canvas.width / 2, ctx.canvas.height / 2);
    }
});
</script>