<?php
// qbo_ajax_fetch_month.php
// This is the background worker script that fetches data for ONE month.

header('Content-Type: application/json');

// Use required QBO SDK classes
use QuickBooksOnline\API\Core\OAuth\OAuth2\OAuth2LoginHelper;
use QuickBooksOnline\API\DataService\DataService;
use QuickBooksOnline\API\ReportService\ReportService;

// Load required files
require_once __DIR__ . '/../../vendor/autoload.php';
require_once __DIR__ . '/../qbo_functions.php';
require_once __DIR__ . '/../config/qbo_config.php';

global $qboBaseConfig;

// --- Get parameters from the AJAX request ---
$startDate = $_GET['month_start'] ?? null;
$endDate = $_GET['month_end'] ?? null;
$accountingMethod = $_GET['accounting_method'] ?? 'Accrual';

if (!$startDate || !$endDate) {
    echo json_encode(['error' => 'Missing date parameters.']);
    exit;
}

// --- Token and Connection Logic ---
$tokenStorageFile = __DIR__ . '/../tokens/qbo_token.json';
$currentTokens = file_exists($tokenStorageFile) ? json_decode(file_get_contents($tokenStorageFile), true) : null;

if (!$currentTokens || empty($currentTokens['refresh_token'])) {
    echo json_encode(['error' => 'Authentication token not found or invalid.']);
    exit;
}

try {
    $oauth2LoginHelper = new OAuth2LoginHelper($qboBaseConfig['ClientID'], $qboBaseConfig['ClientSecret']);
    $refreshedAccessTokenObj = $oauth2LoginHelper->refreshAccessTokenWithRefreshToken($currentTokens['refresh_token']);
    $newAccessToken = $refreshedAccessTokenObj->getAccessToken();
    $newRefreshToken = $refreshedAccessTokenObj->getRefreshToken();
    if ($newAccessToken && $newRefreshToken) {
        $currentTokens['access_token'] = $newAccessToken;
        file_put_contents($tokenStorageFile, json_encode(['access_token' => $newAccessToken, 'refresh_token' => $newRefreshToken], JSON_PRETTY_PRINT));
    }
} catch (Exception $e) {
    echo json_encode(['error' => 'Token refresh failed: ' . $e->getMessage()]);
    exit;
}

$qboBaseConfig['accessTokenKey'] = $currentTokens['access_token'];
try {
    $dataService = DataService::Configure($qboBaseConfig);
} catch (Exception $e) {
    echo json_encode(['error' => 'DataService configuration failed: ' . $e->getMessage()]);
    exit;
}

// --- Fetch the single month's report ---
$monthlyData = [];
try {
    $reportService = new ReportService($dataService->getServiceContext());
    $reportService->setStartDate($startDate);
    $reportService->setEndDate($endDate);
    $reportService->setAccountingMethod($accountingMethod);
    $report = $reportService->executeReport("ProfitAndLoss");
    
    if ($report && isset($report->Rows->Row)) {
        parseAndExtractData($report->Rows->Row, $monthlyData);
    }
} catch (Exception $e) {
    echo json_encode(['error' => 'Report fetching failed for period ' . $startDate . ': ' . $e->getMessage()]);
    exit;
}

// Simplified parsing function for the AJAX worker
function parseAndExtractData($rows, &$data) {
    $rows = is_array($rows) ? $rows : [$rows];
    foreach ($rows as $row) {
        if (isset($row->type) && $row->type === 'Data') {
            $accountName = $row->ColData[0]->value;
            $amount = (float)($row->ColData[1]->value ?? 0);
            $data[$accountName] = $amount;
        }
        if (isset($row->Summary->ColData)) {
            $summaryName = $row->Summary->ColData[0]->value;
            $summaryAmount = (float)($row->Summary->ColData[1]->value ?? 0);
            $data[$summaryName] = $summaryAmount;
        }
        if (isset($row->Rows->Row)) {
            parseAndExtractData($row->Rows->Row, $data);
        }
    }
}

echo json_encode(['success' => true, 'data' => $monthlyData]);