<?php
// qbo_ap_ageing_report.php
// This script is designed to be INCLUDED by report_viewer.php

// Use required QBO SDK classes
use QuickBooksOnline\API\Core\OAuth\OAuth2\OAuth2LoginHelper;
use QuickBooksOnline\API\Exception\SdkException;
use QuickBooksOnline\API\Exception\ServiceException;
use QuickBooksOnline\API\DataService\DataService;
use QuickBooksOnline\API\ReportService\ReportService;
use QuickBooksOnline\API\ReportService\ReportName;

// --- Load required files ---
require_once __DIR__ . '/../dbconn.php';
require_once __DIR__ . '/../../vendor/autoload.php';
require_once __DIR__ . '/../qbo_functions.php';
require_once __DIR__ . '/../config/qbo_config.php';

global $conn;
global $qboBaseConfig;

// --- Helper Functions (wrapped to prevent re-declaration) ---
$tokenStorageFile = __DIR__ . '/../tokens/qbo_token.json';
if (!function_exists('reportLog')) { function reportLog($message) { $timestamp = date('Y-M-d H:i:s'); echo "<!-- " . htmlspecialchars("[$timestamp] " . $message . "\n") . " -->\n"; } }
if (!function_exists('loadTokens')) { function loadTokens() { global $tokenStorageFile; if (file_exists($tokenStorageFile)) { $json = file_get_contents($tokenStorageFile); $tokens = json_decode($json, true); if (json_last_error() === JSON_ERROR_NONE) return $tokens; reportLog("ERROR: Failed to decode token storage file: " . json_last_error_msg()); return null; } reportLog("ERROR: Token storage file not found at: " . $tokenStorageFile); return null; } }
if (!function_exists('saveTokens')) { function saveTokens($accessToken, $refreshToken) { global $tokenStorageFile; $tokens = ['access_token' => $accessToken, 'refresh_token' => $refreshToken, 'last_updated' => date('Y-m-d H:i:s')]; $tokenDir = dirname($tokenStorageFile); if (!is_dir($tokenDir)) { mkdir($tokenDir, 0755, true); } if (file_put_contents($tokenStorageFile, json_encode($tokens, JSON_PRETTY_PRINT))) { reportLog("Tokens successfully saved to storage."); return true; } reportLog("FATAL: Could not write tokens to storage file: " . $tokenStorageFile); return false; } }
if (!function_exists('getQBOBillLink')) { function getQBOBillLink($billId, $baseUrl) { if (empty($billId)) return '#'; $isSandbox = (strcasecmp($baseUrl, 'Development') == 0); $linkBase = $isSandbox ? "https://app.sandbox.qbo.intuit.com/app/bill?txnId=" : "https://app.qbo.intuit.com/app/bill?txnId="; return $linkBase . htmlspecialchars($billId); } }
if (!function_exists('getColIndexByName')) { function getColIndexByName($report, $colName) { if (!isset($report->Columns->Column) || !is_array($report->Columns->Column)) return -1; foreach ($report->Columns->Column as $index => $column) { if (isset($column->ColTitle) && strcasecmp($column->ColTitle, $colName) == 0) return $index; } return -1; } }

// --- Report Variables & Logic ---
$dataService = null;
$apAgeingSummaryReport = null;
$apAgeingDetailReport = null;
$reportDate = date('Y-m-d');
$selectedVendorId = isset($_GET['vendor_id']) ? $_GET['vendor_id'] : null;
$selectedVendorName = isset($_GET['vendor_name']) ? $_GET['vendor_name'] : null;
$targetSupplierForDetail = $selectedVendorName ? $selectedVendorName : 'Select a supplier from the summary above';

reportLog("A/P Ageing Report Module Started.");
$currentTokens = loadTokens();
$qboConfigForDataService = $qboBaseConfig;
if ($currentTokens && !empty($currentTokens['refresh_token'])) {
    try { $oauth2LoginHelper = new OAuth2LoginHelper($qboBaseConfig['ClientID'], $qboBaseConfig['ClientSecret']); $refreshedAccessTokenObj = $oauth2LoginHelper->refreshAccessTokenWithRefreshToken($currentTokens['refresh_token']); $newAccessToken = $refreshedAccessTokenObj->getAccessToken(); $newRefreshToken = $refreshedAccessTokenObj->getRefreshToken(); if ($newAccessToken && $newRefreshToken) { if (saveTokens($newAccessToken, $newRefreshToken)) { $currentTokens['access_token'] = $newAccessToken; } } } catch (Exception $e) { reportLog("ERROR refreshing tokens: " . $e->getMessage()); }
}
if (!empty($currentTokens['access_token']) && !empty($qboBaseConfig['QBORealmID']) && !empty($qboBaseConfig['baseUrl'])) {
    $qboConfigForDataService['accessTokenKey'] = $currentTokens['access_token'];
    try { $dataService = DataService::Configure($qboConfigForDataService); if (isset($qboBaseConfig['logLocation'])) { $dataService->setLogLocation($qboBaseConfig['logLocation']); } reportLog("DataService instantiated successfully for A/P Report."); } catch (Exception $e) { exit("Error initializing QBO DataService connection: " . $e->getMessage()); }
} else { exit("QBO Authentication failed. Unable to load necessary parameters."); }

$summaryReportFetchSuccess = false;
if ($dataService) { try { reportLog("Fetching Aged Payables Summary Report for date: {$reportDate}"); $reportService = new ReportService($dataService->getServiceContext()); $reportService->setReportDate($reportDate); $apAgeingSummaryReport = $reportService->executeReport("AgedPayables"); if ($apAgeingSummaryReport) { $summaryReportFetchSuccess = true; reportLog("Successfully fetched Aged Payables Summary Report."); } } catch (Exception $e) { reportLog("Error fetching Aged Payables Summary Report: " . $e->getMessage()); } }
$detailReportFetchSuccess = false;
if ($dataService && $selectedVendorId) { try { reportLog("Fetching Aged Payables Detail Report for supplier ID: {$selectedVendorId}"); $reportServiceDetail = new ReportService($dataService->getServiceContext()); $reportServiceDetail->setReportDate($reportDate); $reportServiceDetail->setVendor($selectedVendorId); $apAgeingDetailReport = $reportServiceDetail->executeReport("AgedPayableDetail"); if ($apAgeingDetailReport) { $detailReportFetchSuccess = true; reportLog("Successfully fetched Aged Payables Detail Report for {$targetSupplierForDetail}."); } } catch (Exception $e) { reportLog("Error fetching Aged Payables Detail Report: " . $e->getMessage()); } }
?>

<!-- Report Header -->
<div class="report-header">
    <h2><?php echo htmlspecialchars($qboBaseConfig['QBORealmID'] ?? 'QuickBooks Online Company'); ?></h2>
</div>

<!-- A/P Ageing Summary Report Section -->
<section id="ap-ageing-summary">
    <div class="report-header">
        <!-- MODIFIED: Added class, data-target attribute, and span for the icon -->
        <h3 class="collapsible-header" data-target="#ap-summary-table">
            A/P Ageing Summary
            <span class="toggle-icon">▼</span>
        </h3>
        <p>As of <?php echo htmlspecialchars(date("F j, Y", strtotime($reportDate))); ?></p>
    </div>
    <!-- MODIFIED: Added an ID to the table and a class to make it the collapsible content -->
    <table id="ap-summary-table" class="collapsible-content">
        <thead>
            <tr>
                <th>Supplier</th>
                <th class="currency">CURRENT</th>
                <th class="currency">1 - 30</th>
                <th class="currency">31 - 60</th>
                <th class="currency">61 - 90</th>
                <th class="currency">91 AND OVER</th>
                <th class="currency">TOTAL</th>
            </tr>
        </thead>
        <tbody>
            <?php
            if ($summaryReportFetchSuccess && isset($apAgeingSummaryReport->Rows->Row)) {
                $summaryRows = is_array($apAgeingSummaryReport->Rows->Row) ? $apAgeingSummaryReport->Rows->Row : [$apAgeingSummaryReport->Rows->Row];
                $renderedSummaryRowsCount = 0;
                
                foreach ($summaryRows as $row) {
                    if (isset($row->group) && strcasecmp($row->group, 'GrandTotal') == 0) continue;
                    if (isset($row->type) && strcasecmp($row->type, 'Summary') == 0) continue;
                    $vendorId = null; $vendorName = null; $rowData = null;
                    if (isset($row->Header->ColData[0])) { $headerColData = $row->Header->ColData[0]; $vendorName = $headerColData->value ?? 'Unknown Vendor'; $vendorId = $headerColData->id ?? null; if (isset($row->Rows->Row)) { $nestedRows = is_array($row->Rows->Row) ? $row->Rows->Row : [$row->Rows->Row]; foreach ($nestedRows as $subRow) { if (isset($subRow->ColData) && count($subRow->ColData) > 1) { $rowData = $subRow->ColData; if (!$vendorId && isset($rowData[0]->id)) $vendorId = $rowData[0]->id; break; } } } } elseif (isset($row->ColData) && count($row->ColData) > 1) { $rowData = $row->ColData; $vendorName = $rowData[0]->value ?? 'Unknown Vendor'; $vendorId = $rowData[0]->id ?? null; }
                    if ($vendorId && $rowData && count($rowData) >= 7) {
                        $total = (float)($rowData[6]->value ?? 0);
                        if ($total == 0) continue;
                        $renderedSummaryRowsCount++;
                        ?>
                        <tr class='clickable-row' data-vendor-id='<?php echo htmlspecialchars($vendorId); ?>' data-vendor-name='<?php echo htmlspecialchars($vendorName); ?>'>
                            <td><strong><?php echo htmlspecialchars($vendorName); ?></strong></td>
                            <td class='currency'><?php echo htmlspecialchars(number_format((float)($rowData[1]->value ?? 0), 2)); ?></td>
                            <td class='currency'><?php echo htmlspecialchars(number_format((float)($rowData[2]->value ?? 0), 2)); ?></td>
                            <td class='currency'><?php echo htmlspecialchars(number_format((float)($rowData[3]->value ?? 0), 2)); ?></td>
                            <td class='currency'><?php echo htmlspecialchars(number_format((float)($rowData[4]->value ?? 0), 2)); ?></td>
                            <td class='currency'><?php echo htmlspecialchars(number_format((float)($rowData[5]->value ?? 0), 2)); ?></td>
                            <td class='currency'>A$<?php echo htmlspecialchars(number_format($total, 2)); ?></td>
                        </tr>
                        <?php
                    }
                }
                if ($renderedSummaryRowsCount === 0) { echo '<tr><td colspan="7" class="no-data-message">No A/P Ageing Summary data found for the selected date.</td></tr>'; }
            } else { echo '<tr><td colspan="7" class="no-data-message">Failed to fetch or parse A/P Ageing Summary Report. Check logs for details.</td></tr>'; }
            ?>
        </tbody>
        <tfoot>
            <?php
            if ($summaryReportFetchSuccess && isset($apAgeingSummaryReport->Summary->ColData) && count($apAgeingSummaryReport->Summary->ColData) >= 7) { $summaryData = $apAgeingSummaryReport->Summary->ColData; ?>
                <tr class="total-row"><td><strong>TOTAL</strong></td><td class="currency">A$<?php echo htmlspecialchars(number_format((float)($summaryData[1]->value ?? 0), 2)); ?></td><td class="currency">A$<?php echo htmlspecialchars(number_format((float)($summaryData[2]->value ?? 0), 2)); ?></td><td class="currency">A$<?php echo htmlspecialchars(number_format((float)($summaryData[3]->value ?? 0), 2)); ?></td><td class="currency">A$<?php echo htmlspecialchars(number_format((float)($summaryData[4]->value ?? 0), 2)); ?></td><td class="currency">A$<?php echo htmlspecialchars(number_format((float)($summaryData[5]->value ?? 0), 2)); ?></td><td class="currency">A$<?php echo htmlspecialchars(number_format((float)($summaryData[6]->value ?? 0), 2)); ?></td></tr>
            <?php } ?>
        </tfoot>
    </table>
</section>

<hr>

<!-- A/P Ageing Detail Report Section (No changes here) -->
<section id="ap-ageing-detail">
    <div class="report-header">
        <h3>A/P Ageing Detail</h3>
        <p>For Supplier: <?php echo htmlspecialchars($targetSupplierForDetail); ?> <br> As of <?php echo htmlspecialchars(date("F j, Y", strtotime($reportDate))); ?></p>
    </div>
    <?php if ($selectedVendorId): ?>
    <table>
        <thead><tr><th>DATE</th><th>TRANSACTION TYPE</th><th>NO.</th><th>DUE DATE</th><th class="currency">PAST DUE</th><th class="currency">AMOUNT</th><th class="currency">OPEN BALANCE</th></tr></thead>
        <tbody>
            <?php
            if ($detailReportFetchSuccess && isset($apAgeingDetailReport->Rows->Row)) {
                $detailRows = is_array($apAgeingDetailReport->Rows->Row) ? $apAgeingDetailReport->Rows->Row : [$apAgeingDetailReport->Rows->Row];
                $dateCol = getColIndexByName($apAgeingDetailReport, 'Date'); $typeCol = getColIndexByName($apAgeingDetailReport, 'Transaction Type'); $noCol = getColIndexByName($apAgeingDetailReport, 'No.'); $dueDateCol = getColIndexByName($apAgeingDetailReport, 'Due Date'); $pastDueCol = getColIndexByName($apAgeingDetailReport, 'Past Due'); $amountCol = getColIndexByName($apAgeingDetailReport, 'Amount'); $openBalCol = getColIndexByName($apAgeingDetailReport, 'Open Balance'); $renderedDetailRows = 0;
                foreach ($detailRows as $row) {
                    $rowsToProcess = [];
                    if (isset($row->type) && strcasecmp($row->type, 'Data') == 0) { $rowsToProcess[] = $row; } elseif (isset($row->type) && strcasecmp($row->type, 'Section') == 0 && isset($row->Rows->Row)) { $nestedRows = is_array($row->Rows->Row) ? $row->Rows->Row : [$row->Rows->Row]; $rowsToProcess = array_merge($rowsToProcess, $nestedRows); }
                    foreach ($rowsToProcess as $dataRow) {
                        if (isset($dataRow->type) && strcasecmp($dataRow->type, 'Data') == 0 && isset($dataRow->ColData)) {
                            $rowData = $dataRow->ColData; $txnType = $rowData[$typeCol]->value ?? 'N/A'; $docNumRaw = $rowData[$noCol]->value ?? 'N/A'; $billId = null; if (isset($rowData[$noCol]->attr->href)) { parse_str(parse_url($rowData[$noCol]->attr->href, PHP_URL_QUERY), $queryParams); $billId = $queryParams['txnid'] ?? null; } $docNumDisplay = $docNumRaw; if ($billId && strcasecmp($txnType, 'Bill') == 0) { $billLink = getQBOBillLink($billId, $qboBaseConfig['baseUrl']); $docNumDisplay = "<a href='{$billLink}' target='_blank'>" . htmlspecialchars($docNumRaw) . "</a>"; } $renderedDetailRows++;
                            ?>
                            <tr><td><?php echo htmlspecialchars($rowData[$dateCol]->value ?? 'N/A'); ?></td><td><?php echo htmlspecialchars($txnType); ?></td><td><?php echo $docNumDisplay; ?></td><td><?php echo htmlspecialchars($rowData[$dueDateCol]->value ?? 'N/A'); ?></td><td class="currency"><?php echo htmlspecialchars($rowData[$pastDueCol]->value ?? 'N/A'); ?></td><td class="currency"><?php echo htmlspecialchars(number_format((float)($rowData[$amountCol]->value ?? 0), 2)); ?></td><td class="currency"><?php echo htmlspecialchars(number_format((float)($rowData[$openBalCol]->value ?? 0), 2)); ?></td></tr>
                            <?php
                        }
                    }
                }
                if ($renderedDetailRows === 0) { echo '<tr><td colspan="7" class="no-data-message">No transaction details found for this supplier.</td></tr>'; }
            } elseif ($detailReportFetchSuccess) { echo '<tr><td colspan="7" class="no-data-message">No transaction details found for this supplier.</td></tr>'; } else { echo '<tr><td colspan="7" class="no-data-message">Could not load detail report for this supplier. Check logs.</td></tr>'; }
            ?>
        </tbody>
        <tfoot>
            <?php
            if ($detailReportFetchSuccess && isset($apAgeingDetailReport->Summary->ColData)) { $detailSummaryData = $apAgeingDetailReport->Summary->ColData; $amountCol = getColIndexByName($apAgeingDetailReport, 'Amount'); $openBalCol = getColIndexByName($apAgeingDetailReport, 'Open Balance'); ?>
                <tr class="total-row"><td colspan="5"><strong>Total for <?php echo htmlspecialchars($targetSupplierForDetail); ?></strong></td><td class="currency"><strong>A$<?php echo htmlspecialchars(number_format((float)($detailSummaryData[$amountCol]->value ?? 0), 2)); ?></strong></td><td class="currency"><strong>A$<?php echo htmlspecialchars(number_format((float)($detailSummaryData[$openBalCol]->value ?? 0), 2)); ?></strong></td></tr>
            <?php } ?>
        </tfoot>
    </table>
    <?php else: ?>
        <p class="no-data-message">Select a supplier from the A/P Ageing Summary above to view their detailed ageing report.</p>
    <?php endif; ?>
</section>

<!-- Report Generation Timestamp -->
<div class="small-text">
    A/P Report generated on <?php echo date('l j F Y H:i A T'); ?>
</div>

<!-- JavaScript for This Specific Report -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    const summaryRows = document.querySelectorAll('#ap-ageing-summary .clickable-row');
    summaryRows.forEach(row => {
        row.addEventListener('click', function() {
            const vendorId = this.getAttribute('data-vendor-id');
            const vendorName = this.getAttribute('data-vendor-name');
            if (vendorId) {
                const urlParams = new URLSearchParams(window.location.search);
                urlParams.set('vendor_id', vendorId);
                urlParams.set('vendor_name', vendorName);
                window.location.href = `${window.location.pathname}?${urlParams.toString()}`;
            }
        });
    });

    if (window.location.search.includes('vendor_id=')) {
        setTimeout(() => {
            const detailSection = document.getElementById('ap-ageing-detail');
            if (detailSection) {
                detailSection.scrollIntoView({ behavior: 'smooth', block: 'start' });
            }
        }, 100);
    }
});
</script>