<?php
// qbo_income_by_customer_report.php
// FINAL CORRECTED VERSION: Adds the necessary token refresh logic to the main page to ensure the customer list always loads.

use QuickBooksOnline\API\Core\OAuth\OAuth2\OAuth2LoginHelper;
use QuickBooksOnline\API\DataService\DataService;

require_once __DIR__ . '/../../vendor/autoload.php';
require_once __DIR__ . '/../qbo_functions.php';
require_once __DIR__ . '/../config/qbo_config.php';
global $qboBaseConfig;

// --- Report Filters & Variables ---
$finYearStart = (date('m') < 7) ? date('Y-07-01', strtotime('-1 year')) : date('Y-07-01');
$startDate = isset($_GET['start_date']) && !empty($_GET['start_date']) ? $_GET['start_date'] : $finYearStart;
$endDate = isset($_GET['end_date']) && !empty($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-d');
$accountingMethod = isset($_GET['accounting_method']) ? $_GET['accounting_method'] : 'Accrual';

$dataService = null;
$allCustomers = [];

// --- QBO Connection & Token Refresh ---
// THIS IS THE CRITICAL FIX: Ensure the token is refreshed before trying to use the DataService.
$tokenStorageFile = __DIR__ . '/../tokens/qbo_token.json';
$currentTokens = file_exists($tokenStorageFile) ? json_decode(file_get_contents($tokenStorageFile), true) : null;
if ($currentTokens && !empty($currentTokens['refresh_token'])) {
    try {
        $oauth2LoginHelper = new OAuth2LoginHelper($qboBaseConfig['ClientID'], $qboBaseConfig['ClientSecret']);
        $refreshedAccessTokenObj = $oauth2LoginHelper->refreshAccessTokenWithRefreshToken($currentTokens['refresh_token']);
        $newAccessToken = $refreshedAccessTokenObj->getAccessToken();
        $newRefreshToken = $refreshedAccessTokenObj->getRefreshToken();
        if ($newAccessToken && $newRefreshToken) {
            file_put_contents($tokenStorageFile, json_encode(['access_token' => $newAccessToken, 'refresh_token' => $newRefreshToken], JSON_PRETTY_PRINT));
            $currentTokens['access_token'] = $newAccessToken;
        }
    } catch (Exception $e) {
        // Suppress error and continue, subsequent calls will fail gracefully
    }
}

if ($currentTokens && !empty($currentTokens['access_token'])) {
    $qboBaseConfig['accessTokenKey'] = $currentTokens['access_token'];
    try { $dataService = DataService::Configure($qboBaseConfig); } catch (Exception $e) { $dataService = null; }
}

if ($dataService) {
    try {
        $allCustomers = $dataService->Query("SELECT Id, DisplayName FROM Customer WHERE Active = true ORDERBY DisplayName ASC");
    } catch (Exception $e) { $allCustomers = []; }
}
?>

<div class="report-header">
    <h2>Income by Customer Summary</h2>
    <p>For period: <?php echo htmlspecialchars(date("F j, Y", strtotime($startDate))) . " - " . htmlspecialchars(date("F j, Y", strtotime($endDate))); ?> </p>
</div>

<form method="get" class="filter-form">
    <input type="hidden" name="report" value="income_by_customer">
    <div>
        <label>Report period:</label>
        <select id="report_period_selector">
            <option value="This_month_to_date">This month to date</option><option value="This_Month">This Month</option>
            <option value="This_Fiscal_Year" selected>This Financial Year</option><option value="Last_Month">Last Month</option>
            <option value="Last_Fiscal_Year">Last Financial Year</option><option value="custom">Custom</option>
        </select>
    </div>
    <div><label>From:</label><input type="date" name="start_date" id="start_date" value="<?php echo htmlspecialchars($startDate); ?>"></div>
    <div><label>To:</label><input type="date" name="end_date" id="end_date" value="<?php echo htmlspecialchars($endDate); ?>"></div>
    <div>
        <label>Accounting Method:</label>
        <select name="accounting_method">
            <option value="Accrual" <?php if($accountingMethod == 'Accrual') echo 'selected'; ?>>Accrual</option>
            <option value="Cash" <?php if($accountingMethod == 'Cash') echo 'selected'; ?>>Cash</option>
        </select>
    </div>
    <button type="submit">Run Report</button>
</form>

<div id="loading-indicator" style="text-align: center; padding: 40px; font-size: 1.2em;"><p>Loading report data, please wait...</p></div>
<section id="income-summary-report" style="display:none;">
    <table>
        <thead><tr><th>Customer</th><th class="currency">Income</th><th class="currency">Expenses</th><th class="currency">Net Income</th></tr></thead>
        <tbody id="report-body">
            <?php if (!empty($allCustomers)): ?>
                <?php foreach ($allCustomers as $customer): ?>
                    <tr class="customer-row" data-customer-id="<?php echo $customer->Id; ?>" data-customer-name="<?php echo htmlspecialchars($customer->DisplayName); ?>">
                        <td><strong><a href="#" class="customer-detail-link"><?php echo htmlspecialchars($customer->DisplayName); ?></a></strong></td>
                        <td class="currency income-cell">...</td>
                        <td class="currency expenses-cell">...</td>
                        <td class="currency net-cell">...</td>
                    </tr>
                <?php endforeach; ?>
            <?php else: ?>
                <tr><td colspan="4" class="no-data-message">Could not retrieve customer list. Please check your QBO connection.</td></tr>
            <?php endif; ?>
        </tbody>
        <tfoot>
            <tr class="total-row">
                <td><strong>TOTAL</strong></td>
                <td class="currency" id="total-income">0.00</td>
                <td class="currency" id="total-expenses">0.00</td>
                <td class="currency" id="total-net">0.00</td>
            </tr>
        </tfoot>
    </table>
</section>

<script>
document.addEventListener('DOMContentLoaded', async function() {
    const reportSection = document.getElementById('income-summary-report');
    const loadingIndicator = document.getElementById('loading-indicator');
    const tbody = document.getElementById('report-body');
    const customerRows = document.querySelectorAll('.customer-row');

    if(customerRows.length === 0){
        loadingIndicator.innerHTML = '<p>Could not load customer list. Please check your QBO connection.</p>';
        return;
    }

    reportSection.style.display = 'block';
    loadingIndicator.style.display = 'none';
    
    const startDate = '<?php echo $startDate; ?>';
    const endDate = '<?php echo $endDate; ?>';
    const accountingMethod = '<?php echo $accountingMethod; ?>';
    let totalIncome = 0, totalExpenses = 0, totalNet = 0;

    // --- CHANGE #1: Add a delay helper function ---
    const delay = ms => new Promise(resolve => setTimeout(resolve, ms));

    // --- CHANGE #2: Replace the old fetch function with this smarter one ---
    async function fetchCustomerData(customerId, retries = 3, backoff = 1000) {
        const url = `qbo_ajax_fetch_customer_pnl.php?customer_id=${customerId}&start_date=${startDate}&end_date=${endDate}&accounting_method=${accountingMethod}`;
        try {
            const response = await fetch(url);
            // If we are rate-limited (429) and have retries left, wait and try again
            if (response.status === 429 && retries > 0) {
                console.warn(`Rate limit hit for customer ${customerId}. Retrying in ${backoff / 1000}s...`);
                await delay(backoff);
                // Retry with one less attempt and double the wait time for the next potential failure
                return fetchCustomerData(customerId, retries - 1, backoff * 2);
            }
            // For any other non-successful response, return an error
            if (!response.ok) {
                return { error: `HTTP error! status: ${response.status}` };
            }
            // If successful, return the JSON data
            return await response.json();
        } catch (e) {
            // Handle network errors
            return { error: e.message };
        }
    }

    const processChunk = async (chunk) => {
        // This part of the code does not need to change at all
        const promises = chunk.map(row => fetchCustomerData(row.dataset.customerId));
        const results = await Promise.all(promises);
        chunk.forEach((row, index) => {
            const result = results[index];
            const customerId = row.dataset.customerId;
            const customerName = row.dataset.customerName;
            const incomeCell = row.querySelector('.income-cell');
            const expensesCell = row.querySelector('.expenses-cell');
            const netCell = row.querySelector('.net-cell');
            const customerLink = row.querySelector('.customer-detail-link');

            const detailUrl = `?report=sales_by_customer_detail&customer_id=${customerId}&start_date=${startDate}&end_date=${endDate}&accounting_method=${accountingMethod}`;
            customerLink.href = detailUrl;

            if (result && result.success) {
                const income = result.income || 0, expenses = result.expenses || 0, net = income - expenses;
                if (income === 0 && expenses === 0) { row.style.display = 'none'; return; }
                totalIncome += income; totalExpenses += expenses; totalNet += net;
                
                const pnlUrl = `?report=pl_report&customer=${customerId}&start_date=${startDate}&end_date=${endDate}&accounting_method=${accountingMethod}`;
                
                incomeCell.innerHTML = income !== 0 ? `<a href="${pnlUrl}">${income.toFixed(2)}</a>` : '-';
                expensesCell.innerHTML = expenses !== 0 ? `<a href="${pnlUrl}">${expenses.toFixed(2)}</a>` : '-';
                netCell.innerHTML = net !== 0 ? `<a href="${pnlUrl}">${net.toFixed(2)}</a>` : '-';
            } else {
                console.error(`Failed to load data for ${customerName}:`, (result ? result.error : 'Unknown error'));
                netCell.innerHTML = `<span style="color: red;">Failed</span>`;
            }
        });
    };

    // --- CHANGE #3: The main loop can be simplified again ---
    // We can go back to a larger chunk size and remove the artificial delay
    // because the fetch function now handles its own rate limiting.
    const chunkSize = 5; 

    for (let i = 0; i < customerRows.length; i += chunkSize) {
        try {
            const chunk = Array.from(customerRows).slice(i, i + chunkSize);
            await processChunk(chunk); // No more artificial delay needed here
        } catch (error) {
            console.error("A critical error occurred while processing a chunk, but continuing...", error);
        }
    }
    
    document.getElementById('total-income').textContent = totalIncome.toFixed(2);
    document.getElementById('total-expenses').textContent = totalExpenses.toFixed(2);
    document.getElementById('total-net').textContent = totalNet.toFixed(2);

    // Date logic remains the same...
    document.getElementById('report_period_selector').addEventListener('change', function() {
        const selectedPeriod = this.value; const today = new Date();
        let startDate, endDate;
        const formatDate = (d) => { const pad = (num) => num.toString().padStart(2, '0'); return `${d.getFullYear()}-${pad(d.getMonth() + 1)}-${pad(d.getDate())}`; }
        const finYearStartMonth = 6;
        switch (selectedPeriod) {
            case 'This_month_to_date': startDate = new Date(today.getFullYear(), today.getMonth(), 1); endDate = today; break;
            case 'This_Month': startDate = new Date(today.getFullYear(), today.getMonth(), 1); endDate = new Date(today.getFullYear(), today.getMonth() + 1, 0); break;
            case 'This_Fiscal_Year':
                let finYear = today.getFullYear();
                if (today.getMonth() < finYearStartMonth) finYear--;
                startDate = new Date(finYear, finYearStartMonth, 1);
                endDate = new Date(finYear + 1, finYearStartMonth, 0);
                break;
            case 'Last_Month': startDate = new Date(today.getFullYear(), today.getMonth() - 1, 1); endDate = new Date(today.getFullYear(), today.getMonth(), 0); break;
            case 'Last_Fiscal_Year':
                let lastFinYear = today.getFullYear();
                if (today.getMonth() < finYearStartMonth) lastFinYear--;
                startDate = new Date(lastFinYear - 1, finYearStartMonth, 1);
                endDate = new Date(lastFinYear, finYearStartMonth, 0);
                break;
            case 'custom': default: return;
        }
        if(startDate && endDate){
            document.getElementById('start_date').value = formatDate(startDate);
            document.getElementById('end_date').value = formatDate(endDate);
        }
    });
});
</script>