<?php
// qbo_pl_by_month_report.php
// FINAL POLISHED VERSION: Corrects the fatal API error for report filtering.

// Use required QBO SDK classes
use QuickBooksOnline\API\Core\OAuth\OAuth2\OAuth2LoginHelper;
use QuickBooksOnline\API\DataService\DataService;
use QuickBooksOnline\API\ReportService\ReportService;

// --- Load required files ---
require_once __DIR__ . '/../dbconn.php';
require_once __DIR__ . '/../../vendor/autoload.php';
require_once __DIR__ . '/../qbo_functions.php';
require_once __DIR__ . '/../config/qbo_config.php';

global $conn;
global $qboBaseConfig;

// --- Helper Functions ---
$tokenStorageFile = __DIR__ . '/../tokens/qbo_token.json';
if (!function_exists('reportLog')) { function reportLog($message) { $timestamp = date('Y-m-d H:i:s'); echo "<!-- " . htmlspecialchars("[$timestamp] " . $message . "\n") . " -->\n"; } }
if (!function_exists('loadTokens')) { function loadTokens() { global $tokenStorageFile; if (file_exists($tokenStorageFile)) { return json_decode(file_get_contents($tokenStorageFile), true); } return null; } }
if (!function_exists('saveTokens')) { function saveTokens($accessToken, $refreshToken) { global $tokenStorageFile; $tokens = ['access_token' => $accessToken, 'refresh_token' => $refreshToken, 'last_updated' => date('Y-m-d H:i:s')]; file_put_contents($tokenStorageFile, json_encode($tokens, JSON_PRETTY_PRINT)); return true; } }
if (!function_exists('getQBOTransactionLink')) { function getQBOTransactionLink($txnId, $txnType, $baseUrl) { if (empty($txnId)) return '#'; $isSandbox = (strcasecmp($baseUrl, 'Development') == 0); $qboDomain = $isSandbox ? "https://app.sandbox.qbo.intuit.com/app/" : "https://app.qbo.intuit.com/app/"; $page = strtolower(str_replace(' ', '', $txnType)); if ($page === 'journalentry') $page = 'journal'; return $qboDomain . $page . "?txnId=" . htmlspecialchars($txnId); } }
if (!function_exists('getColIndexByName')) { function getColIndexByName($report, $colName) { if (!isset($report->Columns->Column) || !is_array($report->Columns->Column)) return -1; foreach ($report->Columns->Column as $index => $column) { if (isset($column->ColTitle) && strcasecmp($column->ColTitle, $colName) == 0) return $index; } return -1; } }

// --- Report Filters & Variables ---
$finYearStart = (date('m') < 7) ? date('Y-07-01', strtotime('-1 year')) : date('Y-07-01');
$startDate = isset($_GET['start_date']) ? $_GET['start_date'] : $finYearStart;
$endDate = isset($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-d');
$accountingMethod = isset($_GET['accounting_method']) ? $_GET['accounting_method'] : 'Accrual';
$detailAccountId = isset($_GET['account_id']) ? $_GET['account_id'] : null;
$detailAccountName = isset($_GET['account_name']) ? $_GET['account_name'] : null;
$detailStartDate = isset($_GET['detail_start_date']) ? $_GET['detail_start_date'] : null;
$detailEndDate = isset($_GET['detail_end_date']) ? $_GET['detail_end_date'] : null;
$detailMonthName = isset($_GET['month']) ? $_GET['month'] : null;

$dataService = null;
$detailReport = null;
$masterReportData = [];
$columnHeaders = ['Account'];
$months = [];

// --- QBO Connection & Token Refresh ---
reportLog("P&L Report Module Started.");
$currentTokens = loadTokens();
if ($currentTokens && !empty($currentTokens['refresh_token'])) { try { $oauth2LoginHelper = new OAuth2LoginHelper($qboBaseConfig['ClientID'], $qboBaseConfig['ClientSecret']); $refreshedAccessTokenObj = $oauth2LoginHelper->refreshAccessTokenWithRefreshToken($currentTokens['refresh_token']); $newAccessToken = $refreshedAccessTokenObj->getAccessToken(); $newRefreshToken = $refreshedAccessTokenObj->getRefreshToken(); if ($newAccessToken && $newRefreshToken) { saveTokens($newAccessToken, $newRefreshToken); $currentTokens['access_token'] = $newAccessToken; } } catch (Exception $e) { reportLog("ERROR refreshing tokens: " . $e->getMessage()); } }
if (!empty($currentTokens['access_token']) && !empty($qboBaseConfig['QBORealmID'])) { $qboBaseConfig['accessTokenKey'] = $currentTokens['access_token']; try { $dataService = DataService::Configure($qboBaseConfig); reportLog("DataService instantiated successfully for P&L Report."); } catch (Exception $e) { exit("Error initializing QBO DataService connection: " . $e->getMessage()); }
} else { exit("QBO Authentication failed. Unable to load necessary parameters."); }

// --- Data Fetching Logic (Multi-Query Workaround) ---
$fetchSuccess = false;
if ($dataService) {
    $period = new DatePeriod(new DateTime($startDate), new DateInterval('P1M'), (new DateTime($endDate))->modify('+1 day'));
    foreach ($period as $date) {
        $monthKey = $date->format('F Y');
        $months[] = ['key' => $monthKey, 'start' => $date->format('Y-m-01'), 'end' => $date->format('Y-m-t')];
        $columnHeaders[] = $monthKey;
    }
    $columnHeaders[] = 'TOTAL';
    $allPeriodsToQuery = array_merge($months, [['key' => 'TOTAL', 'start' => $startDate, 'end' => $endDate]]);
    foreach ($allPeriodsToQuery as $queryPeriod) {
        reportLog("Fetching P&L for " . $queryPeriod['key']);
        try {
            $reportService = new ReportService($dataService->getServiceContext());
            $reportService->setStartDate($queryPeriod['start']); $reportService->setEndDate($queryPeriod['end']);
            $reportService->setAccountingMethod($accountingMethod);
            $report = $reportService->executeReport("ProfitAndLoss");
            if ($report && isset($report->Rows->Row)) {
                parseAndMergeReport($report->Rows->Row, $masterReportData, $queryPeriod['key']);
                $fetchSuccess = true;
            }
        } catch (Exception $e) { reportLog("Error fetching P&L for " . $queryPeriod['key'] . ": " . $e->getMessage()); }
    }
}

function parseAndMergeReport($rows, &$masterData, $colKey) {
    $rows = is_array($rows) ? $rows : [$rows];
    foreach ($rows as $row) {
        if (isset($row->type) && $row->type === 'Data') {
            $accountName = $row->ColData[0]->value; $accountId = $row->ColData[0]->id ?? null;
            $amount = (float)($row->ColData[1]->value ?? 0);
            if (!isset($masterData[$accountName])) $masterData[$accountName] = ['id' => $accountId, 'values' => []];
            $masterData[$accountName]['values'][$colKey] = $amount;
        }
        if (isset($row->Summary->ColData)) {
            $summaryName = $row->Summary->ColData[0]->value;
            $summaryAmount = (float)($row->Summary->ColData[1]->value ?? 0);
            if (!isset($masterData[$summaryName])) $masterData[$summaryName] = ['id' => null, 'values' => []];
            $masterData[$summaryName]['values'][$colKey] = $summaryAmount;
        }
        if (isset($row->Rows->Row)) parseAndMergeReport($row->Rows->Row, $masterData, $colKey);
    }
}

// ** THE FIX IS HERE **
// Fetch Drill-Down Report using GENERAL LEDGER
if ($dataService && $detailAccountId && $detailStartDate && $detailEndDate) {
    try {
        reportLog("Fetching General Ledger Report for Account ID(s) {$detailAccountId}");
        $reportService = new ReportService($dataService->getServiceContext());
        $reportService->setStartDate($detailStartDate);
        $reportService->setEndDate($detailEndDate);
        
        // The setAccount() method correctly handles a single ID or a comma-separated list of IDs.
        // The undefined method `setAccountFilter()` has been removed.
        $reportService->setAccount($detailAccountId);
        
        $detailReport = $reportService->executeReport("GeneralLedger");
    } catch (Exception $e) {
        reportLog("Error fetching General Ledger Report: " . $e->getMessage());
    }
}
?>

<!-- Report Header & Filters -->
<div class="report-header">
    <h2>Profit & Loss by Month</h2>
    <p>For period: <?php echo htmlspecialchars(date("F j, Y", strtotime($startDate))) . " - " . htmlspecialchars(date("F j, Y", strtotime($endDate))); ?> </p>
</div>

<form method="get" class="filter-form">
    <input type="hidden" name="report" value="pl_by_month">
    <div>
        <label for="report_period">Report period:</label>
        <select id="report_period" name="report_period">
            <option value="custom">Custom</option>
            <option value="today">Today</option>
            <option value="this_week">This Week</option>
            <option value="this_month">This Month</option>
            <option value="this_quarter">This Quarter</option>
            <option value="this_fin_year">This Financial Year</option>
            <option value="last_month">Last Month</option>
            <option value="last_quarter">Last Quarter</option>
            <option value="last_fin_year">Last Financial Year</option>
        </select>
    </div>
    <div><label>From:</label><input type="date" id="start_date" name="start_date" value="<?php echo htmlspecialchars($startDate); ?>"></div>
    <div><label>To:</label><input type="date" id="end_date" name="end_date" value="<?php echo htmlspecialchars($endDate); ?>"></div>
    <div><label>Method:</label><select name="accounting_method"><option value="Accrual" <?php echo ($accountingMethod==='Accrual' ? 'selected' : ''); ?>>Accrual</option><option value="Cash" <?php echo ($accountingMethod==='Cash' ? 'selected' : ''); ?>>Cash</option></select></div>
    <button type="submit">Run Report</button>
</form>

<!-- Profit & Loss Report Section -->
<section id="pl-report">
    <table>
        <thead><tr><?php foreach ($columnHeaders as $header) echo "<th>" . str_replace(' ', ' ', htmlspecialchars($header)) . "</th>"; ?></tr></thead>
        <tbody>
            <?php
            function getAccountIdsFromSection($section, &$masterData) {
                $ids = [];
                if (isset($section->Rows->Row)) {
                    $rows = is_array($section->Rows->Row) ? $section->Rows->Row : [$section->Rows->Row];
                    foreach ($rows as $row) {
                        if (isset($row->type) && $row->type === 'Data') {
                            $accountName = $row->ColData[0]->value;
                            if (isset($masterData[$accountName]['id'])) {
                                $ids[] = $masterData[$accountName]['id'];
                            }
                        } elseif (isset($row->type) && $row->type === 'Section') {
                            $ids = array_merge($ids, getAccountIdsFromSection($row, $masterData));
                        }
                    }
                }
                return array_unique($ids);
            }

            if ($fetchSuccess && !empty($masterReportData)) {
                $reportService = new ReportService($dataService->getServiceContext());
                $reportService->setStartDate($startDate);
                $reportService->setEndDate($endDate);
                $reportService->setAccountingMethod($accountingMethod);
                $finalReportStructure = $reportService->executeReport("ProfitAndLoss");
                if ($finalReportStructure) {
                    renderFinalReport($finalReportStructure->Rows->Row, $masterReportData, $columnHeaders, $months, $startDate, $endDate);
                } else {
                    echo "<tr><td colspan='" . count($columnHeaders) . "' class='no-data-message'>Error building final report structure.</td></tr>";
                }
            } else {
                echo "<tr><td colspan='" . count($columnHeaders) . "' class='no-data-message'>Could not fetch report data.</td></tr>";
            }

            function renderFinalReport($rows, &$masterData, $columnHeaders, $months, $reportStartDate, $reportEndDate, $indent = 0) {
                 $rows = is_array($rows) ? $rows : [$rows];
                 foreach($rows as $row) {
                    $rowType = $row->type ?? 'Data';
                    if ($rowType === 'Section') {
                        if (isset($row->Header->ColData)) {
                            $accountName = htmlspecialchars($row->Header->ColData[0]->value);
                            echo "<tr class='pl-header-row'><td class='pl-indent-{$indent}'><strong>{$accountName}</strong></td>";
                            for ($i = 1; $i < count($columnHeaders); $i++) echo "<td></td>";
                            echo "</tr>";
                        }
                        if (isset($row->Rows->Row)) {
                            renderFinalReport($row->Rows->Row, $masterData, $columnHeaders, $months, $reportStartDate, $reportEndDate, $indent + 1);
                        }
                        if (isset($row->Summary->ColData)) {
                            $summaryName = htmlspecialchars($row->Summary->ColData[0]->value);
                            $amounts = $masterData[$summaryName]['values'] ?? [];
                            $summaryAccountIds = getAccountIdsFromSection($row, $masterData);
                            $summaryAccountIdString = implode(',', $summaryAccountIds);

                            echo "<tr class='pl-total-row pl-indent-{$indent}'><td><strong>{$summaryName}</strong></td>";
                            
                            foreach ($months as $month) {
                                $val = $amounts[$month['key']] ?? 0;
                                if ($val != 0 && !empty($summaryAccountIdString)) {
                                    echo "<td class='currency clickable-amount'><strong><a href='#' data-account-id='{$summaryAccountIdString}' data-account-name='{$summaryName}' data-start-date='{$month['start']}' data-end-date='{$month['end']}' data-month='{$month['key']}'>" . number_format($val, 2) . "</a></strong></td>";
                                } else {
                                    echo "<td class='currency'><strong>" . ($val != 0 ? number_format($val, 2) : '-') . "</strong></td>";
                                }
                            }
                            $total = $amounts['TOTAL'] ?? 0;
                            if ($total != 0 && !empty($summaryAccountIdString)) {
                                 echo "<td class='currency clickable-amount'><strong><a href='#' data-account-id='{$summaryAccountIdString}' data-account-name='{$summaryName}' data-start-date='{$reportStartDate}' data-end-date='{$reportEndDate}' data-month='Total'>" . number_format($total, 2) . "</a></strong></td>";
                            } else {
                                echo "<td class='currency'><strong>" . ($total != 0 ? number_format($total, 2) : '-') . "</strong></td>";
                            }
                            echo "</tr>";
                        }
                    } elseif ($rowType === 'Data') {
                        $accountName = htmlspecialchars($row->ColData[0]->value);
                        $accountId = $masterData[$accountName]['id'] ?? null;
                        $amounts = $masterData[$accountName]['values'] ?? [];
                        echo "<tr class='pl-data-row'><td class='pl-indent-{$indent}'>{$accountName}</td>";
                        foreach ($months as $month) {
                            $val = $amounts[$month['key']] ?? 0;
                            if ($val != 0 && $accountId) {
                                echo "<td class='currency clickable-amount'><a href='#' data-account-id='{$accountId}' data-account-name='{$accountName}' data-start-date='{$month['start']}' data-end-date='{$month['end']}' data-month='{$month['key']}'>" . number_format($val, 2) . "</a></td>";
                            } else {
                                echo "<td class='currency'>" . ($val != 0 ? number_format($val, 2) : '-') . "</td>";
                            }
                        }
                        $total = $amounts['TOTAL'] ?? 0;
                        if ($total != 0 && $accountId) {
                             echo "<td class='currency clickable-amount'><a href='#' data-account-id='{$accountId}' data-account-name='{$accountName}' data-start-date='{$reportStartDate}' data-end-date='{$reportEndDate}' data-month='Total'>" . number_format($total, 2) . "</a></td>";
                        } else {
                            echo "<td class='currency'>" . ($total != 0 ? number_format($total, 2) : '-') . "</td>";
                        }
                        echo "</tr>";
                    }
                 }
            }
            ?>
        </tbody>
    </table>
</section>

<hr>

<!-- Transaction Detail Report Section -->
<section id="detail-report">
     <?php if (isset($detailReport) && $detailAccountId): ?>
        <div class="report-header">
            <h3>General Ledger Report</h3>
            <p>For Account: <strong><?php echo htmlspecialchars($detailAccountName); ?></strong> (<?php echo htmlspecialchars($detailMonthName); ?>)</p>
        </div>
        <table>
            <thead><tr><?php foreach($detailReport->Columns->Column as $column) echo "<th>" . htmlspecialchars($column->ColTitle) . "</th>"; ?></tr></thead>
            <tbody>
                <?php
                if (isset($detailReport->Rows->Row)) {
                    $detailRows = is_array($detailReport->Rows->Row) ? $detailReport->Rows->Row : [$detailReport->Rows->Row];
                    foreach ($detailRows as $row) {
                        if (isset($row->type) && $row->type === 'Section' && isset($row->Rows->Row)) {
                            $accountRows = is_array($row->Rows->Row) ? $row->Rows->Row : [$row->Rows->Row];
                            foreach($accountRows as $dataRow) {
                                if(isset($dataRow->type) && $dataRow->type === 'Data') {
                                    $name = $dataRow->ColData[getColIndexByName($detailReport, 'Name')]->value ?? '';
                                    $rowClass = (stripos($name, '(deleted)') !== false) ? 'class="deleted-row"' : '';
                                    echo "<tr {$rowClass}>";
                                    foreach($dataRow->ColData as $cell) { echo "<td>" . htmlspecialchars($cell->value) . "</td>"; }
                                    echo "</tr>";
                                }
                            }
                        }
                    }
                } else echo "<tr><td colspan='8' class='no-data-message'>No transactions found.</td></tr>";
                ?>
            </tbody>
        </table>
    <?php elseif ($detailAccountId): ?>
        <div class="report-header"><h3>General Ledger Report</h3><p class="no-data-message">Could not load transaction details. The API call may have failed.</p></div>
    <?php endif; ?>
</section>

<!-- CSS and JavaScript -->
<style>
    .pl-header-row td { font-weight: bold; background-color: #f8f8f8; } 
    .pl-total-row td, .pl-grand-total-row td { font-weight: bold; border-top: 1px solid #ccc; border-bottom: 2px solid #333; }
    .pl-grand-total-row td { background-color: #f2f2f2; } 
    .clickable-amount a { cursor: pointer; text-decoration: underline; color: #0056b3; display: block; padding: 5px; margin: -5px; }
    .clickable-amount a:hover { color: #003d80; }
    .pl-total-row .clickable-amount a { color: inherit; text-decoration: underline; }
    .pl-indent-0 { padding-left: 12px !important; } 
    .pl-indent-1 { padding-left: 30px !important; }
    .pl-indent-2 { padding-left: 50px !important; } 
    .pl-indent-3 { padding-left: 70px !important; } 
    .currency { text-align: right; }
    .deleted-row { text-decoration: line-through; color: #999; }
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const plTable = document.getElementById('pl-report');
    if (plTable) {
        plTable.addEventListener('click', function(e) {
            const link = e.target.closest('a');
            if (link) {
                e.preventDefault(); 
                const url = new URL(window.location.href);
                url.searchParams.set('account_id', link.dataset.accountId); 
                url.searchParams.set('account_name', link.dataset.accountName);
                url.searchParams.set('detail_start_date', link.dataset.startDate); 
                url.searchParams.set('detail_end_date', link.dataset.endDate);
                url.searchParams.set('month', link.dataset.month); 
                window.location.href = url.toString();
            }
        });
    }
    if (window.location.search.includes('account_id=')) {
        setTimeout(() => {
            const detailSection = document.getElementById('detail-report');
            if (detailSection) detailSection.scrollIntoView({ behavior: 'smooth', block: 'start' });
        }, 100);
    }
    
    const periodSelect = document.getElementById('report_period');
    const startDateInput = document.getElementById('start_date');
    const endDateInput = document.getElementById('end_date');
    periodSelect.addEventListener('change', function() {
        const selectedPeriod = this.value; const today = new Date();
        let startDate = new Date(); let endDate = new Date();
        const formatDate = (d) => d.toISOString().slice(0, 10);
        const finYearStartMonth = 6;

        switch (selectedPeriod) {
            case 'today': startDate = today; endDate = today; break;
            case 'this_week':
                const firstDay = today.getDate() - today.getDay() + (today.getDay() == 0 ? -6 : 1);
                startDate = new Date(today.setDate(firstDay));
                endDate = new Date(startDate); endDate.setDate(startDate.getDate() + 6);
                break;
            case 'this_month':
                startDate = new Date(today.getFullYear(), today.getMonth(), 1);
                endDate = new Date(today.getFullYear(), today.getMonth() + 1, 0);
                break;
            case 'this_quarter':
                const quarter = Math.floor(today.getMonth() / 3);
                startDate = new Date(today.getFullYear(), quarter * 3, 1);
                endDate = new Date(today.getFullYear(), quarter * 3 + 3, 0);
                break;
            case 'this_fin_year':
                let finYear = today.getFullYear();
                if (today.getMonth() < finYearStartMonth) { finYear--; }
                startDate = new Date(finYear, finYearStartMonth, 1);
                endDate = new Date(finYear + 1, finYearStartMonth, 0);
                break;
            case 'last_month':
                startDate = new Date(today.getFullYear(), today.getMonth() - 1, 1);
                endDate = new Date(today.getFullYear(), today.getMonth(), 0);
                break;
            case 'last_quarter':
                const lastQDate = new Date(today.getFullYear(), today.getMonth() - 3, 1);
                const lastQ = Math.floor(lastQDate.getMonth() / 3);
                startDate = new Date(lastQDate.getFullYear(), lastQ * 3, 1);
                endDate = new Date(lastQDate.getFullYear(), lastQ * 3 + 3, 0);
                break;
            case 'last_fin_year':
                let lastFinYear = today.getFullYear();
                if (today.getMonth() < finYearStartMonth) { lastFinYear--; }
                startDate = new Date(lastFinYear - 1, finYearStartMonth, 1);
                endDate = new Date(lastFinYear, finYearStartMonth, 0);
                break;
            case 'custom': default: return;
        }
        startDateInput.value = formatDate(startDate);
        endDateInput.value = formatDate(endDate);
    });
});
</script>