<?php
// qbo_sales_by_customer_detail_report.php
// FINAL POLISHED VERSION: Implements all features, including a new transaction type filter.

use QuickBooksOnline\API\DataService\DataService;

require_once __DIR__ . '/../../vendor/autoload.php';
require_once __DIR__ . '/../qbo_functions.php';
require_once __DIR__ . '/../config/qbo_config.php';
global $qboBaseConfig;

// This part of the PHP is unchanged
if (!function_exists('getQBOTransactionLink')) { function getQBOTransactionLink($txnId, $txnType, $baseUrl) { if (empty($txnId)) return '#'; $isSandbox = (strcasecmp($baseUrl, 'Development') == 0); $qboDomain = $isSandbox ? "https://app.sandbox.qbo.intuit.com/app/" : "https://app.qbo.intuit.com/app/"; $page = strtolower(str_replace(' ', '', $txnType)); if (in_array($page, ['invoice', 'salesreceipt', 'creditmemo', 'payment'])) { return $qboDomain . $page . "?txnId=" . htmlspecialchars($txnId); } return '#'; } }

$finYearStart = (date('m') < 7) ? date('Y-07-01', strtotime('-1 year')) : date('Y-07-01');
$startDate = isset($_GET['start_date']) && !empty($_GET['start_date']) ? $_GET['start_date'] : $finYearStart;
$endDate = isset($_GET['end_date']) && !empty($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-d');
$accountingMethod = isset($_GET['accounting_method']) ? $_GET['accounting_method'] : 'Accrual';

$dataService = null; $allCustomers = [];

$tokenStorageFile = __DIR__ . '/../tokens/qbo_token.json';
$currentTokens = file_exists($tokenStorageFile) ? json_decode(file_get_contents($tokenStorageFile), true) : null;
if ($currentTokens && !empty($currentTokens['access_token'])) {
    $qboBaseConfig['accessTokenKey'] = $currentTokens['access_token'];
    try { $dataService = DataService::Configure($qboBaseConfig); } catch (Exception $e) { $dataService = null; }
}

if ($dataService) {
    try {
        $allCustomers = $dataService->Query("SELECT Id, DisplayName FROM Customer WHERE Active = true ORDERBY DisplayName ASC");
    } catch (Exception $e) { $allCustomers = []; }
}
?>

<!-- NEW: CSS for the filter dropdown -->
<style>
    .multiselect-dropdown { position: relative; display: inline-block; font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif; }
    .multiselect-dropdown-btn { padding: 6px 12px; font-size: 14px; border: 1px solid #ccc; border-radius: 4px; background-color: #fff; cursor: pointer; width: 200px; text-align: left; }
    .multiselect-dropdown-content { display: none; position: absolute; background-color: #f9f9f9; min-width: 200px; box-shadow: 0px 8px 16px 0px rgba(0,0,0,0.2); z-index: 10; border-radius: 4px; padding: 5px; max-height: 250px; overflow-y: auto; }
    .multiselect-dropdown-content label { display: block; padding: 8px 10px; cursor: pointer; }
    .multiselect-dropdown-content label:hover { background-color: #f1f1f1; }
    .multiselect-dropdown-content input { margin-right: 10px; }
</style>

<div class="report-header">
    <h2>Sales by Customer Detail</h2>
    <p>For period: <?php echo htmlspecialchars(date("F j, Y", strtotime($startDate))) . " - " . htmlspecialchars(date("F j, Y", strtotime($endDate))); ?> </p>
</div>
<form method="get" class="filter-form">
    <input type="hidden" name="report" value="sales_by_customer_detail">
    <div>
        <label>Report period:</label>
        <select id="report_period_selector">
            <option value="This_month_to_date">This month to date</option><option value="This_Month">This Month</option>
            <option value="This_Fiscal_Year" selected>This Financial Year</option><option value="Last_Month">Last Month</option>
            <option value="Last_Fiscal_Year">Last Financial Year</option><option value="custom">Custom</option>
        </select>
    </div>
    <div><label>From:</label><input type="date" name="start_date" id="start_date" value="<?php echo htmlspecialchars($startDate); ?>"></div>
    <div><label>To:</label><input type="date" name="end_date" id="end_date" value="<?php echo htmlspecialchars($endDate); ?>"></div>
    <div>
        <label>Accounting Method:</label>
        <select name="accounting_method">
            <option value="Accrual" <?php if($accountingMethod == 'Accrual') echo 'selected'; ?>>Accrual</option>
            <option value="Cash" <?php if($accountingMethod == 'Cash') echo 'selected'; ?>>Cash</option>
        </select>
    </div>
    <!-- NEW: Transaction Type Filter HTML -->
    <div>
        <label>Transaction Type:</label>
        <div class="multiselect-dropdown" id="transaction-type-filter">
            <button type="button" class="multiselect-dropdown-btn">Select Types</button>
            <div class="multiselect-dropdown-content">
                <label><input type="checkbox" name="txn_type" value="Invoice" checked> Invoice</label>
                <label><input type="checkbox" name="txn_type" value="Journal Entry"> Journal Entry</label>
                <label><input type="checkbox" name="txn_type" value="Deposit"> Deposit</label>
            </div>
        </div>
    </div>
    <button type="submit">Run Report</button>
</form>

<div id="loading-indicator" style="text-align: center; padding: 40px; font-size: 1.2em;"><p>Loading report data, please wait...</p></div>
<section id="sales-detail-report" style="display:none;">
    <table>
        <thead id="report-header-row"></thead>
        <tbody id="report-body"></tbody>
        <tfoot id="report-footer"></tfoot>
    </table>
</section>

<script>
document.addEventListener('DOMContentLoaded', async function() {
    const reportSection = document.getElementById('sales-detail-report');
    const loadingIndicator = document.getElementById('loading-indicator');
    const tbody = document.getElementById('report-body');
    const tfoot = document.getElementById('report-footer');
    const headerRow = document.getElementById('report-header-row');
    
    // NEW: Get a reference to the transaction filter elements
    const transactionFilter = document.getElementById('transaction-type-filter');
    const transactionCheckboxes = transactionFilter.querySelectorAll('input[type="checkbox"]');

    const customers = <?php echo json_encode($allCustomers); ?>;
    const startDate = '<?php echo $startDate; ?>';
    const endDate = '<?php echo $endDate; ?>';
    const accountingMethod = '<?php echo $accountingMethod; ?>';
    const qboBaseUrl = '<?php echo $qboBaseConfig['baseUrl']; ?>';
    let grandTotal = 0;

    async function fetchDetails(customerId) {
        const url = `qbo_ajax_fetch_sales_detail.php?customer_id=${customerId}&start_date=${startDate}&end_date=${endDate}&accounting_method=${accountingMethod}`;
        const response = await fetch(url);
        return await response.json();
    }

    // NEW: Client-side filtering function
    function applyFilters() {
        const selectedTypes = Array.from(transactionCheckboxes)
            .filter(cb => cb.checked)
            .map(cb => cb.value);

        let newGrandTotal = 0;
        const allCustomerHeaders = document.querySelectorAll('.customer-header');

        allCustomerHeaders.forEach(header => {
            const targetClass = header.dataset.targetClass;
            const detailRows = document.querySelectorAll(`.${targetClass}`);
            let visibleRowsCount = 0;
            let customerSubtotal = 0;

            detailRows.forEach(row => {
                const transactionType = row.dataset.transactionType;
                const transactionAmount = parseFloat(row.dataset.transactionAmount) || 0;

                if (selectedTypes.includes(transactionType)) {
                    row.style.display = 'table-row'; // Show this row if it was hidden
                    visibleRowsCount++;
                    customerSubtotal += transactionAmount;
                } else {
                    row.style.display = 'none'; // Hide this row
                }
            });
            
            // Update the customer's total row
            const totalCell = header.querySelector('.currency');
            const toggleArrow = header.querySelector('.toggle-arrow');
            const transactionCountSpan = header.querySelector('strong');

            if (visibleRowsCount > 0) {
                header.style.display = 'table-row';
                totalCell.textContent = customerSubtotal.toFixed(2);
                transactionCountSpan.innerHTML = `<span class="toggle-arrow">${toggleArrow.textContent}</span> ${header.dataset.customerName} (${visibleRowsCount})`;
                newGrandTotal += customerSubtotal;
            } else {
                header.style.display = 'none';
            }
        });
        
        // Update the grand total in the footer
        const grandTotalCell = document.querySelector('#report-footer .currency');
        if (grandTotalCell) {
            grandTotalCell.textContent = newGrandTotal.toFixed(2);
        }
    }

    const processCustomer = async (customer) => {
        const result = await fetchDetails(customer.Id);
        if (result.success && result.data && result.data.Rows && result.data.Rows.Row) {
            if (headerRow.innerHTML === '') {
                result.data.Columns.Column.forEach(col => {
                    const th = document.createElement('th');
                    th.textContent = col.ColTitle;
                    if(col.ColType === 'Money') th.classList.add('currency');
                    headerRow.appendChild(th);
                });
            }
            
            const rows = Array.isArray(result.data.Rows.Row) ? result.data.Rows.Row : [result.data.Rows.Row];
            let customerHtml = '';
            let totalForCustomer = 0;
            let transactionCount = 0;
            let detailRowsHtml = '';

            rows.forEach(section => {
                if(section.Rows && section.Rows.Row){
                     const dataRows = Array.isArray(section.Rows.Row) ? section.Rows.Row : [section.Rows.Row];
                     dataRows.forEach(dataRow => {
                        if (dataRow.type === 'Data') {
                            const transactionType = dataRow.ColData[1]?.value || 'N/A';
                            const transactionAmount = dataRow.ColData[7]?.value || '0';
                            transactionCount++;

                            // MODIFIED: Add data attributes to the row for filtering
                            detailRowsHtml += `<tr class="sub-row detail-row-${customer.Id}" style="display:none;" data-transaction-type="${transactionType}" data-transaction-amount="${transactionAmount}">`;
                            dataRow.ColData.forEach((cell, idx) => {
                                const cellValue = cell.value || '';
                                // Display as plain text as previously requested
                                detailRowsHtml += `<td>${cellValue}</td>`;
                            });
                            detailRowsHtml += `</tr>`;
                        }
                     });
                }
                if(section.Summary){
                    totalForCustomer = parseFloat(section.Summary.ColData[7].value) || 0;
                }
            });
            
            // If customer has no transactions at all, don't add them to the DOM.
            if (transactionCount === 0) return;

            grandTotal += totalForCustomer;
            
            // MODIFIED: Add data attributes to the header row
            customerHtml += `<tr class="total-row customer-header" data-customer-name="${customer.DisplayName}" data-target-class="detail-row-${customer.Id}" style="cursor:pointer;">`;
            customerHtml += `<td colspan="7"><strong><span class="toggle-arrow">▶</span> ${customer.DisplayName} (${transactionCount})</strong></td>`;
            customerHtml += `<td class="currency"><strong>${totalForCustomer.toFixed(2)}</strong></td><td></td></tr>`;
            
            tbody.innerHTML += customerHtml + detailRowsHtml;
        }
    };

    if (customers.length > 0) {
        loadingIndicator.textContent = 'Loading customer data...';
        const allPromises = customers.map(c => processCustomer(c));
        await Promise.all(allPromises);

        loadingIndicator.style.display = 'none';
        reportSection.style.display = 'block';
        
        // MODIFIED: Apply initial filter state (Invoice only)
        applyFilters();

    } else {
        loadingIndicator.innerHTML = '<p>Could not load customer list.</p>';
    }

    // NEW: Event listener for the transaction filter dropdown
    transactionFilter.addEventListener('click', function(e) {
        const content = this.querySelector('.multiselect-dropdown-content');
        if (e.target.classList.contains('multiselect-dropdown-btn')) {
            content.style.display = content.style.display === 'block' ? 'none' : 'block';
        } else if (e.target.type === 'checkbox') {
            applyFilters();
        }
    });
    // Close dropdown if clicked outside
    document.addEventListener('click', function(e) {
        if (!transactionFilter.contains(e.target)) {
            transactionFilter.querySelector('.multiselect-dropdown-content').style.display = 'none';
        }
    });

    tbody.addEventListener('click', function(e) {
        const header = e.target.closest('.customer-header');
        if (header) {
            const targetClass = header.dataset.targetClass;
            const arrow = header.querySelector('.toggle-arrow');
            
            // MODIFIED: Only toggle rows that are visible based on the current filter
            const detailRows = document.querySelectorAll(`.${targetClass}[style*="display: table-row"]`);
            let isVisible = false;
            
            // Check visibility of the FIRST visible row
            if (detailRows.length > 0 && detailRows[0].classList.contains('expanded')) {
                isVisible = true;
            }

            // Toggle all visible rows
            document.querySelectorAll(`.${targetClass}`).forEach(row => {
                 if (row.style.display === 'table-row') { // only affect currently visible rows
                    if (isVisible) {
                        row.classList.remove('expanded');
                        // No need to change style, applyFilters already handles this
                    } else {
                        row.classList.add('expanded');
                    }
                 }
            });

            // This logic to expand/collapse is now a bit more complex with filtering.
            // A simpler approach for now is just to expand/collapse what is visible.
            // Let's refine this to be clearer. We'll simply toggle the *visibility* of the sub-rows
            const allDetailRows = document.querySelectorAll(`.${targetClass}`);
            allDetailRows.forEach(row => {
                const currentDisplay = row.style.display;
                if(currentDisplay !== 'none') { // only toggle if it's meant to be visible
                    // This part is now tricky. Let's simplify. Clicking expands/collapses ALL for that customer
                }
            });
            // Let's revert to a simpler expand/collapse for all children
            const allChildren = document.querySelectorAll(`.${targetClass}`);
            let anyRowVisible = false;
            allChildren.forEach(row => {
                if (row.style.display !== 'none') {
                    anyRowVisible = true;
                }
            });

            // Check if it's currently expanded or collapsed
            const isNowExpanded = header.classList.toggle('expanded');
            arrow.textContent = isNowExpanded ? '▼ ' : '▶ ';

            allChildren.forEach(row => {
                // The applyFilters function controls the base visibility.
                // This event listener only controls the expand/collapse state.
                const transactionType = row.dataset.transactionType;
                const selectedTypes = Array.from(transactionCheckboxes)
                    .filter(cb => cb.checked)
                    .map(cb => cb.value);

                if (selectedTypes.includes(transactionType)) {
                    row.style.display = isNowExpanded ? 'table-row' : 'none';
                } else {
                    row.style.display = 'none';
                }
            });
        }
    });
    
    // Date logic unchanged...
    document.getElementById('report_period_selector').addEventListener('change', function() {
        const selectedPeriod = this.value; const today = new Date();
        let startDate, endDate;
        const formatDate = (d) => d.toISOString().slice(0, 10);
        const finYearStartMonth = 6;
        switch (selectedPeriod) {
            case 'This_month_to_date': startDate = new Date(today.getFullYear(), today.getMonth(), 1); endDate = today; break;
            case 'This_Month': startDate = new Date(today.getFullYear(), today.getMonth(), 1); endDate = new Date(today.getFullYear(), today.getMonth() + 1, 0); break;
            case 'Last_Month': startDate = new Date(today.getFullYear(), today.getMonth() - 1, 1); endDate = new Date(today.getFullYear(), today.getMonth(), 0); break;
            case 'This_Fiscal_Year':
                let finYear = today.getFullYear();
                if (today.getMonth() < finYearStartMonth) finYear--;
                startDate = new Date(finYear, finYearStartMonth, 1);
                endDate = new Date(finYear + 1, finYearStartMonth, 0);
                break;
            case 'Last_Fiscal_Year':
                let lastFinYear = today.getFullYear();
                if (today.getMonth() < finYearStartMonth) lastFinYear--;
                startDate = new Date(lastFinYear - 1, finYearStartMonth, 1);
                endDate = new Date(lastFinYear, finYearStartMonth, 0);
                break;
            case 'custom': default: return;
        }
        if(startDate && endDate){
            document.getElementById('start_date').value = formatDate(startDate);
            document.getElementById('end_date').value = formatDate(endDate);
        }
    });
});
</script>