<?php
// qbo_supplier_cost_report.php
// REFACTORED VERSION: Uses the local database as the "master" for vendor names and fuzzy-matches QBO data to it.

// --- Essential Includes & Setup ---
require_once __DIR__ . '/dbconn.php';
require_once __DIR__ . '/../vendor/autoload.php';
require_once __DIR__ . '/qbo_functions.php';
require_once __DIR__ . '/config/qbo_config.php';
global $conn, $qboBaseConfig;
require_once __DIR__ . '/qbo_bootstrap.php';
global $qboUtil;

// --- Filter Logic ---
$filters = [
    'date_from' => $_GET['date_from'] ?? date('Y-m-d', strtotime('-90 days')),
    'date_to' => $_GET['date_to'] ?? date('Y-m-d'),
    'country' => $_GET['country'] ?? 'All'
];

// --- Data Fetching Logic ---
$final_report_data = [];
$error_message = null;

try {
    // --- Step 1: Fetch LOCAL vendors based on filters to build the MASTER search map ---
    // <<<<<<<<<<<<<<< START OF FIX >>>>>>>>>>>>>>>
    $local_supplier_data_for_map = [];
    $local_supplier_data_for_report = []; // New array to hold full vendor details for report building
    $sql_local_vendors = "SELECT vendorid, vendorName, vendorCountry FROM tdu_vendors WHERE vendorName IS NOT NULL AND vendorName != ''";
    $sql_local_vendors_params = [];
    $sql_local_vendors_types = "";

    if ($filters['country'] !== 'All') {
        $sql_local_vendors .= " AND vendorCountry = ?";
        $sql_local_vendors_params[] = $filters['country'];
        $sql_local_vendors_types .= "s";
    }

    $stmt_local_vendors = $conn->prepare($sql_local_vendors);
    if (!$stmt_local_vendors) {
        throw new Exception("Failed to prepare SQL statement for local vendors: " . $conn->error);
    }

    if (!empty($sql_local_vendors_params)) {
        $stmt_local_vendors->bind_param($sql_local_vendors_types, ...$sql_local_vendors_params);
    }

    $stmt_local_vendors->execute();
    $result_local_vendors = $stmt_local_vendors->get_result();

    while ($row = $result_local_vendors->fetch_assoc()) {
        // We need the full details for building the report later
        $local_supplier_data_for_report[$row['vendorName']] = $row; 
        // We only need the name for the search map
        $local_supplier_data_for_map[] = (object)['DisplayName' => $row['vendorName']];
    }
    $stmt_local_vendors->close();
    
    // The local vendor search map is now our "source of truth", correctly filtered
    $localVendorSearchMap = $qboUtil->buildReportSearchMap($local_supplier_data_for_map);
    // <<<<<<<<<<<<<<< END OF FIX >>>>>>>>>>>>>>>

    // --- Step 2: Fetch all QBO transactions (Bills and Payments) ---
    $qbo_balance_map = [];
    if ($qboUtil) {
        $vendor_ids_from_txns = [];
        $all_qbo_bills = []; $startPosition = 1;
        while (true) { /* ... Pagination logic for Bills ... */ $bill_query = "SELECT VendorRef, TotalAmt FROM Bill WHERE TxnDate >= '{$filters['date_from']}' AND TxnDate <= '{$filters['date_to']}' STARTPOSITION {$startPosition} MAXRESULTS 1000"; $billsBatch = $qboUtil->getDataService()->Query($bill_query); if (empty($billsBatch)) break; foreach ($billsBatch as $bill) { $all_qbo_bills[] = $bill; if (isset($bill->VendorRef)) $vendor_ids_from_txns[] = $bill->VendorRef; } if (count($billsBatch) < 1000) break; $startPosition += 1000; }
        $all_qbo_payments = []; $startPosition = 1;
        while (true) { /* ... Pagination logic for Payments ... */ $payment_query = "SELECT VendorRef, TotalAmt FROM BillPayment WHERE TxnDate >= '{$filters['date_from']}' AND TxnDate <= '{$filters['date_to']}' STARTPOSITION {$startPosition} MAXRESULTS 1000"; $paymentsBatch = $qboUtil->getDataService()->Query($payment_query); if (empty($paymentsBatch)) break; foreach ($paymentsBatch as $payment) { $all_qbo_payments[] = $payment; if (isset($payment->VendorRef)) $vendor_ids_from_txns[] = $payment->VendorRef; } if (count($paymentsBatch) < 1000) break; $startPosition += 1000; }
        
        $vendor_id_to_name_map = [];
        $unique_vendor_ids = array_unique(array_filter($vendor_ids_from_txns));
        if (!empty($unique_vendor_ids)) { $id_list_string = "'" . implode("','", $unique_vendor_ids) . "'"; $vendor_query = "SELECT Id, DisplayName FROM Vendor WHERE Id IN ({$id_list_string})"; $qbo_vendors = $qboUtil->getDataService()->Query($vendor_query); foreach ($qbo_vendors as $vendor) { $vendor_id_to_name_map[$vendor->Id] = $vendor->DisplayName; } }
        
        foreach ($all_qbo_bills as $bill) { $vendorName = $vendor_id_to_name_map[$bill->VendorRef] ?? null; if ($vendorName) { $qbo_balance_map[$vendorName] = ($qbo_balance_map[$vendorName] ?? 0) + (float)$bill->TotalAmt; } }
        foreach ($all_qbo_payments as $payment) { $vendorName = $vendor_id_to_name_map[$payment->VendorRef] ?? null; if ($vendorName) { $qbo_balance_map[$vendorName] = ($qbo_balance_map[$vendorName] ?? 0) - (float)$payment->TotalAmt; } }
    }

    // --- Step 3: Fetch Local Pax Data ---
    $pax_map = [];
    $sql_params = []; $sql_types = "";
    $sql_pax = "SELECT v.vendorName, SUM(q.adults + q.children) AS total_pax FROM tdu_vendors v JOIN tdu_products p ON v.vendorid = p.vendorid JOIN vtiger_inventoryproductrel i ON p.productid = i.productid JOIN vtiger_quotes q ON i.id = q.quoteid JOIN vtiger_quotescf qcf ON q.quoteid = qcf.quoteid WHERE qcf.cf_1162 BETWEEN ? AND ? ";
    $sql_params[] = $filters['date_from']; $sql_params[] = $filters['date_to']; $sql_types .= "ss";
    if ($filters['country'] !== 'All') { $sql_pax .= " AND v.vendorCountry = ?"; $sql_params[] = $filters['country']; $sql_types .= "s"; }
    $sql_pax .= " GROUP BY v.vendorName";
    $stmt_pax = $conn->prepare($sql_pax);
    if ($stmt_pax) { $stmt_pax->bind_param($sql_types, ...$sql_params); $stmt_pax->execute(); $pax_data = $stmt_pax->get_result()->fetch_all(MYSQLI_ASSOC); $stmt_pax->close(); }
    foreach($pax_data as $pax_row) {
        $pax_map[$pax_row['vendorName']] = (int)$pax_row['total_pax'];
    }
    
    // --- Step 4: Aggregate all data, using the now-filtered local database list as the master list ---
    $aggregated_report_data = [];
    // Initialize the report, but ONLY with local vendors who have a pax count > 0
    foreach ($local_supplier_data_for_report as $localName => $vendorDetails) {
        $paxCount = $pax_map[$localName] ?? 0;
        // --- THIS IS THE NEW CONDITION ---
        if ($paxCount > 0) {
            $aggregated_report_data[$localName] = [
                'vendor_id' => $vendorDetails['vendorid'],
                'vendor_name' => $localName,
                'qbo_total_amount' => 0,
                'total_pax' => $paxCount,
                'country' => $vendorDetails['vendorCountry']
            ];
        }
    }
    
    // Now, loop through the QBO data and map it back to the local master list
    foreach ($qbo_balance_map as $qboVendorName => $totalAmount) {
        // Use fuzzy matching to find the official local name for this QBO vendor
        $matchedLocalName = $qboUtil->findBestEntityMatch($qboVendorName, $localVendorSearchMap);
        
        if ($matchedLocalName && isset($aggregated_report_data[$matchedLocalName])) {
            // Found a match, add the amount to the existing record
            $aggregated_report_data[$matchedLocalName]['qbo_total_amount'] += $totalAmount;
        } else {
            // This QBO vendor could not be matched to a local vendor.
            // We can optionally add it as a new "unmatched" row if desired.
            // For now, we will ignore it to keep the report clean based on the local master list.
        }
    }
    
    $final_report_data = array_values($aggregated_report_data);

} catch (Exception $e) {
    $error_message = "An error occurred: " . htmlspecialchars($e->getMessage());
}
?>

<!-- ... (The entire HTML/JavaScript section remains unchanged) ... -->
<!-- ========================================================================= -->
<!-- UI, LIBRARIES, TABLE, AND JAVASCRIPT (No changes needed below this line)  -->
<!-- ========================================================================= -->
<link rel="stylesheet" type="text/css" href="https://cdn.datatables.net/1.13.6/css/jquery.dataTables.min.css">
<script type="text/javascript" charset="utf8" src="https://code.jquery.com/jquery-3.7.0.js"></script>
<script type="text/javascript" charset="utf8" src="https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"></script>

<style>
    #supplierReportTable_filter { margin-bottom: 10px; } tr.dt-hasChild td.dt-control { background-image: url('https://datatables.net/examples/resources/details_close.png'); } td.dt-control { background: url('https://datatables.net/examples/resources/details_open.png') no-repeat center center; cursor: pointer; } .detail-container { padding: 10px 15px; background-color: #fafafa; display: flex; gap: 30px;} .detail-section { flex: 1; } .detail-container h4 { margin-top: 0; margin-bottom: 10px; border-bottom: 1px solid #ddd; padding-bottom: 5px; } .detail-table { width: 100%; font-size: 0.9em; } .detail-table th, .detail-table td { border: none; border-bottom: 1px solid #e9ecef; padding: 6px 4px; } .payment-row, .status-paid { color: #2d8a32; } .status-open { color: #0077c5; } .status-overdue { color: #d93025; font-weight: bold; }
</style>

<div class="report-header"><h2>Supplier Cost Report</h2></div>

<?php if ($error_message): ?>
    <div class='no-data-message' style='color:red; text-align:center; padding: 10px; border: 1px solid red; background: #ffebee;'><?= $error_message ?></div>
<?php endif; ?>

<form method="GET" class="filter-form">
    <input type="hidden" name="report" value="supplier_cost">
    <div><label for="date_from">Transaction Date From:</label><input type="date" id="date_from" name="date_from" value="<?= htmlspecialchars($filters['date_from']) ?>"></div>
    <div><label for="date_to">To:</label><input type="date" id="date_to" name="date_to" value="<?= htmlspecialchars($filters['date_to']) ?>"></div>
    <div><label for="country">Country:</label><select id="country" name="country"><option value="All">All Countries</option><option value="Australia" <?= $filters['country'] === 'Australia' ? 'selected' : '' ?>>Australia</option><option value="New Zealand" <?= $filters['country'] === 'New Zealand' ? 'selected' : '' ?>>New Zealand</option></select></div>
    <button type="submit">Filter</button>
</form>

<table id="supplierReportTable" class="display" style="width:100%">
    <thead><tr><th></th><th>Supplier Name</th><th>Outstanding Balance (from QBO)</th><th>Pax (by Travel Date)</th><th>Country</th></tr></thead>
    <tfoot><tr><th></th><th>Page Total:</th><th></th><th></th><th></th></tr></tfoot>
</table>

<script>
    const reportData = <?= json_encode($final_report_data); ?>;
    $(document).ready(function() {
        const table = $('#supplierReportTable').DataTable({
            data: reportData,
            columns: [ { className: 'dt-control', orderable: false, data: null, defaultContent: '' }, { data: 'vendor_name' }, { data: 'qbo_total_amount', render: function(data) { return '$' + parseFloat(data).toFixed(2).replace(/\d(?=(\d{3})+\.)/g, '$&,'); }, className: 'currency' }, { data: 'total_pax' }, { data: 'country' } ],
            order: [[1, 'asc']], paging: true, pageLength: 50, searching: true, info: true,
            footerCallback: function (row, data, start, end, display) { var api = this.api(); const total = (colIndex) => api.column(colIndex, { page: 'current' }).data().reduce((a, b) => a + parseFloat(b || 0), 0); $(api.column(1).footer()).html('Page Total:'); $(api.column(2).footer()).html('$' + total(2).toFixed(2).replace(/\d(?=(\d{3})+\.)/g, '$&,')); $(api.column(3).footer()).html(total(3)); }
        });
        $('#supplierReportTable tbody').on('click', 'td.dt-control', async function () {
            var tr = $(this).closest('tr'); var row = table.row(tr);
            if (row.child.isShown()) { row.child.hide(); tr.removeClass('dt-hasChild'); }
            else {
                tr.addClass('dt-hasChild'); row.child('<div>Loading details...</div>').show();
                const rowData = row.data(); const vendorId = rowData.vendor_id; const vendorName = rowData.vendor_name;
                const dateFrom = $('#date_from').val(); const dateTo = $('#date_to').val();
                try {
                    const apiUrl = `api_get_supplier_transactions.php?id=${vendorId}&name=${encodeURIComponent(vendorName)}&from=${dateFrom}&to=${dateTo}`;
                    const response = await fetch(apiUrl); const data = await response.json();
                    if (!response.ok || data.error) throw new Error(data.error || `API returned status: ${response.status}`);
                    const childHtml = formatChildRow(data); row.child(childHtml).show();
                } catch (error) { console.error('Failed to fetch details:', error); row.child(`<div style="color:red;">Failed to load details: ${error.message}</div>`).show(); }
            }
        });
    });
    function formatChildRow(data) {
        let transactionHtml = '<h4>Transaction Breakdown (QBO)</h4>';
        if (data.transaction_history && data.transaction_history.length > 0) {
            transactionHtml += '<table class="detail-table"><thead><tr><th>Date</th><th>Type</th><th>#</th><th>Quote #</th><th style="text-align:right;">Amount</th></tr></thead><tbody>';
            data.transaction_history.forEach(txn => { const amount = parseFloat(txn.amount) || 0; const typeString = (txn.type || '').toLowerCase(); const isPayment = typeString.includes('payment'); const amountDisplay = isPayment ? `-$${amount.toFixed(2)}` : `+$${amount.toFixed(2)}`; const docNum = txn.doc_num || 'N/A'; const quoteNum = txn.quote_no || 'N/A'; transactionHtml += `<tr class="${isPayment ? 'payment-row' : ''}"><td>${txn.date || 'N/A'}</td><td>${txn.type || 'N/A'}</td><td>${docNum}</td><td>${quoteNum}</td><td style="text-align:right;">${amountDisplay}</td></tr>`; });
            transactionHtml += '</tbody></table>';
        } else { transactionHtml += '<p>No transaction history found in QBO for this period.</p>'; }
        let quoteHtml = '<h4>Quote Breakdown</h4>';
        if (data.quote_details && data.quote_details.length > 0) {
            quoteHtml += '<table class="detail-table"><thead><tr><th>Quote #</th><th>Travel Date</th><th>Subject</th><th>Status</th><th>Pax</th></tr></thead><tbody>';
            data.quote_details.forEach(quote => { quoteHtml += `<tr><td><a href="quote.php?opt=summary&quoteNumber=${encodeURIComponent(quote.quote_no)}" target="_blank">${quote.quote_no}</a></td><td>${quote.travel_date}</td><td>${quote.subject}</td><td>${quote.quotestage || 'N/A'}</td><td>${quote.pax}</td></tr>`; });
            quoteHtml += '</tbody></table>';
        } else { quoteHtml += '<p>No quotes featuring this supplier\'s products found for the selected travel date range.</p>'; }
        return `<div class="detail-container"><div class="detail-section">${transactionHtml}</div><div class="detail-section">${quoteHtml}</div></div>`;
    }
</script>