<?php
// qbo_supplier_expense_detail_report.php
// FINAL CORRECTED VERSION: Fixes the fatal error by including the missing helper function.

use QuickBooksOnline\API\DataService\DataService;
use QuickBooksOnline\API\ReportService\ReportService;
require_once __DIR__ . '/../../vendor/autoload.php';
require_once __DIR__ . '/../qbo_functions.php';
require_once __DIR__ . '/../config/qbo_config.php';
global $qboBaseConfig;

// --- CRITICAL FIX: Add the missing helper function ---
if (!function_exists('getColIndexByName')) {
    function getColIndexByName($report, $colName) {
        if (!isset($report->Columns->Column) || !is_array($report->Columns->Column)) return -1;
        foreach ($report->Columns->Column as $index => $column) {
            if (isset($column->ColTitle) && strcasecmp($column->ColTitle, $colName) == 0) return $index;
        }
        return -1;
    }
}
if (!function_exists('getQBOTransactionLink')) { function getQBOTransactionLink($txnId, $txnType, $baseUrl) { if (empty($txnId)) return '#'; $isSandbox = (strcasecmp($baseUrl, 'Development') == 0); $qboDomain = $isSandbox ? "https://app.sandbox.qbo.intuit.com/app/" : "https://app.qbo.intuit.com/app/"; $page = strtolower(str_replace(' ', '', $txnType)); if (in_array($page, ['bill', 'purchase', 'expense', 'check'])) { return $qboDomain . $page . "?txnId=" . htmlspecialchars($txnId); } return '#'; } }

// --- Report Filters & Variables ---
$vendorId = isset($_GET['vendor_id']) ? $_GET['vendor_id'] : null;
$vendorName = isset($_GET['vendor_name']) ? $_GET['vendor_name'] : 'N/A';
$finYearStart = (date('m') < 7) ? date('Y-07-01', strtotime('-1 year')) : date('Y-07-01');
$startDate = isset($_GET['start_date']) && !empty($_GET['start_date']) ? $_GET['start_date'] : $finYearStart;
$endDate = isset($_GET['end_date']) && !empty($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-d');
$accountingMethod = isset($_GET['accounting_method']) ? $_GET['accounting_method'] : 'Accrual';
$dataService = null; $report = null; $groupedTransactions = []; $grandTotal = 0;

if (!$vendorId) { exit("Error: Vendor ID is required."); }

// --- QBO Connection & Token Refresh ---
$tokenStorageFile = __DIR__ . '/../tokens/qbo_token.json';
$currentTokens = file_exists($tokenStorageFile) ? json_decode(file_get_contents($tokenStorageFile), true) : null;
if ($currentTokens && !empty($currentTokens['access_token'])) {
    $qboBaseConfig['accessTokenKey'] = $currentTokens['access_token'];
    try { $dataService = DataService::Configure($qboBaseConfig); } catch (Exception $e) { $dataService = null; }
} else { exit("QBO Authentication failed."); }

// --- Data Fetching Logic ---
$reportFetchSuccess = false;
if ($dataService) {
    try {
        $reportService = new ReportService($dataService->getServiceContext());
        $reportService->setStartDate($startDate);
        $reportService->setEndDate($endDate);
        $reportService->setAccountingMethod($accountingMethod);
        $reportService->setVendor($vendorId);
        $report = $reportService->executeReport("TransactionList");
        if ($report) { $reportFetchSuccess = true; }
    } catch (Exception $e) { /* Suppress error */ }
}

// Group transactions by Account
if ($reportFetchSuccess && isset($report->Rows->Row)) {
    $rows = is_array($report->Rows->Row) ? $report->Rows->Row : [$report->Rows->Row];
    foreach($rows as $row) {
        if(isset($row->type) && $row->type === 'Data') {
            $account = $row->ColData[getColIndexByName($report, 'Account')]->value ?? 'Uncategorized';
            if(!isset($groupedTransactions[$account])) {
                $groupedTransactions[$account] = ['transactions' => [], 'total' => 0];
            }
            $groupedTransactions[$account]['transactions'][] = $row->ColData;
            $amount = (float)($row->ColData[getColIndexByName($report, 'Amount')]->value ?? 0);
            $groupedTransactions[$account]['total'] += $amount;
            $grandTotal += $amount;
        }
    }
}
?>
<div class="report-header">
    <p><a href="?report=supplier_expense&start_date=<?php echo $startDate; ?>&end_date=<?php echo $endDate; ?>&accounting_method=<?php echo $accountingMethod; ?>">« Back to Summary</a></p>
    <h2>Transaction Report</h2>
    <h3><?php echo htmlspecialchars($vendorName); ?></h3>
    <p>For period: <?php echo htmlspecialchars(date("F j, Y", strtotime($startDate))) . " - " . htmlspecialchars(date("F j, Y", strtotime($endDate))); ?> </p>
</div>
<form method="get" class="filter-form">
    <input type="hidden" name="report" value="supplier_expense_detail">
    <input type="hidden" name="vendor_id" value="<?php echo htmlspecialchars($vendorId); ?>">
    <input type="hidden" name="vendor_name" value="<?php echo htmlspecialchars($vendorName); ?>">
    <div><label>Report period:</label><select id="report_period_selector"><option value="This_Fiscal_Year" selected>This Financial Year</option><option value="Last_Fiscal_Year">Last Financial Year</option><option value="This_Month">This Month</option><option value="Last_Month">Last Month</option><option value="custom">Custom</option></select></div>
    <div><label>From:</label><input type="date" name="start_date" id="start_date" value="<?php echo htmlspecialchars($startDate); ?>"></div>
    <div><label>To:</label><input type="date" name="end_date" id="end_date" value="<?php echo htmlspecialchars($endDate); ?>"></div>
    <div><label>Method:</label><select name="accounting_method"><option value="Accrual" <?php if($accountingMethod == 'Accrual') echo 'selected';?>>Accrual</option><option value="Cash" <?php if($accountingMethod == 'Cash') echo 'selected';?>>Cash</option></select></div>
    <button type="submit">Run Report</button>
</form>
<table>
    <thead><tr>
        <th>Transaction Date</th><th>Transaction Type</th><th>No.</th><th>Name</th><th>Memo/Description</th><th>Split</th><th class="currency">Amount</th><th class="currency">Balance</th>
    </tr></thead>
    <tbody>
        <?php if (!empty($groupedTransactions)): ?>
            <?php foreach($groupedTransactions as $accountName => $group): ?>
                <tr class="total-row"><td colspan="8"><strong><?php echo htmlspecialchars($accountName); ?></strong></td></tr>
                <?php foreach($group['transactions'] as $txnData): ?>
                    <tr>
                        <?php
                        $txnType = $txnData[getColIndexByName($report, 'Transaction Type')]->value ?? '';
                        $txnId = $txnData[getColIndexByName($report, 'Transaction Type')]->id ?? null;
                        $link = getQBOTransactionLink($txnId, $txnType, $qboBaseConfig['baseUrl']);
                        ?>
                        <td><?php echo htmlspecialchars($txnData[getColIndexByName($report, 'Date')]->value ?? ''); ?></td>
                        <td><a href="<?php echo $link; ?>" target="_blank"><?php echo htmlspecialchars($txnType); ?></a></td>
                        <td><?php echo htmlspecialchars($txnData[getColIndexByName($report, 'No.')]->value ?? ''); ?></td>
                        <td><?php echo htmlspecialchars($txnData[getColIndexByName($report, 'Name')]->value ?? ''); ?></td>
                        <td><?php echo htmlspecialchars($txnData[getColIndexByName($report, 'Memo/Description')]->value ?? ''); ?></td>
                        <td><?php echo htmlspecialchars($txnData[getColIndexByName($report, 'Split')]->value ?? ''); ?></td>
                        <td class="currency"><?php echo number_format((float)($txnData[getColIndexByName($report, 'Amount')]->value ?? 0), 2); ?></td>
                        <td class="currency"><?php echo number_format((float)($txnData[getColIndexByName($report, 'Balance')]->value ?? 0), 2); ?></td>
                    </tr>
                <?php endforeach; ?>
                <tr class="total-row">
                    <td colspan="6"><strong>Total for <?php echo htmlspecialchars($accountName); ?></strong></td>
                    <td class="currency"><strong><?php echo number_format($group['total'], 2); ?></strong></td>
                    <td></td>
                </tr>
            <?php endforeach; ?>
        <?php else: ?>
            <tr><td colspan="8" class="no-data-message">No transactions found for this supplier in the selected period.</td></tr>
        <?php endif; ?>
    </tbody>
    <tfoot>
        <tr class="total-row">
            <td colspan="6"><strong>TOTAL</strong></td>
            <td class="currency"><strong><?php echo number_format($grandTotal, 2); ?></strong></td>
            <td></td>
        </tr>
    </tfoot>
</table>
<script>
    document.getElementById('report_period_selector')?.addEventListener('change', function() {
        const selectedPeriod = this.value; const today = new Date();
        let startDate, endDate;
        const formatDate = (d) => { const pad = (num) => num.toString().padStart(2, '0'); return `${d.getFullYear()}-${pad(d.getMonth() + 1)}-${pad(d.getDate())}`; }
        const finYearStartMonth = 6;
        switch (selectedPeriod) {
            case 'This_Month': startDate = new Date(today.getFullYear(), today.getMonth(), 1); endDate = new Date(today.getFullYear(), today.getMonth() + 1, 0); break;
            case 'This_Fiscal_Year':
                let finYear = today.getFullYear(); if (today.getMonth() < finYearStartMonth) finYear--;
                startDate = new Date(finYear, finYearStartMonth, 1); endDate = new Date(finYear + 1, finYearStartMonth, 0); break;
            case 'Last_Month': startDate = new Date(today.getFullYear(), today.getMonth() - 1, 1); endDate = new Date(today.getFullYear(), today.getMonth(), 0); break;
            case 'Last_Fiscal_Year':
                let lastFinYear = today.getFullYear(); if (today.getMonth() < finYearStartMonth) lastFinYear--;
                startDate = new Date(lastFinYear - 1, finYearStartMonth, 1); endDate = new Date(lastFinYear, finYearStartMonth, 0); break;
            case 'custom': default: return;
        }
        if(startDate && endDate){
            document.getElementById('start_date').value = formatDate(startDate);
            document.getElementById('end_date').value = formatDate(endDate);
        }
    });
</script>