<?php
// ai_email_functions.php

require_once "dbconn.php";
require_once "ai_keys.php";
require_once "file_extract_functions.php"; // Assumed to contain getFileTypeFromString, extractTextFromPDF, extractTextfromWordDoc, etc.

$log_file_path = __DIR__ . '/email_ai_functions_error.log'; // Or a more central log path
ini_set('error_log', $log_file_path);
ini_set('display_errors', '0'); // For production
ini_set('log_errors', '1');
error_reporting(E_ALL); // Or your preferred level

// define $common_terms globally, pass as param, or define inside function if static
global $common_org_filter_terms; // Or pass as argument
$common_org_filter_terms = [
    'pvt ltd', 'private limited', 'ltd', 'limited', 'inc', 'incorporated', 'llc',
    'corp', 'corporation', 'tours & travels', 'tours and travels', 'travels', 'tours',
    'group', 'solutions', 'services', 'enterprises', 'company', 'co', 'agency',
    'international', 'global', 'national', 'associates', 'partners', 'sons',
    '(india)', '(australia)', 'operations', 'pty', // etc.
];

global $common_vendor_filter_terms; // Or pass as argument
$common_vendor_filter_terms = [
    'pvt ltd', 'private limited', 'ltd', 'limited', 'inc', 'incorporated', 'llc',
    'corp', 'corporation', 'tours & travels', 'tours and travels', 'travels', 'tours',
    'group', 'solutions', 'services', 'enterprises', 'company', 'co', 'agency',
    'international', 'global', 'national', 'associates', 'partners', 'sons',
    '(india)', '(australia)', 'operations', 'pty', // etc.
];

/**
 * Creates the 'tdu_tai_assigned_role' table if it does not already exist.
 * This function is intended for one-time setup or to ensure the table exists.
 * It's recommended to call this manually via a setup script or comment out/delete
 * after the table is confirmed to be created to avoid running it on every request.
 *
 * @param mysqli $conn The database connection object.
 * @return bool True on success (table exists or was created), false on failure.
 */
function setup_tdu_tai_assigned_role_table(mysqli $conn): bool {
    if (!$conn || $conn->connect_error) {
        error_log("setup_tdu_tai_assigned_role_table: Database connection failed: " . ($conn->connect_error ?? 'Unknown DB error'));
        echo "Error: Database connection failed. Cannot setup table.\n";
        return false;
    }

    $sql_create_table = "
    CREATE TABLE IF NOT EXISTS tdu_tai_assigned_role (
        user_id INT NOT NULL,
        tag_id INT NOT NULL,
        PRIMARY KEY (user_id, tag_id),
        FOREIGN KEY (user_id) REFERENCES vtiger_users(id) ON DELETE CASCADE ON UPDATE CASCADE
        -- If you have a central tags table (e.g., vtiger_freetags or a custom one)
        -- you might want to add a foreign key for tag_id as well:
        -- FOREIGN KEY (tag_id) REFERENCES your_tags_table(id) ON DELETE CASCADE ON UPDATE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    ";

    if (mysqli_query($conn, $sql_create_table)) {
        // Check if the table was actually created or already existed
        // (mysqli_query for CREATE TABLE IF NOT EXISTS returns true even if it already existed)
        $check_sql = "SHOW TABLES LIKE 'tdu_tai_assigned_role'";
        $result = mysqli_query($conn, $check_sql);
        if ($result && mysqli_num_rows($result) > 0) {
            // echo "Notice: Table 'tdu_tai_assigned_role' checked/created successfully.\n"; // For CLI or direct script run
            // error_log("setup_tdu_tai_assigned_role_table: Table 'tdu_tai_assigned_role' checked/created successfully."); // For server logs
            return true;
        } else {
            // This case is unlikely if CREATE TABLE IF NOT EXISTS succeeded but good for thoroughness
            error_log("setup_tdu_tai_assigned_role_table: CREATE TABLE query succeeded but table 'tdu_tai_assigned_role' not found afterwards.");
            echo "Error: Table 'tdu_tai_assigned_role' creation reported success, but table not found.\n";
            return false;
        }
    } else {
        error_log("setup_tdu_tai_assigned_role_table: Error creating table 'tdu_tai_assigned_role': " . mysqli_error($conn));
        echo "Error: Failed to create table 'tdu_tai_assigned_role'. Check error log. Details: " . mysqli_error($conn) . "\n";
        return false;
    }
}

/**
 * Normalizes a vendor name specifically for creating SQL LIKE search patterns.
 * - Converts to lowercase.
 * - Removes common vendor terms.
 * - Replaces space sequences with '%' wildcard.
 * - Keeps alphanumeric characters and '%'.
 */
function normalize_vendor_name_for_sql_search(string $name): string {
    global $common_vendor_filter_terms;
    $terms_to_remove = $common_vendor_filter_terms ?? [];

    $processed_name = strtolower(trim($name));

    // Remove common terms using word boundaries
    foreach ($terms_to_remove as $term) {
        $processed_name = preg_replace('/\b' . preg_quote(strtolower($term), '/') . '\b/u', '', $processed_name);
    }

    // Remove punctuation except for what might be part of a name or wildcard
    // Keep letters, numbers, spaces, and allow % (though we add it next)
    $processed_name = preg_replace('/[^\p{L}\p{N}\s]/u', '', $processed_name);

    // Consolidate multiple spaces/tabs into a single space, then trim
    $processed_name = trim(preg_replace('/\s+/', ' ', $processed_name));
    
    // Replace remaining spaces with '%' for LIKE query
    $processed_name = str_replace(' ', '%', $processed_name);

    return $processed_name;
}

function getBestMatchingVendor(string $extracted_name, mysqli $conn): ?string {
    // Input validation
    if (empty(trim($extracted_name)) || strtolower(trim($extracted_name)) === 'n/a') {
        error_log("getBestMatchingVendor (SQL Only): Input name is empty or N/A for '{$extracted_name}'.");
        return null;
    }

    $original_trimmed_extracted_name = trim($extracted_name);

    // 1. Normalize for SQL LIKE search
    $sql_search_term_parts = normalize_vendor_name_for_sql_search($original_trimmed_extracted_name);
    if (empty($sql_search_term_parts)) {
        error_log("getBestMatchingVendor (SQL Only): Normalized SQL search term is empty for '{$original_trimmed_extracted_name}'.");
        return null;
    }
    $search_pattern = '%' . $sql_search_term_parts . '%';

    // 2. Create a condensed search pattern
    $condensed_name = strtolower(trim($original_trimmed_extracted_name));
    foreach (($GLOBALS['common_vendor_filter_terms'] ?? []) as $term) {
        $condensed_name = preg_replace('/\b' . preg_quote(strtolower($term), '/') . '\b/u', '', $condensed_name);
    }
    $condensed_name = preg_replace('/[^\p{L}\p{N}]/u', '', $condensed_name);
    $condensed_search_pattern = '%' . $condensed_name . '%';

    // 3. Fetch potential vendors using LIKE and an ordering preference
    $sql = "SELECT vendorName 
            FROM tdu_vendors 
            WHERE vendorName IS NOT NULL AND vendorName != ''
              AND (
                LOWER(vendorName) LIKE ? 
                OR LOWER(REPLACE(vendorName, ' ', '')) LIKE ?
              )
            ORDER BY 
              CASE
                WHEN LOWER(vendorName) = LOWER(?) THEN 1 -- Exact match (case-insensitive) to original
                WHEN LOWER(REPLACE(vendorName, ' ', '')) = ? THEN 2 -- Exact match to condensed original
                WHEN LOWER(vendorName) LIKE ? THEN 3 -- Starts with the core part of the search term (e.g., 'sunlover%')
                WHEN INSTR(LOWER(vendorName), ?) > 0 THEN 4 -- Contains the core part of the search term (e.g., contains 'sunlover')
                ELSE 5
              END,
              LENGTH(vendorName) ASC -- Prefer shorter matches if other things are equal
            LIMIT 1"; // Fetch only the single best candidate based on SQL ordering

    $stmt = $conn->prepare($sql);
    if (!$stmt) {
        error_log("getBestMatchingVendor (SQL Only): DB Prepare failed - " . $conn->error);
        return null;
    }

    $lower_original_extracted_name = strtolower($original_trimmed_extracted_name);
    $core_search_term_for_ordering = str_replace('%', '', $sql_search_term_parts); // e.g., "sunlover" from "sun%lover"
    $starts_with_pattern_for_ordering = $core_search_term_for_ordering . '%';
    // For the condensed match in ORDER BY
    $condensed_original_for_ordering = preg_replace('/[^\p{L}\p{N}]/u', '', strtolower($original_trimmed_extracted_name));


    // Parameters for bind_param:
    // 1. $search_pattern (for main LIKE)
    // 2. $condensed_search_pattern (for condensed LIKE)
    // 3. $lower_original_extracted_name (for ORDER BY exact original)
    // 4. $condensed_original_for_ordering (for ORDER BY exact condensed)
    // 5. $starts_with_pattern_for_ordering (for ORDER BY starts with core)
    // 6. $core_search_term_for_ordering (for ORDER BY contains core)
    $stmt->bind_param("ssssss", 
        $search_pattern, 
        $condensed_search_pattern, 
        $lower_original_extracted_name,
        $condensed_original_for_ordering, 
        $starts_with_pattern_for_ordering,
        $core_search_term_for_ordering
    );
    
    $best_db_match = null;
    if ($stmt->execute()) {
        $result = $stmt->get_result();
        if ($row = $result->fetch_assoc()) {
            $best_db_match = $row['vendorName'];
        }
        if ($result instanceof mysqli_result) $result->free();
    } else {
        error_log("getBestMatchingVendor (SQL Only): DB Execute failed - " . $stmt->error);
        $stmt->close();
        return null;
    }
    $stmt->close();

    if ($best_db_match) {
        error_log("getBestMatchingVendor (SQL Only): SQL query returned '{$best_db_match}' as the best match for '{$original_trimmed_extracted_name}'.");
        return $best_db_match;
    } else {
        error_log("getBestMatchingVendor (SQL Only): No matches found in DB by SQL for '{$original_trimmed_extracted_name}' (Patterns: '{$search_pattern}', '{$condensed_search_pattern}').");
        return null;
    }
}

/**
 * Normalizes vendor names for similarity matching.
 * Adapt rules as needed (e.g., different common terms, maybe keep spaces).
 * For now, using similar logic to organization normalization including space removal.
 */
function normalize_vendor_name_for_similarity(string $name): string {
    global $common_vendor_filter_terms; // Use if defined
    $terms_to_remove = $common_vendor_filter_terms ?? []; // Example if using separate vendor terms

    $processed_name = strtolower(trim($name));
    $processed_name = preg_replace('/[^\p{L}\p{N}\s]/u', '', $processed_name); // Keep letters, numbers, spaces

    // Example: No common term removal for vendors (adjust if needed)
    foreach ($terms_to_remove as $term) {
         $processed_name = preg_replace('/\b' . preg_quote(strtolower($term), '/') . '\b/u', '', $processed_name);
     }

    $processed_name = trim(preg_replace('/\s+/', ' ', $processed_name)); // Consolidate spaces
    $processed_name = str_replace(' ', '', $processed_name); // Remove all spaces
    return $processed_name;
}

/**
 * Uses AI to determine which expected products are confirmed in a document.
 * Internally asks AI for reasoning for logging purposes, but only returns the sequence string.
 *
 * @param string $confirmation_document_text The text of the confirmation document.
 * @param array  $expected_products_for_ai_prompt An array of strings, each representing a product.
 * @param mysqli $conn Database connection.
 * @return string Comma-separated string of confirmed sequence numbers (e.g., "1,3") or "N/A".
 */
function getConfirmedProductSequencesViaAI(string $confirmation_document_text, array $expected_products_for_ai_prompt, mysqli $conn): string {
    $default_sequences = "N/A";

    if (empty(trim($confirmation_document_text))) {
        error_log("getConfirmedProductSequencesViaAI: Confirmation document text is empty. Returning '{$default_sequences}'.");
        return $default_sequences;
    }
    if (empty($expected_products_for_ai_prompt)) {
        error_log("getConfirmedProductSequencesViaAI: Expected products list is empty. Returning '{$default_sequences}'.");
        return $default_sequences;
    }

    $products_list_string = implode("\n", $expected_products_for_ai_prompt);

    $prompt = "Objective: Analyze the 'Confirmation Document Text' to identify which products from the 'Expected Product List' are confirmed, and provide reasoning.\n\n";
    $prompt .= "Expected Product List (from quote - each item has a 'Seq' number and a descriptive name):\n" . htmlspecialchars($products_list_string) . "\n\n";
    $prompt .= "Confirmation Document Text:\n" . htmlspecialchars($confirmation_document_text) . "\n\n";
    $prompt .= "Task: For each product in the 'Expected Product List', determine if the core service/activity and primary location are confirmed in the 'Confirmation Document Text'.\n";
    $prompt .= " - The 'Expected Product List' items may contain internal codes (like 'FDBC PAK 4') or classifications (like 'SIC'). You should IGNORE these codes and classifications for primary matching.\n";
    $prompt .= " - Focus ONLY on whether the main described activity (e.g., 'Full Day Green Island Reef Cruise') and location (e.g., 'from Cairns') are present and confirmed in the document. For example, if the expected product is 'FULL DAY GREEN ISLAND REEF CRUISE (SIC)' and the document confirms 'Full Day Green Island tour', this should be considered a match for the core activity if the location also aligns.\n";
    $prompt .= " - Specifically, if an expected product includes 'REEF CRUISE' (e.g., 'GREEN ISLAND REEF CRUISE') and the document confirms a tour to 'GREEN ISLAND' from the same location, consider this a match for the core service, especially if the document is associated with an operator known for 'Reef Cruises' to that island.\n";
    $prompt .= " - The confirmation document may list specific inclusions (like lunch, semi-submarine); if the core activity and location match an expected product, consider it a match even if the expected product name doesn't detail all those specific inclusions but implies a package.\n\n";
    $prompt .= "Output Format: Return a single, valid JSON object with the following fields ONLY:\n";
    $prompt .= "1. \"confirmed_sequences\": A comma-separated string of the 'Seq' numbers for the products you identify as confirmed based on this core matching logic (e.g., \"1,3,5\"). If NO products from the list appear to be confirmed, this value should be the string \"N/A\".\n";
    $prompt .= "2. \"reasoning\": A brief explanation for your decision. If sequences are confirmed, explain which parts of the document text matched which expected products. If 'N/A' for sequences, explain why no products were considered confirmed (e.g., 'Document text did not clearly mention services X, Y, or Z from the expected list.', or 'While Green Island Tour was mentioned, the key Reef Cruise aspect from Seq 13 was missing, making the match uncertain.').\n\n";
    $prompt .= "Example Output (Match Found):\n";
    $prompt .= "```json\n";
    $prompt .= "{\n";
    $prompt .= "  \"confirmed_sequences\": \"13\",\n";
    $prompt .= "  \"reasoning\": \"The document text mentions 'Full Day Green Island departing Cairns', which strongly matches the core service and location of 'Seq: 13 - Product: ...FULL DAY GREEN ISLAND REEF CRUISE (SIC) from Cairns', especially given the instruction to ignore codes and focus on core activity. The listed inclusions in the document further support this.\"\n";
    $prompt .= "}\n";
    $prompt .= "```\n\n";
    $prompt .= "Example Output (No Match Found):\n";
    $prompt .= "```json\n";
    $prompt .= "{\n";
    $prompt .= "  \"confirmed_sequences\": \"N/A\",\n";
    $prompt .= "  \"reasoning\": \"The confirmation document mentions a general booking but does not contain specific details that clearly match any core services or locations from the expected product list. For instance, while Seq 13 is 'FULL DAY GREEN ISLAND REEF CRUISE', the document only mentioned 'Island trip' without specific reference to 'Green Island' or 'Reef Cruise'.\"\n";
    $prompt .= "}\n";
    $prompt .= "```\n";
    $prompt .= "Ensure your entire response is ONLY the JSON object described, with no text before or after the ```json block if you use it, or just the raw JSON object.";

    $apiUrl = apiURL_Flash();
    if (!$apiUrl) {
        error_log("getConfirmedProductSequencesViaAI: API URL is missing. Returning '{$default_sequences}'.");
        return $default_sequences;
    }

    $payload = [
        'contents' => [['parts' => [['text' => $prompt]]]],
        'generationConfig' => [
            'temperature' => 0.2,
            'response_mime_type' => "application/json", // Expect JSON output
            'maxOutputTokens' => 50000, // Increased for JSON with reasoning
            'topP' => 0.95,
            'topK' => 40
        ]
    ];
    $headers = ['Content-Type: application/json'];

    $ch = curl_init($apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 20);
    curl_setopt($ch, CURLOPT_TIMEOUT, 75);

    $response_body = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    if ($curlError || $httpCode != 200) {
        error_log("getConfirmedProductSequencesViaAI: API Error - HTTP $httpCode, cURL: $curlError, Resp: $response_body. Returning '{$default_sequences}'.");
        return $default_sequences;
    }

    $responseData = json_decode($response_body, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("getConfirmedProductSequencesViaAI: Failed to decode main API JSON. Error: " . json_last_error_msg() . ". Raw: " . $response_body . ". Returning '{$default_sequences}'.");
        return $default_sequences;
    }
    if (isset($responseData['promptFeedback']['blockReason'])) {
        error_log("getConfirmedProductSequencesViaAI: Gemini Content Blocked: " . ($responseData['promptFeedback']['blockReason'] ?? 'Unknown') . ". Returning '{$default_sequences}'.");
        return $default_sequences;
    }

    $extractedJsonText = null;
    // --- CORRECTED PATH TO EXTRACT TEXT ---
    if (isset($responseData['candidates'][0]['content']['parts'][0]['text'])) {
        $generatedText = $responseData['candidates'][0]['content']['parts'][0]['text'];
        // --- END CORRECTION ---
        $trimmedText = trim($generatedText);
        // Remove markdown ```json if present (though API should return raw with response_mime_type)
        if (substr($trimmedText, 0, 7) === "```json") {
            $trimmedText = substr($trimmedText, 7);
            if (substr($trimmedText, -3) === "```") {
                $trimmedText = substr($trimmedText, 0, -3);
            }
            $trimmedText = trim($trimmedText);
        } elseif (strpos($trimmedText, '```') === 0 && strrpos($trimmedText, '```') === (strlen($trimmedText) - 3) ) {
             $trimmedText = substr($trimmedText, 3, -3);
             $trimmedText = trim($trimmedText);
        }
        $extractedJsonText = $trimmedText;
    } else {
        error_log("getConfirmedProductSequencesViaAI: Could not extract text part from Gemini response (candidates[0]['content']['parts'][0]['text'] was not set). Raw: " . json_encode($responseData) . ". Returning '{$default_sequences}'.");
        return $default_sequences;
    }
    
    if ($extractedJsonText === null) { // This check is largely redundant if the above else catches it.
        error_log("getConfirmedProductSequencesViaAI: Extracted JSON text is null after attempting to find it. Raw full response: " . $response_body . ". Returning '{$default_sequences}'.");
        return $default_sequences;
    }

    $parsedResult = json_decode($extractedJsonText, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("getConfirmedProductSequencesViaAI: Failed to decode extracted JSON from AI. Error: " . json_last_error_msg() . ". JSON Text: " . $extractedJsonText . ". Returning '{$default_sequences}'.");
        return $default_sequences;
    }

    $ai_sequences = $parsedResult['confirmed_sequences'] ?? 'N/A'; 
    $ai_reasoning = $parsedResult['reasoning'] ?? 'No reasoning provided by AI.';
    error_log("getConfirmedProductSequencesViaAI (Internal Log): Sequences='{$ai_sequences}', Reasoning='{$ai_reasoning}'");

    $sequences_to_return = trim($ai_sequences);
    if (strtoupper($sequences_to_return) === 'N/A') {
        return "N/A";
    }
    if (preg_match('/^(\d+(,\s*\d+)*)?$/', $sequences_to_return) || empty($sequences_to_return) ) {
        return empty($sequences_to_return) ? "N/A" : $sequences_to_return;
    } else {
        error_log("getConfirmedProductSequencesViaAI: AI returned invalid format for 'confirmed_sequences' ('{$sequences_to_return}') even after JSON parse. Defaulting to 'N/A'.");
        return "N/A";
    }
}

function create_new_organization(mysqli $conn, array $aiData): ?int {
    $sql = "INSERT INTO tdu_organisation (organization_name, phone, email, website, address, region, country, preferred, membershipid, active, merged)
            VALUES (?, ?, ?, ?, ?, ?, ?, 0, 1, 1, 0)"; // Defaults for preferred, membershipid, active, merged

    $stmt = $conn->prepare($sql);
    if (!$stmt) {
        error_log("create_new_organization: Prepare failed: " . $conn->error);
        return null;
    }

    // The AI prompt for extractContactAndOrganizationInfo provides:
    // 'organization_name', 'contact_phone', 'contact_email', 'organization_website', 'organization_address', 'organization_country', 'organization_region'
    // tdu_organisation has: organization_name, phone, email, website, address, region, country

    // We need to decide if contact_phone/contact_email from AI should map to tdu_organisation.phone/email
    // For now, let's assume they *don't* unless explicitly stated.
    // If AI provides org-specific phone/email, use that. Otherwise, null.
    // Your current AI prompt does not explicitly ask for organization_phone or organization_email, only contact_phone/email.
    // So, org phone/email will be NULL unless you change the AI prompt or map contact details here.

    $orgName = trim($aiData['organization_name'] ?? '');
    // For organization table, we might not want to use the *contact's* phone/email.
    // If your AI prompt doesn't give specific org_phone/org_email, these should probably be null.
    $orgPhone = (isset($aiData['organization_phone']) && strtolower(trim($aiData['organization_phone'])) !== 'n/a') ? trim($aiData['organization_phone']) : null; // Assuming AI might give this
    $orgEmail = (isset($aiData['organization_email']) && strtolower(trim($aiData['organization_email'])) !== 'n/a') ? trim($aiData['organization_email']) : null; // Assuming AI might give this

    $orgWebsite = (isset($aiData['organization_website']) && strtolower(trim($aiData['organization_website'])) !== 'n/a') ? trim($aiData['organization_website']) : null;
    $orgAddress = (isset($aiData['organization_address']) && strtolower(trim($aiData['organization_address'])) !== 'n/a') ? trim($aiData['organization_address']) : null;
    $orgRegion = (isset($aiData['organization_region']) && strtolower(trim($aiData['organization_region'])) !== 'n/a') ? trim($aiData['organization_region']) : null;
    $orgCountry = (isset($aiData['organization_country']) && strtolower(trim($aiData['organization_country'])) !== 'n/a') ? trim($aiData['organization_country']) : null;

    if (empty($orgName)) {
        error_log("create_new_organization: Organization name is empty or N/A. Cannot create.");
        return null;
    }

    $stmt->bind_param("sssssss",
        $orgName,
        $orgPhone, // This might be null if AI doesn't provide a specific org phone
        $orgEmail, // This might be null if AI doesn't provide a specific org email
        $orgWebsite,
        $orgAddress,
        $orgRegion,
        $orgCountry
    );

    if ($stmt->execute()) {
        $newOrgId = $conn->insert_id;
        $stmt->close();
        return $newOrgId;
    } else {
        error_log("create_new_organization: Execute failed for org '{$orgName}': " . $stmt->error);
        $stmt->close();
        return null;
    }
}

function create_new_contact(mysqli $conn, int $organizationId, array $aiData): ?int {
    $sql = "INSERT INTO tdu_contacts (organizationid, name, email, mobile, vendorid) VALUES (?, ?, ?, ?, NULL)"; // vendorid is NULL

    $stmt = $conn->prepare($sql);
    if (!$stmt) {
        error_log("create_new_contact: Prepare failed: " . $conn->error);
        return null;
    }

    $contactName = (isset($aiData['contact_name']) && strtolower(trim($aiData['contact_name'])) !== 'n/a') ? trim($aiData['contact_name']) : null;
    $contactEmail = (isset($aiData['contact_email']) && strtolower(trim($aiData['contact_email'])) !== 'n/a') ? trim($aiData['contact_email']) : null;
    $contactMobile = (isset($aiData['contact_phone']) && strtolower(trim($aiData['contact_phone'])) !== 'n/a') ? trim($aiData['contact_phone']) : null; // AI gives 'contact_phone', DB has 'mobile'

    // Require at least a name or email to create a contact
    if (empty($contactName) && empty($contactEmail)) {
        error_log("create_new_contact: Contact name and email are both missing or N/A. Cannot create contact for Org ID {$organizationId}.");
        return null;
    }
     // DB schema shows name, email, mobile as NOT NULL. Ensure AI provides valid values or adjust schema/logic.
    // For now, if AI gives N/A, we pass null. If the DB strictly enforces NOT NULL without defaults, this will fail.
    // Let's assume for required fields, 'N/A' means we should not proceed or use a placeholder if DB allows.
    // Given the schema, 'name' and 'email' are NOT NULL. 'mobile' also NOT NULL.
    // This function should ensure valid values are passed for these or handle it.
    // If AI gives N/A for a NOT NULL field, you must decide: skip creation, or insert a placeholder (e.g. "Unknown").
    // For now, this code assumes if it's N/A from AI, it will be null, which will violate NOT NULL constraint.
    // A better approach for NOT NULL fields:
    if (empty($contactName)) $contactName = "Unknown Contact (AI)"; // Placeholder if DB requires it
    if (empty($contactEmail)) $contactEmail = "unknown_" . uniqid() . "@example.com"; // Placeholder if DB requires it, ensure unique
    if (empty($contactMobile)) $contactMobile = "N/A"; // Placeholder if DB requires it


    $stmt->bind_param("isss", $organizationId, $contactName, $contactEmail, $contactMobile);

    if ($stmt->execute()) {
        $newContactId = $conn->insert_id;
        $stmt->close();
        return $newContactId;
    } else {
        error_log("create_new_contact: Execute failed for Org ID {$organizationId}, Contact Name '{$contactName}': " . $stmt->error);
        $stmt->close();
        return null;
    }
}

function find_best_match_org_via_ai_disambiguation(
    string $extracted_org_name_from_ai, // The organization name extracted by the initial AI call
    array $ai_extracted_full_details,  // The complete details array from extractContactAndOrganizationInfo
    mysqli $conn,
    int $max_candidates_for_ai = 5,    // Max candidates to send to AI for disambiguation
    float $min_confidence_for_match = 0.7 // Minimum confidence from AI to consider it a match
): ?int {
    if (empty(trim($extracted_org_name_from_ai)) || strtolower(trim($extracted_org_name_from_ai)) === 'n/a') {
        error_log("find_best_match_org_via_ai_disambiguation: AI extracted org name is empty or N/A.");
        return null;
    }

    // --- Stage 1: Initial Candidate Retrieval from Database ---
    $db_candidates_with_details = get_initial_organisation_candidates(
        $extracted_org_name_from_ai,
        $conn,
        $max_candidates_for_ai // Use this to limit SQL results too
    );

    if (empty($db_candidates_with_details)) {
        error_log("find_best_match_org_via_ai_disambiguation: No initial DB candidates found for '{$extracted_org_name_from_ai}'.");
        return null; // No candidates to send to AI
    }

    // --- Stage 2: AI-Powered Disambiguation ---
    // $ai_extracted_full_details should be the output from your existing extractContactAndOrganizationInfo()
    // which contains keys like 'organization_name', 'contact_name', 'contact_email', etc.

    $ai_disambiguation_result = call_gemini_for_organisation_disambiguation(
        $ai_extracted_full_details,
        $db_candidates_with_details,
        $conn // Pass $conn if your API key function or other helpers within it need it
    );

    if (!$ai_disambiguation_result || !isset($ai_disambiguation_result['best_match_organizationid'])) {
        error_log("find_best_match_org_via_ai_disambiguation: AI disambiguation failed or returned invalid format for '{$extracted_org_name_from_ai}'. Raw AI result: " . json_encode($ai_disambiguation_result));
        return null;
    }

    $best_match_id = $ai_disambiguation_result['best_match_organizationid'];
    $confidence = $ai_disambiguation_result['confidence_score'] ?? 0.0;
    $reasoning = $ai_disambiguation_result['reasoning'] ?? 'N/A';

    error_log("find_best_match_org_via_ai_disambiguation: AI Disambiguation for '{$extracted_org_name_from_ai}': Matched ID '{$best_match_id}', Confidence {$confidence}, Reason: {$reasoning}");

    if ($best_match_id !== null && $confidence >= $min_confidence_for_match) {
        return (int)$best_match_id;
    } else {
        if ($best_match_id !== null && $confidence < $min_confidence_for_match) {
            error_log("find_best_match_org_via_ai_disambiguation: AI found a match (ID: {$best_match_id}) but confidence ({$confidence}) is below threshold ({$min_confidence_for_match}).");
        }
        return null; // No match or low confidence
    }
}

/**
 * Helper function to get initial organisation candidates from the DB.
 * Fetches organisation details and their associated contacts.
 */
function get_initial_organisation_candidates(
    string $org_name_for_search,
    mysqli $conn,
    int $limit = 10
): array {
    global $common_org_filter_terms; // Assuming this is available

    $candidates = [];
    if (empty(trim($org_name_for_search))) {
        return $candidates;
    }

    // Normalize for SQL LIKE search - keep some spaces as wildcards potentially
    $normalized_for_sql = strtolower(trim($org_name_for_search));
    // Remove common terms that might hinder broad matching in SQL
    foreach ($common_org_filter_terms as $term) {
        // Simpler removal for SQL, might need refinement
        $normalized_for_sql = str_replace(strtolower($term), '', $normalized_for_sql);
    }
    $normalized_for_sql = preg_replace('/[^\p{L}\p{N}\s%]/u', '', $normalized_for_sql); // Keep letters, numbers, spaces, and percent for wildcards
    $normalized_for_sql = trim(preg_replace('/\s+/', '%', $normalized_for_sql)); // Replace spaces with %

    if (empty($normalized_for_sql)) {
        return $candidates;
    }
    $search_pattern = '%' . $normalized_for_sql . '%';
    // Also create a condensed version for matching names like MakeMyTrip vs Make My Trip
    $condensed_search_pattern = '%' . str_replace('%', '', $normalized_for_sql) . '%';


    // SQL to fetch candidate organizations
    // This query can be refined further based on your specific needs
    $sql_org = "SELECT organizationid, organization_name, website, address, region, country
                FROM tdu_organisation
                WHERE (LOWER(organization_name) LIKE ? OR LOWER(REPLACE(organization_name, ' ', '')) LIKE ?)
                AND organization_name IS NOT NULL AND organization_name != ''
                ORDER BY CASE
                    WHEN LOWER(organization_name) = ? THEN 1 -- Exact match (case insensitive)
                    WHEN LOWER(organization_name) LIKE ? THEN 2 -- Starts with
                    ELSE 3
                END
                LIMIT ?";

    $stmt_org = $conn->prepare($sql_org);
    if (!$stmt_org) {
        error_log("get_initial_organisation_candidates: Prepare failed (org): " . $conn->error);
        return $candidates;
    }

    $exact_match_term = strtolower(trim($org_name_for_search)); // For exact full name match prio
    $starts_with_pattern = strtolower(trim($org_name_for_search)) . '%';
    $stmt_org->bind_param("ssssi", $search_pattern, $condensed_search_pattern, $exact_match_term, $starts_with_pattern, $limit);

    if ($stmt_org->execute()) {
        $result_org = $stmt_org->get_result();
        while ($org_row = $result_org->fetch_assoc()) {
            $candidate_org_id = $org_row['organizationid'];
            $org_details = [
                'organizationid' => (int)$candidate_org_id,
                'organization_name' => $org_row['organization_name'],
                'website' => $org_row['website'],
                'address' => $org_row['address'],
                'region' => $org_row['region'],
                'country' => $org_row['country'],
                'contacts' => [] // Initialize contacts array
            ];

            // Fetch associated contacts for this organization
            $sql_contacts = "SELECT name, email, mobile FROM tdu_contacts WHERE organizationid = ?";
            $stmt_contacts = $conn->prepare($sql_contacts);
            if ($stmt_contacts) {
                $stmt_contacts->bind_param("i", $candidate_org_id);
                if ($stmt_contacts->execute()) {
                    $result_contacts = $stmt_contacts->get_result();
                    while ($contact_row = $result_contacts->fetch_assoc()) {
                        $org_details['contacts'][] = [
                            'name' => $contact_row['name'],
                            'email' => $contact_row['email'],
                            'mobile' => $contact_row['mobile']
                        ];
                    }
                    if ($result_contacts instanceof mysqli_result) $result_contacts->free();
                } else {
                    error_log("get_initial_organisation_candidates: Execute failed (contacts for org ID {$candidate_org_id}): " . $stmt_contacts->error);
                }
                $stmt_contacts->close();
            } else {
                 error_log("get_initial_organisation_candidates: Prepare failed (contacts for org ID {$candidate_org_id}): " . $conn->error);
            }
            $candidates[] = $org_details;
        }
        if ($result_org instanceof mysqli_result) $result_org->free();
    } else {
        error_log("get_initial_organisation_candidates: Execute failed (org): " . $stmt_org->error);
    }
    $stmt_org->close();
    return $candidates;
}


/**
 * Helper function to call Gemini for organisation disambiguation.
 */
function call_gemini_for_organisation_disambiguation(
    array $ai_extracted_details,
    array $db_candidates_details,
    mysqli $conn // Or remove if not needed by API key function
): ?array {
    // Construct the prompt for Gemini
    $prompt = "Objective:\n";
    $prompt .= "Based on the 'AI Extracted Organization Details' below, identify the best matching organization from the 'Database Candidate List'.\n";
    $prompt .= "Consider all fields for matching: organization name (allowing for variations like 'MakeMyTrip' vs 'Make My Trip', or slight misspellings), website, address components, and especially contact information (name, email, phone/mobile).\n\n";

    $prompt .= "AI Extracted Organization Details:\n";
    $prompt .= "- Organization Name: \"" . htmlspecialchars($ai_extracted_details['organization_name'] ?? 'N/A') . "\"\n";
    $prompt .= "- Contact Name: \"" . htmlspecialchars($ai_extracted_details['contact_name'] ?? 'N/A') . "\"\n";
    $prompt .= "- Contact Email: \"" . htmlspecialchars($ai_extracted_details['contact_email'] ?? 'N/A') . "\"\n";
    $prompt .= "- Contact Phone: \"" . htmlspecialchars($ai_extracted_details['contact_phone'] ?? 'N/A') . "\"\n"; // AI gives contact_phone
    $prompt .= "- Website: \"" . htmlspecialchars($ai_extracted_details['organization_website'] ?? 'N/A') . "\"\n";
    $prompt .= "- Address: \"" . htmlspecialchars($ai_extracted_details['organization_address'] ?? 'N/A') . "\"\n";
    $prompt .= "- Country: \"" . htmlspecialchars($ai_extracted_details['organization_country'] ?? 'N/A') . "\"\n";
    $prompt .= "- Region: \"" . htmlspecialchars($ai_extracted_details['organization_region'] ?? 'N/A') . "\"\n\n";
    $prompt .= "If any of the details not the organization name is blank, disregard for comparison, organization name is the most important, address is second, email domain is also a good tell\n\n";
    $prompt .= "Database Candidate List (organizations from our system):\n";
    $prompt .= json_encode($db_candidates_details, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES) . "\n\n";

    $prompt .= "Task:\n";
    $prompt .= "Return a single, valid JSON object with the following fields ONLY:\n";
    $prompt .= "- \"best_match_organizationid\": The integer 'organizationid' of the best matching candidate from the 'Database Candidate List'. If no candidate is a reasonably good match, return null (not the string 'null', but actual JSON null).\n";
    $prompt .= "- \"confidence_score\": A score from 0.0 to 1.0 indicating your confidence in the match. If no match, this can be low (e.g., 0.0 or 0.1).\n";
    $prompt .= "- \"reasoning\": A brief explanation of why you chose this candidate (or why no match was found), highlighting the key matching or mismatching fields.\n\n";

    $prompt .= "Important Comparison Guidelines:\n";
    $prompt .= "*   Name Variations: Consider organization names equivalent if they differ only by spacing (e.g., \"MakeMyTrip\" matches \"Make My Trip\"), capitalization, or common business suffixes (like Ltd, Inc, Pvt), unless these create a clear distinction between different entities in the candidate list.\n";
    $prompt .= "*   Contact Information: Pay close attention to matching contact names, emails, and phone numbers. A strong contact match can significantly increase confidence.\n";
    $prompt .= "*   Website & Address: Use website and address components as secondary confirmation points.\n";
    $prompt .= "*   Focus: Your primary goal is to find the single best match from the provided database list for the AI Extracted details. Do not invent new organizations.\n\n";

    $prompt .= "Example Output (Match Found):\n";
    $prompt .= "```json\n";
    $prompt .= "{\n";
    $prompt .= "  \"best_match_organizationid\": 101,\n";
    $prompt .= "  \"confidence_score\": 0.9,\n";
    $prompt .= "  \"reasoning\": \"Strong match on organization name variant 'MakeMyTrip' vs 'Make My Trip (India) Pvt. Ltd.', identical website, and the extracted contact email 'rajesh.k@makemytrip.com' is present in the candidate's contact list.\"\n";
    $prompt .= "}\n";
    $prompt .= "```\n\n";
    $prompt .= "Example Output (No Good Match Found):\n";
    $prompt .= "```json\n";
    $prompt .= "{\n";
    $prompt .= "  \"best_match_organizationid\": null,\n";
    $prompt .= "  \"confidence_score\": 0.2,\n";
    $prompt .= "  \"reasoning\": \"While 'Make My Travels Agency' has a similar name, the website, address, and contact details do not align with the AI extracted information. None of the candidates are a strong fit.\"\n";
    $prompt .= "}\n";
    $prompt .= "```\n";


    $apiUrl = apiURL_Flash(); // Defined in ai_keys.php
    if (!$apiUrl) {
        error_log("call_gemini_for_organisation_disambiguation: API URL is missing.");
        return ['error' => 'API URL not configured for Gemini.', 'best_match_organizationid' => null, 'confidence_score' => 0.0, 'reasoning' => 'API URL missing.'];
    }
    //error_log($prompt);
    $payload = [
        'contents' => [['parts' => [['text' => $prompt]]]],
        'generationConfig' => [
            'temperature' => 0.2, // Lower temperature for more deterministic output
            'response_mime_type' => "application/json",
            'maxOutputTokens' => 50000, // Adjust as needed based on typical response size
            'topP' => 0.95,
            'topK' => 40
        ]
    ];
    $headers = ['Content-Type: application/json'];

    $ch = curl_init($apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 25); // Increased timeout
    curl_setopt($ch, CURLOPT_TIMEOUT, 100);      // Increased timeout

    $response_body = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    if ($curlError || $httpCode != 200) {
        $error_message = "Gemini API Error (Org Disambiguation HTTP $httpCode, cURL $curlError): " . $response_body;
        error_log("call_gemini_for_organisation_disambiguation: " . $error_message);
        $details = json_decode($response_body, true);
        $reason = "Gemini API Error (HTTP $httpCode)" . (isset($details['error']['message']) ? ': ' . $details['error']['message'] : '');
        return ['error' => $reason, 'best_match_organizationid' => null, 'confidence_score' => 0.0, 'reasoning' => $reason];
    }

    $responseData = json_decode($response_body, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("call_gemini_for_organisation_disambiguation: Failed to decode API JSON (Outer response). Error: " . json_last_error_msg() . ". Raw: " . $response_body);
        return ['error' => 'Gemini API: Failed to decode main JSON.', 'best_match_organizationid' => null, 'confidence_score' => 0.0, 'reasoning' => 'Failed to decode API JSON.'];
    }

    if (isset($responseData['promptFeedback']['blockReason'])) {
        $blockReason = $responseData['promptFeedback']['blockReason'] ?? 'Unknown';
        error_log("call_gemini_for_organisation_disambiguation: Gemini Content Blocked: " . $blockReason . ". Details: " . json_encode($responseData['promptFeedback']));
        $reason = 'Gemini: Content blocked (' . $blockReason . ')';
        return ['error' => $reason, 'best_match_organizationid' => null, 'confidence_score' => 0.0, 'reasoning' => $reason];
    }

    $extractedJsonText = null;
    if (isset($responseData['candidates'][0]['content']['parts'][0]['text'])) {
        $generatedText = $responseData['candidates'][0]['content']['parts'][0]['text'];
        $trimmedText = trim($generatedText);
        // Remove markdown ```json if present
        if (substr($trimmedText, 0, 7) === "```json") {
            $trimmedText = substr($trimmedText, 7);
            if (substr($trimmedText, -3) === "```") {
                $trimmedText = substr($trimmedText, 0, -3);
            }
            $trimmedText = trim($trimmedText);
        } elseif (strpos($trimmedText, '```') === 0 && strrpos($trimmedText, '```') === (strlen($trimmedText) - 3) ) {
             $trimmedText = substr($trimmedText, 3, -3);
             $trimmedText = trim($trimmedText);
        }
        $extractedJsonText = $trimmedText;
    } else {
        error_log("call_gemini_for_organisation_disambiguation: Could not extract text part from Gemini response. Raw: " . $response_body);
         return ['error' => 'Gemini API: No text part.', 'best_match_organizationid' => null, 'confidence_score' => 0.0, 'reasoning' => 'Gemini API: No text part in response.'];
    }

    if ($extractedJsonText === null) {
        error_log("call_gemini_for_organisation_disambiguation: Extracted JSON text is null. Raw Gemini Response: " . $response_body);
        return ['error' => 'Gemini API: Failed to extract JSON string.', 'best_match_organizationid' => null, 'confidence_score' => 0.0, 'reasoning' => 'Gemini API: Failed to extract JSON content.'];
    }

    $parsedResult = json_decode($extractedJsonText, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("call_gemini_for_organisation_disambiguation: Failed to decode extracted JSON from AI. Error: " . json_last_error_msg() . ". JSON Text: " . $extractedJsonText);
        return ['error' => 'Failed to parse AI JSON.', 'best_match_organizationid' => null, 'confidence_score' => 0.0, 'reasoning' => 'Failed to parse AI JSON. Text: ' . substr($extractedJsonText, 0, 200)];
    }

    // Ensure the expected fields are present, defaulting if not
    return [
        'best_match_organizationid' => isset($parsedResult['best_match_organizationid']) ? (is_numeric($parsedResult['best_match_organizationid']) ? (int)$parsedResult['best_match_organizationid'] : null) : null,
        'confidence_score' => isset($parsedResult['confidence_score']) ? (float)$parsedResult['confidence_score'] : 0.0,
        'reasoning' => $parsedResult['reasoning'] ?? 'N/A - AI did not provide reasoning.',
        'error' => null // No error if we reached here and parsed
    ];
}

function normalize_org_name_for_similarity(string $name, array $common_terms_to_remove): string {
    $processed_name = strtolower(trim($name));

    // Remove punctuation (allow spaces, numbers, letters)
    $processed_name = preg_replace('/[^\p{L}\p{N}\s]/u', '', $processed_name);

    // Remove common terms using word boundaries
    foreach ($common_terms_to_remove as $term) {
        $processed_name = preg_replace('/\b' . preg_quote(strtolower($term), '/') . '\b/u', '', $processed_name);
    }

    // Remove extra spaces that might have resulted from removals
    $processed_name = trim(preg_replace('/\s+/', ' ', $processed_name));
    
    // Remove all remaining spaces
    $processed_name = str_replace(' ', '', $processed_name);

    return $processed_name;
}

/**
 * Defines the prompt for the AI to classify the email category.
 * MODIFIED: 'unsure' removed from allowed keywords.
 */
function get_ai_email_category_prompt_text(): string {
    return "Analyze the following email subject and body.
What is the primary travel category for this request?
Respond with ONLY one of the following keywords (all lowercase): 'sic', 'private', 'sic+private', 'group'.
Do not include any other text, explanations, or JSON formatting. Just the single category keyword.

For example, if it's a shared tour, respond: sic
If it's for a private vehicle, respond: private
If it's for a large group, respond: group

Email Content is below:
";
}

/**
 * Calls Gemini AI to classify the email's category.
 * MODIFIED: 'unsure' removed from valid_categories. Returns empty string for invalid/unexpected AI output.
 */
function determine_email_category_via_ai(string $subject, string $body, mysqli $conn): string {
    if (empty(trim($subject)) && empty(trim($body))) {
         error_log("determine_email_category_via_ai: Subject and body are empty.");
        return ''; // Return empty if no content
    }

    $full_prompt_text = get_ai_email_category_prompt_text() . "\nSubject: " . $subject . "\nBody: " . $body;
    $apiUrl = apiURL_Flash();
    if (!$apiUrl) {
        error_log("determine_email_category_via_ai: API URL is missing.");
        return '';
    }

    $payload = [
        'contents' => [['parts' => [['text' => $full_prompt_text]]]],
        'generationConfig' => ['temperature' => 0.1, 'maxOutputTokens' => 50000, 'topP' => 0.95, 'topK' => 40]
    ];
    $headers = ['Content-Type: application/json'];

    $ch = curl_init($apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 15);
    curl_setopt($ch, CURLOPT_TIMEOUT, 45);

    $response_body = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    if ($curlError || $httpCode !== 200) {
        error_log("determine_email_category_via_ai: cURL/API Error - HTTP $httpCode, cURL: $curlError, Resp: $response_body");
        return '';
    }

    $responseData = json_decode($response_body, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("determine_email_category_via_ai: Failed to decode API JSON. Raw: " . $response_body);
        return '';
    }
    if (isset($responseData['promptFeedback']['blockReason'])) {
        error_log("determine_email_category_via_ai: Gemini Content Blocked: " . ($responseData['promptFeedback']['blockReason'] ?? 'Unknown'));
        return '';
    }

    $generated_text = $responseData['candidates'][0]['content']['parts'][0]['text'] ?? null;
    if ($generated_text === null) {
        error_log("determine_email_category_via_ai: Could not extract text. Resp: " . $response_body);
        return '';
    }

    $cleaned_category = strtolower(trim($generated_text));
    // MODIFIED: Valid categories list updated.
    $valid_categories = ['sic', 'private', 'sic+private', 'group'];

    if (in_array($cleaned_category, $valid_categories)) {
        // error_log("determine_email_category_via_ai: AI classified as: " . $cleaned_category);
        return $cleaned_category;
    } else {
        error_log("determine_email_category_via_ai: AI returned unexpected '{$cleaned_category}'. Raw: " . $generated_text);
        return ''; // Return empty string for unexpected categories
    }
}

/**
 * Defines instructions for Gemini AI for travel segment data extraction.
 * MODIFIED: Added mailbox parameter to influence category output.
 */
function gemini_instructions_forSegments(string $mailbox = null): string { // Added $mailbox parameter
    $category_constraint_text = "Determine `SIC`, `Private`, `SIC+Private`, `Group` based on mentions or context."; // Default

    if ($mailbox === 'sales@turtledownunder.com.au') {
        $category_constraint_text = "The email is from the 'sales' mailbox. Therefore, the 'category' MUST BE one of 'SIC', 'Private', or 'SIC+Private'. DO NOT output 'Group'. Determine the most appropriate among these three based on mentions or context.";
    } elseif ($mailbox === 'groupsales@turtledownunder.com.au') {
        $category_constraint_text = "The email is from the 'groupsales' mailbox. Therefore, the 'category' MUST BE 'Group'. DO NOT output 'SIC', 'Private', or 'SIC+Private'.";
    }

    return "
**Objective:** Analyze the provided email subject, body, and any included attachment text to extract travel itinerary details. Focus on identifying structured trip segments first. Output the information strictly in the JSON format below.

**CRITICAL OUTPUT CONSTRAINT:** The absolute highest priority is ensuring that the final comma-separated lists for 'duration', 'cities', and 'country' have the EXACT SAME number of entries. If one has 3 entries, ALL THREE must have 3 entries. This applies even if the entries are 'N/A'. Verify this count match rigorously before finalizing the output. Mismatched counts are incorrect.

**Mailbox Context:** The email being analyzed is associated with the mailbox: " . ($mailbox ?? 'Unknown/General') . ". This may influence category selection as per rules below.

**Output JSON Format:**

```json
{
  'category': 'SIC or Private or SIC+Private or Group', /* THIS FIELD IS CONSTRAINED BY MAILBOX RULES BELOW */
  'travel_date': 'Primary overall travel start date (YYYY-MM-DD) OR \\'N/A\\'',
  'duration': 'Comma-separated list of nights per aggregated city stay (e.g., \\'3, 2, 4\\') OR single total OR \\'N/A\\' OR list of \\'N/A\\'s. MUST match entry count of cities/country.',
  'cities': 'Comma-separated list of Major Cities/Destinations OR \\'N/A\\'. MUST match entry count of duration/country.',
  'country': 'Comma-separated list of countries (repeated per city) OR single country. MUST match entry count of duration/cities.', cannot be N/A, has to be Australia or New Zealand.
  'seaters': 'Number of passengers. if its a range, extract the highest number, it can only be a single number (e.g., \\'7\\') OR \\'N/A\\'',
  'itinerary_items': 'Comma-separated list of activities, tours, transfers, requests, non-major destinations, hotels OR \\'N/A\\'',
  'reasoning': 'Detailed explanation for how each field\\'s value was derived (source text, calculation, logic applied), without naming specific internal rule numbers. Specifically mention how mailbox context influenced category choice if applicable.'
}```

**Key Definitions:**

*   **Major City/Destination:** Includes major cities (e.g., Sydney, Melbourne, Brisbane, Cairns, Gold Coast, Perth, Adelaide) AND significant multi-day tourist destinations used as a base (e.g., Byron Bay, Ayers Rock/Uluru, Airlie Beach/Whitsundays). Exclude specific attractions or day trip locations (like a specific national park visited for a day) from the `cities` list; these belong in `itinerary_items`.
*   **Aggregated Stay:** A continuous period spent overnight in the *same* Major City/Destination. Day trips *do not* break an aggregated stay block. Travel *to a different overnight location* ends one aggregated stay block and begins the next.

**Extraction Strategy (Apply in Order):**

**Phase 1: Segmented Itinerary Extraction (Highest Priority for `duration`, `cities`, `country`)**
*   **Attempt 1: Explicit Nights Format ('X nights City')**
    *   Search for patterns like '3 nights Sydney', '2 nights Cairns'. Use the Major City/Destination definition.
    *   If found: Extract night counts, location names, and repeat country. **Crucially, ensure the generated lists for duration, cities, and country have the exact same number of entries.**
    *   Populate `duration`, `cities`, `country` lists accordingly (e.g., duration: '3, 2', cities: 'Sydney, Cairns', country: 'Australia, Australia').
    *   **If this attempt yields valid, count-matched results for `duration`, `cities`, `country`, STOP attempts for these fields and proceed to Phase 3 for other fields.**

*   **Attempt 2: Date/Day Range Format**
    *   Look for itineraries specified by date ranges (e.g., 'Oct 1-4: Sydney', 'Oct 4-6: Melbourne') or sequential days ('Day 1-3: Sydney', 'Day 4-5: Melbourne').
    *   **Step A: Identify and Aggregate Continuous Stays:** Group consecutive days/dates for overnight stays in the *same* Major City/Destination. Calculate the **single TOTAL number of nights** for each **entire continuous block** (Check-Out Date - Check-In Date = Nights). Aggregate ALL consecutive nights in one location into ONE duration entry.
    *   **Step B: Record Location and Country per Block:** Identify the single Major City/Destination and country for each **aggregated stay block**.
    *   **Step C: Assemble Output Lists:** Create comma-separated strings for `duration` (total nights per block), `cities` (location per block), `country` (repeated country per block). **CRITICAL: Verify these three lists have the exact same number of entries and maintain chronological order.**
    *   **Example:** 'Oct 1-4: Sydney stay, Oct 4 Travel to Mel, Oct 4-6: Melbourne stay' -> Aggregated blocks: Sydney (Oct 1 check-in, Oct 4 check-out = 3 nights), Melbourne (Oct 4 check-in, Oct 6 check-out = 2 nights). Output -> duration: '3, 2', cities: 'Sydney, Melbourne', country: 'Australia, Australia'. (Counts match: 2 entries each).
    *   **If this attempt yields valid, count-matched results (identifies at least one aggregated stay block) for `duration`, `cities`, `country`, STOP attempts for these fields and proceed to Phase 3 for other fields.**

**Phase 2: Fallback Extraction (Use ONLY if Phase 1 yielded NO count-matched results for `duration`, `cities`, `country`)**

*   **NON-NEGOTIABLE CONSTRAINT:** `duration`, `cities`, and `country` outputs generated in this phase MUST have the exact same number of comma-separated entries. This ensures consistency even if specific details aren't available (e.g., outputting 'N/A', 'Sydney, Melbourne', 'Australia, Australia' is WRONG; it should be 'N/A, N/A', 'Sydney, Melbourne', 'Australia, Australia' if duration details are missing but two cities are found).
*   **Step A: Identify Cities (Determine Count 'N'):** Find all distinct **Major Cities/Destinations** mentioned. List them comma-separated in `cities`. Record the count 'N'. If none, `cities` is 'N/A' (N=0).
*   **Step B: Determine Duration (Match Count 'N'):**
    *   Check for a single total duration (e.g., '10-day trip').
    *   If found AND N <= 1, set `duration` to this single value (e.g., '10').
    *   Otherwise (N > 1 OR no total duration found), set `duration` to a comma-separated list of 'N/A' repeated 'N' times. **This forces the duration count to match the city count.**
    *   If `cities` was 'N/A' (N=0) and no total duration, `duration` is 'N/A'. (Count is 1, matching N=0 fallback logic).
*   **Step C: Determine Country (Match Count 'N'):**
    *   Identify the primary country.
    *   If N >= 1, set `country` to the country name repeated 'N' times, comma-separated. **This forces the country count to match the city count.**
    *   If `cities` was 'N/A' (N=0), set `country` to the single country name (or 'N/A' if unknown). (Count is 1, matching N=0 fallback logic).
    *   **Crucially re-verify:** Ensure the final `duration`, `cities`, `country` lists derived from Steps A, B, C have the exact same number of entries before finalizing.

**Phase 3: Extract Other Fields (Apply regardless of Phase 1 or 2 outcome)**

*   **Travel Date:** Extract the primary start date of the overall travel or main tour. If multiple distinct start dates for different legs of a trip are mentioned, try to identify the very first significant travel date. If a date range for the whole trip is given (e.g., 'Travel from Oct 10-20'), use the start of that range. Format strictly as YYYY-MM-DD. Output 'N/A' if no clear primary start date is found or if it's too ambiguous.
*   **Seaters:** Extract total guests ('No of Pax - X', 'X adults', 'X travellers'). Return number (e.g., '7', '2'). 'N/A' if none.
*   **Itinerary Items:** List key activities, tours, transfers, requests (e.g., 'best price', 'With Hotel', 'airport transfer'), hotels, sights, parks, lookouts, AND non-major-destination locations. Comma-separated. 'N/A' if none.
*   **Category:** " . $category_constraint_text . " /* THIS IS A CRITICAL FIELD CONSTRAINED BY MAILBOX CONTEXT */

**Phase 4: Reasoning**

*   **Crucial:** For *every* field in the JSON, provide a detailed step-by-step reasoning in the `reasoning` field. Explain *how* the value was derived:
    *   For `travel_date`: Explain where the date was found or why it's 'N/A'.
    *   For `duration`, `cities`, `country`: Explain the source (e.g., 'extracted directly from '3 nights Sydney' text', 'calculated nights based on Oct 1-4 dates', 'identified cities X, Y and generated N/A list for duration to match count'). Explicitly state if the fallback logic (Phase 2) was necessary because no structured segments were found.
    *   For `seaters`, `category`, `itinerary_items`: Explain the evidence found in the text (e.g., 'found text 'No of Pax - 2'', 'category based on mention of 'private tour' AND mailbox context allows private', 'items listed are tours and activities mentioned'). Specifically for 'category', explain how the mailbox constraint was applied if applicable.

**General Notes:**

*   Prioritize the most current information.
*   Infer the primary country if needed.
*   Ignore greetings, closings, signatures, filler. Focus on itinerary details.
*   **Final Check:** Before outputting the JSON, perform one last explicit check that the number of comma-separated elements in `duration`, `cities`, `country` are identical, and that the `category` adheres to the mailbox constraints if specified.
";
}

/**
 * Sends prompt to Gemini AI for segment extraction, returns JSON response string.
 */
function promptGeminiAI_forSegments(string $prompt): string {
    $apiUrl = apiURL_Flash();
    if (!$apiUrl) {
        error_log("promptGeminiAI_forSegments: API URL is missing.");
        return json_encode(['error' => 'API URL not configured for Gemini.']);
    }

    $payload = [
        'contents' => [['parts' => [['text' => $prompt]]]],
        'generationConfig' => ['temperature' => 0.3, 'response_mime_type' => "application/json", 'maxOutputTokens' => 50000]
    ];
    $headers = ['Content-Type: application/json'];

    $ch = curl_init($apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 20);
    curl_setopt($ch, CURLOPT_TIMEOUT, 100);

    $response_body = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    if ($curlError || $httpCode != 200) {
        $error_message = "Gemini API Error (Segments HTTP $httpCode, cURL $curlError): " . $response_body;
        error_log($error_message);
        $details = json_decode($response_body, true);
        return json_encode(['error' => "Gemini API Error (Segments HTTP $httpCode)", 'details' => $details ?: strip_tags($response_body)]);
    }

    $responseData = json_decode($response_body, true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("Gemini JSON Decode Error (Segments main - outer response): " . json_last_error_msg() . " | Raw: " . $response_body);
        return json_encode(['error' => 'Gemini API: Failed to decode main JSON (Segments outer response)', 'raw_response' => $response_body]);
    }

    if (isset($responseData['promptFeedback']['blockReason'])) {
        $blockReason = $responseData['promptFeedback']['blockReason'] ?? 'Unknown';
        error_log("Gemini Content Blocked (Segments): " . $blockReason);
        return json_encode(['error' => 'Gemini: Content blocked (Segments)', 'details' => $responseData['promptFeedback']]);
    }

    if (isset($responseData['candidates'][0]['content']['parts']) && is_array($responseData['candidates'][0]['content']['parts'])) {
        $finalJsonText = null;
        $allPartsCombinedTextForLog = "";

        foreach ($responseData['candidates'][0]['content']['parts'] as $partIndex => $part) {
            $currentText = $part['text'] ?? '';
            $allPartsCombinedTextForLog .= "Part #{$partIndex}: " . $currentText . "\n---\n";

            if (isset($part['text'])) {
                $isThoughtPart = isset($part['thought']) && $part['thought'] === true;

                if (!$isThoughtPart) {
                    $trimmedPartText = trim($currentText);
                    if (substr($trimmedPartText, 0, 7) === "```json") {
                        $trimmedPartText = substr($trimmedPartText, 7);
                        if (substr($trimmedPartText, -3) === "```") {
                            $trimmedPartText = substr($trimmedPartText, 0, -3);
                        }
                        $trimmedPartText = trim($trimmedPartText);
                    } elseif (strpos($trimmedPartText, '```') === 0 && strrpos($trimmedPartText, '```') === (strlen($trimmedPartText) - 3) ) {
                        $trimmedPartText = substr($trimmedPartText, 3, -3);
                        $trimmedPartText = trim($trimmedPartText);
                    }

                    json_decode($trimmedPartText);
                    if (json_last_error() === JSON_ERROR_NONE) {
                        $finalJsonText = $trimmedPartText;
                        // error_log("promptGeminiAI_forSegments: Successfully extracted JSON from part #{$partIndex}.");
                        break;
                    } else {
                         error_log("promptGeminiAI_forSegments: Part #{$partIndex} was not valid JSON. Error: ".json_last_error_msg().". Content: ".substr($trimmedPartText, 0, 200)."...");
                    }
                } else {
                     // error_log("promptGeminiAI_forSegments: Part #{$partIndex} was marked as a 'thought'. Skipping direct JSON use for now.");
                }
            }
        }

        if ($finalJsonText !== null) {
            return $finalJsonText;
        } else {
            error_log("Gemini response did not contain a valid JSON part after checking all parts. All parts combined: " . $allPartsCombinedTextForLog . "Raw full response: " . $response_body);
            return json_encode(['error' => 'Gemini API: No valid JSON found in response parts', 'raw_response' => $response_body, 'all_parts_text' => $allPartsCombinedTextForLog]);
        }
    }

    error_log("Gemini response format unexpected (Segments - 'parts' structure missing or issue in processing). Raw: " . $response_body);
    return json_encode(['error' => 'Gemini API: Unexpected response format (parts structure issue)', 'raw_response' => $response_body]);
}

/**
 * Fetches latest message details for segment extraction context.
 */
function getMessageFromConversationID_forSegments(string $conversationID, mysqli $conn): ?array {
    if (empty($conversationID) || !$conn) return null;
    $conversationID_safe = mysqli_real_escape_string($conn, $conversationID);
    $sql = "SELECT subject, plaintext_body, full_body, sender
            FROM tdu_emails
            WHERE conversation_id = '$conversationID_safe'
            ORDER BY received_datetime DESC
            LIMIT 1";
    $result = mysqli_query($conn, $sql);
    if ($result && mysqli_num_rows($result) > 0) {
        $messageData = mysqli_fetch_assoc($result);
        mysqli_free_result($result);
        return $messageData;
    } elseif (!$result) {
        error_log("DB Error fetching message for Segments (Conv ID $conversationID_safe): " . mysqli_error($conn));
    }
    return null;
}

/**
 * Fetches product and vendor names for a template ID.
 */
function getTemplateItems_forSegments(int $templateID, mysqli $conn): array {
     $items = [];
     if (empty($templateID) || !is_numeric($templateID) || !$conn) return $items;
     $templateID_safe = (int)$templateID;
     $sql = "SELECT p.productName, v.vendorName
             FROM tdu_template_products tp
             LEFT JOIN tdu_products p ON tp.productid = p.productid
             LEFT JOIN tdu_vendors v ON p.vendorid = v.vendorid
             WHERE tp.templateid = $templateID_safe
             ORDER BY CAST(tp.day AS UNSIGNED) ASC, tp.sequence_no ASC";
     $result = mysqli_query($conn, $sql);
     if ($result) {
         while ($row = mysqli_fetch_assoc($result)) { $items[] = $row; }
         mysqli_free_result($result);
     } else {
         error_log("DB Error fetching template items for Segments (ID $templateID_safe): " . mysqli_error($conn));
     }
     return $items;
}

/**
 * Searches for the most relevant active template for segment matching.
 */
function relevancy_search_template_forSegments(string $sender_email, string $seaters, string $duration, string $country, string $tour_type_from_main_ai, string $cities, string $itinerary_items, mysqli $conn, string $categoryFilterToUse): ?array {
    if (!$conn) { error_log("relevancy_search_template_forSegments: Invalid DB connection."); return null; }

    $where_clause_parts = ["t.active = 1"];

    $duration_clean = trim($duration);
    if ($duration_clean != 'N/A' && is_numeric($duration_clean) && intval($duration_clean) > 0) {
        $where_clause_parts[] = "t.nights = " . intval($duration_clean);
    }

    $country_clean = trim($country);
    if ($country_clean != 'N/A' && !empty($country_clean)) {
        $where_clause_parts[] = "t.country = '" . mysqli_real_escape_string($conn, $country_clean) . "'";
    }

    // Use the rule-adjusted category for filtering templates
    if (!empty($categoryFilterToUse)) {
        $cats = [];
        if (strtolower($categoryFilterToUse) === 'sic') $cats[] = "t.category = 'SIC'";
        elseif (strtolower($categoryFilterToUse) === 'private') $cats[] = "t.category = 'Private'";
        elseif (strtolower($categoryFilterToUse) === 'group') $cats[] = "t.category = 'Group'";
        elseif (strtolower($categoryFilterToUse) === 'sic+private') $cats[] = "t.category = 'SIC+Private'";
        if (!empty($cats)) {
            $where_clause_parts[] = "(" . implode(" OR ", $cats) . ")";
        }
    }

    $ai_seats_num = ($seaters != 'N/A' && is_numeric($seaters)) ? intval($seaters) : 0;
    if ((strtolower($categoryFilterToUse) != 'sic' || empty($categoryFilterToUse)) && $ai_seats_num > 0) {
        $where_clause_parts[] = "t.seaters >= " . $ai_seats_num;
    }

    $city_clean = trim($cities);
    if ($city_clean != 'N/A' && !empty($city_clean)) {
        $where_clause_parts[] = "t.city LIKE '%" . mysqli_real_escape_string($conn, $city_clean) . "%'";
    }

    $base_sql = "SELECT t.* FROM tdu_templates t WHERE " . implode(" AND ", $where_clause_parts);
    $min_seaters_val = null;

    if ((strtolower($categoryFilterToUse) != 'sic' || empty($categoryFilterToUse)) && $ai_seats_num > 0) {
        $min_seaters_sql = "SELECT MIN(sub.seaters) FROM ($base_sql) as sub WHERE sub.seaters >= $ai_seats_num";
        $min_result = mysqli_query($conn, $min_seaters_sql);
        if ($min_result) {
            $min_row = mysqli_fetch_row($min_result);
            if ($min_row && $min_row[0] !== null) { $min_seaters_val = intval($min_row[0]); }
            mysqli_free_result($min_result);
        } else {
            error_log("DB Error finding min seaters for Segments: " . mysqli_error($conn));
        }
    }

    $potential_templates = [];
    $final_sql = $base_sql;
    if ($min_seaters_val !== null && (strtolower($categoryFilterToUse) != 'sic' || empty($categoryFilterToUse))) {
        $final_sql .= " AND t.seaters = " . $min_seaters_val;
    }
    $final_sql .= " ORDER BY t.preferred DESC, t.templateid ASC";

    $result = mysqli_query($conn, $final_sql);
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) { $potential_templates[] = $row; }
        mysqli_free_result($result);
    } else {
        error_log("DB Error fetching potential templates for Segments SQL [$final_sql]: " . mysqli_error($conn));
        return null;
    }

    if (empty($potential_templates)) return null;

    $gemini_itinerary_array = [];
    if ($itinerary_items != 'N/A' && !empty(trim($itinerary_items))) {
        $gemini_itinerary_array = array_filter(array_map('trim', explode(',', $itinerary_items)));
        $gemini_itinerary_array = array_map('strtolower', $gemini_itinerary_array);
    }

    $combined_results = [];
    foreach ($potential_templates as $template) {
        $item_relevance_score = 0;
        $template_items = getTemplateItems_forSegments($template['templateid'], $conn);
        $template['template_items'] = $template_items;
        if (!empty($gemini_itinerary_array) && !empty($template_items)) {
            $template_item_content = strtolower($template['templatename'] ?? '');
            foreach ($template_items as $item) {
                $template_item_content .= " " . strtolower(($item['productName'] ?? '') . " " . ($item['vendorName'] ?? ''));
            }
            $template_item_content = preg_replace('/\s+/', ' ', trim($template_item_content));
            foreach ($gemini_itinerary_array as $gemini_item) {
                if (!empty($gemini_item) && stripos($template_item_content, $gemini_item) !== false) {
                    $item_relevance_score++;
                }
            }
        }
        $template['relevance'] = $item_relevance_score;
        $template['preferred'] = $template['preferred'] ?? 0;
        $combined_results[] = $template;
    }

    usort($combined_results, function($a, $b) {
        if ($b['preferred'] != $a['preferred']) return $b['preferred'] - $a['preferred'];
        return $b['relevance'] - $a['relevance'];
    });

    return $combined_results[0] ?? null;
}

/**
 * Logs AI processing error to tdu_ai_error_log for segment extraction.
 */
function log_ai_error_forSegments(mysqli $conn, string $conversationId, ?int $aiRef, string $reason, string $type, string $status): bool {
    $tableName = 'tdu_ai_error_log';
    if (!$conn || empty($conversationId) || empty($reason) || empty($type) || empty($status)) {
        error_log("log_ai_error_forSegments: Missing params.");
        return false;
    }
    $insertSql = "INSERT INTO {$tableName} (ai_ref, conversation_id, reason, type, status, created_at) VALUES (?, ?, ?, ?, ?, NOW())";
    $stmt = mysqli_prepare($conn, $insertSql);
    if ($stmt) {
        mysqli_stmt_bind_param($stmt, "issss", $aiRef, $conversationId, $reason, $type, $status);
        if (mysqli_stmt_execute($stmt)) {
            mysqli_stmt_close($stmt);
            return true;
        } else {
            error_log("log_ai_error_forSegments: DB Error (Execute {$tableName}): " . mysqli_stmt_error($stmt) . " | Conv ID: " . $conversationId);
            mysqli_stmt_close($stmt);
        }
    } else {
        error_log("log_ai_error_forSegments: DB Error (Prepare {$tableName}): " . mysqli_error($conn) . " | Conv ID: " . $conversationId);
    }
    return false;
}

/**
 * Main function to extract travel segments and match templates.
 * MODIFIED: Added $mailbox parameter and logic to apply category rules.
 */
function extractTravelSegmentsAndMatchTemplates(string $conversationID, mysqli $conn, string $userSelectedCategory = '', ?string $mailbox = null): array {
    $default_return = [
        'success' => false, 'error' => null, 'details' => null, 'segments' => null,
        'raw_ai_response' => null, 'ai_parsed_data' => null, 'message' => null,
        'category_used_for_filter' => null, // Will store the final category after rules
        'initial_ai_category_log' => null, // Will store the log of AI determination and rule application
        'ai_detailed_category' => null // From segment AI (if different)
    ];

    if (!$conn || $conn->connect_error) {
         $default_return['error'] = 'Database connection failed: ' . ($conn->connect_error ?? 'Unknown DB error');
         error_log("extractTravelSegmentsAndMatchTemplates: " . $default_return['error']);
         return $default_return;
    }
    if (empty($conversationID)) {
        $default_return['error'] = 'Conversation ID not provided.';
        return $default_return;
    }

    $latestMessageForContext = getMessageFromConversationID_forSegments($conversationID, $conn);
    if (!$latestMessageForContext) {
        $default_return['error'] = "No messages found for Conversation ID: " . htmlspecialchars($conversationID);
        return $default_return;
    }

    $subjectForAI = $latestMessageForContext['subject'] ?? 'N/A';
    $sender_email = $latestMessageForContext['sender'] ?? 'N/A';

    // Fetch all conversation content for segment AI
    $allConversationMessagesAndAttachments = getAllEmailMessagesAndAttachmentsInConversation($conversationID, $conn);
    $fullContentForSegmentAI = "";
    foreach($allConversationMessagesAndAttachments['emails'] as $email) {
        $fullContentForSegmentAI .= "\n\n--- Email Start (Received: " . $email['received_datetime'] . ") ---\n";
        $fullContentForSegmentAI .= "Subject: " . $email['subject'] . "\n";
        $fullContentForSegmentAI .= "Body:\n" . $email['body_clean'] . "\n";
        $fullContentForSegmentAI .= "--- Email End ---\n";
    }
    foreach ($allConversationMessagesAndAttachments['attachments'] as $attachment) {
        if (!$attachment['error'] && !empty($attachment['extracted_text'])) {
            $fullContentForSegmentAI .= "\n\n--- Attachment: " . htmlspecialchars($attachment['filename']) . " (From email received: " . $attachment['parent_email_received_datetime'] . ") ---\n";
            $fullContentForSegmentAI .= $attachment['extracted_text'] . "\n";
            $fullContentForSegmentAI .= "--- Attachment End ---\n";
        }
    }
    $fullContentForSegmentAI = trim($fullContentForSegmentAI);


    if (empty($fullContentForSegmentAI) && ($subjectForAI === 'N/A' || empty($subjectForAI))) {
        $default_return['error'] = 'Cannot analyze: Email bodies, subject, and attachments are effectively empty for segment extraction.';
        error_log("extractTravelSegmentsAndMatchTemplates: " . $default_return['error'] . " for ConvID {$conversationID}");
        return $default_return;
    }

    // --- Category Determination with Mailbox Rules ---
    $categoryForFilter = '';
    $initial_ai_category_determination_log = "";
    $latestEmailBodyForCategory = $latestMessageForContext['plaintext_body'] ?? ($latestMessageForContext['full_body'] ?? '');
    $bodyForCategoryAI = $latestEmailBodyForCategory; // Use cleaned body for category AI

    $categoryForFilter = $aiDetailedCategoryFromSegments; // Using the category from segment AI
    if (empty($categoryForFilter) || $categoryForFilter === 'N/A') {
        // Fallback if segment AI fails to provide a category or if you want to apply a default
        if ($mailbox === 'sales@turtledownunder.com.au') {
            $categoryForFilter = 'private'; // Or 'sic' as a general default for sales
             error_log("extractTravelSegmentsAndMatchTemplates: Segment AI category N/A for sales mailbox, defaulting to 'private' for template filter.");
        } elseif ($mailbox === 'groupsales@turtledownunder.com.au') {
            $categoryForFilter = 'group';
             error_log("extractTravelSegmentsAndMatchTemplates: Segment AI category N/A for groupsales mailbox, defaulting to 'group' for template filter.");
        } else {
            $categoryForFilter = 'private'; // General default
             error_log("extractTravelSegmentsAndMatchTemplates: Segment AI category N/A, defaulting to 'private' for template filter.");
        }
    }
    $default_return['category_used_for_filter'] = $categoryForFilter;
    $default_return['initial_ai_category_log'] = $initial_ai_category_determination_log;
    // --- End Category Determination ---

    $geminiStructuredPrompt = gemini_instructions_forSegments($mailbox) . // <<< MODIFIED HERE
                              "\nConversation Subject (latest): " . $subjectForAI . 
                              "\nFull Conversation Body and Attachment Text:\n" . $fullContentForSegmentAI;

    $rawGeminiResponseString = promptGeminiAI_forSegments($geminiStructuredPrompt);
    $default_return['raw_ai_response'] = $rawGeminiResponseString;
    $structuredDataArray = json_decode($rawGeminiResponseString, true);
    $default_return['ai_parsed_data'] = $structuredDataArray;

    if ($structuredDataArray === null || json_last_error() !== JSON_ERROR_NONE) {
        $default_return['error'] = 'Error decoding Gemini JSON response (Segments).';
        $default_return['details'] = 'Raw AI response was not valid JSON. Error: ' . json_last_error_msg();
        error_log("extractTravelSegmentsAndMatchTemplates: " . $default_return['error'] . $default_return['details'] . " | Conv ID: " . $conversationID);
        return $default_return;
    }
    if (isset($structuredDataArray['error'])) {
        $default_return['error'] = $structuredDataArray['error'];
        $default_return['details'] = is_string($structuredDataArray['details'] ?? null) ? $structuredDataArray['details'] : json_encode($structuredDataArray['details'] ?? null);
        error_log("extractTravelSegmentsAndMatchTemplates: Gemini API Error (Segments): " . $default_return['error'] . " | Details: " . $default_return['details'] . " | Conv ID: " . $conversationID);
        return $default_return;
    }

    // This is the category from the *segment extraction* AI, which might be different from the initial classification
    $aiDetailedCategoryFromSegments = $structuredDataArray['category'] ?? ($structuredDataArray['category(testing)'] ?? 'N/A');
    $default_return['ai_detailed_category'] = $aiDetailedCategoryFromSegments;
    $seaters = $structuredDataArray['seaters'] ?? 'N/A';
    $durationStr = $structuredDataArray['duration'] ?? 'N/A';
    $citiesStr = $structuredDataArray['cities'] ?? 'N/A';
    $countryStr = $structuredDataArray['country'] ?? 'N/A';
    $itinerary_items_ai = $structuredDataArray['itinerary_items'] ?? 'N/A';
    $default_return['itinerary_items_used'] = $itinerary_items_ai;

    $segmentResults = [];
    $is_segmented = (strpos($durationStr, ',') !== false && strpos($citiesStr, ',') !== false && strpos($countryStr, ',') !== false);
    $force_process_as_segments = true; // Process even if not explicitly comma-separated, to handle single segment cases

    if ($is_segmented || $force_process_as_segments) {
        $durations = array_values(array_filter(array_map('trim', explode(',', $durationStr))));
        $cities = array_values(array_filter(array_map('trim', explode(',', $citiesStr))));
        $countries = array_values(array_filter(array_map('trim', explode(',', $countryStr))));
        $segment_count = count($cities);
        // Ensure all arrays are at least of size 1 if their string was not 'N/A' to begin with
        if ($segment_count === 0 && $citiesStr !== 'N/A' && !empty($citiesStr)) $segment_count = 1;
        if (empty($durations) && $durationStr !== 'N/A' && !empty($durationStr)) $durations = [$durationStr];
        if (empty($cities) && $citiesStr !== 'N/A' && !empty($citiesStr)) $cities = [$citiesStr];
        if (empty($countries) && $countryStr !== 'N/A' && !empty($countryStr)) $countries = [$countryStr];

        // Re-evaluate counts after potential single-element adjustments
        $segment_count = max(count($cities), count($durations), count($countries));
        if ($segment_count == 0 && ($citiesStr != 'N/A' || $durationStr != 'N/A' || $countryStr != 'N/A') && (!empty($citiesStr) || !empty($durationStr) || !empty($countryStr)) ){
             $segment_count = 1; // If any field has a non-N/A value, assume at least one segment.
        }


        // Pad arrays to match the max segment_count with 'N/A' if needed, for consistency in loop
        $durations = array_pad($durations, $segment_count, 'N/A');
        $cities = array_pad($cities, $segment_count, 'N/A');
        $countries = array_pad($countries, $segment_count, 'N/A');

        $counts_match = (count($durations) === $segment_count && count($countries) === $segment_count && count($cities) === $segment_count);


        if ($segment_count > 0 && $counts_match) {
            for ($i = 0; $i < $segment_count; $i++) {
                $current_duration = $durations[$i] ?? 'N/A';
                $current_city = $cities[$i] ?? 'N/A';
                $current_country = $countries[$i] ?? 'N/A';

                // Use $categoryForFilter (rule-adjusted) for template searching.
                // $aiDetailedCategoryFromSegments is from the segment AI, $categoryForFilter is from initial AI + rules.
                // The prompt for relevancy_search_template_forSegments uses $tour_type_from_main_ai, which should be $categoryForFilter.
                $best_template_details = relevancy_search_template_forSegments(
                    $sender_email, $seaters, $current_duration, $current_country,
                    $categoryForFilter, // Use the rule-adjusted category here
                    $current_city, $itinerary_items_ai, $conn, $categoryForFilter // Pass it as $categoryFilterToUse as well
                );
                if ($best_template_details === null) {
                    $error_reason = sprintf("No template for seg %d (Dur=%s, City=%s, Country=%s, Seats=%s, FilterCatUsed=%s, SegAICat=%s)",
                        $i + 1, $current_duration, $current_city, $current_country, $seaters,
                        empty($categoryForFilter) ? "N/A" : $categoryForFilter, $aiDetailedCategoryFromSegments);
                    log_ai_error_forSegments($conn, $conversationID, null, $error_reason, 'AUTO_NO_TEMPLATE', 'PENDING');
                }
                $segmentResults[] = ['segment_number' => $i + 1, 'input_duration' => $current_duration, 'input_city' => $current_city, 'input_country' => $current_country, 'matched_template' => $best_template_details];
            }
        } elseif ($segment_count > 0 && !$counts_match) { // Mismatch in counts
            $error_reason = sprintf("Segment count mismatch. Dur: %d, Cities: %d, Countries: %d. Raw D:'%s', C:'%s', Co:'%s'",
                count($durations), count($cities), count($countries), $durationStr, $citiesStr, $countryStr);
            error_log("[AI Segments - Mismatch] ConvID: {$conversationID} | {$error_reason}");
            log_ai_error_forSegments($conn, $conversationID, null, $error_reason, 'AUTO_MISMATCH', 'PENDING');
            $default_return['error'] = 'Segmented data count mismatch in AI response.';
            $default_return['details'] = $error_reason;
            // Do not return immediately, let it try to process what it can or return empty segments if it falls through
        } else {
            // This case means $segment_count is 0, or some other non-processing scenario.
            // If AI returns N/A for all (duration, cities, country), this will be hit.
            // error_log("[AI Segments - Non-Processable or No Segments] ConvID: {$conversationID} | Dur: '{$durationStr}', Cities: '{$citiesStr}', Country: '{$countryStr}'");
        }
    } else { // Should not be hit if $force_process_as_segments = true
         // error_log("[AI Segments - Not Segmented by AI criteria] ConvID: {$conversationID} | Raw D:'{$durationStr}', C:'{$citiesStr}', Co:'{$countryStr}'");
    }


    $default_return['success'] = true;
    $default_return['segments'] = $segmentResults; // Will be empty if no segments were processed
    $default_return['message'] = empty($segmentResults) ? 'AI analysis processed, but no valid segments with matching templates were finalized.' : 'AI analysis and template matching complete.';
    return $default_return;
}


/**
 * Fetches email data, extracts attachment text, calls AI for booking/quote/invoice details.
 */
function ai_confirmationExtract($conversationID, $email_domain, $date, array $products, string $signature, mysqli $conn){
    if (empty($conversationID) || !$conn) {
        error_log("ai_confirmationExtract: Missing conversationID or DB connection.");
        return NULL;
    }
    $messageData = null; $allMessagesData = []; $extractedText = []; $vendorList = [];
    $conversationID_safe = mysqli_real_escape_string($conn, $conversationID);
    $testid='13'; // This seems to be a hardcoded ID for fetching vendors, which is unusual

    // Fetch vendor names associated with a specific test ID
    $vendorSql = "SELECT v.vendorName FROM vtiger_inventoryproductrel i LEFT JOIN tdu_vendors v ON i.vendorid = v.vendorid WHERE i.id = '$testid'";
    $vendorResult = mysqli_query($conn, $vendorSql);
    if ($vendorResult) {
        while ($vendorRow = mysqli_fetch_assoc($vendorResult)) {
            if (!empty($vendorRow['vendorName'])) { $vendorList[] = $vendorRow['vendorName']; }
        }
        mysqli_free_result($vendorResult);
    } else {
        error_log("ai_confirmationExtract: DB Error fetching vendor list for Test ID $testid: " . mysqli_error($conn));
    }

    // Fetch all emails and their attachments for the conversation
    $sql = "SELECT e.subject, e.plaintext_body, e.full_body, e.sender, a.attachment_id, a.file_name, e.message_id as messageID, e.msgid, e.received_datetime
            FROM tdu_emails e LEFT JOIN tdu_attachments a ON e.message_id = a.message_id
            WHERE e.conversation_id = '$conversationID_safe' ORDER BY e.received_datetime DESC";
    $result = mysqli_query($conn, $sql);
    if (!$result) {
        error_log("ai_confirmationExtract: DB Error fetching messages for Conv ID $conversationID_safe: " . mysqli_error($conn));
        return NULL;
    }
    if (mysqli_num_rows($result) > 0) {
        while ($row = mysqli_fetch_assoc($result)) { $allMessagesData[] = $row; }
    }
    mysqli_free_result($result);
    if (empty($allMessagesData)) {
        error_log("ai_confirmationExtract: No messages found for Conv ID $conversationID_safe.");
        return NULL;
    }
    $messageData = $allMessagesData[0]; // Use the latest message for primary subject/body context

    // Process attachments from all messages in the conversation
    $processedAttachments = []; // To avoid processing the same attachment multiple times if linked to multiple emails (though unlikely with current schema)
    foreach ($allMessagesData as $message_entry) {
        if (!empty($message_entry['attachment_id']) && !isset($processedAttachments[$message_entry['attachment_id']])) {
            $processedAttachments[$message_entry['attachment_id']] = true;
            $endpoint = "https://dashboard.turtledownunder.com.au/ajax_outlook_attachment_download.php?attachment_id=".urlencode($message_entry['attachment_id'])."&message_id=".urlencode($message_entry['msgid'])."&file_name=".urlencode($message_entry['file_name']);

            $ch = curl_init($endpoint);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true); curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            $response = curl_exec($ch); $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE); $curlError = curl_error($ch);
            curl_close($ch);

            if ($curlError || $httpCode !== 200) {
                error_log("ai_confirmationExtract: cURL/HTTP Error downloading attach ID {$message_entry['attachment_id']}: $curlError, HTTP $httpCode");
            } else {
                $fileType = function_exists('getFileTypeFromString') ? getFileTypeFromString($response) : 'unknown'; $text = '';
                try {
                    if ($fileType == "PDF Document" && function_exists('extractTextFromPDF')) { $text = extractTextFromPDF($response); }
                    elseif (($fileType == "Word Document (Modern Format - DOCX)" || $fileType == "Word Document (Legacy Format - DOC)") && function_exists('extractTextfromWordDoc')) { $text = extractTextfromWordDoc($response); }
                    elseif ($fileType == "Plain Text File" || $fileType == "HTML Web Page File") { $text = strip_tags($response); }
                } catch (Exception $e) {
                     error_log("ai_confirmationExtract: Error extracting text from attach ID {$message_entry['attachment_id']} (Type: $fileType): " . $e->getMessage());
                }
                if (!empty(trim($text))) {
                    $extractedText[] = "--- Attachment: " . ($message_entry['file_name'] ?? 'Unknown') . " (ID: " . $message_entry['attachment_id'] . ") ---\n" . trim($text);
                }
            }
        }
    }

    // Call AI with context from the latest email and text from all attachments
    $AI_response = findConfirmationNumber(
        $email_domain, $date, $products, $signature, $vendorList, // $vendorList is from the hardcoded $testid
        $messageData['subject'] ?? '',
        $messageData['full_body'] ?? $messageData['plaintext_body'] ?? '',
        implode("\n\n", $extractedText)
    );

    // Example of how invoice data might be used (currently commented out in original)
    if (isset($AI_response['isInvoice']) && $AI_response['isInvoice'] === 'yes' && isset($AI_response['invoiceNumber']) && $AI_response['invoiceNumber'] !== 'N/A' && isset($AI_response['invoiceTotal']) && $AI_response['invoiceTotal'] !== 'N/A' && isset($AI_response['invoiceVendorName']) && $AI_response['invoiceVendorName'] !== 'N/A' ) {
        // error_log("ai_confirmationExtract: Invoice detected. Vendor: " . $AI_response['invoiceVendorName']);
        // Potentially save invoice details to a database here.
    }
    return $AI_response;
}

/**
 * Analyzes email content using AI to extract booking, quote, invoice details, and vendor.
 */
function findConfirmationNumber(string $email_domain, string $date, array $products, string $signature, array $validVendorNames, string $email_subject = "", string $email_body = "", string $attachment_text = ""): array {
    $full_email_content = trim($email_subject . "\n\n" . $email_body . "\n\n" . $attachment_text);
    $default_response_structure = [
        'confirmationNumber' => 'N/A', 'time' => 'N/A', 'description' => 'N/A',
        'quoteNumber' => 'N/A', 'matches' => 'no', 'log' => 'N/A',
        'isInvoice' => 'no', 'invoiceNumber' => 'N/A', 'invoiceTotal' => 'N/A',
        'invoiceVendorName' => 'N/A'
    ];

    if (empty(trim($full_email_content))) {
        $default_response_structure['log'] = 'Validation failed: No email content provided.';
        foreach (array_keys($default_response_structure) as $key) { if ($key !== 'log' && $key !== 'matches' && $key !== 'isInvoice') $default_response_structure[$key] = 'N/A: No input content';}
        return $default_response_structure;
    }

    $products_string = !empty($products) ? implode(', ', $products) : 'N/A';
    $vendorListString = !empty($validVendorNames) ? "\nValid Vendor Names: [" . implode(', ', array_map(function($name) { return "'".addslashes($name)."'"; }, $validVendorNames)) . "]" : "\nValid Vendor Names: N/A";

    $preprompt = "Analyze 'Email Content'. Extract booking, quote, and invoice details. Validate against 'Context'. Choose vendor ONLY from 'Valid Vendor Names' list if applicable.\n\nTasks:\n1. **Booking Confirmation:** Extract primary **confirmation number** (external preferred).\n2. **Booking Time:** Extract primary **start time/date**.\n3. **Booking Description:** Extract concise **description** (supplier, location).\n4. **Quote Number:** Extract the quote number. It always starts with 'TDU or tdu'(convert to capital letter) and is frequently in the email subject. Try to find and use the latest confirmed one found in the email chain. If not found, use 'N/A' .\n5. **Invoice Detection:** Is this primarily an **invoice**, bill, or receipt? Output 'yes'/'no' for `isInvoice`.\n6. **Invoice Number:** If `isInvoice`='yes', extract **invoice number**. Else 'N/A'.\n7. **Invoice Total:** If `isInvoice`='yes', extract **invoice total** (NUMERIC VALUE ONLY, e.g., '123.45'). No symbols/commas. Else 'N/A'.\n8. **Invoice Vendor Name:** If `isInvoice`='yes', identify the vendor. If the vendor's name exactly matches one in the 'Valid Vendor Names' list provided in the Context, output that exact name. Otherwise (if no match in the list or not an invoice), output 'N/A'.\n9. **Context Validation:** Does content seem relevant/consistent with 'Context' (domain, date, product, signature)? Output 'yes'/'no' for `matches`.\n10. **Validation Log:** Briefly explain `matches` decision. State mismatches if 'no'.\n\nOutput Format Constraint:\nFormat ENTIRE output STRICTLY as a single line tuple:\n('confirmationNumber_value', 'time_value', 'description_value', 'quoteNumber_value', 'matches_value', 'log_value', 'isInvoice_value', 'invoiceNumber_value', 'invoiceTotal_value', 'invoiceVendorName_value')\n\nFormatting Rules:\n- Use 'N/A' (string, single quotes) for undetermined/inapplicable values.\n- `isInvoice`: 'yes' or 'no' (lowercase, single quotes).\n- `invoiceTotal`: NUMERIC (digits, optional decimal point) or 'N/A', single quotes.\n- `invoiceVendorName`: MUST be from the 'Valid Vendor Names' list (in Context) or 'N/A'.\n- Use ONLY single quotes (') for values and outer parentheses (). NO double quotes (\").\n- Separate elements ONLY with comma + single space (', ').\n- NO other text, explanations, markdown, code blocks, line breaks before/after tuple string.\n\nExamples:\nEx 1 (Booking+Quote): ('PPNP-56789', '2025-04-13 18:30', '17x Penguin Premium tickets', 'tdu12345', 'yes', 'Match confirmed.', 'no', 'N/A', 'N/A', 'N/A')\nEx 2 (Invoice from List): ('N/A', 'N/A', 'Service Invoice', 'N/A', 'yes', 'Match confirmed. Vendor from list.', 'yes', 'INV-2024-00123', '75.50', 'Valid Vendor A')\nEx 3 (Invoice not on List): ('N/A', 'N/A', 'Hardware Invoice', 'tdu-q987', 'yes', 'Match confirmed, but vendor \\'Some Hardware Inc\\' not in Valid list.', 'yes', 'HW-5566', '1500.00', 'N/A')\nEx 4 (Irrelevant): ('N/A', 'N/A', 'Newsletter', 'N/A', 'no', 'Match failed: Newsletter content.', 'no', 'N/A', 'N/A', 'N/A')\nEx 5 (Invoice+Booking): ('BOOK-T567', '2024-11-01', 'Hotel Stay Invoice - Grand Hyatt', 'tdu77889', 'yes', 'Match confirmed. Vendor \\'Grand Hyatt Sydney\\' matches list.', 'yes', 'GH-INV-9876', '1250.00', 'Grand Hyatt Sydney')\n\nContext:\nClient Email Domain: $email_domain\nProduct Start Date: $date\nProduct title/description: $products_string\nClient Signature: $signature$vendorListString\n\nEmail Content:\n";
    $prompt_for_ai_utf8 = $preprompt . $full_email_content;
    if (!mb_check_encoding($prompt_for_ai_utf8, 'UTF-8')) {
        $detected_encoding = mb_detect_encoding($prompt_for_ai_utf8, 'UTF-8, Windows-1252, ISO-8859-1', true);
        $converted = mb_convert_encoding($prompt_for_ai_utf8, 'UTF-8', $detected_encoding ?: 'UTF-8');
        if ($converted === false) { error_log("findConfirmationNumber Error: Failed to convert prompt to UTF-8 from '$detected_encoding'."); return ['error' => 'Data encoding error.']; }
        $prompt_for_ai_utf8 = $converted;
    }

    $apiUrl = apiURL_Flash();
    if (!$apiUrl) { error_log("findConfirmationNumber: API URL is missing."); return ['error' => 'API URL not configured.']; }
    $payload = ['contents' => [['parts' => [['text' => $prompt_for_ai_utf8]]]], 'generationConfig' => ['temperature' => 0.2, 'maxOutputTokens' => 50000, 'topP' => 0.95, 'topK' => 40]];
    $headers = ['Content-Type: application/json'];

    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [CURLOPT_RETURNTRANSFER => true, CURLOPT_POST => true, CURLOPT_POSTFIELDS => json_encode($payload), CURLOPT_HTTPHEADER => $headers, CURLOPT_CONNECTTIMEOUT => 20, CURLOPT_TIMEOUT => 120]);
    $response_body = curl_exec($ch); $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE); $curlError = curl_error($ch); curl_close($ch);

    if ($curlError || $httpCode !== 200) {
        error_log("findConfirmationNumber: cURL/API Error - HTTP $httpCode, cURL: $curlError, Resp: $response_body");
        $errorDetails = json_decode($response_body, true); $errorMessage = $errorDetails['error']['message'] ?? 'Unknown API error';
        return ['error' => "API error ($httpCode): $errorMessage"];
    }

    $responseData = json_decode($response_body, true);
    if (json_last_error() !== JSON_ERROR_NONE || !isset($responseData['candidates'][0]['content']['parts'][0]['text'])) {
        error_log("findConfirmationNumber: Failed to decode API JSON or extract text. Raw: " . $response_body);
        return ['error' => 'API response parsing error.'];
    }
    $generated_text = $responseData['candidates'][0]['content']['parts'][0]['text'] ?? null;
    if ($generated_text === null) {
        error_log("findConfirmationNumber: Could not extract text. Resp: " . $response_body);
         if (isset($responseData['promptFeedback']['blockReason'])) {
            error_log("findConfirmationNumber: Gemini Content Blocked: " . ($responseData['promptFeedback']['blockReason'] ?? 'Unknown'));
             $default_response_structure['log'] = 'N/A: AI Content Blocked';
        } else {
            $default_response_structure['log'] = 'N/A: AI Text Error';
        }
        foreach (array_keys($default_response_structure) as $key) { if ($key !== 'log' && $key !== 'matches' && $key !== 'isInvoice') $default_response_structure[$key] = $default_response_structure['log'];}
        return $default_response_structure;
    }

    $trimmed_text = trim(str_replace("\xEF\xBB\xBF", '', $generated_text)); // Remove BOM
    // Regex to parse the specific tuple format, allowing for escaped single quotes within values
    $pattern = "/^\(\s*'((?:\\\\'|[^'\\\\])*)'\s*,\s*'((?:\\\\'|[^'\\\\])*)'\s*,\s*'((?:\\\\'|[^'\\\\])*)'\s*,\s*'((?:\\\\'|[^'\\\\])*)'\s*,\s*'((?:yes|no|N\/A))'\s*,\s*'((?:\\\\'|[^'\\\\])*)'\s*,\s*'((?:yes|no|N\/A))'\s*,\s*'((?:\\\\'|[^'\\\\])*)'\s*,\s*'((?:N\/A|(?:-?\d+(?:\.\d+)?)))'\s*,\s*'((?:\\\\'|[^'\\\\])*)'\s*\)$/u";
    $matches_arr = [];

    if (empty($trimmed_text) || !preg_match($pattern, $trimmed_text, $matches_arr)) {
        $parseFailReason = empty($trimmed_text) ? "AI response empty" : "AI format mismatch";
        error_log("findConfirmationNumber: Failed to parse AI response. Reason: $parseFailReason. Received: [$trimmed_text]. Pattern: $pattern");
        foreach (array_keys($default_response_structure) as $key) { if ($key !== 'log' && $key !== 'matches' && $key !== 'isInvoice') $default_response_structure[$key] = 'N/A: ' . $parseFailReason;}
        $default_response_structure['log'] = 'N/A: ' . $parseFailReason . ". AI Out: " . substr($trimmed_text, 0, 200);
        return $default_response_structure;
    }
    // Return associative array with stripslashes to handle escaped quotes from AI
    return [
        'confirmationNumber' => stripslashes($matches_arr[1]), 'time' => stripslashes($matches_arr[2]),
        'description' => stripslashes($matches_arr[3]), 'quoteNumber' => stripslashes($matches_arr[4]),
        'matches' => stripslashes($matches_arr[5]), 'log' => stripslashes($matches_arr[6]),
        'isInvoice' => stripslashes($matches_arr[7]), 'invoiceNumber' => stripslashes($matches_arr[8]),
        'invoiceTotal' => stripslashes($matches_arr[9]), 'invoiceVendorName' => stripslashes($matches_arr[10])
    ];
}

/**
 * Fetches all attachments for a conversation ID, extracts their text content.
 */
function getAttachmentFiles(string $conversationID, mysqli $conn): array {
    $attachmentItems = [];
    if (empty($conversationID) || !$conn) { error_log("getAttachmentFiles: Missing params."); return []; }
    $conversationID_safe = mysqli_real_escape_string($conn, $conversationID);

    // Select distinct attachments for the conversation
    $sql = "SELECT DISTINCT a.attachment_id, a.file_name, e.msgid
            FROM tdu_attachments a
            JOIN tdu_emails e ON a.message_id = e.message_id
            WHERE e.conversation_id = '$conversationID_safe'";
    $result = mysqli_query($conn, $sql);

    if (!$result) { error_log("getAttachmentFiles: DB Error for Conv ID $conversationID_safe: " . mysqli_error($conn)); return []; }

    if (mysqli_num_rows($result) > 0) {
        while ($row = mysqli_fetch_assoc($result)) {
            $filename = $row['file_name'] ?? 'Unknown Attachment';
            $endpoint = "https://dashboard.turtledownunder.com.au/ajax_outlook_attachment_download.php?attachment_id=".urlencode($row['attachment_id'])."&message_id=".urlencode($row['msgid'])."&file_name=".urlencode($row['file_name']);

            $ch = curl_init($endpoint);
            curl_setopt_array($ch, [CURLOPT_RETURNTRANSFER => true, CURLOPT_TIMEOUT => 60]); // Increased timeout
            $raw_content = curl_exec($ch); $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE); $curlError = curl_error($ch); curl_close($ch);

            if ($curlError || $httpCode !== 200 || empty($raw_content)) {
                error_log("getAttachmentFiles: cURL/HTTP Error downloading attach ID {$row['attachment_id']} ('{$filename}'): $curlError, HTTP $httpCode, Empty: ".(empty($raw_content)?'yes':'no'));
                continue; // Skip to next attachment
            }

            $text = '';
            $fileType = function_exists('getFileTypeFromString') ? getFileTypeFromString($raw_content) : 'unknown';
            try {
                if ($fileType == "PDF Document" && function_exists('extractTextFromPDF')) { $text = extractTextFromPDF($raw_content); }
                elseif (($fileType == "Word Document (Modern Format - DOCX)" || $fileType == "Word Document (Legacy Format - DOC)") && function_exists('extractTextfromWordDoc')) { $text = extractTextfromWordDoc($raw_content); }
                elseif ($fileType == "Plain Text File" || $fileType == "HTML Web Page File") { $text = strip_tags($raw_content); }
                // Add more types as needed (e.g., Excel, images with OCR if available)
            } catch (Exception $e) {
                error_log("getAttachmentFiles: Error extracting text from attach ID {$row['attachment_id']} ('{$filename}', Type: $fileType): " . $e->getMessage());
            }

            $trimmedText = trim($text);
            if (!empty($trimmedText)) {
                $attachmentItems[] = ['filename' => $filename, 'text' => $trimmedText];
            }
        }
        mysqli_free_result($result);
    }
    return $attachmentItems;
}


/**
 * Fetches content of the latest email and raw attachment data for the whole conversation.
 * This seems to fetch ONLY the latest email and ALL attachments.
 */
function getFullConversationWithAttachments(string $conversationID, mysqli $conn): array {
    $conversationData = ['emails' => [], 'attachments' => []];
    if (empty($conversationID) || !$conn) { error_log("getFullConversationWithAttachments: Missing params."); return $conversationData; }
    $conversationID_safe = mysqli_real_escape_string($conn, $conversationID);

    // Fetch the LATEST email in the conversation
    $emailSql = "SELECT sender, subject, received_datetime, plaintext_body, full_body, message_id
                 FROM tdu_emails
                 WHERE conversation_id = '$conversationID_safe'
                 ORDER BY received_datetime DESC
                 LIMIT 1";
    $emailResult = mysqli_query($conn, $emailSql);
    if ($emailResult && mysqli_num_rows($emailResult) > 0) {
        $emailRow = mysqli_fetch_assoc($emailResult);
        $body_content = (!empty(trim($emailRow['plaintext_body']))) ? $emailRow['plaintext_body'] : strip_tags($emailRow['full_body'] ?? '');
        $conversationData['emails'][] = [ // Stored as an array, but only one email
            'sender' => $emailRow['sender'],
            'subject' => $emailRow['subject'],
            'received_datetime' => $emailRow['received_datetime'],
            'body' => trim($body_content),
            'message_id' => $emailRow['message_id']
        ];
        mysqli_free_result($emailResult);
    } elseif (!$emailResult) {
        error_log("getFullConversationWithAttachments: DB Error fetching email for Conv ID $conversationID_safe: " . mysqli_error($conn));
    }

    // Fetch ALL distinct attachments associated with the conversation
    $attachmentSql = "SELECT DISTINCT a.attachment_id, a.file_name, e.msgid, e.message_id as parent_message_id
                      FROM tdu_attachments a
                      JOIN tdu_emails e ON a.message_id = e.message_id
                      WHERE e.conversation_id = '$conversationID_safe'";
    $attachmentResult = mysqli_query($conn, $attachmentSql);

    if (!$attachmentResult) {
        error_log("getFullConversationWithAttachments: DB Error fetching attachments for Conv ID $conversationID_safe: " . mysqli_error($conn));
        return $conversationData; // Return what we have (possibly just email)
    }

    if (mysqli_num_rows($attachmentResult) > 0) {
        while ($attRow = mysqli_fetch_assoc($attachmentResult)) {
            $filename = $attRow['file_name'] ?? 'Unknown Attachment';
            $endpoint = "https://dashboard.turtledownunder.com.au/ajax_outlook_attachment_download.php?attachment_id=".urlencode($attRow['attachment_id'])."&message_id=".urlencode($attRow['msgid'])."&file_name=".urlencode($attRow['file_name']);

            $ch = curl_init($endpoint);
            curl_setopt_array($ch, [CURLOPT_RETURNTRANSFER => true, CURLOPT_TIMEOUT => 60]);
            $raw_content = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $curlError = curl_error($ch);
            curl_close($ch);

            $attachmentEntry = [
                'attachment_id' => $attRow['attachment_id'],
                'parent_message_id' => $attRow['parent_message_id'],
                'filename' => $filename,
                'msgid_for_download' => $attRow['msgid'], // Useful for direct download later if needed
                'content' => '', // Raw content
                'error' => false,
                'error_message' => ''
            ];
            if ($curlError || $httpCode !== 200 || empty($raw_content)) {
                $error_detail = "cURL Error: $curlError, HTTP Code: $httpCode, Empty Response: ".(empty($raw_content)?'yes':'no');
                error_log("getFullConversationWithAttachments: Failed to download attach ID {$attRow['attachment_id']} ('{$filename}'): $error_detail");
                $attachmentEntry['content'] = "Error downloading attachment."; // Placeholder
                $attachmentEntry['error'] = true;
                $attachmentEntry['error_message'] = $error_detail;
            } else {
                $attachmentEntry['content'] = $raw_content; // Store raw binary/string content
            }
            $conversationData['attachments'][] = $attachmentEntry;
        }
        mysqli_free_result($attachmentResult);
    }
    return $conversationData;
}

/**
 * Fetches all email messages and their associated attachments (with extracted text) for a conversation.
 * This is the most comprehensive fetch function for getting all textual content.
 */
function getAllEmailMessagesAndAttachmentsInConversation(string $conversationID, mysqli $conn): array {
    $conversationData = ['emails' => [], 'attachments' => []];
    if (empty($conversationID) || !$conn) {
        error_log("getAllEmailMessagesAndAttachmentsInConversation: Missing params.");
        return $conversationData;
    }
    $conversationID_safe = mysqli_real_escape_string($conn, $conversationID);

    // Fetch all emails in the conversation, ordered by received time (oldest first for chronological context)
    $emailSql = "SELECT message_id, msgid, sender, subject, received_datetime, plaintext_body, full_body
                 FROM tdu_emails
                 WHERE conversation_id = '$conversationID_safe'
                 ORDER BY received_datetime ASC"; // ASC for chronological order if processing as a thread
    $emailResult = mysqli_query($conn, $emailSql);

    if ($emailResult) {
        while ($emailRow = mysqli_fetch_assoc($emailResult)) {
            // Prefer plaintext, fallback to stripped HTML
            $body_clean = (!empty(trim($emailRow['plaintext_body']))) ? trim($emailRow['plaintext_body']) : strip_tags(trim($emailRow['full_body'] ?? ''));
            $conversationData['emails'][$emailRow['message_id']] = [ // Use message_id as key for easy lookup
                'message_id' => $emailRow['message_id'],
                'msgid_for_download' => $emailRow['msgid'], // Original message ID for downloads
                'sender' => $emailRow['sender'],
                'subject' => $emailRow['subject'] ?? '',
                'received_datetime' => $emailRow['received_datetime'],
                'plaintext_body' => $emailRow['plaintext_body'] ?? '',
                'full_body' => $emailRow['full_body'] ?? '',
                'body_clean' => $body_clean // The cleaned body for AI
            ];

            // Fetch attachments for this specific email message
            $attachmentSql = "SELECT attachment_id, file_name
                              FROM tdu_attachments
                              WHERE message_id = '" . mysqli_real_escape_string($conn, $emailRow['message_id']) . "'";
            $attachmentResult = mysqli_query($conn, $attachmentSql);
            if ($attachmentResult) {
                while ($attRow = mysqli_fetch_assoc($attachmentResult)) {
                    $filename = $attRow['file_name'] ?? 'Unknown Attachment';
                    // Endpoint to download the attachment
                    $endpoint = "https://dashboard.turtledownunder.com.au/ajax_outlook_attachment_download.php?attachment_id=".urlencode($attRow['attachment_id'])."&message_id=".urlencode($emailRow['msgid'])."&file_name=".urlencode($filename);

                    $ch = curl_init($endpoint);
                    curl_setopt_array($ch, [CURLOPT_RETURNTRANSFER => true, CURLOPT_TIMEOUT => 60]);
                    $raw_content = curl_exec($ch);
                    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                    $curlError = curl_error($ch);
                    curl_close($ch);

                    $extracted_text = '';
                    $attachment_error = false;
                    $attachment_error_message = '';

                    if ($curlError || $httpCode !== 200 || empty($raw_content)) {
                        $error_detail = "cURL Error: $curlError, HTTP Code: $httpCode, Empty Response: ".(empty($raw_content)?'yes':'no');
                        error_log("getAllEmailMessagesAndAttachmentsInConversation: Failed to download attach ID {$attRow['attachment_id']} ('{$filename}'): $error_detail");
                        $attachment_error = true;
                        $attachment_error_message = $error_detail;
                    } else {
                        // Try to extract text from the downloaded content
                        $fileType = function_exists('getFileTypeFromString') ? getFileTypeFromString($raw_content) : 'unknown';
                        try {
                            if ($fileType == "PDF Document" && function_exists('extractTextFromPDF')) { $extracted_text = extractTextFromPDF($raw_content); }
                            elseif (($fileType == "Word Document (Modern Format - DOCX)" || $fileType == "Word Document (Legacy Format - DOC)") && function_exists('extractTextfromWordDoc')) { $extracted_text = extractTextfromWordDoc($raw_content); }
                            elseif ($fileType == "Plain Text File" || $fileType == "HTML Web Page File") { $extracted_text = strip_tags($raw_content); }
                        } catch (Exception $e) {
                            error_log("getAllEmailMessagesAndAttachmentsInConversation: Error extracting text from attach ID {$attRow['attachment_id']} ('{$filename}', Type: $fileType): " . $e->getMessage());
                            $attachment_error = true;
                            $attachment_error_message = "Text extraction failed: " . $e->getMessage();
                        }
                    }
                    $conversationData['attachments'][] = [ // Add to a flat list of all attachments in conversation
                        'attachment_id' => $attRow['attachment_id'],
                        'parent_message_id' => $emailRow['message_id'], // Link back to the email it came from
                        'parent_email_received_datetime' => $emailRow['received_datetime'], // For chronological context of attachment
                        'filename' => $filename,
                        'raw_content' => $raw_content ?: null, // Store raw content if successfully fetched
                        'extracted_text' => trim($extracted_text),
                        'error' => $attachment_error,
                        'error_message' => $attachment_error_message
                    ];
                }
                mysqli_free_result($attachmentResult);
            } else {
                 error_log("getAllEmailMessagesAndAttachmentsInConversation: DB Error fetching attachments for Msg ID {$emailRow['message_id']}: " . mysqli_error($conn));
            }
        }
        mysqli_free_result($emailResult);
    } else {
        error_log("getAllEmailMessagesAndAttachmentsInConversation: DB Error fetching emails for Conv ID $conversationID_safe: " . mysqli_error($conn));
    }
    // Convert emails from associative array (keyed by message_id) to simple indexed array
    $conversationData['emails'] = array_values($conversationData['emails']);
    return $conversationData;
}


/**
 * Fetches all public tags from vtiger_freetags, grouped by category (raw_tag).
 */
function getAllTags(mysqli $conn): array {
    $sql = "SELECT id, tag, raw_tag FROM vtiger_freetags WHERE visibility = 'public' ORDER BY raw_tag, tag";
    $groupedTags = []; $result = mysqli_query($conn, $sql);
    if (!$result) { error_log("Error fetching tags: " . mysqli_error($conn)); return []; }
    if (mysqli_num_rows($result) > 0) {
        while ($row = mysqli_fetch_assoc($result)) {
            $category = $row['raw_tag']; // Use raw_tag as the grouping key
            if (!isset($groupedTags[$category])) { $groupedTags[$category] = []; }
            $groupedTags[$category][] = ['id' => $row['id'], 'tag' => $row['tag']];
        }
    }
    mysqli_free_result($result);
    return $groupedTags;
}

/**
 * Fetches a mapping of "type" tag names (like 'Refund', 'Complaint')
 * to the user name(s) assigned to handle them via tdu_tai_assigned_role.
 *
 * @param mysqli $conn Database connection
 * @param array $availableSystemTypes Mapping of type tag_id to type_name (e.g., [105 => 'Refund'])
 * @return array Example: ['Refund' => 'Job Mathew', 'Complaint' => 'Arya Prajapati, Komal Maheshwari']
 */
function getTypeToUserAssignments(mysqli $conn, array $availableSystemTypes): array {
    // --- CAUTIOUS SETUP CALL ---
    // This will run every time getTypeToUserAssignments is called.
    // Consider the performance and permission implications mentioned.
    if (function_exists('setup_tdu_tai_assigned_role_table')) {
        setup_tdu_tai_assigned_role_table($conn); // Ensures table exists before querying
    } else {
        error_log("getTypeToUserAssignments: FATAL - setup_tdu_tai_assigned_role_table function is missing!");
        // Optionally, you could try to run the CREATE TABLE IF NOT EXISTS query directly here
        // but it's better to ensure the setup function is available if this pattern is chosen.
    }
    // --- END CAUTIOUS SETUP CALL ---
    
    $assignments = [];
    $typeTagIdsToNames = array_flip($availableSystemTypes); // Map name back to ID for easier lookup if needed, or just use IDs directly

    // We need tag IDs for 'Refund', 'Complaint' etc.
    // Let's assume $availableSystemTypes gives us the tag_id => 'TypeName' mapping.
    // We need to query tdu_tai_assigned_role for these tag_ids and join with vtiger_users.

    $sql = "SELECT
                atr.tag_id,
                CONCAT(u.first_name, ' ', u.last_name) AS assigned_user_name,
                u.id as assigned_user_id -- if your 'person' tags are user IDs
            FROM tdu_tai_assigned_role atr
            JOIN vtiger_users u ON atr.user_id = u.id
            WHERE u.status = 'Active' AND atr.tag_id IN (" . implode(',', array_map('intval', array_keys($availableSystemTypes))) . ")
            ORDER BY atr.tag_id, assigned_user_name"; // Ensure consistent ordering for multiple assignees

    $result = mysqli_query($conn, $sql);
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $typeTagId = (int)$row['tag_id'];
            $typeName = $availableSystemTypes[$typeTagId] ?? null; // Get 'Refund', 'Complaint'

            if ($typeName) {
                $personIdentifier = $row['assigned_user_name']; // Or $row['assigned_user_id'] if person tags are user IDs

                if (!isset($assignments[$typeName])) {
                    $assignments[$typeName] = $personIdentifier;
                } else {
                    // If multiple users can be assigned to a type, concatenate them or handle as needed.
                    // For the AI prompt, a simple comma-separated list might be best.
                    $assignments[$typeName] .= ", " . $personIdentifier;
                }
            }
        }
        mysqli_free_result($result);
    } else {
        error_log("getTypeToUserAssignments: Error fetching assignments: " . mysqli_error($conn));
    }
    return $assignments;
}

/**
 * Analyzes conversation content and suggests relevant tags using AI.
 */
function getSuggestedTagsForConversation(string $conversationID, mysqli $conn, array $excludedCategories = []): array {
    if (empty($conversationID) || !$conn) { error_log("getSuggestedTagsForConversation: Missing params."); return []; }

    // Get content for AI (latest email subject/body + all attachments text)
    $conversationContentForAI = "";
    $latestEmailData = getMessageFromConversationID_forSegments($conversationID, $conn); // Fetches latest email
    if ($latestEmailData) {
        $emailSubject = $latestEmailData['subject'] ?? '';
        $emailBodyRaw = $latestEmailData['plaintext_body'] ?? ($latestEmailData['full_body'] ?? '');
        $emailBodyClean = strip_tags($emailBodyRaw); // Clean body
        $conversationContentForAI .= "Email Subject: " . trim($emailSubject) . "\nEmail Body:\n" . trim($emailBodyClean) . "\n\n";
    }

    $attachmentItems = getAttachmentFiles($conversationID, $conn); // Fetches text from all attachments
    if (!empty($attachmentItems)) {
        $conversationContentForAI .= "--- Attachments Text ---\n";
        foreach ($attachmentItems as $item) {
            $conversationContentForAI .= "\n--- Attachment: " . htmlspecialchars($item['filename']) . " ---\n" . $item['text'] . "\n";
        }
    }
    $trimmedCombinedContent = trim($conversationContentForAI);
    if (empty($trimmedCombinedContent)) { error_log("getSuggestedTagsForConversation: Combined content (email + attachments) is empty for ID: $conversationID."); return []; }

    // Get all available tags from DB
    $allTagsGrouped = getAllTags($conn);
    if (empty($allTagsGrouped)) { error_log("getSuggestedTagsForConversation: No tags in DB."); return []; }

    // Prepare tag list for AI prompt, excluding specified categories
    $tagListForPrompt = ""; $idToNameMap = [];
    foreach ($allTagsGrouped as $category => $tags) {
        if (in_array($category, $excludedCategories)) continue; // Skip excluded categories
        $tagListForPrompt .= "Category '$category':\n";
        foreach ($tags as $tag) {
            $tagListForPrompt .= "- '" . addslashes($tag['tag']) . "' (ID: " . $tag['id'] . ")\n";
            $idToNameMap[$tag['id']] = $tag['tag']; // Map ID to name for later use
        }
        $tagListForPrompt .= "\n";
    }
    if (empty($idToNameMap)) { error_log("getSuggestedTagsForConversation: No tags after exclusion."); return []; }

    $currentAvailableSystemTypes = [ 
        110 => 'Complaint', 92  => 'Confirmation', 108 => 'Following up Vouchers',
        109 => 'Payment on Arrival', 88  => 'Post Sales', 101 => 'Post Sales In Progress',
        89  => 'Pre Sales', 90  => 'Quotes', 105 => 'Refund', 91  => 'Requote',
        107 => 'Requote after Confirmation'
    ];

    $typeUserAssignments = getTypeToUserAssignments($conn, $currentAvailableSystemTypes);
    $personRulesString = "Rules for person assignment based on specific types:\n";
    if (!empty($typeUserAssignments)) {
        foreach ($typeUserAssignments as $type => $users) {
            $personRulesString .= "- If the conversation is related to '$type', assign person tag for: '$users'.\n";
        }
    } else {
        $personRulesString .= "- No specific dynamic person assignment rules found from database. Use general knowledge if applicable.\n";
    }
    // Add your static rule if it's still relevant or if the dynamic one might not cover it
    // $personRulesString .= "- For general refunds not covered above, consider: Job Matthew.\n";
    // --- END NEW ---


    // Construct the AI prompt
    $preprompt = "Analyze 'Conversation Content'.
Suggest relevant tags (by ID) from 'Available Tags List'.
Output ONLY a valid JSON array of strings (tag IDs).
Empty array [] if none. No other text/markdown. \n
Information: The North, West, East, South tags correspond to regions of India that the vendor is from (anyone not TDU), so look for indian regions or zipcode and assign tags accordingly, the latest one takes precedence.
Exempt 'Turtle Down Under' from the analysis for all tags.
If the assign person is blank, that means no one is assigned to it and can be safely ignored
$personRulesString  // <<<< MODIFIED: Insert dynamic rules here
\nThe following are considered types Post Sales, Pre Sales, Quotes, Requote, Confirmation, Post Sales In Progress, Refund, Requote after Confirmation, Following up Vouchers, Payment on Arrival, Complaint. There can only be 1 types, consider the latest situation within the conversation to determine the most appropriate type\nAvailable Tags List:\n$tagListForPrompt\nConversation Content:\n"; // $tagListForPrompt is from your existing code

    $prompt_for_ai_utf8 = $preprompt . $trimmedCombinedContent;
    //error_log($prompt_for_ai_utf8);
    // Ensure UTF-8 encoding for AI
    if (!mb_check_encoding($prompt_for_ai_utf8, 'UTF-8')) {
        $detected_encoding = mb_detect_encoding($prompt_for_ai_utf8, 'UTF-8, Windows-1252, ISO-8859-1', true);
        $converted = mb_convert_encoding($prompt_for_ai_utf8, 'UTF-8', $detected_encoding ?: 'UTF-8');
        if ($converted === false) { error_log("getSuggestedTagsForConversation: Failed to convert prompt to UTF-8 from '$detected_encoding'."); return []; }
        $prompt_for_ai_utf8 = $converted;
    }

    // Call Gemini AI
    $apiUrl = apiURL_Flash(); if (!$apiUrl) { error_log("getSuggestedTagsForConversation: API URL missing."); return []; }
    $payload = ['contents' => [['parts' => [['text' => $prompt_for_ai_utf8]]]], 'generationConfig' => ['temperature' => 0.3, 'maxOutputTokens' => 50000, 'topP' => 0.95, 'topK' => 40, 'response_mime_type' => "application/json"]]; // Expect JSON response
    $headers = ['Content-Type: application/json'];
    $ch = curl_init($apiUrl); curl_setopt_array($ch, [CURLOPT_RETURNTRANSFER => true, CURLOPT_POST => true, CURLOPT_POSTFIELDS => json_encode($payload), CURLOPT_HTTPHEADER => $headers, CURLOPT_CONNECTTIMEOUT => 20, CURLOPT_TIMEOUT => 120]);
    $response_body = curl_exec($ch); $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE); $curlError = curl_error($ch); curl_close($ch);

    if ($curlError || $httpCode !== 200) { error_log("getSuggestedTagsForConversation: API/cURL Error. HTTP: $httpCode, cURL: $curlError, Resp: $response_body"); return []; }

    $responseData = json_decode($response_body, true);
    if (json_last_error() !== JSON_ERROR_NONE || !isset($responseData['candidates'][0]['content']['parts'][0]['text'])) {
        error_log("getSuggestedTagsForConversation: Failed to decode API JSON or extract text. Raw: " . $response_body);
        return [];
    }
    $generated_text = $responseData['candidates'][0]['content']['parts'][0]['text'] ?? null;
    if ($generated_text === null) { error_log("getSuggestedTagsForConversation: Could not extract text (null). Resp: $response_body"); return []; }

    // Clean the AI response (remove BOM, markdown) and parse JSON
    $cleaned_json_text = trim(str_replace(["\xEF\xBB\xBF", "```json", "```"], '', $generated_text));
    $suggestedTagIds = json_decode($cleaned_json_text, true);

    // Fallback if JSON parsing fails but might contain IDs in a string
    if (!(json_last_error() === JSON_ERROR_NONE && is_array($suggestedTagIds))) {
        if (preg_match_all('/"(\d+)"/', $cleaned_json_text, $matches_arr_tags)) {
             $suggestedTagIds = $matches_arr_tags[1]; // Use matched IDs
        } else {
             error_log("getSuggestedTagsForConversation: Fallback regex also failed to find Tag IDs.");
             return [];
        }
    }
    if (!is_array($suggestedTagIds)) { // Ensure it's an array after all processing
        error_log("getSuggestedTagsForConversation: \$suggestedTagIds is not an array after processing. Value: " . print_r($suggestedTagIds, true));
        return [];
    }

    // Validate suggested IDs against known tags
    $finalSuggestedTags = [];
    foreach ($suggestedTagIds as $tagId) {
        $tagIdStr = (string) trim($tagId);
        if (isset($idToNameMap[$tagIdStr])) { // Check if ID is valid
            $finalSuggestedTags[] = ['id' => $tagIdStr, 'name' => $idToNameMap[$tagIdStr]];
        } else {
            error_log("getSuggestedTagsForConversation: AI suggested unknown or invalid Tag ID '$tagIdStr'.");
        }
    }
    return $finalSuggestedTags;
}

/**
 * Uses Gemini AI to generate a concise summary of text content.
 */
function generateSummary(string $fileString): string {
    if (empty(trim($fileString))) { error_log("generateSummary: Input empty."); return "Error: Input empty."; }
    $preprompt = "Analyze 'Document Text'. Provide a concise, one or two sentence summary. Constraint: Output ONLY summary text. No intro/labels/markdown.\n\nDocument Text:\n";
    $prompt_for_ai_utf8 = $preprompt . $fileString;
    // UTF-8 check and conversion
    if (!mb_check_encoding($prompt_for_ai_utf8, 'UTF-8')) {
        $detected_encoding = mb_detect_encoding($prompt_for_ai_utf8, 'UTF-8, Windows-1252, ISO-8859-1', true);
        $converted = mb_convert_encoding($prompt_for_ai_utf8, 'UTF-8', $detected_encoding ?: 'UTF-8');
        if ($converted === false) { error_log("generateSummary Error: UTF-8 conversion failed from '$detected_encoding'."); return "Error: Encoding failed."; }
        $prompt_for_ai_utf8 = $converted;
    }

    $apiUrl = apiURL_Flash(); if (!$apiUrl) { error_log("generateSummary Error: API URL missing."); return "Error: API URL missing."; }
    $payload = ['contents' => [['parts' => [['text' => $prompt_for_ai_utf8]]]], 'generationConfig' => ['temperature' => 0.4, 'maxOutputTokens' => 50000, 'topP' => 0.95, 'topK' => 40]];
    $headers = ['Content-Type: application/json'];
    $ch = curl_init($apiUrl); curl_setopt_array($ch, [CURLOPT_RETURNTRANSFER => true, CURLOPT_POST => true, CURLOPT_POSTFIELDS => json_encode($payload), CURLOPT_HTTPHEADER => $headers, CURLOPT_CONNECTTIMEOUT => 15, CURLOPT_TIMEOUT => 60]);
    $response_body = curl_exec($ch); $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE); $curlError = curl_error($ch); curl_close($ch);

    if ($curlError || $httpCode !== 200) {
        error_log("generateSummary: cURL/API Error HTTP $httpCode, cURL: $curlError, Resp: $response_body");
        $apiErrorDetails = json_decode($response_body, true);
        $apiErrorMessage = $apiErrorDetails['error']['message'] ?? 'Unknown API error';
        return "Error: API failed ($httpCode) - " . substr(htmlspecialchars($apiErrorMessage), 0, 100);
    }

    $responseData = json_decode($response_body, true);
    if (json_last_error() !== JSON_ERROR_NONE || !isset($responseData['candidates'][0]['content']['parts'][0]['text'])) {
         error_log("generateSummary: Failed to decode API JSON or extract text. Raw: " . $response_body);
        return "Error: API response parse failed.";
    }
    $generated_text = $responseData['candidates'][0]['content']['parts'][0]['text'] ?? null;
    if ($generated_text === null) {
        error_log("generateSummary: Could not extract text. Resp: " . $response_body);
        if (isset($responseData['promptFeedback']['blockReason'])) { error_log("generateSummary: Gemini Content Blocked: " . ($responseData['promptFeedback']['blockReason'] ?? 'Unknown')); return "Error: AI Content Blocked.";}
        return "Error: AI response parse failed (null text).";
    }

    $trimmed_text = trim(str_replace("\xEF\xBB\xBF", '', $generated_text)); // Remove BOM
    if (empty($trimmed_text)) { error_log("generateSummary: AI returned empty summary. Original: " . $generated_text); return "Error: AI empty summary."; }
    return $trimmed_text;
}

/**
 * Uses Gemini AI to classify text as 'invoice', 'confirmation', or 'other'.
 */
function classifyAttachmentContent(string $fileString): string {
    if (empty(trim($fileString))) { error_log("classifyAttachmentContent: Input empty."); return 'error'; } // Return 'error' for empty input
    $preprompt = "Analyze 'Document Text'. Classify primary purpose: invoice/bill/receipt, booking/order confirmation, or something else. Task: If invoice, output ONLY 'invoice'. If confirmation, output ONLY 'confirmation'. Otherwise, output ONLY 'other'. Constraint: Respond ONLY 'invoice', 'confirmation', or 'other' (lowercase). No explanations/markdown.\n\nDocument Text:\n";
    $prompt_for_ai_utf8 = $preprompt . $fileString;
    // UTF-8 check and conversion
     if (!mb_check_encoding($prompt_for_ai_utf8, 'UTF-8')) {
        $detected_encoding = mb_detect_encoding($prompt_for_ai_utf8, 'UTF-8, Windows-1252, ISO-8859-1', true);
        $converted = mb_convert_encoding($prompt_for_ai_utf8, 'UTF-8', $detected_encoding ?: 'UTF-8');
        if ($converted === false) { error_log("classifyAttachmentContent Error: UTF-8 conversion failed from '$detected_encoding'."); return 'error'; }
        $prompt_for_ai_utf8 = $converted;
    }

    $apiUrl = apiURL_Flash(); if (!$apiUrl) { error_log("classifyAttachmentContent Error: API URL missing."); return 'error'; }
    $payload = ['contents' => [['parts' => [['text' => $prompt_for_ai_utf8]]]], 'generationConfig' => ['temperature' => 0.1, 'maxOutputTokens' => 50000, 'topP' => 0.95, 'topK' => 40]];
    $headers = ['Content-Type: application/json'];
    $ch = curl_init($apiUrl); curl_setopt_array($ch, [CURLOPT_RETURNTRANSFER => true, CURLOPT_POST => true, CURLOPT_POSTFIELDS => json_encode($payload), CURLOPT_HTTPHEADER => $headers, CURLOPT_CONNECTTIMEOUT => 10, CURLOPT_TIMEOUT => 30]);
    $response_body = curl_exec($ch); $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE); $curlError = curl_error($ch); curl_close($ch);

    if ($curlError || $httpCode !== 200) { error_log("classifyAttachmentContent: cURL/API Error HTTP $httpCode, cURL: $curlError, Resp: $response_body"); return 'error'; }

    $responseData = json_decode($response_body, true);
    if (json_last_error() !== JSON_ERROR_NONE || !isset($responseData['candidates'][0]['content']['parts'][0]['text'])) {
         error_log("classifyAttachmentContent: Failed to decode API JSON or extract text. Raw: " . $response_body);
        return 'error';
    }
    $generated_text = $responseData['candidates'][0]['content']['parts'][0]['text'] ?? null;

    if ($generated_text === null) {
        error_log("classifyAttachmentContent: Could not extract text. Check safety blocks. Resp: " . $response_body);
        if (isset($responseData['candidates'][0]['finishReason']) && $responseData['candidates'][0]['finishReason'] !== 'STOP') { error_log("classifyAttachmentContent: AI finishReason " . $responseData['candidates'][0]['finishReason']);}
        if (isset($responseData['promptFeedback']['blockReason'])) { error_log("classifyAttachmentContent: Prompt blocked: " . $responseData['promptFeedback']['blockReason']);}
        return 'error'; // Return 'error' if AI text is null
    }

    $trimmed_text = trim(strtolower(str_replace("\xEF\xBB\xBF", '', $generated_text))); // Remove BOM, lowercase
    if ($trimmed_text === '') { error_log("classifyAttachmentContent: AI returned empty string after cleaning."); return 'error'; }
    if (in_array($trimmed_text, ['invoice', 'confirmation', 'other'])) { return $trimmed_text; }
    else { error_log("classifyAttachmentContent: AI unexpected response: '{$trimmed_text}'"); return 'other'; } // Default to 'other' for unexpected
}

/**
 * Defines the prompt for the AI to classify if an email is spam or an actionable auto-reply.
 * This consolidated prompt now covers both traditional spam and various types of automated messages
 * that should be filtered out.
 */
function get_ai_spam_classification_prompt_text(): string {
    return <<<PROMPT
Analyze the following email content (subject, body, and any attachment text context).
Your goal is to determine if this email is SPAM or an AUTOMATIC SYSTEM-GENERATED REPLY / NON-ACTIONABLE INFORMATIONAL MESSAGE that requires no immediate human business action from our team.

Classify as 'yes' (effectively spam/filterable for our processing purposes) if the email is ANY of the following:

1.  **TRADITIONAL SPAM:**
    *   Unsolicited commercial email (UCE), junk mail.
    *   Phishing attempts, scams, fraudulent messages.
    *   Malware distribution attempts.
    *   Irrelevant bulk marketing emails or newsletters the recipient likely did not subscribe to.
    Examples: Offers for dubious products, requests for personal info, suspicious links.

2.  **AUTOMATIC REPLIES INDICATING SENDER UNAVAILABILITY (Out of Office):**
    *   Common phrases: 'out of office', 'away from my desk', 'on leave', 'will return on [date]', 'limited access to email', 'ooo'.
    *   Subject lines like 'Automatic reply' or 'Auto-Reply' when combined with messages of unavailability for general correspondence.
    Examples:
    *   Subject: `Out of Office: Annual Leave` Body: `I am currently out of the office and will return on July 15th.`
    *   Body: `Thank you for your message. I am currently on vacation with no email access.`

3.  **SIMPLE ACKNOWLEDGEMENTS OF RECEIPT (Non-Actionable):**
    *   Emails whose *sole or primary new contribution* to the conversation is a generic confirmation that a message was received, or a statement about the sender's own process (e.g., "we will review"), without adding further business-relevant information, questions, or requests that require our team's immediate action or change the status of an ongoing transaction significantly.
    Examples of such standalone messages or latest replies in a thread that would typically be 'yes':
    *   'Your message has been received.'
    *   'Thank you for your email. We will get back to you within 24-48 hours.'
    *   'Acknowledged.' / 'Noted.' / 'Received.'
    *   'We are looking into your request and will update you soon.'
    *   Subject: `Automatic reply: Your inquiry` Body: `Thank you for your email. Your message has been received, and we will respond within 24 hours.`
    *   **Important Distinction for Threads & Overall Actionability:**
        *   **Context is King for Threads:** An entire email thread should NOT be classified as 'yes' simply because an acknowledgement (e.g., "Noted", "Acknowledged receipt") appears somewhere within it. If the overall conversation (the email content you are analyzing) contains actionable business communication, outstanding questions that require a response from our team, requests, or substantive new information from any participant – especially if these elements are more recent than a simple acknowledgement or are part of the same message that also contains an acknowledgement – then the email should be classified as 'no'.
        *   **When to Apply This Rule #3 for 'yes':** This rule applies if, after considering the entire conversation, the *net actionable status* indicates no further business action is required from our team because the latest substantive input from other parties is purely a simple acknowledgement or a non-critical status update (as exemplified above), and there are no other unresolved actionable items for our team presented in the email content.
        *   **Substantive Automated Messages are Different:** Always distinguish these simple/non-actionable acknowledgements from *substantive* automated messages (like usable password reset links, detailed support ticket updates providing new resolution steps, specific order shipment details) which are generally 'no' as they convey new, often critical or actionable, information.

4.  **DELIVERY STATUS NOTIFICATIONS (DSN):**
    *   Messages indicating the status of an email that *we previously sent*.
    Examples:
    *   'Delivery Status Notification (Failure)', 'Delivery Status Notification (Success)', 'Delivery Status Notification (Delay)'
    *   'Undeliverable mail', 'Message delivery failed', 'The following message to <recipient@example.com> was undeliverable.'
    *   'Read receipt for: [Our Subject]'

5.  **NON-ACTIONABLE ADMINISTRATIVE / CONTACT DETAIL CHANGE NOTIFICATIONS:**
    *   Emails whose primary purpose is to inform about a change in contact details (e.g., email address, phone number, company name/ rebranding) and do not contain a new business inquiry or an update on an ongoing business transaction.
    Examples:
    *   Subject: `Email Address Change` Body: `Dear Sir / Mam Please note our e-mail ( old@example.com) has been changed to new@example.com. So please save this email id for future communications.`
    *   Subject: `New Contact Information` Body: `Please update your records, my new phone number is XXXXX.`
    *   Body: `Our company, ExampleCorp, will now be known as NovaSolutions.`
    *   **Important Distinction:** If an email about a new business inquiry also happens to mention an email change in the signature, it should still be 'no'. This rule applies if the *main point* of the email is the contact change.

Classify as 'no' (legitimate and likely requires human attention) if the email is:

1.  **A NORMAL, HUMAN-WRITTEN EMAIL** directly related to our business inquiries, bookings, support, ongoing discussions, etc.
2.  **AN AUTOMATED EMAIL THAT IS PRIMARILY TRANSACTIONAL, INFORMATIONAL, OR REQUIRES ACTION, and is NOT one of the ignorable types defined above.**
    Examples (these should be 'no' unless their *primary message* also includes explicit out-of-office content for general correspondence or is a simple DSN for an email *we* sent, or fits one of the 'yes' categories above):
    *   Order confirmations from suppliers for bookings we made ('Your booking #XYZ is confirmed')
    *   Invoices from vendors for services we used ('Your invoice INV-005 is attached')
    *   Shipping notifications for items relevant to us.
    *   Password reset emails for systems we use.
    *   Proactive announcements from a known service or partner (e.g., 'Important update about your account', 'Newsletter from a relevant industry source we opted into').
    *   Meeting invitations or calendar updates that are not just OOO notifications.

Essentially, if the email is something our team needs to read and potentially act upon because it contains new, relevant business information or a direct communication, classify it as 'no'. If it's junk, an OOO, a simple "got your email" (when that is the primary, non-actionable content of the latest communication as detailed in rule #3), a DSN for something we sent, or a purely administrative contact update, classify it as 'yes'.

Respond with ONLY one of the following keywords (all lowercase):
*   'yes' (if SPAM or an AUTOMATIC SYSTEM-GENERATED REPLY / NON-ACTIONABLE INFORMATIONAL MESSAGE as defined above)
*   'no' (if legitimate and requires attention as defined above)

Do not include any other text, explanations, or JSON formatting. Just the single keyword.
The content you are receiving is one whole conversation, therefore there should only be 1 yes or 1 no

Email and Attachment Content is below:
PROMPT;
}

/**
 * Analyzes conversation using AI to determine if content is likely spam or an ignorable auto-reply.
 * Returns 'yes' if spam/auto-reply, 'no' if legitimate, 'error' on failure.
 */
function classifyAsSpam(string $conversationID, mysqli $conn): string {
    if (empty($conversationID) || !$conn || $conn->connect_error) {
        error_log("classifyAsSpam: Missing conversationID or invalid DB connection.");
        return 'error'; // error, yes, no
    }
    $conversationID_safe = mysqli_real_escape_string($conn, $conversationID);
    $emailSubject = '';
    $combinedBodyAndAttachmentText = "";

    // Get all email bodies and attachment texts for the conversation
    // It's important to use getAllEmailMessagesAndAttachmentsInConversation to get the full context
    if (!function_exists('getAllEmailMessagesAndAttachmentsInConversation')) {
        error_log("classifyAsSpam: Helper function 'getAllEmailMessagesAndAttachmentsInConversation' is missing.");
        return 'error';
    }
    $allConversationData = getAllEmailMessagesAndAttachmentsInConversation($conversationID_safe, $conn);

    if (!empty($allConversationData['emails'])) {
        // Use subject from the latest email in the conversation for primary context,
        // but all content is passed to AI.
        $latestEmailForSubject = end($allConversationData['emails']); // Get the last email
        reset($allConversationData['emails']); // Reset array pointer if needed elsewhere
        $emailSubject = $latestEmailForSubject['subject'] ?? '';

        foreach ($allConversationData['emails'] as $email) {
            $combinedBodyAndAttachmentText .= "\n\n--- Email Start (Sender: {$email['sender']}, Received: {$email['received_datetime']}) ---\n";
            $combinedBodyAndAttachmentText .= "Subject: " . ($email['subject'] ?? '') . "\n";
            $combinedBodyAndAttachmentText .= "Body:\n" . ($email['body_clean'] ?? ''); // Use the cleaned body
            $combinedBodyAndAttachmentText .= "\n--- Email End ---";
        }
    }
    if (!empty($allConversationData['attachments'])) {
        foreach ($allConversationData['attachments'] as $attachment) {
            if (!$attachment['error'] && !empty(trim($attachment['extracted_text']))) {
                 $combinedBodyAndAttachmentText .= "\n\n--- Attachment: " . htmlspecialchars($attachment['filename']) . " (From email received: " . ($attachment['parent_email_received_datetime'] ?? 'N/A') . ") ---\n" . trim($attachment['extracted_text']);
                 $combinedBodyAndAttachmentText .= "\n--- Attachment End ---";
            }
        }
    }

    $combinedContentForAI = trim($combinedBodyAndAttachmentText); // The prompt will refer to this as "Email and Attachment Content"

    if (empty($combinedContentForAI) && empty($emailSubject)) { // If absolutely no text content from any source
        error_log("classifyAsSpam: No text content (subject, body, attachments) for Conv ID $conversationID_safe.");
        return 'no'; // Default to 'no' (not spam) if there's nothing to analyze, to be safe. Or 'error'.
    }

    // The prompt is designed to work with the $combinedContentForAI.
    // The $emailSubject is implicitly part of $combinedContentForAI if emails were found.
    $full_prompt_text = get_ai_spam_classification_prompt_text() . "\n" . $combinedContentForAI;

    // UTF-8 check
    if (!mb_check_encoding($full_prompt_text, 'UTF-8')) {
        $detected_encoding = mb_detect_encoding($full_prompt_text, 'UTF-8, Windows-1252, ISO-8859-1', true);
        $converted = mb_convert_encoding($full_prompt_text, 'UTF-8', $detected_encoding ?: 'UTF-8');
        if ($converted === false) {
            error_log("classifyAsSpam Error: UTF-8 conversion failed for Conv ID {$conversationID_safe} from '$detected_encoding'.");
            return 'error';
        }
        $full_prompt_text = $converted;
    }

    $apiUrl = apiURL_Flash();
    if (!$apiUrl) {
        error_log("classifyAsSpam: API URL is missing for Conv ID {$conversationID_safe}.");
        return 'error';
    }
    //error_log($full_prompt_text);
    $payload = [
        'contents' => [['parts' => [['text' => $full_prompt_text]]]],
        'generationConfig' => ['temperature' => 0.1, 'maxOutputTokens' => 50000, 'topP' => 0.95, 'topK' => 40] // Expect short "yes"/"no"
    ];
    $headers = ['Content-Type: application/json'];

    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true, CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($payload), CURLOPT_HTTPHEADER => $headers,
        CURLOPT_CONNECTTIMEOUT => 15, CURLOPT_TIMEOUT => 45
    ]);
    $response_body = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    if ($curlError || $httpCode !== 200) {
        error_log("classifyAsSpam: cURL/API Error for Conv ID {$conversationID_safe} - HTTP $httpCode, cURL: $curlError, Resp: $response_body");
        return 'error';
    }

    $responseData = json_decode($response_body, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
         error_log("classifyAsSpam: Failed to decode API JSON for Conv ID {$conversationID_safe}. Raw: " . $response_body);
        return 'error';
    }

    if (isset($responseData['promptFeedback']['blockReason'])) {
        error_log("classifyAsSpam: Gemini Content Blocked for Conv ID {$conversationID_safe}: " . ($responseData['promptFeedback']['blockReason'] ?? 'Unknown'));
        return 'error'; // Treat blocked content as an error in classification
    }

    $generated_text = $responseData['candidates'][0]['content']['parts'][0]['text'] ?? null;

    if ($generated_text === null) {
        error_log("classifyAsSpam: Could not extract text from AI response for Conv ID {$conversationID_safe}. Check safety blocks. Resp: " . $response_body);
        return 'error';
    }

    $cleaned_response = strtolower(trim(str_replace("\xEF\xBB\xBF", '', $generated_text))); // Remove BOM, lowercase

    if ($cleaned_response === 'yes') {
        error_log("classifyAsSpam: AI classified as SPAM/AUTO-REPLY for Conv ID {$conversationID_safe}.");
        return 'yes';
    } elseif ($cleaned_response === 'no') {
        error_log("classifyAsSpam: AI classified as NOT SPAM/AUTO-REPLY for Conv ID {$conversationID_safe}.");
        return 'no';
    } else {
        error_log("classifyAsSpam: AI returned unexpected response '{$cleaned_response}' for Conv ID {$conversationID_safe}. Raw AI: " . $generated_text);
        return 'error'; // Default to error for unexpected responses
    }
}

/**
 * Extracts unique quote numbers from a conversation.
 * PRIORITIZES the quote_no from vtiger_support if available and valid.
 * Otherwise, uses an AI call to scan all conversation text for TDU-prefixed quote numbers.
 *
 * @param string $conversationID The ID of the conversation to process.
 * @param mysqli $conn The database connection object.
 * @return string JSON encoded array of unique quote numbers (e.g., "[\"TDU123\"]" or "[\"TDU456\",\"TDU789\"]")
 *                or an empty JSON array "[]" if none are found or on error.
 */
function extractQuoteNumbersFromConversation(string $conversationID, mysqli $conn): string {
    if (empty($conversationID) || !$conn || $conn->connect_error) {
        error_log("extractQuoteNumbersFromConversation: Missing conversationID or invalid DB connection.");
        return json_encode([]);
    }
    $conversationID_safe = mysqli_real_escape_string($conn, $conversationID);

    // --- Stage 1: Check vtiger_support for an authoritative quote_no ---
    $db_quote_no_from_vtiger = null;
    $sql_vtiger_quote = "SELECT quote_no FROM vtiger_support WHERE ticketid = ?";
    $stmt_vtiger = $conn->prepare($sql_vtiger_quote);

    if ($stmt_vtiger) {
        $stmt_vtiger->bind_param("s", $conversationID_safe);
        if ($stmt_vtiger->execute()) {
            $result_vtiger = $stmt_vtiger->get_result();
            if ($row_vtiger = $result_vtiger->fetch_assoc()) {
                // Check if DB quote is valid (not empty, not 'N/A', starts with TDU)
                if (!empty($row_vtiger['quote_no']) &&
                    strtoupper(trim($row_vtiger['quote_no'])) !== 'N/A' &&
                    stripos(strtoupper(trim($row_vtiger['quote_no'])), 'TDU') === 0) {
                    $db_quote_no_from_vtiger = strtoupper(trim($row_vtiger['quote_no']));
                    error_log("extractQuoteNumbersFromConversation: Found authoritative quote_no '{$db_quote_no_from_vtiger}' in vtiger_support for Conv ID {$conversationID_safe}. Prioritizing this.");
                    return json_encode([$db_quote_no_from_vtiger]); // Return only this quote
                }
            }
            if ($result_vtiger instanceof mysqli_result) $result_vtiger->free();
        } else {
            error_log("extractQuoteNumbersFromConversation: Execute failed (vtiger_support quote check): " . $stmt_vtiger->error);
        }
        $stmt_vtiger->close();
    } else {
        error_log("extractQuoteNumbersFromConversation: Prepare failed (vtiger_support quote check): " . $conn->error);
    }
    // --- END vtiger_support check ---

    // --- Stage 2: If no DB quote, proceed with AI-based extraction from full conversation text ---
    error_log("extractQuoteNumbersFromConversation: No authoritative quote in vtiger_support for Conv ID {$conversationID_safe}. Proceeding with AI extraction.");

    $aggregatedContent = "";
    // Ensure 'getAllEmailMessagesAndAttachmentsInConversation' function exists and is callable
    if (!function_exists('getAllEmailMessagesAndAttachmentsInConversation')) {
        error_log("extractQuoteNumbersFromConversation: CRITICAL - Helper function 'getAllEmailMessagesAndAttachmentsInConversation' is missing.");
        return json_encode([]);
    }
    $conversationSources = getAllEmailMessagesAndAttachmentsInConversation($conversationID_safe, $conn);

    // Aggregate all subjects, email bodies, and attachment texts
    if (!empty($conversationSources['emails'])) {
        foreach ($conversationSources['emails'] as $email) {
            if (!empty(trim($email['subject']))) {
                $aggregatedContent .= "Email Subject: " . trim($email['subject']) . "\n\n";
            }
            if (!empty(trim($email['body_clean']))) {
                $aggregatedContent .= "Email Body (Received: " . $email['received_datetime'] . "):\n" . trim($email['body_clean']) . "\n\n";
            }
        }
    }
    if (!empty($conversationSources['attachments'])) {
        foreach ($conversationSources['attachments'] as $attachment) {
            if (!$attachment['error'] && !empty(trim($attachment['extracted_text']))) {
                $aggregatedContent .= "Attachment Text (Filename: " . htmlspecialchars($attachment['filename']) . " From email received: " . $attachment['parent_email_received_datetime'] . "):\n" . trim($attachment['extracted_text']) . "\n\n";
            }
        }
    }
    $aggregatedContent = trim($aggregatedContent);

    if (empty($aggregatedContent)) {
        error_log("extractQuoteNumbersFromConversation (AI Scan): No text content (subject, body, attachments) found for Conv ID {$conversationID_safe}.");
        return json_encode([]);
    }

    // Define the AI prompt for quote number extraction
    $ai_prompt = "Analyze the 'Full Conversation Text' provided below.\n";
    $ai_prompt .= "Your task is to identify and extract all unique quote numbers.\n\n";
    $ai_prompt .= "Quote Number Format:\n";
    $ai_prompt .= "- Quote numbers typically start with the prefix \"TDU\" (case-insensitive).\n";
    $ai_prompt .= "- The \"TDU\" prefix is followed by a sequence of 5 numbers and then a G denoting a group quote or no G meaning its a FIT quote.\n";
    $ai_prompt .= "- Examples: TDU12345, tdu67890, TDU42324G, tDu55555g.\n\n";
    $ai_prompt .= "Output Instructions:\n";
    $ai_prompt .= "- Return a single, valid JSON array containing all unique quote numbers found in the text.\n";
    $ai_prompt .= "- Convert all extracted quote numbers to UPPERCASE before including them in the array.\n";
    $ai_prompt .= "- If no quote numbers matching the format are found, return an empty JSON array: [].\n";
    $ai_prompt .= "- Do not include any other text, explanations, or markdown before or after the JSON array.\n\n";
    $ai_prompt .= "Example Output (Quotes Found):\n";
    $ai_prompt .= "[\"TDU12345\", \"TDU67890A\"]\n\n";
    $ai_prompt .= "Example Output (No Quotes Found):\n";
    $ai_prompt .= "[]\n\n";
    $ai_prompt .= "Full Conversation Text:\n" . $aggregatedContent;

    // UTF-8 check and conversion for the prompt
    if (!mb_check_encoding($ai_prompt, 'UTF-8')) {
        $detected_encoding = mb_detect_encoding($ai_prompt, 'UTF-8, Windows-1252, ISO-8859-1', true);
        $converted_prompt = mb_convert_encoding($ai_prompt, 'UTF-8', $detected_encoding ?: 'UTF-8');
        if ($converted_prompt === false) {
            error_log("extractQuoteNumbersFromConversation (AI Scan) Error: Failed to convert prompt to UTF-8 from '$detected_encoding' for Conv ID {$conversationID_safe}.");
            return json_encode([]);
        }
        $ai_prompt = $converted_prompt;
    }
    
    // Call Gemini AI
    $apiUrl = apiURL_Flash(); // Assumed to be defined in ai_keys.php
    if (!$apiUrl) {
        error_log("extractQuoteNumbersFromConversation (AI Scan) Error: API URL is missing for Conv ID {$conversationID_safe}.");
        return json_encode([]);
    }

    $payload = [
        'contents' => [['parts' => [['text' => $ai_prompt]]]],
        'generationConfig' => [
            'temperature' => 0.1, // Low temperature for factual extraction
            'response_mime_type' => "application/json", // Request JSON directly
            'maxOutputTokens' => 50024, // Sufficient for a list of quote numbers
            'topP' => 0.95,
            'topK' => 40
        ]
    ];
    $headers = ['Content-Type: application/json'];

    $ch = curl_init($apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 20); // Connection timeout
    curl_setopt($ch, CURLOPT_TIMEOUT, 90);       // Total timeout for the call

    $response_body = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    if ($curlError || $httpCode != 200) {
        error_log("extractQuoteNumbersFromConversation (AI Scan) API Error for Conv ID {$conversationID_safe}: HTTP $httpCode, cURL: $curlError, Response: $response_body");
        return json_encode([]);
    }

    $responseData = json_decode($response_body, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("extractQuoteNumbersFromConversation (AI Scan) Error: Failed to decode main API JSON for Conv ID {$conversationID_safe}. Raw: " . $response_body);
        return json_encode([]);
    }

    if (isset($responseData['promptFeedback']['blockReason'])) {
        $blockReason = $responseData['promptFeedback']['blockReason'] ?? 'Unknown';
        error_log("extractQuoteNumbersFromConversation (AI Scan) Error: Gemini Content Blocked for Conv ID {$conversationID_safe}. Reason: " . $blockReason);
        return json_encode([]);
    }

    $extractedJsonText = null;
    if (isset($responseData['candidates'][0]['content']['parts'][0]['text'])) {
        $generatedText = $responseData['candidates'][0]['content']['parts'][0]['text'];
        $trimmedText = trim($generatedText);
        // Clean potential markdown (though API should return raw JSON with response_mime_type)
        if (substr($trimmedText, 0, 7) === "```json") {
            $trimmedText = substr($trimmedText, 7);
            if (substr($trimmedText, -3) === "```") {
                $trimmedText = substr($trimmedText, 0, -3);
            }
            $trimmedText = trim($trimmedText);
        } elseif (strpos($trimmedText, '```') === 0 && strrpos($trimmedText, '```') === (strlen($trimmedText) - 3) ) {
             $trimmedText = substr($trimmedText, 3, -3);
             $trimmedText = trim($trimmedText);
        }
        $extractedJsonText = $trimmedText;
    } else {
        error_log("extractQuoteNumbersFromConversation (AI Scan) Error: Could not extract text part from Gemini response for Conv ID {$conversationID_safe}. Raw: " . $response_body);
        return json_encode([]);
    }

    if ($extractedJsonText === null) {
         error_log("extractQuoteNumbersFromConversation (AI Scan) Error: Extracted JSON text is null for Conv ID {$conversationID_safe}. Raw Full Response: " . $response_body);
        return json_encode([]);
    }

    $quoteNumbersArray = json_decode($extractedJsonText, true);
    if (json_last_error() !== JSON_ERROR_NONE && is_array($quoteNumbersArray)) {
        // Ensure all quotes are uppercase and unique
        $processedQuotes = [];
        foreach ($quoteNumbersArray as $quote) {
            if (is_string($quote) && !empty(trim($quote)) && stripos(strtoupper(trim($quote)), 'TDU') === 0) {
                $processedQuotes[] = strtoupper(trim($quote));
            }
        }
        $uniqueQuoteNumbers = array_values(array_unique($processedQuotes));
        
        if (!empty($uniqueQuoteNumbers)) {
            error_log("extractQuoteNumbersFromConversation (AI Scan): Found quote numbers for Conv ID {$conversationID_safe}: " . implode(', ', $uniqueQuoteNumbers));
        } else {
            error_log("extractQuoteNumbersFromConversation (AI Scan): AI returned a valid JSON array, but it was empty or contained no valid TDU quotes for Conv ID {$conversationID_safe}. JSON from AI: " . $extractedJsonText);
        }
        return json_encode($uniqueQuoteNumbers);
    } else {
        error_log("extractQuoteNumbersFromConversation (AI Scan) Error: Failed to decode extracted JSON from AI, or it's not an array for Conv ID {$conversationID_safe}. Error: " . json_last_error_msg() . ". JSON Text: " . $extractedJsonText);
        // Fallback to regex on the AI's text output if JSON parsing failed but it's a string
        if (is_string($extractedJsonText)) {
            $pattern = '/\b(TDU[a-zA-Z0-9]+)\b/i';
            $matches = [];
            if (preg_match_all($pattern, $extractedJsonText, $matches)) {
                if (isset($matches[1]) && !empty($matches[1])) {
                    $uniqueQuoteNumbers = array_values(array_unique(array_map('strtoupper', $matches[1])));
                    error_log("extractQuoteNumbersFromConversation (AI Scan - Fallback Regex): Extracted quotes from AI's non-JSON string output for Conv ID {$conversationID_safe}: " . implode(', ', $uniqueQuoteNumbers));
                    return json_encode($uniqueQuoteNumbers);
                }
            }
        }
        error_log("extractQuoteNumbersFromConversation (AI Scan): No TDU quote numbers found after AI processing and fallback for Conv ID {$conversationID_safe}.");
        return json_encode([]);
    }
}


/**
 * [MODIFIED FUNCTION with Function Calling for Vendor Validation and Quote Number Prioritization]
 * Processes entire conversation, saves LATEST valid invoice to a local directory relative to this script,
 * extracts details.
 * Returns a single associative array with a 'saved_file_path' relative to this script's directory.
 */
function processConversationForInvoice(string $conversationID, mysqli $conn): ?array {
    if (empty($conversationID) || !$conn || $conn->connect_error) {
        error_log("processConversationForInvoice: Missing conversationID or invalid DB connection.");
        return null;
    }
    $conversationID_safe = mysqli_real_escape_string($conn, $conversationID); // For logging and paths
    $allPotentialInvoices = [];

    // Get all email bodies and attachment texts
    $conversationSources = getAllEmailMessagesAndAttachmentsInConversation($conversationID, $conn);
    if (empty($conversationSources['emails']) && empty($conversationSources['attachments'])) {
         error_log("processConversationForInvoice: No text content found for Conv ID {$conversationID_safe}.");
        return null; // No content to process
    }

    // Define save directory relative to this script (ai_email_functions.php)
    $baseSavePath = __DIR__;
    $invoiceSubDir = 'InvoiceFiles'; // Subdirectory name for invoices
    $targetDir = $baseSavePath . DIRECTORY_SEPARATOR . $invoiceSubDir;

    // Create directory if it doesn't exist and check writability
    if (!is_dir($targetDir)) {
        if (!mkdir($targetDir, 0777, true)) { // Create if not exists, recursive
             error_log("processConversationForInvoice: Failed to create directory '$targetDir' for Conv ID {$conversationID_safe}. Check permissions.");
             $targetDir = null; // Can't save if directory creation fails
        }
    } elseif (!is_writable($targetDir)) {
         error_log("processConversationForInvoice: Directory '$targetDir' is not writable for Conv ID {$conversationID_safe}.");
         $targetDir = null; // Can't save if not writable
    }
    
    // Consolidate all text sources (email bodies, attachment texts)
    $textSources = [];
    foreach ($conversationSources['emails'] as $email) {
        if (!empty($email['body_clean'])) {
            $textSources[] = [
                'type' => 'email_body',
                'content_text' => $email['body_clean'],
                'raw_content_for_saving' => null,
                'identifier' => $email['message_id'],
                'filename' => 'Email Body - MsgID ' . $email['message_id'],
                'received_datetime' => $email['received_datetime'],
                'email_subject_context' => $email['subject'] ?? 'N/A',
                'email_body_snippet_context' => mb_substr($email['body_clean'], 0, 500) . '...'
            ];
        }
    }
    foreach ($conversationSources['attachments'] as $attachment) {
        if (!$attachment['error'] && !empty($attachment['extracted_text'])) {
            $parentEmailContext = ['subject' => 'N/A', 'body_snippet' => 'N/A'];
            foreach($conversationSources['emails'] as $email_ctx) { // Renamed $email to avoid conflict in this inner loop
                if ($email_ctx['message_id'] === $attachment['parent_message_id']) {
                    $parentEmailContext['subject'] = $email_ctx['subject'] ?? 'N/A';
                    $parentEmailContext['body_snippet'] = mb_substr($email_ctx['body_clean'], 0, 500) . '...';
                    break;
                }
            }
            $textSources[] = [
                'type' => 'attachment',
                'content_text' => $attachment['extracted_text'],
                'raw_content_for_saving' => $attachment['raw_content'],
                'identifier' => $attachment['attachment_id'],
                'filename' => $attachment['filename'],
                'received_datetime' => $attachment['parent_email_received_datetime'],
                'email_subject_context' => $parentEmailContext['subject'],
                'email_body_snippet_context' => $parentEmailContext['body_snippet']
            ];
        }
    }

    // --- Fetch authoritative quote_no from vtiger_support ONCE per conversationID ---
    $db_quote_no_from_vtiger = null;
    $sql_vtiger_quote = "SELECT quote_no FROM vtiger_support WHERE ticketid = ?";
    $stmt_vtiger = $conn->prepare($sql_vtiger_quote);
    if ($stmt_vtiger) {
        $stmt_vtiger->bind_param("s", $conversationID);
        if ($stmt_vtiger->execute()) {
            $result_vtiger = $stmt_vtiger->get_result();
            if ($row_vtiger = $result_vtiger->fetch_assoc()) {
                 // Check if DB quote is valid (not empty, not 'N/A', starts with TDU)
                 if (!empty($row_vtiger['quote_no']) &&
                    strtoupper(trim($row_vtiger['quote_no'])) !== 'N/A' &&
                    stripos(strtoupper(trim($row_vtiger['quote_no'])), 'TDU') === 0) {
                    $db_quote_no_from_vtiger = strtoupper(trim($row_vtiger['quote_no']));
                    error_log("processConversationForInvoice: Found authoritative quote_no '{$db_quote_no_from_vtiger}' in vtiger_support for Conv ID {$conversationID}.");
                }
            }
            if ($result_vtiger instanceof mysqli_result) $result_vtiger->free();
        } else {
            error_log("processConversationForInvoice: Execute failed (vtiger_support quote check): " . $stmt_vtiger->error);
        }
        $stmt_vtiger->close();
    } else {
        error_log("processConversationForInvoice: Prepare failed (vtiger_support quote check): " . $conn->error);
    }
    // --- END Vtiger quote check ---

    // Process each text source for invoice details
    foreach ($textSources as $source) {
        $classification = classifyAttachmentContent($source['content_text']);
        error_log("Invoice Test - Source: {$source['filename']}, Classification: {$classification} for Conv ID {$conversationID_safe}"); // Added for debugging

        if ($classification === 'invoice') {
            error_log("processConversationForInvoice: Source '{$source['filename']}' classified as INVOICE for Conv ID {$conversationID_safe}. Attempting detail extraction.");
            
            // Define the Tool (Function Declaration) for the Gemini API
            $getVendorToolDeclaration = [
                'function_declarations' => [
                    [
                        'name' => 'getBestMatchingVendor',
                        'description' => "Takes an extracted vendor name from an invoice and finds the closest official match in the vendor database using fuzzy matching. Returns the official database name if found, otherwise null or 'N/A'.",
                        'parameters' => [
                            'type' => 'OBJECT',
                            'properties' => [
                                'extracted_name' => [
                                    'type' => 'STRING',
                                    'description' => 'The vendor name as extracted directly from the invoice document text.'
                                ]
                            ],
                            'required' => ['extracted_name']
                        ]
                    ]
                ]
            ];
            
            // Create the Prompt for the AI
            $extract_prompt = "Analyze 'Invoice Document Text'. Extract details, format dates YYYY-MM-DD. If you identify a vendor name, use the `getBestMatchingVendor` tool to validate it and get the official name.\n\nEmail Context:\nSubject: " . $source['email_subject_context'] . "\nEmail Body Snippet: " . $source['email_body_snippet_context'] . "\n\nTasks:\n1. **Invoice Number:** Primary invoice number.\n2. **Invoice Total:** Final total (NUMERIC ONLY, e.g., '123.45'). Use '.' decimal separator. Remove currency/commas. N/A if none.\n3. **Invoice Date:** Issue date. Format YYYY-MM-DD. N/A if none.\n4. **Quote Number:** Extract the quote number if present from the Document Text OR the Email Context. It typically starts with 'TDU' (case-insensitive) followed by digits (e.g., TDU12345, tdu00567). Convert to uppercase. If multiple are found, prefer the latest or most prominent one. If none, output 'N/A'.\n5. **Invoice Vendor Name (Extraction & Validation Call):** Identify the vendor name *exactly as written* in the document. Then, call the `getBestMatchingVendor` tool with this extracted name. Use the tool's result (the official/matched name or 'N/A') as the value for the final `invoiceVendorName` field.\n6. **Context Match:** Is this invoice relevant to the 'Email Context'? 'yes'/'no'.\n7. **Match Log:** Explain `matches` decision and clearly state the outcome of vendor validation via the tool.\n\nOutput Format Constraint:\nSTRICTLY single line tuple: ('invoiceNumber_value', 'invoiceTotal_value', 'invoiceDate_value', 'quoteNumber_value', 'invoiceVendorName_value', 'matches_value', 'log_value')\n\nFormatting Rules:\n- 'N/A' for undetermined. `invoiceTotal`: NUMERIC string or 'N/A'. `invoiceDate` as YYYY-MM-DD. `quoteNumber_value` should be uppercase 'TDU' followed by digits, or 'N/A'. `invoiceVendorName`: Use the result FROM THE TOOL, or 'N/A'. `matches`: 'yes'/'no'. ONLY single quotes. Comma+space. NO other text/markdown.\n\nExample (Tool finds vendor, Quote present): ('INV-123', '450.00', '2024-10-30', 'TDU54321', 'Official Vendor Name From DB', 'yes', 'Vendor \\'Original Extracted Name\\' validated via tool to \\'Official Vendor Name From DB\\'. Quote TDU54321 found. Details fit.')\nExample (Tool does not find vendor, No Quote): ('AB/456', '95.50', '2024-11-01', 'N/A', 'N/A', 'no', 'Vendor \\'Original Extracted Name\\' not found via tool. Content unrelated.')\n\nInvoice Document Text:\n" . $source['content_text'];
            
            $prompt_for_ai_utf8 = $extract_prompt;
            // Ensure UTF-8 encoding
            if (!mb_check_encoding($prompt_for_ai_utf8, 'UTF-8')) {
                $detected_encoding = mb_detect_encoding($prompt_for_ai_utf8, 'UTF-8, ISO-8859-1, Windows-1252', true);
                $converted = mb_convert_encoding($prompt_for_ai_utf8, 'UTF-8', $detected_encoding ?: 'UTF-8');
                if ($converted === false) { error_log("processConversationForInvoice: UTF-8 convert fail for source '{$source['filename']}' from '$detected_encoding' for Conv ID $conversationID_safe."); continue; }
                $prompt_for_ai_utf8 = $converted;
            }
             //error_log($prompt_for_ai_utf8);
            $apiUrl = apiURL_Flash();
            if (!$apiUrl) { error_log("processConversationForInvoice: API URL missing for Conv ID {$conversationID_safe}. Skipping source '{$source['filename']}'."); continue; }

            // API Call Sequence (potentially multi-turn for function calling)
            $apiCallTry = 0;
            $maxApiCallTries = 2;
            $finalAiGeneratedTupleText = null;
            $conversationHistory = [
                [
                    'role' => 'user',
                    'parts' => [['text' => $prompt_for_ai_utf8]]
                ]
            ];

            while ($apiCallTry < $maxApiCallTries) {
                $apiCallTry++;
                error_log("processConversationForInvoice: API Call Try #{$apiCallTry} for source '{$source['filename']}', Conv ID {$conversationID_safe}");

                $currentApiPayload = [
                    'contents' => $conversationHistory,
                    'tools' => [$getVendorToolDeclaration],
                    'generationConfig' => [ 'temperature' => 0.1, 'maxOutputTokens' => 50000, 'topP' => 0.95, 'topK' => 40 ]
                ];

                $headers = ['Content-Type: application/json'];
                $ch_extract = curl_init($apiUrl);
                curl_setopt_array($ch_extract, [
                    CURLOPT_RETURNTRANSFER => true, CURLOPT_POST => true,
                    CURLOPT_POSTFIELDS => json_encode($currentApiPayload),
                    CURLOPT_HTTPHEADER => $headers,
                    CURLOPT_CONNECTTIMEOUT => 20, CURLOPT_TIMEOUT => 120
                ]);
                $extract_response_body = curl_exec($ch_extract);
                $extract_httpCode = curl_getinfo($ch_extract, CURLINFO_HTTP_CODE);
                $extract_curlError = curl_error($ch_extract);
                curl_close($ch_extract);

                if ($extract_curlError || $extract_httpCode !== 200) {
                    error_log("processConversationForInvoice: AI API Error (Try {$apiCallTry}) for source '{$source['filename']}'. Err: {$extract_curlError}, HTTP: {$extract_httpCode}, Resp: {$extract_response_body}");
                    if ($apiCallTry >= $maxApiCallTries) break;
                    sleep(1); continue;
                }
                $extract_responseData = json_decode($extract_response_body, true);
                if (json_last_error() !== JSON_ERROR_NONE) {
                    error_log("processConversationForInvoice: Failed to decode AI JSON (Try {$apiCallTry}) for source '{$source['filename']}'. Raw: " . $extract_response_body);
                    if ($apiCallTry >= $maxApiCallTries) break;
                    sleep(1); continue;
                }
                 if (!isset($extract_responseData['candidates'][0]['content']['parts'][0])) {
                     if (isset($extract_responseData['promptFeedback']['blockReason'])) {
                          error_log("processConversationForInvoice: AI Content Blocked (Try {$apiCallTry}) for source '{$source['filename']}'. Reason: ". ($extract_responseData['promptFeedback']['blockReason'] ?? 'Unknown'));
                     } else {
                          error_log("processConversationForInvoice: AI response (Try {$apiCallTry}) for source '{$source['filename']}' missing expected content structure. Raw: ". $extract_response_body);
                     }
                     break;
                 }
                
                $modelResponsePart = $extract_responseData['candidates'][0]['content']; // This contains the 'role' and 'parts'
                $conversationHistory[] = $modelResponsePart; // Add model's full response part to history

                if (isset($modelResponsePart['parts'][0]['functionCall'])) {
                    $functionCall = $modelResponsePart['parts'][0]['functionCall'];
                    if ($functionCall['name'] === 'getBestMatchingVendor') {
                        error_log("processConversationForInvoice: Received function call request for getBestMatchingVendor from AI for source '{$source['filename']}'.");
                        $args = $functionCall['args'] ?? [];
                        $extractedNameArg = $args['extracted_name'] ?? null;
                        $functionResultContent = 'N/A';
                        if ($extractedNameArg !== null) {
                            if (!function_exists('getBestMatchingVendor')) {
                                 error_log("processConversationForInvoice: FATAL - Helper function getBestMatchingVendor is MISSING! Cannot execute function call.");
                            } else {
                                error_log("processConversationForInvoice: Calling local getBestMatchingVendor with name: '{$extractedNameArg}' for source '{$source['filename']}'.");
                                //$matchedVendorName = getBestMatchingVendor($extractedNameArg, $conn);
                                $matchedVendorName = getBestMatchingVendor($extractedNameArg, $conn);
                                $functionResultContent = $matchedVendorName ?? 'N/A';
                                error_log("processConversationForInvoice: Local getBestMatchingVendor returned: '{$functionResultContent}' for source '{$source['filename']}'.");
                            }
                        } else {
                            error_log("processConversationForInvoice: Missing 'extracted_name' argument from AI for getBestMatchingVendor call. Source '{$source['filename']}'.");
                        }
                        
                        $conversationHistory[] = [ // Add tool's response to history
                            'role' => 'tool',
                            'parts' => [[
                                'functionResponse' => [
                                    'name' => 'getBestMatchingVendor',
                                    'response' => [ 'content' => $functionResultContent ]
                                ]
                            ]]
                        ];
                        error_log("processConversationForInvoice: Sending function response back to API for source '{$source['filename']}'.");
                        continue; // Continue to the next API call with updated history
                    } else {
                         error_log("processConversationForInvoice: AI requested unknown function '{$functionCall['name']}'. Source '{$source['filename']}'. Breaking loop.");
                         break; // Unknown function requested
                    }
                }
                elseif (isset($modelResponsePart['parts'][0]['text'])) { // This is the final text response
                    $finalAiGeneratedTupleText = $modelResponsePart['parts'][0]['text'];
                    error_log("processConversationForInvoice: Received final text response from AI for source '{$source['filename']}'.");
                    break; // Exit the API call loop, we have the final text
                }
                else { // Unexpected response structure from AI
                     error_log("processConversationForInvoice: AI response (Try {$apiCallTry}) for source '{$source['filename']}' had no text or function call. Raw: " . $extract_response_body);
                     if ($apiCallTry >= $maxApiCallTries) break;
                     sleep(1); continue;
                }
            } // End API call loop

            if ($finalAiGeneratedTupleText === null) {
                error_log("processConversationForInvoice: Null AI tuple extract after all tries for source '{$source['filename']}'. Skipping this source.");
                continue; // Skip to next source
            }

            $trimmed_extract_text = trim(preg_replace("/^\xEF\xBB\xBF/", '', $finalAiGeneratedTupleText));
            // Regex for: ('invNum', 'total', 'date', 'quoteNum', 'vendorName', 'matches', 'log')
            $extract_pattern = "/^\(\s*'((?:\\\\'|[^'\\\\])*)'\s*,\s*'((?:N\/A|(?:-?\d+(?:\.\d+)?)))'\s*,\s*'((?:\\\\'|[^'\\\\])*)'\s*,\s*'((?:TDU\d+|N\/A))'\s*,\s*'((?:\\\\'|[^'\\\\])*)'\s*,\s*'((?:yes|no|N\/A))'\s*,\s*'((?:\\\\'|[^'\\\\])*)'\s*\)$/ui";
            $extract_matches = [];

            if (preg_match($extract_pattern, $trimmed_extract_text, $extract_matches)) {
                $ai_extracted_quote_number = strtoupper(stripslashes($extract_matches[4]));

                // Prioritize DB quote number
                $final_quote_number = $db_quote_no_from_vtiger ?? $ai_extracted_quote_number;
                if ($db_quote_no_from_vtiger && $db_quote_no_from_vtiger !== $ai_extracted_quote_number && $ai_extracted_quote_number !== 'N/A') {
                    error_log("processConversationForInvoice: Overriding AI extracted quote '{$ai_extracted_quote_number}' with DB quote '{$db_quote_no_from_vtiger}' for ConvID {$conversationID}.");
                } elseif ($db_quote_no_from_vtiger && $ai_extracted_quote_number === 'N/A') {
                     error_log("processConversationForInvoice: Using DB quote '{$db_quote_no_from_vtiger}'. AI did not find a quote for ConvID {$conversationID}.");
                }

                $allPotentialInvoices[] = [
                    'source_type'         => $source['type'],
                    'source_identifier'   => $source['identifier'],
                    'source_filename'     => $source['filename'],
                    'received_datetime'   => $source['received_datetime'],
                    'invoiceNumber'       => stripslashes($extract_matches[1]),
                    'invoiceTotal'        => stripslashes($extract_matches[2]),
                    'invoiceDate'         => stripslashes($extract_matches[3]),
                    'quoteNumber'         => $final_quote_number, // Use the prioritized one
                    'invoiceVendorName'   => stripslashes($extract_matches[5]), // This comes from the function call result via AI
                    'matches'             => stripslashes($extract_matches[6]),
                    'log'                 => stripslashes($extract_matches[7]),
                    'ai_parse_status'     => 'success',
                    'saved_file_path'     => null,
                    'raw_content_for_saving' => $source['raw_content_for_saving']
                ];
                error_log("processConversationForInvoice: Successfully parsed invoice tuple for source '{$source['filename']}'. Vendor: '" . stripslashes($extract_matches[5]) . "', Final Quote: '" . $final_quote_number . "'");
            } else {
                error_log("processConversationForInvoice: Failed to parse FINAL AI invoice tuple for source '{$source['filename']}'. Received: " . $trimmed_extract_text . ". Regex: " . $extract_pattern);
            }
        } // End if classification is 'invoice'
    } // End foreach $textSource loop

    if (empty($allPotentialInvoices)) {
         error_log("processConversationForInvoice: No valid invoices identified or parsed for Conv ID {$conversationID_safe}.");
        return null;
    }

    // Sort invoices to get the latest one
    usort($allPotentialInvoices, function($a, $b) {
        $timeA = strtotime($a['received_datetime']);
        $timeB = strtotime($b['received_datetime']);
        if ($timeB != $timeA) return $timeB - $timeA;
        if ($a['source_type'] === 'attachment' && $b['source_type'] === 'email_body') return -1;
        if ($a['source_type'] === 'email_body' && $b['source_type'] === 'attachment') return 1;
        return 0;
    });

    $latestInvoice = $allPotentialInvoices[0];
    error_log("processConversationForInvoice: Selected latest invoice source '{$latestInvoice['source_filename']}' received at {$latestInvoice['received_datetime']} for Conv ID {$conversationID_safe}.");

    // Save the attachment if it's the source of the latest invoice
    if ($latestInvoice['source_type'] === 'attachment' && $targetDir !== null && !empty($latestInvoice['raw_content_for_saving'])) {
        $originalFilename = $latestInvoice['source_filename'];
        $filenameWithoutExt = pathinfo($originalFilename, PATHINFO_FILENAME);
        $extension = pathinfo($originalFilename, PATHINFO_EXTENSION);
        $safeFilenameBase = preg_replace('/[^a-zA-Z0-9._-]/', '_', $filenameWithoutExt);
        $formattedTimestamp = date('YmdHis'); // Using YYYYMMDDHHMMSS
        $newFilename = $safeFilenameBase . "_" . $formattedTimestamp . ($extension ? '.' . $extension : '');
        
        $relativeSavePath = $invoiceSubDir . DIRECTORY_SEPARATOR . $newFilename;
        $fullSavePath = $targetDir . DIRECTORY_SEPARATOR . $newFilename;

        if (file_put_contents($fullSavePath, $latestInvoice['raw_content_for_saving']) !== false) {
            $latestInvoice['saved_file_path'] = $relativeSavePath; // Store the relative path
            error_log("processConversationForInvoice: Saved latest invoice attachment '{$originalFilename}' to '{$fullSavePath}' (reported as '{$relativeSavePath}').");
        } else {
            error_log("processConversationForInvoice: FAILED to write latest invoice attachment '{$originalFilename}' to '{$fullSavePath}'. Check permissions/disk space.");
        }
    }
    unset($latestInvoice['raw_content_for_saving']);
    // Clean up old key if it was ever used/present
    if (isset($latestInvoice['temporary_file_path'])) {
        unset($latestInvoice['temporary_file_path']);
    }

    return $latestInvoice;
}

/**
 * Classifies the primary purpose of an entire conversation.
 *
 * Iterates through all email bodies and attachment texts in a conversation,
 * classifies each piece using `classifyAttachmentContent`, and then determines
 * if the conversation as a whole primarily relates to an invoice, a confirmation,
 * both, or something else.
 *
 * @param string $conversationID The ID of the conversation to classify.
 * @param mysqli $conn The database connection object.
 * @return string Returns 'invoice', 'confirmation', 'both', or 'other'.
 *                Returns 'other' if no definitive classification can be made or on error.
 */
function classifyConversationPurpose(string $conversationID, mysqli $conn): string {
    if (empty($conversationID) || !$conn || $conn->connect_error) {
        error_log("classifyConversationPurpose: Missing conversationID or invalid DB connection.");
        return 'other'; // Default to 'other' on critical input/DB error
    }

    $conversationID_safe = mysqli_real_escape_string($conn, $conversationID);
    $conversationSources = getAllEmailMessagesAndAttachmentsInConversation($conversationID_safe, $conn);

    if (empty($conversationSources['emails']) && empty($conversationSources['attachments'])) {
        error_log("classifyConversationPurpose: No text content found for Conv ID {$conversationID_safe} to classify.");
        return 'other'; // No content to analyze
    }

    $foundInvoice = false;
    $foundConfirmation = false;

    // Analyze email bodies
    foreach ($conversationSources['emails'] as $email) {
        if (!empty(trim($email['body_clean']))) {
            $classification = classifyAttachmentContent(trim($email['body_clean']));
            // error_log("classifyConversationPurpose: Email body (MsgID {$email['message_id']}) classified as: {$classification}");
            if ($classification === 'invoice') {
                $foundInvoice = true;
            } elseif ($classification === 'confirmation') {
                $foundConfirmation = true;
            }
            // If both found, we can potentially short-circuit, but let's check all content
            // for a more comprehensive view, in case one classification was weak.
        }
    }

    // Analyze attachment texts
    foreach ($conversationSources['attachments'] as $attachment) {
        if (!$attachment['error'] && !empty(trim($attachment['extracted_text']))) {
            $classification = classifyAttachmentContent(trim($attachment['extracted_text']));
            // error_log("classifyConversationPurpose: Attachment ('{$attachment['filename']}') classified as: {$classification}");
            if ($classification === 'invoice') {
                $foundInvoice = true;
            } elseif ($classification === 'confirmation') {
                $foundConfirmation = true;
            }
        }
    }

    // Determine final classification
    if ($foundInvoice && $foundConfirmation) {
        error_log("classifyConversationPurpose: Conv ID {$conversationID_safe} classified as 'both'.");
        return 'both';
    } elseif ($foundInvoice) {
        error_log("classifyConversationPurpose: Conv ID {$conversationID_safe} classified as 'invoice'.");
        return 'invoice';
    } elseif ($foundConfirmation) {
        error_log("classifyConversationPurpose: Conv ID {$conversationID_safe} classified as 'confirmation'.");
        return 'confirmation';
    } else {
        error_log("classifyConversationPurpose: Conv ID {$conversationID_safe} classified as 'other'.");
        return 'other';
    }
}


/**
 * Defines the prompt for the AI to extract contact and organization information
 * with a focus on email signatures.
 */
function get_ai_contact_extraction_prompt_text(): string {
    return "Analyze the following email content, which may include multiple email messages in a thread and text from attachments. Pay close attention to email signatures to extract contact and organization details.

    Your goal is to identify the most relevant or most recent complete set of contact/organization information, likely from the primary sender or a key correspondent within the provided text.

    Extract the following information:
    - organization_name: The name of the company or organization, give full name instead of abbreviation if possible.
    - contact_name: The full name of the primary contact person.
    - contact_email: The email address of the contact person.
    - contact_phone: The phone number of the contact person.
    - organization_website: The website URL of the organization.
    - organization_address: The full street address of the organization.
    - organization_country: The country of the organization, derived from the address.
    - organization_region:
        - If 'organization_country' is 'India' (or common variations like 'IN'), determine the region (South, West, North, East) based on the 'organization_address'.
            - South India typically includes states/cities like Karnataka (Bangalore), Kerala, Tamil Nadu (Chennai), Andhra Pradesh, Telangana (Hyderabad).
            - West India typically includes states/cities like Maharashtra (Mumbai, Pune), Gujarat (Ahmedabad), Goa.
            - North India typically includes states/cities like Delhi, Punjab, Haryana, Uttar Pradesh (Noida), Rajasthan (Jaipur).
            - East India typically includes states/cities like West Bengal (Kolkata), Odisha, Bihar, Jharkhand.
        - If the country is not India, or if the region within India cannot be reliably determined from the address, output 'N/A'.

    CRITICAL INSTRUCTIONS:
    0.  Turtle Down Under is exempted from this, anything relating to this organisation is not to be extracted
    1.  If any piece of information is not found or cannot be reliably determined from the provided text, use the string 'N/A' as its value.
    2.  Focus on information present in email signatures first. If multiple signatures exist, try to determine the most relevant or latest one from the entire text.
    3.  The output MUST be a single, valid JSON object. Do not include any text before or after the JSON object, and do not use markdown code blocks like ```json.
    4.  Ensure all field names in the JSON output are exactly as listed above (e.g., 'organization_name').

    Example of a valid JSON output:
    ```json
    {
      \"organization_name\": \"Acme Innovations Ltd.\",
      \"contact_name\": \"Priya Sharma\",
      \"contact_email\": \"priya.sharma@acmeinnovations.co.in\",
      \"contact_phone\": \"+91 80 1234 5678\",
      \"organization_website\": \"www.acmeinnovations.co.in\",
      \"organization_address\": \"#42, MG Road, Bangalore, Karnataka 560001, India\",
      \"organization_country\": \"India\",
      \"organization_region\": \"South\"
    }
    Example with missing information:
    {
      \"organization_name\": \"Global Exports\",
      \"contact_name\": \"John Smith\",
      \"contact_email\": \"jsmith@globalexports.com\",
      \"contact_phone\": \"N/A\",
      \"organization_website\": \"N/A\",
      \"organization_address\": \"1500 Commerce St, Dallas, TX, USA\",
      \"organization_country\": \"USA\",
      \"organization_region\": \"N/A\"
    }

    Email Content is below:
    ";
}


function extractContactAndOrganizationInfo(string $conversationID, mysqli $conn): array {
    $default_result = [
    'success' => false,
    'error' => null,
    'data' => [ // Default structure with 'N/A'
        'organization_name' => 'N/A',
        'contact_name' => 'N/A',
        'contact_email' => 'N/A',
        'contact_phone' => 'N/A',
        'organization_website' => 'N/A',
        'organization_address' => 'N/A',
        'organization_country' => 'N/A',
        'organization_region' => 'N/A'
    ],
    'raw_ai_response' => null
    ];
    if (!$conn || $conn->connect_error) {
         $default_result['error'] = 'Database connection failed: ' . ($conn->connect_error ?? 'Unknown DB error');
         error_log("extractContactAndOrganizationInfo: " . $default_result['error']);
         return $default_result;
    }
    if (empty($conversationID)) {
        $default_result['error'] = 'Conversation ID not provided.';
        error_log("extractContactAndOrganizationInfo: " . $default_result['error']);
        return $default_result;
    }

    // Fetch all email messages and attachments content for the conversation
    $allConversationMessagesAndAttachments = getAllEmailMessagesAndAttachmentsInConversation($conversationID, $conn);
    $fullContentForAI = "";

    // Concatenate all email bodies
    foreach($allConversationMessagesAndAttachments['emails'] as $email) {
        $fullContentForAI .= "\n\n--- Email Start (Received: " . $email['received_datetime'] . " Sender: " . $email['sender'] . ") ---\n";
        $fullContentForAI .= "Subject: " . $email['subject'] . "\n";
        $fullContentForAI .= "Body:\n" . $email['body_clean'] . "\n"; // Use cleaned body
        $fullContentForAI .= "--- Email End ---\n";
    }

    // Concatenate all attachment texts
    foreach ($allConversationMessagesAndAttachments['attachments'] as $attachment) {
        if (!$attachment['error'] && !empty($attachment['extracted_text'])) {
            $fullContentForAI .= "\n\n--- Attachment: " . htmlspecialchars($attachment['filename']) . " (From email received: " . $attachment['parent_email_received_datetime'] . ") ---\n";
            $fullContentForAI .= $attachment['extracted_text'] . "\n";
            $fullContentForAI .= "--- Attachment End ---\n";
        }
    }
    $fullContentForAI = trim($fullContentForAI);

    if (empty($fullContentForAI)) {
        $default_result['error'] = 'Cannot analyze: Email bodies and attachments are effectively empty for contact extraction.';
        error_log("extractContactAndOrganizationInfo: " . $default_result['error'] . " for ConvID {$conversationID}");
        return $default_result;
    }

    $promptText = get_ai_contact_extraction_prompt_text() . $fullContentForAI;

    $apiUrl = apiURL_Flash();
    if (!$apiUrl) {
        error_log("extractContactAndOrganizationInfo: API URL is missing.");
        $default_result['error'] = 'API URL not configured for Gemini.';
        return $default_result;
    }

    $payload = [
        'contents' => [['parts' => [['text' => $promptText]]]],
        'generationConfig' => [
            'temperature' => 0.1,
            'response_mime_type' => "application/json", // Request JSON directly
            'maxOutputTokens' => 50048,
            'topP' => 0.95,
            'topK' => 40
        ]
    ];
    $headers = ['Content-Type: application/json'];

    $ch = curl_init($apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 20);
    curl_setopt($ch, CURLOPT_TIMEOUT, 90);

    $response_body = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    $default_result['raw_ai_response'] = $response_body;

    if ($curlError || $httpCode != 200) {
        $api_error_message = "Gemini API Error (Contact Extraction HTTP $httpCode, cURL $curlError): " . $response_body;
        error_log("extractContactAndOrganizationInfo: " . $api_error_message);
        $details = json_decode($response_body, true);
        $default_result['error'] = "Gemini API Error (Contact Extraction HTTP $httpCode)" . (isset($details['error']['message']) ? ': ' . $details['error']['message'] : '');
        return $default_result;
    }

    $responseData = json_decode($response_body, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("extractContactAndOrganizationInfo: Failed to decode API JSON (Outer response). Error: " . json_last_error_msg() . ". Raw: " . $response_body);
        $default_result['error'] = 'Gemini API: Failed to decode main JSON response.';
        return $default_result;
    }

    if (isset($responseData['promptFeedback']['blockReason'])) {
        $blockReason = $responseData['promptFeedback']['blockReason'] ?? 'Unknown';
        error_log("extractContactAndOrganizationInfo: Gemini Content Blocked: " . $blockReason . ". Details: " . json_encode($responseData['promptFeedback']));
        $default_result['error'] = 'Gemini: Content blocked (' . $blockReason . ')';
        return $default_result;
    }

    $extractedJsonText = null;
    // Path to text when response_mime_type is application/json
    if (isset($responseData['candidates'][0]['content']['parts'][0]['text'])) {
        $generatedText = $responseData['candidates'][0]['content']['parts'][0]['text'];
        $trimmedText = trim($generatedText);
        // Remove markdown ```json if present (though API should return raw with response_mime_type)
        if (substr($trimmedText, 0, 7) === "```json") {
            $trimmedText = substr($trimmedText, 7);
            if (substr($trimmedText, -3) === "```") {
                $trimmedText = substr($trimmedText, 0, -3);
            }
            $trimmedText = trim($trimmedText);
        } elseif (strpos($trimmedText, '```') === 0 && strrpos($trimmedText, '```') === (strlen($trimmedText) - 3) ) {
             $trimmedText = substr($trimmedText, 3, -3);
             $trimmedText = trim($trimmedText);
        }
        $extractedJsonText = $trimmedText;
    } else {
        error_log("extractContactAndOrganizationInfo: Could not extract text part from Gemini response. Raw: " . $response_body);
        $default_result['error'] = 'Gemini API: No text part in response candidates.';
        return $default_result;
    }

    if ($extractedJsonText === null) {
        error_log("extractContactAndOrganizationInfo: Extracted JSON text is null. Raw Gemini Response: " . $response_body);
        $default_result['error'] = 'Gemini API: Failed to extract JSON content string from response.';
        return $default_result;
    }

    $parsedData = json_decode($extractedJsonText, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("extractContactAndOrganizationInfo: Failed to decode extracted JSON from AI. Error: " . json_last_error_msg() . ". JSON Text attempted: " . $extractedJsonText);
        $default_result['error'] = 'Failed to parse AI JSON response.';
        return $default_result;
    }

    // Populate result with data from AI, ensuring all expected keys are present
    $expectedKeys = array_keys($default_result['data']);
    foreach ($expectedKeys as $key) {
        if (isset($parsedData[$key])) {
            $default_result['data'][$key] = $parsedData[$key];
        } // If not set, it keeps the default 'N/A'
    }

    $default_result['success'] = true;
    return $default_result;
}

/**
 * Sends an entire file (e.g., PDF, Word, Excel) directly to the Gemini AI for statement data extraction.
 * This method is more robust for files with complex layouts where local text extraction might fail,
 * as it allows the AI to analyze the file's structure and content together.
 *
 * @param string $filePath The absolute path to the file to be processed.
 * @return array An associative array containing:
 *               'success' (bool): True if the process completed and returned valid data.
 *               'error' (string|null): An error message if something went wrong.
 *               'data' (array): The structured data extracted by the AI as an array of line items.
 *               'raw_ai_json_text' (string|null): The raw JSON string returned by the AI for debugging.
 */
function extractStatementDataAI_fromFile(string $filePath): array {
    $default_response = [
        'success' => false,
        'error' => 'AI processing not initiated.',
        'data' => [],
        'raw_ai_json_text' => null
    ];

    // 1. Validate the file path
    if (!file_exists($filePath) || !is_readable($filePath)) {
        $default_response['error'] = "File does not exist or is not readable at path: " . htmlspecialchars($filePath);
        error_log("extractStatementDataAI_fromFile Error: " . $default_response['error']);
        return $default_response;
    }

    // 2. Get file content and MIME type
    $fileContent = file_get_contents($filePath);
    if ($fileContent === false) {
        $default_response['error'] = "Failed to read file content from path: " . htmlspecialchars($filePath);
        error_log("extractStatementDataAI_fromFile Error: " . $default_response['error']);
        return $default_response;
    }

    $mime_type = mime_content_type($filePath);
    if ($mime_type === false) {
        $default_response['error'] = "Could not determine MIME type for file: " . htmlspecialchars($filePath);
        error_log("extractStatementDataAI_fromFile Error: " . $default_response['error']);
        return $default_response;
    }

    // 3. Encode file content to Base64
    $base64EncodedData = base64_encode($fileContent);

    // 4. Construct the AI prompt and payload
    $prompt = "
    Analyze the attached statement file. The file could be a PDF, an image, or a spreadsheet.
    Extract all applicable financial transaction line items you can find within the document.

    Fields to extract for EACH line item:
    - 'supplier_name': The name of the supplier or vendor.
    - 'supplier_email_domain': The email domain (e.g., 'example.com') from any supplier email address found.
    - 'tdu_class_number': A specific reference number that always starts with 'TDU', case-insensitive, followed by numbers, and sometimes ending with 'G'. There should be no spaces (e.g., 'TDU12345G', 'tdu88333').
    - 'passenger_name': The name of the passenger or client associated with the transaction.
    - 'confirmation_number': Any booking or confirmation reference number.
    - 'invoice_number': The specific invoice number for the line item.
    - 'service_date': The date the service was rendered or the transaction occurred (Format: YYYY-MM-DD).
    - 'invoice_amount': The monetary value of the line item. Extract only the numeric value. Do not add amounts together.
    - 'description': A brief description of the service or product.

    Important Rules:
    1.  Our company is 'Turtle Down Under'. Never extract 'Turtle Down Under' as the 'supplier_name'. It is the recipient.
    2.  The final output MUST be a single, valid JSON array where each element is an object representing one extracted line item.
    3.  If a specific field for a line item cannot be found in the document, use the string 'N/A' as its value.
    4.  Ensure your entire response is ONLY the JSON array, with no explanatory text or markdown code blocks before or after it.

    Example Output:
    [{\"supplier_name\": \"Touring Melbourne Group\", \"supplier_email_domain\": \"touringmelbourne.com.au\", \"tdu_class_number\": \"TDU27014G\", \"passenger_name\": \"John Smith\", \"confirmation_number\": \"CONF555\", \"invoice_number\": \"INV-101\", \"service_date\": \"2024-10-15\", \"invoice_amount\": \"150.00\", \"description\": \"City Tour\"}]
    ";

    $apiUrl = apiURL_Flash();
    if (!$apiUrl) {
        $default_response['error'] = 'AI API URL is not configured.';
        error_log("extractStatementDataAI_fromFile Error: " . $default_response['error']);
        return $default_response;
    }

    // The payload now includes separate parts for the text prompt and the file data.
    $payload = [
        'contents' => [[
            'parts' => [
                ['text' => $prompt],
                [
                    'inline_data' => [
                        'mime_type' => $mime_type,
                        'data' => $base64EncodedData
                    ]
                ]
            ]
        ]],
        'generationConfig' => [
            'temperature' => 0.2,
            'response_mime_type' => "application/json",
            'maxOutputTokens' => 28192 // Ensure this is adequate for large statements
        ]
    ];

    // 5. Make the API call
    $headers = ['Content-Type: application/json'];
    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_TIMEOUT => 300 // Increased timeout for file processing
    ]);

    $response_body = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    // 6. Process the API response
    if ($curlError || $httpCode != 200) {
        $default_response['error'] = "AI API call failed (HTTP $httpCode). " . $curlError;
        error_log("extractStatementDataAI_fromFile Error: " . $default_response['error'] . " | Response: " . $response_body);
        return $default_response;
    }

    $responseData = json_decode($response_body, true);
    if (isset($responseData['promptFeedback']['blockReason'])) {
        $default_response['error'] = "AI: Content blocked due to: " . ($responseData['promptFeedback']['blockReason'] ?? 'Unknown reason');
        error_log("extractStatementDataAI_fromFile Error: " . $default_response['error']);
        return $default_response;
    }

    $extractedJsonText = null;
    if (isset($responseData['candidates'][0]['content']['parts'][0]['text'])) {
        $extractedJsonText = trim($responseData['candidates'][0]['content']['parts'][0]['text']);
        $default_response['raw_ai_json_text'] = $extractedJsonText;
    } else {
        $default_response['error'] = 'AI API: No text part in response.';
        error_log("extractStatementDataAI_fromFile Error: " . $default_response['error'] . " | Response: " . $response_body);
        return $default_response;
    }

    $parsedData = json_decode($extractedJsonText, true);
    if (json_last_error() === JSON_ERROR_NONE && is_array($parsedData)) {
        $default_response['success'] = true;
        $default_response['error'] = null;
        $default_response['data'] = $parsedData;
    } else {
        $default_response['error'] = 'Failed to parse AI JSON response. Error: ' . json_last_error_msg();
        error_log("extractStatementDataAI_fromFile Error: " . $default_response['error'] . " | Raw JSON Text: " . $extractedJsonText);
    }

    return $default_response;
}

?>