<?php
// --- Database Configuration ---
require_once "dbconn.php"; // Assuming dbconn.php establishes $conn
include "header.php";

// --- Pagination Configuration ---
define('ITEMS_PER_PAGE', 25); // Number of items to display per page for the detailed log

// --- Check Connection ---
if ($conn->connect_error) {
    die("Connection failed: " . $conn->connect_error);
}

// --- Helper function to safely prepare data for htmlspecialchars ---
function make_string_for_html($value, $default = 'N/A') {
    if (is_null($value) || $value === '') {
        return $default;
    }
    if (is_string($value) || is_numeric($value)) {
        return (string) $value;
    }
    if (is_array($value) || is_object($value)) {
        return json_encode($value, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    }
    return (string) $value;
}

// --- Function to parse the 'reason' string ---
function parseReasonDetails($reasonString) {
    $details = [
        'segment' => null,
        'duration' => null,
        'city' => null,
        'country' => null,
        'seats' => null,
        'cat_filter' => null,
        'full_input_string' => null,
        'error_summary' => $reasonString
    ];
    if (empty($reasonString)) return $details;
    if (preg_match('/segment\s+(\d+)/i', $reasonString, $segmentMatches)) {
        $details['segment'] = trim($segmentMatches[1]);
    }
    if (preg_match('/\(Input:\s*([^\)]+)\)/i', $reasonString, $inputMatches)) {
        $details['full_input_string'] = trim($inputMatches[1]);
        $pairs = explode(',', $details['full_input_string']);
        $parsed_params = [];
        foreach ($pairs as $pair) {
            $parts = explode('=', $pair, 2);
            if (count($parts) === 2) {
                $key = strtolower(trim($parts[0]));
                $value = trim($parts[1]);
                $parsed_params[$key] = $value;
            }
        }
        $details['duration']   = $parsed_params['dur'] ?? ($parsed_params['duration'] ?? null);
        $details['city']       = $parsed_params['city'] ?? null;
        $details['country']    = $parsed_params['country'] ?? null;
        $details['seats']      = $parsed_params['seats'] ?? null;
        $details['cat_filter'] = $parsed_params['catfilter'] ?? null;
        if (stripos($reasonString, "No template found") === 0) {
             $details['error_summary'] = "Missing template for Segment " . ($details['segment'] ?? 'N/A');
        }
    } else {
        $details['error_summary'] = $reasonString;
    }
    return $details;
}

// --- Fetch ALL data for summaries (runs once) ---
$sql_all_for_summary = "SELECT reason, type, status FROM tdu_ai_error_log";
$result_all_for_summary = $conn->query($sql_all_for_summary);

$type_counts = [];
$status_counts = [];
$no_template_param_counts = [];
$other_reason_counts = [];
$grand_total_errors = 0;

if ($result_all_for_summary && $result_all_for_summary->num_rows > 0) {
    $grand_total_errors = $result_all_for_summary->num_rows;
    while($summary_row = $result_all_for_summary->fetch_assoc()) {
        $parsed_reason_details = parseReasonDetails($summary_row['reason']);
        
        $type_key = $summary_row['type'] ?? 'UNKNOWN_TYPE';
        $status_key = $summary_row['status'] ?? 'UNKNOWN_STATUS';

        $type_counts[$type_key] = ($type_counts[$type_key] ?? 0) + 1;
        $status_counts[$status_key] = ($status_counts[$status_key] ?? 0) + 1;

        if ($type_key === 'AUTO_NO_TEMPLATE' && !empty($parsed_reason_details['full_input_string'])) {
            $param_key_parts = [];
            if (!empty($parsed_reason_details['duration'])) $param_key_parts[] = "Dur=" . $parsed_reason_details['duration'];
            if (!empty($parsed_reason_details['city'])) $param_key_parts[] = "City=" . $parsed_reason_details['city'];
            if (!empty($parsed_reason_details['country'])) $param_key_parts[] = "Country=" . $parsed_reason_details['country'];
            if (!empty($parsed_reason_details['seats'])) $param_key_parts[] = "Seats=" . $parsed_reason_details['seats'];
            if (!empty($parsed_reason_details['cat_filter'])) $param_key_parts[] = "CatFilter=" . $parsed_reason_details['cat_filter'];
            $param_key = implode(", ", $param_key_parts);
            
            if (!empty($param_key)) {
                 $no_template_param_counts[$param_key] = ($no_template_param_counts[$param_key] ?? 0) + 1;
            } else {
                $fallback_reason = $summary_row['reason'] ?? 'REASON_NOT_PROVIDED';
                $other_reason_key = $type_key . '|' . $fallback_reason;
                $other_reason_counts[$other_reason_key] = ($other_reason_counts[$other_reason_key] ?? 0) + 1;
            }
        } elseif ($type_key !== 'AUTO_NO_TEMPLATE') {
            $reason_for_key = $summary_row['reason'] ?? 'REASON_NOT_PROVIDED';
            $reason_key = $type_key . '|' . $reason_for_key;
            $other_reason_counts[$reason_key] = ($other_reason_counts[$reason_key] ?? 0) + 1;
        }
    }
    arsort($no_template_param_counts);
    arsort($other_reason_counts);
}
$result_all_for_summary->free(); // Free result memory


// --- Pagination Logic for Detailed Log ---
$current_page = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
if ($current_page < 1) $current_page = 1;

// $grand_total_errors is already available from the summary query.
$total_pages = ceil($grand_total_errors / ITEMS_PER_PAGE);
if ($current_page > $total_pages && $total_pages > 0) $current_page = $total_pages; // Don't go past the last page

$offset = ($current_page - 1) * ITEMS_PER_PAGE;

// --- SQL Query to Fetch Paginated Data for Detailed Log ---
$sql_paginated_detail = "SELECT id, ai_ref, conversation_id, reason, type, status, created_at 
                         FROM tdu_ai_error_log 
                         ORDER BY created_at DESC, id DESC
                         LIMIT " . ITEMS_PER_PAGE . " OFFSET " . $offset;
$result_paginated_detail = $conn->query($sql_paginated_detail);

$detailed_errors_current_page = [];
if ($result_paginated_detail && $result_paginated_detail->num_rows > 0) {
    while($raw_row = $result_paginated_detail->fetch_assoc()) { 
        $current_error_data = $raw_row; 
        $parsed_reason_details = parseReasonDetails($current_error_data['reason']);
        $current_error_data['parsed_reason'] = $parsed_reason_details; 
        $detailed_errors_current_page[] = $current_error_data; 
    }
}

// --- Function to Generate Pagination Links ---
function generatePaginationLinks($current_page, $total_pages, $url_base = 'ai_error_report.php') {
    $links = '<div class="pagination">';
    $max_visible_pages = 7; // Number of page links to show (e.g., 1 ... 4 5 6 ... 10)

    if ($total_pages <= 1) return '';

    // Previous Link
    if ($current_page > 1) {
        $links .= '<a href="' . $url_base . '?page=' . ($current_page - 1) . '">« Previous</a>';
    } else {
        $links .= '<span class="disabled">« Previous</span>';
    }

    // Page Number Links
    if ($total_pages <= $max_visible_pages) {
        for ($i = 1; $i <= $total_pages; $i++) {
            if ($i == $current_page) {
                $links .= '<span class="current">' . $i . '</span>';
            } else {
                $links .= '<a href="' . $url_base . '?page=' . $i . '">' . $i . '</a>';
            }
        }
    } else {
        // Logic for more complex pagination (with ellipses)
        $start_page = max(1, $current_page - floor(($max_visible_pages - 3) / 2));
        $end_page = min($total_pages, $start_page + $max_visible_pages - 3);
        
        if ($start_page > 1) {
             $links .= '<a href="' . $url_base . '?page=1">1</a>';
            if ($start_page > 2) {
                $links .= '<span class="ellipsis">...</span>';
            }
        }

        for ($i = $start_page; $i <= $end_page; $i++) {
            if ($i == $current_page) {
                $links .= '<span class="current">' . $i . '</span>';
            } else {
                $links .= '<a href="' . $url_base . '?page=' . $i . '">' . $i . '</a>';
            }
        }
        
        if ($end_page < $total_pages) {
            if ($end_page < $total_pages -1) {
                 $links .= '<span class="ellipsis">...</span>';
            }
            $links .= '<a href="' . $url_base . '?page=' . $total_pages . '">' . $total_pages . '</a>';
        }
    }


    // Next Link
    if ($current_page < $total_pages) {
        $links .= '<a href="' . $url_base . '?page=' . ($current_page + 1) . '">Next »</a>';
    } else {
        $links .= '<span class="disabled">Next »</span>';
    }

    $links .= '</div>';
    return $links;
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>AI Error Log Analysis</title>
    <style>
        h1, h2, h3 { 
            color: #343a40; 
            border-bottom: 1px solid #007bff; 
            padding-bottom: 5px; 
            margin-top: 15px; 
            margin-bottom: 10px; 
        }
        h1 { text-align: center; font-size: 1.6em; }
        h2 { font-size: 1.3em; }
        h3 { font-size: 1.1em; border-bottom: none; margin-bottom: 5px;}

        table { 
            width: 100%; 
            border-collapse: collapse; 
            margin-top: 10px; 
            box-shadow: 0 1px 5px rgba(0,0,0,0.07); 
            background-color: #fff; 
        }
        th, td { 
            border: 1px solid #dee2e6; 
            padding: 5px 7px; 
            text-align: left; 
            vertical-align: top; 
            word-wrap: break-word; 
        }
        th { 
            background-color: #007bff; 
            color: white; 
            font-weight: bold; 
        }
        tr:nth-child(even) { background-color: #f2f2f2; }

        .summary-section { 
            margin-bottom: 20px; 
            padding: 10px 15px; 
            background-color: #fff; 
            box-shadow: 0 1px 5px rgba(0,0,0,0.07); 
            border-radius: 4px;
        }
        .grid-container { 
            display: grid; 
            grid-template-columns: repeat(auto-fit, minmax(280px, 1fr)); 
            gap: 10px; 
            margin-bottom: 10px; 
        }
        .grid-item { 
            padding: 10px; 
            background-color: #f8f9fa; 
            border: 1px solid #dee2e6; 
            border-radius: 3px;
        }
        .error-message { color: #D8000C; background-color: #FFD2D2; padding: 8px; border-radius: 3px; margin-top:15px; }
        .no-data { color: #6c757d; font-style: italic; padding: 5px 0; }
        
        .reason-params { 
            font-size: 0.9em; 
            color: #495057; 
            line-height: 1.3; 
        }
        .reason-params span { 
            display: inline-block;
            margin-right: 8px;
            padding: 1px 3px;
            margin-bottom: 2px;
        }
        .reason-params strong { color: #343a40; }

        .full-reason-cell { 
            max-height: 100px; 
            overflow-y: auto;
            font-size: 0.85em;
        }
        .small-text { font-size: 0.9em; }

        /* Pagination Styles */
        .pagination {
            margin: 20px 0;
            text-align: center;
        }
        .pagination a, .pagination span {
            display: inline-block;
            padding: 6px 12px;
            margin: 0 2px;
            border: 1px solid #dee2e6;
            text-decoration: none;
            color: #007bff;
            border-radius: 3px;
            font-size: 0.9em;
        }
        .pagination a:hover {
            background-color: #e9ecef;
        }
        .pagination .current {
            background-color: #007bff;
            color: white;
            border-color: #007bff;
        }
        .pagination .disabled {
            color: #6c757d;
            pointer-events: none;
            border-color: #dee2e6;
        }
        .pagination .ellipsis {
            padding: 6px 0px; /* Adjust padding for ellipsis to align better */
            border: none;
        }
    </style>
</head>
<body>

<h1>AI Error Log Analysis</h1>

<div class="summary-section">
    <h2>Overall Summary</h2>
    <div class="grid-container">
        <div class="grid-item"><h3>Total Errors Logged</h3><p style="font-size: 1.2em; font-weight: bold;"><?php echo $grand_total_errors; ?></p></div>
    </div>

    <div class="grid-container">
        <div class="grid-item">
            <h3>Errors by Type</h3>
            <?php if (!empty($type_counts)): ?>
            <table><thead><tr><th>Type</th><th>Count</th><th>%</th></tr></thead><tbody>
            <?php foreach ($type_counts as $type => $count): ?>
                <tr>
                    <td><?php echo htmlspecialchars(make_string_for_html($type), ENT_QUOTES, 'UTF-8'); ?></td>
                    <td><?php echo $count; ?></td>
                    <td class="small-text"><?php echo $grand_total_errors ? round(($count / $grand_total_errors) * 100, 1) : 0; ?>%</td>
                </tr>
            <?php endforeach; ?>
            </tbody></table>
            <?php else: echo "<p class='no-data'>No type data available.</p>"; endif; ?>
        </div>
        <div class="grid-item">
            <h3>Errors by Status</h3>
            <?php if (!empty($status_counts)): ?>
            <table><thead><tr><th>Status</th><th>Count</th><th>%</th></tr></thead><tbody>
            <?php foreach ($status_counts as $status => $count): ?>
                <tr>
                    <td><?php echo htmlspecialchars(make_string_for_html($status), ENT_QUOTES, 'UTF-8'); ?></td>
                    <td><?php echo $count; ?></td>
                    <td class="small-text"><?php echo $grand_total_errors ? round(($count / $grand_total_errors) * 100, 1) : 0; ?>%</td>
                </tr>
            <?php endforeach; ?>
            </tbody></table>
            <?php else: echo "<p class='no-data'>No status data available.</p>"; endif; ?>
        </div>
    </div>
</div>

<div class="summary-section">
    <h2>Most Common "AUTO_NO_TEMPLATE" Issues (Top 20)</h2>
    <p class="small-text">Parameter combinations frequently missing templates (based on all data).</p>
    <?php if (!empty($no_template_param_counts)): ?>
    <table>
        <thead><tr><th>Missing Template Parameters</th><th>Freq.</th></tr></thead>
        <tbody>
        <?php $i = 0; foreach ($no_template_param_counts as $params => $count): if ($i++ >= 20) break; ?>
            <tr>
                <td class="small-text"><?php echo htmlspecialchars(make_string_for_html($params), ENT_QUOTES, 'UTF-8'); ?></td>
                <td><?php echo $count; ?></td>
            </tr>
        <?php endforeach; ?>
        </tbody>
    </table>
    <?php else: echo "<p class='no-data'>No 'AUTO_NO_TEMPLATE' errors with parsed parameters found, or no such errors logged.</p>"; endif; ?>
</div>

<div class="summary-section">
    <h2>Most Common Other Reasons (Top 20)</h2>
    <p class="small-text">Frequent reasons for types like AUTO_MISMATCH, MANUAL, etc. (based on all data).</p>
    <?php if (!empty($other_reason_counts)): ?>
    <table>
        <thead><tr><th>Type</th><th>Reason</th><th>Freq.</th></tr></thead>
        <tbody>
        <?php $i = 0; foreach ($other_reason_counts as $key => $count): if ($i++ >= 20) break; 
            list($type, $reason_text) = explode('|', $key, 2);
        ?>
            <tr>
                <td><?php echo htmlspecialchars(make_string_for_html($type), ENT_QUOTES, 'UTF-8'); ?></td>
                <td class="small-text full-reason-cell"><?php echo htmlspecialchars(make_string_for_html($reason_text), ENT_QUOTES, 'UTF-8'); ?></td>
                <td><?php echo $count; ?></td>
            </tr>
        <?php endforeach; ?>
        </tbody>
    </table>
    <?php else: echo "<p class='no-data'>No other error types/reasons found or aggregated.</p>"; endif; ?>
</div>

<h2>Detailed Error Log (Page <?php echo $current_page . ' of ' . $total_pages; ?> - <?php echo $grand_total_errors; ?> total entries)</h2>

<?php echo generatePaginationLinks($current_page, $total_pages); ?>

<?php
if (!empty($detailed_errors_current_page)) {
    echo "<table>";
    echo "<thead><tr>";
    echo "<th>ID</th>";
    echo "<th>AI Ref</th>";
    echo "<th>Conv. ID</th>";
    echo "<th>Type</th>";
    echo "<th>Status</th>";
    echo "<th>Parsed Details</th>";
    echo "<th>Full Reason</th>";
    echo "<th>Created At</th>";
    echo "</tr></thead>";
    echo "<tbody>";

    foreach($detailed_errors_current_page as $error_row) { // Iterate over the paginated subset
        echo "<tr>";
        echo "<td>" . htmlspecialchars(make_string_for_html($error_row["id"]), ENT_QUOTES, 'UTF-8') . "</td>";
        echo "<td>" . htmlspecialchars(make_string_for_html($error_row["ai_ref"]), ENT_QUOTES, 'UTF-8') . "</td>";
        echo "<td>" . htmlspecialchars(make_string_for_html($error_row["conversation_id"]), ENT_QUOTES, 'UTF-8') . "</td>";
        echo "<td>" . htmlspecialchars(make_string_for_html($error_row["type"]), ENT_QUOTES, 'UTF-8') . "</td>";
        echo "<td>" . htmlspecialchars(make_string_for_html($error_row["status"]), ENT_QUOTES, 'UTF-8') . "</td>";
        
        echo "<td><div class='reason-params'>";
        $parsed_any = false;
        if(!empty($error_row['parsed_reason']['segment'])) { echo "<span><strong>Seg:</strong> " . htmlspecialchars(make_string_for_html($error_row['parsed_reason']['segment']), ENT_QUOTES, 'UTF-8') . "</span>"; $parsed_any = true; }
        if(!empty($error_row['parsed_reason']['duration'])) { echo "<span><strong>Dur:</strong> " . htmlspecialchars(make_string_for_html($error_row['parsed_reason']['duration']), ENT_QUOTES, 'UTF-8') . "</span>"; $parsed_any = true; }
        if(!empty($error_row['parsed_reason']['city'])) { echo "<span><strong>City:</strong> " . htmlspecialchars(make_string_for_html($error_row['parsed_reason']['city']), ENT_QUOTES, 'UTF-8') . "</span>"; $parsed_any = true; }
        if(!empty($error_row['parsed_reason']['country'])) { echo "<span><strong>Country:</strong> " . htmlspecialchars(make_string_for_html($error_row['parsed_reason']['country']), ENT_QUOTES, 'UTF-8') . "</span>"; $parsed_any = true; }
        if(!empty($error_row['parsed_reason']['seats'])) { echo "<span><strong>Seats:</strong> " . htmlspecialchars(make_string_for_html($error_row['parsed_reason']['seats']), ENT_QUOTES, 'UTF-8') . "</span>"; $parsed_any = true; }
        if(!empty($error_row['parsed_reason']['cat_filter'])) { echo "<span><strong>Cat:</strong> " . htmlspecialchars(make_string_for_html($error_row['parsed_reason']['cat_filter']), ENT_QUOTES, 'UTF-8') . "</span>"; $parsed_any = true; }
        
        if (!$parsed_any && make_string_for_html($error_row['type']) === 'AUTO_NO_TEMPLATE' && !empty($error_row['parsed_reason']['full_input_string'])) {
             echo "<span class='no-data'>Params not individually parsed.</span>";
        } elseif (!$parsed_any && empty($error_row['parsed_reason']['full_input_string'])) { // if no input string at all
            echo "<span class='no-data'>-</span>";
        }

        echo "</div></td>";

        echo "<td class='full-reason-cell'>" . htmlspecialchars(make_string_for_html($error_row["reason"]), ENT_QUOTES, 'UTF-8') . "</td>";
        echo "<td class='small-text'>" . htmlspecialchars(make_string_for_html($error_row["created_at"]), ENT_QUOTES, 'UTF-8') . "</td>";
        echo "</tr>";
    }
    echo "</tbody>";
    echo "</table>";
} elseif ($grand_total_errors > 0 && empty($detailed_errors_current_page)) { // Should not happen if page logic is correct
    echo "<p class='no-data'>No errors found for this page, but errors exist in the log. Check page number.</p>";
} elseif ($grand_total_errors == 0) {
    echo "<p class='no-data'>No records found in the 'tdu_ai_error_log' table.</p>";
} else { 
    echo "<p class='error-message'>Error executing query for detailed log: " . htmlspecialchars(make_string_for_html($conn->error), ENT_QUOTES, 'UTF-8') . "</p>";
}

echo generatePaginationLinks($current_page, $total_pages); 
?>

<?php
// --- Close Connection ---
if ($conn) {
    $conn->close();
}
?>
</body>
</html>