<?php

require "ai_keys.php"; // API keys & URLs
include "dbconn.php"; // DB connection ($conn)

$descSuccess = databaseDescriptionUpdate($conn);

/**
 * Fetches active products from DB based on specified criteria.
 */
function getProducts($conn, $productid = null, $criteria = null): array
{
    $products = [];
    $limit = " LIMIT 1000"; // Process batch size
    $sql = "
        SELECT p.productid, p.productName, p.country, p.city, v.address, p.description, p.productImage, v.vendorName, v.website
        FROM tdu_products p
        LEFT JOIN tdu_vendors v ON p.vendorid = v.vendorid
        WHERE p.productActive='Yes' AND p.category != 'Transfers'
    ";

    // Apply specific filters
    if ($productid !== null) {
        $sql .= " AND v.vendorName NOT LIKE '%Turtle Down Under%' AND p.productid = '" . mysqli_real_escape_string($conn, $productid) . "'";
    } elseif ($criteria == "description") {
        $sql .= " AND (p.description IS NULL OR p.description = '') AND v.vendorName NOT LIKE '%Turtle Down Under%'"; // Description filters
    } elseif ($criteria == "productImage") {
        $sql .= " AND (p.productImage IS NULL OR p.productImage = '')"; // Image filter
    }
    $sql .= $limit; // Add limit

    $result = mysqli_query($conn, $sql);
    if (!$result) {
        error_log("DB Error [getProducts]: " . mysqli_error($conn));
        return [];
    }
    while ($row = mysqli_fetch_assoc($result)) {
        $products[] = $row;
    }
    return $products;
}

/**
 * Generates AI description (Optional Function, uses <br>).
 */
function ai_generateDescriptionForProduct($conn, array $product): ?string
{
    $finalResult = null;
    // --- AI Prompt construction ---
    $rules = "
        IMPORTANT RULES FOR OUTPUT FORMATTING:
        1. The output MUST NOT contain the '*' character anywhere. (HIGHEST PRIORITY).
        2. The output MUST be a single block of descriptive text (prose). No headings, lists.
        3. Structure the output as 1 to 3 paragraphs.
        4. DO NOT BE CONVERSATIONAL. Output only the description directly. ABSOLUTELY VITAL.
    ";
    $preprompt = $rules . " give me as detailed of a description of the product as possible based on the information given and use google search to find the latest information: 
                            Instruction:
                            DO NOT including any product pricing in description
                            ";

    // Append product details for AI context
    $preprompt .= "Vendor Name: " . ($product['vendorName'] ?? 'N/A') . ", ";
    $preprompt .= "Product Name: " . ($product['productName'] ?? 'N/A') . ", ";
    $preprompt .= "Country: " . ($product['country'] ?? 'N/A') . ", ";
    $preprompt .= "City: " . ($product['city'] ?? 'N/A') . ", ";
    $preprompt .= "Address: " . ($product['address'] ?? 'N/A') . ", ";
    $preprompt .= "Website: " . ($product['website'] ?? 'N/A') . ", ";
    $prompt = trim($preprompt);

    $apiCallTask = function () use ($prompt) { return promptGeminiAI($prompt); };

    echo "Attempting AI description for Product ID: " . ($product['productid'] ?? 'N/A') . "...<br>"; // HTML output
    $result = callApiWithBackoff($apiCallTask, 5, 1.5, 60.0, true); // Call API with retry

    // Process result
    if ($result['status'] === 'SUCCESS' && isset($result['data'])) {
        echo "<br>--- Generated Description ---<br>"; // HTML output
        $cleanedText = str_replace('*', '', $result['data']);
        $trimmedText = trim($cleanedText);
        echo nl2br(htmlspecialchars($trimmedText)) . "<br>"; // HTML output (convert internal newlines)
        $finalResult = $trimmedText;
    } else {
        echo "<br>--- Failed to generate description ---<br>"; // HTML output
        echo "Status: " . ($result['status'] ?? 'UNKNOWN') . ", Msg: " . ($result['message'] ?? 'N/A') . "<br>"; // HTML output
        if (isset($result['details'])) { error_log("Detailed Error [ai_generateDescriptionForProduct]... " . print_r($result['details'], true)); }
    }
    echo "------------------------------------<br><br>"; // HTML break
    return $finalResult;
}

/**
 * Calls Gemini API with Google Search tool (e.g., for descriptions).
 */
function promptGeminiAI(string $prompt): array
{
    // Config checks
    if (!function_exists('apiURL_Flash')) return ['status' => 'CONFIG_ERROR', 'message' => 'apiURL_Flash missing.'];
    $apiUrl = apiURL_Flash();
    if (empty($apiUrl)) return ['status' => 'CONFIG_ERROR', 'message' => 'API URL empty.'];

    // API Payload
    $payload = [
        'contents' => [['parts' => [['text' => $prompt]]]],
        'generationConfig' => [ 'temperature' => 0.2, 'maxOutputTokens' => 2048, 'topP' => 0.95, 'topK' => 40 ],
        'tools' => [ ['google_search' => new stdClass()] ] // Enable Google Search
    ];
    $headers = ['Content-Type: application/json'];
    $ch = curl_init();
    // cURL options
    curl_setopt($ch, CURLOPT_URL, $apiUrl); curl_setopt($ch, CURLOPT_RETURNTRANSFER, true); curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload)); curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 30); curl_setopt($ch, CURLOPT_TIMEOUT, 150);
    // Execute & get info
    $response_body = curl_exec($ch); $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE); $curlError = curl_error($ch); curl_close($ch);
    // Error handling
    if ($curlError) return ['status' => 'CURL_ERROR', 'message' => $curlError];
    if ($httpCode == 429) return ['status' => 'RATE_LIMIT_ERROR', 'message' => 'HTTP 429'];
    if ($httpCode >= 500) return ['status' => 'SERVER_ERROR', 'message' => "HTTP $httpCode", 'details' => $response_body];
    if ($httpCode >= 400) return ['status' => 'CLIENT_ERROR', 'message' => "HTTP $httpCode", 'details' => $response_body];
    // Process success
    if ($httpCode >= 200) {
        $responseData = json_decode($response_body, true);
        if (json_last_error() !== JSON_ERROR_NONE) return ['status' => 'JSON_DECODE_ERROR', 'message' => json_last_error_msg(), 'details' => $response_body];
        // Extract text safely
        if (isset($responseData['candidates'][0]['content']['parts'][0]['text'])) {
            $finishReason = $responseData['candidates'][0]['finishReason'] ?? 'UNKNOWN';
            if ($finishReason === 'STOP' || $finishReason === 'MAX_TOKENS') { return ['status' => 'SUCCESS', 'data' => $responseData['candidates'][0]['content']['parts'][0]['text']]; }
            else { return ['status' => 'API_FILTERED', 'message' => "Reason: $finishReason", 'details' => $responseData]; }
        } elseif (isset($responseData['candidates'][0]['finishReason'])) {
            $reason = $responseData['candidates'][0]['finishReason'];
            return ['status' => 'API_FILTERED', 'message' => "Reason: $reason", 'details' => $responseData];
        } else { return ['status' => 'UNEXPECTED_RESPONSE', 'message' => 'No text part.', 'details' => $responseData]; }
    }
    return ['status' => 'UNEXPECTED_HTTP_CODE', 'message' => "Code: $httpCode", 'details' => $response_body];
}

/**
 * Generates a concise image search query using Gemini.
 * CORRECTED to dynamically build prompt with product details. Uses <br>.
 */
function ai_generateOptimalImageSearchQuery(array $product): ?string
{
    $finalQuery = null;

    // Dynamically construct prompt using HEREDOC
    $productName = $product['productName'] ?? 'N/A';
    $city = $product['city'] ?? 'N/A';
    $country = $product['country'] ?? 'N/A';
    $vendorName = $product['vendorName'] ?? 'N/A';

    $prompt = <<<PROMPT
TASK: Generate the best possible concise Google Image search query (5-10 words)
      to find a visually representative photo for the described product.

INPUT PRODUCT DETAILS:
- Product Name: $productName
- City: $city
- Country: $country
- Vendor: $vendorName

INSTRUCTIONS:
1. Focus ONLY on the core visual subject/activity, vendor and the primary location/landmark.
2. Create a query likely to yield a high-quality photograph of the experience.
3. IGNORE logistical details (duration, price, capacity like '1pax', specific times, complex codes).
4. IGNORE generic words unless essential (e.g., 'tour' might be useful, 'from'/'at' are not).
5. Keep it short and effective.
6. Output ONLY the generated search query string. No explanations, no conversational text. ABSOLUTELY VITAL.
7. The vendor is important because it helps narrow down the images for more relevancy

EXAMPLE:
If input details lead to a Sydney Helicopter tour by 'Sydney Heli Tours',
a good output would be 'Sydney Harbour helicopter tour Sydney Heli Tours' or 'Helicopter over Sydney Opera House Sydney Heli Tours'.

YOUR GENERATED QUERY:
PROMPT;
    $prompt = trim($prompt);

    echo "Attempting AI query generation for Product ID: " . ($product['productid'] ?? 'N/A') . "...<br>"; // HTML output
    $apiCallTask = function () use ($prompt) { // Capture dynamic prompt
         $payload = [
            'contents' => [['parts' => [['text' => $prompt]]]],
            'generationConfig' => [ 'temperature' => 0.3, 'maxOutputTokens' => 60, 'topP' => 0.95, 'topK' => 40 ]
        ];
         return promptGeminiAI_NoTools($payload); // Use NoTools version
    };
    $result = callApiWithBackoff($apiCallTask, 3, 1.5, 30.0, true); // Retry logic

    // Process result, clean, and validate
    if ($result['status'] === 'SUCCESS' && isset($result['data'])) {
        $cleanedQuery = trim(str_replace(['*', '"', "'", 'QUERY:', 'Generated Query:'], '', $result['data']));
        $cleanedQuery = trim(preg_replace('/\s+/', ' ', $cleanedQuery));
        if (!empty($cleanedQuery) && strlen($cleanedQuery) > 3 && strlen($cleanedQuery) < 150) {
            $finalQuery = $cleanedQuery;
            echo "AI Generated Image Search Query: '{$finalQuery}'<br>"; // HTML output
        } else {
            error_log("AI generated invalid query ('{$cleanedQuery}') for Product ID " . ($product['productid'] ?? 'N/A') . ". Rejected.");
            $finalQuery = null;
        }
    } else {
        echo "-- Failed AI query gen. Status: ".($result['status']??'UNKNOWN').", Msg: ".($result['message']??'N/A')."<br>"; // HTML output
    }
    return $finalQuery;
}


/**
 * Calls Gemini API - Does NOT include 'tools' parameter (for query generation).
 */
function promptGeminiAI_NoTools(array $payload): array
{
    // Config checks
    if (!function_exists('apiURL_Flash')) return ['status' => 'CONFIG_ERROR', 'message' => 'apiURL_Flash missing.'];
    $apiUrl = apiURL_Flash();
    if (empty($apiUrl)) return ['status' => 'CONFIG_ERROR', 'message' => 'API URL empty.'];

    $headers = ['Content-Type: application/json'];
    $ch = curl_init();
    // cURL options
    curl_setopt($ch, CURLOPT_URL, $apiUrl); curl_setopt($ch, CURLOPT_RETURNTRANSFER, true); curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload)); curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 30); curl_setopt($ch, CURLOPT_TIMEOUT, 90);
    // Execute & get info
    $response_body = curl_exec($ch); $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE); $curlError = curl_error($ch); curl_close($ch);
    // Error handling
    if ($curlError) return ['status' => 'CURL_ERROR', 'message' => $curlError];
    if ($httpCode == 429) return ['status' => 'RATE_LIMIT_ERROR', 'message' => 'HTTP 429'];
    if ($httpCode >= 500) return ['status' => 'SERVER_ERROR', 'message' => "HTTP $httpCode", 'details' => $response_body];
    if ($httpCode >= 400) return ['status' => 'CLIENT_ERROR', 'message' => "HTTP $httpCode", 'details' => $response_body];
    // Process success
    if ($httpCode >= 200) {
        $responseData = json_decode($response_body, true);
        if (json_last_error() !== JSON_ERROR_NONE) return ['status' => 'JSON_DECODE_ERROR', 'message' => json_last_error_msg(), 'details' => $response_body];
        if (isset($responseData['candidates'][0]['content']['parts'][0]['text'])) {
            $finishReason = $responseData['candidates'][0]['finishReason'] ?? 'UNKNOWN';
            if ($finishReason === 'STOP' || $finishReason === 'MAX_TOKENS') { return ['status' => 'SUCCESS', 'data' => $responseData['candidates'][0]['content']['parts'][0]['text']]; }
            else { return ['status' => 'API_FILTERED', 'message' => "Reason: $finishReason", 'details' => $responseData]; }
        } elseif (isset($responseData['candidates'][0]['finishReason'])) {
            $reason = $responseData['candidates'][0]['finishReason'];
            return ['status' => 'API_FILTERED', 'message' => "Reason: $reason", 'details' => $responseData];
        } else { return ['status' => 'UNEXPECTED_RESPONSE', 'message' => 'No text part.', 'details' => $responseData]; }
    }
    return ['status' => 'UNEXPECTED_HTTP_CODE', 'message' => "Code: $httpCode", 'details' => $response_body];
}

/**
 * Wraps a callable API function with exponential backoff retry logic. Uses <br>.
 */
function callApiWithBackoff( callable $apiCallClosure, int $maxRetries = 5, float $initialDelaySeconds = 1.5, float $maxDelaySeconds = 60.0, bool $retryOnServerErrors = true ): array
{
    $attempt = 0; $delay = $initialDelaySeconds; $lastResult = [];
    while ($attempt <= $maxRetries) {
        echo "API Call Attempt #" . ($attempt + 1) . "...<br>"; // HTML output
        $lastResult = $apiCallClosure(); $status = $lastResult['status'] ?? 'UNKNOWN_ERROR';
        $isRetryable = ($status === 'RATE_LIMIT_ERROR') || ($retryOnServerErrors && $status === 'SERVER_ERROR');
        if (!$isRetryable) { if ($status !== 'SUCCESS') error_log("Non-retryable error ($status) on attempt " . ($attempt + 1)); break; }
        $attempt++; if ($attempt > $maxRetries) { error_log("Max retries ($maxRetries) reached. Last status: $status"); break; }
        $waitTime = min($delay, $maxDelaySeconds); $jitterMs = mt_rand(0, 1000); $actualWaitSeconds = $waitTime + ($jitterMs / 1000.0);
        error_log("Retryable error ($status) on attempt #$attempt. Waiting " . round($actualWaitSeconds, 2) . "s...");
        usleep(intval($actualWaitSeconds * 1000000)); // Pause
        $delay *= 2;
    }
    return $lastResult;
}

/**
 * Updates a product's image filename(s) in the DB. Uses <br>.
 */
function updateProductImage($conn, int $productid, string $newImageFileString): bool
{
    $sql = "UPDATE tdu_products SET productImage = ? WHERE productid = ?";
    $stmt = mysqli_prepare($conn, $sql);
    if (!$stmt) { error_log("DB Prepare Error [updateProductImage] ID $productid: " . mysqli_error($conn)); return false; }
    mysqli_stmt_bind_param($stmt, "si", $newImageFileString, $productid); // Bind comma-separated string
    $success = mysqli_stmt_execute($stmt);
    if (!$success) { error_log("DB Execute Error [updateProductImage] ID $productid: " . mysqli_stmt_error($stmt)); }
    else { echo "DB: ".(mysqli_stmt_affected_rows($stmt) > 0 ? "Successfully updated" : "Update executed (0 rows affected for") ." productImage ID: $productid<br>"; } // HTML output
    mysqli_stmt_close($stmt);
    return $success;
}

/**
 * Updates a product's description in the DB. (Optional Function, uses <br>).
 */
function updateProductDescription($conn, int $productid, string $newDescription): bool
{
    $sql = "UPDATE tdu_products SET description = ?, ai_description = 1 WHERE productid = ?";
    $stmt = mysqli_prepare($conn, $sql);
    if (!$stmt) { error_log("DB Prepare Error [updateProductDescription] ID $productid: " . mysqli_error($conn)); return false; }
    mysqli_stmt_bind_param($stmt, "si", $newDescription, $productid);
    $success = mysqli_stmt_execute($stmt);
    if (!$success) { error_log("DB Execute Error [updateProductDescription] ID $productid: " . mysqli_stmt_error($stmt)); }
    else { echo "DB: ".(mysqli_stmt_affected_rows($stmt) > 0 ? "Successfully updated" : "Update executed (0 rows affected for") ." description ID: $productid<br>"; } // HTML output
    mysqli_stmt_close($stmt);
    return $success;
}

/**
 * Main process for updating product descriptions via AI. (Optional Function, uses <br>).
 */
function databaseDescriptionUpdate($conn, $productid = null): bool
{
    $anyFailures = false;
    if (!isset($conn) || !($conn instanceof mysqli) || $conn->connect_error) { die("DB connection failed."); }
    $products = getProducts($conn, $productid, "description");
    if (empty($products)) { echo "No products need description updates.<br>"; if ($conn) mysqli_close($conn); return true; }

    echo "Fetched " . count($products) . " for descriptions...<br>"; // HTML output
    foreach ($products as $product) {
        echo "<br><br>"; // Keep HTML break
        $newDescription = ai_generateDescriptionForProduct($conn, $product); // Calls func using <br>
        if ($newDescription !== null && $newDescription !== '') {
            echo "Updating DB Description ID: " . $product['productid'] . "<br>"; // HTML output
            $updateSuccess = updateProductDescription($conn, $product['productid'], $newDescription);
            if ($updateSuccess === false) $anyFailures = true;
        } else {
            echo "Skipping DB update Description ID: " . $product['productid'] . " (AI failed).<br>"; // HTML output
            $anyFailures = true;
        }
        echo "<br><br>"; // Keep HTML break
    }
    echo "Finished description batch.<br>"; // HTML output
    if ($conn) mysqli_close($conn);
    return !$anyFailures;
}

/**
 * Searches Google Images via Custom Search API. Uses <br>.
 */
function getImage(string $search, int $numResults = 3): ?array
{
    $apiKey = apikey_Unrestrained(); $searchEngineId = searchEngineID();
    // Config / Input validation
    if (empty(trim($search))) { error_log("Error [getImage]: Query empty."); return null; }
    if (empty($apiKey) || $apiKey === 'YOUR_GOOGLE_API_KEY' || empty($searchEngineId) || $searchEngineId === 'YOUR_SEARCH_ENGINE_ID') { error_log("Error [getImage]: API/CX Key not configured."); return null; }

    // Build request URL
    $endpoint = 'https://www.googleapis.com/customsearch/v1';
    $queryParams = [ 'key' => $apiKey, 'cx' => $searchEngineId, 'q' => $search, 'searchType' => 'image', 'num' => max(1, min(10, $numResults)), 'safe' => 'active' ];
    $url = $endpoint . '?' . http_build_query($queryParams);

    // Execute cURL
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url); curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10); curl_setopt($ch, CURLOPT_TIMEOUT, 20);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true); curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
    $response_body = curl_exec($ch); $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE); $curlError = curl_error($ch); curl_close($ch);

    // Handle errors
    if ($curlError) { error_log("cURL Error [getImage] '{$search}': " . $curlError); return null; }
    if ($httpCode !== 200) { error_log("HTTP Error {$httpCode} [getImage] '{$search}'. Resp: " . $response_body); return null; }

    // Decode and extract URLs
    $responseData = json_decode($response_body);
    if (json_last_error() !== JSON_ERROR_NONE) { error_log("JSON Error [getImage] '{$search}': " . json_last_error_msg()); return null; }
    $imageUrls = [];
    if (isset($responseData->items) && is_array($responseData->items) && !empty($responseData->items)) {
        foreach ($responseData->items as $item) { if (isset($item->link) && is_string($item->link) && !empty($item->link)) $imageUrls[] = $item->link; }
    }

    // Return results
    if (empty($imageUrls)) { error_log("No valid image URLs found [getImage] '{$search}'."); return null; }
    else { echo count($imageUrls) . " image URL(s) found for '{$search}'.<br>"; return $imageUrls; } // HTML output
}

/**
 * Downloads an image, saves it locally with a clean name based on URL path.
 * Returns the clean relative path on success, null on failure. Uses <br>.
 */
function downloadImage(string $imageUrl): ?string
{
    $targetDir = __DIR__ . '/product_image/';
    $originalImageUrl = trim($imageUrl);
    // URL Validation
    if (empty($originalImageUrl) || !filter_var($originalImageUrl, FILTER_VALIDATE_URL)) { error_log("Error [downloadImage]: Invalid URL: " . $originalImageUrl); return null; }

    // Directory Checks
    if (!is_dir($targetDir)) { if (!mkdir($targetDir, 0755, true)) { error_log("Error [downloadImage]: Cannot create dir: " . $targetDir); return null; } echo "Created dir: " . $targetDir . "<br>"; } // HTML output
    if (!is_writable($targetDir)) { error_log("Error [downloadImage]: Dir not writable: " . $targetDir); return null; }

    // Generate clean filename from URL path
    $urlParts = parse_url($originalImageUrl); $pathComponent = $urlParts['path'] ?? '/unknown.jpg';
    $urlPathInfo = pathinfo($pathComponent);
    $originalFilename = $urlPathInfo['filename'] ?? 'image'; $extension = $urlPathInfo['extension'] ?? 'jpg';
    $safeOriginalFilename = preg_replace('/[^a-zA-Z0-9_.-]/', '_', $originalFilename);
    $uniqueFilename = uniqid('img_', true) . '_' . $safeOriginalFilename . '.' . $extension; // Unique name
    $targetFilePath = $targetDir . $uniqueFilename; // Full save path

    // Download using ORIGINAL URL
    echo "Attempting download: {$originalImageUrl}<br>"; // HTML output
    $ch = curl_init();
    // cURL options
    curl_setopt($ch, CURLOPT_URL, $originalImageUrl); curl_setopt($ch, CURLOPT_RETURNTRANSFER, true); curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 15); curl_setopt($ch, CURLOPT_TIMEOUT, 45);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true); curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
    // Execute
    $imageData = curl_exec($ch); $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE); $curlError = curl_error($ch); curl_close($ch);

    // Handle download errors
    if ($curlError) { error_log("cURL Error DL '{$originalImageUrl}': " . $curlError); if (strpos($curlError, 'SSL_ERROR_SYSCALL') !== false) error_log("--> SSL_ERROR_SYSCALL"); return null; }
    if ($httpCode !== 200) { error_log("HTTP Error {$httpCode} DL '{$originalImageUrl}'"); return null; }
    if (empty($imageData)) { error_log("Empty data DL '{$originalImageUrl}'"); return null; }

    // Save file
    echo "Attempting save: {$targetFilePath}<br>"; // HTML output
    $bytesWritten = file_put_contents($targetFilePath, $imageData);
    if ($bytesWritten === false) { error_log("Failed write: " . $targetFilePath); return null; }

    // Success
    $cleanRelativePath = "product_image/" . $uniqueFilename; // Path for DB
    echo "Success save ({$bytesWritten}b): " . $cleanRelativePath . "<br>"; // HTML output
    return $cleanRelativePath;
}

/**
 * Main process: AI query -> Fallback -> Search -> Download Multiple -> Update DB Comma-Separated.
 * Uses <br> for HTML output.
 */
function databaseProductImageUpdate($conn, $productid = null): bool
{
    $anyFailures = false;
    $maxImagesToSave = 3; // Config: How many images to try saving per product
    $numResultsToRequest = 10; // Config: How many URLs to get from Google

    if (!isset($conn) || !($conn instanceof mysqli) || $conn->connect_error) { die("DB connection failed."); }

    $products = getProducts($conn, $productid, "productImage"); // Get products
    if (empty($products)) { echo "No products need image updates.<br>"; if ($conn) mysqli_close($conn); return true; }

    echo "Fetched " . count($products) . " products for images...<br>"; // HTML output

    foreach ($products as $product) {
        echo "<br><br>Processing Product ID: " . $product['productid'] . "<br>"; // HTML output

        // Generate/Fallback Search Query (Calls function using <br>)
        $imageSearch = ai_generateOptimalImageSearchQuery($product);
        if (empty($imageSearch)) {
            echo "AI query failed. Using fallback...<br>"; // HTML output
            $queryParts = []; // Build fallback query
            if (!empty(trim($product['productName'] ?? ''))) $queryParts[] = trim($product['productName']);
            if (!empty(trim($product['city'] ?? ''))) $queryParts[] = trim($product['city']);
            if (!empty(trim($product['country'] ?? ''))) $queryParts[] = trim($product['country']);
            $imageSearch = implode(' ', $queryParts);
            $imageSearch = trim(preg_replace('/\s+/', ' ', $imageSearch));
            $maxLength = 150; if (strlen($imageSearch) > $maxLength) { $lastSpace = strrpos(substr($imageSearch, 0, $maxLength), ' '); $imageSearch = substr($imageSearch, 0, ($lastSpace !== false ? $lastSpace : $maxLength)); }
            echo "Fallback Query: '{$imageSearch}'<br>"; // HTML output
        }
        if (empty($imageSearch)) { echo "Cannot build query. Skipping.<br>"; $anyFailures = true; continue; } // HTML output

        // Search Images (Calls function using <br>)
        echo "Searching images: '{$imageSearch}'<br>"; // HTML output
        $imageUrlArray = getImage($imageSearch, $numResultsToRequest); // Get URL list

        $downloadedImagePaths = []; // Store paths of successful downloads
        $successfulDownloadsCount = 0;
        $firstSuccessfulImageUrl = null; // For display

        // Attempt Downloads
        if ($imageUrlArray !== null && !empty($imageUrlArray)) {
            echo "Found " . count($imageUrlArray) . " URLs. Trying up to {$maxImagesToSave} downloads...<br>"; // HTML output
            foreach ($imageUrlArray as $index => $potentialImageUrl) {
                echo "Attempt #" . ($index + 1) . ": URL: {$potentialImageUrl}<br>"; // HTML output
                $downloadedFilename = downloadImage($potentialImageUrl); // Try download (uses <br>)

                if ($downloadedFilename !== null) { // Success?
                    $downloadedImagePaths[] = $downloadedFilename; // Add path to list
                    $successfulDownloadsCount++;
                    if ($firstSuccessfulImageUrl === null) $firstSuccessfulImageUrl = $potentialImageUrl;

                    echo "Success download #{$successfulDownloadsCount} ({$downloadedFilename}).<br>"; // HTML output

                    if ($successfulDownloadsCount >= $maxImagesToSave) { // Reached target?
                        echo "Reached max images ({$maxImagesToSave}).<br>"; // HTML output
                        break; // Stop downloading for this product
                    }
                } else {
                    echo "Failed download #" . ($index + 1) . ". Trying next...<br>"; // HTML output
                }
            } // End URL loop

            // Display first downloaded image (optional) - HTML Tag
            if ($firstSuccessfulImageUrl !== null) {
                 echo '<img src="' . htmlspecialchars($firstSuccessfulImageUrl) . '" alt="' . htmlspecialchars($imageSearch). '" style="width: 250px; height: auto; display: block; margin-top: 10px;">';
            }
             if (empty($downloadedImagePaths)) { echo "Could not download any usable image.<br>"; } // HTML output

        } else {
             error_log("No URLs from Google Search for '{$imageSearch}'.");
             echo "No URLs from Google Search for '{$imageSearch}'.<br>"; // HTML output
        }

        // Update Database
        if (!empty($downloadedImagePaths)) { // If we got at least one image
            $dbImageString = implode(',', $downloadedImagePaths); // Create comma-separated string
            echo "Updating DB ID: " . $product['productid'] . " with: " . $dbImageString . "<br>"; // HTML output
            $updateSuccess = updateProductImage($conn, $product['productid'], $dbImageString); // Store the string (uses <br>)
            if ($updateSuccess === false) $anyFailures = true;
        } else {
            echo "Skipping DB update ID: " . $product['productid'] . " (No images saved).<br>"; // HTML output
            $anyFailures = true; // Count as failure if no image saved
        }
        echo "<br><br>"; // Keep HTML break

    } // End product loop

    // Finalize
    echo "Finished image batch.<br>"; // HTML output
    if ($conn) mysqli_close($conn);
    return !$anyFailures;
}


// --- Script Execution ---

//if (!isset($conn) || !($conn instanceof mysqli) || $conn->connect_error) { die("DB connection failed."); }

// Choose which main function to run
//$imgSuccess = databaseProductImageUpdate($conn); // Run image update process
//$descSuccess = databaseDescriptionUpdate($conn); // Run description update
//$descSuccess = databaseDescriptionUpdate($conn, 100004237); // Run targeted description update
// Report status
//if ($imgSuccess) { echo "<br>Image update process completed successfully for the batch.<br>"; } // HTML output
//else { echo "<br>Image update process encountered one or more errors.<br>"; } // HTML output

?>