<?php
/**
 * ajax_ai_extract.php
 *
 * Receives a conversationID and an optional category filter.
 * 1. Fetches the latest email from the conversation.
 * 2. Calls Gemini AI to extract structured data (segments, itinerary items, etc.).
 * 3. If segmented data is found, iterates through segments.
 * 4. For each segment, searches for the best matching template based on criteria.
 * 5. **Automatically logs errors to tdu_ai_error_log for:**
 *    - Segment count mismatch (duration/city/country). (Type: AUTO_MISMATCH)
 *    - Failure to find a template for a segment. (Type: AUTO_NO_TEMPLATE)
 * 6. Returns JSON response with success/error, segment results, AI data.
 */

// --- Error Reporting (Development: uncomment, Production: comment/log) ---
// ini_set('display_errors', 1);
// ini_set('display_startup_errors', 1);
// error_reporting(E_ALL);
ini_set('display_errors', 0); // Production: Log errors instead of displaying
error_reporting(E_ALL);
ini_set('log_errors', 1);
// --- Includes and Session Start ---
require_once "dbconn.php"; // Establishes $conn
include_once "dictionaries.php"; // Optional
require_once "cities_countries_functions.php"; // Optional
require_once __DIR__ . '/google-api-php-client/vendor/autoload.php'; // Google API
require_once "ai_keys.php";

session_start();

// --- Authentication Check ---
if (!isset($_SESSION['user_name'])) {
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    header('Expires: Mon, 26 Jul 1997 05:00:00 GMT');
    http_response_code(401); // Unauthorized
    echo json_encode(['success' => false, 'error' => 'Authentication required.']);
    exit;
}

// --- Timezone ---
date_default_timezone_set('Australia/Melbourne');



/* ========================================================================== */
/* --- FUNCTIONS --- */
/* ========================================================================== */

/**
 * Defines the instructions for the Gemini AI model for data extraction.
 * (Keep this function as it is)
 * @return string The prompt instructions.
 */
function gemini_instructions() {
    /*
    $gemini_instructions = "
Deeply Analyze everything about the email subject and body carefully and extract the following information. If a piece of information is not found, use 'N/A'. Respond in JSON format.

Specifically identify and output in JSON format:
{
    category(testing): SIC or Private or SIC+Private or Group or unsure if unsure but only as a last resort make your best guess based on the analysis of the email,
    duration: Comma-separated list of durations (as nights) per aggregated city stay (e.g., '3, 2, 4') OR a comma-separated list of 'N/A' matching the city count OR a single total number/range OR 'N/A',
    cities: Comma-separated list of major cities/destinations corresponding to aggregated stays OR a single list OR 'N/A',
    country: Comma-separated list of countries corresponding to aggregated stays (repeating the country for each city) OR 'N/A',
    seaters: The amount of passengers or PAX (e.g., '7') or N/A,
    itinerary_items: List of itinerary items as a comma separated string or N/A if no itinerary items,
    reasoning: give reasonings for all the output as detailed as possible, do not use asterisk for formatting
}

**Definition of Major City/Destination:** Include major cities (Sydney, Melbourne, Brisbane, Cairns, Gold Coast, etc.) AND significant multi-day tourist destinations used as a base (Byron Bay, Ayers Rock/Uluru, Airlie Beach/Whitsundays). Exclude specific attractions or day trip locations.

**Overall Priority:** Check for segmented itineraries first (Rules 1 & 2). Prioritize extracting `duration`, `cities`, `country` based on **aggregated continuous stays** in the same Major City/Destination. If no segmented format yields results, fall back to Rules 3+.

**Instructions for Segmented Duration, Cities, and Country Extraction (Priority Order):**

1.  **Segmented Itinerary (XN format - HIGHEST PRIORITY):**
    *   Look for 'X nights City/Destination' format. Use the Major City/Destination definition.
    *   If found: Extract night counts, location names, and repeat country as comma-separated lists. Ensure counts match.
    *   **If this rule yields results, STOP and use these outputs.**

2.  **Segmented Itinerary (Date Range / Sequential Day format - SECOND PRIORITY):**
    *   Look for itinerary specified with locations associated with date ranges OR sequential day entries.
    *   **Step A: Identify Stay Blocks:** Mentally (or actually) group all consecutive days/date ranges where the guest is based overnight in the *same* Major City/Destination. A travel day marks the end of one block and the beginning of the next. **Day trips do NOT break a stay block.**
    *   **Step B: Calculate Nights per AGGREGATED Block:** For each identified **continuous stay block** in a single Major City/Destination:
        *   Determine the check-in date (first day associated with the block) and the check-out date (the day the guest *departs* from that location to the *next* different location or home).
        *   Calculate the **single TOTAL number of nights** spent during that **entire continuous block** (Check-Out Date - Check-In Date). **CRITICAL: Do NOT split a continuous stay in the same Major City/Destination into multiple segments in the output. Aggregate ALL consecutive nights in that single location into ONE duration entry.**
    *   **Step C: Record Location per Block:** Identify the single Major City/Destination for each **aggregated stay block**. Assume all segments are in the same primary country unless specified otherwise.
    *   **Step D: Assemble Output Lists:** Create the comma-separated strings for `duration`, `cities`, and `country`. Use the calculated **total nights** (from Step B) and the single location name (from Step C) for each identified **aggregated stay block**, maintaining chronological order. **CRITICAL for `country`: Repeat the identified country name for EACH segment identified in Step A/C.** Ensure `duration`, `cities`, and `country` lists have the same number of entries.
    *   **If this rule yields results (identifies at least one stay block), STOP and use these segmented outputs.**

**Instructions for Duration, Cities, Country (Fallback - If Rules 1 & 2 Yield No Results):**

    *   **NON-NEGOTIABLE CONSTRAINT:** If applying these fallback rules, the final output for `duration`, `cities`, and `country` **MUST** have the exact same number of comma-separated entries.

    3.  **Cities Extraction (Fallback Rule A - Determine Count):** Identify all distinct **Major Cities/Destinations** mentioned anywhere in the email. Output as a comma-separated list. If no Major Cities/Destinations are found, output 'N/A'. **Record the number of cities found (let this be 'N').**
    4.  **Duration Extraction (Fallback Rule B - Match Count):**
        *   First, check for an explicitly stated *single* total duration (e.g., 10-day trip, 2 weeks). If found, output this single value *only if* Rule A found 0 or 1 city.
        *   If Rule A found multiple cities (N > 1) OR if no single total duration was found, output a comma-separated list of 'N/A' repeated 'N' times (matching the count from Rule A).
        *   If Rule A found 'N/A' (N=0) and no single total duration was found, output 'N/A'.
    5.  **Country Extraction (Fallback Rule C - Match Count):**
        *   Identify the single primary country mentioned.
        *   If Rule A found one or more cities (N >= 1), output the identified country name repeated 'N' times in a comma-separated list (matching the count from Rule A).
        *   If Rule A found 'N/A' (N=0), output the single country name once.
        *   If no country is identifiable, output 'N/A' (this will automatically match the count if Rule A also resulted in 'N/A').

**Instructions for Seaters Extraction:**
6.  Extract total guests ('No of Pax - X', 'X adults'). Return number (e.g., '7'). 'N/A' if none.

**Instructions for Itinerary Items Extraction:**
7.  Identify key activities, tours, transfers, requests ('best price', 'With Hotel', etc.), specific hotel options, sights, parks, lookouts, AND locations not included in the 'cities' list (if Rule 1 or 2 applied). Comma-separated. 'N/A' if none.

**General Instructions:**
*   Latest info focus, spell check, infer country, ignore irrelevant.
*   **Reasoning:** MUST explain the identified stay blocks and night calculations if Rule 2 was used. If Fallback Rules were used, MUST state this and explain how the count matching was enforced for duration, cities, and country based on the identified cities. Explain which rule was applied for each field.
*   **REMINDER:** CITIES, COUNTRY, AND DURATION MUST HAVE THE SAME COUNT.

Ambiguity: Best guess, explain in reasoning.
    ";
    */
    
    $gemini_instructions = 
    "
**Objective:** Analyze the provided email subject and body to extract travel itinerary details. Focus on identifying structured trip segments first. Output the information strictly in the JSON format below.

**CRITICAL OUTPUT CONSTRAINT:** The absolute highest priority is ensuring that the final comma-separated lists for 'duration', 'cities', and 'country' have the EXACT SAME number of entries. If one has 3 entries, ALL THREE must have 3 entries. This applies even if the entries are 'N/A'. Verify this count match rigorously before finalizing the output. Mismatched counts are incorrect.

**Output JSON Format:**

```json
{
  'category': 'SIC or Private or SIC+Private or Group or unsure',
  'duration': 'Comma-separated list of nights per aggregated city stay (e.g., \'3, 2, 4\') OR single total OR \'N/A\' OR list of \'N/A\'s. MUST match entry count of cities/country. The output has to be number only',
  'cities': 'Comma-separated list of Major Cities/Destinations OR \'N/A\'. MUST match entry count of duration/country.',
  'country': 'Comma-separated list of countries (repeated per city) OR single country OR \'N/A\'. MUST match entry count of duration/cities.',
  'seaters': 'Number of passengers. if its a range, extract the highest number, it can only be a single number (e.g., \'7\') OR \'N/A\'',
  'itinerary_items': 'Comma-separated list of activities, tours, transfers, requests, non-major destinations, hotels OR \'N/A\'',
  'reasoning': 'Detailed explanation for how each field\'s value was derived (source text, calculation, logic applied), without naming specific internal rule numbers.'
}```

**Key Definitions:**

*   **Major City/Destination:** Includes major cities (e.g., Sydney, Melbourne, Brisbane, Cairns, Gold Coast, Perth, Adelaide) AND significant multi-day tourist destinations used as a base (e.g., Byron Bay, Ayers Rock/Uluru, Airlie Beach/Whitsundays). Exclude specific attractions or day trip locations (like a specific national park visited for a day) from the `cities` list; these belong in `itinerary_items`.
*   **Aggregated Stay:** A continuous period spent overnight in the *same* Major City/Destination. Day trips *do not* break an aggregated stay block. Travel *to a different overnight location* ends one aggregated stay block and begins the next.

**Extraction Strategy (Apply in Order):**

**Phase 1: Segmented Itinerary Extraction (Highest Priority for `duration`, `cities`, `country`)**

*   **Attempt 1: Explicit Nights Format ('X nights City')**
    *   Search for patterns like '3 nights Sydney', '2 nights Cairns'. Use the Major City/Destination definition.
    *   If found: Extract night counts, location names, and repeat country. **Crucially, ensure the generated lists for duration, cities, and country have the exact same number of entries.**
    *   Populate `duration`, `cities`, `country` lists accordingly (e.g., duration: '3, 2', cities: 'Sydney, Cairns', country: 'Australia, Australia').
    *   **If this attempt yields valid, count-matched results for `duration`, `cities`, `country`, STOP attempts for these fields and proceed to Phase 3 for other fields.**

*   **Attempt 2: Date/Day Range Format**
    *   Look for itineraries specified by date ranges (e.g., 'Oct 1-4: Sydney', 'Oct 4-6: Melbourne') or sequential days ('Day 1-3: Sydney', 'Day 4-5: Melbourne').
    *   **Step A: Identify and Aggregate Continuous Stays:** Group consecutive days/dates for overnight stays in the *same* Major City/Destination. Calculate the **single TOTAL number of nights** for each **entire continuous block** (Check-Out Date - Check-In Date = Nights). Aggregate ALL consecutive nights in one location into ONE duration entry.
    *   **Step B: Record Location and Country per Block:** Identify the single Major City/Destination and country for each **aggregated stay block**.
    *   **Step C: Assemble Output Lists:** Create comma-separated strings for `duration` (total nights per block), `cities` (location per block), and `country` (repeated country per block). **CRITICAL: Verify these three lists have the exact same number of entries and maintain chronological order.**
    *   **Example:** 'Oct 1-4: Sydney stay, Oct 4 Travel to Mel, Oct 4-6: Melbourne stay' -> Aggregated blocks: Sydney (Oct 1 check-in, Oct 4 check-out = 3 nights), Melbourne (Oct 4 check-in, Oct 6 check-out = 2 nights). Output -> duration: '3, 2', cities: 'Sydney, Melbourne', country: 'Australia, Australia'. (Counts match: 2 entries each).
    *   **If this attempt yields valid, count-matched results (identifies at least one aggregated stay block) for `duration`, `cities`, `country`, STOP attempts for these fields and proceed to Phase 3 for other fields.**

**Phase 2: Fallback Extraction (Use ONLY if Phase 1 yielded NO count-matched results for `duration`, `cities`, `country`)**

*   **NON-NEGOTIABLE CONSTRAINT:** `duration`, `cities`, and `country` outputs generated in this phase MUST have the exact same number of comma-separated entries. This ensures consistency even if specific details aren't available (e.g., outputting 'N/A', 'Sydney, Melbourne', 'Australia, Australia' is WRONG; it should be 'N/A, N/A', 'Sydney, Melbourne', 'Australia, Australia' if duration details are missing but two cities are found).
*   **Step A: Identify Cities (Determine Count 'N'):** Find all distinct **Major Cities/Destinations** mentioned. List them comma-separated in `cities`. Record the count 'N'. If none, `cities` is 'N/A' (N=0).
*   **Step B: Determine Duration (Match Count 'N'):**
    *   Check for a single total duration (e.g., '10-day trip').
    *   If found AND N <= 1, set `duration` to this single value (e.g., '10').
    *   Otherwise (N > 1 OR no total duration found), set `duration` to a comma-separated list of 'N/A' repeated 'N' times. **This forces the duration count to match the city count.**
    *   If `cities` was 'N/A' (N=0) and no total duration, `duration` is 'N/A'. (Count is 1, matching N=0 fallback logic).
*   **Step C: Determine Country (Match Count 'N'):**
    *   Identify the primary country.
    *   If N >= 1, set `country` to the country name repeated 'N' times, comma-separated. **This forces the country count to match the city count.**
    *   If `cities` was 'N/A' (N=0), set `country` to the single country name (or 'N/A' if unknown). (Count is 1, matching N=0 fallback logic).
    *   **Crucially re-verify:** Ensure the final `duration`, `cities`, and `country` lists derived from Steps A, B, C have the exact same number of entries before finalizing.

**Phase 3: Extract Other Fields (Apply regardless of Phase 1 or 2 outcome)**

*   **Seaters:** Extract total guests ('No of Pax - X', 'X adults', 'X travellers'). Return number (e.g., '7', '2'). 'N/A' if none.
*   **Itinerary Items:** List key activities, tours, transfers, requests (e.g., 'best price', 'With Hotel', 'airport transfer'), hotels, sights, parks, lookouts, AND non-major-destination locations. Comma-separated. 'N/A' if none.
*   **Category:** Determine `SIC`, `Private`, `SIC+Private`, `Group` based on mentions or context. Use `unsure` as a last resort.

**Phase 4: Reasoning**

*   **Crucial:** For *every* field in the JSON, provide a detailed step-by-step reasoning in the `reasoning` field. Explain *how* the value was derived:
    *   For `duration`, `cities`, `country`: Explain the source (e.g., 'extracted directly from '3 nights Sydney' text', 'calculated nights based on Oct 1-4 dates', 'identified cities X, Y and generated N/A list for duration to match count'). Explicitly state if the fallback logic (Phase 2) was necessary because no structured segments were found.
    *   For `seaters`, `category`, `itinerary_items`: Explain the evidence found in the text (e.g., 'found text 'No of Pax - 2'', 'category based on mention of 'private tour'', 'items listed are tours and activities mentioned').

**General Notes:**

*   Prioritize the most current information.
*   Infer the primary country if needed.
*   Ignore greetings, closings, signatures, filler. Focus on itinerary details.
*   Keep in mind that the final message take priority if there are any conflicts as customers can change their mind
*   **Final Check:** Before outputting the JSON, perform one last explicit check that the number of comma-separated elements in `duration`, `cities`, and `country` are identical.
";
    return $gemini_instructions;
}


/**
 * Sends a prompt to the Gemini API and returns the JSON response string.
 * (Keep this function as it is)
 * @param string $prompt The text prompt to send.
 * @return string JSON string containing the response or an error structure.
 */
function promptGeminiAI($prompt) {
    $apiUrl = apiURL_Flash(); // Assuming this function returns your API URL
    if (!$apiUrl) {
        error_log("promptGeminiAI_forSegments: API URL is missing.");
        return json_encode(['error' => 'API URL not configured for Gemini.']);
    }
    //error_log($prompt);
    $payload = [
        'contents' => [['parts' => [['text' => $prompt]]]],
        'generationConfig' => ['temperature' => 0.3, 'response_mime_type' => "application/json", 'maxOutputTokens' => 8192]
    ];
    $headers = ['Content-Type: application/json'];

    $ch = curl_init($apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 20); // Connection timeout
    curl_setopt($ch, CURLOPT_TIMEOUT, 100);       // Total cURL execution timeout

    $response_body = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    if ($curlError || $httpCode != 200) {
        $error_message = "Gemini API Error (Segments HTTP $httpCode, cURL $curlError): " . $response_body;
        error_log($error_message);
        $details = json_decode($response_body, true);
        return json_encode(['error' => "Gemini API Error (Segments HTTP $httpCode)", 'details' => $details ?: strip_tags($response_body)]);
    }

    $responseData = json_decode($response_body, true);
    // var_dump($responseData); // Keep this for debugging if needed, then remove/comment out

    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("Gemini JSON Decode Error (Segments main - outer response): " . json_last_error_msg() . " | Raw: " . $response_body);
        return json_encode(['error' => 'Gemini API: Failed to decode main JSON (Segments outer response)', 'raw_response' => $response_body]);
    }

    if (isset($responseData['promptFeedback']['blockReason'])) {
        $blockReason = $responseData['promptFeedback']['blockReason'] ?? 'Unknown';
        error_log("Gemini Content Blocked (Segments): " . $blockReason);
        return json_encode(['error' => 'Gemini: Content blocked (Segments)', 'details' => $responseData['promptFeedback']]);
    }

    // --- MODIFICATION START: Handle multi-part response ---
    if (isset($responseData['candidates'][0]['content']['parts']) && is_array($responseData['candidates'][0]['content']['parts'])) {
        $finalJsonText = null;
        $allPartsCombinedTextForLog = ""; // For logging if no JSON found

        foreach ($responseData['candidates'][0]['content']['parts'] as $partIndex => $part) {
            $currentText = $part['text'] ?? '';
            $allPartsCombinedTextForLog .= "Part #{$partIndex}: " . $currentText . "\n---\n";

            if (isset($part['text'])) {
                // Check if this part is explicitly marked as a "thought" (heuristic)
                // The more reliable check is trying to decode it as JSON.
                $isThoughtPart = isset($part['thought']) && $part['thought'] === true;

                if (!$isThoughtPart) { // Prioritize parts not explicitly marked as thoughts
                    $trimmedPartText = trim($currentText);

                    // Remove common markdown wrappers for JSON
                    if (substr($trimmedPartText, 0, 7) === "```json") {
                        $trimmedPartText = substr($trimmedPartText, 7);
                        if (substr($trimmedPartText, -3) === "```") {
                            $trimmedPartText = substr($trimmedPartText, 0, -3);
                        }
                        $trimmedPartText = trim($trimmedPartText); // Trim again
                    } elseif (strpos($trimmedPartText, '```') === 0 && strrpos($trimmedPartText, '```') === (strlen($trimmedPartText) - 3) ) {
                        // Handle case where only ``` is used without json specifier
                        $trimmedPartText = substr($trimmedPartText, 3, -3);
                        $trimmedPartText = trim($trimmedPartText);
                    }


                    // Attempt to decode this part.
                    json_decode($trimmedPartText);
                    if (json_last_error() === JSON_ERROR_NONE) {
                        $finalJsonText = $trimmedPartText; // Found a valid JSON part
                        error_log("promptGeminiAI_forSegments: Successfully extracted JSON from part #{$partIndex}.");
                        break; // Assume the first valid non-thought JSON part is the one we want
                    } else {
                         error_log("promptGeminiAI_forSegments: Part #{$partIndex} was not valid JSON. Error: ".json_last_error_msg().". Content: ".substr($trimmedPartText, 0, 200)."...");
                    }
                } else {
                     error_log("promptGeminiAI_forSegments: Part #{$partIndex} was marked as a 'thought'. Skipping direct JSON use for now.");
                }
            }
        }

        if ($finalJsonText !== null) {
            return $finalJsonText; // Return the AI's valid JSON string
        } else {
            error_log("Gemini response did not contain a valid JSON part after checking all parts. All parts combined: " . $allPartsCombinedTextForLog . "Raw full response: " . $response_body);
            return json_encode(['error' => 'Gemini API: No valid JSON found in response parts', 'raw_response' => $response_body, 'all_parts_text' => $allPartsCombinedTextForLog]);
        }
    }
    // --- MODIFICATION END ---

    // Fallback if the expected 'parts' structure isn't found or the above logic didn't return
    error_log("Gemini response format unexpected (Segments - 'parts' structure missing or issue in processing). Raw: " . $response_body);
    // Return the raw response or a generic error. Returning the raw response might help in debugging.
    // However, the calling function expects a JSON string (either the AI's JSON or an error JSON).
    return json_encode(['error' => 'Gemini API: Unexpected response format (parts structure issue)', 'raw_response' => $response_body]);
}


/**
 * Fetches the latest message details from a conversation ID.
 * (Keep this function as it is)
 * @param string $conversationID
 * @param mysqli $conn Database connection object
 * @return array|null Message details or null.
 */
function getMessageFromConversationID($conversationID, $conn) {
     // ... (getMessageFromConversationID function remains unchanged) ...
    if (empty($conversationID) || !$conn) return null;

    $conversationID_safe = mysqli_real_escape_string($conn, $conversationID);
    $sql = "SELECT subject, plaintext_body, full_body, sender
            FROM tdu_emails
            WHERE conversation_id = '$conversationID_safe'
            ORDER BY received_datetime DESC
            LIMIT 1";

    $result = mysqli_query($conn, $sql);
    if ($result && mysqli_num_rows($result) > 0) {
        $messageData = mysqli_fetch_assoc($result);
        mysqli_free_result($result);
        return $messageData;
    } elseif (!$result) {
        error_log("DB Error fetching message for Conv ID $conversationID_safe: " . mysqli_error($conn));
    }
    return null;
}

/**
 * Fetches product and vendor names for a template ID.
 * (Keep this function as it is)
 * @param int $templateID
 * @param mysqli $conn Database connection object
 * @return array List of items [{'productName': ..., 'vendorName': ...}] or empty array.
 */
function getTemplateItems($templateID, $conn) {
      // ... (getTemplateItems function remains unchanged) ...
     $items = [];
     if (empty($templateID) || !is_numeric($templateID) || !$conn) return $items;

     $templateID_safe = (int)$templateID;
     $sql = "SELECT p.productName, v.vendorName
             FROM tdu_template_products tp
             LEFT JOIN tdu_products p ON tp.productid = p.productid
             LEFT JOIN tdu_vendors v ON p.vendorid = v.vendorid
             WHERE tp.templateid = $templateID_safe
             ORDER BY CAST(tp.day AS UNSIGNED) ASC, tp.sequence_no ASC";

     $result = mysqli_query($conn, $sql);
     if ($result) {
         while ($row = mysqli_fetch_assoc($result)) {
             $items[] = $row;
         }
         mysqli_free_result($result);
     } else {
         error_log("DB Error fetching template items for ID $templateID_safe: " . mysqli_error($conn));
     }
     return $items;
}

/**
 * Searches for the most relevant active template based on criteria and relevance scoring.
 * (Keep this function as it is)
 * @param string $sender_email Sender's email address.
 * @param string $seaters AI extracted seaters ('N/A' or number).
 * @param string $duration AI extracted duration for segment ('N/A' or number).
 * @param string $country AI extracted country for segment ('N/A' or name).
 * @param string $tour_type AI extracted tour type (context only).
 * @param string $cities AI extracted city for segment ('N/A' or name).
 * @param string $itinerary_items AI extracted itinerary items (comma-separated string or 'N/A').
 * @param mysqli $conn Database connection object.
 * @param string $selectedCategory Category selected from dropdown ('', 'SIC', 'Private', 'Group', 'SIC+Private').
 * @return array|null Best matching template details (associative array including 'template_items') or null.
 */
function relevancy_search_template($sender_email, $seaters, $duration, $country, $tour_type, $cities, $itinerary_items, $conn, $selectedCategory) {
     // ... (relevancy_search_template function remains unchanged) ...
    if (!$conn) { error_log("relevancy_search_template: Invalid DB connection."); return null; }

    $where_clause_parts = ["t.active = 1"]; // Start with active templates

    // Duration Filter
    $duration_clean = trim($duration);
    if ($duration_clean != 'N/A' && is_numeric($duration_clean) && intval($duration_clean) > 0) {
        $where_clause_parts[] = "t.nights = " . intval($duration_clean);
    }

    // Country Filter
    $country_clean = trim($country);
    if ($country_clean != 'N/A' && !empty($country_clean)) {
        $where_clause_parts[] = "t.country = '" . mysqli_real_escape_string($conn, $country_clean) . "'";
    }

    // Category Filter (Dropdown Priority)
    if (!empty($selectedCategory)) {
        $cats = [];
        if ($selectedCategory === 'SIC') $cats[] = "t.category = 'SIC'";
        elseif ($selectedCategory === 'Private') $cats[] = "t.category = 'Private'";
        elseif ($selectedCategory === 'Group') $cats[] = "t.category = 'Group'";
        elseif ($selectedCategory === 'SIC+Private') $cats[] = "(t.category = 'SIC+Private')"; // Adjusted based on previous feedback
        if (!empty($cats)) {
            $where_clause_parts[] = "(" . implode(" OR ", $cats) . ")";
        }
    }
    // Note: Fallback category logic based on email/seaters is removed as dropdown is primary

    // Seaters Filter (Apply ONLY if category is NOT SIC)
    $ai_seats_num = ($seaters != 'N/A' && is_numeric($seaters)) ? intval($seaters) : 0;
    if ($selectedCategory != 'SIC' && $ai_seats_num > 0) {
        $where_clause_parts[] = "t.seaters >= " . $ai_seats_num;
    }

    // City Filter (LIKE match)
    $city_clean = trim($cities);
    if ($city_clean != 'N/A' && !empty($city_clean)) {
        $where_clause_parts[] = "t.city LIKE '%" . mysqli_real_escape_string($conn, $city_clean) . "%'";
    }

    // Build SQL for potential matches
    $base_sql = "SELECT t.* FROM tdu_templates t WHERE " . implode(" AND ", $where_clause_parts);

    // Find Minimum Applicable Seaters (Apply ONLY if category is NOT SIC and seaters > 0)
    $min_seaters_val = null;
    if ($selectedCategory != 'SIC' && $ai_seats_num > 0) {
        $min_seaters_sql = "SELECT MIN(sub.seaters) FROM ($base_sql) as sub WHERE sub.seaters >= $ai_seats_num"; // Ensure min is >= required
        $min_result = mysqli_query($conn, $min_seaters_sql);
        if ($min_result) {
            $min_row = mysqli_fetch_row($min_result);
            if ($min_row && $min_row[0] !== null) { // Check if a minimum was found
                 $min_seaters_val = intval($min_row[0]);
            }
            mysqli_free_result($min_result);
        } else {
            error_log("DB Error finding min seaters: " . mysqli_error($conn));
        }
    }

    // Fetch Potential Templates (apply min seaters filter ONLY if applicable)
    $potential_templates = [];
    $final_sql = $base_sql;
    // Apply the specific minimum seater value ONLY if it was found AND category is NOT SIC
    if ($min_seaters_val !== null && $selectedCategory != 'SIC') {
        $final_sql .= " AND t.seaters = " . $min_seaters_val;
    }
    $final_sql .= " ORDER BY t.preferred DESC, t.templateid ASC"; // Prioritize preferred, then ID

    $result = mysqli_query($conn, $final_sql);
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $potential_templates[] = $row;
        }
        mysqli_free_result($result);
    } else {
        error_log("DB Error fetching potential templates with final SQL [$final_sql]: " . mysqli_error($conn));
        return null;
    }

    if (empty($potential_templates)) return null;

    // Prepare AI itinerary items for comparison
    $gemini_itinerary_array = [];
    if ($itinerary_items != 'N/A' && !empty(trim($itinerary_items))) {
        $gemini_itinerary_array = array_filter(array_map('trim', explode(',', $itinerary_items)));
        $gemini_itinerary_array = array_map('strtolower', $gemini_itinerary_array); // Lowercase for comparison
    }

    // Calculate Relevance, Add Items, and Prepare for Sorting
    $combined_results = [];
    foreach ($potential_templates as $template) {
        $item_relevance_score = 0;
        $template_items = getTemplateItems($template['templateid'], $conn); // Fetch items
        $template['template_items'] = $template_items; // Embed items into the template data

        // Calculate relevance score based on itinerary overlap
        if (!empty($gemini_itinerary_array) && !empty($template_items)) {
            $template_item_content = strtolower($template['templatename'] ?? '');
            foreach ($template_items as $item) {
                $template_item_content .= " " . strtolower(($item['productName'] ?? '') . " " . ($item['vendorName'] ?? ''));
            }
            $template_item_content = preg_replace('/\s+/', ' ', trim($template_item_content));

            foreach ($gemini_itinerary_array as $gemini_item) {
                if (!empty($gemini_item) && stripos($template_item_content, $gemini_item) !== false) {
                    $item_relevance_score++;
                }
            }
        }
        $template['relevance'] = $item_relevance_score;
        $template['preferred'] = $template['preferred'] ?? 0; // Default preferred to 0 if null

        $combined_results[] = $template;
    }

    // Sort by Preferred (desc), then Relevance (desc)
    usort($combined_results, function($a, $b) {
        if ($b['preferred'] != $a['preferred']) {
            return $b['preferred'] - $a['preferred'];
        }
        return $b['relevance'] - $a['relevance'];
    });

    // Return the top result
    return $combined_results[0];
}


/**
 * ---- NEW HELPER FUNCTION ----
 * Logs an error message to the tdu_ai_error_log table.
 *
 * @param mysqli $conn Database connection object.
 * @param string $conversationId The conversation ID related to the error.
 * @param ?int $aiRef The reference ID (from tdu_temp_ai_quotes), if applicable (can be null).
 * @param string $reason A description of the error.
 * @param string $type The type of error (e.g., 'AUTO_MISMATCH', 'AUTO_NO_TEMPLATE').
 * @param string $status The initial status (e.g., 'PENDING').
 * @return bool True on success, false on failure.
 */
function log_ai_error($conn, $conversationId, $aiRef, $reason, $type, $status) {
    $tableName = 'tdu_ai_error_log'; // Ensure table name is correct

    // Basic validation
    if (!$conn || empty($conversationId) || empty($reason) || empty($type) || empty($status)) {
        error_log("log_ai_error: Missing required parameters.");
        return false;
    }

    $insertSql = "INSERT INTO {$tableName} (ai_ref, conversation_id, reason, type, status, created_at)
                  VALUES (?, ?, ?, ?, ?, NOW())";

    $stmt = mysqli_prepare($conn, $insertSql);

    if ($stmt) {
        // Bind parameters: 'i' for integer aiRef (null handled correctly), 's' for strings
        mysqli_stmt_bind_param($stmt, "issss",
            $aiRef,          // Can be null
            $conversationId,
            $reason,
            $type,
            $status
        );

        if (mysqli_stmt_execute($stmt)) {
            mysqli_stmt_close($stmt);
            return true; // Log successful
        } else {
            error_log("log_ai_error: Database Error (Execute Insert {$tableName}): " . mysqli_stmt_error($stmt) . " | Conv ID: " . $conversationId);
            mysqli_stmt_close($stmt);
            return false; // Log failed
        }
    } else {
        error_log("log_ai_error: Database Error (Prepare Insert {$tableName}): " . mysqli_error($conn) . " | Conv ID: " . $conversationId);
        return false; // Log failed
    }
}


/* ========================================================================== */
/* --- Main AJAX Request Handling --- */
/* ========================================================================== */

// Set JSON headers
header('Content-Type: application/json');
header('Cache-Control: no-cache, must-revalidate');
header('Expires: Mon, 26 Jul 1997 05:00:00 GMT');

// Get POST data
$conversationID = $_POST['conversationID'] ?? null;
$selectedCategory = trim($_POST['category'] ?? ''); // Category filter from frontend

// Validate Database Connection
if (!$conn || $conn->connect_error) {
     $db_error = $conn->connect_error ?? 'Unknown DB connection error';
     error_log("Database connection failed in ajax_ai_extract.php: " . $db_error);
     http_response_code(500);
     echo json_encode(['success' => false, 'error' => 'Database connection failed.']);
     exit;
}

// Validate Input
if (empty($conversationID)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Conversation ID not provided.']);
    exit;
}

// 1. Fetch Email Details
$message = getMessageFromConversationID($conversationID, $conn);
if (!$message) {
    http_response_code(404); // Not Found
    echo json_encode(['success' => false, 'error' => "Email not found for Conversation ID: " . htmlspecialchars($conversationID)]);
    exit;
}

$subject = $message['subject'] ?? 'N/A';
$body_raw = $message['plaintext_body'] ?? ($message['full_body'] ?? '');
$sender_email = $message['sender'] ?? 'N/A';

// Prepare body text for AI
$body = '';
if (!empty($body_raw)) {
    $body = (isset($message['plaintext_body']) && !empty($message['plaintext_body'])) ? $body_raw : strip_tags($body_raw);
    $body = preg_replace(['/\s+/', '/\n+/'], ' ', trim($body));
}
if (empty($body) && ($subject === 'N/A' || empty($subject))) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Cannot analyze: Email body and subject are empty.']);
    exit;
}

// 2. Call Gemini AI for Structured Extraction
$geminiStructuredPrompt = gemini_instructions() . "\nSubject: " . $subject . "\nBody: " . $body;
$rawGeminiResponseString = promptGeminiAI($geminiStructuredPrompt); // Gets JSON string (or error JSON)
$structuredDataArray = json_decode($rawGeminiResponseString, true); // Decode the result

// Robust Error Handling for Gemini Response
if ($structuredDataArray === null || json_last_error() !== JSON_ERROR_NONE) {
    error_log("Fatal JSON Decode Error for Gemini Response. Raw: " . $rawGeminiResponseString . " | Conv ID: " . $conversationID);
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Error decoding Gemini JSON response.', 'raw_ai_response' => $rawGeminiResponseString]);
    exit;
}
if (isset($structuredDataArray['error'])) { // Check for functional errors from promptGeminiAI
    error_log("Gemini API Error: " . $structuredDataArray['error'] . " | Details: " . json_encode($structuredDataArray['details'] ?? null) . " | Conv ID: " . $conversationID);
    http_response_code(502); // Bad Gateway (error from upstream API)
    echo json_encode(['success' => false, 'error' => $structuredDataArray['error'], 'details' => $structuredDataArray['details'] ?? null, 'raw_ai_response' => $rawGeminiResponseString ]);
    exit;
}

// 3. Extract Key Data Points from AI Response
$tourType = $structuredDataArray['category(testing)'] ?? 'N/A'; // Using corrected key
$seaters = $structuredDataArray['seaters'] ?? 'N/A';
$durationStr = $structuredDataArray['duration'] ?? 'N/A';
$citiesStr = $structuredDataArray['cities'] ?? 'N/A';
$countryStr = $structuredDataArray['country'] ?? 'N/A';
$itinerary_items_ai = $structuredDataArray['itinerary_items'] ?? 'N/A'; // Raw AI string

// 4. Process Segments and Search for Templates
$segmentResults = [];
$is_segmented = (strpos($durationStr, ',') !== false &&
                 strpos($citiesStr, ',') !== false &&
                 strpos($countryStr, ',') !== false);

// Use a flag to decide if we should always try to process, even if not strictly segmented by commas
// This matches the original logic `if ($is_segmented or true)`
$force_process_as_segments = true; // Set to true to always attempt processing

if ($is_segmented || $force_process_as_segments) {
    // Split comma-separated values (trim ensures no empty elements from double commas)
    $durations = array_filter(array_map('trim', explode(',', $durationStr)));
    $cities = array_filter(array_map('trim', explode(',', $citiesStr)));
    $countries = array_filter(array_map('trim', explode(',', $countryStr)));

    // Use a consistent count, preferably cities as it's often the most reliable anchor
    $segment_count = count($cities);

    // Ensure counts match before proceeding *if* data was expected to be segmented
    // If counts are 1 and it wasn't comma-separated, it's okay.
    $counts_match = (count($durations) === $segment_count && count($countries) === $segment_count);

    if ($segment_count > 0 && $counts_match) {
        for ($i = 0; $i < $segment_count; $i++) {
            // Use array indexing based on the loop counter $i
            // Note: If arrays were filtered, direct indexing might be wrong.
            // It's safer to re-index or access based on a reliable source like cities array keys if filtered.
            // For simplicity here, assuming explode gives arrays ready for 0-based indexing.
            // If using array_filter, it's better to use array_values() first:
             $durations_val = array_values($durations);
             $cities_val = array_values($cities);
             $countries_val = array_values($countries);
             // Now use $durations_val[$i], etc.

            $current_duration = $durations_val[$i] ?? 'N/A';
            $current_city = $cities_val[$i] ?? 'N/A';
            $current_country = $countries_val[$i] ?? 'N/A';

            // Perform relevance search for each segment
            $best_template_details = relevancy_search_template(
                $sender_email, $seaters, $current_duration, $current_country,
                $tourType, $current_city, $itinerary_items_ai, $conn, $selectedCategory
            );

            // Log if no template was found for a segment
            if ($best_template_details === null) {
                 $error_reason = sprintf(
                    "No template found for segment %d (Input: Dur=%s, City=%s, Country=%s, Seats=%s, CatFilter=%s)",
                    $i + 1,
                    $current_duration,
                    $current_city,
                    $current_country,
                    $seaters,
                    empty($selectedCategory) ? $tourType : $selectedCategory // Use AI category if filter empty
                );
                error_log("[AI Extract - No Template] ConvID: {$conversationID} | {$error_reason}"); // Keep server log

                // --- AUTOMATICALLY LOG 'NO TEMPLATE' ERROR ---
                log_ai_error(
                    $conn,
                    $conversationID,
                    null, // ai_ref is null here
                    $error_reason,
                    'AUTO_NO_TEMPLATE', // Specific type for this error
                    'PENDING'           // Initial status
                );
                // --- END OF AUTO LOGGING ---
            }

            // Store segment input and the matched template (or null)
            $segmentResults[] = [
                'segment_number' => $i + 1,
                'input_duration' => $current_duration,
                'input_city' => $current_city,
                'input_country' => $current_country,
                'matched_template' => $best_template_details // Contains full template details + 'template_items' array or null
            ];
        }
    } elseif ($segment_count > 0 && !$counts_match) {
        // Mismatch in segment counts - Log error and return error response
        $error_reason = sprintf(
            "Segmented data count mismatch. Durations: %d, Cities: %d, Countries: %d. Raw Dur: '%s', Raw Cities: '%s', Raw Country: '%s'",
            count($durations),
            count($cities),
            count($countries),
            $durationStr,
            $citiesStr,
            $countryStr
        );
        error_log("[AI Extract - Mismatch] ConvID: {$conversationID} | {$error_reason}"); // Keep server log

        // --- AUTOMATICALLY LOG 'MISMATCH' ERROR ---
        log_ai_error(
            $conn,
            $conversationID,
            null, // ai_ref is null here
            $error_reason,
            'AUTO_MISMATCH', // Specific type for this error
            'PENDING'        // Initial status
        );
        // --- END OF AUTO LOGGING ---

         http_response_code(400); // Bad Request due to data inconsistency
         echo json_encode([
             'success' => false,
             'error' => 'Segmented data count mismatch detected in AI response.',
             'details' => $error_reason, // Provide details in the response too
             'raw_ai_response' => $rawGeminiResponseString // Include raw response for debugging
            ]);
         mysqli_close($conn);
         exit;
    } else {
         // Case: AI didn't provide comma-separated lists, or they were empty after trimming/filtering.
         // This path might be hit if $force_process_as_segments was true but the data wasn't suitable.
         // Consider if this scenario should also log an error or just return an empty segment list.
         // For now, let it proceed to return success with empty segments.
         error_log("[AI Extract - Non-Segmented Data] ConvID: {$conversationID} | AI data did not yield processable segments. Dur: '{$durationStr}', Cities: '{$citiesStr}', Country: '{$countryStr}'");
    }

} else {
    // Handle case where AI response was clearly not segmented AND $force_process_as_segments was false
    // This part of the logic might not be reached if $force_process_as_segments is always true.
     error_log("[AI Extract - Not Segmented] ConvID: {$conversationID} | AI analysis did not produce a comma-separated segmented itinerary.");
     // No error is logged here by default, just info in server logs. Proceed to return success.
}

// 5. Return Success Response with Segment Results (or empty if no segments processed/found)
echo json_encode([
    'success' => true,
    'segments' => $segmentResults,           // Array of segment results
    'itinerary_items_used' => $itinerary_items_ai, // Raw string from AI used for relevance
    'raw_ai_response' => $rawGeminiResponseString,  // Full raw AI JSON string
    'message' => empty($segmentResults) ? 'AI analysis processed, but no valid segments with matching templates were finalized.' : 'AI analysis and template matching complete.'
]);

mysqli_close($conn);
exit;

?>