<?php
require_once "ai_keys.php"; // Defines $apiKey and apiURL_Flash()

/**
 * Sends a prompt to the Gemini API, handles multi-part responses,
 * cleans the identified final output, and echoes it.
 *
 * @param string $prompt The text prompt to send.
 */
function promptGeminiAI(string $prompt)
{
    $apiUrl = apiURL_Flash(); // Gemini Flash model endpoint
    if (empty($apiUrl)) {
        error_log("promptGeminiAI Error: API URL is missing."); // Log error
        echo "Error: API URL is missing.\n";
        return;
    }

    $payload = [
        'contents' => [
            [
                'parts' => [
                    ['text' => $prompt]
                ]
            ]
        ],
        'generationConfig' => [
            'temperature' => 0.3, // Keep temperature low for more predictable output
            'maxOutputTokens' => 15000,
            // 'response_mime_type' => "text/plain", // Or "text/html" if you specifically expect HTML
                                                   // If you expect JSON, it should be "application/json"
                                                   // For general text, "text/plain" is safer if you don't want it trying to force JSON.
                                                   // If you're getting HTML, "text/html" might be better.
                                                   // Given the AI's output, it seems to be aiming for HTML.
            'response_mime_type' => "text/plain",

        ]
    ];

    $headers = [
        'Content-Type: application/json',
        // The API key is often included in the $apiUrl string from apiURL_Flash()
        // e.g., $apiUrl = "https://generativelanguage.googleapis.com/.../gemini-1.5-flash-latest:generateContent?key=" . $apiKey;
        // If not, you might need to add it here, but usually it's in the URL for direct REST calls like this.
        // 'x-goog-api-key: ' . $apiKey, // Uncomment if your apiURL_Flash() doesn't include the key
    ];

    $ch = curl_init($apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 20);
    curl_setopt($ch, CURLOPT_TIMEOUT, 120); // Increased timeout for potentially larger responses

    $response_body = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlErrNo = curl_errno($ch);
    $curlErrMsg = curl_error($ch);

    if ($curlErrNo) {
        error_log("promptGeminiAI cURL Error ({$curlErrNo}): " . $curlErrMsg);
        echo "cURL Error: " . $curlErrMsg . "\n";
        curl_close($ch);
        return;
    }

    curl_close($ch); // Close curl handle after execution

    if ($httpCode == 200) {
        $responseData = json_decode($response_body, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            error_log("promptGeminiAI Failed to decode API's main JSON response. Error: " . json_last_error_msg() . " | Raw: " . $response_body);
            echo "Error: Failed to parse API response.\nRaw Response: " . htmlspecialchars($response_body) . "\n";
            return;
        }

        if (isset($responseData['promptFeedback']['blockReason'])) {
            $blockReason = $responseData['promptFeedback']['blockReason'] ?? 'Unknown';
            error_log("promptGeminiAI Gemini Content Blocked: " . $blockReason . " | Details: " . json_encode($responseData['promptFeedback']));
            echo "Error: Gemini Content Blocked - " . htmlspecialchars($blockReason) . "\n";
            return;
        }

        $finalTextOutput = null;

        if (isset($responseData['candidates'][0]['content']['parts']) && is_array($responseData['candidates'][0]['content']['parts'])) {
            $parts = $responseData['candidates'][0]['content']['parts'];
            
            // Strategy:
            // 1. If there's only one part, use it.
            // 2. If multiple parts, try to find one that is NOT a "thought".
            // 3. If all parts seem to be "thoughts" or no clear distinction, take the LAST part,
            //    as the final output often comes after the reasoning.

            if (count($parts) === 1 && isset($parts[0]['text'])) {
                $finalTextOutput = $parts[0]['text'];
                error_log("promptGeminiAI: Using single part response.");
            } else {
                $nonThoughtPartsText = [];
                $allPartsTexts = []; // To store all texts for fallback

                foreach ($parts as $partIndex => $part) {
                    if (isset($part['text'])) {
                        $allPartsTexts[] = $part['text']; // Store all texts
                        if (!(isset($part['thought']) && $part['thought'] === true)) {
                            $nonThoughtPartsText[] = $part['text'];
                        }
                    }
                }

                if (!empty($nonThoughtPartsText)) {
                    // If there are non-thought parts, take the last one
                    // (assuming final output is the last non-thought part)
                    $finalTextOutput = end($nonThoughtPartsText);
                    error_log("promptGeminiAI: Using the last non-thought part. Found " . count($nonThoughtPartsText) . " non-thought parts.");
                } elseif (!empty($allPartsTexts)) {
                    // If all parts were thoughts (or no thought flag), take the text from the very last part as a fallback
                    $finalTextOutput = end($allPartsTexts);
                    error_log("promptGeminiAI: No explicit non-thought parts found. Using the very last part of the response.");
                }
            }
        }

        if ($finalTextOutput !== null) {
            // **String Cleaning - Remove code block markers and 'html' (if still needed)**
            // Be cautious with these general replacements if the AI is supposed to return actual code.
            $cleanedText = $finalTextOutput;
            
            // If the AI is wrapping its HTML output in markdown for code blocks.
            if (strpos(trim($cleanedText), '```html') === 0) {
                 $cleanedText = preg_replace('/^```html\s*/', '', $cleanedText, 1);
                 $cleanedText = preg_replace('/\s*```$/', '', $cleanedText, 1);
            } elseif (strpos(trim($cleanedText), '```') === 0) {
                 $cleanedText = preg_replace('/^```\s*/', '', $cleanedText, 1);
                 $cleanedText = preg_replace('/\s*```$/', '', $cleanedText, 1);
            }
            
            // Your original 'html' removal logic (if the word "html" itself needs to be removed, not the tags)
            // This seems unlikely to be what you want if the AI is generating HTML.
            // Commenting out unless you have a specific reason for this.
            /*
            $pos = strpos($cleanedText, "html");
            if ($pos !== false) {
                // This removes the literal string "html". If the AI generates "<html>...", this would break it.
                // $cleanedText = substr_replace($cleanedText, "", $pos, strlen("html"));
            }
            */
            $cleanedText = trim($cleanedText);

            echo $cleanedText; // **Important: Echo the cleaned response (no extra newline needed if source has it)**
            return;
        } else {
            error_log("promptGeminiAI: No usable text content found in API response parts. Raw: " . $response_body);
            echo "Error: Could not extract a usable response from the AI.\nRaw Response: " . htmlspecialchars($response_body) . "\n";
            return;
        }

    } else { // HTTP Code is not 200
        error_log("promptGeminiAI API Request Failed. HTTP Code: " . $httpCode . " | Response: " . $response_body);
        echo "Error: API Request Failed.\n";
        echo "HTTP Code: " . $httpCode . "\n";
        echo "API Response:\n" . htmlspecialchars($response_body) . "\n";
        return;
    }
}

// Check if the prompt is sent via POST request
if (isset($_POST['prompt'])) {
    $prompt = $_POST['prompt'];
    // Set header for the output if you intend it to be HTML
    // header('Content-Type: text/html; charset=utf-8'); // Uncomment if the output is HTML
    promptGeminiAI($prompt); // Call the function and the response will be echoed
} else {
    error_log("promptGeminiAI AJAX Call: No prompt received via POST.");
    echo "Error: No prompt received."; // Handle case where no prompt is sent
}
?>