<?php
// --- Error Reporting (Log for Production) ---
ini_set('display_errors', 0);
error_reporting(E_ALL);
ini_set('log_errors', 1);
// ---------------------------------------------------------------------

// --- Includes and Session Start ---
require_once "dbconn.php"; // Establishes $conn
session_start();

// --- Set Headers ---
header('Content-Type: application/json');
header('Cache-Control: no-cache, must-revalidate');
header('Expires: Mon, 26 Jul 1997 05:00:00 GMT');

// --- Authentication Check ---
if (!isset($_SESSION['user_name'])) {
    http_response_code(401); // Unauthorized
    echo json_encode(['success' => false, 'error' => 'Authentication required.']);
    exit;
}

// --- Validate Database Connection ---
if (!$conn || $conn->connect_error) {
     $db_error = $conn->connect_error ?? 'Unknown DB connection error';
     error_log("Database connection failed in ajax_create_ai_error_log.php: " . $db_error);
     http_response_code(500); // Internal Server Error
     echo json_encode(['success' => false, 'error' => 'Database connection failed.']);
     exit;
}

// --- Define Table Name ---
$tableName = 'tdu_ai_error_log';

// --- ****** START: Check if Table Exists and Create if Not ****** ---
try {
    // Correct way to check for table existence: Direct query with escaping
    $escapedTableName = mysqli_real_escape_string($conn, $tableName);
    $checkTableSql = "SHOW TABLES LIKE '" . $escapedTableName . "'"; // Construct the query string directly

    $checkTableResult = mysqli_query($conn, $checkTableSql);

    if ($checkTableResult === false) {
        // Handle query error specifically
        throw new Exception("Error executing SHOW TABLES query: " . mysqli_error($conn));
    }

    $tableExists = (mysqli_num_rows($checkTableResult) > 0);
    mysqli_free_result($checkTableResult); // Free the result set

    if (!$tableExists) {
        error_log("Table '$tableName' does not exist. Attempting to create...");

        // CREATE TABLE statement (keep this as is)
        $createTableSql = "
            CREATE TABLE {$tableName} (
                id INT AUTO_INCREMENT PRIMARY KEY,
                ai_ref INT NULL DEFAULT NULL,
                conversation_id VARCHAR(255) NOT NULL,
                reason TEXT NULL,
                type VARCHAR(50) NOT NULL,
                status VARCHAR(50) NOT NULL DEFAULT 'PENDING',
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                INDEX idx_conversation_id (conversation_id),
                INDEX idx_ai_ref (ai_ref),
                INDEX idx_status_type (status, type)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        ";

        if (mysqli_query($conn, $createTableSql)) {
            error_log("Table '$tableName' created successfully.");
        } else {
            throw new Exception("Error creating table '$tableName': " . mysqli_error($conn));
        }
    }
} catch (Exception $e) {
    error_log("Database setup error in ajax_create_ai_error_log.php: " . $e->getMessage());
    http_response_code(500);
    // Pass the actual MySQL error detail if available
    $details = ($e instanceof mysqli_sql_exception) ? $e->getMessage() : 'Specific MySQL error not available.';
     if (strpos($e->getMessage(), 'syntax') !== false) { // Check if it's likely a syntax error
          $details = $e->getMessage(); // Use the caught exception message directly
     }

    echo json_encode(['success' => false, 'error' => 'Database setup error occurred.', 'details' => $details]); // Send detailed error back
    mysqli_close($conn); // Close connection on error
    exit;
}
// --- ****** END: Table should now exist ****** ---


// --- Get POST data ---
$ai_ref_input = $_POST['ai_ref'] ?? null;
$conversation_id = trim($_POST['conversation_id'] ?? '');
$reason = trim($_POST['reason'] ?? '');
$type = trim($_POST['type'] ?? ''); // Should be 'MANUAL' from the report button
$status = trim($_POST['status'] ?? ''); // Should be 'PENDING' from the report button

// --- Validate Input ---
if (empty($conversation_id) || empty($reason) || empty($type) || empty($status)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Missing required report data (Conversation ID, Reason, Type, or Status).']);
    mysqli_close($conn); exit;
}

if ($type !== 'MANUAL' || $status !== 'PENDING') {
     http_response_code(400);
     error_log("Invalid type/status received for manual report: Type='{$type}', Status='{$status}'");
     echo json_encode(['success' => false, 'error' => 'Invalid report type or status.']);
     mysqli_close($conn); exit;
}

// Convert ai_ref_input to an integer ID or null
$ai_ref_id = null;
if ($ai_ref_input !== null && $ai_ref_input !== '' && is_numeric($ai_ref_input)) {
    $ai_ref_id = (int)$ai_ref_input;
    // Optional: Add a check here to see if this ai_ref_id actually exists in tdu_temp_ai_quotes
    // if you want strict referential integrity (though a foreign key constraint is better for that).
}

// --- Prepare SQL Statement ---
$insertSql = "INSERT INTO {$tableName} (ai_ref, conversation_id, reason, type, status, created_at)
              VALUES (?, ?, ?, ?, ?, NOW())";

$stmt = mysqli_prepare($conn, $insertSql);

if ($stmt) {
    // Bind parameters: 'i' for integer ai_ref_id (or null), 's' for strings
    mysqli_stmt_bind_param($stmt, "issss",
        $ai_ref_id,
        $conversation_id,
        $reason,
        $type,
        $status
    );

    // --- Execute the INSERT Statement ---
    if (mysqli_stmt_execute($stmt)) {
        $log_id = mysqli_insert_id($conn);
        echo json_encode(['success' => true, 'log_id' => $log_id, 'message' => 'Report submitted successfully.']);
    } else {
        error_log("Database Error (Execute Insert {$tableName}): " . mysqli_stmt_error($stmt) . " | Conv ID: " . $conversation_id);
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => 'Database error while saving the report.']);
    }
    mysqli_stmt_close($stmt);

} else {
    // Statement Preparation Failed
    error_log("Database Error (Prepare Insert {$tableName}): " . mysqli_error($conn) . " | Conv ID: " . $conversation_id);
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Database error preparing the report save operation.']);
}

// --- Clean Up Connection ---
mysqli_close($conn);
exit;

?>