<?php
/**
 * ajax_create_temp_quote.php
 *
 * Receives data via POST:
 * - templateref: Comma-separated string of template IDs or 'N/A'.
 * - country: Country name (typically from the first segment).
 * - ai_response: The full raw AI JSON response string.
 * - conversation_id: The original email conversation ID.
 *
 * Checks if the tdu_temp_ai_quotes table exists. If not, it creates it.
 * Fetches the original email sender based on conversation_id.
 * If the original email folder is 'Inbox', the sender's email is stored in the 'sender' column.
 * Inserts a new row into the tdu_temp_ai_quotes table including the sender's email (if applicable).
 */

// --- Error Reporting (Enable for Debugging, Disable/Log for Production) ---
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
// ---------------------------------------------------------------------

// --- Includes and Session Start ---
require_once "dbconn.php"; // Establishes $conn
session_start();

// --- Authentication Check ---
if (!isset($_SESSION['user_name'])) {
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    header('Expires: Mon, 26 Jul 1997 05:00:00 GMT');
    http_response_code(401); // Unauthorized
    echo json_encode(['success' => false, 'error' => 'Authentication required.']);
    exit;
}

// --- Set Headers ---
header('Content-Type: application/json');
header('Cache-Control: no-cache, must-revalidate');
header('Expires: Mon, 26 Jul 1997 05:00:00 GMT');

// --- Validate Database Connection ---
if (!$conn || $conn->connect_error) {
     $db_error = $conn->connect_error ?? 'Unknown DB connection error';
     error_log("Database connection failed in ajax_create_temp_quote.php: " . $db_error);
     http_response_code(500); // Internal Server Error
     echo json_encode(['success' => false, 'error' => 'Database connection failed. Please check server logs.']);
     exit;
}

// --- Get POST data ---
$templateRef = $_POST['templateref'] ?? null;
$country = $_POST['country'] ?? null;
$aiResponse = $_POST['ai_response'] ?? null;
$conversationId = $_POST['conversation_id'] ?? null;
$sender = $_POST['sender'] ?? null;

// --- Validate Input ---
if (empty($templateRef)) {
    http_response_code(400); // Bad Request
    echo json_encode(['success' => false, 'error' => 'Template reference string not provided.']);
    exit;
}
if ($country === null) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Country not provided.']);
    exit;
}
if (empty($aiResponse)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'AI response data not provided.']);
    exit;
}
if (empty($conversationId) && $conversationId !== '0') {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Conversation ID not provided.']);
    exit;
}

// --- Define Table Name ---
$tableName = 'tdu_temp_ai_quotes';
$escapedTableName = $conn->real_escape_string($tableName);

// --- Check if Table Exists and Create if Not ---
try {
    $checkTableSql = "SHOW TABLES LIKE '" . $escapedTableName . "'";
    $tableResult = $conn->query($checkTableSql);

    if ($tableResult === false) {
        throw new Exception("Error checking table existence: " . $conn->error);
    }

    if ($tableResult->num_rows == 0) {
        error_log("Table '$tableName' does not exist. Attempting to create...");

        $createTableSql = "
            CREATE TABLE {$escapedTableName} (
                id INT AUTO_INCREMENT PRIMARY KEY,
                templateref VARCHAR(1000) NULL,
                quoteref VARCHAR(255) NULL,
                country VARCHAR(255) NULL,
                ai_response LONGTEXT NULL,
                conversation_id VARCHAR(255) NOT NULL,
                created_at TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP,
                sender VARCHAR(255) NULL DEFAULT NULL      -- Renamed from 'email'
                -- ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
            );
        ";

        if ($conn->query($createTableSql) === TRUE) {
            error_log("Table '$tableName' created successfully.");
        } else {
            throw new Exception("Error creating table '$tableName': " . $conn->error);
        }
    }
    if ($tableResult instanceof mysqli_result) {
        $tableResult->free();
    }

} catch (Exception $e) {
    error_log("Database setup error in ajax_create_temp_quote.php: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Database setup error occurred.', 'details' => $e->getMessage()]);
    exit;
}
// --- Table should now exist ---


// --- Prepare for Database Insertion into tdu_temp_ai_quotes ---
// Updated SQL to include the 'sender' column (instead of 'email')
$sql = "INSERT INTO {$escapedTableName} (templateref, quoteref, country, ai_response, conversation_id, created_at, sender)
        VALUES (?, NULL, ?, ?, ?, NOW(), ?)"; // Note the last placeholder corresponds to the 'sender' column

$stmt = $conn->prepare($sql);

if ($stmt === false) {
    $prepareError = $conn->error;
    error_log("SQL Prepare Error (Insert Quote) in ajax_create_temp_quote.php: " . $prepareError);
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Database prepare statement failed for quote insertion.', 'details' => $prepareError]);
    exit;
}

// Bind parameters. 5 placeholders: templateRef, country, aiResponse, conversationId, senderValueToInsert
$bindResult = $stmt->bind_param("sssss", $templateRef, $country, $aiResponse, $conversationId, $sender);

if ($bindResult === false) {
    $bindError = $stmt->error;
    error_log("SQL bind_param Error (Insert Quote) in ajax_create_temp_quote.php: " . $bindError);
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Database bind parameters failed for quote insertion.', 'details' => $bindError]);
    $stmt->close();
    exit;
}

// --- Execute and Check Result ---
if ($stmt->execute()) {
    $newId = $conn->insert_id;
    // Updated JSON key to reflect the column name
    echo json_encode(['success' => true, 'temp_quote_id' => $newId, 'inserted_sender' => $sender]);
} else {
    $executeError = $stmt->error;
    error_log("SQL Execute Error (Insert Quote) in ajax_create_temp_quote.php: " . $executeError);
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Failed to save temporary quote to database.', 'details' => $executeError]);
}

// --- Cleanup ---
$stmt->close();
$conn->close();
exit;
?>