<?php
// ajax_fetch_confirmed_quotes.php
// AJAX worker to fetch a LIST of confirmed quotes by date range.

header('Content-Type: application/json');
include "dbconn.php"; // Include your standard database connection.

// --- Get Parameters ---
$start_date = $_GET['start_date'] ?? null;
$end_date = $_GET['end_date'] ?? null;

if (!$start_date || !$end_date) {
    echo json_encode(['success' => false, 'error' => 'Missing start_date or end_date parameter.']);
    exit;
}

if (!$conn) {
    echo json_encode(['success' => false, 'error' => 'Database connection failed.']);
    exit;
}

$response_data = ['success' => true, 'data' => []];

try {
    // --- Define Confirmed Stages ---
    $confirmed_stages = [
        'Accepted', 'PRE QA - pending', 'PRE QA - completed', 
        'Payment Received - Release Vouchers', 'Final QA', 'Delivered', 
        'On Ground', 'Accounts - Reconciliation', 'Completed (Accounts)'
    ];
    $escaped_stages = array_map([$conn, 'real_escape_string'], $confirmed_stages);
    $confirmed_stages_sql_in = "('" . implode("','", $escaped_stages) . "')";

    // --- Subquery to get the first, non-deleted version of a quote ---
    $unique_quotes_subquery = "(
        SELECT inner_vq.quoteid, inner_vq.created_at, inner_vq.deleted, inner_vq.quote_no,
        inner_vq.quotestage, inner_vq.contactid
        FROM vtiger_quotes inner_vq
        WHERE inner_vq.quoteid = (
            SELECT MIN(sub.quoteid) FROM vtiger_quotes AS sub WHERE sub.quote_no = inner_vq.quote_no AND sub.deleted = 0
        )
    ) vq";

    // --- Main Query to fetch a LIST of quotes ---
    $sql = "SELECT 
                vq.quote_no, 
                vq.created_at,
                o.organization_name,
                vqinfo.assigned_to_region
            FROM " . $unique_quotes_subquery . "
            LEFT JOIN vtiger_quotes_info vqinfo ON vq.quoteid = vqinfo.quoteid
            LEFT JOIN tdu_contacts c ON c.auto_id = vq.contactid
            LEFT JOIN tdu_organisation o ON o.organizationid = c.organizationid
            WHERE vq.deleted = 0 
            AND vq.quotestage IN " . $confirmed_stages_sql_in . "
            AND vq.created_at BETWEEN ? AND ?
            ORDER BY vq.created_at DESC";

    $stmt = $conn->prepare($sql);
    $start_datetime = $start_date . " 00:00:00";
    $end_datetime = $end_date . " 23:59:59";
    $stmt->bind_param("ss", $start_datetime, $end_datetime);
    
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result && $result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $response_data['data'][] = $row;
        }
    }
    
    $stmt->close();
    $conn->close();

} catch (Exception $e) {
    $response_data['success'] = false;
    $response_data['error'] = 'Database query failed: ' . $e->getMessage();
}

echo json_encode($response_data);