<?php
// ajax_get_quote_details_for_report.php
// New AJAX worker to fetch agent and priority details for a list of quote numbers.

header('Content-Type: application/json');
include "dbconn.php"; // Include your standard database connection.

// --- Get Parameters ---
$quote_numbers = $_GET['quote_no'] ?? [];

if (empty($quote_numbers) || !is_array($quote_numbers)) {
    echo json_encode(['success' => false, 'error' => 'Missing or invalid quote_no parameter array.']);
    exit;
}

if (!$conn) {
    echo json_encode(['success' => false, 'error' => 'Database connection failed.']);
    exit;
}

$response_data = ['success' => true, 'details' => []];

try {
    // --- Sanitize input quote numbers to prevent SQL injection ---
    $placeholders = implode(',', array_fill(0, count($quote_numbers), '?'));
    $types = str_repeat('s', count($quote_numbers));

    // --- Subquery to get the first, non-deleted version of a quote ---
    $unique_quotes_subquery = "(
        SELECT inner_vq.quoteid, inner_vq.quote_no
        FROM vtiger_quotes inner_vq
        WHERE inner_vq.quoteid = (
            SELECT MIN(sub.quoteid) FROM vtiger_quotes AS sub WHERE sub.quote_no = inner_vq.quote_no AND sub.deleted = 0
        )
    ) vq";

    // --- Main Query ---
    // Fetches the raw agent and priority data for the given quote numbers.
    $sql = "SELECT 
                vq.quote_no,
                COALESCE(vqinfo.assigned_to_sales_agent, '') AS internal_agent,
                COALESCE(vqinfo.assigned_to_external_sales_agent, '') AS external_agent,
                COALESCE(vqinfo.priority, 'N/A') AS priority
            FROM " . $unique_quotes_subquery . "
            LEFT JOIN vtiger_quotes_info vqinfo ON vq.quoteid = vqinfo.quoteid
            WHERE vq.quote_no IN ($placeholders)";

    $stmt = $conn->prepare($sql);
    $stmt->bind_param($types, ...$quote_numbers);
    
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result && $result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $agentString = '';
            $internal = trim($row['internal_agent']);
            $external = trim($row['external_agent']);

            // Construct the agent string based on what's available
            if (!empty($internal) && !empty($external)) {
                $agentString = "Internal: {$internal} | External: {$external}";
            } elseif (!empty($internal)) {
                $agentString = "Internal: {$internal}";
            } elseif (!empty($external)) {
                $agentString = "External: {$external}";
            } else {
                $agentString = 'Unassigned';
            }

            // Map the details to the quote number
            $response_data['details'][$row['quote_no']] = [
                'agentString' => $agentString,
                'priority'    => ($row['priority'] === '' || $row['priority'] === null) ? 'N/A' : $row['priority']
            ];
        }
    }
    
    $stmt->close();
    $conn->close();

} catch (Exception $e) {
    $response_data['success'] = false;
    $response_data['error'] = 'Database query failed: ' . $e->getMessage();
}

echo json_encode($response_data);