<?php
// ajax_qbo_bill_update.php
// MODIFIED: Now logs changes to the history table with robust validation.

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

header('Content-Type: application/json');

// --- Essential Includes ---
require_once __DIR__ . '/qbo_functions.php';
require_once __DIR__ . '/../vendor/autoload.php';
require_once __DIR__ . '/dbconn.php';
global $conn;

// MODIFIED: Include the new helper file for history logging.
require_once __DIR__ . '/qbo_reconciliation_helpers.php';

use QuickBooksOnline\API\Core\OAuth\OAuth2\OAuth2LoginHelper;

// --- Function to get a live, refreshed QBO Utility Library instance ---
function getQboUtils() {
    $qboBaseConfigFile = __DIR__ . '/config/qbo_config.php';
    $qboTokenStorageFile = __DIR__ . '/tokens/qbo_token.json';
    $qboBaseConfig = null;

    if (file_exists($qboBaseConfigFile)) { require $qboBaseConfigFile; }
    else { throw new Exception("QBO base config file not found."); }

    if ($qboBaseConfig) {
        $json = file_exists($qboTokenStorageFile) ? file_get_contents($qboTokenStorageFile) : null;
        $currentTokens = $json ? json_decode($json, true) : null;
        $qboConfigForLibrary = $qboBaseConfig;

        if ($currentTokens && !empty($currentTokens['refresh_token'])) {
            $qboConfigForLibrary['refreshTokenKey'] = $currentTokens['refresh_token'];
            try {
                $oauth2LoginHelper = new OAuth2LoginHelper($qboBaseConfig['ClientID'], $qboBaseConfig['ClientSecret']);
                $refreshedAccessTokenObj = $oauth2LoginHelper->refreshAccessTokenWithRefreshToken($qboConfigForLibrary['refreshTokenKey']);
                $newAccessToken = $refreshedAccessTokenObj->getAccessToken();
                $newRefreshToken = $refreshedAccessTokenObj->getRefreshToken();
                
                $newTokensArray = ['access_token' => $newAccessToken, 'refresh_token' => $newRefreshToken, 'last_updated' => date('Y-m-d H:i:s')];
                file_put_contents($qboTokenStorageFile, json_encode($newTokensArray, JSON_PRETTY_PRINT));

                $currentTokens['access_token'] = $newAccessToken;
            } catch (Exception $e) { 
                // Log but don't kill the script, old token might be valid.
                error_log("AJAX QBO Refresh Error: " . $e->getMessage()); 
            }
        }

        if ($currentTokens && !empty($currentTokens['access_token'])) {
            $qboConfigForLibrary['accessTokenKey'] = $currentTokens['access_token'];
            $qboConfigForLibrary['refreshTokenKey'] = $currentTokens['refresh_token'] ?? null;
            return new QBOUtilityLibrary($qboConfigForLibrary);
        }
    }
    throw new Exception("Could not get a valid QBO connection.");
}

// --- Main AJAX Logic ---
$response = ['success' => false, 'error' => 'An unknown error occurred.'];

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    $response['error'] = 'Invalid request method.';
    echo json_encode($response);
    exit;
}

try {
    // 1. Get data from the modal form
    $recId = $_POST['reconciliation_id'] ?? null;
    $itemIndex = $_POST['item_index'] ?? null; // Can be 0, so check with isset
    $billId = $_POST['bill_id'] ?? null;
    $syncToken = $_POST['sync_token'] ?? null;
    $lineId = $_POST['line_id'] ?? null;
    $newAmount = $_POST['bill_amount'] ?? null;
    $newDocNumber = $_POST['doc_number'] ?? null;
    $fileName = $_POST['file_name'] ?? null;
    
    // --- MODIFIED: More Robust and Specific Validation Block ---
    if (empty($recId) || !is_numeric($recId)) { throw new Exception("Invalid Reconciliation ID provided."); }
    if (!isset($itemIndex) || !is_numeric($itemIndex)) { throw new Exception("Invalid Item Index provided."); }
    if (empty($billId) || !is_numeric($billId)) { throw new Exception("Invalid Bill ID provided."); }
    if (!isset($syncToken) || $syncToken === '') { throw new Exception("Missing Sync Token for the bill."); } // Sync token can be "0" but not empty
    if (empty($lineId)) { throw new Exception("Invalid Line ID provided."); }
    if (!isset($newAmount) || !is_numeric($newAmount)) { throw new Exception("A valid numeric Bill Amount is required."); }
    if (empty($fileName)) { throw new Exception("Could not determine the context filename."); }
    
    // 2. Get a live QBO connection
    $qboUtil = getQboUtils();
    $dataService = $qboUtil->getDataService();

    // 3. Find the existing Bill in QBO
    $billToUpdate = $dataService->FindById('Bill', $billId);
    if (!$billToUpdate) {
        throw new Exception("Could not find Bill with ID {$billId} in QuickBooks.");
    }
    
    if ($billToUpdate->SyncToken != $syncToken) {
        throw new Exception("Sync token mismatch. The bill may have been updated by someone else. Please refresh the page and try again.");
    }

    // 4. Modify the Bill object after capturing the 'before' state
    $lineFound = false;
    $actualLines = is_array($billToUpdate->Line) ? $billToUpdate->Line : [$billToUpdate->Line];

    $detailsBefore = [
        'doc_number' => $billToUpdate->DocNumber ?? '',
        'amount' => null // Set in the loop below
    ];
    
    foreach ($actualLines as &$line) {
        if (isset($line->Id) && $line->Id == $lineId) {
            $lineFound = true;
            $detailsBefore['amount'] = $line->Amount ?? 'N/A'; // Capture original line amount

            if (isset($line->ItemBasedExpenseLineDetail)) {
                $line->ItemBasedExpenseLineDetail->UnitPrice = (float)$newAmount;
                $line->Amount = ($line->ItemBasedExpenseLineDetail->Qty ?? 1) * (float)$newAmount;
            } elseif(isset($line->AccountBasedExpenseLineDetail)) {
                 $line->Amount = (float)$newAmount;
            }
        }
    }
    unset($line); 
    
    if (!$lineFound) {
        throw new Exception("Could not find Line ID {$lineId} within Bill ID {$billId}.");
    }

    $billToUpdate->DocNumber = $newDocNumber;
    $billToUpdate->Line = $actualLines;
    
    // Recalculate total amount
    $totalAmt = 0;
    foreach($actualLines as $line) { $totalAmt += $line->Amount; }
    $billToUpdate->TotalAmt = $totalAmt;

    // 5. Send the update to QBO
    $updatedBillFromQBO = $qboUtil->updateBill($billToUpdate);
    
    // 6. UPDATE THE LOCAL DATABASE CACHE
    $stmt = mysqli_prepare($conn, "SELECT qbo_bills_json FROM tdu_qbo_reconciliation WHERE file_name = ?");
    mysqli_stmt_bind_param($stmt, "s", $fileName);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    $row = mysqli_fetch_assoc($result);

    if ($row) {
        $qboDataFromDb = json_decode($row['qbo_bills_json'], false);

        if (is_array($qboDataFromDb) && isset($qboDataFromDb[(int)$itemIndex])) {
            $qboDataFromDb[(int)$itemIndex]->bill_object = $updatedBillFromQBO;
            $qboDataFromDb[(int)$itemIndex]->matched_line_amount = (float)$newAmount;
            
            $newJsonToStore = json_encode($qboDataFromDb);

            $updateStmt = mysqli_prepare($conn, "UPDATE tdu_qbo_reconciliation SET qbo_bills_json = ? WHERE file_name = ?");
            mysqli_stmt_bind_param($updateStmt, "ss", $newJsonToStore, $fileName);
            mysqli_stmt_execute($updateStmt);
        }
    }

    // 7. LOG THE CHANGE TO THE HISTORY TABLE
    $detailsAfter = [
        'doc_number' => $newDocNumber,
        'amount' => (float)$newAmount
    ];
    log_reconciliation_change($conn, (int)$recId, (int)$itemIndex, 'Bill Update', $detailsBefore, $detailsAfter);

    // 8. Send success response
    $response['success'] = true;
    $response['message'] = 'Bill updated successfully.';

} catch (Exception $e) {
    $response['error'] = $e->getMessage();
    error_log("AJAX Update Error: " . $e->getMessage());
}

echo json_encode($response);