<?php
include "dbconn.php"; // Your database connection script
// include "dictionaries.php"; // If you need $availableSystemTypes here for response
require_once "ai_email_functions.php";

session_start();

// Security check: Ensure user is logged in and is an admin
if (!isset($_SESSION['user_name']) || $_SESSION['title'] != 'admin') {
    header('Content-Type: application/json');
    echo json_encode(['status' => 'error', 'message' => 'Unauthorized']);
    exit;
}

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['id'])) {
    // --- CAUTIOUS SETUP CALL ---
    // This will run every time this AJAX endpoint is hit.
    if (function_exists('setup_tdu_tai_assigned_role_table')) {
        if (!setup_tdu_tai_assigned_role_table($conn)) {
            // If table setup fails, we probably shouldn't proceed with DB operations
            echo json_encode(['status' => 'error', 'message' => 'Database table setup failed. Cannot update user types.']);
            mysqli_close($conn);
            exit;
        }
    } else {
        error_log("ajax_update_user_types.php: FATAL - setup_tdu_tai_assigned_role_table function is missing!");
        echo json_encode(['status' => 'error', 'message' => 'Critical setup function missing. Cannot update user types.']);
        mysqli_close($conn);
        exit;
    }
    // --- END CAUTIOUS SETUP CALL ---
    
    
    $userId = intval($_POST['id']); // Ensure it's an integer
    $tagIdsString = isset($_POST['tag_ids']) ? $_POST['tag_ids'] : ''; // Comma-separated string of tag_ids

    // Begin transaction for atomicity (all or nothing)
    mysqli_begin_transaction($conn);

    try {
        // 1. Delete existing type assignments for this user
        $sql_delete = "DELETE FROM tdu_tai_assigned_role WHERE user_id = ?";
        $stmt_delete = mysqli_prepare($conn, $sql_delete);
        if (!$stmt_delete) {
            throw new Exception("Prepare delete failed: " . mysqli_error($conn));
        }
        mysqli_stmt_bind_param($stmt_delete, "i", $userId);
        if (!mysqli_stmt_execute($stmt_delete)) {
            throw new Exception("Execute delete failed: " . mysqli_stmt_error($stmt_delete));
        }
        mysqli_stmt_close($stmt_delete);

        // 2. Insert new assignments if any are provided
        $newly_assigned_tag_ids = []; // To keep track for response if needed

        if (!empty($tagIdsString)) {
            $tagIdsArray = explode(',', $tagIdsString);
            $sql_insert = "INSERT INTO tdu_tai_assigned_role (user_id, tag_id) VALUES (?, ?)";
            $stmt_insert = mysqli_prepare($conn, $sql_insert);
            if (!$stmt_insert) {
                throw new Exception("Prepare insert failed: " . mysqli_error($conn));
            }

            foreach ($tagIdsArray as $tagIdStr) {
                if (!empty(trim($tagIdStr))) {
                    $currentTagId = intval(trim($tagIdStr)); // Ensure it's an integer
                    mysqli_stmt_bind_param($stmt_insert, "ii", $userId, $currentTagId);
                    if (!mysqli_stmt_execute($stmt_insert)) {
                        // Optional: could log which one failed, or just let the transaction rollback
                        // For simplicity, we'll let it rollback on any insert failure
                        throw new Exception("Execute insert failed for tag_id $currentTagId: " . mysqli_stmt_error($stmt_insert));
                    }
                    $newly_assigned_tag_ids[] = $currentTagId;
                }
            }
            mysqli_stmt_close($stmt_insert);
        }

        // If all operations were successful, commit the transaction
        mysqli_commit($conn);

        // OPTIONAL: Prepare names for the response to update UI accurately
        // This requires $availableSystemTypes to be available here
        // include_once "dictionaries.php"; // or however you access $availableSystemTypes
        // global $availableSystemTypes; // if it's global in dictionaries.php

        $assigned_names_for_response = [];
        // Make sure $availableSystemTypes is loaded if you use this part
        // For example, if you define it directly in this file or include dictionaries.php
        // For this example, let's assume $availableSystemTypes is accessible
        // If not, you'd need to load it or query a tags table
        $tempAvailableSystemTypes = [ /* copy your $availableSystemTypes definition here if not included */
            110 => 'Complaint', 92  => 'Confirmation', 108 => 'Following up Vouchers',
            109 => 'Payment on Arrival', 88  => 'Post Sales', 101 => 'Post Sales In Progress',
            89  => 'Pre Sales', 90  => 'Quotes', 105 => 'Refund', 91  => 'Requote',
            107 => 'Requote after Confirmation'
        ];
        foreach ($newly_assigned_tag_ids as $id) {
            if (isset($tempAvailableSystemTypes[$id])) {
                $assigned_names_for_response[] = $tempAvailableSystemTypes[$id];
            } else {
                $assigned_names_for_response[] = "TypeID-" . $id; // Fallback
            }
        }


        echo json_encode(['status' => 'success', 'message' => 'User types updated successfully.', 'assigned_names' => $assigned_names_for_response]);

    } catch (Exception $e) {
        // If any operation failed, roll back the transaction
        mysqli_rollback($conn);
        echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
    }

} else {
    echo json_encode(['status' => 'error', 'message' => 'Invalid request method or missing parameters.']);
}

mysqli_close($conn);
?>