<?php
// api_get_reconciliation_details.php
// This API endpoint returns data for the sales report drill-down.
// MODIFIED: Now includes QBO invoice breakdown data alongside Vtiger quote details.

header('Content-Type: application/json');

require_once __DIR__ . '/dbconn.php';
require_once __DIR__ . '/qbo_bootstrap.php';
global $conn, $qboUtil;

// --- Read GET parameters ---
$organisation_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$org_name = $_GET['name'] ?? null;
$date_from = $_GET['from'] ?? date('Y-m-d', strtotime('-90 days'));
$date_to = $_GET['to'] ?? date('Y-m-d');

$response = [
    'invoice_details' => [], // Added key for QBO invoice data
    'quote_details' => []
];

if ($organisation_id === 0 || !$org_name) {
    http_response_code(400);
    $response['error'] = 'Missing Organization ID or Name.';
    echo json_encode($response);
    exit;
}

try {
    // --- Step 1: Fetch QBO Invoice Details ---
    if ($qboUtil) {
        // Use the robust fuzzy search to find the customer in QBO
        $customer = $qboUtil->findEntityByFuzzyName('Customer', $org_name);
        if ($customer) {
            // This single, reliable function gets all the necessary numbers.
            $financialData = $qboUtil->getFinancialDetailData($customer->Id);
            $response['invoice_details'] = $financialData['invoice_details'] ?? [];
        }
    }

    // --- Step 2: Fetch Vtiger Quote Details ---
    $sql_params = [$organisation_id, $date_from, $date_to];
    $sql_types = "iss";
    $sql = "
        SELECT 
            q.quote_no, 
            q.quotestage, 
            (q.adults + q.children) AS total_pax,
            qcf.cf_1162 AS travel_date
        FROM vtiger_quotes AS q 
        JOIN vtiger_quotescf AS qcf ON q.quoteid = qcf.quoteid
        WHERE q.accountid = ? AND qcf.cf_1162 BETWEEN ? AND ?
        ORDER BY qcf.cf_1162 DESC
    ";
    
    $stmt = $conn->prepare($sql);
    if ($stmt) {
        $stmt->bind_param($sql_types, ...$sql_params);
        $stmt->execute();
        $response['quote_details'] = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmt->close();
    }

} catch (Exception $e) {
    http_response_code(500);
    $response['error'] = 'API Error: ' . $e->getMessage();
}

echo json_encode($response);