<?php
// api_get_supplier_details.php
// This API endpoint fetches and returns detailed bill and quote data for a single supplier.

header('Content-Type: application/json');

require_once __DIR__ . '/dbconn.php';
require_once __DIR__ . '/qbo_bootstrap.php';
global $conn, $qboUtil;

$vendor_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$vendor_name = $_GET['name'] ?? null;
$date_from = $_GET['from'] ?? date('Y-m-d', strtotime('-90 days'));
$date_to = $_GET['to'] ?? date('Y-m-d');

$response = [
    'bill_details' => [],
    'quote_details' => []
];

if ($vendor_id === 0 || !$vendor_name) {
    http_response_code(400);
    $response['error'] = 'Missing Vendor ID or Name.';
    echo json_encode($response);
    exit;
}

try {
    // --- Step 1: Fetch QBO Bill Details ---
    if ($qboUtil) {
        $qboVendor = $qboUtil->findEntityByFuzzyName('Vendor', $vendor_name);
        if ($qboVendor) {
            $financialData = $qboUtil->getVendorFinancialDetailData($qboVendor->Id);
            // We only need the bill_details part for the drill-down
            $response['bill_details'] = $financialData['bill_details'] ?? [];
        }
    }

    // --- Step 2: Fetch Vtiger Quote Details ---
    $sql_params = [$vendor_id, $date_from, $date_to];
    $sql_types = "iss";
    $sql = "
        SELECT DISTINCT
            q.quote_no,
            q.subject,
            qcf.cf_1162 AS travel_date,
            (q.adults + q.children) AS pax
        FROM tdu_vendors v
        JOIN tdu_products p ON v.vendorid = p.vendorid
        JOIN vtiger_inventoryproductrel i ON p.productid = i.productid
        JOIN vtiger_quotes q ON i.id = q.quoteid
        JOIN vtiger_quotescf qcf ON q.quoteid = qcf.quoteid
        WHERE v.vendorid = ? AND qcf.cf_1162 BETWEEN ? AND ?
        ORDER BY qcf.cf_1162 DESC
    ";

    $stmt = $conn->prepare($sql);
    if ($stmt) {
        $stmt->bind_param($sql_types, ...$sql_params);
        $stmt->execute();
        $response['quote_details'] = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmt->close();
    }

} catch (Exception $e) {
    http_response_code(500);
    $response['error'] = 'API Error: ' . $e->getMessage();
}

echo json_encode($response);