<?php
/**
 * Cron Runner Script: cronjob_ai_comparison_processing.php
 * 1. Checks vendors with contracts (using tdu_vendors_contracts schema), queues new PENDING jobs (linked to contract_auto_id) if needed.
 * 2. Finds and processes ONE pending AI comparison job per execution.
 * --- Updated with full error handling and contract_auto_id ---
 */

// --- Basic Setup ---
date_default_timezone_set('UTC'); // Set consistent timezone
require_once __DIR__ . "/dbconn.php";       // Provides $conn
require_once __DIR__ . "/../ai_keys.php";      // Provides apiURL_thinking() (needed by callGeminiAPI)
require_once __DIR__ . "/../ContractCompare/contract_comparison_functions.php"; // Load all helper functions
ini_set('error_log', '/contract_compare_error.log');
// --- Configuration ---
$max_jobs_to_process_per_run = 1; // Limit processing

// --- Script Start & DB Connection ---
//error_log("Cron runner started (Check/Queue & Process - Contract Link Update).");
if (!isset($conn) || !($conn instanceof mysqli) || $conn->connect_error) {
    error_log("CRITICAL: Cron runner DB connection failed. Error: " . ($conn->connect_error ?? 'Unknown error'));
    exit(1); // Exit if DB connection fails
}
if (!$conn->set_charset('utf8mb4')) { // Set encoding
     error_log("CRITICAL: Error loading character set utf8mb4: " . $conn->error);
     $conn->close(); exit(1); // Exit if charset fails
}
//error_log("Database connection successful.");

// --- Ensure Database Schema ---
// Creates jobs table and vendor contracts table (with correct schema) if they don't exist
if (!createDatabaseTablesIfNeeded($conn)) {
    error_log("CRITICAL: Failed to ensure required database tables exist. Cron cannot proceed.");
    $conn->close(); exit(1); // Exit if table creation/verification fails
}
//error_log("Required tables verified/created.");


// --- Step 1: Check Vendors & Queue New Jobs ---
//error_log("Step 1: Checking vendors and queuing new jobs...");
$vendors_checked = 0;
$new_jobs_created = 0;

// Query the vendor contracts table using correct column names
$sql_get_contract_vendors = "SELECT auto_id, vendorid, attachment_path FROM tdu_vendors_contracts"; // Use schema names
$vendor_result = $conn->query($sql_get_contract_vendors);

if ($vendor_result === false) {
    error_log("ERROR: Failed to query `tdu_vendors_contracts`: " . $conn->error . ". Skipping Step 1."); // Log error but continue to Step 2 maybe?
} elseif ($vendor_result->num_rows > 0) {
    // Loop through each vendor found in the contracts table
    while ($vendor_row = $vendor_result->fetch_assoc()) {
        $contractAutoId = (int)$vendor_row['auto_id']; // The contract's own ID
        $vendorId = (int)$vendor_row['vendorid'];       // The associated vendor ID
        $contractPath = trim($vendor_row['attachment_path']); // The file path
        $vendors_checked++;

        // Skip if essential IDs or path is invalid
        if ($vendorId <= 0 || $contractAutoId <= 0 || empty($contractPath)) {
            error_log("Skipping invalid entry from `tdu_vendors_contracts` (AutoID: {$contractAutoId}, VendorID: {$vendorId}).");
            continue; // Skip this row
        }

        // Check the status of the most recent job for the VENDOR to decide if *any* new job is needed
        $latest_vendor_status = getLatestVendorJobStatus($vendorId, $conn); // Use the function checking vendor status

        if ($latest_vendor_status === null) {
            error_log("Skipping vendor {$vendorId}: Error checking latest job status.");
            continue; // Skip this vendor if status check failed
        }

        // Decide whether to queue a NEW job for THIS SPECIFIC CONTRACT
        // Queue if vendor's latest job is DONE, ERROR, or PENDING (includes vendor never processed)
        if ($latest_vendor_status === 'PENDING' || $latest_vendor_status === 'DONE' || $latest_vendor_status === 'ERROR') {
             // **Add check:** See if a job for THIS specific contract_auto_id already exists and is PENDING/PROCESSING
             $existing_job_for_contract = getLatestJobByContractID($contractAutoId, $conn);
             if ($existing_job_for_contract && ($existing_job_for_contract['status'] === 'PENDING' || $existing_job_for_contract['status'] === 'PROCESSING' || $existing_job_for_contract['status'] === 'DONE' || $existing_job_for_contract['status'] === 'ERROR')) {
                  // Job for this specific contract already exists and is active, don't queue another
                  //error_log("Vendor {$vendorId}, Contract {$contractAutoId}: Job #{$existing_job_for_contract['id']} already exists with status {$existing_job_for_contract['status']}. No new job queued.");
             } else {
                  // Okay to queue: Vendor needs processing AND specific contract job is Missing
                  $log_reason = ($latest_vendor_status === 'PENDING') ? "No active job found for vendor" : "Vendor's latest job is {$latest_vendor_status}";
                  //error_log("Vendor {$vendorId}, Contract {$contractAutoId}: {$log_reason}. Queuing new job for contract path '{$contractPath}'.");

                  // Insert new job, passing contractAutoId now
                  if (insertNewJob($vendorId, $contractAutoId, $contractPath, $conn)) {
                      $new_jobs_created++;
                  } // Error logged within insertNewJob if it fails
             }
        } else {
            // Vendor's latest job is 'PROCESSING'
            //error_log("Vendor {$vendorId}: Latest job status is '{$latest_vendor_status}'. No new job queued for Contract {$contractAutoId}.");
        }
    }
    $vendor_result->free_result(); // Free result memory
} else {
    //error_log("No vendors found in `tdu_vendors_contracts` to check.");
}
//error_log("Step 1 finished. Checked: {$vendors_checked}, New Jobs Queued: {$new_jobs_created}.");


// --- Step 2: Find & Process ONE Pending Job ---
//error_log("Step 2: Attempting to process 1 pending job...");
$processed_count = 0;
$job_id_to_process = null; // ID of the job we will process

// Use a transaction to find and lock a job atomically
if (!$conn->begin_transaction()) {
    error_log("CRITICAL: Failed to begin transaction for job processing. Skipping Step 2.");
} else {
    $transaction_active = true; // Flag to track transaction state
    try {
        // Find the oldest PENDING job, lock it, skip if already locked by another process
        $sql_find_job = "SELECT id FROM tdu_ai_contract_comparison_jobs
                         WHERE status = 'PENDING'
                         ORDER BY created_at ASC, id ASC
                         LIMIT 1
                         FOR UPDATE SKIP LOCKED"; // Important for concurrency
        $result = $conn->query($sql_find_job);

        if ($result === false) { // Check for query error itself
             throw new Exception("Failed to query for pending jobs: " . $conn->error);
        }

        if ($result->num_rows > 0) {
            // Found and locked a job
            $job_row = $result->fetch_assoc();
            $job_id_to_process = (int)$job_row['id']; // Store the ID to process
            $result->free_result();
            //error_log("[Job {$job_id_to_process}] Locked PENDING job.");

            // Mark as PROCESSING using prepared statement
            $sql_update_status = "UPDATE tdu_ai_contract_comparison_jobs
                                  SET status = 'PROCESSING', processing_started_at = NOW()
                                  WHERE id = ? AND status = 'PENDING'"; // Verify status again
            $stmt_update = $conn->prepare($sql_update_status);
            if(!$stmt_update) {
                // Rollback immediately if prepare fails within transaction
                $conn->rollback();
                $transaction_active = false;
                throw new Exception("Prepare failed (mark processing): " . $conn->error);
            }

            $stmt_update->bind_param('i', $job_id_to_process); // Bind the job ID

            if ($stmt_update->execute()) {
                 if ($stmt_update->affected_rows === 1) {
                     // Successfully updated the status, commit the transaction
                     if (!$conn->commit()) {
                         $transaction_active = false; // Commit failed, transaction is implicitly rolled back or in error state
                         throw new Exception("Failed to commit transaction after marking PROCESSING.");
                     } else {
                         //error_log("[Job {$job_id_to_process}] Marked as PROCESSING. Transaction committed.");
                         $transaction_active = false; // Commit successful
                         // job_id_to_process is now ready to be processed outside transaction
                     }
                 } else {
                      // Should ideally not happen with FOR UPDATE lock, but handle defensively
                      throw new Exception("Failed to mark as PROCESSING (affected_rows!=1). Status might have changed. Rolling back.");
                 }
            } else {
                 // The UPDATE query execution failed
                 throw new Exception("Failed SQL to mark as PROCESSING: " . $stmt_update->error . ". Rolling back.");
            }
            $stmt_update->close(); // Close the statement

        } else {
            // No PENDING jobs found or all were locked
            //error_log("No unlocked PENDING jobs found to process in this run.");
            $conn->rollback(); // Release locks if any were held implicitly
            $transaction_active = false;
            $job_id_to_process = null; // Ensure no job ID is set
        }
    } catch (Exception $e) {
        error_log("[Job " . ($job_id_to_process ?? 'N/A') . "] Exception during job lock/update: " . $e->getMessage());
        if ($transaction_active) { $conn->rollback(); } // Rollback explicitly if transaction started and exception occurred
        $transaction_active = false;
        $job_id_to_process = null; // Do not proceed
    } finally {
        // Final safety check for rollback if transaction somehow still active (e.g., uncaught throwable)
        if ($transaction_active) {
             error_log("Transaction was still active in finally block, rolling back.");
             $conn->rollback();
        }
    }
} // End transaction block

// Process the job if one was successfully locked and marked
if ($job_id_to_process !== null) {
    //error_log("[Job {$job_id_to_process}] Calling main processing function...");
    try {
        // processAiComparisonJob handles the core logic and updates status to DONE/ERROR
        processAiComparisonJob($job_id_to_process, $conn);
        // Log completion, final status is set within processAiComparisonJob->updateJobStatus
        //error_log("[Job {$job_id_to_process}] Processing function call completed (check final status).");
        $processed_count++; // Increment count as processing was attempted

    } catch (Throwable $t) { // Catch unexpected fatal errors during processing
        error_log("[Job {$job_id_to_process}] CRITICAL UNCAUGHT ERROR during processing: " . $t->getMessage() . "\nTrace: " . $t->getTraceAsString());
        // Attempt to mark the job as ERROR
        updateJobStatus($job_id_to_process, 'ERROR', $conn, null, "Fatal error during processing: " . substr($t->getMessage(), 0, 500));
        $processed_count++; // Count it as processed (attempted)
    }
} else {
     //error_log("No job was processed in this run (either none pending or lock failed).");
}
//error_log("Step 2 finished. Jobs processed in this run: {$processed_count}.");


// --- Clean Up ---
if ($conn && $conn instanceof mysqli && !$conn->connect_error) {
    $conn->close();
    //error_log("Database connection closed.");
}
//error_log("Cron runner finished successfully.");
exit(0); // Exit indicating success
?>