<?php
// --- Cron Job: Process Unprocessed Emails for AI Quotes ---

// Error logging setup
ini_set('display_errors', 0); // Do not display errors to screen
ini_set('log_errors', 1);     // Log errors to a file
$errorLogPath = __DIR__ . '/cron_ai_quote_errors.log'; // Path to error log file
ini_set('error_log', $errorLogPath);
error_reporting(E_ALL);      // Report all PHP errors
set_time_limit(0);           // No time limit for script execution
ignore_user_abort(true);     // Continue script even if client disconnects

// --- Includes ---
$baseDir = __DIR__ . "/../"; // Assuming this cron script is in a subdirectory like 'cron'
require_once $baseDir . "dbconn.php"; // Database connection
require_once $baseDir . "ai_keys.php";    // For apiURL_Flash()
require_once $baseDir . "file_extract_functions.php"; // For text extraction from attachments
require_once $baseDir . "ai_email_functions.php"; // For all AI logic

// --- Configuration ---
define('BATCH_SIZE', 1); // Number of emails to process per run in batch mode
if (!defined('VTIGER_SUPPORT_STATUS_AI_PROCESSED')) define('VTIGER_SUPPORT_STATUS_AI_PROCESSED', 'AI Processed');
if (!defined('VTIGER_SUPPORT_STATUS_SPAM')) define('VTIGER_SUPPORT_STATUS_SPAM', 'SPAM');
$tempAiQuotesTableName = 'tdu_temp_ai_quotes';   // Name of the temporary table for AI quote data
$excludedSenderDomain = '@turtledownunder.com.au'; // Sender domain to handle specially
$skippedAlreadyActionedCount = 0;                  // Counter for skipped emails

// --- Modular Feature Configuration ---
$enableSpamDetection = true;        // Enable/disable spam detection module
$enableQuoteGeneration = true;      // Enable/disable core AI for segments, travel dates etc. (Now an umbrella for Org/Contact too)
$enableAiTagging = true;            // Enable/disable AI tagging module
$enableOrgContactManagement = true; // Enable/disable Organization/Contact extraction & creation

// --- Tagging Specific Configuration (if tagging is enabled) ---
$aiTaggerUserId = 0;      // User ID for vtiger_freetagged_objects.tagger_id (0 for 'system')
$tagModule = 'Dashboard'; // Module name for vtiger_freetagged_objects.module

// --- Validate DB Connection ---
if (!$conn || $conn->connect_error) {
    error_log("CRON JOB: Database connection failed: " . ($conn->connect_error ?? 'Unknown DB error') . ". Check dbconn.php.");
    exit(1); // Exit if DB connection fails
}
//error_log("CRON JOB: Starting processing of emails at " . date('Y-m-d H:i:s'));
//error_log("CRON JOB CONFIG: Spam Detection " . ($enableSpamDetection ? "ENABLED" : "DISABLED"));
//error_log("CRON JOB CONFIG: Quote Generation Umbrella " . ($enableQuoteGeneration ? "ENABLED" : "DISABLED"));
//error_log("CRON JOB CONFIG: AI Tagging " . ($enableAiTagging ? "ENABLED (TaggerID: {$aiTaggerUserId}, Module: {$tagModule})" : "DISABLED"));
//error_log("CRON JOB CONFIG: Org/Contact Management " . ($enableOrgContactManagement ? "ENABLED (under Quote Gen umbrella)" : "DISABLED"));


// --- Ensure Temp Table and Columns Exist (Only if Quote Generation related features are enabled) ---
if ($enableQuoteGeneration) { // Schema for tdu_temp_ai_quotes is mainly for quote gen outputs
    //error_log("CRON JOB: Performing schema check for table '{$tempAiQuotesTableName}' (Quote Generation is ENABLED).");
    $escapedTableName = $conn->real_escape_string($tempAiQuotesTableName);
    try {
        $checkTableSql = "SHOW TABLES LIKE '" . $escapedTableName . "'";
        $tableResult = $conn->query($checkTableSql);

        if (!$tableResult) {
            throw new Exception("Error checking if table '$escapedTableName' exists: " . $conn->error);
        }

        if ($tableResult->num_rows == 0) {
           // error_log("CRON JOB: Table '$escapedTableName' does not exist. Attempting to create it.");
            $createTableSql = "CREATE TABLE {$escapedTableName} (
                id INT AUTO_INCREMENT PRIMARY KEY,
                templateref VARCHAR(1000) NULL,
                quoteref TEXT NULL,
                country VARCHAR(255) NULL,
                travel_date DATE NULL DEFAULT NULL,
                ai_response LONGTEXT NULL,
                conversation_id VARCHAR(255) NOT NULL,
                organizationid INT NULL DEFAULT NULL,
                contactid INT NULL DEFAULT NULL,
                created_at TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP,
                sender VARCHAR(255) NULL DEFAULT NULL,
                UNIQUE KEY (conversation_id)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;";
            if (!$conn->query($createTableSql)) {
                throw new Exception("Error creating table '$escapedTableName': " . $conn->error);
            }
            //error_log("CRON JOB: Table '$escapedTableName' created successfully.");
        } else {
            //error_log("CRON JOB: Table '$escapedTableName' exists.");
            // Check if quoteref needs to be altered to TEXT
            $checkQuoterefColSql = "SHOW COLUMNS FROM {$escapedTableName} LIKE 'quoteref'";
            $quoterefColResult = $conn->query($checkQuoterefColSql);
            if ($quoterefColResult && $quoterefRow = $quoterefColResult->fetch_assoc()) {
                if (strtolower(strtok($quoterefRow['Type'], '(')) !== 'text' && strtolower(strtok($quoterefRow['Type'], '(')) !== 'longtext' && strtolower(strtok($quoterefRow['Type'], '(')) !== 'mediumtext') {
                    //error_log("CRON JOB: Column 'quoteref' in '{$escapedTableName}' is not TEXT type. Attempting to alter.");
                    $alterQuoterefSql = "ALTER TABLE {$escapedTableName} MODIFY COLUMN `quoteref` TEXT NULL";
                    if (!$conn->query($alterQuoterefSql)) {
                        error_log("CRON JOB: Error altering column 'quoteref' to TEXT in '{$escapedTableName}': " . $conn->error);
                    } else {
                        error_log("CRON JOB: Column 'quoteref' altered to TEXT successfully in '{$escapedTableName}'.");
                    }
                }
            }
            if ($quoterefColResult instanceof mysqli_result) $quoterefColResult->free();
        }
        if ($tableResult instanceof mysqli_result) $tableResult->free();

        $colsToCheck = [
            'organizationid' => "INT NULL DEFAULT NULL AFTER conversation_id",
            'contactid' => "INT NULL DEFAULT NULL AFTER organizationid",
            'sender' => "VARCHAR(255) NULL DEFAULT NULL AFTER created_at",
            'travel_date' => "DATE NULL DEFAULT NULL AFTER country"
        ];

        foreach ($colsToCheck as $colName => $colDef) {
            $checkColSql = "SHOW COLUMNS FROM {$escapedTableName} LIKE '{$conn->real_escape_string($colName)}'";
            $colResult = $conn->query($checkColSql);
            if (!$colResult) {
                 error_log("CRON JOB: Error checking for column '{$colName}' in '{$escapedTableName}': " . $conn->error);
                 continue;
            }
            if ($colResult->num_rows == 0) {
                error_log("CRON JOB: Column '{$colName}' missing in '{$escapedTableName}'. Attempting to add.");
                $addColSql = "ALTER TABLE {$escapedTableName} ADD COLUMN `{$conn->real_escape_string($colName)}` {$colDef}";
                if (!$conn->query($addColSql)) {
                    error_log("CRON JOB: Error adding column '{$colName}' to '{$escapedTableName}': " . $conn->error . ". SQL: " . $addColSql);
                } else {
                    error_log("CRON JOB: Column '{$colName}' added successfully to '{$escapedTableName}'.");
                }
            }
            if ($colResult instanceof mysqli_result) $colResult->free();
        }
        //error_log("CRON JOB: Schema check for table '{$escapedTableName}' completed.");

    } catch (Exception $e) {
        error_log("CRON JOB: CRITICAL DATABASE SETUP ERROR for '{$escapedTableName}': " . $e->getMessage());
    }
} else {
    //error_log("CRON JOB: Schema check for table '{$tempAiQuotesTableName}' SKIPPED (Quote Generation is DISABLED).");
}


// --- Determine Mode: Single ID Test (from CLI arg) or Batch ---
$emailsToProcess = [];
$specificConversationId = null;

if (isset($argv[1]) && !empty(trim($argv[1]))) {
    // CLI mode: process a single conversation ID passed as an argument
    $specificConversationId = trim($argv[1]);
    //error_log("CRON JOB (CLI TEST MODE): Processing specific Conversation ID: " . $specificConversationId);

    $detailsSql = "SELECT sender, mailbox FROM tdu_emails WHERE conversation_id = ? ORDER BY received_datetime DESC LIMIT 1";
    $stmtDetails = $conn->prepare($detailsSql);
    $originalSender = 'N/A';
    $mailboxForThisEmail = null;
    if ($stmtDetails) {
        $stmtDetails->bind_param("s", $specificConversationId);
        if ($stmtDetails->execute()) {
            $resultDetails = $stmtDetails->get_result();
            if ($rowDetails = $resultDetails->fetch_assoc()) {
                $originalSender = $rowDetails['sender'];
                $mailboxForThisEmail = $rowDetails['mailbox'];
            }
            if ($resultDetails instanceof mysqli_result) $resultDetails->free();
        } else {
            error_log("CRON JOB (CLI TEST MODE): Failed execute details fetch for {$specificConversationId}: " . $stmtDetails->error);
        }
        $stmtDetails->close();
    } else {
        error_log("CRON JOB (CLI TEST MODE): Failed prepare details fetch: " . $conn->error);
    }

    $emailsToProcess[] = ['conversation_id' => $specificConversationId, 'original_sender' => $originalSender, 'mailbox' => $mailboxForThisEmail];
    //error_log("CRON JOB: TEST MODE - ID: {$specificConversationId}, Sender: {$originalSender}, Mailbox: " . ($mailboxForThisEmail ?? 'N/A'));

} else {
    // Batch mode: fetch a batch of unprocessed emails
    //error_log("CRON JOB: BATCH MODE - Fetching unprocessed emails.");
    $sqlFetchUnprocessed = "
        SELECT e.conversation_id, e.sender as original_sender, e.mailbox, e.received_datetime
        FROM tdu_emails e
        LEFT JOIN vtiger_support s ON e.conversation_id = s.ticketid
        WHERE
            (s.ticketid IS NULL OR (s.ai_actioned IS NULL OR s.ai_actioned = 0))
            AND (s.ticketid IS NULL OR (s.status IS NULL OR s.status NOT IN ('Closed', ?)))
            AND e.folder != 'Sent Items'
            AND DATE(e.received_datetime) = CURDATE()
            AND (e.mailbox = 'sales@turtledownunder.com.au' or e.mailbox = 'groupsales@turtledownunder.com.au' or e.mailbox = 'asia@turtledownunder.com.au')
        GROUP BY e.conversation_id, e.sender, e.mailbox
        ORDER BY e.received_datetime ASC
        LIMIT " . BATCH_SIZE;

    $stmtFetch = $conn->prepare($sqlFetchUnprocessed);
    if ($stmtFetch === false) {
        error_log("CRON JOB: SQL Prepare Error (Fetch Batch): " . $conn->error . ". SQL: " . $sqlFetchUnprocessed);
        if ($conn) $conn->close();
        exit(1);
    }

    $spamStatusConstant = VTIGER_SUPPORT_STATUS_SPAM;
    $stmtFetch->bind_param("s", $spamStatusConstant);
    if (!$stmtFetch->execute()) {
        error_log("CRON JOB: SQL Execute Error (Fetch Batch): " . $stmtFetch->error);
        $stmtFetch->close();
        if ($conn) $conn->close();
        exit(1);
    }

    $resultUnprocessed = $stmtFetch->get_result();
    while ($row = $resultUnprocessed->fetch_assoc()) {
        $emailsToProcess[] = $row;
    }
    if ($resultUnprocessed instanceof mysqli_result) $resultUnprocessed->free();
    $stmtFetch->close();
}

if (empty($emailsToProcess)) {
    //error_log("CRON JOB: No emails found to process.");
    if ($conn) $conn->close();
    exit(0);
}
//error_log("CRON JOB: Found " . count($emailsToProcess) . " email(s) to process.");

// --- Process Each Email ---
$mainProcessedCount = 0;            // Count for actual AI segment/template processing
$taggingProcessedCount = 0;
$orgContactProcessedCount = 0;      // Count for Org/Contact processing attempts
$skippedForExistingQuotesCount = 0; // Count for emails where segment/template was skipped due to existing quotes
$errorCount = 0;
$spamCount = 0;
$spamErrorCount = 0;

foreach ($emailsToProcess as $emailData) {
    $conversationId = $emailData['conversation_id'];
    $originalSenderForThisEmail = $emailData['original_sender'];
    $mailboxForThisEmail = $emailData['mailbox'];

    //error_log("CRON JOB: ----- Processing Conv ID: {$conversationId}, Original Sender: {$originalSenderForThisEmail}, Mailbox: " . ($mailboxForThisEmail ?? 'N/A') . " -----");

    // Check if already marked as ai_actioned=1
    $skipThisEmail = false;
    $sqlCheckIfActioned = "SELECT ticketid FROM vtiger_support WHERE ticketid = ? AND ai_actioned = 1 LIMIT 1";
    $stmtCheckActioned = $conn->prepare($sqlCheckIfActioned);
    if ($stmtCheckActioned) {
        $stmtCheckActioned->bind_param("s", $conversationId);
        if ($stmtCheckActioned->execute()) {
            $resultCheckActioned = $stmtCheckActioned->get_result();
            if ($resultCheckActioned->num_rows > 0) {
                //error_log("CRON JOB: Conv ID {$conversationId} is already marked as ai_actioned=1 in vtiger_support. Skipping further processing stages.");
                $skipThisEmail = true;
                $skippedAlreadyActionedCount++;
            }
            if ($resultCheckActioned instanceof mysqli_result) $resultCheckActioned->free();
        } else { error_log("CRON JOB: WARNING - Failed to EXECUTE check if already actioned for Conv ID {$conversationId}: " . $stmtCheckActioned->error); }
        $stmtCheckActioned->close();
    } else { error_log("CRON JOB: WARNING - Failed to PREPARE check if already actioned for Conv ID {$conversationId}: " . $conn->error); }

    if ($skipThisEmail) {
        continue; // Skip to the next email
    }

    // Pre-set ai_actioned=1 for the conversation in vtiger_support
    //error_log("CRON JOB: Attempting to pre-set ai_actioned=1 for Conv ID: {$conversationId}");
    $sqlPreSetAiActioned = "INSERT INTO vtiger_support (ticketid, ai_actioned) VALUES (?, 1) ON DUPLICATE KEY UPDATE ai_actioned = 1";
    $stmtPreSet = $conn->prepare($sqlPreSetAiActioned);
    if ($stmtPreSet) {
        $stmtPreSet->bind_param("s", $conversationId);
        if ($stmtPreSet->execute()) {
             //error_log("CRON JOB: Successfully pre-set/updated ai_actioned=1 for Conv ID {$conversationId}. Affected rows: ".$stmtPreSet->affected_rows);
        } else {
            error_log("CRON JOB: CRITICAL - Failed to EXECUTE pre-set of ai_actioned=1 for Conv ID {$conversationId}: " . $stmtPreSet->error . ". Skipping this email.");
            $stmtPreSet->close(); continue;
        }
        $stmtPreSet->close();
    } else {
        //error_log("CRON JOB: CRITICAL - Failed to PREPARE pre-set of ai_actioned=1 for Conv ID {$conversationId}: " . $conn->error . ". Skipping this email.");
        continue;
    }

    // Spam Detection Module
    if ($enableSpamDetection) {
        //error_log("CRON JOB: Spam Detection ENABLED for Conv ID {$conversationId}.");
        $spamClassification = 'error'; // Default to error
        if (function_exists('classifyAsSpam')) {
            $spamClassification = classifyAsSpam($conversationId, $conn);
        } else {
            error_log("CRON JOB: classifyAsSpam function missing for Conv ID {$conversationId}. Assuming not spam.");
            $spamClassification = 'no';
        }

        if ($spamClassification === 'yes') {
            //error_log("CRON JOB: Conv ID {$conversationId} classified as SPAM.");
            $spamCount++;
            // Update vtiger_support status to SPAM
            $sqlSetSpamStatus = "INSERT INTO vtiger_support (ticketid, status, ai_actioned) VALUES (?, ?, 1) ON DUPLICATE KEY UPDATE status = VALUES(status), ai_actioned = VALUES(ai_actioned)";
            $stmtSetSpam = $conn->prepare($sqlSetSpamStatus);
            if ($stmtSetSpam) {
                $statusSpam = VTIGER_SUPPORT_STATUS_SPAM;
                $stmtSetSpam->bind_param("ss", $conversationId, $statusSpam);
                if ($stmtSetSpam->execute()) { error_log("CRON JOB: Ensured vtiger status is SPAM & ai_actioned=1 for Ticket ID {$conversationId}. Affected rows: ".$stmtSetSpam->affected_rows);
                } else { error_log("CRON JOB: ERROR setting SPAM status (exec) for Conv ID {$conversationId}: " . $stmtSetSpam->error); }
                $stmtSetSpam->close();
            } else { error_log("CRON JOB: ERROR setting SPAM status (prep) for Conv ID {$conversationId}: " . $conn->error); }
            continue; // Skip to next email if it's SPAM
        } elseif ($spamClassification === 'error') {
            error_log("CRON JOB: Error during spam classification for Conv ID {$conversationId}. Skipping.");
            $spamErrorCount++; continue; // Skip on spam check error
        }
        //error_log("CRON JOB: Conv ID {$conversationId} is NOT spam.");
    } else {
        //error_log("CRON JOB: Spam Detection DISABLED for Conv ID {$conversationId}. Proceeding.");
    }

    // Determine sender email, excluding internal domain if possible
    $senderEmailForDb = $originalSenderForThisEmail ?? null;
    if ($senderEmailForDb !== null && strpos(strtolower($senderEmailForDb), strtolower($excludedSenderDomain)) !== false) {
        //error_log("CRON JOB: Initial sender '{$senderEmailForDb}' for Conv ID {$conversationId} is from excluded domain '{$excludedSenderDomain}'. Searching alternative...");
        $foundAlternativeSender = false;
        $sqlAltSender = "SELECT sender FROM tdu_emails WHERE conversation_id = ? AND (sender IS NULL OR sender = '' OR LOWER(sender) NOT LIKE ?) ORDER BY received_datetime DESC LIMIT 1";
        $stmtAltSender = $conn->prepare($sqlAltSender);
        if ($stmtAltSender) {
            $excludedPattern = '%' . strtolower($excludedSenderDomain);
            $stmtAltSender->bind_param("ss", $conversationId, $excludedPattern);
            if ($stmtAltSender->execute()) {
                $resultAltSender = $stmtAltSender->get_result();
                if ($rowAlt = $resultAltSender->fetch_assoc()) {
                    if (!empty($rowAlt['sender'])) { $senderEmailForDb = $rowAlt['sender']; $foundAlternativeSender = true; error_log("CRON JOB: Found alternative sender '{$senderEmailForDb}' for Conv ID {$conversationId}."); }
                }
                if (!$foundAlternativeSender) { $senderEmailForDb = null; error_log("CRON JOB: No suitable alternative sender found for Conv ID {$conversationId}. Sender set to NULL."); }
                if ($resultAltSender instanceof mysqli_result) $resultAltSender->free();
            } else { error_log("CRON JOB: Exec error finding alt sender for Conv ID {$conversationId}: " . $stmtAltSender->error); }
            $stmtAltSender->close();
        } else { error_log("CRON JOB: Prep error finding alt sender for Conv ID {$conversationId}: " . $conn->error); }
    }
    error_log("CRON JOB: Final Sender Email for DB operations for Conv ID {$conversationId}: " . ($senderEmailForDb ?? 'NULL'));

    // Quote Pre-detection Step
    $quotesAlreadyExist = false;
    $detectedQuoteNumbersJson = '[]';
    if (function_exists('extractQuoteNumbersFromConversation')) {
        $detectedQuoteNumbersJson = extractQuoteNumbersFromConversation($conversationId, $conn);
        $decodedQuotes = json_decode($detectedQuoteNumbersJson, true);
        if (is_array($decodedQuotes) && !empty($decodedQuotes)) {
            $quotesAlreadyExist = true;
            // Log message changed to reflect that Org/Contact might still run
            //error_log("CRON JOB: Conv ID {$conversationId} - Pre-existing quotes detected: " . $detectedQuoteNumbersJson . ". Core AI Segment Generation will be SKIPPED. Org/Contact may still run.");
        } else {
            error_log("CRON JOB: Conv ID {$conversationId} - No pre-existing quotes detected. Proceeding with potential Core AI Segment Generation and Org/Contact.");
        }
    } else {
        error_log("CRON JOB: Conv ID {$conversationId} - Function 'extractQuoteNumbersFromConversation' is missing. Assuming no pre-existing quotes.");
    }

    // --- MODIFICATION: Initialize Org/Contact variables before the main AI block ---
    $extractedOrgId = null;
    $extractedContactId = null;
    $aiContactOrgData = null; // Store AI extracted contact/org data

    // --- MODIFICATION: Organization & Contact Management (runs if enabled, before segment/quote specific logic) ---
    if ($enableQuoteGeneration && $enableOrgContactManagement) {
        error_log("CRON JOB: Org/Contact Management ENABLED for Conv ID {$conversationId}. Attempting to process.");
        try {
            if (function_exists('extractContactAndOrganizationInfo')) {
                $contactOrgDetails = extractContactAndOrganizationInfo($conversationId, $conn);
                $aiContactOrgData = $contactOrgDetails['data'] ?? null;
                $extractedOrgNameFromAI = $aiContactOrgData['organization_name'] ?? null;

                if ($contactOrgDetails['success'] && $extractedOrgNameFromAI && strtolower(trim($extractedOrgNameFromAI)) !== 'n/a') {
                    error_log("CRON JOB: Conv ID {$conversationId} - Initial AI Extracted Org Name: '{$extractedOrgNameFromAI}'");
                    if (function_exists('find_best_match_org_via_ai_disambiguation')) {
                        $extractedOrgId = find_best_match_org_via_ai_disambiguation($extractedOrgNameFromAI, $aiContactOrgData, $conn);
                        if ($extractedOrgId) { error_log("CRON JOB: Conv ID {$conversationId} - AI Disambiguation Matched Org ID: {$extractedOrgId}");
                        } else { error_log("CRON JOB: Conv ID {$conversationId} - AI Disambiguation no match for '{$extractedOrgNameFromAI}'.");}
                    } else { error_log("CRON JOB: Missing func 'find_best_match_org_via_ai_disambiguation'."); }
                } elseif (!$contactOrgDetails['success']) { error_log("CRON JOB: AI contact/org extraction failed: " . ($contactOrgDetails['error'] ?? 'Unknown'));}
            } else { error_log("CRON JOB: Missing func 'extractContactAndOrganizationInfo'.");}

            $aiOrgNameValid = ($aiContactOrgData && !empty($aiContactOrgData['organization_name']) && strtolower(trim($aiContactOrgData['organization_name'])) !== 'n/a');
            if (!$extractedOrgId && $aiOrgNameValid) {
                if (function_exists('create_new_organization')) {
                    error_log("CRON JOB: Conv ID {$conversationId} - Org '{$aiContactOrgData['organization_name']}' not matched. Attempting creation.");
                    $newOrgId = create_new_organization($conn, $aiContactOrgData);
                    if ($newOrgId) { $extractedOrgId = $newOrgId; error_log("CRON JOB: Created new Org ID: {$extractedOrgId}");
                    } else { error_log("CRON JOB: Failed to create Org '{$aiContactOrgData['organization_name']}'.");}
                } else { error_log("CRON JOB: Missing func 'create_new_organization'.");}
            }

            // --- FULLY EXPANDED: Contact matching/creation logic from original script ---
            $aiContactNameValid = ($aiContactOrgData && !empty($aiContactOrgData['contact_name']) && strtolower(trim($aiContactOrgData['contact_name'])) !== 'n/a');
            $aiContactEmailValid = ($aiContactOrgData && !empty($aiContactOrgData['contact_email']) && filter_var($aiContactOrgData['contact_email'], FILTER_VALIDATE_EMAIL));
            $canCreateOrMatchContact = $aiContactNameValid || $aiContactEmailValid;

            if ($extractedOrgId && $canCreateOrMatchContact) {
                error_log("CRON JOB: Conv ID {$conversationId} - Org ID {$extractedOrgId}. Attempting contact match/creation.");
                $contactFoundInDb = false;
                $sqlFindContact = "SELECT auto_id FROM tdu_contacts WHERE organizationid = ?";
                $params = [$extractedOrgId]; $types = "i"; $conditions = [];

                if ($aiContactEmailValid) { $conditions[] = "email = ?"; $params[] = $aiContactOrgData['contact_email']; $types .= "s"; }
                if ($aiContactNameValid) {
                    $nameCondition = "LOWER(REPLACE(name,'.','')) = LOWER(REPLACE(?,'.',''))";
                    if(empty($conditions)) { // If email was not valid, $conditions is empty
                        $conditions[] = $nameCondition; // $conditions = [$nameCondition]
                        $params[] = $aiContactOrgData['contact_name']; $types .= "s";
                    } else { // Email was valid, $conditions = ["email = ?"]
                        $conditions[0] = "(" . $conditions[0] . " OR " . $nameCondition . ")"; // $conditions[0] becomes "(email = ? OR name_condition)"
                        // $params already has orgId and email. Add name parameter.
                        $params[] = $aiContactOrgData['contact_name'];
                        $types .= "s";
                    }
                }

                if(!empty($conditions)){
                    // If $conditions[0] is the primary (potentially compound) condition
                    $sqlFindContact .= " AND (" . $conditions[0] . ")";
                } else {
                    $sqlFindContact .= " AND 1=0"; // No valid criteria to find contact if both email and name are invalid/missing
                }
                $sqlFindContact .= " ORDER BY auto_id DESC LIMIT 1";

                $stmtContact = $conn->prepare($sqlFindContact);
                if ($stmtContact) {
                    if (!empty($types)) { // Bind parameters if there are any
                        $stmtContact->bind_param($types, ...$params);
                    }

                    if($stmtContact->execute()){
                        $resultContact = $stmtContact->get_result();
                        if ($rowContact = $resultContact->fetch_assoc()) { $extractedContactId = $rowContact['auto_id']; $contactFoundInDb = true; error_log("CRON JOB: Conv ID {$conversationId} - Found existing Contact ID: {$extractedContactId}");}
                        if ($resultContact instanceof mysqli_result) $resultContact->free();
                    } else { error_log("CRON JOB: SQL Exec Error (Find Contact) for ConvID {$conversationId}: " . $stmtContact->error . " SQL: " . $sqlFindContact . " Params: " . json_encode($params)); }
                    $stmtContact->close();
                } else { error_log("CRON JOB: SQL PrepError (Find Contact) for ConvID {$conversationId}: " . $conn->error); }

                if (!$contactFoundInDb && $canCreateOrMatchContact) { // Check $canCreateOrMatchContact again before attempting creation
                    if (function_exists('create_new_contact')) {
                        error_log("CRON JOB: Conv ID {$conversationId} - Contact not found. Attempting creation for Org ID {$extractedOrgId}.");
                        $newContactId = create_new_contact($conn, $extractedOrgId, $aiContactOrgData);
                        if ($newContactId) { $extractedContactId = $newContactId; error_log("CRON JOB: Conv ID {$conversationId} - Created new Contact ID: {$extractedContactId}");
                        } else { error_log("CRON JOB: Conv ID {$conversationId} - Failed to create contact."); }
                    } else { error_log("CRON JOB: Conv ID {$conversationId} - func 'create_new_contact' missing."); }
                }
            } elseif ($extractedOrgId) { error_log("CRON JOB: Org ID {$extractedOrgId} identified, but not enough AI contact data (name/email) for contact actions."); }
            // --- END OF FULLY EXPANDED Contact matching/creation ---
            $orgContactProcessedCount++; // Increment if Org/Contact management block was run
        } catch (Exception $ocException) {
            error_log("CRON JOB: ERROR during Organization/Contact Management for Conv ID {$conversationId}: " . $ocException->getMessage());
        }
        error_log("CRON JOB: Org/Contact Management finished for Conv ID {$conversationId}. OrgID: ".($extractedOrgId ?? 'None').", ContactID: ".($extractedContactId ?? 'None'));
    } elseif ($enableQuoteGeneration && !$enableOrgContactManagement) {
         error_log("CRON JOB: Org/Contact Management DISABLED for Conv ID {$conversationId}, though Quote Generation umbrella is enabled.");
    }


    // --- (MODULAR & CONDITIONAL) AI Quote Segment Generation OR Handling Pre-existing Quotes ---
    if ($enableQuoteGeneration) {
        $conn->begin_transaction();
        try {
            $orgIdForDb = $extractedOrgId ? (int)$extractedOrgId : null;
            $contactIdForDb = $extractedContactId ? (int)$extractedContactId : null;

            if (!$quotesAlreadyExist) {
                //error_log("CRON JOB: Core AI Segment Generation ENABLED and NO pre-existing quotes found for Conv ID {$conversationId}. Running segment extraction.");

                if (!function_exists('extractTravelSegmentsAndMatchTemplates')) { throw new Exception("Function extractTravelSegmentsAndMatchTemplates does not exist. Conv ID {$conversationId}.");}
                $aiExtractionResult = extractTravelSegmentsAndMatchTemplates($conversationId, $conn, '', $mailboxForThisEmail);
                if (!$aiExtractionResult['success']) { throw new Exception("AI segment extraction failed for Conv ID {$conversationId}: " . ($aiExtractionResult['error'] ?? 'AI error') . " Details: " . ($aiExtractionResult['details'] ?? 'N/A'));}

                $aiParsedDataFromSegments = $aiExtractionResult['ai_parsed_data'] ?? null;
                $extractedTravelDate = null;
                if ($aiParsedDataFromSegments && isset($aiParsedDataFromSegments['travel_date'])) {
                    $dateValue = trim($aiParsedDataFromSegments['travel_date']);
                    if (strtolower($dateValue) !== 'n/a' && !empty($dateValue)) {
                        $timestamp = strtotime($dateValue);
                        if ($timestamp !== false) { $extractedTravelDate = date('Y-m-d', $timestamp); }
                    }
                }

                // --- FULLY EXPANDED: $dataForDbAiResponse construction ---
                $dataForDbAiResponse = [
                    'success' => $aiExtractionResult['success'],
                    'segments' => ($aiExtractionResult['segments'] ?? []),
                    'itinerary_items_used' => $aiExtractionResult['itinerary_items_used'] ?? 'N/A',
                    'raw_ai_response' => $aiExtractionResult['raw_ai_response'] ?? null,
                    'ai_parsed_data' => $aiParsedDataFromSegments,
                    'message' => $aiExtractionResult['message'] ?? 'Processed.',
                    'category_used_for_filter' => $aiExtractionResult['category_used_for_filter'] ?? 'N/A',
                    'initial_ai_category_log' => $aiExtractionResult['initial_ai_category_log'] ?? null,
                    'processed_organizationid' => $orgIdForDb,
                    'processed_contactid' => $contactIdForDb
                ];
                $jsonForDbAiResponse = json_encode($dataForDbAiResponse);
                if ($jsonForDbAiResponse === false) { throw new Exception("JSON encode error for AI response, Conv ID {$conversationId}: " . json_last_error_msg());}

                // --- FULLY EXPANDED: $templateRefString construction ---
                $templateRefString = 'N/A';
                $segmentsFromExtraction = $aiExtractionResult['segments'] ?? [];
                if (!empty($segmentsFromExtraction)) {
                    $templateIdsCollector = [];
                    foreach ($segmentsFromExtraction as $segment) { if (isset($segment['matched_template']['templateid']) && !empty($segment['matched_template']['templateid'])) { $templateIdsCollector[] = $segment['matched_template']['templateid'];}}
                    $uniqueFilteredIds = array_unique(array_filter($templateIdsCollector));
                    if (!empty($uniqueFilteredIds)) { $templateRefString = implode(',', $uniqueFilteredIds); }
                }

                // --- FULLY EXPANDED: $country determination ---
                $country = 'N/A';
                if ($aiParsedDataFromSegments !== null && isset($aiParsedDataFromSegments['country'])) {
                    $countryList = array_map('trim', explode(',', $aiParsedDataFromSegments['country']));
                    $validCountries = array_filter($countryList, function($c) { return strtolower($c) !== 'n/a' && !empty($c); });
                    if (!empty($validCountries)) { $country = reset($validCountries); }
                }
                if (($country === 'N/A' || empty($country)) && !empty($segmentsFromExtraction[0]['input_country']) && strtolower($segmentsFromExtraction[0]['input_country']) !== 'n/a') {
                    $country = trim($segmentsFromExtraction[0]['input_country']);
                     error_log("CRON JOB: Conv ID {$conversationId} - Used fallback country '{$country}' from segment 1 input.");
                }
                error_log("CRON JOB: Conv ID {$conversationId} - Country Determined for DB: " . $country);
                // If no valid templates were matched (templateref is 'N/A' or empty), skip updating tdu_temp_ai_quotes.
                if ($templateRefString === 'N/A' || empty($templateRefString)) {
                    error_log("CRON JOB: Conv ID {$conversationId} - Skipping upsert into {$escapedTableName} because no valid templates were matched (templateref is 'N/A' or empty).");
                    // Do not increment $mainProcessedCount here, as no segment-related upsert occurred.
                } else {
                    // Upsert into tdu_temp_ai_quotes (WITH AI SEGMENTS)
                    $sqlInsertOrUpdateQuote = "INSERT INTO {$escapedTableName} (templateref, quoteref, country, travel_date, ai_response, conversation_id, organizationid, contactid, created_at, sender) VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW(), ?) ON DUPLICATE KEY UPDATE templateref = VALUES(templateref), quoteref = VALUES(quoteref), country = VALUES(country), travel_date = VALUES(travel_date), ai_response = VALUES(ai_response), organizationid = VALUES(organizationid), contactid = VALUES(contactid), created_at = NOW(), sender = VALUES(sender)";
                    $stmtInsertUpdate = $conn->prepare($sqlInsertOrUpdateQuote);
                    if ($stmtInsertUpdate === false) { throw new Exception("SQL Prep Error (Insert/Update Quote - Segments) for Conv ID {$conversationId}: " . $conn->error); }
    
                    $travelDateForDb = ($extractedTravelDate === 'N/A' || empty($extractedTravelDate)) ? null : $extractedTravelDate;
                    $quoteRefForDbThisPath = null;
    
                    $stmtInsertUpdate->bind_param("ssssssiss", $templateRefString, $quoteRefForDbThisPath, $country, $travelDateForDb, $jsonForDbAiResponse, $conversationId, $orgIdForDb, $contactIdForDb, $senderEmailForDb);
                    if (!$stmtInsertUpdate->execute()) { $stmtError = $stmtInsertUpdate->error; $stmtInsertUpdate->close(); throw new Exception("SQL Exec Error (Insert/Update Quote - Segments): " . $stmtError); }
                    error_log("CRON JOB: Upserted record into {$escapedTableName} with AI Segments for Conv ID {$conversationId}. Affected Rows: ".$stmtInsertUpdate->affected_rows);
                    $stmtInsertUpdate->close();
                    $mainProcessedCount++;
                }

            } else { // $quotesAlreadyExist is TRUE
                //error_log("CRON JOB: Core AI Segment Generation SKIPPED for Conv ID {$conversationId} due to pre-detected quotes: " . $detectedQuoteNumbersJson . ". Storing pre-detected quote info.");
                $skippedForExistingQuotesCount++;
                /*
                $aiResponseMessageArray = [
                    'message' => 'Core AI segment generation skipped due to pre-detected quotes by extractQuoteNumbersFromConversation. Org/Contact management may have run if enabled. AI Tagging may still apply.',
                    'detected_quotes_json' => json_decode($detectedQuoteNumbersJson),
                    'processed_organizationid' => $orgIdForDb,
                    'processed_contactid' => $contactIdForDb
                ];
                $aiResponseMessage = json_encode($aiResponseMessageArray);
                 if ($aiResponseMessage === false) { throw new Exception("JSON encode error for skipped AI response, Conv ID {$conversationId}: " . json_last_error_msg());}

                $sqlInsertSkipped = "INSERT INTO {$escapedTableName}
                    (templateref, quoteref, country, travel_date, ai_response, conversation_id, organizationid, contactid, created_at, sender)
                    VALUES (NULL, ?, NULL, NULL, ?, ?, ?, ?, NOW(), ?)
                    ON DUPLICATE KEY UPDATE
                    quoteref = VALUES(quoteref), ai_response = VALUES(ai_response),
                    organizationid = VALUES(organizationid), contactid = VALUES(contactid),
                    sender = VALUES(sender), created_at = NOW(),
                    templateref = NULL, country = NULL, travel_date = NULL";

                $stmtInsertSkipped = $conn->prepare($sqlInsertSkipped);
                if ($stmtInsertSkipped) {
                    $decodedQuotesArray = json_decode($detectedQuoteNumbersJson, true);
                    $quoterefToStore = null;
                    if (is_array($decodedQuotesArray)) {
                        if (count($decodedQuotesArray) === 1 && !empty($decodedQuotesArray[0])) {
                            $quoterefToStore = $decodedQuotesArray[0];
                        } elseif (count($decodedQuotesArray) > 1) {
                            $quoterefToStore = implode(',', array_filter($decodedQuotesArray));
                            if(empty($quoterefToStore)) $quoterefToStore = null;
                        }
                    } else {
                        error_log("CRON JOB: Conv ID {$conversationId} - Pre-detected quotes ('{$detectedQuoteNumbersJson}') was not valid JSON or not an array. Storing NULL in quoteref.");
                    }

                    $stmtInsertSkipped->bind_param("sssiis",
                        $quoterefToStore,
                        $aiResponseMessage,
                        $conversationId,
                        $orgIdForDb,
                        $contactIdForDb,
                        $senderEmailForDb
                    );

                    if ($stmtInsertSkipped->execute()) { error_log("CRON JOB: Conv ID {$conversationId} - Upserted {$escapedTableName} with pre-detected quotes and Org/Contact. Affected rows: " . $stmtInsertSkipped->affected_rows);
                    } else { error_log("CRON JOB: Conv ID {$conversationId} - FAILED to upsert {$escapedTableName} (skipped case): " . $stmtInsertSkipped->error); throw new Exception($stmtInsertSkipped->error); }
                    $stmtInsertSkipped->close();
                } else { error_log("CRON JOB: Conv ID {$conversationId} - FAILED to prepare {$escapedTableName} (skipped case): " . $conn->error); throw new Exception($conn->error); }
                */
            }

            // Update vtiger_support status (common to both paths if $enableQuoteGeneration is true)
            $statusAiProcessed = VTIGER_SUPPORT_STATUS_AI_PROCESSED;
            $baseSqlInsertVtiger = "INSERT INTO vtiger_support"; $insertColumnsVtiger = ["ticketid", "ai_actioned", "status"]; $insertPlaceholdersVtiger = ["?", "?", "?"]; $insertParamsVtiger = [$conversationId, 1, $statusAiProcessed]; $insertTypesVtiger = "sis";
            $updateSetClausesVtiger = ["ai_actioned = ?", "status = ?"]; $updateParamsVtiger = [1, $statusAiProcessed]; $updateTypesVtiger = "is";
            $sqlVtiger = $baseSqlInsertVtiger . " (" . implode(", ", $insertColumnsVtiger) . ") VALUES (" . implode(", ", $insertPlaceholdersVtiger) . ") ON DUPLICATE KEY UPDATE " . implode(", ", $updateSetClausesVtiger);
            $allParamsVtiger = array_merge($insertParamsVtiger, $updateParamsVtiger); $allTypesVtiger = $insertTypesVtiger . $updateTypesVtiger;

            $stmtUpdateVtiger = $conn->prepare($sqlVtiger);
            if ($stmtUpdateVtiger === false) { throw new Exception("SQL Prep Error (Update Vtiger Support): " . $conn->error); }
            $stmtUpdateVtiger->bind_param($allTypesVtiger, ...$allParamsVtiger);
            if (!$stmtUpdateVtiger->execute()) { $stmtErrorVtiger = $stmtUpdateVtiger->error; $stmtUpdateVtiger->close(); throw new Exception("SQL Exec Error (Update Vtiger Support): " . $stmtErrorVtiger); }
            error_log("CRON JOB: Ensured vtiger_support status is '" . $statusAiProcessed . "' (ai_actioned=1) for Ticket ID {$conversationId}. Affected rows: ".$stmtUpdateVtiger->affected_rows);
            $stmtUpdateVtiger->close();

            $conn->commit();
        } catch (Exception $e) {
            if ($conn->get_transaction_status()) { $conn->rollback(); }
            error_log("CRON JOB: ERROR during AI Quote/Segment or Pre-detected Quote Handling for Conv ID {$conversationId}: " . $e->getMessage() . "\nTrace: " . substr($e->getTraceAsString(), 0, 500));
            $errorCount++;
        }
    } else { // $enableQuoteGeneration is false
         error_log("CRON JOB: AI Quote Generation features (including Org/Contact and Segments) are DISABLED for Conv ID {$conversationId}. No core processing performed.");
    }


    // AI Tagging Module
    if ($enableAiTagging) {
        $tagsAttempted = false; $tagsSuccessfullyApplied = false;
        $coreAiBlockLikelyUpdatedVtiger = ($enableQuoteGeneration && (!isset($e) || $e === null)); // Check if the above try block completed without throwing an exception into $e

        try {
            if (function_exists('getSuggestedTagsForConversation')) {
                $excludedTagCategories = [];
                $suggestedTags = getSuggestedTagsForConversation($conversationId, $conn, $excludedTagCategories);
                $tagsAttempted = true;

                if (!empty($suggestedTags)) {
                    $sqlInsertTag = "INSERT IGNORE INTO vtiger_freetagged_objects (tag_id, tagger_id, object_id, module) VALUES (?, ?, ?, ?)";
                    $stmtInsertTag = $conn->prepare($sqlInsertTag);
                    if ($stmtInsertTag) {
                        foreach ($suggestedTags as $tagInfo) {
                            if (isset($tagInfo['id']) && is_numeric($tagInfo['id'])) {
                                $tagIdToInsert = (int)$tagInfo['id'];
                                $stmtInsertTag->bind_param("iiss", $tagIdToInsert, $aiTaggerUserId, $conversationId, $tagModule);
                                if ($stmtInsertTag->execute()) {
                                    if ($stmtInsertTag->affected_rows > 0) { error_log("CRON JOB: Conv ID {$conversationId} - Successfully tagged with Tag ID: {$tagIdToInsert}."); $tagsSuccessfullyApplied = true; }
                                } else { error_log("CRON JOB: Conv ID {$conversationId} - FAILED tag insert exec TagID {$tagIdToInsert}: " . $stmtInsertTag->error); }
                            } else { error_log("CRON JOB: Conv ID {$conversationId} - Invalid tag ID in suggestions: " . print_r($tagInfo, true));}
                        }
                        $stmtInsertTag->close();
                    } else { error_log("CRON JOB: Conv ID {$conversationId} - FAILED prepare tag insert: " . $conn->error); throw new Exception("Tag insert prep failed."); }
                } else { error_log("CRON JOB: Conv ID {$conversationId} - No tags suggested or error during suggestion.");}

                if ($tagsSuccessfullyApplied) {
                    $taggingProcessedCount++;
                    if (!$coreAiBlockLikelyUpdatedVtiger) { // If core AI block might not have set vtiger status
                        $sqlSetVtigerStatusTagging = "INSERT INTO vtiger_support (ticketid, status, ai_actioned) VALUES (?, ?, 1) ON DUPLICATE KEY UPDATE status = IF(status IS NULL OR status != ?, VALUES(status), status), ai_actioned = 1";
                        $stmtSetVtigerTagging = $conn->prepare($sqlSetVtigerStatusTagging);
                        if ($stmtSetVtigerTagging) {
                            $statusAiProcessed = VTIGER_SUPPORT_STATUS_AI_PROCESSED;
                            $statusSpamConst = VTIGER_SUPPORT_STATUS_SPAM;
                            $stmtSetVtigerTagging->bind_param("sss", $conversationId, $statusAiProcessed, $statusSpamConst);
                            if ($stmtSetVtigerTagging->execute()) { error_log("CRON JOB: Conv ID {$conversationId} - Ensured vtiger_support 'AI Processed' after tagging (if not already set by core AI). Affected: " . $stmtSetVtigerTagging->affected_rows);
                            } else { error_log("CRON JOB: Conv ID {$conversationId} - FAILED update vtiger_support status post-tagging: " . $stmtSetVtigerTagging->error); }
                            $stmtSetVtigerTagging->close();
                        } else { error_log("CRON JOB: Conv ID {$conversationId} - FAILED prepare vtiger_support status (post-tagging): " . $conn->error); }
                    }
                } elseif ($tagsAttempted) {
                    $taggingProcessedCount++;
                }
                // Reset $e for the next iteration or module, if it was caught above.
                unset($e);


            } else { error_log("CRON JOB: Missing func 'getSuggestedTagsForConversation'."); throw new Exception("Tagging function missing."); }
        } catch (Exception $taggingException) { // Use a different variable for exception in tagging block
            error_log("CRON JOB: ERROR during AI Tagging block for Conv ID {$conversationId}: " . $taggingException->getMessage());
            $errorCount++;
        }
    } else {
        //error_log("CRON JOB: AI Tagging DISABLED for Conv ID {$conversationId}.");
    }

} // End foreach email loop

error_log("CRON JOB: Finished at " . date('Y-m-d H:i:s') . ". Summary: AI Segments Ran: {$mainProcessedCount}, Segments Skipped (Quotes Exist): {$skippedForExistingQuotesCount}, Org/Contact Ran: {$orgContactProcessedCount}, Tagging Ran: {$taggingProcessedCount}, Skipped (already actioned): {$skippedAlreadyActionedCount}, SPAM: {$spamCount}, Module Errors: {$errorCount}, Spam Check Errors: {$spamErrorCount}.");
if ($conn && $conn->ping()) {
    $conn->close(); // Close the database connection
}
exit(0); // Successful exit
?>