<?php
// cronjob_fetch_qbo_data.php
// Nightly cron job to sync QuickBooks Online data to a local database.
// FINAL PRODUCTION VERSION: Implements "Plan B" (Last Modified Time) and is
// self-initializing, creating its own database tables if they do not exist.

set_time_limit(3600); // 1 hour timeout.
ini_set('memory_limit', '256M'); // Allow more memory for processing.

require_once __DIR__ . '/../qbo_bootstrap.php';
require_once __DIR__ . '/../dbconn.php'; // Your existing DB connection file.

$logFile = __DIR__ . '/qbo_cron.log';
$stateFileDir = __DIR__ . '/cron_state';

// --- Helper & Setup Functions ---

function write_log($message) {
    global $logFile;
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($logFile, "[$timestamp] $message\n", FILE_APPEND);
}

function ensureTablesExist($conn) {
    write_log("Verifying database schema...");

    $queries = [
        "CREATE TABLE IF NOT EXISTS `qbo_customers` (
          `id` INT(11) NOT NULL AUTO_INCREMENT,
          `qbo_id` INT(11) NOT NULL,
          `display_name` VARCHAR(255) DEFAULT NULL,
          `is_active` TINYINT(1) NOT NULL DEFAULT '1',
          `last_updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
          PRIMARY KEY (`id`),
          UNIQUE KEY `qbo_id` (`qbo_id`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;",

        "CREATE TABLE IF NOT EXISTS `qbo_items` (
          `id` INT(11) NOT NULL AUTO_INCREMENT,
          `qbo_id` INT(11) NOT NULL,
          `name` VARCHAR(255) DEFAULT NULL,
          `is_active` TINYINT(1) NOT NULL DEFAULT '1',
          `last_updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
          PRIMARY KEY (`id`),
          UNIQUE KEY `qbo_id` (`qbo_id`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;",

        "CREATE TABLE IF NOT EXISTS `qbo_invoices` (
          `id` INT(11) NOT NULL AUTO_INCREMENT,
          `qbo_id` INT(11) NOT NULL,
          `customer_qbo_id` INT(11) NOT NULL,
          `doc_number` VARCHAR(50) DEFAULT NULL,
          `txn_date` DATE DEFAULT NULL,
          `due_date` DATE DEFAULT NULL,
          `total_amount` DECIMAL(10,2) DEFAULT '0.00',
          `status` VARCHAR(25) DEFAULT NULL,
          `qbo_create_time` DATETIME DEFAULT NULL,
          `qbo_last_modified_time` DATETIME DEFAULT NULL,
          `last_updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
          PRIMARY KEY (`id`),
          UNIQUE KEY `qbo_id` (`qbo_id`),
          KEY `txn_date` (`txn_date`),
          KEY `qbo_last_modified_time` (`qbo_last_modified_time`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;",

        "CREATE TABLE IF NOT EXISTS `qbo_invoice_lines` (
          `id` INT(11) NOT NULL AUTO_INCREMENT,
          `invoice_qbo_id` INT(11) NOT NULL,
          `item_qbo_id` INT(11) DEFAULT NULL,
          `class_name` VARCHAR(255) DEFAULT NULL,
          `qty` DECIMAL(10,2) DEFAULT '1.00', /* <-- This line must be present */
          `line_amount` DECIMAL(10,2) DEFAULT '0.00',
          PRIMARY KEY (`id`),
          KEY `invoice_qbo_id` (`invoice_qbo_id`),
          KEY `item_qbo_id` (`item_qbo_id`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;"
    ];

    foreach ($queries as $query) {
        if ($conn->query($query) === FALSE) {
            throw new Exception("Database schema setup failed: " . $conn->error);
        }
    }
    write_log("Database schema verified successfully.");
}

function getLastSyncTime($stateFile) {
    if (!file_exists($stateFile)) {
        return '2000-01-01T00:00:00Z';
    }
    return trim(file_get_contents($stateFile));
}

function setLastSyncTime($stateFile, $timestamp) {
    file_put_contents($stateFile, $timestamp);
}

// --- Main Execution ---
write_log("--- Cron Job Started ---");

if (!file_exists($stateFileDir)) {
    mkdir($stateFileDir, 0755, true);
}
if (!$conn) {
    write_log("CRITICAL DB ERROR: Database connection failed. Check dbconn.php.");
    exit(1);
}

try {
    // 1. Ensure database tables exist before doing anything else.
    ensureTablesExist($conn);
    
    $dataService = $qboUtil->getDataService();
    
    // 2. Sync Invoices
    $invoiceStateFile = $stateFileDir . '/invoice_last_sync.log';
    $lastInvoiceSync = getLastSyncTime($invoiceStateFile);
    write_log("Syncing Invoices modified since: " . $lastInvoiceSync);
    $maxInvoiceSyncTime = $lastInvoiceSync;

    $i = 1;
    while (true) {
        $query = "SELECT * FROM Invoice WHERE MetaData.LastUpdatedTime > '{$lastInvoiceSync}' ORDERBY MetaData.LastUpdatedTime ASC STARTPOSITION {$i} MAXRESULTS 100";
        $invoices = $dataService->Query($query);
        if (empty($invoices)) { break; }

        write_log("Processing batch of " . count($invoices) . " invoices...");
        foreach ($invoices as $invoice) {
            // Business logic for status, based on our sanity check
            $status = ($invoice->Balance == 0 && $invoice->TotalAmt > 0) ? 'Paid' : 'Open';
            if (floatval($invoice->TotalAmt) == 0) $status = 'Zero Value';

            $stmt = $conn->prepare("INSERT INTO qbo_invoices (qbo_id, customer_qbo_id, doc_number, txn_date, due_date, total_amount, status, qbo_create_time, qbo_last_modified_time) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                                   ON DUPLICATE KEY UPDATE customer_qbo_id=VALUES(customer_qbo_id), doc_number=VALUES(doc_number), txn_date=VALUES(txn_date), due_date=VALUES(due_date),
                                   total_amount=VALUES(total_amount), status=VALUES(status), qbo_create_time=VALUES(qbo_create_time), qbo_last_modified_time=VALUES(qbo_last_modified_time)");
            $stmt->bind_param("iisssdsss", $invoice->Id, $invoice->CustomerRef, $invoice->DocNumber, $invoice->TxnDate, $invoice->DueDate, $invoice->TotalAmt, $status, $invoice->MetaData->CreateTime, $invoice->MetaData->LastUpdatedTime);
            $stmt->execute();
            
            // Re-sync lines
            $conn->prepare("DELETE FROM qbo_invoice_lines WHERE invoice_qbo_id = ?")->execute([$invoice->Id]);
            if (!empty($invoice->Line)) {
                // PRE-FETCH THE HEADER CLASS ID AND NAME (as our top priority)
                $headerClassName = null;
                if (!empty($invoice->ClassRef)) {
                    try {
                        $classObj = $dataService->FindById('Class', $invoice->ClassRef);
                        $headerClassName = $classObj ? $classObj->Name : null;
                    } catch (\Exception $classEx) {
                        write_log("Warning: Could not fetch header ClassRef {$invoice->ClassRef} for Invoice ID {$invoice->Id}. Error: " . $classEx->getMessage());
                    }
                }
            
                // Prepare the statement with all the correct columns
                $insStmt = $conn->prepare("INSERT INTO qbo_invoice_lines (invoice_qbo_id, item_qbo_id, class_name, qty, line_amount) VALUES (?, ?, ?, ?, ?)");
                
                foreach ($invoice->Line as $line) {
                    if ($line->DetailType !== 'SalesItemLineDetail' || empty($line->SalesItemLineDetail->ItemRef)) continue;
                    
                    // --- THIS IS THE NEW HIERARCHY LOGIC ---
                    $finalClassName = null;
            
                    // Priority 1: Use the Header-level Class if it exists.
                    if (!empty($headerClassName)) {
                        $finalClassName = $headerClassName;
                    } 
                    // Priority 2 (Fallback): If the header is empty, check the Line-level Class.
                    else if (!empty($line->SalesItemLineDetail->ClassRef)) {
                        try {
                            $lineClassObj = $dataService->FindById('Class', $line->SalesItemLineDetail->ClassRef);
                            $finalClassName = $lineClassObj ? $lineClassObj->Name : null;
                        } catch (\Exception $lineClassEx) {
                            write_log("Warning: Could not fetch line-level ClassRef {$line->SalesItemLineDetail->ClassRef} for Invoice ID {$invoice->Id}. Error: " . $lineClassEx->getMessage());
                            // $finalClassName remains null
                        }
                    }
                    // Priority 3 (Final Fallback): If both are empty, $finalClassName is already null.
            
                    // Get the quantity and default to 1 if it's empty
                    $quantity = $line->SalesItemLineDetail->Qty ?? 1;
                    if (empty($quantity)) {
                        $quantity = 1;
                    }
                    
                    // Use the final determined class name and quantity in the database insert
                    $insStmt->bind_param("iisdd", $invoice->Id, $line->SalesItemLineDetail->ItemRef, $finalClassName, $quantity, $line->Amount);
                    $insStmt->execute();
                }
            }
            $maxInvoiceSyncTime = $invoice->MetaData->LastUpdatedTime;
        }
        $i += count($invoices);
    }
    setLastSyncTime($invoiceStateFile, $maxInvoiceSyncTime);

    // 3. Sync Customers with full pagination
    $customerStateFile = $stateFileDir . '/customer_last_sync.log';
    $lastCustomerSync = getLastSyncTime($customerStateFile);
    write_log("Syncing Customers modified since: " . $lastCustomerSync);
    $maxCustomerSyncTime = $lastCustomerSync;

    $i = 1;
    while (true) {
        $query = "SELECT * FROM Customer WHERE MetaData.LastUpdatedTime > '{$lastCustomerSync}' ORDERBY MetaData.LastUpdatedTime ASC STARTPOSITION {$i} MAXRESULTS 100";
        $customers = $dataService->Query($query);
        if (empty($customers)) { break; }

        write_log("Processing batch of " . count($customers) . " customers...");
        $stmt = $conn->prepare("INSERT INTO qbo_customers (qbo_id, display_name, is_active) VALUES (?, ?, ?) ON DUPLICATE KEY UPDATE display_name=VALUES(display_name), is_active=VALUES(is_active)");
        foreach ($customers as $customer) {
            $stmt->bind_param("isi", $customer->Id, $customer->DisplayName, $customer->Active);
            $stmt->execute();
            $maxCustomerSyncTime = $customer->MetaData->LastUpdatedTime;
        }
        $i += count($customers);
    }
    setLastSyncTime($customerStateFile, $maxCustomerSyncTime);

    // 4. Sync Items with full pagination
    $itemStateFile = $stateFileDir . '/item_last_sync.log';
    $lastItemSync = getLastSyncTime($itemStateFile);
    write_log("Syncing Items modified since: " . $lastItemSync);
    $maxItemSyncTime = $lastItemSync;

    $i = 1;
    while (true) {
        $query = "SELECT * FROM Item WHERE MetaData.LastUpdatedTime > '{$lastItemSync}' ORDERBY MetaData.LastUpdatedTime ASC STARTPOSITION {$i} MAXRESULTS 100";
        $items = $dataService->Query($query);
        if (empty($items)) { break; }

        write_log("Processing batch of " . count($items) . " items...");
        $stmt = $conn->prepare("INSERT INTO qbo_items (qbo_id, name, is_active) VALUES (?, ?, ?) ON DUPLICATE KEY UPDATE name=VALUES(name), is_active=VALUES(is_active)");
        foreach ($items as $item) {
            $stmt->bind_param("isi", $item->Id, $item->Name, $item->Active);
            $stmt->execute();
            $maxItemSyncTime = $item->MetaData->LastUpdatedTime;
        }
        $i += count($items);
    }
    setLastSyncTime($itemStateFile, $maxItemSyncTime);
    
} catch (\Exception $e) {
    write_log("!! SCRIPT FAILED !! Error: " . $e->getMessage());
    write_log("File: " . $e->getFile() . " on Line: " . $e->getLine());
    exit(1);
}

write_log("--- Cron Job Finished Successfully ---");
exit(0);
?>