<?php
// cronjob_qbo_processing.php
// CLI script to process Vtiger quotes for QBO Sync.
// Usage:
// php cronjob_qbo_processing.php [quote_no]
// If [quote_no] is provided, it processes that specific quote.
// If no [quote_no] is provided, it (will eventually) process a batch.

// --- Essential Setup ---
ini_set('display_errors', 1); // Show errors on CLI
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Set a longer execution time if batch processing might take a while
set_time_limit(0); // 0 = no time limit for CLI scripts

// Define a base path for includes if this script is in a subdirectory

require_once __DIR__ . '/../dbconn.php';
require_once __DIR__ . '/../../vendor/autoload.php'; // QBO SDK Autoloader
require_once __DIR__ . '/../qbo_functions.php';
require_once __DIR__ . '/../quote_pricing.php';   // For getQuotePricing()
require_once __DIR__ . '/qbo_sync_processor.php'; // The combined algorithm file
require_once __DIR__ . '/../config/qbo_config.php';

use QuickBooksOnline\API\Core\OAuth\OAuth2\OAuth2LoginHelper;
use QuickBooksOnline\API\Exception\SdkException;
use QuickBooksOnline\API\Exception\ServiceException;

global $conn; // From dbconn.php

// --- Configuration & Helpers ---
$tokenStorageFile = __DIR__ . '/../tokens/qbo_token.json';
$logFileCli = __DIR__ . '/qbo_cli_cron.log'; // Specific log for this CLI script

function cliLog($message) {
    global $logFileCli;
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[$timestamp] " . $message . "\n";
    echo $logEntry; // Output to console
    file_put_contents($logFileCli, $logEntry, FILE_APPEND); // Also write to log file
}

function loadTokensCli() {
    global $tokenStorageFile;
    if (file_exists($tokenStorageFile)) {
        $json = file_get_contents($tokenStorageFile);
        return $json ? json_decode($json, true) : null;
    }
    cliLog("ERROR: Token storage file not found at: " . $tokenStorageFile);
    return null;
}

function saveTokensCli($accessToken, $refreshToken) {
    global $tokenStorageFile;
    $tokens = [
        'access_token' => $accessToken,
        'refresh_token' => $refreshToken,
        'last_updated' => date('Y-m-d H:i:s')
    ];
    $tokenDir = dirname($tokenStorageFile);
    if (!is_dir($tokenDir)) {
        if (!mkdir($tokenDir, 0755, true) && !is_dir($tokenDir)) {
            cliLog("FATAL: Could not create token storage directory: " . $tokenDir);
            return false;
        }
    }
    if (file_put_contents($tokenStorageFile, json_encode($tokens, JSON_PRETTY_PRINT))) {
        cliLog("Tokens successfully saved to storage by CLI script.");
        return true;
    } else {
        cliLog("FATAL: Could not write tokens to storage file by CLI script: " . $tokenStorageFile);
        return false;
    }
}
/*
$qboBaseConfig = [
    'auth_mode'       => 'oauth2',
    'ClientID'        => "AB3WEwQtfG1Ws43pqxY7Y2Ok3s7QfN97VniZxSssaQhl9PA4JE", // Your Client ID
    'ClientSecret'    => "GRjmj4WxpMo5ZkMOjfnEQ5LmzXkAna1xb9YpwEWL", // Your Client Secret
    'QBORealmID'      => "9341454710400258",          // Your Realm ID
    'baseUrl'         => "Development",             // Or "Production"
    // 'logLocation'     => BASE_PATH . "/logs/qbo_sdk_cli_logs" // Optional SDK log location
];
// --- End Configuration & Helpers ---
*/

cliLog("--------------------------------------------------");
cliLog("CLI QBO Processing Job Started.");

// 1. Attempt to Refresh Tokens
cliLog("Attempting to refresh QBO tokens...");
$currentTokens = loadTokensCli();
$qboConfigForLibrary = $qboBaseConfig; // Base config for refresh helper

if ($currentTokens && !empty($currentTokens['refresh_token'])) {
    $qboConfigForLibrary['refreshTokenKey'] = $currentTokens['refresh_token']; // Needed by refresh helper
    try {
        $oauth2LoginHelper = new OAuth2LoginHelper($qboBaseConfig['ClientID'], $qboBaseConfig['ClientSecret']);
        $refreshedAccessTokenObj = $oauth2LoginHelper->refreshAccessTokenWithRefreshToken($qboConfigForLibrary['refreshTokenKey']);

        $newAccessToken = $refreshedAccessTokenObj->getAccessToken();
        $newRefreshToken = $refreshedAccessTokenObj->getRefreshToken();

        if (saveTokensCli($newAccessToken, $newRefreshToken)) {
            cliLog("SUCCESS: QBO Tokens refreshed and saved.");
            // Update currentTokens for this run
            $currentTokens['access_token'] = $newAccessToken;
            $currentTokens['refresh_token'] = $newRefreshToken;
        } else {
            cliLog("ERROR: Tokens refreshed but failed to save. Proceeding with potentially old tokens if loaded.");
        }
    } catch (SdkException $e) {
        cliLog("ERROR refreshing tokens (SDK Exception): " . $e->getMessage() . " | Details: " . ($e->getPrevious() ? $e->getPrevious()->getMessage() : 'No details'));
        cliLog("Proceeding with potentially old tokens if loaded.");
    } catch (ServiceException $e) {
        cliLog("ERROR refreshing tokens (Service Exception): " . $e->getMessage());
        cliLog("Proceeding with potentially old tokens if loaded.");
    } catch (Exception $e) {
        cliLog("An unexpected error occurred during token refresh: " . $e->getMessage());
        cliLog("Proceeding with potentially old tokens if loaded.");
    }
} else {
    cliLog("Warning: Refresh token not found in storage. Cannot refresh. Will attempt to use existing tokens if available.");
}

// Prepare the final QBO config for the QBOUtilityLibrary
if ($currentTokens && !empty($currentTokens['access_token']) && !empty($currentTokens['refresh_token'])) {
    $qboConfigForLibrary['accessTokenKey'] = $currentTokens['access_token'];
    $qboConfigForLibrary['refreshTokenKey'] = $currentTokens['refresh_token']; // Should be the latest if refresh succeeded
} else {
    cliLog("CRITICAL: No valid access/refresh tokens available after refresh attempt. Cannot proceed with QBO operations.");
    exit(1); // Exit with an error code
}

// 2. Check Database Connection
if (!$conn) {
    cliLog("CRITICAL: Database connection failed. Check dbconn.php.");
    exit(1);
}
cliLog("Database connection successful.");

// 3. Initialize QBOUtilityLibrary
try {
    $qboLibrary = new QBOUtilityLibrary($qboConfigForLibrary);
    $qboLibrary->setDatabaseConnection($conn);

    cliLog("QBOUtilityLibrary instantiated successfully.");
} catch (InvalidArgumentException $e) {
    cliLog("CRITICAL: QBOUtilityLibrary Configuration Error: " . $e->getMessage());
    exit(1);
} catch (Exception $e) {
    cliLog("CRITICAL: Error instantiating QBOUtilityLibrary: " . $e->getMessage());
    exit(1);
}


// 4. Determine processing mode (single quote or batch)
$quote_identifier_to_process = null;
if (isset($argv[1]) && !empty(trim($argv[1]))) {
    // $argv[0] is the script name, $argv[1] is the first argument
    $quote_identifier_to_process = trim($argv[1]);
    cliLog("Processing single identifier from command line: " . $quote_identifier_to_process);

    // Process the single quote
    $result_data = processQuoteForQBOSync($quote_identifier_to_process, $conn, $qboLibrary, $qboConfigForLibrary); // Pass full config for baseUrl access

    cliLog("--- Result for " . $quote_identifier_to_process . " ---");
    // Output key information to CLI, full details to log potentially
    cliLog("Vtiger Quote ID: " . ($result_data['vtiger_quote_id'] ?? 'N/A'));
    cliLog("Vtiger Quote No: " . ($result_data['vtiger_quote_no'] ?? 'N/A'));
    cliLog("Customer Sync: " . ($result_data['customer_sync_status']['status'] ?? 'N/A') . " - " . ($result_data['customer_sync_status']['message'] ?? ''));
    cliLog("Invoice Sync: " . ($result_data['invoice_sync_status']['status'] ?? 'N/A') . " - " . ($result_data['invoice_sync_status']['message'] ?? '') . " (Link: " . ($result_data['invoice_sync_status']['qbo_invoice_link'] ?? 'N/A') . ")");
    foreach ($result_data['vendors_sync_status'] as $vendor_status) {
        cliLog("Vendor '" . ($vendor_status['original_vtiger_vendor_name'] ?? $vendor_status['vtiger_vendor_id']) . "' Bill Sync: " . ($vendor_status['status'] ?? 'N/A') . " - " . ($vendor_status['message'] ?? '') . " (Link: " . ($vendor_status['qbo_bill_link'] ?? 'N/A') .")");
    }
    if (!empty($result_data['debug_notes'])) {
        cliLog("Debug Notes Encountered:");
        foreach ($result_data['debug_notes'] as $note) {
            cliLog("  - " . $note);
        }
    }
    if (!empty($result_data['error'])) {
        cliLog("Overall Processing Error for {$quote_identifier_to_process}: " . $result_data['error']);
    }
    // For full JSON output, you might write it to a separate result file or just rely on the main log
    // file_put_contents(__DIR__ . '/result_' . preg_replace("/[^a-zA-Z0-9]/", "_", $quote_identifier_to_process) . '.json', json_encode($result_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));

} else {
    cliLog("No quote_no provided. Entering batch processing mode (placeholder).");
    // --- PLACEHOLDER FOR BATCH PROCESSING ---
    // 1. SQL to fetch a batch of quote_no or quoteid that need processing
    //    (e.g., based on a status field, last sync date, etc.)
    //    $sql_batch = "SELECT quote_no FROM vtiger_quotes WHERE qbo_sync_status = 'pending' LIMIT 10"; // Example
    //    $result_batch = mysqli_query($conn, $sql_batch);
    //
    // 2. Loop through the results
    //    if ($result_batch && mysqli_num_rows($result_batch) > 0) {
    //        while ($row_batch = mysqli_fetch_assoc($result_batch)) {
    //            $batch_quote_no = $row_batch['quote_no'];
    //            cliLog("Processing quote_no from batch: " . $batch_quote_no);
    //            $result_data = processQuoteForQBOSync($batch_quote_no, $conn, $qboLibrary, $qboConfigForLibrary);
    //
    //            // Log summary for this quote (similar to single processing)
    //            cliLog("  Result for batch item " . $batch_quote_no . ": Invoice Status - " . ($result_data['invoice_sync_status']['status'] ?? 'N/A'));
    //
    //            // Optionally, update a status in Vtiger for this quote (e.g., qbo_sync_status = 'processed' or 'error')
    //            // $update_sql = "UPDATE vtiger_quotescf SET cf_qbo_sync_status = '...' WHERE quoteid = '...'";
    //            // mysqli_query($conn, $update_sql);
    //
    //            cliLog("---"); // Separator for batch items
    //            sleep(1); // Optional: pause between processing items in a batch
    //        }
    //        mysqli_free_result($result_batch);
    //    } else {
    //        cliLog("No quotes found for batch processing.");
    //    }
    cliLog("Batch processing logic is not yet implemented.");
    // --- END PLACEHOLDER ---
}

cliLog("CLI QBO Processing Job Finished.");
cliLog("--------------------------------------------------\n");
exit(0); // Successful exit

?>