<?php
// cronjob_qbo_reconciliation.php
// CLI script to process statement files in the background.
// FINAL, CORRECTED VERSION with a three-layer matching process: Auto -> AI Disambiguation -> Manual Review.

// --- Basic Setup ---
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
set_time_limit(600); // 10 minutes

// --- Essential Includes ---
require_once __DIR__ . '/../ai_email_functions.php';
require_once __DIR__ . '/../qbo_functions.php';
require_once __DIR__ . '/../../vendor/autoload.php';
require_once __DIR__ . '/../dbconn.php';
require_once __DIR__ . '/../quote_pricing.php';
global $conn;

use QuickBooksOnline\API\Core\OAuth\OAuth2\OAuth2LoginHelper;
use QuickBooksOnline\API\Exception\IdsException;

// --- Configuration Paths ---
$qboBaseConfigFile = __DIR__ . '/../config/qbo_config.php';
$qboTokenStorageFile = __DIR__ . '/../tokens/qbo_token.json';
$uploadDirectory = __DIR__ . '/../statements/';
$qboBaseConfig = null;

// --- CLI Argument Check ---
if ($argc < 2) {
    $log_msg = "CRON ERROR: No filename provided. Usage: php " . basename(__FILE__) . " <filename>";
    error_log($log_msg);
    die($log_msg . "\n");
}
$filename_to_process = basename($argv[1]);
$full_file_path = $uploadDirectory . $filename_to_process;

if (!file_exists($full_file_path)) {
    $log_msg = "CRON ERROR: File not found: " . htmlspecialchars($full_file_path);
    error_log($log_msg);
    die($log_msg . "\n");
}


// =================================================================================
// SECTION 1: DATABASE & HELPER FUNCTIONS
// =================================================================================

function create_reconciliation_table_if_not_exists(mysqli $dbConn) {
    $tableName = 'tdu_qbo_reconciliation';
    if (mysqli_num_rows(mysqli_query($dbConn, "SHOW TABLES LIKE '{$tableName}'")) == 0) {
        $sql = "CREATE TABLE `{$tableName}` (
            `id` INT(11) NOT NULL AUTO_INCREMENT,
            `file_name` VARCHAR(255) NOT NULL,
            `ai_raw_json` LONGTEXT NULL,
            `vtiger_quote_amounts_json` LONGTEXT NULL,
            `qbo_bills_json` LONGTEXT NULL,
            `processing_status` VARCHAR(50) NOT NULL DEFAULT 'processing',
            `error_message` TEXT NULL,
            `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            `processed_at` TIMESTAMP NULL ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            UNIQUE KEY `file_name` (`file_name`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";
        if (!mysqli_query($dbConn, $sql)) { die("CRITICAL ERROR: Could not create table '{$tableName}': " . mysqli_error($dbConn)); }
    }
}

function loadQBOTokens(): ?array {
    global $qboTokenStorageFile;
    if (!file_exists($qboTokenStorageFile)) return null;
    $json = file_get_contents($qboTokenStorageFile);
    if ($json === false) return null;
    return json_decode($json, true) ?: null;
}

function saveQBOTokens(string $accessToken, string $refreshToken): bool {
    global $qboTokenStorageFile;
    $tokens = ['access_token' => $accessToken, 'refresh_token' => $refreshToken, 'last_updated' => date('Y-m-d H:i:s')];
    $tokenDir = dirname($qboTokenStorageFile);
    if (!is_dir($tokenDir)) { if (!mkdir($tokenDir, 0755, true) && !is_dir($tokenDir)) { return false; } }
    return (bool)file_put_contents($qboTokenStorageFile, json_encode($tokens, JSON_PRETTY_PRINT));
}

function generateSearchVariations(string $name): array {
    $noise = [' pty ltd', ' pty', ' ltd', ' inc', ' llc', ' group', ' hotel'];
    $searchTerms = [trim($name)];
    if (preg_match('/t\s?\/?\s?a\s(.+)/i', $name, $matches)) { array_unshift($searchTerms, trim($matches[1])); }
    $cleanName = str_ireplace($noise, '', $name);
    if (trim($cleanName) !== trim($name)) { $searchTerms[] = trim($cleanName); }
    $noParenthesesName = trim(preg_replace('/\s*\(.*?\)\s*/', ' ', $name));
    if (trim($noParenthesesName) !== trim($name)) { $searchTerms[] = trim($noParenthesesName); }
    return array_unique($searchTerms);
}

function buildVendorSearchMap(array $vendors): array {
    $searchMap = [];
    foreach ($vendors as $vendor) {
        if (!empty($vendor->DisplayName)) {
            $searchMap[$vendor->DisplayName] = generateSearchVariations($vendor->DisplayName);
        }
    }
    return $searchMap;
}

function findBestVendorMatch(string $targetName, array $vendorSearchMap): ?string {
    $bestMatch = null; $highestPercentage = 0.0;
    $targetVariations = generateSearchVariations($targetName);
    foreach ($vendorSearchMap as $originalName => $dbSearchTerms) {
        foreach ($targetVariations as $targetTerm) {
            foreach ($dbSearchTerms as $dbTerm) {
                similar_text(strtolower($targetTerm), strtolower($dbTerm), $percent);
                if ($percent > 99) return $originalName;
                if ($percent > $highestPercentage) { $highestPercentage = $percent; $bestMatch = $originalName; }
            }
        }
    }
    return ($highestPercentage > 80) ? $bestMatch : null;
}

function getSupplierQuoteCostAmount(mysqli $dbConn, array $dataRow, array $dbVendorSearchMap) {
    // This entire function is unchanged.
    $vtigerQuoteNo = $dataRow['tdu_class_number'] ?? 'N/A';
    $supplierNameFromAI = $dataRow['supplier_name'] ?? 'N/A';
    $confirmationNum = $dataRow['confirmation_number'] ?? 'N/A';

    if (empty($vtigerQuoteNo) || $vtigerQuoteNo === 'N/A') { return 'N/A Input'; }
    $escapedQuoteNo = mysqli_real_escape_string($dbConn, strtoupper(str_replace(' ', '', $vtigerQuoteNo)));
    $escapedConfirmationNum = mysqli_real_escape_string($dbConn, $confirmationNum);

    $sql_quote_details = "SELECT quoteid, adults, children, infants FROM vtiger_quotes WHERE quote_no = '{$escapedQuoteNo}' LIMIT 1";
    $result_quote_details = mysqli_query($dbConn, $sql_quote_details);
    if (!$result_quote_details || mysqli_num_rows($result_quote_details) == 0) { return 'V.Quote N/A'; }
    $quote_data = mysqli_fetch_assoc($result_quote_details);
    $quoteid = $quote_data['quoteid'];
    mysqli_free_result($result_quote_details);

    $adults_from_quote = (int)($quote_data['adults'] ?? 0);
    $children_from_quote = (int)($quote_data['children'] ?? 0);
    $infants_from_quote = (int)($quote_data['infants'] ?? 0);
    $single_rooms_for_cost = 0; $double_rooms_for_cost = 0; $triple_rooms_for_cost = 0; $child_no_bed_for_cost = 0;
    $sql_rooms_cost = "SELECT meta_key, meta_value FROM vtiger_itinerary WHERE quoteid = '$quoteid' AND meta_key IN ('single_rooms', 'double_rooms', 'triple_rooms', 'child_without_bed') GROUP BY meta_key";
    $result_rooms_cost = mysqli_query($dbConn, $sql_rooms_cost);
    if ($result_rooms_cost) {
        while ($row_room_cost = mysqli_fetch_assoc($result_rooms_cost)) {
            if ($row_room_cost['meta_key'] == 'single_rooms') $single_rooms_for_cost = (int)($row_room_cost['meta_value'] ?? 0);
            elseif ($row_room_cost['meta_key'] == 'double_rooms') $double_rooms_for_cost = (int)($row_room_cost['meta_value'] ?? 0);
            elseif ($row_room_cost['meta_key'] == 'triple_rooms') $triple_rooms_for_cost = (int)($row_room_cost['meta_value'] ?? 0);
            elseif ($row_room_cost['meta_key'] == 'child_without_bed') $child_no_bed_for_cost = (int)($row_room_cost['meta_value'] ?? 0);
        }
    }

    $result_supplier_lines = null;
    if (!empty($supplierNameFromAI) && $supplierNameFromAI !== 'N/A') {
        $matchedDbVendorName = findBestVendorMatch($supplierNameFromAI, $dbVendorSearchMap);
        if ($matchedDbVendorName) {
            $escapedMatchedName = mysqli_real_escape_string($dbConn, $matchedDbVendorName);
            $vendorid = null;
            $sql_vendor_id = "SELECT vendorid FROM vtiger_vendor WHERE vendorname = '{$escapedMatchedName}' UNION SELECT vendorid FROM tdu_vendors WHERE vendorname = '{$escapedMatchedName}' UNION SELECT vendorid FROM vtiger_vendor_custom WHERE vendorname = '{$escapedMatchedName}' LIMIT 1";
            $result_vendor_id = mysqli_query($dbConn, $sql_vendor_id);
            if ($result_vendor_id && mysqli_num_rows($result_vendor_id) > 0) { $vendorid = mysqli_fetch_assoc($result_vendor_id)['vendorid']; }
            if ($vendorid) {
                $sql_to_execute = "SELECT vi.productid, vi.cf_928 AS item_type, vi.quantity AS vtiger_line_item_quantity, vi.checkin, vi.checkout, vps.sale_price AS cost_price_adult, vps.sale_price_child AS cost_price_child, vps.sale_price_infant AS cost_price_infant, vps.sale_price_single AS cost_price_single, vps.sale_price_double AS cost_price_double, vps.sale_price_triple AS cost_price_triple, vps.sale_price_child_no_bed AS cost_price_child_no_bed FROM vtiger_inventoryproductrel vi LEFT JOIN vtiger_products_saleprice vps ON vi.id = vps.quoteid AND vi.sequence_no = vps.sequence_no AND vps.subquoteid <= 1 WHERE vi.id = '{$quoteid}' AND vi.vendorid = '{$vendorid}'";
                $result_supplier_lines = mysqli_query($dbConn, $sql_to_execute);
            }
        }
    }

    if (!$result_supplier_lines || mysqli_num_rows($result_supplier_lines) == 0) {
        if (!empty($confirmationNum) && $confirmationNum !== 'N/A') {
            $sql_itinerary = "SELECT sequence_no FROM vtiger_itinerary WHERE meta_key = 'confirmation_number' AND TRIM(meta_value) = '{$escapedConfirmationNum}' AND quoteid = '{$quoteid}' LIMIT 1";
            $result_itinerary = mysqli_query($dbConn, $sql_itinerary);
            if ($result_itinerary && mysqli_num_rows($result_itinerary) > 0) {
                $sequence_no = mysqli_fetch_assoc($result_itinerary)['sequence_no'];
                $sql_to_execute = "SELECT vi.productid, vi.cf_928 AS item_type, vi.quantity AS vtiger_line_item_quantity, vi.checkin, vi.checkout, vps.sale_price AS cost_price_adult, vps.sale_price_child AS cost_price_child, vps.sale_price_infant AS cost_price_infant, vps.sale_price_single AS cost_price_single, vps.sale_price_double AS cost_price_double, vps.sale_price_triple AS cost_price_triple, vps.sale_price_child_no_bed AS cost_price_child_no_bed FROM vtiger_inventoryproductrel vi LEFT JOIN vtiger_products_saleprice vps ON vi.id = vps.quoteid AND vi.sequence_no = vps.sequence_no AND vps.subquoteid <= 1 WHERE vi.id = '{$quoteid}' AND vi.sequence_no = '{$sequence_no}' LIMIT 1";
                $result_supplier_lines = mysqli_query($dbConn, $sql_to_execute);
            }
        }
    }

    $totalSupplierCost = 0.0;
    if ($result_supplier_lines && mysqli_num_rows($result_supplier_lines) > 0) {
        while ($line_row = mysqli_fetch_assoc($result_supplier_lines)) {
            $item_cost_for_this_line = 0.0;
            $item_type = $line_row['item_type'] ?? 'Unknown';
            if ($item_type == 'Hotel') {
                $nights = 1;
                if (!empty($line_row['checkin']) && !empty($line_row['checkout']) && $line_row['checkin'] !== '0000-00-00' && $line_row['checkout'] !== '0000-00-00') {
                    try { $cin = new DateTime($line_row['checkin']); $cout = new DateTime($line_row['checkout']); $nights = max(1, $cin->diff($cout)->days); } catch (Exception $e) { $nights = 1; }
                }
                $item_cost_for_this_line += ($single_rooms_for_cost * (float)($line_row['cost_price_single'] ?? 0) + $double_rooms_for_cost * (float)($line_row['cost_price_double'] ?? 0) + $triple_rooms_for_cost * (float)($line_row['cost_price_triple'] ?? 0) + $child_no_bed_for_cost * (float)($line_row['cost_price_child_no_bed'] ?? 0)) * $nights;
            } elseif ($item_type == 'Transfers' || $item_type == 'Guide') {
                $item_cost_for_this_line = (float)($line_row['cost_price_adult'] ?? 0);
            } else {
                $item_cost_for_this_line += ($adults_from_quote * (float)($line_row['cost_price_adult'] ?? 0) + $children_from_quote * (float)($line_row['cost_price_child'] ?? 0) + $infants_from_quote * (float)($line_row['cost_price_infant'] ?? 0));
            }
            $totalSupplierCost += $item_cost_for_this_line;
        }
        mysqli_free_result($result_supplier_lines);
    } else {
        return 'Line N/F';
    }

    return round($totalSupplierCost, 2);
}

function extractStatementDataAI(string $statementText): array {
    $statementText = mb_convert_encoding($statementText, 'UTF-8', 'UTF-8');
    $default_response = ['success' => false, 'error' => 'AI processing not initiated.', 'data' => [], 'raw_ai_json_text' => null ];
    if (empty(trim($statementText))) {
        $default_response['error'] = 'Input statement text is empty after cleaning.';
        return $default_response;
    }
    $prompt = "
    Analyze the following 'Statement Text'.
    Extract all applicable line items.
    Fields to extract for EACH item:
        'supplier_name', 'supplier_email_domain', 'tdu_class_number', 'passenger_name', 'confirmation_number',
        'invoice_number', 'service_date', 'invoice_amount', 'description'.
    Output Instructions:
        Return a single, valid JSON array. Each element an object. If not found, use 'N/A'.
        For 'supplier_email_domain', extract the email domain (e.g., 'example.com') from any supplier email address found.
        For 'invoice_amount', extract the amount for each separate item, don't add them together
        class number starts with a TDU and maybe ends in a G or not depending if its FIT or Group and there will never be whitespace in between example: TDU12345G(correct), TDU88333(correct), TDU 12345(wrong), TDU 12345 G(wrong).
        Our company is Turtle Down Under, it is exempt as supplier, never extract it as a supplier.
    Example:
        [{\"supplier_name\": \"Touring Melbourne Group\", \"supplier_email_domain\": \"touringmelbourne.com.au\", \"tdu_class_number\": \"TDU27014G\", ...}]
    Statement Text:\n" . $statementText;

    $apiUrl = apiURL_Flash();
    if (!$apiUrl) {
        $default_response['error'] = 'AI API URL is not configured.';
        return $default_response;
    }
    //error_log($prompt);
    $payload = ['contents' => [['parts' => [['text' => $prompt]]]], 'generationConfig' => ['temperature' => 0.2, 'response_mime_type' => "application/json", 'maxOutputTokens' => 28192]];
    $headers = ['Content-Type: application/json'];
    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [CURLOPT_RETURNTRANSFER => true, CURLOPT_POST => true, CURLOPT_POSTFIELDS => json_encode($payload), CURLOPT_HTTPHEADER => $headers, CURLOPT_TIMEOUT => 300]);
    $response_body = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    if ($curlError || $httpCode != 200) {
        $default_response['error'] = "AI API call failed (HTTP $httpCode). " . $curlError;
        return $default_response;
    }
    //error_log($responseData);
    $responseData = json_decode($response_body, true);
    if (isset($responseData['promptFeedback']['blockReason'])) {
        $default_response['error'] = "AI: Content blocked due to: " . ($responseData['promptFeedback']['blockReason'] ?? 'Unknown reason');
        return $default_response;
    }

    $extractedJsonText = null;
    if (isset($responseData['candidates'][0]['content']['parts'][0]['text'])) {
        $extractedJsonText = trim($responseData['candidates'][0]['content']['parts'][0]['text']);
        $default_response['raw_ai_json_text'] = $extractedJsonText;
    } else {
        $default_response['error'] = 'AI API: No text part in response.';
        return $default_response;
    }

    $parsedData = json_decode($extractedJsonText, true);
    if (json_last_error() === JSON_ERROR_NONE && is_array($parsedData)) {
        $default_response['success'] = true;
        $default_response['error'] = null;
        $default_response['data'] = $parsedData;
    } else {
        $default_response['error'] = 'Failed to parse AI JSON response. Error: ' . json_last_error_msg();
    }

    return $default_response;
}

/**
 * FINAL VERSION (V3): Second AI call with MAXIMUM context, including the Class on each line.
 *
 * @param array $statementItem The original line item extracted from the statement.
 * @param array $candidateLines The array of potential matching QBO bill lines.
 * @return string|null The Line ID of the best match, or null if undecided.
 */
function resolveAmbiguityWithAI(array $statementItem, array $candidateLines): ?string {
    $prompt_context = "You are an expert accounting assistant. Your task is to resolve an ambiguity with high precision. I have one item from a supplier statement and several potential matching bill lines from my accounting software (QuickBooks). All candidates provided are already confirmed to have the correct 'Class'. Based on all the other context provided, you must select the single best match.

Analyze all details. The amount, description, invoice number, and dates are primary indicators. Secondary indicators like the specific 'Product/Service' or 'Expense Account' on the line are also very important clues.";

    // Provide the AI with every piece of data from the source statement item.
    $statement_details = "STATEMENT ITEM (Source of Truth):\n";
    $statement_details .= "- Supplier: " . ($statementItem['supplier_name'] ?? 'N/A') . "\n";
    $statement_details .= "- TDU Class Number: " . ($statementItem['tdu_class_number'] ?? 'N/A') . "\n";
    $statement_details .= "- Passenger: " . ($statementItem['passenger_name'] ?? 'N/A') . "\n";
    $statement_details .= "- Service Date: " . ($statementItem['service_date'] ?? 'N/A') . "\n";
    $statement_details .= "- Confirmation #: " . ($statementItem['confirmation_number'] ?? 'N/A') . "\n";
    $statement_details .= "- Invoice #: " . ($statementItem['invoice_number'] ?? 'N/A') . "\n";
    $statement_details .= "- Description: " . ($statementItem['description'] ?? 'N/A') . "\n";
    $statement_details .= "- Amount: " . ($statementItem['invoice_amount'] ?? 'N/A') . "\n";

    // Provide a rich, detailed breakdown of every candidate bill and line from QBO.
    $candidate_details = "CANDIDATE QUICKBOOKS BILL LINES (All have the correct Class):\n";
    foreach ($candidateLines as $index => $match) {
        $candidate_details .= "-------------------------------------\n";
        $candidate_details .= "Candidate " . ($index + 1) . ":\n";
        $candidate_details .= "  BILL-LEVEL DETAILS:\n";
        $candidate_details .= "    - Bill Doc #: " . ($match['bill']->DocNumber ?? 'N/A') . "\n";
        $candidate_details .= "    - Bill Date: " . ($match['bill']->TxnDate ?? 'N/A') . "\n";
        $candidate_details .= "    - Bill Total Amount: " . ($match['bill']->TotalAmt ?? 'N/A') . "\n";
        $candidate_details .= "    - Bill Private Note: " . ($match['bill']->PrivateNote ?? 'N/A') . "\n";
        $candidate_details .= "  LINE-LEVEL DETAILS (This is the specific line to consider):\n";
        $candidate_details .= "    - Line ID: " . ($match['line']->Id) . "\n"; // This is the critical ID to return
        $candidate_details .= "    - Line Amount: " . ($match['line']->Amount ?? 'N/A') . "\n";
        $candidate_details .= "    - Line Description: " . ($match['line']->Description ?? 'N/A') . "\n";

        // Add deeper details based on the line type, including the class name.
        if (isset($match['line']->ItemBasedExpenseLineDetail)) {
            $candidate_details .= "    - Line Type: Item-Based\n";
            $candidate_details .= "    - Line Class: " . ($match['line']->ItemBasedExpenseLineDetail->ClassRef ?? 'N/A') . "\n";
            $candidate_details .= "    - Product/Service: " . ($match['line']->ItemBasedExpenseLineDetail->ItemRef ?? 'N/A') . "\n";
            $candidate_details .= "    - Quantity: " . ($match['line']->ItemBasedExpenseLineDetail->Qty ?? 'N/A') . "\n";
            $candidate_details .= "    - Unit Price: " . ($match['line']->ItemBasedExpenseLineDetail->UnitPrice ?? 'N/A') . "\n";
        } elseif (isset($match['line']->AccountBasedExpenseLineDetail)) {
            $candidate_details .= "    - Line Type: Account-Based\n";
            $candidate_details .= "    - Line Class: " . ($match['line']->AccountBasedExpenseLineDetail->ClassRef ?? 'N/A') . "\n";
            $candidate_details .= "    - Expense Account: " . ($match['line']->AccountBasedExpenseLineDetail->AccountRef ?? 'N/A') . "\n";
        }
    }
    $candidate_details .= "-------------------------------------\n";


    $instructions = "OUTPUT INSTRUCTIONS:\nReturn a single, valid JSON object with one key: 'best_match_line_id'. The value MUST be the Line ID of the best candidate from the list above. If you cannot determine a clear best match with very high confidence, the value MUST be null. Do not explain your reasoning. Your entire response must be only the JSON object.";

    $final_prompt = $prompt_context . "\n\n" . $statement_details . "\n" . $candidate_details . "\n" . $instructions;

    $apiUrl = apiURL_Flash();
    if (!$apiUrl) {
        error_log("Disambiguation AI Error: API URL not configured.");
        return null;
    }
    //error_log($final_prompt);
    $payload = ['contents' => [['parts' => [['text' => $final_prompt]]]], 'generationConfig' => ['temperature' => 0.1, 'response_mime_type' => "application/json"]];
    $headers = ['Content-Type: application/json'];
    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [CURLOPT_RETURNTRANSFER => true, CURLOPT_POST => true, CURLOPT_POSTFIELDS => json_encode($payload), CURLOPT_HTTPHEADER => $headers, CURLOPT_TIMEOUT => 120]);
    
    $response_body = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    if ($curlError || $httpCode != 200) {
        error_log("Disambiguation AI API call failed (HTTP $httpCode). " . $curlError);
        return null;
    }
    
    $responseData = json_decode($response_body, true);
    $aiJsonText = $responseData['candidates'][0]['content']['parts'][0]['text'] ?? null;
    if (!$aiJsonText) {
        error_log("Disambiguation AI: No text part in response.");
        return null;
    }
    
    $parsedResponse = json_decode($aiJsonText, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("Disambiguation AI: Failed to parse JSON response: " . $aiJsonText);
        return null;
    }

    // Return the chosen line ID, or null if the AI couldn't decide
    return $parsedResponse['best_match_line_id'] ?? null;
}

// =================================================================================
// SECTION 2: MAIN PROCESSING LOGIC
// =================================================================================

create_reconciliation_table_if_not_exists($conn);

error_log("CRON START: Processing file '{$filename_to_process}'");

$stmt_insert = mysqli_prepare($conn, "INSERT INTO tdu_qbo_reconciliation (file_name, processing_status) VALUES (?, 'processing') ON DUPLICATE KEY UPDATE processing_status='processing', ai_raw_json=NULL, vtiger_quote_amounts_json=NULL, qbo_bills_json=NULL, error_message=NULL, processed_at=NULL");
mysqli_stmt_bind_param($stmt_insert, "s", $filename_to_process);
mysqli_stmt_execute($stmt_insert);
mysqli_stmt_close($stmt_insert);

try {
    // --- Initialize QBO Connection ---
    $qboUtil = null;
    if (file_exists($qboBaseConfigFile)) { require $qboBaseConfigFile; } else { throw new Exception("CRITICAL: QBO base config file not found."); }
    if ($qboBaseConfig) {
        $currentTokens = loadQBOTokens();
        $qboConfigForLibrary = $qboBaseConfig;
        if ($currentTokens && !empty($currentTokens['refresh_token'])) {
            $qboConfigForLibrary['refreshTokenKey'] = $currentTokens['refresh_token'];
            try {
                $oauth2LoginHelper = new OAuth2LoginHelper($qboBaseConfig['ClientID'], $qboBaseConfig['ClientSecret']);
                $refreshedAccessTokenObj = $oauth2LoginHelper->refreshAccessTokenWithRefreshToken($qboConfigForLibrary['refreshTokenKey']);
                if (saveQBOTokens($refreshedAccessTokenObj->getAccessToken(), $refreshedAccessTokenObj->getRefreshToken())) {
                    $currentTokens['access_token'] = $refreshedAccessTokenObj->getAccessToken();
                }
            } catch (Exception $e) { error_log("CRON WARNING: Could not refresh QBO token. " . $e->getMessage()); }
        }
        if ($currentTokens && !empty($currentTokens['access_token'])) {
            $qboConfigForLibrary['accessTokenKey'] = $currentTokens['access_token'];
            $qboConfigForLibrary['refreshTokenKey'] = $currentTokens['refresh_token'] ?? null;
            $qboUtil = new QBOUtilityLibrary($qboConfigForLibrary);
        } else { throw new Exception("CRITICAL: No valid QBO access token available."); }
    }

    // --- Build Vendor Search Maps ---
    $vendorSearchMap = []; $allQboVendors = []; $startPosition = 1;
    while (true) {
        $vendorsBatch = $qboUtil->getDataService()->Query("SELECT * FROM Vendor WHERE Active = true STARTPOSITION {$startPosition} MAXRESULTS 1000");
        if (empty($vendorsBatch)) break;
        $allQboVendors = array_merge($allQboVendors, $vendorsBatch);
        $startPosition += count($vendorsBatch);
    }
    $vendorSearchMap = buildVendorSearchMap($allQboVendors);

    $dbVendorSearchMap = [];
    $sql_db_vendors = "SELECT vendorname FROM vtiger_vendor WHERE vendorname IS NOT NULL AND vendorname != '' UNION SELECT vendorname FROM tdu_vendors WHERE vendorname IS NOT NULL AND vendorname != '' UNION SELECT vendorname FROM vtiger_vendor_custom WHERE vendorname IS NOT NULL AND vendorname != ''";
    $result_db_vendors = mysqli_query($conn, $sql_db_vendors);
    $dbVendorObjects = [];
    if ($result_db_vendors) {
        while ($row = mysqli_fetch_assoc($result_db_vendors)) { $dbVendorObjects[] = (object)['DisplayName' => $row['vendorname']]; }
        mysqli_free_result($result_db_vendors);
    }
    $dbVendorSearchMap = buildVendorSearchMap($dbVendorObjects);
    /*
    // --- Extract Text from File ---
    $raw_file_content = file_get_contents($full_file_path);
    if ($raw_file_content === false) throw new Exception("Could not read file content.");
    $friendly_file_type = getFileTypeFromString($raw_file_content);
    $extracted_text = null;
    switch ($friendly_file_type) {
        case 'PDF Document': $extracted_text = extractTextFromPDF_withLayout($raw_file_content); break;
        case 'Word Document (Modern Format - DOCX)': case 'Word Document (Old Format - DOC)': $extracted_text = extractTextfromWordDoc($raw_file_content); break;
        case 'Excel Spreadsheet (Modern Format - XLSX)': case 'Excel Spreadsheet (Old Format - XLS)': $extracted_text = extractTextFromExcelSheet($raw_file_content); break;
        default: $extracted_text = (in_array($friendly_file_type, ['JPEG Image', 'PNG Image'])) ? null : $raw_file_content; break;
    }
    if (empty(trim($extracted_text))) throw new Exception("Could not extract any text from the file.");
    
    // --- AI Data Extraction ---
    $ai_extracted_data_list = extractStatementDataAI(trim($extracted_text));
    */
    $ai_extracted_data_list = extractStatementDataAI_fromFile($full_file_path);
    if (!$ai_extracted_data_list['success'] || !is_array($ai_extracted_data_list['data'])) { throw new Exception("AI processing failed. Error: " . ($ai_extracted_data_list['error'] ?? 'Unknown AI error')); }
    
    // --- Main Processing Loop ---
    $vtiger_quote_amounts_to_store = [];
    $qbo_bills_to_store = [];
    $vendorBillCache = [];

    // MODIFIED: Loop with a reference (&) to allow direct modification of the $item array.
    foreach ($ai_extracted_data_list['data'] as $index => &$item) {

        // =================================================================
        // MODIFIED: STEP 1: FIND AND FINALIZE THE SUPPLIER NAME FIRST
        // =================================================================
        $supplierNameFromAI = $item['supplier_name'] ?? 'N/A';
        $emailDomainFromAI = $item['supplier_email_domain'] ?? 'N/A';
        
        $qboVendorObject = null;
        $matchMethod = '';

        if ($qboUtil) {
            // Try to match by name similarity first
            $matchedQboVendorName = findBestVendorMatch($supplierNameFromAI, $vendorSearchMap);
            if ($matchedQboVendorName) {
                $qboVendorObject = $qboUtil->queryEntityByField('Vendor', 'DisplayName', $matchedQboVendorName);
                if ($qboVendorObject) {
                    $matchMethod = 'Name';
                    // MODIFIED: Ensure the item has the official capitalized/formatted name
                    $item['supplier_name'] = $qboVendorObject->DisplayName; 
                    error_log("CRON INFO: Vendor '{$qboVendorObject->DisplayName}' found by name match.");
                }
            }
            
            // If name match failed, try domain match
            if (!$qboVendorObject && $emailDomainFromAI !== 'N/A' && !empty(trim($emailDomainFromAI))) {
                $qboVendorObject = $qboUtil->findVendorByEmailDomain(trim($emailDomainFromAI));
                if ($qboVendorObject) {
                    $matchMethod = 'Domain';
                    $originalAIName = $supplierNameFromAI;
                    // MODIFIED: Update the item data array directly
                    $item['supplier_name'] = $qboVendorObject->DisplayName; 
                    error_log("CRON INFO: Vendor '{$qboVendorObject->DisplayName}' found by domain match. Original AI name was '{$originalAIName}'.");
                }
            }
        }

        // =================================================================
        // MODIFIED: STEP 2: PERFORM VTIGER LOOKUP USING THE FINALIZED SUPPLIER NAME
        // =================================================================
        $vtiger_quote_amounts_to_store[$index] = getSupplierQuoteCostAmount($conn, $item, $dbVendorSearchMap);
        
        // =================================================================
        // MODIFIED: STEP 3: PROCEED WITH QBO BILL MATCHING
        // =================================================================
        $qbo_data_for_row = null;
        $tduClassNumberFromAI = $item['tdu_class_number'] ?? 'N/A';

        if (!$qboUtil) {
            $qbo_data_for_row = ['status' => 'error', 'message' => 'QBO Conn Err'];
        } else {
            try {
                // The vendor object is already found. We can proceed directly to bill lookup.
                if (!$qboVendorObject) {
                    $qbo_data_for_row = ['status' => 'vendor_not_found'];
                } elseif ($tduClassNumberFromAI === 'N/A') {
                    $qbo_data_for_row = ['status' => 'error', 'message' => 'Missing TDU Class #'];
                } else {
                    // All subsequent logic uses the single, correct $qboVendorObject
                    $vendorId = $qboVendorObject->Id;
                    if (!isset($vendorBillCache[$vendorId])) {
                        error_log("CRON INFO: Caching bills for Vendor ID: {$vendorId} ('{$qboVendorObject->DisplayName}')");
                        $vendorBillCache[$vendorId] = $qboUtil->getDataService()->Query("SELECT * FROM Bill WHERE VendorRef = '{$vendorId}' MAXRESULTS 1000");
                    }
                    
                    $candidateBills = $vendorBillCache[$vendorId] ?? [];
                    $classEntity = $qboUtil->queryEntityByField('Class', 'Name', $tduClassNumberFromAI);
                    $matchingBillsAndLines = [];

                    if ($classEntity && !empty($candidateBills)) {
                        $targetClassId = $classEntity->Id;
                        foreach ($candidateBills as $bill) {
                            if (empty($bill->Line)) continue;
                            $actualLines = is_array($bill->Line) ? $bill->Line : [$bill->Line];
                            foreach ($actualLines as $line) {
                                $lineClassRefValue = null;
                                if (isset($line->ItemBasedExpenseLineDetail->ClassRef)) { $lineClassRefValue = (string) $line->ItemBasedExpenseLineDetail->ClassRef; }
                                elseif (isset($line->AccountBasedExpenseLineDetail->ClassRef)) { $lineClassRefValue = (string) $line->AccountBasedExpenseLineDetail->ClassRef; }
                                if ($lineClassRefValue !== null && trim($lineClassRefValue) === trim((string)$targetClassId)) {
                                    $matchingBillsAndLines[] = ['bill' => $bill, 'line' => $line];
                                }
                            }
                        }
                    }

                    // 3. The Three-Tier Decision Logic (This block is unchanged)
                    if (empty($matchingBillsAndLines)) {
                        $qbo_data_for_row = ['status' => 'bill_line_not_found'];
                    } elseif (count($matchingBillsAndLines) === 1) {
                        $bestMatch = reset($matchingBillsAndLines);
                        $qbo_data_for_row = [
                            'status' => 'matched',
                            'bill_object' => $bestMatch['bill'],
                            'matched_line_id' => $bestMatch['line']->Id,
                            'matched_line_amount' => $bestMatch['line']->Amount ?? null,
                            'match_method' => $matchMethod . ' (Unique Class Match)'
                        ];
                    } else {
                        error_log("CRON INFO: Ambiguity detected for TDU '{$tduClassNumberFromAI}'. Calling disambiguation AI...");
                        $chosenLineId = resolveAmbiguityWithAI($item, $matchingBillsAndLines);
                        
                        $aiResolvedMatch = null;
                        if ($chosenLineId !== null) {
                            foreach ($matchingBillsAndLines as $candidateMatch) {
                                if ($candidateMatch['line']->Id == $chosenLineId) {
                                    $aiResolvedMatch = $candidateMatch;
                                    break;
                                }
                            }
                        }

                        if ($aiResolvedMatch) {
                            error_log("CRON INFO: Disambiguation AI successfully selected Line ID: {$chosenLineId}");
                            $qbo_data_for_row = [
                                'status' => 'matched',
                                'bill_object' => $aiResolvedMatch['bill'],
                                'matched_line_id' => $aiResolvedMatch['line']->Id,
                                'matched_line_amount' => $aiResolvedMatch['line']->Amount ?? null,
                                'match_method' => 'AI Disambiguation'
                            ];
                        } else {
                            error_log("CRON INFO: Disambiguation AI could not decide. Flagging for manual review.");
                            $potential_matches_data = [];
                            foreach ($matchingBillsAndLines as $match) {
                                $potential_matches_data[] = [
                                    'bill_id' => $match['bill']->Id,
                                    'sync_token' => $match['bill']->SyncToken,
                                    'line_id' => $match['line']->Id,
                                    'amount' => $match['line']->Amount ?? 'N/A',
                                    'description' => $match['line']->Description ?? 'N/A',
                                    'doc_number' => $match['bill']->DocNumber ?? 'N/A'
                                ];
                            }
                            $qbo_data_for_row = [
                                'status' => 'needs_review',
                                'reason' => 'multiple_lines_match_class',
                                'potential_matches' => $potential_matches_data
                            ];
                        }
                    }
                }
            } catch (Exception $e) {
                // MODIFIED: Use the finalized supplier name for more accurate error logging.
                $finalSupplierNameForError = $item['supplier_name'] ?? $supplierNameFromAI;
                $qbo_data_for_row = ['status' => 'error', 'message' => substr($e->getMessage(), 0, 150) . '...'];
                error_log("CRON QBO ERROR for supplier '{$finalSupplierNameForError}': " . $e->getMessage());
            }
        }
        $qbo_bills_to_store[$index] = $qbo_data_for_row;
    }
    
    // MODIFIED: Unset the reference to prevent potential bugs after the loop.
    unset($item);

    // --- Save final results to the database ---
    $vtiger_json_to_store = json_encode($vtiger_quote_amounts_to_store);
    $qbo_json_to_store = json_encode($qbo_bills_to_store);
    
    // MODIFIED: Re-encode the data list that may have been modified in the loop to store the corrected supplier name.
    $final_ai_data_to_store = json_encode($ai_extracted_data_list['data']);

    $update_stmt = mysqli_prepare($conn, "UPDATE tdu_qbo_reconciliation SET processing_status='completed', ai_raw_json=?, vtiger_quote_amounts_json=?, qbo_bills_json=?, processed_at=CURRENT_TIMESTAMP WHERE file_name=?");
    
    // MODIFIED: Use the new variable for storing the corrected data in the first parameter.
    mysqli_stmt_bind_param($update_stmt, "ssss", $final_ai_data_to_store, $vtiger_json_to_store, $qbo_json_to_store, $filename_to_process);
    mysqli_stmt_execute($update_stmt);
    mysqli_stmt_close($update_stmt);

    error_log("CRON SUCCESS: Finished processing '{$filename_to_process}'");

} catch (Exception $e) {
    // --- Handle fatal errors during processing ---
    $error_message = $e->getMessage();
    error_log("CRON FAILED for file '{$filename_to_process}': " . $error_message);
    $error_stmt = mysqli_prepare($conn, "UPDATE tdu_qbo_reconciliation SET processing_status='error', error_message=? WHERE file_name=?");
    mysqli_stmt_bind_param($error_stmt, "ss", $error_message, $filename_to_process);
    mysqli_stmt_execute($error_stmt);
    mysqli_stmt_close($error_stmt);
    die();
}
?>