<?php
// cronjob_qbo_reverse_sync.php
// Final production script to sync paid bills from QBO to the local dashboard.
// Version 3: Fixes object parsing for BillPayment Line items.

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
set_time_limit(0);

// --- Essential Includes ---
require_once __DIR__ . '/../dbconn.php';
require_once __DIR__ . '/../qbo_bootstrap.php';

// --- DATABASE SETUP FUNCTION ---
function ensureSyncTablesExist(mysqli $dbConn) {
    $paymentLogTable = 'tdu_qbo_payment_log';
    if (mysqli_num_rows(mysqli_query($dbConn, "SHOW TABLES LIKE '{$paymentLogTable}'")) == 0) {
        echo "INFO: Table '{$paymentLogTable}' not found. Creating it...\n";
        $sqlCreateLog = "CREATE TABLE `{$paymentLogTable}` (
                          `id` INT(11) NOT NULL AUTO_INCREMENT,
                          `qbo_bill_id` VARCHAR(50) NOT NULL,
                          `qbo_bill_payment_id` VARCHAR(50) NOT NULL,
                          `vtiger_quote_id` INT(11) NOT NULL,
                          `vtiger_vendor_id` INT(11) NOT NULL,
                          `payment_date` DATE NOT NULL,
                          `payment_amount` DECIMAL(12, 2) NOT NULL,
                          `sync_timestamp` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
                          PRIMARY KEY (`id`),
                          UNIQUE KEY `qbo_bill_payment_id` (`qbo_bill_payment_id`),
                          KEY `qbo_bill_id` (`qbo_bill_id`),
                          KEY `vtiger_quote_id` (`vtiger_quote_id`)
                        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";
        if (!mysqli_query($dbConn, $sqlCreateLog)) { die("CRITICAL: Could not create table '{$paymentLogTable}'. Error: " . mysqli_error($dbConn) . "\n"); }
        echo "SUCCESS: Table '{$paymentLogTable}' created.\n";
    }

    $syncJobsTable = 'tdu_sync_jobs';
    if (mysqli_num_rows(mysqli_query($dbConn, "SHOW TABLES LIKE '{$syncJobsTable}'")) == 0) {
        echo "INFO: Table '{$syncJobsTable}' not found. Creating it...\n";
        $sqlCreateJobs = "CREATE TABLE `{$syncJobsTable}` (
                              `id` INT(11) NOT NULL AUTO_INCREMENT,
                              `job_name` VARCHAR(100) NOT NULL,
                              `last_successful_run` DATETIME NOT NULL,
                              `status_message` VARCHAR(255) NULL,
                              `last_attempt` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                              PRIMARY KEY (`id`),
                              UNIQUE KEY `job_name` (`job_name`)
                            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";
        if (!mysqli_query($dbConn, $sqlCreateJobs)) { die("CRITICAL: Could not create table '{$syncJobsTable}'. Error: " . mysqli_error($dbConn) . "\n"); }
        echo "SUCCESS: Table '{$syncJobsTable}' created.\n";

        $initialTimestamp = '2025-06-18 00:00:00';
        $sqlInsertInitial = "INSERT INTO `{$syncJobsTable}` (job_name, last_successful_run) VALUES ('qbo_paid_bill_sync', '{$initialTimestamp}')";
        if (!mysqli_query($dbConn, $sqlInsertInitial)) { die("CRITICAL: Could not insert initial record into '{$syncJobsTable}'. Error: " . mysqli_error($dbConn) . "\n"); }
        echo "INFO: Initial record for 'qbo_paid_bill_sync' job created.\n";
    }
}

// =========================================================================
// --- PRODUCTION CRON JOB LOGIC ---
// =========================================================================

echo "--- [".date('Y-m-d H:i:s')."] Starting QBO Paid Bill Sync Cron Job ---\n";

global $conn, $qboUtil;

if (!$conn) { die("CRITICAL: Database connection failed.\n"); }
if (!$qboUtil) { die("CRITICAL: QBOUtilityLibrary not initialized.\n"); }

ensureSyncTablesExist($conn);

$jobStartTime = new DateTime("now", new DateTimeZone("UTC"));
$lastRunTimestamp = '2024-01-01T00:00:00Z'; 

$sql_get_ts = "SELECT last_successful_run FROM tdu_sync_jobs WHERE job_name = 'qbo_paid_bill_sync' LIMIT 1";
$result_ts = mysqli_query($conn, $sql_get_ts);
if ($result_ts && mysqli_num_rows($result_ts) > 0) {
    $lastRunDateTime = new DateTime(mysqli_fetch_assoc($result_ts)['last_successful_run'], new DateTimeZone("UTC"));
    $lastRunTimestamp = $lastRunDateTime->format('Y-m-d\TH:i:s\Z');
}
echo "INFO: Checking for payments modified since: {$lastRunTimestamp}\n";

try {
    // --- Fetch Changed BillPayments from QBO ---
    $entitiesToQuery = ['BillPayment'];
    $dataService = $qboUtil->getDataService();
    $cdcResponse = $dataService->CDC($entitiesToQuery, $lastRunTimestamp);
    
    $changedBillPayments = $cdcResponse->entities['BillPayment'] ?? [];

    if (empty($changedBillPayments)) {
        echo "INFO: No new or updated BillPayments found since the last run.\n";
    } else {
        echo "SUCCESS: Found " . count($changedBillPayments) . " new/updated BillPayments to process.\n";
        
        foreach ($changedBillPayments as $payment) {
            
            mysqli_begin_transaction($conn);

            try {
                // --- Extract and Link Data ---
                $qbo_bill_payment_id = $payment->Id;
                $payment_amount = (float)($payment->TotalAmt ?? 0);
                $payment_date = (new DateTime($payment->TxnDate))->format('Y-m-d');
                $linked_bill_id = null;

                // =========================================================================
                // --- THE FIX IS HERE ---
                // Safely navigate the object structure without assuming arrays.
                // The SDK often returns a single line as an object, not an array with one element.
                if (isset($payment->Line, $payment->Line->LinkedTxn) &&
                    $payment->Line->LinkedTxn->TxnType === 'Bill') 
                {
                    $linked_bill_id = $payment->Line->LinkedTxn->TxnId;
                }
                // =========================================================================


                if (!$linked_bill_id || $payment_amount <= 0) {
                    echo "WARNING: Skipping QBO Payment ID {$qbo_bill_payment_id} - missing linked Bill ID or zero amount.\n";
                    mysqli_rollback($conn);
                    continue;
                }
                
                $sql_link = "SELECT source_identifier, organisation_name FROM tdu_qbo_sync_history WHERE qbo_entity_id = ? AND qbo_entity_type = 'Bill' LIMIT 1";
                $stmt_link = mysqli_prepare($conn, $sql_link);
                mysqli_stmt_bind_param($stmt_link, "s", $linked_bill_id);
                mysqli_stmt_execute($stmt_link);
                $link_data = mysqli_fetch_assoc(mysqli_stmt_get_result($stmt_link));
                mysqli_stmt_close($stmt_link);
                
                if (!$link_data) {
                    echo "WARNING: Skipping QBO Payment ID {$qbo_bill_payment_id} - could not find a matching Bill (ID: {$linked_bill_id}) in local sync history.\n";
                    mysqli_rollback($conn);
                    continue;
                }
                
                $vendor_name = $link_data['organisation_name'];
                $quote_no = $link_data['source_identifier'];
                
                echo "-> Processing Payment ID: {$qbo_bill_payment_id} | Amount: {$payment_amount} | Quote: {$quote_no} | Vendor: {$vendor_name}\n";

                // --- Update Database ("Upsert" Logic) ---
                $sql_get_quote_id = "SELECT quoteid FROM vtiger_quotes WHERE quote_no = ? LIMIT 1";
                $stmt_q = mysqli_prepare($conn, $sql_get_quote_id);
                mysqli_stmt_bind_param($stmt_q, "s", $quote_no);
                mysqli_stmt_execute($stmt_q);
                $vtiger_quote_id = mysqli_fetch_assoc(mysqli_stmt_get_result($stmt_q))['quoteid'] ?? null;
                mysqli_stmt_close($stmt_q);

                $sql_get_vendor_id = "(SELECT vendorid FROM tdu_vendors WHERE vendorName = ?) UNION (SELECT vendorid FROM vtiger_vendor WHERE vendorname = ?) LIMIT 1";
                $stmt_v = mysqli_prepare($conn, $sql_get_vendor_id);
                mysqli_stmt_bind_param($stmt_v, "ss", $vendor_name, $vendor_name);
                mysqli_stmt_execute($stmt_v);
                $vtiger_vendor_id = mysqli_fetch_assoc(mysqli_stmt_get_result($stmt_v))['vendorid'] ?? null;
                mysqli_stmt_close($stmt_v);

                if (!$vtiger_quote_id || !$vtiger_vendor_id) {
                     throw new Exception("Could not resolve internal IDs for Quote '{$quote_no}' or Vendor '{$vendor_name}'.");
                }
                
                $sql_insert_log = "INSERT IGNORE INTO tdu_qbo_payment_log (qbo_bill_id, qbo_bill_payment_id, vtiger_quote_id, vtiger_vendor_id, payment_date, payment_amount) VALUES (?, ?, ?, ?, ?, ?)";
                $stmt_log = mysqli_prepare($conn, $sql_insert_log);
                mysqli_stmt_bind_param($stmt_log, "ssiisd", $linked_bill_id, $qbo_bill_payment_id, $vtiger_quote_id, $vtiger_vendor_id, $payment_date, $payment_amount);
                mysqli_stmt_execute($stmt_log);
                
                if (mysqli_stmt_affected_rows($stmt_log) === 0) {
                    throw new Exception("This payment (QBO ID: {$qbo_bill_payment_id}) has already been processed. Halting this item.");
                }
                mysqli_stmt_close($stmt_log);

                $sql_check = "SELECT paid_amount FROM vtiger_supplier_payment WHERE quoteid = ? AND vendorid = ? LIMIT 1";
                $stmt_check = mysqli_prepare($conn, $sql_check);
                mysqli_stmt_bind_param($stmt_check, "ii", $vtiger_quote_id, $vtiger_vendor_id);
                mysqli_stmt_execute($stmt_check);
                $existing_row = mysqli_fetch_assoc(mysqli_stmt_get_result($stmt_check));
                mysqli_stmt_close($stmt_check);

                if ($existing_row) {
                    $current_paid_amount = (float) $existing_row['paid_amount'];
                    $new_total_paid = $current_paid_amount + $payment_amount;
                    $sql_update = "UPDATE vtiger_supplier_payment SET paid_amount = ? WHERE quoteid = ? AND vendorid = ?";
                    $stmt_upsert = mysqli_prepare($conn, $sql_update);
                    mysqli_stmt_bind_param($stmt_upsert, "sii", $new_total_paid, $vtiger_quote_id, $vtiger_vendor_id);
                } else {
                    $sql_insert = "INSERT INTO vtiger_supplier_payment (quoteid, vendorid, paid_amount, total_amount) VALUES (?, ?, ?, '0')";
                    $stmt_upsert = mysqli_prepare($conn, $sql_insert);
                    mysqli_stmt_bind_param($stmt_upsert, "iid", $vtiger_quote_id, $vtiger_vendor_id, $payment_amount);
                }
                mysqli_stmt_execute($stmt_upsert);
                mysqli_stmt_close($stmt_upsert);

                mysqli_commit($conn);
                echo "   ...SUCCESS: Database updated.\n";

            } catch (Exception $e) {
                mysqli_rollback($conn);
                echo "   ...ERROR processing payment ID {$qbo_bill_payment_id}: " . $e->getMessage() . "\n";
            }
        }
    }

    // --- Update Master Timestamp ---
    $finalTimestamp = $jobStartTime->format('Y-m-d H:i:s');
    $sql_update_ts = "UPDATE tdu_sync_jobs SET last_successful_run = ? WHERE job_name = 'qbo_paid_bill_sync'";
    $stmt_ts = mysqli_prepare($conn, $sql_update_ts);
    mysqli_stmt_bind_param($stmt_ts, "s", $finalTimestamp);
    mysqli_stmt_execute($stmt_ts);
    mysqli_stmt_close($stmt_ts);
    
    echo "INFO: Job finished. Last successful run timestamp updated to: {$finalTimestamp}\n";

} catch (Exception $e) {
    echo "CRITICAL JOB FAILURE: An unrecoverable error occurred during the QBO API call. The master timestamp will NOT be updated. Error: " . $e->getMessage() . "\n";
    die(1);
}

echo "--- [".date('Y-m-d H:i:s')."] QBO Paid Bill Sync Cron Job Finished ---\n\n";
?>