<?php
// --- ai_test_processor.php ---

// Error logging setup for the script itself (PHP errors)
ini_set('display_errors', 1); // Display errors to screen for testing
ini_set('log_errors', 1);     // Log errors to a file
$errorLogPathWebTest = __DIR__ . '/web_ai_processor_errors.log'; // Path to error log file for this test page
ini_set('error_log', $errorLogPathWebTest);
error_reporting(E_ALL);      // Report all PHP errors

// --- Global array to capture logs for web display ---
$web_logs = [];

// --- Custom logger function for web display ---
function web_logger($message) {
    global $web_logs;
    $logEntry = "[" . date('Y-m-d H:i:s') . "] " . $message;
    $web_logs[] = $logEntry;
}

session_start(); // Start session for storing undo data

// --- Includes ---
$baseDir = __DIR__ . "/../"; // Assuming this test page is in 'cron' and includes are in parent
require_once $baseDir . "dbconn.php"; // Database connection
require_once $baseDir . "ai_keys.php";    // For apiURL_Flash()
require_once $baseDir . "file_extract_functions.php"; // For text extraction from attachments
require_once $baseDir . "ai_email_functions.php"; // For all AI logic

// --- Configuration (Copied from cron) ---
if (!defined('VTIGER_SUPPORT_STATUS_AI_PROCESSED')) define('VTIGER_SUPPORT_STATUS_AI_PROCESSED', 'AI Processed');
if (!defined('VTIGER_SUPPORT_STATUS_SPAM')) define('VTIGER_SUPPORT_STATUS_SPAM', 'SPAM');
$tempAiQuotesTableName = 'tdu_temp_ai_quotes';   // Name of the temporary table for AI quote data
$excludedSenderDomain = '@turtledownunder.com.au'; // Sender domain to handle specially

// --- Modular Feature Configuration ---
$enableSpamDetection = true;        // Enable/disable spam detection module
$enableQuoteGeneration = true;      // Enable/disable core AI (umbrella for Segments and Org/Contact)
$enableAiTagging = true;            // Enable/disable AI tagging module
$enableOrgContactManagement = true; // Enable/disable Organization/Contact extraction & creation

// --- Tagging Specific Configuration ---
$aiTaggerUserId = 0;      // User ID for vtiger_freetagged_objects.tagger_id (0 for 'system')
$tagModule = 'Dashboard'; // Module name for vtiger_freetagged_objects.module

// --- Validate DB Connection ---
$db_error_message = null;
if (!$conn || $conn->connect_error) {
    $db_error_message = "Database connection failed: " . ($conn->connect_error ?? 'Unknown DB error') . ". Check dbconn.php.";
    web_logger($db_error_message); // Log it for web display too
}

// --- Ensure Temp Table and Columns Exist (Schema Check - adapted from cron) ---
// For a web test page, DDL (CREATE/ALTER) operations are risky.
// This section will log what it *would* do but by default won't execute DDL.
// It's assumed the main cron job manages the schema.
function checkAndSetupSchema($conn_ref, $tableName, $isQuoteGenerationEnabled) {
    global $web_logs;

    if (!$isQuoteGenerationEnabled) {
        web_logger("SCHEMA_CHECK: Skipped for table '{$tableName}' (Quote Generation features are DISABLED).");
        return;
    }
    web_logger("SCHEMA_CHECK: Performing schema check for table '{$tableName}'.");
    $escapedTableName = $conn_ref->real_escape_string($tableName);
    try {
        $checkTableSql = "SHOW TABLES LIKE '" . $escapedTableName . "'";
        $tableResult = $conn_ref->query($checkTableSql);

        if (!$tableResult) {
            throw new Exception("Error checking if table '$escapedTableName' exists: " . $conn_ref->error);
        }

        if ($tableResult->num_rows == 0) {
            web_logger("SCHEMA_CHECK: Table '$escapedTableName' does not exist. SQL to create (NOT EXECUTING on web):");
            // --- FULLY EXPANDED: createTableSql ---
            $createTableSql = "CREATE TABLE {$escapedTableName} (
                id INT AUTO_INCREMENT PRIMARY KEY,
                templateref VARCHAR(1000) NULL,
                quoteref TEXT NULL,
                country VARCHAR(255) NULL,
                travel_date DATE NULL DEFAULT NULL,
                ai_response LONGTEXT NULL,
                conversation_id VARCHAR(255) NOT NULL,
                organizationid INT NULL DEFAULT NULL,
                contactid INT NULL DEFAULT NULL,
                created_at TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP,
                sender VARCHAR(255) NULL DEFAULT NULL,
                UNIQUE KEY (conversation_id)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;";
            web_logger($createTableSql);
            // To enable creation (use with caution on a web server):
            // if (!$conn_ref->query($createTableSql)) { throw new Exception("Error creating table '$escapedTableName': " . $conn_ref->error); }
            // web_logger("SCHEMA_CHECK: Table '$escapedTableName' would be created (if DDL enabled).");
        } else {
            web_logger("SCHEMA_CHECK: Table '$escapedTableName' exists.");
            // --- FULLY EXPANDED: colsToCheck ---
            $colsToCheck = [
                'quoteref' => "TEXT NULL", // Special handling for type check below
                'organizationid' => "INT NULL DEFAULT NULL AFTER conversation_id",
                'contactid' => "INT NULL DEFAULT NULL AFTER organizationid",
                'sender' => "VARCHAR(255) NULL DEFAULT NULL AFTER created_at",
                'travel_date' => "DATE NULL DEFAULT NULL AFTER country"
            ];

            // --- FULLY EXPANDED: Check and potentially alter 'quoteref' column type ---
            $checkQuoterefColSql = "SHOW COLUMNS FROM {$escapedTableName} LIKE 'quoteref'";
            $quoterefColResult = $conn_ref->query($checkQuoterefColSql);
            if ($quoterefColResult && $quoterefRow = $quoterefColResult->fetch_assoc()) {
                $currentType = strtolower(strtok($quoterefRow['Type'], '('));
                if ($currentType !== 'text' && $currentType !== 'longtext' && $currentType !== 'mediumtext') {
                    web_logger("SCHEMA_CHECK: Column 'quoteref' in '{$escapedTableName}' is not TEXT/MEDIUMTEXT/LONGTEXT type (current: {$currentType}). SQL to alter (NOT EXECUTING on web):");
                    $alterQuoterefSql = "ALTER TABLE {$escapedTableName} MODIFY COLUMN `quoteref` TEXT NULL";
                    web_logger($alterQuoterefSql);
                    // To enable alteration (use with caution):
                    // if (!$conn_ref->query($alterQuoterefSql)) { web_logger("SCHEMA_CHECK: Error altering 'quoteref': " . $conn_ref->error); }
                }
            }
            if ($quoterefColResult instanceof mysqli_result) $quoterefColResult->free();

            // --- FULLY EXPANDED: Check for other missing columns ---
            foreach ($colsToCheck as $colName => $colDef) {
                 if ($colName === 'quoteref') continue; // Already handled for type check

                $checkColSql = "SHOW COLUMNS FROM {$escapedTableName} LIKE '{$conn_ref->real_escape_string($colName)}'";
                $colResult = $conn_ref->query($checkColSql);
                if (!$colResult) {
                     web_logger("SCHEMA_CHECK: Error checking for column '{$colName}' in '{$escapedTableName}': " . $conn_ref->error);
                     continue;
                }
                if ($colResult->num_rows == 0) {
                    web_logger("SCHEMA_CHECK: Column '{$colName}' missing in '{$escapedTableName}'. SQL to add (NOT EXECUTING on web):");
                    $addColSql = "ALTER TABLE {$escapedTableName} ADD COLUMN `{$conn_ref->real_escape_string($colName)}` {$colDef}";
                    web_logger($addColSql);
                    // To enable addition (use with caution):
                    // if (!$conn_ref->query($addColSql)) { web_logger("SCHEMA_CHECK: Error adding column '{$colName}': " . $conn_ref->error . ". SQL: " . $addColSql); }
                }
                if ($colResult instanceof mysqli_result) $colResult->free();
            }
        }
        if ($tableResult instanceof mysqli_result) $tableResult->free();
        web_logger("SCHEMA_CHECK: Schema check for table '{$escapedTableName}' completed.");
    } catch (Exception $e) {
        web_logger("SCHEMA_CHECK: CRITICAL DATABASE SETUP ERROR for '{$escapedTableName}': " . $e->getMessage());
    }
}

// Perform schema check once if DB connection is okay
if (!$db_error_message) {
    checkAndSetupSchema($conn, $tempAiQuotesTableName, $enableQuoteGeneration);
}


// --- Main Processing Function ---
// (This function is now fully expanded as per previous complete responses)
function process_single_conversation_id($conversationId, $conn, &$undo_data_capture) {
    global $web_logs, $tempAiQuotesTableName, $excludedSenderDomain,
           $enableSpamDetection, $enableQuoteGeneration, $enableAiTagging, $enableOrgContactManagement,
           $aiTaggerUserId, $tagModule, $baseDir;

    web_logger("PROCESS: Starting for Conv ID: {$conversationId}");

    $undo_data_capture = [ /* ... (undo data initialization as before) ... */
        'conversation_id' => $conversationId,
        'vtiger_support_original' => null,
        'tdu_temp_ai_quotes_original' => null,
        'newly_created_org_id' => null,
        'newly_created_contact_id' => null,
        'newly_added_tags' => [],
        'processed_successfully' => false
    ];

    // Capture initial state for UNDO
    $stmt_undo_vs = $conn->prepare("SELECT status, ai_actioned FROM vtiger_support WHERE ticketid = ?");
    if ($stmt_undo_vs) {
        $stmt_undo_vs->bind_param("s", $conversationId);
        if ($stmt_undo_vs->execute()) {
            $result_undo_vs = $stmt_undo_vs->get_result();
            if ($row_undo_vs = $result_undo_vs->fetch_assoc()) {
                $undo_data_capture['vtiger_support_original'] = ['exists' => true, 'status' => $row_undo_vs['status'], 'ai_actioned' => $row_undo_vs['ai_actioned']];
            } else {
                $undo_data_capture['vtiger_support_original'] = ['exists' => false, 'status' => null, 'ai_actioned' => null];
            }
            if ($result_undo_vs instanceof mysqli_result) $result_undo_vs->free();
        } else { web_logger("UNDO_PREP_ERROR (vtiger_support EXEC): " . $stmt_undo_vs->error); }
        $stmt_undo_vs->close();
    } else { web_logger("UNDO_PREP_ERROR (vtiger_support PREPARE): " . $conn->error); }

    $escapedTempTable = $conn->real_escape_string($tempAiQuotesTableName);
    $stmt_undo_tdaq = $conn->prepare("SELECT * FROM {$escapedTempTable} WHERE conversation_id = ?");
    if ($stmt_undo_tdaq) {
        $stmt_undo_tdaq->bind_param("s", $conversationId);
        if ($stmt_undo_tdaq->execute()) {
            $result_undo_tdaq = $stmt_undo_tdaq->get_result();
            if ($row_undo_tdaq = $result_undo_tdaq->fetch_assoc()) {
                $undo_data_capture['tdu_temp_ai_quotes_original'] = ['exists' => true, 'data' => $row_undo_tdaq];
            } else {
                $undo_data_capture['tdu_temp_ai_quotes_original'] = ['exists' => false, 'data' => null];
            }
            if ($result_undo_tdaq instanceof mysqli_result) $result_undo_tdaq->free();
        } else { web_logger("UNDO_PREP_ERROR ({$escapedTempTable} EXEC): " . $stmt_undo_tdaq->error); }
        $stmt_undo_tdaq->close();
    } else { web_logger("UNDO_PREP_ERROR ({$escapedTempTable} PREPARE): " . $conn->error); }

    // Fetch email details
    $originalSenderForThisEmail = 'N/A'; $mailboxForThisEmail = null;
    $detailsSql = "SELECT sender, mailbox FROM tdu_emails WHERE conversation_id = ? ORDER BY received_datetime DESC LIMIT 1";
    $stmtDetails = $conn->prepare($detailsSql);
    if ($stmtDetails) {
        $stmtDetails->bind_param("s", $conversationId);
        if ($stmtDetails->execute()) {
            $resultDetails = $stmtDetails->get_result();
            if ($rowDetails = $resultDetails->fetch_assoc()) {
                $originalSenderForThisEmail = $rowDetails['sender'];
                $mailboxForThisEmail = $rowDetails['mailbox'];
            }
            if ($resultDetails instanceof mysqli_result) $resultDetails->free();
        } else { web_logger("PROCESS_ERROR: Failed to execute details fetch for {$conversationId}: " . $stmtDetails->error); }
        $stmtDetails->close();
    } else { web_logger("PROCESS_ERROR: Failed to prepare details fetch: " . $conn->error); }
    web_logger("PROCESS: Details - ConvID: {$conversationId}, Sender: {$originalSenderForThisEmail}, Mailbox: " . ($mailboxForThisEmail ?? 'N/A'));

    // Pre-set ai_actioned=1
    web_logger("PROCESS: Attempting to pre-set ai_actioned=1 for Conv ID: {$conversationId}");
    $sqlPreSetAiActioned = "INSERT INTO vtiger_support (ticketid, ai_actioned) VALUES (?, 1) ON DUPLICATE KEY UPDATE ai_actioned = 1";
    $stmtPreSet = $conn->prepare($sqlPreSetAiActioned);
    if ($stmtPreSet) {
        $stmtPreSet->bind_param("s", $conversationId);
        if ($stmtPreSet->execute()) {
             web_logger("PROCESS: Successfully pre-set/updated ai_actioned=1 for Conv ID {$conversationId}. Affected rows: ".$stmtPreSet->affected_rows);
        } else {
            web_logger("PROCESS_CRITICAL_ERROR: Failed to EXECUTE pre-set of ai_actioned=1 for Conv ID {$conversationId}: " . $stmtPreSet->error . ". Halting this process.");
            $stmtPreSet->close(); return false;
        }
        $stmtPreSet->close();
    } else {
        web_logger("PROCESS_CRITICAL_ERROR: Failed to PREPARE pre-set of ai_actioned=1 for Conv ID {$conversationId}: " . $conn->error . ". Halting this process.");
        return false;
    }

    // Spam Detection
    if ($enableSpamDetection) {
        web_logger("PROCESS: Spam Detection ENABLED for Conv ID {$conversationId}.");
        $spamClassification = 'error';
        if (function_exists('classifyAsSpam')) {
            $spamClassification = classifyAsSpam($conversationId, $conn); // Assumes classifyAsSpam uses web_logger or its logs won't show here
        } else {
            web_logger("PROCESS_ERROR: classifyAsSpam function missing. Assuming not spam.");
            $spamClassification = 'no';
        }
        if ($spamClassification === 'yes') {
            web_logger("PROCESS: Conv ID {$conversationId} classified as SPAM.");
            $statusSpam = VTIGER_SUPPORT_STATUS_SPAM;
            $sqlSetSpamStatus = "INSERT INTO vtiger_support (ticketid, status, ai_actioned) VALUES (?, ?, 1) ON DUPLICATE KEY UPDATE status = VALUES(status), ai_actioned = VALUES(ai_actioned)";
            $stmtSetSpam = $conn->prepare($sqlSetSpamStatus);
            if ($stmtSetSpam) {
                $stmtSetSpam->bind_param("ss", $conversationId, $statusSpam);
                if ($stmtSetSpam->execute()) { web_logger("PROCESS: Ensured vtiger_support status is SPAM & ai_actioned=1 for Ticket ID {$conversationId}. Affected rows: ".$stmtSetSpam->affected_rows);
                } else { web_logger("PROCESS_ERROR: Setting SPAM status (exec) for Conv ID {$conversationId}: " . $stmtSetSpam->error); }
                $stmtSetSpam->close();
            } else { web_logger("PROCESS_ERROR: Setting SPAM status (prep) for Conv ID {$conversationId}: " . $conn->error); }
            $undo_data_capture['processed_successfully'] = true; // Mark as "processed" (as spam)
            return true; // Stop further processing if SPAM
        } elseif ($spamClassification === 'error') {
            web_logger("PROCESS_ERROR: Error during spam classification for Conv ID {$conversationId}. Halting further processing.");
            return false; // Stop on spam check error
        }
        web_logger("PROCESS: Conv ID {$conversationId} is NOT spam.");
    } else {
        web_logger("PROCESS: Spam Detection DISABLED for Conv ID {$conversationId}.");
    }

    // Determine sender email
    $senderEmailForDb = $originalSenderForThisEmail ?? null;
    if ($senderEmailForDb !== null && strpos(strtolower($senderEmailForDb), strtolower($excludedSenderDomain)) !== false) {
        web_logger("PROCESS: Initial sender '{$senderEmailForDb}' (ConvID {$conversationId}) is from excluded domain '{$excludedSenderDomain}'. Searching alternative...");
        $foundAlternativeSender = false;
        $sqlAltSender = "SELECT sender FROM tdu_emails WHERE conversation_id = ? AND (sender IS NULL OR sender = '' OR LOWER(sender) NOT LIKE ?) ORDER BY received_datetime DESC LIMIT 1";
        $stmtAltSender = $conn->prepare($sqlAltSender);
        if ($stmtAltSender) {
            $excludedPattern = '%' . strtolower($excludedSenderDomain);
            $stmtAltSender->bind_param("ss", $conversationId, $excludedPattern);
            if ($stmtAltSender->execute()) {
                $resultAltSender = $stmtAltSender->get_result();
                if ($rowAlt = $resultAltSender->fetch_assoc()) {
                    if (!empty($rowAlt['sender'])) {
                        $senderEmailForDb = $rowAlt['sender'];
                        $foundAlternativeSender = true;
                        web_logger("PROCESS: Found alternative sender '{$senderEmailForDb}' for Conv ID {$conversationId}.");
                    }
                }
                if (!$foundAlternativeSender) {
                    $senderEmailForDb = null;
                    web_logger("PROCESS: No suitable alternative sender found for Conv ID {$conversationId}. Sender set to NULL for DB operations.");
                }
                if ($resultAltSender instanceof mysqli_result) $resultAltSender->free();
            } else { web_logger("PROCESS_ERROR: Exec error finding alt sender for Conv ID {$conversationId}: " . $stmtAltSender->error); }
            $stmtAltSender->close();
        } else { web_logger("PROCESS_ERROR: Prep error finding alt sender for Conv ID {$conversationId}: " . $conn->error); }
    }
    web_logger("PROCESS: Final Sender Email for DB for Conv ID {$conversationId}: " . ($senderEmailForDb ?? 'NULL'));

    // Quote Pre-detection
    $quotesAlreadyExist = false; $detectedQuoteNumbersJson = '[]';
    if (function_exists('extractQuoteNumbersFromConversation')) {
        $detectedQuoteNumbersJson = extractQuoteNumbersFromConversation($conversationId, $conn); // Assumes this logs internally or doesn't need specific web_log here
        if ($detectedQuoteNumbersJson === null || $detectedQuoteNumbersJson === false) {
            web_logger("PROCESS_WARNING: extractQuoteNumbersFromConversation returned null/false for Conv ID {$conversationId}. Assuming no quotes or error.");
            $detectedQuoteNumbersJson = '[]';
        }
        $decodedQuotes = json_decode($detectedQuoteNumbersJson, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            web_logger("PROCESS_WARNING: Invalid JSON from extractQuoteNumbersFromConversation for Conv ID {$conversationId}: " . json_last_error_msg() . ". Content: " . $detectedQuoteNumbersJson);
            $decodedQuotes = [];
            $detectedQuoteNumbersJson = '[]';
        }
        if (is_array($decodedQuotes) && !empty($decodedQuotes)) {
            $quotesAlreadyExist = true;
            web_logger("PROCESS: Conv ID {$conversationId} - Pre-existing quotes detected: " . $detectedQuoteNumbersJson . ". Core AI Segment Generation will be SKIPPED. Org/Contact may still run.");
        } else {
            web_logger("PROCESS: Conv ID {$conversationId} - No pre-existing quotes detected. Proceeding with potential Core AI Segment Generation and Org/Contact.");
        }
    } else {
        web_logger("PROCESS_ERROR: Function 'extractQuoteNumbersFromConversation' is missing for Conv ID {$conversationId}. Assuming no pre-existing quotes.");
    }

    // Initialize Org/Contact variables
    $extractedOrgId = null; $extractedContactId = null; $aiContactOrgData = null;

    // Organization & Contact Management
    if ($enableQuoteGeneration && $enableOrgContactManagement) {
        web_logger("PROCESS: Org/Contact Management ENABLED for Conv ID {$conversationId}. Attempting to process.");
        try {
            if (function_exists('extractContactAndOrganizationInfo')) {
                $contactOrgDetails = extractContactAndOrganizationInfo($conversationId, $conn); // Assumes this logs internally via web_logger if needed
                $aiContactOrgData = $contactOrgDetails['data'] ?? null;
                $extractedOrgNameFromAI = $aiContactOrgData['organization_name'] ?? null;
                if ($contactOrgDetails['success'] && $extractedOrgNameFromAI && strtolower(trim($extractedOrgNameFromAI)) !== 'n/a') {
                    web_logger("PROCESS: Conv ID {$conversationId} - Initial AI Extracted Org Name: '{$extractedOrgNameFromAI}'");
                    if (function_exists('find_best_match_org_via_ai_disambiguation')) {
                        $extractedOrgId = find_best_match_org_via_ai_disambiguation($extractedOrgNameFromAI, $aiContactOrgData, $conn);
                        if ($extractedOrgId) { web_logger("PROCESS: Conv ID {$conversationId} - AI Disambiguation Matched Org ID: {$extractedOrgId}");
                        } else { web_logger("PROCESS: Conv ID {$conversationId} - AI Disambiguation no match for '{$extractedOrgNameFromAI}'.");}
                    } else { web_logger("PROCESS_ERROR: Missing func 'find_best_match_org_via_ai_disambiguation' for Conv ID {$conversationId}."); }
                } elseif (!$contactOrgDetails['success']) { web_logger("PROCESS_ERROR: AI contact/org extraction failed for Conv ID {$conversationId}: " . ($contactOrgDetails['error'] ?? 'Unknown'));}
            } else { web_logger("PROCESS_ERROR: Missing func 'extractContactAndOrganizationInfo' for Conv ID {$conversationId}.");}

            $aiOrgNameValid = ($aiContactOrgData && !empty($aiContactOrgData['organization_name']) && strtolower(trim($aiContactOrgData['organization_name'])) !== 'n/a');
            if (!$extractedOrgId && $aiOrgNameValid) {
                if (function_exists('create_new_organization')) {
                    web_logger("PROCESS: Conv ID {$conversationId} - Org '{$aiContactOrgData['organization_name']}' not matched. Attempting creation.");
                    $newOrgId = create_new_organization($conn, $aiContactOrgData); // Assumes this returns ID or null
                    if ($newOrgId) {
                        $extractedOrgId = $newOrgId;
                        $undo_data_capture['newly_created_org_id'] = $newOrgId; // Capture for UNDO
                        web_logger("PROCESS: Created new Org ID: {$extractedOrgId} for Conv ID {$conversationId}");
                    } else { web_logger("PROCESS_ERROR: Failed to create Org '{$aiContactOrgData['organization_name']}' for Conv ID {$conversationId}.");}
                } else { web_logger("PROCESS_ERROR: Missing func 'create_new_organization' for Conv ID {$conversationId}.");}
            }

            $aiContactNameValid = ($aiContactOrgData && !empty($aiContactOrgData['contact_name']) && strtolower(trim($aiContactOrgData['contact_name'])) !== 'n/a');
            $aiContactEmailValid = ($aiContactOrgData && !empty($aiContactOrgData['contact_email']) && filter_var($aiContactOrgData['contact_email'], FILTER_VALIDATE_EMAIL));
            $canCreateOrMatchContact = $aiContactNameValid || $aiContactEmailValid;

            if ($extractedOrgId && $canCreateOrMatchContact) {
                web_logger("PROCESS: Conv ID {$conversationId} - Org ID {$extractedOrgId}. Attempting contact match/creation.");
                $contactFoundInDb = false;
                $sqlFindContact = "SELECT auto_id FROM tdu_contacts WHERE organizationid = ?";
                $params = [$extractedOrgId]; $types = "i"; $conditions = [];
                if ($aiContactEmailValid) { $conditions[] = "email = ?"; $params[] = $aiContactOrgData['contact_email']; $types .= "s"; }
                if ($aiContactNameValid) {
                    $nameCondition = "LOWER(REPLACE(name,'.','')) = LOWER(REPLACE(?,'.',''))";
                    if(empty($conditions)) { $conditions[] = $nameCondition; $params[] = $aiContactOrgData['contact_name']; $types .= "s"; }
                    else { $conditions[0] = "(" . $conditions[0] . " OR " . $nameCondition . ")"; $params[] = $aiContactOrgData['contact_name']; $types .= "s"; }
                }
                if(!empty($conditions)){ $sqlFindContact .= " AND (" . $conditions[0] . ")"; } else { $sqlFindContact .= " AND 1=0"; }
                $sqlFindContact .= " ORDER BY auto_id DESC LIMIT 1";
                $stmtContact = $conn->prepare($sqlFindContact);
                if ($stmtContact) {
                    if (!empty($types) && count($params) > 0) { $stmtContact->bind_param($types, ...$params); }
                    if($stmtContact->execute()){
                        $resultContact = $stmtContact->get_result();
                        if ($rowContact = $resultContact->fetch_assoc()) { $extractedContactId = $rowContact['auto_id']; $contactFoundInDb = true; web_logger("PROCESS: Conv ID {$conversationId} - Found existing Contact ID: {$extractedContactId}");}
                        if ($resultContact instanceof mysqli_result) $resultContact->free();
                    } else { web_logger("PROCESS_ERROR: SQL Exec Error (Find Contact) for ConvID {$conversationId}: " . $stmtContact->error); }
                    $stmtContact->close();
                } else { web_logger("PROCESS_ERROR: SQL Prep Error (Find Contact) for ConvID {$conversationId}: " . $conn->error); }
                if (!$contactFoundInDb && $canCreateOrMatchContact) {
                    if (function_exists('create_new_contact')) {
                        web_logger("PROCESS: Conv ID {$conversationId} - Contact not found. Attempting creation for Org ID {$extractedOrgId}.");
                        $newContactId = create_new_contact($conn, $extractedOrgId, $aiContactOrgData); // Assumes this returns ID or null
                        if ($newContactId) {
                            $extractedContactId = $newContactId;
                            $undo_data_capture['newly_created_contact_id'] = $newContactId; // Capture for UNDO
                            web_logger("PROCESS: Created new Contact ID: {$extractedContactId} for Conv ID {$conversationId}");
                        } else { web_logger("PROCESS_ERROR: Failed to create contact for Conv ID {$conversationId}."); }
                    } else { web_logger("PROCESS_ERROR: func 'create_new_contact' missing for Conv ID {$conversationId}."); }
                }
            } elseif ($extractedOrgId) { web_logger("PROCESS_INFO: Org ID {$extractedOrgId} identified, but not enough AI contact data for contact actions for Conv ID {$conversationId}."); }
        } catch (Exception $ocException) {
            web_logger("PROCESS_ERROR: During Organization/Contact Management for Conv ID {$conversationId}: " . $ocException->getMessage());
        }
        web_logger("PROCESS: Org/Contact Management finished for Conv ID {$conversationId}. OrgID: ".($extractedOrgId ?? 'None').", ContactID: ".($extractedContactId ?? 'None'));
    } elseif ($enableQuoteGeneration && !$enableOrgContactManagement) {
         web_logger("PROCESS_INFO: Org/Contact Management DISABLED for Conv ID {$conversationId}, though Quote Generation umbrella is enabled.");
    }

    // AI Quote Segment Generation OR Handling Pre-existing Quotes
    if ($enableQuoteGeneration) {
        web_logger("PROCESS: Quote Generation features are ENABLED for Conv ID {$conversationId}.");
        $conn->begin_transaction();
        try {
            $orgIdForDb = $extractedOrgId ? (int)$extractedOrgId : null;
            $contactIdForDb = $extractedContactId ? (int)$extractedContactId : null;

            if (!$quotesAlreadyExist) {
                web_logger("PROCESS: Core AI Segment Generation ENABLED and NO pre-existing quotes found for Conv ID {$conversationId}. Running segment extraction.");
                if (!function_exists('extractTravelSegmentsAndMatchTemplates')) { throw new Exception("Function extractTravelSegmentsAndMatchTemplates does not exist. Conv ID {$conversationId}.");}
                $aiExtractionResult = extractTravelSegmentsAndMatchTemplates($conversationId, $conn, '', $mailboxForThisEmail); // Assumes this logs internally
                if (!$aiExtractionResult || !isset($aiExtractionResult['success'])) { throw new Exception("AI segment extraction function returned invalid result for Conv ID {$conversationId}");}
                if (!$aiExtractionResult['success']) { throw new Exception("AI segment extraction failed for Conv ID {$conversationId}: " . ($aiExtractionResult['error'] ?? 'AI error') . " Details: " . ($aiExtractionResult['details'] ?? 'N/A'));}

                $aiParsedDataFromSegments = $aiExtractionResult['ai_parsed_data'] ?? null;
                $extractedTravelDate = null;
                if ($aiParsedDataFromSegments && isset($aiParsedDataFromSegments['travel_date'])) {
                    $dateValue = trim($aiParsedDataFromSegments['travel_date']);
                    if (strtolower($dateValue) !== 'n/a' && !empty($dateValue)) {
                        $timestamp = strtotime($dateValue);
                        if ($timestamp !== false) { $extractedTravelDate = date('Y-m-d', $timestamp); }
                    }
                }
                $dataForDbAiResponse = [
                    'success' => $aiExtractionResult['success'], 'segments' => ($aiExtractionResult['segments'] ?? []),
                    'itinerary_items_used' => $aiExtractionResult['itinerary_items_used'] ?? 'N/A',
                    'raw_ai_response' => $aiExtractionResult['raw_ai_response'] ?? null,
                    'ai_parsed_data' => $aiParsedDataFromSegments, 'message' => $aiExtractionResult['message'] ?? 'Processed.',
                    'category_used_for_filter' => $aiExtractionResult['category_used_for_filter'] ?? 'N/A',
                    'initial_ai_category_log' => $aiExtractionResult['initial_ai_category_log'] ?? null,
                    'processed_organizationid' => $orgIdForDb, 'processed_contactid' => $contactIdForDb
                ];
                $jsonForDbAiResponse = json_encode($dataForDbAiResponse);
                if ($jsonForDbAiResponse === false) { throw new Exception("JSON encode error for AI response, Conv ID {$conversationId}: " . json_last_error_msg());}
                $templateRefString = 'N/A';
                $segmentsFromExtraction = $aiExtractionResult['segments'] ?? [];
                if (!empty($segmentsFromExtraction)) {
                    $templateIdsCollector = [];
                    foreach ($segmentsFromExtraction as $segment) { if (isset($segment['matched_template']['templateid']) && !empty($segment['matched_template']['templateid'])) { $templateIdsCollector[] = $segment['matched_template']['templateid'];}}
                    $uniqueFilteredIds = array_unique(array_filter($templateIdsCollector));
                    if (!empty($uniqueFilteredIds)) { $templateRefString = implode(',', $uniqueFilteredIds); }
                }
                $country = 'N/A';
                if ($aiParsedDataFromSegments !== null && isset($aiParsedDataFromSegments['country'])) {
                    $countryList = array_map('trim', explode(',', $aiParsedDataFromSegments['country']));
                    $validCountries = array_filter($countryList, function($c) { return strtolower($c) !== 'n/a' && !empty($c); });
                    if (!empty($validCountries)) { $country = reset($validCountries); }
                }
                if (($country === 'N/A' || empty($country)) && !empty($segmentsFromExtraction[0]['input_country']) && strtolower($segmentsFromExtraction[0]['input_country']) !== 'n/a') {
                    $country = trim($segmentsFromExtraction[0]['input_country']);
                     web_logger("PROCESS: Conv ID {$conversationId} - Used fallback country '{$country}' from segment 1 input.");
                }
                web_logger("PROCESS: Conv ID {$conversationId} - Country Determined for DB: " . $country);
                $sqlInsertOrUpdateQuote = "INSERT INTO {$escapedTempTable} (templateref, quoteref, country, travel_date, ai_response, conversation_id, organizationid, contactid, created_at, sender) VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW(), ?) ON DUPLICATE KEY UPDATE templateref = VALUES(templateref), quoteref = VALUES(quoteref), country = VALUES(country), travel_date = VALUES(travel_date), ai_response = VALUES(ai_response), organizationid = VALUES(organizationid), contactid = VALUES(contactid), created_at = NOW(), sender = VALUES(sender)";
                $stmtInsertUpdate = $conn->prepare($sqlInsertOrUpdateQuote);
                if ($stmtInsertUpdate === false) { throw new Exception("SQL Prep Error (Insert/Update Quote - Segments) for Conv ID {$conversationId}: " . $conn->error); }
                $travelDateForDb = ($extractedTravelDate === 'N/A' || empty($extractedTravelDate)) ? null : $extractedTravelDate;
                $quoteRefForDbThisPath = null;
                $stmtInsertUpdate->bind_param("ssssssiss", $templateRefString, $quoteRefForDbThisPath, $country, $travelDateForDb, $jsonForDbAiResponse, $conversationId, $orgIdForDb, $contactIdForDb, $senderEmailForDb);
                if (!$stmtInsertUpdate->execute()) { $stmtError = $stmtInsertUpdate->error; $stmtInsertUpdate->close(); throw new Exception("SQL Exec Error (Insert/Update Quote - Segments) for Conv ID {$conversationId}: " . $stmtError); }
                web_logger("PROCESS: Upserted record into {$escapedTempTable} with AI Segments for Conv ID {$conversationId}. Affected Rows: ".$stmtInsertUpdate->affected_rows);
                $stmtInsertUpdate->close();
            } else { // $quotesAlreadyExist is TRUE
                web_logger("PROCESS: Core AI Segment Generation SKIPPED for Conv ID {$conversationId} due to pre-detected quotes: " . $detectedQuoteNumbersJson . ". Storing pre-detected quote info.");
                $aiResponseMessageArray = [
                    'message' => 'Core AI segment generation skipped due to pre-detected quotes. Org/Contact management may have run. AI Tagging may still apply.',
                    'detected_quotes_json' => json_decode($detectedQuoteNumbersJson),
                    'processed_organizationid' => $orgIdForDb, 'processed_contactid' => $contactIdForDb
                ];
                $aiResponseMessage = json_encode($aiResponseMessageArray);
                if ($aiResponseMessage === false) { throw new Exception("JSON encode error for skipped AI response, Conv ID {$conversationId}: " . json_last_error_msg());}
                $sqlInsertSkipped = "INSERT INTO {$escapedTempTable} (templateref, quoteref, country, travel_date, ai_response, conversation_id, organizationid, contactid, created_at, sender) VALUES (NULL, ?, NULL, NULL, ?, ?, ?, ?, NOW(), ?) ON DUPLICATE KEY UPDATE quoteref = VALUES(quoteref), ai_response = VALUES(ai_response), organizationid = VALUES(organizationid), contactid = VALUES(contactid), sender = VALUES(sender), created_at = NOW(), templateref = NULL, country = NULL, travel_date = NULL";
                $stmtInsertSkipped = $conn->prepare($sqlInsertSkipped);
                if ($stmtInsertSkipped) {
                    $decodedQuotesArray = json_decode($detectedQuoteNumbersJson, true);
                    $quoterefToStore = null;
                    if (is_array($decodedQuotesArray)) {
                        if (count($decodedQuotesArray) === 1 && !empty($decodedQuotesArray[0])) { $quoterefToStore = $decodedQuotesArray[0];
                        } elseif (count($decodedQuotesArray) > 1) { $quoterefToStore = implode(',', array_filter($decodedQuotesArray)); if(empty($quoterefToStore)) $quoterefToStore = null; }
                    } else { web_logger("PROCESS_WARNING: Conv ID {$conversationId} - Pre-detected quotes ('{$detectedQuoteNumbersJson}') not valid JSON array. Storing NULL in quoteref.");}
                    $stmtInsertSkipped->bind_param("sssiis", $quoterefToStore, $aiResponseMessage, $conversationId, $orgIdForDb, $contactIdForDb, $senderEmailForDb);
                    if ($stmtInsertSkipped->execute()) { web_logger("PROCESS: Conv ID {$conversationId} - Upserted {$escapedTempTable} with pre-detected quotes and Org/Contact. Affected rows: " . $stmtInsertSkipped->affected_rows);
                    } else { throw new Exception("FAILED to upsert {$escapedTempTable} (skipped case) for Conv ID {$conversationId}: " . $stmtInsertSkipped->error); }
                    $stmtInsertSkipped->close();
                } else { throw new Exception("FAILED to prepare {$escapedTempTable} (skipped case) for Conv ID {$conversationId}: " . $conn->error); }
            }
            // Update vtiger_support status
            $statusAiProcessed = VTIGER_SUPPORT_STATUS_AI_PROCESSED;
            $sqlSetVtigerStatus = "INSERT INTO vtiger_support (ticketid, status, ai_actioned) VALUES (?, ?, 1) ON DUPLICATE KEY UPDATE status = VALUES(status), ai_actioned = 1";
            $stmtSetVtiger = $conn->prepare($sqlSetVtigerStatus);
            if ($stmtSetVtiger === false) { throw new Exception("SQL Prep Error (Update Vtiger Support) for Conv ID {$conversationId}: " . $conn->error); }
            $stmtSetVtiger->bind_param("ss", $conversationId, $statusAiProcessed);
            if (!$stmtSetVtiger->execute()) { $stmtErrorVtiger = $stmtSetVtiger->error; $stmtSetVtiger->close(); throw new Exception("SQL Exec Error (Update Vtiger Support) for Conv ID {$conversationId}: " . $stmtErrorVtiger); }
            web_logger("PROCESS: Ensured vtiger_support status is '{$statusAiProcessed}' (ai_actioned=1) for Ticket ID {$conversationId}. Affected rows: ".$stmtSetVtiger->affected_rows);
            $stmtSetVtiger->close();
            $conn->commit();
            web_logger("PROCESS: Quote Generation block (either segments or skipped info) committed successfully for Conv ID {$conversationId}.");
        } catch (Exception $e_main_ai) { // Catch exception from main AI processing block
            if ($conn->get_transaction_status()) { $conn->rollback(); }
            web_logger("PROCESS_CRITICAL_ERROR: During AI Quote/Segment or Pre-detected Quote Handling for Conv ID {$conversationId}: " . $e_main_ai->getMessage());
            // This exception will prevent 'processed_successfully' from being true
        }
    } else {
         web_logger("PROCESS_INFO: AI Quote Generation features (including Org/Contact and Segments) are DISABLED for Conv ID {$conversationId}.");
    }

    // AI Tagging Module
    $coreQuoteBlockLikelyCommitted = (!isset($e_main_ai) && $enableQuoteGeneration); // True if the try-catch above didn't catch an error
    if ($enableAiTagging) {
        web_logger("PROCESS: AI Tagging ENABLED for Conv ID {$conversationId}.");
        $tagsAttempted = false; $tagsSuccessfullyApplied = false;
        try {
            if (function_exists('getSuggestedTagsForConversation')) {
                $excludedTagCategories = [];
                $suggestedTags = getSuggestedTagsForConversation($conversationId, $conn, $excludedTagCategories); // Assumes this logs internally
                if ($suggestedTags === false || $suggestedTags === null) {
                    web_logger("PROCESS_WARNING: getSuggestedTagsForConversation returned an error or no tags for Conv ID {$conversationId}.");
                    $suggestedTags = [];
                }
                $tagsAttempted = true;
                if (!empty($suggestedTags)) {
                    web_logger("PROCESS: Conv ID {$conversationId} - AI Suggested tags: " . json_encode($suggestedTags));
                    $sqlInsertTag = "INSERT IGNORE INTO vtiger_freetagged_objects (tag_id, tagger_id, object_id, module) VALUES (?, ?, ?, ?)";
                    $stmtInsertTag = $conn->prepare($sqlInsertTag);
                    if ($stmtInsertTag) {
                        foreach ($suggestedTags as $tagInfo) {
                            if (is_array($tagInfo) && isset($tagInfo['id']) && is_numeric($tagInfo['id'])) {
                                $tagIdToInsert = (int)$tagInfo['id'];
                                $stmtInsertTag->bind_param("iiss", $tagIdToInsert, $aiTaggerUserId, $conversationId, $tagModule);
                                if ($stmtInsertTag->execute()) {
                                    if ($stmtInsertTag->affected_rows > 0) {
                                        web_logger("PROCESS: Conv ID {$conversationId} - Successfully tagged with Tag ID: {$tagIdToInsert}.");
                                        $undo_data_capture['newly_added_tags'][] = [
                                            'tag_id' => $tagIdToInsert, 'object_id' => $conversationId,
                                            'tagger_id' => $aiTaggerUserId, 'module' => $tagModule
                                        ];
                                        $tagsSuccessfullyApplied = true;
                                    }
                                } else { web_logger("PROCESS_ERROR: Conv ID {$conversationId} - FAILED tag insert exec TagID {$tagIdToInsert}: " . $stmtInsertTag->error); }
                            } else { web_logger("PROCESS_WARNING: Conv ID {$conversationId} - Invalid tag data in suggestions: " . print_r($tagInfo, true));}
                        }
                        $stmtInsertTag->close();
                    } else { throw new Exception("Tag insert prep failed for Conv ID {$conversationId}: " . $conn->error); }
                } else { web_logger("PROCESS_INFO: Conv ID {$conversationId} - No valid tags suggested or error during suggestion.");}
                if ($tagsSuccessfullyApplied) {
                    if (!$coreQuoteBlockLikelyCommitted) { // If core AI block didn't set vtiger_support status or failed
                        $statusAiProcessed = VTIGER_SUPPORT_STATUS_AI_PROCESSED;
                        $sqlSetVtigerStatusTagging = "INSERT INTO vtiger_support (ticketid, status, ai_actioned) VALUES (?, ?, 1) ON DUPLICATE KEY UPDATE status = IF(status IS NULL OR status != ?, VALUES(status), status), ai_actioned = 1";
                        $stmtSetVtigerTagging = $conn->prepare($sqlSetVtigerStatusTagging);
                        if ($stmtSetVtigerTagging) {
                            $statusSpamConst = VTIGER_SUPPORT_STATUS_SPAM; // Avoid overwriting SPAM status
                            $stmtSetVtigerTagging->bind_param("sss", $conversationId, $statusAiProcessed, $statusSpamConst);
                            if ($stmtSetVtigerTagging->execute()) { web_logger("PROCESS: Conv ID {$conversationId} - Ensured vtiger_support '{$statusAiProcessed}' after tagging. Affected: " . $stmtSetVtigerTagging->affected_rows);
                            } else { web_logger("PROCESS_ERROR: Conv ID {$conversationId} - FAILED update vtiger_support status post-tagging: " . $stmtSetVtigerTagging->error); }
                            $stmtSetVtigerTagging->close();
                        } else { web_logger("PROCESS_ERROR: Conv ID {$conversationId} - FAILED prepare vtiger_support status (post-tagging): " . $conn->error); }
                    }
                } elseif ($tagsAttempted) { web_logger("PROCESS_INFO: Conv ID {$conversationId} - Tagging attempted but no new tags were applied.");}
                // Clear any exception from the main AI block if it was caught and we reached here.
                if(isset($e_main_ai)) unset($e_main_ai);
            } else { web_logger("PROCESS_ERROR: Missing func 'getSuggestedTagsForConversation'."); throw new Exception("Tagging function missing for Conv ID {$conversationId}."); }
        } catch (Exception $e_tagging) { // Catch exception from tagging block
            web_logger("PROCESS_ERROR: During AI Tagging block for Conv ID {$conversationId}: " . $e_tagging->getMessage());
        }
    } else {
        web_logger("PROCESS_INFO: AI Tagging DISABLED for Conv ID {$conversationId}.");
    }

    // Determine overall success for undo purposes
    if (!isset($e_main_ai) && !isset($e_tagging)) { // If no exceptions were caught and held by these specific variables
        web_logger("PROCESS: Finished processing successfully for Conv ID: {$conversationId}");
        $undo_data_capture['processed_successfully'] = true;
    } else {
        web_logger("PROCESS_WARNING: Processing for Conv ID: {$conversationId} completed with some errors in sub-modules. Check logs.");
        // processed_successfully remains false, undo data still captures what was done.
    }
    return true; // The function process_single_conversation_id itself completed its flow.
}


// --- Undo Function ---
// (This function remains the same as the last fully verified version)
function perform_undo($conn, $undo_data) {
    global $web_logs, $tempAiQuotesTableName;
    if (!$undo_data || !isset($undo_data['conversation_id'])) {
        web_logger("UNDO_ERROR: Invalid or missing undo data.");
        return false;
    }
    $conversationId = $undo_data['conversation_id'];
    web_logger("UNDO: Starting for Conversation ID: " . $conversationId);
    $overall_undo_success = true;
    $conn->begin_transaction();
    try {
        // 1. Revert/delete tdu_temp_ai_quotes
        $escapedTempTable = $conn->real_escape_string($tempAiQuotesTableName);
        if (isset($undo_data['tdu_temp_ai_quotes_original'])) {
            $original_tdaq = $undo_data['tdu_temp_ai_quotes_original'];
            if ($original_tdaq['exists'] && is_array($original_tdaq['data'])) {
                $data = $original_tdaq['data'];
                $updateFields = []; $bindTypes = ""; $bindValues = [];
                foreach ($data as $key => $value) {
                    if ($key === 'id' || $key === 'conversation_id') continue;
                    $updateFields[] = "`" . $conn->real_escape_string($key) . "` = ?";
                    if (is_int($value)) $bindTypes .= "i";
                    elseif (is_double($value)) $bindTypes .= "d";
                    elseif ($value === null) $bindTypes .= "s";
                    else $bindTypes .= "s";
                    $bindValues[] = $value;
                }
                $bindValues[] = $conversationId; $bindTypes .= "s";
                if (!empty($updateFields)) {
                    $sql_restore_tdaq = "UPDATE {$escapedTempTable} SET " . implode(", ", $updateFields) . " WHERE conversation_id = ?";
                    $stmt_restore_tdaq = $conn->prepare($sql_restore_tdaq);
                    if ($stmt_restore_tdaq) {
                        $stmt_restore_tdaq->bind_param($bindTypes, ...$bindValues);
                        if (!$stmt_restore_tdaq->execute()) { web_logger("UNDO_ERROR (tdu_temp_ai_quotes UPDATE): " . $stmt_restore_tdaq->error); $overall_undo_success = false; }
                        else { web_logger("UNDO: Restored tdu_temp_ai_quotes for " . $conversationId . ". Affected: " . $stmt_restore_tdaq->affected_rows); }
                        $stmt_restore_tdaq->close();
                    } else { web_logger("UNDO_ERROR (tdu_temp_ai_quotes UPDATE PREPARE): " . $conn->error); $overall_undo_success = false; }
                } else { web_logger("UNDO_INFO: No fields to update for tdu_temp_ai_quotes."); }
            } else {
                $stmt_delete_tdaq = $conn->prepare("DELETE FROM {$escapedTempTable} WHERE conversation_id = ?");
                if ($stmt_delete_tdaq) {
                    $stmt_delete_tdaq->bind_param("s", $conversationId);
                    if (!$stmt_delete_tdaq->execute()) { web_logger("UNDO_ERROR (tdu_temp_ai_quotes DELETE): " . $stmt_delete_tdaq->error); $overall_undo_success = false; }
                    else { web_logger("UNDO: Deleted tdu_temp_ai_quotes for " . $conversationId . ". Affected: " . $stmt_delete_tdaq->affected_rows); }
                    $stmt_delete_tdaq->close();
                } else { web_logger("UNDO_ERROR (tdu_temp_ai_quotes DELETE PREPARE): " . $conn->error); $overall_undo_success = false; }
            }
        }
        // 2. Revert/delete vtiger_support
        if (isset($undo_data['vtiger_support_original'])) {
            $original_vs = $undo_data['vtiger_support_original'];
            if ($original_vs['exists']) {
                $stmt_restore_vs = $conn->prepare("UPDATE vtiger_support SET status = ?, ai_actioned = ? WHERE ticketid = ?");
                if ($stmt_restore_vs) {
                    $aiActionedInt = $original_vs['ai_actioned'] === null ? null : (int)$original_vs['ai_actioned'];
                    $statusVal = $original_vs['status'];
                    $stmt_restore_vs->bind_param("sis", $statusVal, $aiActionedInt, $conversationId);
                    if (!$stmt_restore_vs->execute()) { web_logger("UNDO_ERROR (vtiger_support UPDATE): " . $stmt_restore_vs->error); $overall_undo_success = false; }
                    else { web_logger("UNDO: Restored vtiger_support for " . $conversationId . ". Affected: " . $stmt_restore_vs->affected_rows); }
                    $stmt_restore_vs->close();
                } else { web_logger("UNDO_ERROR (vtiger_support UPDATE PREPARE): " . $conn->error); $overall_undo_success = false; }
            } else {
                 $stmt_delete_vs = $conn->prepare("DELETE FROM vtiger_support WHERE ticketid = ?");
                 if ($stmt_delete_vs) {
                    $stmt_delete_vs->bind_param("s", $conversationId);
                    if (!$stmt_delete_vs->execute()) { web_logger("UNDO_ERROR (vtiger_support DELETE): " . $stmt_delete_vs->error); $overall_undo_success = false; }
                    else { web_logger("UNDO: Deleted vtiger_support for " . $conversationId . ". Affected: " . $stmt_delete_vs->affected_rows); }
                    $stmt_delete_vs->close();
                 } else { web_logger("UNDO_ERROR (vtiger_support DELETE PREPARE): " . $conn->error); $overall_undo_success = false; }
            }
        }
        // 3. Delete newly added tags
        if (!empty($undo_data['newly_added_tags'])) {
            $sqlDeleteTag = "DELETE FROM vtiger_freetagged_objects WHERE tag_id = ? AND object_id = ? AND tagger_id = ? AND module = ?";
            $stmtDeleteTag = $conn->prepare($sqlDeleteTag);
            if ($stmtDeleteTag) {
                foreach ($undo_data['newly_added_tags'] as $tag) {
                    $stmtDeleteTag->bind_param("isss", $tag['tag_id'], $tag['object_id'], $tag['tagger_id'], $tag['module']);
                    if (!$stmtDeleteTag->execute()) { web_logger("UNDO_ERROR (Deleting Tag ID {$tag['tag_id']}): " . $stmtDeleteTag->error); $overall_undo_success = false; }
                    else { web_logger("UNDO: Deleted Tag ID {$tag['tag_id']} for object {$tag['object_id']}. Affected: " . $stmtDeleteTag->affected_rows); }
                }
                $stmtDeleteTag->close();
            } else { web_logger("UNDO_ERROR (Tag Delete PREPARE): " . $conn->error); $overall_undo_success = false; }
        }
        // 4. Delete newly created organization
        $orgTable = 'vtiger_account'; $orgIdColumn = 'accountid'; // IMPORTANT: Ensure these are correct for your Vtiger
        if (!empty($undo_data['newly_created_org_id'])) {
            $orgIdToDelete = $undo_data['newly_created_org_id'];
            web_logger("UNDO: Attempting to delete newly created Organization ID: {$orgIdToDelete} from table {$orgTable}.{$orgIdColumn}");
            $stmtDeleteOrg = $conn->prepare("DELETE FROM {$orgTable} WHERE {$orgIdColumn} = ?");
            if ($stmtDeleteOrg) {
                $stmtDeleteOrg->bind_param("i", $orgIdToDelete);
                if (!$stmtDeleteOrg->execute()) { web_logger("UNDO_ERROR (Deleting Organization ID {$orgIdToDelete}): " . $stmtDeleteOrg->error . ". May have dependencies or already deleted."); }
                else { web_logger("UNDO: Deleted Organization ID {$orgIdToDelete}. Affected: " . $stmtDeleteOrg->affected_rows); }
                $stmtDeleteOrg->close();
            } else { web_logger("UNDO_ERROR (Organization Delete PREPARE {$orgTable}): " . $conn->error); $overall_undo_success = false; }
        }
        // 5. Delete newly created contact
        $contactTable = 'vtiger_contactdetails'; $contactIdColumn = 'contactid'; // IMPORTANT: Ensure these are correct for your Vtiger
        if (!empty($undo_data['newly_created_contact_id'])) {
            $contactIdToDelete = $undo_data['newly_created_contact_id'];
            web_logger("UNDO: Attempting to delete newly created Contact ID: {$contactIdToDelete} from table {$contactTable}.{$contactIdColumn}");
            $stmtDeleteContact = $conn->prepare("DELETE FROM {$contactTable} WHERE {$contactIdColumn} = ?");
             if ($stmtDeleteContact) {
                $stmtDeleteContact->bind_param("i", $contactIdToDelete);
                if (!$stmtDeleteContact->execute()) { web_logger("UNDO_ERROR (Deleting Contact ID {$contactIdToDelete}): " . $stmtDeleteContact->error . ". May have dependencies or already deleted."); }
                else { web_logger("UNDO: Deleted Contact ID {$contactIdToDelete}. Affected: " . $stmtDeleteContact->affected_rows); }
                $stmtDeleteContact->close();
            } else { web_logger("UNDO_ERROR (Contact Delete PREPARE {$contactTable}): " . $conn->error); $overall_undo_success = false; }
        }
        if ($overall_undo_success) { $conn->commit(); web_logger("UNDO: Committed all undo operations for " . $conversationId); }
        else { $conn->rollback(); web_logger("UNDO_ERROR: Rolled back undo operations for " . $conversationId); }
    } catch (Exception $e) {
        if ($conn->get_transaction_status()) { $conn->rollback(); }
        web_logger("UNDO_CRITICAL_EXCEPTION: " . $e->getMessage());
        $overall_undo_success = false;
    }
    return $overall_undo_success;
}


// --- Handle Form Submissions ---
if ($_SERVER['REQUEST_METHOD'] == 'POST' && !$db_error_message) {
    if (isset($_POST['process_email']) && !empty($_POST['conversation_id'])) {
        $conversationId_input = trim($_POST['conversation_id']);
        web_logger("WEB_REQUEST: Process Email for Conversation ID: " . $conversationId_input);
        $current_undo_data = [];
        process_single_conversation_id($conversationId_input, $conn, $current_undo_data);
        $_SESSION['last_processed_conv_id'] = $conversationId_input;
        $_SESSION['undo_data'] = $current_undo_data;
        web_logger("WEB_REQUEST: Stored undo data in session for " . $conversationId_input . ". Original process success flag: " . ($current_undo_data['processed_successfully'] ? 'true' : 'false'));
    } elseif (isset($_POST['undo_last_process'])) {
        web_logger("WEB_REQUEST: Undo Last Process");
        if (isset($_SESSION['undo_data']) && isset($_SESSION['last_processed_conv_id'])) {
            $undo_data_from_session = $_SESSION['undo_data'];
            if ($undo_data_from_session['conversation_id'] == $_SESSION['last_processed_conv_id']) {
                if (isset($undo_data_from_session['processed_successfully']) && $undo_data_from_session['processed_successfully']) {
                     web_logger("WEB_REQUEST_INFO: Original process for " . $_SESSION['last_processed_conv_id'] . " was marked as successful. Proceeding with Undo.");
                } else {
                     web_logger("WEB_REQUEST_WARNING: Original process for " . $_SESSION['last_processed_conv_id'] . " was NOT marked as fully successful. Undo will attempt to revert known changes. Review logs carefully.");
                }
                if(perform_undo($conn, $undo_data_from_session)) {
                    web_logger("WEB_REQUEST: Undo attempt completed for " . $_SESSION['last_processed_conv_id'] . ". Check logs for details.");
                } else {
                    web_logger("WEB_REQUEST_ERROR: Undo function reported failure for " . $_SESSION['last_processed_conv_id']);
                }
            } else {
                web_logger("WEB_REQUEST_ERROR: Session data mismatch. Aborting undo.");
            }
            unset($_SESSION['last_processed_conv_id']);
            unset($_SESSION['undo_data']);
        } else {
            web_logger("WEB_REQUEST_INFO: No valid undo data found in session.");
        }
    }
}

// Close DB connection
if ($conn && get_class($conn) == 'mysqli' && empty($conn->connect_error) && $conn->thread_id) {
    $conn->close();
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>AI Email Processor Test</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background-color: #f0f2f5; color: #333; font-size: 14px; }
        .container { max-width: 900px; margin: 0 auto; background-color: #fff; padding: 25px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        h1 { color: #333; text-align: center; margin-bottom: 25px; font-size: 24px;}
        h2 { color: #555; margin-top: 20px; margin-bottom: 10px; font-size: 18px; border-bottom: 1px solid #eee; padding-bottom: 5px;}
        label { display: block; margin-bottom: 6px; font-weight: bold; }
        input[type="text"] { width: calc(100% - 20px); padding: 10px; margin-bottom: 18px; border: 1px solid #ccc; border-radius: 4px; box-sizing: border-box; }
        button { padding: 10px 18px; color: white; border: none; border-radius: 4px; cursor: pointer; margin-right: 10px; font-size: 14px; transition: background-color 0.2s ease; }
        button[name="process_email"] { background-color: #007bff; }
        button[name="process_email"]:hover { background-color: #0056b3; }
        button.undo { background-color: #ffc107; color: #212529; }
        button.undo:hover { background-color: #e0a800; }
        .logs { margin-top: 20px; padding: 15px; background-color: #f8f9fa; border: 1px solid #dee2e6; border-radius: 4px; max-height: 500px; overflow-y: auto; white-space: pre-wrap; word-wrap: break-word; font-family: Consolas, 'Courier New', monospace; font-size: 0.85em; line-height: 1.5; }
        .error-message { color: #d9534f; background-color: #f2dede; border: 1px solid #ebccd1; padding: 10px; border-radius: 4px; margin-bottom: 15px; }
    </style>
</head>
<body>
    <div class="container">
        <h1>AI Email Processor Test Page</h1>

        <?php if ($db_error_message): ?>
            <p class="error-message"><strong>Database Connection Error:</strong> <?php echo htmlspecialchars($db_error_message); ?></p>
        <?php else: ?>
            <form method="POST" action="">
                <div>
                    <label for="conversation_id">Conversation ID:</label>
                    <input type="text" id="conversation_id" name="conversation_id" value="<?php echo isset($_POST['conversation_id']) ? htmlspecialchars($_POST['conversation_id']) : (isset($_SESSION['last_processed_conv_id']) ? htmlspecialchars($_SESSION['last_processed_conv_id']) : ''); ?>" required>
                </div>
                <button type="submit" name="process_email">Process Email</button>
                <?php
                $show_undo = false;
                if (isset($_SESSION['undo_data'], $_SESSION['last_processed_conv_id'])) {
                    $s_undo_data = $_SESSION['undo_data'];
                    if (is_array($s_undo_data) && isset($s_undo_data['conversation_id']) && $s_undo_data['conversation_id'] == $_SESSION['last_processed_conv_id']) {
                        $show_undo = true;
                    }
                }
                if ($show_undo):
                ?>
                    <button type="submit" name="undo_last_process" class="undo">Undo Last Process (<?php echo htmlspecialchars($_SESSION['last_processed_conv_id']); ?>)</button>
                <?php endif; ?>
            </form>
        <?php endif; ?>

        <?php if (!empty($web_logs)): ?>
            <h2>Processing Logs:</h2>
            <div class="logs">
                <?php foreach ($web_logs as $log_entry): ?>
                    <?php echo htmlspecialchars($log_entry); ?><br>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</body>
</html>