<?php
// debug_qbo_vendor.php
// Dedicated script to diagnose vendor creation/query issues.
// Usage: php debug_qbo_vendor.php "Vendor Name" [OptionalVendorID]
// Example: php debug_qbo_vendor.php "Merlin Entertainment" 279
// Example: php debug_qbo_vendor.php "Merlin Entertainment"

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
set_time_limit(0);

define('BASE_PATH', dirname(__DIR__));

require_once BASE_PATH . '/vendor/autoload.php';
require_once BASE_PATH . '/qbo_functions.php'; // Assuming QBOUtilityLibrary is here

use QuickBooksOnline\API\DataService\DataService;
use QuickBooksOnline\API\Facades\Vendor;
use QuickBooksOnline\API\Exception\IdsException;

// --- Configuration & Helpers ---
$tokenStorageFile = BASE_PATH . '/tokens/qbo_token.json'; // Centralized token file
$logFileDebug = __DIR__ . '/qbo_vendor_debug.log';

function debugLog($message) {
    global $logFileDebug;
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[$timestamp] " . $message . "\n";
    echo $logEntry; // Output to console
    file_put_contents($logFileDebug, $logEntry, FILE_APPEND);
}

function loadTokensDebug() {
    global $tokenStorageFile;
    if (file_exists($tokenStorageFile)) {
        $json = file_get_contents($tokenStorageFile);
        return $json ? json_decode($json, true) : null;
    }
    debugLog("ERROR: Token storage file not found at: " . $tokenStorageFile);
    return null;
}

// No saveTokens needed for this read-only debug script, but refresh is good.
function refreshTokenDebug($baseConfig, &$loadedTokens) {
    global $tokenStorageFile;
    if (empty($baseConfig['ClientID']) || empty($baseConfig['ClientSecret']) || empty($loadedTokens['refresh_token'])) {
        debugLog("Skipping token refresh: ClientID, ClientSecret, or Refresh Token is missing.");
        return false;
    }
    try {
        $oauth2LoginHelper = new QuickBooksOnline\API\Core\OAuth\OAuth2\OAuth2LoginHelper($baseConfig['ClientID'], $baseConfig['ClientSecret']);
        $refreshedAccessTokenObj = $oauth2LoginHelper->refreshAccessTokenWithRefreshToken($loadedTokens['refresh_token']);
        $newAccessToken = $refreshedAccessTokenObj->getAccessToken();
        $newRefreshToken = $refreshedAccessTokenObj->getRefreshToken();

        $tokensToSave = [
            'access_token' => $newAccessToken,
            'refresh_token' => $newRefreshToken,
            'last_updated' => date('Y-m-d H:i:s')
        ];
        if (file_put_contents($tokenStorageFile, json_encode($tokensToSave, JSON_PRETTY_PRINT))) {
            debugLog("SUCCESS: Tokens refreshed and saved by debug script.");
            $loadedTokens['access_token'] = $newAccessToken;
            $loadedTokens['refresh_token'] = $newRefreshToken;
            return true;
        } else {
            debugLog("ERROR: Tokens refreshed but failed to save to {$tokenStorageFile}.");
            return false;
        }
    } catch (Exception $e) {
        debugLog("ERROR during token refresh: " . $e->getMessage());
        if ($e instanceof IdsException) {
            debugLog("Refresh Response Body: " . $e->getResponseBody());
        }
        return false;
    }
}


$qboBaseConfig = [
    'auth_mode'       => 'oauth2',
    'ClientID'        => "AB3WEwQtfG1Ws43pqxY7Y2Ok3s7QfN97VniZxSssaQhl9PA4JE", // YOUR CLIENT ID
    'ClientSecret'    => "GRjmj4WxpMo5ZkMOjfnEQ5LmzXkAna1xb9YpwEWL", // YOUR CLIENT SECRET
    'QBORealmID'      => "9341454710400258",          // YOUR REALM ID
    'baseUrl'         => "Development",             // Or "Production"
];
// --- End Configuration & Helpers ---

debugLog("===================================================");
debugLog("Starting Vendor Debug Script.");

if ($argc < 2) {
    debugLog("Usage: php debug_qbo_vendor.php \"Vendor Name\" [OptionalVendorID]");
    debugLog("Example: php debug_qbo_vendor.php \"Merlin Entertainment\" 279");
    exit(1);
}

$vendorNameToDebug = trim($argv[1]);
$optionalVendorIdToDebug = isset($argv[2]) ? trim($argv[2]) : null;

debugLog("Vendor Name to Debug: '$vendorNameToDebug'");
if ($optionalVendorIdToDebug) {
    debugLog("Optional Vendor ID to also fetch: '$optionalVendorIdToDebug'");
}

$currentTokens = loadTokensDebug();
if (!$currentTokens) {
    debugLog("CRITICAL: Could not load tokens. Exiting.");
    exit(1);
}

// Attempt to refresh tokens
refreshTokenDebug($qboBaseConfig, $currentTokens);

$qboConfigForLibrary = $qboBaseConfig;
if (empty($currentTokens['access_token']) || empty($currentTokens['refresh_token'])) {
    debugLog("CRITICAL: Access or Refresh token is missing after refresh attempt. Exiting.");
    exit(1);
}
$qboConfigForLibrary['accessTokenKey'] = $currentTokens['access_token'];
$qboConfigForLibrary['refreshTokenKey'] = $currentTokens['refresh_token'];

try {
    $qboLibrary = new QBOUtilityLibrary($qboConfigForLibrary);
    $dataService = $qboLibrary->getDataService(); // Get the underlying DataService
    debugLog("QBOUtilityLibrary instantiated.");
} catch (Exception $e) {
    debugLog("CRITICAL: Failed to instantiate QBOUtilityLibrary: " . $e->getMessage());
    exit(1);
}

// --- 1. Query by DisplayName (Active only - default behavior of QBOUtilityLibrary's queryEntityByField) ---
debugLog("\n--- STEP 1: Querying for ACTIVE Vendor by DisplayName: '$vendorNameToDebug' ---");
try {
    $activeVendor = $qboLibrary->queryEntityByField('Vendor', 'DisplayName', $vendorNameToDebug);
    if ($activeVendor) {
        debugLog("Found ACTIVE Vendor by DisplayName:");
        debugLog("  ID: " . ($activeVendor->Id ?? 'N/A'));
        debugLog("  DisplayName: '" . ($activeVendor->DisplayName ?? 'N/A') . "' (Hex: " . bin2hex($activeVendor->DisplayName ?? '') . ")");
        debugLog("  Active: " . (isset($activeVendor->Active) ? ($activeVendor->Active ? 'true' : 'false') : 'N/A'));
        // debugLog("  Full Object: " . json_encode($activeVendor, JSON_PRETTY_PRINT));
    } else {
        debugLog("No ACTIVE Vendor found with DisplayName: '$vendorNameToDebug'");
    }
} catch (IdsException $e) {
    $responseBody = 'N/A';
    $httpStatusCode = 'N/A';
    $qboErrorCodeDetails = 'N/A'; // To store the code from the XML body if present

    // Try to get HTTP status code
    if (method_exists($e, 'getHttpStatusCode')) {
        $httpStatusCode = $e->getHttpStatusCode();
    } elseif ($e->getPrevious() && method_exists($e->getPrevious(), 'getCode')) { // Guzzle exceptions often have code here
        $httpStatusCode = $e->getPrevious()->getCode();
    }


    // Try to get the response body
    if (method_exists($e, 'getResponseBody') && $e->getResponseBody()) {
        $responseBody = $e->getResponseBody();
    } elseif ($e->getPrevious() && method_exists($e->getPrevious(), 'hasResponse') && $e->getPrevious()->hasResponse()) {
        $response = $e->getPrevious()->getResponse();
        if ($response) {
            $responseBody = (string) $response->getBody();
        }
    }

    // Try to parse error code from XML response body
    if ($responseBody !== 'N/A' && strpos($responseBody, '<?xml') === 0) {
        try {
            $xml = new SimpleXMLElement($responseBody);
            if (isset($xml->Fault) && isset($xml->Fault->Error) && isset($xml->Fault->Error['code'])) {
                $qboErrorCodeDetails = (string) $xml->Fault->Error['code'];
            }
        } catch (Exception $xmlEx) {
            // XML parsing failed, do nothing extra
        }
    }


    debugLog("API Error: " . $e->getMessage());
    debugLog("  HTTP Status: " . $httpStatusCode);
    debugLog("  QBO Error Code from XML (if any): " . $qboErrorCodeDetails);
    debugLog("  Raw Response Body: " . $responseBody); // Log the raw body

    if (strpos($responseBody, "Duplicate Name Exists Error") !== false || strpos($responseBody, "The name supplied already exists") !== false) {
        debugLog("  >> This is the 'Duplicate Name Exists Error'. Check the Response Body for the ID of the conflicting record.");
    } elseif (strpos($responseBody, "Object Not Found") !== false) {
        debugLog("  >> This indicates 'Object Not Found'. The ID likely doesn't exist for this entity type or has been made inactive/deleted.");
    }
} catch (Exception $e) { // Generic PHP exception
    debugLog("PHP Error: " . $e->getMessage());
    // You might want to add $e->getTraceAsString() here for more context on PHP errors
    // debugLog("  Trace: " . $e->getTraceAsString());
}

// --- 2. Query by DisplayName (Including Inactive) ---
debugLog("\n--- STEP 2: Querying for ANY Vendor (Active or Inactive) by DisplayName: '$vendorNameToDebug' ---");
$queryStringInactive = "SELECT * FROM Vendor WHERE DisplayName = '" . addslashes($vendorNameToDebug) . "' AND Active IN (true, false) MAXRESULTS 5";
// Note: QQL does not guarantee finding all if DisplayName is not unique, even with Active IN.
// It's better to use a more specific unique field if possible, or iterate if multiple results.
// For this debug, we'll see what the first few results are.
debugLog("Executing QQL: $queryStringInactive");
try {
    $allMatchingVendors = $dataService->Query($queryStringInactive);
    if (!empty($allMatchingVendors) && count($allMatchingVendors) > 0) {
        debugLog("Found " . count($allMatchingVendors) . " vendor(s) (active or inactive) matching DisplayName '$vendorNameToDebug':");
        foreach ($allMatchingVendors as $i => $vendor) {
            debugLog("  Result " . ($i + 1) . ":");
            debugLog("    ID: " . ($vendor->Id ?? 'N/A'));
            debugLog("    DisplayName: '" . ($vendor->DisplayName ?? 'N/A') . "' (Hex: " . bin2hex($vendor->DisplayName ?? '') . ")");
            debugLog("    Active: " . (isset($vendor->Active) ? ($vendor->Active ? 'true' : 'false') : 'N/A'));
            // debugLog("    Full Object: " . json_encode($vendor, JSON_PRETTY_PRINT));
        }
    } else {
        debugLog("No Vendors (active or inactive) found with DisplayName: '$vendorNameToDebug' using direct query.");
    }
} catch (IdsException $e) {
    $responseBody = 'N/A';
    $httpStatusCode = 'N/A';
    $qboErrorCodeDetails = 'N/A'; // To store the code from the XML body if present

    // Try to get HTTP status code
    if (method_exists($e, 'getHttpStatusCode')) {
        $httpStatusCode = $e->getHttpStatusCode();
    } elseif ($e->getPrevious() && method_exists($e->getPrevious(), 'getCode')) { // Guzzle exceptions often have code here
        $httpStatusCode = $e->getPrevious()->getCode();
    }


    // Try to get the response body
    if (method_exists($e, 'getResponseBody') && $e->getResponseBody()) {
        $responseBody = $e->getResponseBody();
    } elseif ($e->getPrevious() && method_exists($e->getPrevious(), 'hasResponse') && $e->getPrevious()->hasResponse()) {
        $response = $e->getPrevious()->getResponse();
        if ($response) {
            $responseBody = (string) $response->getBody();
        }
    }

    // Try to parse error code from XML response body
    if ($responseBody !== 'N/A' && strpos($responseBody, '<?xml') === 0) {
        try {
            $xml = new SimpleXMLElement($responseBody);
            if (isset($xml->Fault) && isset($xml->Fault->Error) && isset($xml->Fault->Error['code'])) {
                $qboErrorCodeDetails = (string) $xml->Fault->Error['code'];
            }
        } catch (Exception $xmlEx) {
            // XML parsing failed, do nothing extra
        }
    }


    debugLog("API Error: " . $e->getMessage());
    debugLog("  HTTP Status: " . $httpStatusCode);
    debugLog("  QBO Error Code from XML (if any): " . $qboErrorCodeDetails);
    debugLog("  Raw Response Body: " . $responseBody); // Log the raw body

    if (strpos($responseBody, "Duplicate Name Exists Error") !== false || strpos($responseBody, "The name supplied already exists") !== false) {
        debugLog("  >> This is the 'Duplicate Name Exists Error'. Check the Response Body for the ID of the conflicting record.");
    } elseif (strpos($responseBody, "Object Not Found") !== false) {
        debugLog("  >> This indicates 'Object Not Found'. The ID likely doesn't exist for this entity type or has been made inactive/deleted.");
    }
} catch (Exception $e) { // Generic PHP exception
    debugLog("PHP Error: " . $e->getMessage());
    // You might want to add $e->getTraceAsString() here for more context on PHP errors
    // debugLog("  Trace: " . $e->getTraceAsString());
}


// --- 3. If an Optional Vendor ID was provided, fetch it directly ---
if ($optionalVendorIdToDebug) {
    debugLog("\n--- STEP 3: Fetching Vendor directly by Provided ID: '$optionalVendorIdToDebug' ---");
    try {
        $vendorById = $dataService->FindById('Vendor', $optionalVendorIdToDebug);
        if ($vendorById) {
            debugLog("Found Vendor by ID '$optionalVendorIdToDebug':");
            debugLog("  ID: " . ($vendorById->Id ?? 'N/A'));
            debugLog("  DisplayName: '" . ($vendorById->DisplayName ?? 'N/A') . "' (Hex: " . bin2hex($vendorById->DisplayName ?? '') . ")");
            debugLog("  Active: " . (isset($vendorById->Active) ? ($vendorById->Active ? 'true' : 'false') : 'N/A'));
            // debugLog("  Full Object: " . json_encode($vendorById, JSON_PRETTY_PRINT));
        } else {
            // FindById usually throws an exception if not found, so this else might not be hit.
            debugLog("No Vendor found with ID: '$optionalVendorIdToDebug' (FindById returned null/false).");
        }
    } catch (IdsException $e) {
    $responseBody = 'N/A';
    $httpStatusCode = 'N/A';
    $qboErrorCodeDetails = 'N/A'; // To store the code from the XML body if present

    // Try to get HTTP status code
    if (method_exists($e, 'getHttpStatusCode')) {
        $httpStatusCode = $e->getHttpStatusCode();
    } elseif ($e->getPrevious() && method_exists($e->getPrevious(), 'getCode')) { // Guzzle exceptions often have code here
        $httpStatusCode = $e->getPrevious()->getCode();
    }


    // Try to get the response body
    if (method_exists($e, 'getResponseBody') && $e->getResponseBody()) {
        $responseBody = $e->getResponseBody();
    } elseif ($e->getPrevious() && method_exists($e->getPrevious(), 'hasResponse') && $e->getPrevious()->hasResponse()) {
        $response = $e->getPrevious()->getResponse();
        if ($response) {
            $responseBody = (string) $response->getBody();
        }
    }

    // Try to parse error code from XML response body
    if ($responseBody !== 'N/A' && strpos($responseBody, '<?xml') === 0) {
        try {
            $xml = new SimpleXMLElement($responseBody);
            if (isset($xml->Fault) && isset($xml->Fault->Error) && isset($xml->Fault->Error['code'])) {
                $qboErrorCodeDetails = (string) $xml->Fault->Error['code'];
            }
        } catch (Exception $xmlEx) {
            // XML parsing failed, do nothing extra
        }
    }


    debugLog("API Error: " . $e->getMessage());
    debugLog("  HTTP Status: " . $httpStatusCode);
    debugLog("  QBO Error Code from XML (if any): " . $qboErrorCodeDetails);
    debugLog("  Raw Response Body: " . $responseBody); // Log the raw body

    if (strpos($responseBody, "Duplicate Name Exists Error") !== false || strpos($responseBody, "The name supplied already exists") !== false) {
        debugLog("  >> This is the 'Duplicate Name Exists Error'. Check the Response Body for the ID of the conflicting record.");
    } elseif (strpos($responseBody, "Object Not Found") !== false) {
        debugLog("  >> This indicates 'Object Not Found'. The ID likely doesn't exist for this entity type or has been made inactive/deleted.");
    }
} catch (Exception $e) { // Generic PHP exception
    debugLog("PHP Error: " . $e->getMessage());
    // You might want to add $e->getTraceAsString() here for more context on PHP errors
    // debugLog("  Trace: " . $e->getTraceAsString());
}
}

// --- 4. Attempt to CREATE a new vendor with the specified name ---
// (This is what your getOrCreateVendor would do if it doesn't find an active one)
debugLog("\n--- STEP 4: Attempting to CREATE a NEW Vendor with DisplayName: '$vendorNameToDebug' ---");
$vendorDataToCreate = [
    "DisplayName" => $vendorNameToDebug,
    "CompanyName" => $vendorNameToDebug // Often same as DisplayName for vendors
    // Add CurrencyRef here if you test with non-home currency vendors
];
try {
    // We use the Facade directly here for a clean create attempt
    $createdVendorObject = $dataService->Add(Vendor::create($vendorDataToCreate));
    if ($createdVendorObject && !empty($createdVendorObject->Id)) {
        debugLog("SUCCESSFULLY CREATED new Vendor via API with DisplayName '$vendorNameToDebug':");
        debugLog("  New Vendor ID: " . ($createdVendorObject->Id ?? 'N/A'));
        debugLog("  DisplayName: '" . ($createdVendorObject->DisplayName ?? 'N/A') . "'");
        debugLog("  Active: " . (isset($createdVendorObject->Active) ? ($createdVendorObject->Active ? 'true' : 'false') : 'N/A'));
        debugLog("  This new vendor (ID: {$createdVendorObject->Id}) might need to be made inactive or merged in QBO if it's a duplicate of an existing one you intend to use.");
    } else {
        debugLog("Attempt to create vendor with DisplayName '$vendorNameToDebug' did not return a valid object or ID, though no exception was thrown (Unusual).");
    }
} catch (IdsException $e) {
    $responseBody = 'N/A';
    $httpStatusCode = 'N/A';
    $qboErrorCodeDetails = 'N/A'; // To store the code from the XML body if present

    // Try to get HTTP status code
    if (method_exists($e, 'getHttpStatusCode')) {
        $httpStatusCode = $e->getHttpStatusCode();
    } elseif ($e->getPrevious() && method_exists($e->getPrevious(), 'getCode')) { // Guzzle exceptions often have code here
        $httpStatusCode = $e->getPrevious()->getCode();
    }


    // Try to get the response body
    if (method_exists($e, 'getResponseBody') && $e->getResponseBody()) {
        $responseBody = $e->getResponseBody();
    } elseif ($e->getPrevious() && method_exists($e->getPrevious(), 'hasResponse') && $e->getPrevious()->hasResponse()) {
        $response = $e->getPrevious()->getResponse();
        if ($response) {
            $responseBody = (string) $response->getBody();
        }
    }

    // Try to parse error code from XML response body
    if ($responseBody !== 'N/A' && strpos($responseBody, '<?xml') === 0) {
        try {
            $xml = new SimpleXMLElement($responseBody);
            if (isset($xml->Fault) && isset($xml->Fault->Error) && isset($xml->Fault->Error['code'])) {
                $qboErrorCodeDetails = (string) $xml->Fault->Error['code'];
            }
        } catch (Exception $xmlEx) {
            // XML parsing failed, do nothing extra
        }
    }


    debugLog("API Error: " . $e->getMessage());
    debugLog("  HTTP Status: " . $httpStatusCode);
    debugLog("  QBO Error Code from XML (if any): " . $qboErrorCodeDetails);
    debugLog("  Raw Response Body: " . $responseBody); // Log the raw body

    if (strpos($responseBody, "Duplicate Name Exists Error") !== false || strpos($responseBody, "The name supplied already exists") !== false) {
        debugLog("  >> This is the 'Duplicate Name Exists Error'. Check the Response Body for the ID of the conflicting record.");
    } elseif (strpos($responseBody, "Object Not Found") !== false) {
        debugLog("  >> This indicates 'Object Not Found'. The ID likely doesn't exist for this entity type or has been made inactive/deleted.");
    }
} catch (Exception $e) { // Generic PHP exception
    debugLog("PHP Error: " . $e->getMessage());
    // You might want to add $e->getTraceAsString() here for more context on PHP errors
    // debugLog("  Trace: " . $e->getTraceAsString());
}

debugLog("\nVendor Debug Script Finished.");
debugLog("===================================================\n");
exit(0);

?>