<?php
// qbo_sync_processor.php

// It's assumed that dbconn.php, qbo_functions.php, and quote_pricing.php (if used directly)
// are included by the script that calls processQuoteForQBOSync.
// QBOUtilityLibrary and $conn (mysqli object) are passed as arguments.

use QuickBooksOnline\API\Exception\IdsException;

/**
 * Processes a Vtiger quote for synchronization with QuickBooks Online.
 * This includes creating/updating customer, invoice, vendors, and bills.
 *
 * @param mixed $identifier The Vtiger Quote ID (numeric) or Quote Number (string).
 * @param mysqli $conn The active Vtiger database connection.
 * @param QBOUtilityLibrary $qboLibrary An instantiated QBOUtilityLibrary object.
 * @param array $config Optional configuration overrides.
 * @return array A structured array containing the results and logs of the sync process.
 */
function processQuoteForQBOSync($identifier, mysqli $conn, QBOUtilityLibrary $qboLibrary, array $config = []) {
    $output_data = [
        'identifier_processed' => $identifier,
        'vtiger_quote_id' => null,
        'vtiger_quote_no' => null,
        'quote_country' => null,
        'date_of_travel' => null,
        'customer_sync_status' => ['status' => 'Not Processed', 'qbo_customer_id' => null, 'message' => null],
        'invoice_sync_status' => ['status' => 'Not Processed', 'qbo_invoice_id' => null, 'doc_number' => null, 'message' => null, 'qbo_invoice_link' => null],
        'vendors_sync_status' => [],
        'sql_queries_executed' => [],
        'debug_notes' => [],
        'qbo_operations_log' => [],
        'config_used' => [],
        'error' => null
    ];

    // Default configuration values
    $default_config = [
        'QBO_DEFAULT_INCOME_ACCOUNT_ID_FOR_SERVICES' => '84',
        'QBO_DEFAULT_EXPENSE_ACCOUNT_ID_FOR_PURCHASES' => '8',
        'QBO_COST_OF_SALES_ACCOUNT_NAME' => 'Cost of sales',
        'QBO_DEFAULT_TERM_NAME_BILL' => 'Due on receipt',
        'DEFAULT_QBO_HOME_CURRENCY' => 'AUD',
        'TARGET_GST_FREE_TAX_CODE_NAME_BILL' => 'GST-free non-capital',
        'TARGET_GST_FREE_TAX_CODE_NAME_INVOICE' => 'GST free',
        'FALLBACK_NON_TAXABLE_TAX_CODE_ID_INVOICE' => 'NON',
        'SALES_CATEGORY_ITEM_TYPE' => 'Service',
        'DEFAULT_INCOME_ACCOUNT_ID_FOR_SALES_CATEGORY' => '84',
        'QBO_DEFAULT_TERM_NAME_INVOICE' => 'Due on receipt',
        'GENERIC_SUPPLIER_ITEM_NAME' => "General Supplier Services"
    ];
    $current_config = array_merge($default_config, $config);
    $output_data['config_used'] = $current_config;

    $quoteid = null;
    $quote_no_ref_from_db = null;

    // --- 1. Determine Quote ID and Quote Number from identifier ---
    if (is_numeric($identifier)) {
        $quoteid = mysqli_real_escape_string($conn, $identifier);
        $sql_find_q_no = "SELECT quote_no FROM vtiger_quotes WHERE quoteid = '$quoteid' LIMIT 1;";
        $output_data['sql_queries_executed'][] = $sql_find_q_no;
        $result_find_q_no = mysqli_query($conn, $sql_find_q_no);
        if ($result_find_q_no && mysqli_num_rows($result_find_q_no) > 0) {
            $quote_no_ref_from_db = mysqli_fetch_assoc($result_find_q_no)['quote_no'];
        } else {
            $output_data['error'] = "Could not find quote_no for quoteid: $quoteid";
            if($result_find_q_no) mysqli_free_result($result_find_q_no);
            return $output_data;
        }
        if($result_find_q_no) mysqli_free_result($result_find_q_no);
    } elseif (is_string($identifier) && !empty($identifier)) {
        $quote_no_ref_from_db = mysqli_real_escape_string($conn, $identifier);
        $sql_find_qid = "SELECT quoteid FROM vtiger_quotes WHERE quote_no = '$quote_no_ref_from_db' LIMIT 1;";
        $output_data['sql_queries_executed'][] = $sql_find_qid;
        $result_find_qid = mysqli_query($conn, $sql_find_qid);
        if ($result_find_qid && mysqli_num_rows($result_find_qid) > 0) {
            $quoteid = mysqli_fetch_assoc($result_find_qid)['quoteid'];
        } else {
            $output_data['error'] = "Could not find quoteid for quote_no: $quote_no_ref_from_db";
            if($result_find_qid) mysqli_free_result($result_find_qid);
            return $output_data;
        }
        if($result_find_qid) mysqli_free_result($result_find_qid);
    } else {
        $output_data['error'] = "Invalid identifier provided.";
        return $output_data;
    }

    if (empty($quoteid) || empty($quote_no_ref_from_db)) {
         $output_data['error'] = "Failed to resolve Quote ID and Quote Number from identifier: " . htmlspecialchars($identifier);
         return $output_data;
    }
    $output_data['vtiger_quote_id'] = $quoteid;
    $output_data['vtiger_quote_no'] = $quote_no_ref_from_db;
    $output_data['qbo_operations_log'][] = "Resolved: Vtiger Quote ID='{$quoteid}', Quote No='{$quote_no_ref_from_db}'.";

    // Fetch general quote metadata (country, date of travel)
    $sql_quote_meta_for_processor = "SELECT vq.country AS QuoteCountryGlobal, vqcf.cf_1162 AS DateOfTravelActual
                       FROM vtiger_quotes vq
                       LEFT JOIN vtiger_quotescf vqcf ON vq.quoteid = vqcf.quoteid
                       WHERE vq.quoteid = '" . mysqli_real_escape_string($conn, $quoteid) . "';";
    $output_data['sql_queries_executed'][] = $sql_quote_meta_for_processor;
    $result_quote_meta_for_processor = mysqli_query($conn, $sql_quote_meta_for_processor);
    if ($result_quote_meta_for_processor && mysqli_num_rows($result_quote_meta_for_processor) > 0) {
        $row_meta_processor = mysqli_fetch_assoc($result_quote_meta_for_processor);
        $output_data['quote_country'] = $row_meta_processor['QuoteCountryGlobal'];
        if (!empty($row_meta_processor['DateOfTravelActual']) && $row_meta_processor['DateOfTravelActual'] != '0000-00-00') {
            try { $output_data['date_of_travel'] = (new DateTime($row_meta_processor['DateOfTravelActual']))->format('Y-m-d'); }
            catch (Exception $e) { $output_data['date_of_travel'] = date('Y-m-d'); } // Default to today on parse error
        } else { $output_data['date_of_travel'] = date('Y-m-d'); } // Default to today if not set
    } else {
        $output_data['debug_notes'][] = "Could not fetch general meta for quote ID: {$quoteid}. Using defaults.";
        $output_data['date_of_travel'] = date('Y-m-d'); // Default to today
    }
    if($result_quote_meta_for_processor) mysqli_free_result($result_quote_meta_for_processor);


    // --- COMMON PRE-PROCESSING: QBO Accounts and Classes ---
    $qboCostOfSalesAccountId = null;
    try {
        $cosAccountName = $current_config['QBO_COST_OF_SALES_ACCOUNT_NAME'];
        $output_data['qbo_operations_log'][] = "Finding QBO Account: '{$cosAccountName}'.";
        $cosAccountObject = $qboLibrary->findAccountByName($cosAccountName);
        if ($cosAccountObject && !empty($cosAccountObject->Id)) {
            $qboCostOfSalesAccountId = $cosAccountObject->Id;
            $output_data['qbo_operations_log'][] = "QBO Account '{$cosAccountObject->Name}' (ID: {$qboCostOfSalesAccountId}) found.";
        } else { $output_data['debug_notes'][] = "WARNING: QBO Account '{$cosAccountName}' NOT FOUND."; }
    } catch (Exception $e) { $output_data['debug_notes'][] = "Error finding QBO Account '{$cosAccountName}': " . $e->getMessage(); }

    $qboClassRefId = null;
    if (!empty($quote_no_ref_from_db)) {
        try {
            $output_data['qbo_operations_log'][] = "Getting/Creating QBO Class: '{$quote_no_ref_from_db}'.";
            $classObject = $qboLibrary->getOrCreateClassByName($quote_no_ref_from_db);
            if ($classObject && !empty($classObject->Id)) {
                $qboClassRefId = $classObject->Id;
                $output_data['qbo_operations_log'][] = "QBO Class '{$classObject->Name}' (ID: {$qboClassRefId}) processed.";
            } else { $output_data['debug_notes'][] = "Failed to get/create QBO Class '{$quote_no_ref_from_db}'."; }
        } catch (Exception $e) { $output_data['debug_notes'][] = "Error QBO Class '{$quote_no_ref_from_db}': " . $e->getMessage();}
    } else { $output_data['debug_notes'][] = "Vtiger quote_no is empty; lines will not be classified."; }


    // =========================================================================
    // --- Start: Customer and Invoice Processing ---
    // =========================================================================
    $output_data['qbo_operations_log'][] = "--- Starting Customer and Invoice Processing ---";
    try {
        // Fetch Vtiger data for customer and invoice
        $sql_customer_quote_info = "SELECT vq.subject, vq.adults, vq.children, vq.infants, va.organization_name AS accountname, va.address AS full_address, va.country AS bill_country_from_org, va.email AS account_email, vqcf.cf_1182 AS payment_due_date_custom_field, vqcf.cf_1162 AS invoice_date_custom_field, vq.country AS quote_country_for_invoice_logic FROM vtiger_quotes vq LEFT JOIN vtiger_quotescf vqcf ON vq.quoteid = vqcf.quoteid LEFT JOIN tdu_organisation va ON vq.accountid = va.organizationid WHERE vq.quoteid = '$quoteid' LIMIT 1";
        $output_data['sql_queries_executed'][] = $sql_customer_quote_info;
        $result_customer_quote_info = mysqli_query($conn, $sql_customer_quote_info);

        if (!$result_customer_quote_info || mysqli_num_rows($result_customer_quote_info) == 0) {
            if($result_customer_quote_info) mysqli_free_result($result_customer_quote_info);
            throw new Exception("Customer-related quote details not found for ID: {$quoteid}");
        }
        $row_customer_quote = mysqli_fetch_assoc($result_customer_quote_info);
        mysqli_free_result($result_customer_quote_info);

        // Prepare customer and invoice variables
        $accountName_inv = isset($row_customer_quote['accountname']) ? str_replace('&','and',$row_customer_quote['accountname']) : 'Unknown Customer';
        $customer_email_inv = $row_customer_quote['account_email'] ?? null;
        $address_line1_inv = isset($row_customer_quote['full_address']) ? str_replace('&','and',$row_customer_quote['full_address']) : '';
        $address_country_inv = $row_customer_quote['bill_country_from_org'] ??  '';
        $quote_country_inv = $row_customer_quote['quote_country_for_invoice_logic'] ?? $output_data['quote_country'];

        $invoice_currency = $current_config['DEFAULT_QBO_HOME_CURRENCY'];
        if (strcasecmp($quote_country_inv, 'New Zealand') == 0) $invoice_currency = 'NZD';

        $invoice_transaction_date = $output_data['date_of_travel']; // Default to travel date
        if (isset($row_customer_quote['invoice_date_custom_field']) && !empty($row_customer_quote['invoice_date_custom_field']) && $row_customer_quote['invoice_date_custom_field'] != '0000-00-00') {
            try { $invoice_transaction_date = (new DateTime($row_customer_quote['invoice_date_custom_field']))->format('Y-m-d'); } catch (Exception $e) { /* Keep default */ }
        }
        $output_data['invoice_transaction_date_used'] = $invoice_transaction_date;

        $payment_deadline_inv = '';
        if (isset($row_customer_quote['payment_due_date_custom_field']) && $row_customer_quote['payment_due_date_custom_field']!='0000-00-00' && !empty($row_customer_quote['payment_due_date_custom_field'])) {
            try { $payment_deadline_inv = (new DateTime($row_customer_quote['payment_due_date_custom_field']))->format('Y-m-d'); } catch (Exception $e) { /* Keep empty */ }
        }

        // Determine and get/create Sales Category Item for Invoice
        $salesCategoryItemName_inv = '';
        $isGroupQuote_inv = (substr(strtoupper($quote_no_ref_from_db), -1) === 'G');
        if (strcasecmp($quote_country_inv, 'Australia') == 0) { $salesCategoryItemName_inv = $isGroupQuote_inv ? 'Sales - AU Groups' : 'Sales'; }
        elseif (strcasecmp($quote_country_inv, 'New Zealand') == 0) { $salesCategoryItemName_inv = $isGroupQuote_inv ? 'Sales - NZ Groups' : 'Sales - Newzealand'; }
        $salesCategoryItemId_inv = null;

        if (!empty($salesCategoryItemName_inv)) {
             if (empty($current_config['DEFAULT_INCOME_ACCOUNT_ID_FOR_SALES_CATEGORY']) || $current_config['DEFAULT_INCOME_ACCOUNT_ID_FOR_SALES_CATEGORY'] === 'YOUR_INCOME_ACCOUNT_ID') {
                 $output_data['debug_notes'][] = "CRITICAL CONFIG: DEFAULT_INCOME_ACCOUNT_ID_FOR_SALES_CATEGORY not correctly set. Sales Category Item '{$salesCategoryItemName_inv}' cannot be properly created/used.";
            } else {
                $output_data['qbo_operations_log'][] = "Getting/Creating Sales Category Item for Invoice: '{$salesCategoryItemName_inv}'.";
                $defaultItemDataSalesCat = [ 'Type' => $current_config['SALES_CATEGORY_ITEM_TYPE'], 'IncomeAccountRef' => ['value' => $current_config['DEFAULT_INCOME_ACCOUNT_ID_FOR_SALES_CATEGORY']], 'TrackQtyOnHand' => false ];
                $salesCategoryItemObjectInv = $qboLibrary->getOrCreateItemByName($salesCategoryItemName_inv, $defaultItemDataSalesCat);
                if ($salesCategoryItemObjectInv && !empty($salesCategoryItemObjectInv->Id)) {
                    $salesCategoryItemId_inv = $salesCategoryItemObjectInv->Id;
                    $output_data['qbo_operations_log'][] = "Sales Category Item '{$salesCategoryItemObjectInv->Name}' (ID: {$salesCategoryItemId_inv}) processed for invoice.";
                } else { $output_data['debug_notes'][] = "Failed to get/create Sales Category Item '{$salesCategoryItemName_inv}' for invoice."; }
            }
        } else { $output_data['debug_notes'][] = "Sales Category Item Name for invoice not determined for country '{$quote_country_inv}'."; }

        // Get/create QBO Term for Invoice
        $qboTermRefIdForInvoice = null;
        $termNameForInvoice = $current_config['QBO_DEFAULT_TERM_NAME_INVOICE'];
        $defaultTermDataForInvoice = ['Name' => $termNameForInvoice, 'Type' => 'STANDARD', 'DueDays' => 0];
        $output_data['qbo_operations_log'][] = "Getting/Creating QBO Term for Invoices: '{$termNameForInvoice}'.";
        $termInvoiceObject = $qboLibrary->getOrCreateTermByName($termNameForInvoice, $defaultTermDataForInvoice);
        if ($termInvoiceObject && !empty($termInvoiceObject->Id)) {
            $qboTermRefIdForInvoice = $termInvoiceObject->Id;
            $output_data['qbo_operations_log'][] = "QBO Term '{$termInvoiceObject->Name}' (ID: {$qboTermRefIdForInvoice}) processed for invoice.";
        } else { $output_data['debug_notes'][] = "Failed to get/create QBO Term '{$termNameForInvoice}' for invoice.";}

        // Find QBO TaxCode for Invoice lines
        $taxCodeIdForInvoiceLines = $current_config['FALLBACK_NON_TAXABLE_TAX_CODE_ID_INVOICE']; // Fallback
        $taxCodeNameInvoice = $current_config['TARGET_GST_FREE_TAX_CODE_NAME_INVOICE'];
        $output_data['qbo_operations_log'][] = "Finding QBO TaxCode for Invoices: '{$taxCodeNameInvoice}'.";
        $taxCodeInvoiceObject = $qboLibrary->findTaxCodeByName($taxCodeNameInvoice);
        if ($taxCodeInvoiceObject && !empty($taxCodeInvoiceObject->Id)) {
            $taxCodeIdForInvoiceLines = $taxCodeInvoiceObject->Id;
            $output_data['qbo_operations_log'][] = "QBO TaxCode '{$taxCodeInvoiceObject->Name}' (ID: {$taxCodeIdForInvoiceLines}) found for invoice lines.";
        } else { $output_data['debug_notes'][] = "QBO TaxCode '{$taxCodeNameInvoice}' NOT FOUND for invoice. Using fallback '{$taxCodeIdForInvoiceLines}'.";}

        // --- Build Invoice Line Items ---
        $contentForInvoiceLines = [];
        $adults_no_inv = (int)($row_customer_quote['adults']??0);
        $children_no_inv = (int)($row_customer_quote['children']??0);
        $infants_no_inv = (int)($row_customer_quote['infants']??0);
        $single_rooms_inv = 0; $double_rooms_inv = 0; $triple_rooms_inv = 0; $child_no_bed_inv = 0; $child_with_bed_inv = 0;

        // Fetch room configuration for invoice pricing
        $created_at_query_inv_rooms = "";
        $invoice_initial_query_inv = "SELECT MAX(created_at) AS created_at FROM vtiger_invoice WHERE quoteid = '$quoteid' AND type = 'final'";
        $output_data['sql_queries_executed'][] = $invoice_initial_query_inv;
        $result_invoice_initial_inv = mysqli_query($conn, $invoice_initial_query_inv);
        if ($result_invoice_initial_inv && $invoice_initial_inv_row = mysqli_fetch_assoc($result_invoice_initial_inv)) {
            if (!empty($invoice_initial_inv_row['created_at'])) {
                $created_at_query_inv_rooms = " AND created_at < '" . mysqli_real_escape_string($conn, $invoice_initial_inv_row['created_at']) . "' ";
            }
        }
        if($result_invoice_initial_inv) mysqli_free_result($result_invoice_initial_inv);

        $sql_rooms_inv = "SELECT meta_key, meta_value FROM vtiger_itinerary WHERE quoteid = '$quoteid' AND meta_key IN ('single_rooms', 'double_rooms', 'triple_rooms', 'child_without_bed') AND (meta_key, created_at) IN (SELECT meta_key, MAX(created_at) FROM vtiger_itinerary WHERE quoteid = '$quoteid' {$created_at_query_inv_rooms} AND meta_key IN ('single_rooms', 'double_rooms', 'triple_rooms', 'child_without_bed') GROUP BY meta_key)";
        $output_data['sql_queries_executed'][] = $sql_rooms_inv;
        $result_rooms_inv = mysqli_query($conn, $sql_rooms_inv);
        if ($result_rooms_inv) {
            while ($row_room_inv = mysqli_fetch_assoc($result_rooms_inv)) {
                if ($row_room_inv['meta_key'] == 'single_rooms') $single_rooms_inv = (int)($row_room_inv['meta_value'] ?? 0);
                elseif ($row_room_inv['meta_key'] == 'double_rooms') $double_rooms_inv = (int)($row_room_inv['meta_value'] ?? 0);
                elseif ($row_room_inv['meta_key'] == 'triple_rooms') $triple_rooms_inv = (int)($row_room_inv['meta_value'] ?? 0);
                elseif ($row_room_inv['meta_key'] == 'child_without_bed') $child_no_bed_inv = (int)($row_room_inv['meta_value'] ?? 0);
            }
        } else { $output_data['debug_notes'][] = "DB room query failed for invoice: ".mysqli_error($conn); }
        if($result_rooms_inv) mysqli_free_result($result_rooms_inv);
        $child_with_bed_inv = $children_no_inv - $child_no_bed_inv;
        if ($child_with_bed_inv < 0) $child_with_bed_inv = 0;

        // Determine pricing tier
        $quote_pax_for_pricing_inv = -1; // Default for FIT or if tier not found
        $no_pax_calc_for_tier_inv = $adults_no_inv + $children_no_inv;
        $mode_inv = 'group'; // Assuming group pricing logic for now

        if ($mode_inv == 'group' && $no_pax_calc_for_tier_inv > 0) {
            $mode_sql_condition_inv = '>1'; // Condition for group subquotes
            $sql_group_pricing_tier_inv = "SELECT MIN(vps.pax_min) AS pax_min, MAX(vps.pax_max) AS pax_max FROM vtiger_products_saleprice vps WHERE vps.quoteid='$quoteid' AND vps.subquoteid $mode_sql_condition_inv AND vps.cf_928='Transfers' GROUP BY vps.subquoteid ORDER BY pax_min ASC";
            $output_data['sql_queries_executed'][] = $sql_group_pricing_tier_inv;
            $result_group_pricing_tier_inv = mysqli_query($conn, $sql_group_pricing_tier_inv);
            if ($result_group_pricing_tier_inv) {
                if (mysqli_num_rows($result_group_pricing_tier_inv) > 0) {
                    while ($row_tier_inv = mysqli_fetch_assoc($result_group_pricing_tier_inv)) {
                        if ($row_tier_inv['pax_min'] <= $no_pax_calc_for_tier_inv && $no_pax_calc_for_tier_inv <= $row_tier_inv['pax_max']) {
                            $quote_pax_for_pricing_inv = $row_tier_inv['pax_min'];
                            break;
                        }
                    }
                }
                mysqli_free_result($result_group_pricing_tier_inv);
            } else { $output_data['debug_notes'][] = "DB group pricing query failed for invoice: ".mysqli_error($conn); }
        }

        // Get pricing using external function
        if (!function_exists('getQuotePricing')) {
            $output_data['debug_notes'][] = "CRITICAL ERROR: getQuotePricing function not found. Invoice pricing will be incorrect.";
            $pricing_inv_values = ['single_room'=>0,'double_room'=>0,'triple_room'=>0,'child_with_bed'=>0,'child_no_bed'=>0,'infant'=>0, 'adult_no_hotel'=>0, 'child_no_hotel'=>0];
        } else {
            $pricing_inv_values = getQuotePricing($conn, $quoteid, $quote_pax_for_pricing_inv);
        }

        if (!is_array($pricing_inv_values) || empty($pricing_inv_values)) {
            $output_data['debug_notes'][] = "WARNING: Pricing info from getQuotePricing() is invalid or empty for Invoice (QID:{$quoteid}, PaxTier:{$quote_pax_for_pricing_inv}). Using zeros.";
            $pricing_inv_values = ['single_room'=>0,'double_room'=>0,'triple_room'=>0,'child_with_bed'=>0,'child_no_bed'=>0,'infant'=>0, 'adult_no_hotel'=>0, 'child_no_hotel'=>0];
        }

        // Assign pricing values
        $pricing_single_room_inv = $pricing_inv_values['single_room']??0;
        $pricing_double_room_inv = $pricing_inv_values['double_room']??0;
        $pricing_triple_room_inv = $pricing_inv_values['triple_room']??0;
        $pricing_child_with_bed_inv = $pricing_inv_values['child_with_bed']??0;
        $pricing_child_no_bed_inv = $pricing_inv_values['child_no_bed']??0;
        $pricing_infant_inv = $pricing_inv_values['infant']??0;
        $pricing_adult_no_hotel_inv = $pricing_inv_values['adult_no_hotel'] ?? $pricing_single_room_inv; // Fallback for no hotel
        $pricing_child_no_hotel_inv = $pricing_inv_values['child_no_hotel'] ?? $pricing_child_no_bed_inv; // Fallback for no hotel

        // Build content lines based on pax and room types
        if ($single_rooms_inv > 0) $contentForInvoiceLines[] = ['Product'=>'Package Cost Per Adult on Single Occupancy', 'Pricing'=>$pricing_single_room_inv, 'Number'=>$single_rooms_inv];
        if ($double_rooms_inv > 0) $contentForInvoiceLines[] = ['Product'=>'Package Cost Per Adult on Double Occupancy', 'Pricing'=>$pricing_double_room_inv, 'Number'=>($double_rooms_inv * 2)];
        if ($triple_rooms_inv > 0) {
            $adults_in_triple_inv = $triple_rooms_inv * 3 - $child_with_bed_inv; if ($adults_in_triple_inv < 0) $adults_in_triple_inv = 0;
            if ($adults_in_triple_inv > 0) $contentForInvoiceLines[] = ['Product'=>'Package Cost Per Adult on Triple Occupancy', 'Pricing'=>$pricing_triple_room_inv, 'Number'=>$adults_in_triple_inv];
        }
        // Handle "No Hotel" scenarios
        if ($single_rooms_inv == 0 && $double_rooms_inv == 0 && $triple_rooms_inv == 0) {
            if($adults_no_inv > 0) $contentForInvoiceLines[] = ['Product'=>'Package Cost Per Adult No Hotel', 'Pricing'=>$pricing_adult_no_hotel_inv, 'Number'=>$adults_no_inv];
            if($children_no_inv > 0) $contentForInvoiceLines[] = ['Product'=>'Package Cost Per Child No Hotel', 'Pricing'=>$pricing_child_no_hotel_inv, 'Number'=>$children_no_inv];
        } else { // If hotels are present, add child bed types
            if ($child_with_bed_inv > 0) $contentForInvoiceLines[] = ['Product'=>'Package Cost Per Child with Bed', 'Pricing'=>$pricing_child_with_bed_inv, 'Number'=>$child_with_bed_inv];
            if ($child_no_bed_inv > 0) $contentForInvoiceLines[] = ['Product'=>'Package Cost Per Child without Bed', 'Pricing'=>$pricing_child_no_bed_inv, 'Number'=>$child_no_bed_inv];
        }
        if($infants_no_inv > 0) $contentForInvoiceLines[] = ['Product'=>'Package Cost Per Infant', 'Pricing'=>$pricing_infant_inv, 'Number'=>$infants_no_inv];

        // Add adjustments
        $sql_adjustments_inv = "SELECT remark, product_field, price_field, quantity FROM vtiger_invoice_add_products WHERE quoteid='$quoteid' ORDER BY created_at DESC";
        $output_data['sql_queries_executed'][] = $sql_adjustments_inv;
        $result_adjustments_inv = mysqli_query($conn, $sql_adjustments_inv);
        if ($result_adjustments_inv) { while ($row_adj_inv = mysqli_fetch_assoc($result_adjustments_inv)) {
            $adj_desc = str_replace('&', 'and', $row_adj_inv['remark'] ?? 'Adjustment');
            if (!empty($row_adj_inv['product_field']) && $row_adj_inv['product_field'] !== $row_adj_inv['remark']) $adj_desc .= " - " . str_replace('&','and',$row_adj_inv['product_field']);
            $contentForInvoiceLines[] = ['Product' => $adj_desc, 'Pricing' => (float)($row_adj_inv['price_field'] ?? 0), 'Number' => (int)($row_adj_inv['quantity'] ?? 1)];
        }} else { $output_data['debug_notes'][] = "DB adjustments query failed for invoice: ".mysqli_error($conn); }
        if($result_adjustments_inv) mysqli_free_result($result_adjustments_inv);

        // Add refunds (as negative lines)
        $sql_refunds_inv = "SELECT product, refund_amount, refund_reason FROM vtiger_invoice_refunds WHERE quoteid='$quoteid' ORDER BY created_at DESC";
        $output_data['sql_queries_executed'][] = $sql_refunds_inv;
        $result_refunds_inv = mysqli_query($conn, $sql_refunds_inv);
        if ($result_refunds_inv) { while ($row_ref_inv = mysqli_fetch_assoc($result_refunds_inv)) {
            $ref_desc = "Reduction: " . str_replace('&','and',$row_ref_inv['product'] ?? 'Refund');
            if (!empty($row_ref_inv['refund_reason'])) $ref_desc .= " (" . $row_ref_inv['refund_reason'] . ")";
            $contentForInvoiceLines[] = ['Product' => $ref_desc, 'Pricing' => - (float)($row_ref_inv['refund_amount'] ?? 0), 'Number' => 1];
        }} else { $output_data['debug_notes'][] = "DB refunds query failed for invoice: ".mysqli_error($conn); }
        if($result_refunds_inv) mysqli_free_result($result_refunds_inv);

        // Convert content lines to QBO Invoice Line Item format
        $qboInvoiceLineItems = [];
        if (!empty($salesCategoryItemId_inv)) {
            foreach ($contentForInvoiceLines as $contentLine) {
                $desc = $contentLine['Product']; $price = (float)$contentLine['Pricing']; $num = (int)$contentLine['Number'];
                // Basic validation for quantity and price
                if ($price != 0 && $num == 0 && $price > 0) $num = 1; // If price but no num, assume 1
                if ($price < 0 && $num < 0) $num = abs($num);      // Ensure positive num for negative price
                if ($num == 0 && $price == 0 && empty(trim($desc))) continue; // Skip empty lines

                $lineAmountInv = round($price * $num, 2);
                $salesItemLineDetailInv = ["ItemRef"=>["value"=>$salesCategoryItemId_inv], "TaxCodeRef"=>["value"=>$taxCodeIdForInvoiceLines], "Qty"=>$num, "UnitPrice"=>$price];
                if ($qboClassRefId) $salesItemLineDetailInv["ClassRef"] = ["value"=>$qboClassRefId, "name"=>$quote_no_ref_from_db];
                if (!empty($output_data['date_of_travel'])) {
                    $salesItemLineDetailInv["ServiceDate"] = $output_data['date_of_travel']; // Format YYYY-MM-DD
                }
                $qboInvoiceLineItems[] = ["Amount"=>$lineAmountInv, "DetailType"=>"SalesItemLineDetail", "SalesItemLineDetail"=>$salesItemLineDetailInv, "Description"=>$desc];
            }
        } else { $output_data['debug_notes'][] = "CRITICAL: Sales Category Item ID for invoice is missing. Lines not added."; }

        // Add a placeholder line if no valid lines were generated but Sales Category Item exists
        if (empty($qboInvoiceLineItems) && $salesCategoryItemId_inv) {
            $output_data['debug_notes'][] = "Invoice had no valid lines from Vtiger, adding a placeholder $0.01 line.";
             $qboInvoiceLineItems[] = [ "Amount" => 0.01, "DetailType" => "SalesItemLineDetail", "SalesItemLineDetail" => ["ItemRef" => ["value" => $salesCategoryItemId_inv], "TaxCodeRef" => ["value" => $taxCodeIdForInvoiceLines], "Qty" => 1, "UnitPrice" => 0.01], "Description" => "Summary for Quote " . $quote_no_ref_from_db];
             if($qboClassRefId) $qboInvoiceLineItems[0]['SalesItemLineDetail']['ClassRef'] = ["value" => $qboClassRefId, "name" => $quote_no_ref_from_db];
        }
        if (empty($qboInvoiceLineItems)) throw new Exception("No line items could be generated for the invoice for QID {$quoteid}.");

        // Prepare customer data for QBO
        $customerDataForQBO = ['DisplayName' => $accountName_inv, 'CompanyName' => $accountName_inv];
        if ($invoice_currency === 'NZD') $customerDataForQBO['CurrencyRef'] = ['value' => 'NZD'];
        if ($customer_email_inv) $customerDataForQBO['PrimaryEmailAddr'] = ["Address" => $customer_email_inv];
        $billAddrInv = [];
        if (!empty($address_line1_inv)) $billAddrInv['Line1'] = $address_line1_inv;
        if (!empty($address_country_inv)) $billAddrInv['Country'] = $address_country_inv;
        if (!empty($billAddrInv)) $customerDataForQBO['BillAddr'] = $billAddrInv;
        $customerConfigForQBO = ['search_field' => 'DisplayName', 'search_value' => $accountName_inv, 'create_data' => $customerDataForQBO];

        // Prepare invoice metadata for QBO
        $invoiceDocNumber = 'C' . date('YmdHis'); // Generate a unique DocNumber
        $invoiceMetaDataForQBO = ["DocNumber" => $invoiceDocNumber, "TxnDate" => $invoice_transaction_date];
        $invoiceMetaDataForQBO["GlobalTaxCalculation"] = "TaxIncluded";
        if ($payment_deadline_inv) $invoiceMetaDataForQBO["DueDate"] = $invoice_transaction_date; // Corrected: DueDate often same as TxnDate or based on terms
        if ($customer_email_inv) $invoiceMetaDataForQBO["BillEmail"] = ["Address" => $customer_email_inv];
        if ($qboTermRefIdForInvoice) $invoiceMetaDataForQBO["SalesTermRef"] = ["value" => $qboTermRefIdForInvoice];
        if ($qboClassRefId && !empty($quote_no_ref_from_db)) {
            $invoiceMetaDataForQBO["ClassRef"] = [
                "value" => $qboClassRefId,
                "name"  => $quote_no_ref_from_db // QBO often likes the name along with the ID
            ];
            $output_data['qbo_operations_log'][] = "Assigning Class '{$quote_no_ref_from_db}' (ID: {$qboClassRefId}) to the entire Invoice.";
        } else {
            $output_data['debug_notes'][] = "Class ID or Name not available for overall Invoice classification. Quote No: {$quote_no_ref_from_db}";
        }

        // Handle currency and exchange rate for invoice
        if ($invoice_currency !== $current_config['DEFAULT_QBO_HOME_CURRENCY']) {
            $invoiceMetaDataForQBO["CurrencyRef"] = ["value" => $invoice_currency];
            $exchangeRateObjectInv = $qboLibrary->getMostRecentExchangeRate($invoice_currency);
            if ($exchangeRateObjectInv && property_exists($exchangeRateObjectInv, 'Rate') && (float)$exchangeRateObjectInv->Rate > 0) {
                $invoiceMetaDataForQBO["ExchangeRate"] = (float)$exchangeRateObjectInv->Rate;
                $output_data['qbo_operations_log'][] = "Invoice: Applied exchange rate ".(float)$exchangeRateObjectInv->Rate." for {$invoice_currency}.";
            } else { $output_data['debug_notes'][] = "Invoice: Could not get valid exchange rate for {$invoice_currency}. QBO will attempt to auto-apply.";}
        } else { $invoiceMetaDataForQBO["CurrencyRef"] = ["value" => $invoice_currency]; } // Home currency

        // Process customer and invoice creation via QBO library
        $output_data['qbo_operations_log'][] = "Attempting QBO Customer/Invoice workflow.";
        $qboCustomerInvoiceResult = $qboLibrary->processInvoiceCreationWorkflow(
            $customerConfigForQBO, 
            $qboInvoiceLineItems, 
            $invoiceMetaDataForQBO, 
            null, // vendorConfig is null, which is correct
            $quote_no_ref_from_db,  // sourceIdentifier
            $accountName_inv
        ); // No vendor needed here

        // Update sync status for customer and invoice
        $output_data['customer_sync_status'] = ['status' => 'Success', 'qbo_customer_id' => $qboCustomerInvoiceResult['customer']->Id ?? null, 'message' => "Customer '{$accountName_inv}' processed."];
        $invoiceId = $qboCustomerInvoiceResult['invoice']->Id ?? null;
        $output_data['invoice_sync_status'] = ['status' => 'Success', 'qbo_invoice_id' => $invoiceId, 'doc_number' => $qboCustomerInvoiceResult['invoice']->DocNumber ?? null, 'message' => "Invoice created."];

        // Generate QBO link for the created invoice
        $isSandbox = (isset($current_config['baseUrl']) && strcasecmp($current_config['baseUrl'], 'Development') == 0);
        $invoiceLinkBase = $isSandbox ? "https://app.sandbox.qbo.intuit.com/app/invoice?txnId=" : "https://app.qbo.intuit.com/app/invoice?txnId=";
        if($invoiceId) $output_data['invoice_sync_status']['qbo_invoice_link'] = $invoiceLinkBase . $invoiceId;

    } catch (IdsException $e) { // QBO API specific errors
        $errMsg = "QBO API Error (Customer/Invoice): " . $e->getMessage();
        $responseBody = method_exists($e, 'getResponseBody') ? $e->getResponseBody() : 'N/A';
        $httpStatus = method_exists($e, 'getHttpStatusCode') ? $e->getHttpStatusCode() : 'N/A';
        $output_data['debug_notes'][] = $errMsg . " | HTTP: {$httpStatus} | Resp: {$responseBody}";
        $output_data['qbo_operations_log'][] = $errMsg . " | HTTP: {$httpStatus} | Resp: {$responseBody}";
        $output_data['invoice_sync_status'] = ['status' => 'Error (QBO API)', 'message' => $e->getMessage(), 'details' => $responseBody];
    } catch (Exception $e) { // General PHP errors
        $errMsg = "PHP Error (Customer/Invoice): " . $e->getMessage();
        $output_data['debug_notes'][] = $errMsg; $output_data['qbo_operations_log'][] = $errMsg;
        $output_data['invoice_sync_status'] = ['status' => 'Error (PHP)', 'message' => $e->getMessage()];
    }
    $output_data['qbo_operations_log'][] = "--- Finished Customer and Invoice Processing ---";
    // =========================================================================
    // --- End: Customer and Invoice Processing ---
    // =========================================================================


    // =========================================================================
    // --- Start: Supplier and Bill Processing ---
    // =========================================================================
    $output_data['qbo_operations_log'][] = "--- Starting Supplier and Bill Processing ---";
    try {
        $bill_processing_quote_country = $output_data['quote_country']; // Use already fetched quote country

        // Get/create QBO Term for Bills
        $qboTermRefIdForBill = null;
        $termNameBill = $current_config['QBO_DEFAULT_TERM_NAME_BILL'];
        $defaultTermDataBill = ['Name' => $termNameBill, 'Type' => 'STANDARD', 'DueDays' => 0];
        $output_data['qbo_operations_log'][] = "Getting/Creating QBO Term for Bills: '{$termNameBill}'.";
        $termBillObject = $qboLibrary->getOrCreateTermByName($termNameBill, $defaultTermDataBill);
        if ($termBillObject && !empty($termBillObject->Id)) {
            $qboTermRefIdForBill = $termBillObject->Id;
            $output_data['qbo_operations_log'][] = "QBO Term '{$termBillObject->Name}' (ID: {$qboTermRefIdForBill}) processed for bills.";
        } else { $output_data['debug_notes'][] = "Failed to get/create QBO Term '{$termNameBill}' for bills.";}

        // Find QBO TaxCode for Bill lines
        $qboTaxCodeRefIdForBill = null;
        $taxCodeNameBill = $current_config['TARGET_GST_FREE_TAX_CODE_NAME_BILL'];
        $output_data['qbo_operations_log'][] = "Finding QBO TaxCode for Bills: '{$taxCodeNameBill}'.";
        $taxCodeBillObject = $qboLibrary->findTaxCodeByName($taxCodeNameBill);
        if ($taxCodeBillObject && !empty($taxCodeBillObject->Id)) {
            $qboTaxCodeRefIdForBill = $taxCodeBillObject->Id;
            $output_data['qbo_operations_log'][] = "QBO TaxCode '{$taxCodeBillObject->Name}' (ID: {$qboTaxCodeRefIdForBill}) found for bill lines.";
        } else { $output_data['debug_notes'][] = "QBO TaxCode '{$taxCodeNameBill}' NOT FOUND for bill lines.";}

        // Fetch room/pax configuration for bill cost calculation
        $single_rooms_bill = 0; $double_rooms_bill = 0; $triple_rooms_bill = 0; $child_without_bed_bill = 0;
        $created_at_query_filter_for_pax_bill = "";
        $invoice_initial_query_for_pax_bill = "SELECT MAX(created_at) AS created_at FROM vtiger_invoice WHERE quoteid = '" . mysqli_real_escape_string($conn, $quoteid) . "' AND type = 'initial'"; // Use 'initial' invoice for cost basis
        $result_invoice_initial_for_pax_bill = mysqli_query($conn, $invoice_initial_query_for_pax_bill);
        if ($result_invoice_initial_for_pax_bill && $invoice_initial_for_pax_row_bill = mysqli_fetch_assoc($result_invoice_initial_for_pax_bill)) {
            if (!empty($invoice_initial_for_pax_row_bill['created_at'])) {
                $created_at_query_filter_for_pax_bill = " AND created_at < '" . mysqli_real_escape_string($conn, $invoice_initial_for_pax_row_bill['created_at']) . "' ";
            }
        }
        if($result_invoice_initial_for_pax_bill) mysqli_free_result($result_invoice_initial_for_pax_bill);
        $sql_pax_no_bill = "SELECT meta_key, meta_value FROM vtiger_itinerary WHERE quoteid = '" . mysqli_real_escape_string($conn, $quoteid) . "' AND meta_key IN ('single_rooms', 'double_rooms', 'triple_rooms', 'child_without_bed') AND (meta_key, created_at) IN (SELECT meta_key, MAX(created_at) FROM vtiger_itinerary WHERE quoteid = '" . mysqli_real_escape_string($conn, $quoteid) . "' $created_at_query_filter_for_pax_bill AND meta_key IN ('single_rooms', 'double_rooms', 'triple_rooms', 'child_without_bed') GROUP BY meta_key);";
        $output_data['sql_queries_executed'][] = $sql_pax_no_bill;
        $result_pax_no_bill = mysqli_query($conn, $sql_pax_no_bill);
        if ($result_pax_no_bill) { while ($row_pax_no_bill = mysqli_fetch_assoc($result_pax_no_bill)) { switch($row_pax_no_bill['meta_key']) { case 'single_rooms': $single_rooms_bill = (int)($row_pax_no_bill['meta_value'] ?? 0); break; case 'double_rooms': $double_rooms_bill = (int)($row_pax_no_bill['meta_value'] ?? 0); break; case 'triple_rooms': $triple_rooms_bill = (int)($row_pax_no_bill['meta_value'] ?? 0); break; case 'child_without_bed': $child_without_bed_bill = (int)($row_pax_no_bill['meta_value'] ?? 0); break;}}}
        if($result_pax_no_bill) mysqli_free_result($result_pax_no_bill);

        // Fetch all supplier product lines from Vtiger for this quote
        $sql_supplier_all_products = "SELECT DISTINCT vq.adults, vq.children, vq.infants, vv.vendorname, vv.vendorid, vi.productid AS lineitem_productid, vi.day, vi.cf_928, vi.sequence_no, vi.checkin, vi.checkout, vp.productname, vps.sale_price AS sale_price_adult, vps.sale_price_child, vps.sale_price_infant, vps.sale_price_single, vps.sale_price_double, vps.sale_price_triple, vps.sale_price_child_with_bed, vps.sale_price_child_no_bed, vps.subquoteid AS vps_subquoteid, vi.quantity AS vtiger_line_item_quantity FROM (SELECT * FROM vtiger_quotes WHERE quoteid='" . mysqli_real_escape_string($conn, $quoteid) . "') vq LEFT JOIN vtiger_inventoryproductrel vi ON vq.quoteid = vi.id LEFT JOIN ( SELECT productid, productname FROM vtiger_products_custom UNION ALL SELECT productid, productName AS productname FROM tdu_products ) vp ON vi.productid COLLATE utf8mb4_unicode_ci = vp.productid LEFT JOIN ( SELECT vendorid, vendorname FROM vtiger_vendor_custom UNION ALL SELECT vendorid, vendorName AS vendorname FROM tdu_vendors ) vv ON vi.vendorid COLLATE utf8mb4_unicode_ci = vv.vendorid LEFT JOIN vtiger_products_saleprice vps ON vq.quoteid = vps.quoteid AND vi.sequence_no = vps.sequence_no AND vps.subquoteid >= 1 WHERE vv.vendorname IS NOT NULL AND vi.productid IS NOT NULL ORDER BY vv.vendorname ASC, vi.sequence_no ASC;";
        $output_data['sql_queries_executed'][] = $sql_supplier_all_products;
        $result_supplier_all_products = mysqli_query($conn, $sql_supplier_all_products);
        $vendors_temp_data = []; // Holds structured product data per vendor
        $vendor_calculated_line_totals = []; // Holds total cost per vendor from product lines

// --- START: FINAL SUPPLIER COST LOOKUP (v7 - User-Guided Logic Implemented Correctly) ---

// Step 1: PRIMARY ATTEMPT
// Use the query with `subquoteid >= 1`. This is proven to work for correctly configured group quotes like TDU25677G.
$sql_supplier_all_products = "SELECT DISTINCT vq.adults, vq.children, vq.infants, vv.vendorname, vv.vendorid, vi.productid AS lineitem_productid, vi.day, vi.cf_928, vi.sequence_no, vi.checkin, vi.checkout, vp.productname, vps.sale_price AS sale_price_adult, vps.sale_price_child, vps.sale_price_infant, vps.sale_price_single, vps.sale_price_double, vps.sale_price_triple, vps.sale_price_child_with_bed, vps.sale_price_child_no_bed, vps.subquoteid AS vps_subquoteid, vi.quantity AS vtiger_line_item_quantity FROM (SELECT * FROM vtiger_quotes WHERE quoteid='" . mysqli_real_escape_string($conn, $quoteid) . "') vq LEFT JOIN vtiger_inventoryproductrel vi ON vq.quoteid = vi.id LEFT JOIN ( SELECT productid, productname FROM vtiger_products_custom UNION ALL SELECT productid, productName AS productname FROM tdu_products ) vp ON vi.productid COLLATE utf8mb4_unicode_ci = vp.productid LEFT JOIN ( SELECT vendorid, vendorname FROM vtiger_vendor_custom UNION ALL SELECT vendorid, vendorName AS vendorname FROM tdu_vendors ) vv ON vi.vendorid COLLATE utf8mb4_unicode_ci = vv.vendorid LEFT JOIN vtiger_products_saleprice vps ON vq.quoteid = vps.quoteid AND vi.sequence_no = vps.sequence_no AND vps.subquoteid >= 1 WHERE vv.vendorname IS NOT NULL AND vi.productid IS NOT NULL ORDER BY vv.vendorname ASC, vi.sequence_no ASC;";
$output_data['sql_queries_executed'][] = $sql_supplier_all_products;
$result_supplier_all_products = mysqli_query($conn, $sql_supplier_all_products);
$vendors_temp_data = []; // Holds structured product data per vendor
$vendor_calculated_line_totals = []; // Holds total cost per vendor from product lines
$processed_sequences = []; // Used to de-duplicate items in the loop

if ($result_supplier_all_products) {
    while ($row_supp = mysqli_fetch_assoc($result_supplier_all_products)) {
        // For group quotes, this query can return the same product twice (once for subquoteid=1, once for the group tier).
        // We only want to process the one with the higher subquoteid, as it's the more specific price.
        $sequenceNo = $row_supp['sequence_no'];
        if (isset($processed_sequences[$sequenceNo]) && $processed_sequences[$sequenceNo] > $row_supp['vps_subquoteid']) {
            continue; // Skip if we've already processed a more specific tier for this item
        }
        $processed_sequences[$sequenceNo] = $row_supp['vps_subquoteid'];

        $line_items_pax_split_bill = [];
        $adults_bill_line = (int)($row_supp['adults'] ?? 0); $children_bill_line = (int)($row_supp['children'] ?? 0); $infants_bill_line = (int)($row_supp['infants'] ?? 0);
        $item_type_bill = $row_supp['cf_928'];
        if ($item_type_bill == 'Hotel') {
            $total_nights_bill = 1; if (!empty($row_supp['checkin']) && !empty($row_supp['checkout'])) { try { $cin = new DateTime($row_supp['checkin']); $cout = new DateTime($row_supp['checkout']); $total_nights_bill = max(1, $cin->diff($cout)->days); } catch (Exception $e) {}}
            $cost = $single_rooms_bill * (float)($row_supp['sale_price_single'] ?? 0) * $total_nights_bill + $double_rooms_bill * (float)($row_supp['sale_price_double'] ?? 0) * $total_nights_bill + $triple_rooms_bill * (float)($row_supp['sale_price_triple'] ?? 0) * $total_nights_bill + $child_without_bed_bill * (float)($row_supp['sale_price_child_no_bed'] ?? 0) * $total_nights_bill;
            if ($cost > 0) $line_items_pax_split_bill[] = ['pax_type_description' => 'Hotel Room(s)', 'quantity' => 1, 'unit_price' => $cost, 'line_total' => $cost];
        } elseif ($item_type_bill == 'Transfers' || $item_type_bill == 'Guide') {
            $cost = (float)($row_supp['sale_price_adult'] ?? 0);
            $line_qty_bill = (int)($row_supp['vtiger_line_item_quantity'] ?? 1); $line_qty_bill = $line_qty_bill > 0 ? $line_qty_bill : 1;
            if ($cost > 0) $line_items_pax_split_bill[] = ['pax_type_description' => $item_type_bill, 'quantity' => $line_qty_bill, 'unit_price' => ($line_qty_bill > 0 ? $cost / $line_qty_bill : $cost), 'line_total' => $cost];
        } else {
            if ($adults_bill_line > 0 && (float)($row_supp['sale_price_adult'] ?? 0) > 0) $line_items_pax_split_bill[] = ['pax_type_description' => 'Adult(s)', 'quantity' => $adults_bill_line, 'unit_price' => (float)($row_supp['sale_price_adult'] ?? 0), 'line_total' => $adults_bill_line * (float)($row_supp['sale_price_adult'] ?? 0)];
            if ($children_bill_line > 0 && (float)($row_supp['sale_price_child'] ?? 0) > 0) $line_items_pax_split_bill[] = ['pax_type_description' => 'Child(ren)', 'quantity' => $children_bill_line, 'unit_price' => (float)($row_supp['sale_price_child'] ?? 0), 'line_total' => $children_bill_line * (float)($row_supp['sale_price_child'] ?? 0)];
            if ($infants_bill_line > 0 && (float)($row_supp['sale_price_infant'] ?? 0) > 0) $line_items_pax_split_bill[] = ['pax_type_description' => 'Infant(s)', 'quantity' => $infants_bill_line, 'unit_price' => (float)($row_supp['sale_price_infant'] ?? 0), 'line_total' => $infants_bill_line * (float)($row_supp['sale_price_infant'] ?? 0)];
        }
        if (!empty($line_items_pax_split_bill)) {
            $row_to_store_bill = $row_supp; $row_to_store_bill['pax_split_lines'] = $line_items_pax_split_bill;
            $vendors_temp_data[$row_supp['vendorname']]['details']['vtiger_vendor_id'] = $row_supp['vendorid'];
            $vendors_temp_data[$row_supp['vendorname']]['details']['vtiger_vendor_name'] = $row_supp['vendorname'];
            $vendors_temp_data[$row_supp['vendorname']]['products'][] = $row_to_store_bill;
        }
    }
    // Final calculation of totals after de-duplication
    foreach ($vendors_temp_data as $vendorName => &$data) {
        $total = 0;
        if(isset($data['products'])) {
            foreach ($data['products'] as $product) {
                foreach ($product['pax_split_lines'] as $line) {
                    $total += $line['line_total'];
                }
            }
        }
        $vendor_calculated_line_totals[$vendorName] = $total;
    }
    mysqli_free_result($result_supplier_all_products);
}

// Step 2: If the primary attempt failed to find any data, run the fallback.
if (empty($vendors_temp_data)) {
    $output_data['debug_notes'][] = "Primary cost lookup failed. Running fallback for mis-categorized data (like TDU31907).";

    // This is the FALLBACK query, with NO subquoteid constraint at all.
    $sql_supplier_fallback = "SELECT DISTINCT vq.adults, vq.children, vq.infants, vv.vendorname, vv.vendorid, vi.productid AS lineitem_productid, vi.day, vi.cf_928, vi.sequence_no, vi.checkin, vi.checkout, vp.productname, vps.sale_price AS sale_price_adult, vps.sale_price_child, vps.sale_price_infant, vps.sale_price_single, vps.sale_price_double, vps.sale_price_triple, vps.sale_price_child_with_bed, vps.sale_price_child_no_bed, vps.subquoteid AS vps_subquoteid, vi.quantity AS vtiger_line_item_quantity FROM (SELECT * FROM vtiger_quotes WHERE quoteid='" . mysqli_real_escape_string($conn, $quoteid) . "') vq LEFT JOIN vtiger_inventoryproductrel vi ON vq.quoteid = vi.id LEFT JOIN ( SELECT productid, productname FROM vtiger_products_custom UNION ALL SELECT productid, productName AS productname FROM tdu_products ) vp ON vi.productid COLLATE utf8mb4_unicode_ci = vp.productid LEFT JOIN ( SELECT vendorid, vendorname FROM vtiger_vendor_custom UNION ALL SELECT vendorid, vendorName AS vendorname FROM tdu_vendors ) vv ON vi.vendorid COLLATE utf8mb4_unicode_ci = vv.vendorid LEFT JOIN vtiger_products_saleprice vps ON vq.quoteid = vps.quoteid AND vi.sequence_no = vps.sequence_no WHERE vv.vendorname IS NOT NULL AND vi.productid IS NOT NULL ORDER BY vv.vendorname ASC, vi.sequence_no ASC;";
    $output_data['sql_queries_executed'][] = $sql_supplier_fallback;
    $result_supplier_fallback = mysqli_query($conn, $sql_supplier_fallback);

    // Repopulate the data using the fallback results.
    // We can reuse the main processing loop by wrapping it in a function.
    // For simplicity here, we'll just repeat the logic.
    if ($result_supplier_fallback) {
        while ($row_supp = mysqli_fetch_assoc($result_supplier_fallback)) {
            $line_items_pax_split_bill = [];
            $adults_bill_line = (int)($row_supp['adults'] ?? 0); $children_bill_line = (int)($row_supp['children'] ?? 0); $infants_bill_line = (int)($row_supp['infants'] ?? 0);
            $item_type_bill = $row_supp['cf_928'];
            if ($item_type_bill == 'Hotel') {
                $total_nights_bill = 1; if (!empty($row_supp['checkin']) && !empty($row_supp['checkout'])) { try { $cin = new DateTime($row_supp['checkin']); $cout = new DateTime($row_supp['checkout']); $total_nights_bill = max(1, $cin->diff($cout)->days); } catch (Exception $e) {}}
                $cost = $single_rooms_bill * (float)($row_supp['sale_price_single'] ?? 0) * $total_nights_bill + $double_rooms_bill * (float)($row_supp['sale_price_double'] ?? 0) * $total_nights_bill + $triple_rooms_bill * (float)($row_supp['sale_price_triple'] ?? 0) * $total_nights_bill + $child_without_bed_bill * (float)($row_supp['sale_price_child_no_bed'] ?? 0) * $total_nights_bill;
                if ($cost > 0) $line_items_pax_split_bill[] = ['pax_type_description' => 'Hotel Room(s)', 'quantity' => 1, 'unit_price' => $cost, 'line_total' => $cost];
            } elseif ($item_type_bill == 'Transfers' || $item_type_bill == 'Guide') {
                $cost = (float)($row_supp['sale_price_adult'] ?? 0);
                $line_qty_bill = (int)($row_supp['vtiger_line_item_quantity'] ?? 1); $line_qty_bill = $line_qty_bill > 0 ? $line_qty_bill : 1;
                if ($cost > 0) $line_items_pax_split_bill[] = ['pax_type_description' => $item_type_bill, 'quantity' => $line_qty_bill, 'unit_price' => ($line_qty_bill > 0 ? $cost / $line_qty_bill : $cost), 'line_total' => $cost];
            } else {
                if ($adults_bill_line > 0 && (float)($row_supp['sale_price_adult'] ?? 0) > 0) $line_items_pax_split_bill[] = ['pax_type_description' => 'Adult(s)', 'quantity' => $adults_bill_line, 'unit_price' => (float)($row_supp['sale_price_adult'] ?? 0), 'line_total' => $adults_bill_line * (float)($row_supp['sale_price_adult'] ?? 0)];
                if ($children_bill_line > 0 && (float)($row_supp['sale_price_child'] ?? 0) > 0) $line_items_pax_split_bill[] = ['pax_type_description' => 'Child(ren)', 'quantity' => $children_bill_line, 'unit_price' => (float)($row_supp['sale_price_child'] ?? 0), 'line_total' => $children_bill_line * (float)($row_supp['sale_price_child'] ?? 0)];
                if ($infants_bill_line > 0 && (float)($row_supp['sale_price_infant'] ?? 0) > 0) $line_items_pax_split_bill[] = ['pax_type_description' => 'Infant(s)', 'quantity' => $infants_bill_line, 'unit_price' => (float)($row_supp['sale_price_infant'] ?? 0), 'line_total' => $infants_bill_line * (float)($row_supp['sale_price_infant'] ?? 0)];
            }
            if (!empty($line_items_pax_split_bill)) {
                $row_to_store_bill = $row_supp; $row_to_store_bill['pax_split_lines'] = $line_items_pax_split_bill;
                $vendors_temp_data[$row_supp['vendorname']]['details']['vtiger_vendor_id'] = $row_supp['vendorid'];
                $vendors_temp_data[$row_supp['vendorname']]['details']['vtiger_vendor_name'] = $row_supp['vendorname'];
                $vendors_temp_data[$row_supp['vendorname']]['products'][] = $row_to_store_bill;
                foreach ($line_items_pax_split_bill as $split_line_bill) { $vendor_calculated_line_totals[$row_supp['vendorname']] = ($vendor_calculated_line_totals[$row_supp['vendorname']] ?? 0) + $split_line_bill['line_total'];}
            }
        }
        mysqli_free_result($result_supplier_fallback);
    }
}
// --- END: FINAL SUPPLIER COST LOOKUP (v7) ---

        // Fetch summary of payments made to vendors for this quote
        $all_vendor_payments_summary = [];
        $sql_all_payments = "SELECT vendorid, SUM(total_amount) AS total_payable_summary, SUM(paid_amount) AS total_paid_summary, MAX(payment_deadline) AS latest_deadline, MAX(created_at) AS latest_created_at, GROUP_CONCAT(DISTINCT remark SEPARATOR '; ') AS all_remarks, GROUP_CONCAT(DISTINCT invoice_path SEPARATOR ',') AS all_invoice_paths FROM vtiger_supplier_payment sp WHERE sp.quoteid = '" . mysqli_real_escape_string($conn, $quoteid) . "' GROUP BY vendorid;";
        $output_data['sql_queries_executed'][] = $sql_all_payments;
        $result_all_payments = mysqli_query($conn, $sql_all_payments);
        if($result_all_payments){ while($payment_row = mysqli_fetch_assoc($result_all_payments)){ $all_vendor_payments_summary[$payment_row['vendorid']] = $payment_row; }}
        if($result_all_payments) mysqli_free_result($result_all_payments);

        // Get/create a generic QBO Item for summarized bill lines (if no specific product lines)
        $qboGenericPurchaseItem = null;
        $genericItemName = $current_config['GENERIC_SUPPLIER_ITEM_NAME'];
        $genericItemData = ['Name' => $genericItemName, 'Type' => 'Service', 'IncomeAccountRef' => ['value' => $current_config['QBO_DEFAULT_INCOME_ACCOUNT_ID_FOR_SERVICES']], 'ExpenseAccountRef' => ['value' => $current_config['QBO_DEFAULT_EXPENSE_ACCOUNT_ID_FOR_PURCHASES']], 'PurchaseCost' => 0];
        $output_data['qbo_operations_log'][] = "Getting/Creating generic QBO Item for bill summary: {$genericItemName}";
        $qboGenericPurchaseItem = $qboLibrary->getOrCreateItemByName($genericItemName, $genericItemData, $qboCostOfSalesAccountId);
        if ($qboGenericPurchaseItem && !empty($qboGenericPurchaseItem->Id)) { $output_data['qbo_operations_log'][] = "Generic QBO Item '{$qboGenericPurchaseItem->Name}' (ID: {$qboGenericPurchaseItem->Id}) processed for bill summary.";
        } else { $output_data['debug_notes'][] = "Warning: Failed to get/create generic QBO Item '{$genericItemName}' for bill summary."; }

        // Iterate through each vendor and process/create their bill
        foreach($vendors_temp_data as $vendor_name_key => $vendor_detail_lines) {
            $vtiger_vid = $vendor_detail_lines['details']['vtiger_vendor_id'];
            $original_vtiger_vendor_name = $vendor_detail_lines['details']['vtiger_vendor_name'];
            $cleaned_vendor_name = trim($original_vtiger_vendor_name);
            $cleaned_vendor_name = preg_replace('/\s+/', ' ', $cleaned_vendor_name); // Normalize whitespace
            
            $payment_summary = $all_vendor_payments_summary[$vtiger_vid] ?? [];
            $vendor_bill_sync_status = [
                'vtiger_vendor_id' => $vtiger_vid, 'original_vtiger_vendor_name' => $original_vtiger_vendor_name,
                'qbo_vendor_id' => null, 'qbo_bill_id' => null, 'status' => 'Not Processed', 'message' => null, 'qbo_bill_link' => null
            ];

            // Determine if a bill should be created for this vendor
            $createBill = false;
            if (!empty($vendor_detail_lines['products']) && ($vendor_calculated_line_totals[$vendor_name_key] ?? 0) > 0.009) { $createBill = true; }
            elseif (($payment_summary['total_payable_summary'] ?? 0) > 0.009) { $createBill = true; if(empty($vendor_detail_lines['products'])) $output_data['debug_notes'][] = "Vendor '{$original_vtiger_vendor_name}' has amount in payment_summary but no product lines. Bill will be based on summary.";}
            
            if (!$createBill) { 
                $vendor_bill_sync_status['status'] = 'Skipped - No costs/payment summary amount.'; 
                $output_data['vendors_sync_status'][] = $vendor_bill_sync_status; 
                continue; 
            }

            try {
                if (empty($cleaned_vendor_name)) $cleaned_vendor_name = "Unknown Vtiger Vendor {$vtiger_vid}";
                $vendor_bill_sync_status['cleaned_vendor_name_used_for_qbo'] = $cleaned_vendor_name;

                // Determine bill currency
                $current_bill_currency = $current_config['DEFAULT_QBO_HOME_CURRENCY'];
                if (isset($output_data['quote_country']) && strcasecmp($output_data['quote_country'], 'New Zealand') == 0) {
                    $current_bill_currency = 'NZD';
                }

                // Prepare vendor data for QBO
                $vendorQBOCreateData = [ "DisplayName" => $cleaned_vendor_name, "CompanyName" => $cleaned_vendor_name ];
                if ($current_bill_currency !== $current_config['DEFAULT_QBO_HOME_CURRENCY']) {
                   $vendorQBOCreateData['CurrencyRef'] = ["value" => $current_bill_currency];
                }
                
                $output_data['qbo_operations_log'][] = "Getting/Creating QBO Vendor: '{$cleaned_vendor_name}'.";
                $qboVendor = $qboLibrary->getOrCreateVendor($vendorQBOCreateData, 'DisplayName', $cleaned_vendor_name);
                
                if (!$qboVendor || empty($qboVendor->Id)) {
                    throw new Exception("Failed to get/create QBO Vendor '{$cleaned_vendor_name}'.");
                }
                $vendor_bill_sync_status['qbo_vendor_id'] = $qboVendor->Id;
                $output_data['qbo_operations_log'][] = "QBO Vendor '{$qboVendor->DisplayName}' (ID: {$qboVendor->Id}) processed.";

                // --- Build Bill Line Items ---
                $qboBillLineItems = [];
                $total_cost_from_processed_lines_bill = 0.0;

                if (!empty($vendor_detail_lines['products'])) {
                    foreach ($vendor_detail_lines['products'] as $vtiger_product_entry_bill) {
                        $qboItemNameBillRaw = trim($vtiger_product_entry_bill['productname'] ?? "Unknown Product");
                        if (empty($qboItemNameBillRaw)) $qboItemNameBillRaw = "Vtiger Service (PID: {$vtiger_product_entry_bill['lineitem_productid']})";
                        
                        // Replace standard ASCII colon with full-width colon for QBO item name
                        $qboItemNameToUseInQBO = str_replace(':', '：', $qboItemNameBillRaw);

                        if (isset($vtiger_product_entry_bill['pax_split_lines']) && is_array($vtiger_product_entry_bill['pax_split_lines'])) {
                            foreach ($vtiger_product_entry_bill['pax_split_lines'] as $split_line_detail_bill) {
                                $pax_desc_bill = $split_line_detail_bill['pax_type_description'];
                                $qty_bill = (float) $split_line_detail_bill['quantity'];
                                $unit_price_bill = (float) $split_line_detail_bill['unit_price'];
                                $line_amount_bill = (float) $split_line_detail_bill['line_total'];

                                // Skip zero-value lines unless they have a description (e.g. for informational purposes)
                                if ($line_amount_bill <= 0.009 && empty(trim($qboItemNameBillRaw))) continue;
                                if ($line_amount_bill <= 0.009 && !empty(trim($qboItemNameBillRaw))) {
                                    $output_data['debug_notes'][] = "Info: Adding $0.00 or near-zero line for '{$qboItemNameBillRaw} - {$pax_desc_bill}' (Vendor: '{$cleaned_vendor_name}'). Amount: {$line_amount_bill}";
                                }

                                // Prepare data for QBO item (product/service)
                                $defaultQBOItemDataBill = ['Type' => 'Service', 'PurchaseCost' => round($unit_price_bill, 2) ];
                                if ($qboCostOfSalesAccountId) { // If COGS account is configured, use it
                                    $defaultQBOItemDataBill['Type'] = 'NonInventory';
                                    $defaultQBOItemDataBill['IncomeAccountRef'] = ['value' => $qboCostOfSalesAccountId];
                                    $defaultQBOItemDataBill['ExpenseAccountRef'] = ['value' => $qboCostOfSalesAccountId];
                                } else { // Fallback to default service/expense accounts
                                    $defaultQBOItemDataBill['IncomeAccountRef'] = ['value' => $current_config['QBO_DEFAULT_INCOME_ACCOUNT_ID_FOR_SERVICES']];
                                    $defaultQBOItemDataBill['ExpenseAccountRef'] = ['value' => $current_config['QBO_DEFAULT_EXPENSE_ACCOUNT_ID_FOR_PURCHASES']];
                                }
                                
                                // Get/Create QBO Item using the transformed name
                                $qboLineItemBillObject = $qboLibrary->getOrCreateItemByName($qboItemNameToUseInQBO, $defaultQBOItemDataBill, $qboCostOfSalesAccountId);
                
                                if (!$qboLineItemBillObject || empty($qboLineItemBillObject->Id)) { 
                                    $output_data['debug_notes'][] = "Failed get/create QBO Item '{$qboItemNameToUseInQBO}' (original: '{$qboItemNameBillRaw}'). Skipping line part."; 
                                    continue; 
                                }
                                
                                // Add to bill lines
                                $billLineDetail = ["ItemRef" => ["value" => $qboLineItemBillObject->Id, "name" => $qboLineItemBillObject->Name], "Qty" => $qty_bill, "UnitPrice" => round($unit_price_bill, 2)];
                                if ($qboClassRefId) $billLineDetail["ClassRef"] = ["value" => $qboClassRefId, "name" => $quote_no_ref_from_db];
                                if ($qboTaxCodeRefIdForBill) $billLineDetail["TaxCodeRef"] = ["value" => $qboTaxCodeRefIdForBill];
                                $qboBillLineItems[] = ["Amount" => round($line_amount_bill, 2), "DetailType" => "ItemBasedExpenseLineDetail", "ItemBasedExpenseLineDetail" => $billLineDetail, "Description" => $pax_desc_bill];
                                $total_cost_from_processed_lines_bill += $line_amount_bill;
                            }
                        }
                    }
                }
                $vendor_bill_sync_status['qbo_bill_lines_created_count'] = count($qboBillLineItems);

                // If no detailed product lines but payment summary exists, use generic item
                if (empty($qboBillLineItems) && ($payment_summary['total_payable_summary'] ?? 0) > 0.009) {
                    if ($qboGenericPurchaseItem && !empty($qboGenericPurchaseItem->Id)) {
                        $summaryAmountBill = (float)($payment_summary['total_payable_summary'] ?? 0);
                        $summaryDescBill = $payment_summary['all_remarks'] ?? "Summary for Quote ID {$quoteid}";
                        $summaryLineDetailBill = ["ItemRef" => ["value" => $qboGenericPurchaseItem->Id, "name" => $qboGenericPurchaseItem->Name], "Qty" => 1, "UnitPrice" => $summaryAmountBill];
                        if ($qboClassRefId) $summaryLineDetailBill["ClassRef"] = ["value" => $qboClassRefId, "name" => $quote_no_ref_from_db];
                        if ($qboTaxCodeRefIdForBill) $summaryLineDetailBill["TaxCodeRef"] = ["value" => $qboTaxCodeRefIdForBill];
                        $qboBillLineItems[] = ["Amount" => $summaryAmountBill, "DetailType" => "ItemBasedExpenseLineDetail", "ItemBasedExpenseLineDetail" => $summaryLineDetailBill,"Description" => substr($summaryDescBill, 0, 4000)];
                        $total_cost_from_processed_lines_bill = $summaryAmountBill;
                        $vendor_bill_sync_status['qbo_bill_lines_created_count'] = 1;
                    } else { $output_data['debug_notes'][] = "Warning: No detailed lines for '{$cleaned_vendor_name}' and generic summary item not available. Bill might be $0 or skipped."; }
                }
                $vendor_bill_sync_status['calculated_total_from_lines'] = round($total_cost_from_processed_lines_bill, 2);

                // Skip bill creation if no billable lines
                if (empty($qboBillLineItems) || $total_cost_from_processed_lines_bill <= 0.009) {
                     $vendor_bill_sync_status['status'] = 'Skipped';
                     $vendor_bill_sync_status['message'] = 'No billable lines with amount > 0 found.';
                     $output_data['vendors_sync_status'][] = $vendor_bill_sync_status;
                     continue;
                }

                // Prepare bill metadata
                $billTxnDate = $output_data['date_of_travel'] ?? date('Y-m-d');
                $billDocNumber = 'B' . date('YmdHis'); // Base DocNumber
                $vendor_payment_ref_bill = $payment_summary['all_remarks'] ?? null;
                if(!empty($vendor_payment_ref_bill)) $billDocNumber .= "-" . substr(preg_replace("/[^A-Za-z0-9]/", '', $vendor_payment_ref_bill), 0, 5); // Append sanitized remarks
                $billDocNumber = substr($billDocNumber, 0, 21); // Ensure within QBO length limits

                $billDueDate = $payment_summary['latest_deadline'] ?? '';
                if (empty($billDueDate) || $billDueDate == '0000-00-00') $billDueDate = date('Y-m-d', strtotime($billTxnDate . ' +30 days')); // Default due date
                else { try { $billDueDate = (new DateTime($billDueDate))->format('Y-m-d');} catch (Exception $e) { $billDueDate = date('Y-m-d', strtotime($billTxnDate . ' +30 days')); }}

                $billMetaData = [ "DocNumber" => $billDocNumber, "TxnDate" => $billTxnDate, "DueDate" => $billTxnDate, /* Using TxnDate as DueDate for now, adjust if needed: $billDueDate */ "GlobalTaxCalculation" => "TaxIncluded" ];
                if ($qboTermRefIdForBill) $billMetaData["SalesTermRef"] = ["value" => $qboTermRefIdForBill];

                // Handle currency and exchange rate for bill
                if ($current_bill_currency !== $current_config['DEFAULT_QBO_HOME_CURRENCY']) {
                    $billMetaData["CurrencyRef"] = ["value" => $current_bill_currency];
                    $exchangeRateObjectBill = $qboLibrary->getMostRecentExchangeRate($current_bill_currency);
                    if ($exchangeRateObjectBill && property_exists($exchangeRateObjectBill, 'Rate') && (float)$exchangeRateObjectBill->Rate > 0) {
                        $billMetaData["ExchangeRate"] = (float)$exchangeRateObjectBill->Rate;
                        $output_data['qbo_operations_log'][] = "Bill for {$cleaned_vendor_name}: Applied exchange rate ".(float)$exchangeRateObjectBill->Rate." for {$current_bill_currency}.";
                    } else { $output_data['debug_notes'][] = "Bill for {$cleaned_vendor_name}: Could not get valid exchange rate for {$current_bill_currency}. QBO will attempt to auto-apply.";}
                }
                if (!empty($payment_summary['all_invoice_paths'])) $billMetaData["PrivateNote"] = "Vtiger Inv Path(s): " . $payment_summary['all_invoice_paths'];
                
                $output_data['qbo_operations_log'][] = "Attempting create QBO Bill for Vendor ID {$qboVendor->Id}, DocNo: {$billMetaData['DocNumber']}.";
                $output_data['qbo_operations_log'][] = "Bill MetaData Payload for createBill: " . json_encode($billMetaData);
                $output_data['qbo_operations_log'][] = "Bill Line Items Payload for createBill: " . json_encode($qboBillLineItems);
            
                // Create the bill in QBO (single call)
                $qboBill = $qboLibrary->createBill(
                    $qboVendor->Id, 
                    $qboBillLineItems, 
                    $billMetaData, 
                    $quote_no_ref_from_db, // This is the sourceIdentifier
                    $qboVendor->DisplayName // <-- ADD THIS: Pass the vendor name for logging
                );

                if ($qboBill && !empty($qboBill->Id)) {
                    $vendor_bill_sync_status['qbo_bill_id'] = $qboBill->Id;
                    $vendor_bill_sync_status['status'] = 'Success';
                    $vendor_bill_sync_status['message'] = "Bill DocNo {$qboBill->DocNumber} created.";
                    $vendor_bill_sync_status['qbo_bill_total_amount'] = (float)$qboBill->TotalAmt;
                    $isSandboxBill = (isset($current_config['baseUrl']) && strcasecmp($current_config['baseUrl'], 'Development') == 0);
                    $billLinkBase = $isSandboxBill ? "https://app.sandbox.qbo.intuit.com/app/bill?txnId=" : "https://app.qbo.intuit.com/app/bill?txnId=";
                    $vendor_bill_sync_status['qbo_bill_link'] = $billLinkBase . $qboBill->Id;
                } else { throw new Exception("QBO Bill creation failed - no Bill object or ID returned."); }

            } catch (IdsException $e) { // QBO API specific errors for this vendor's bill
                $responseBody = 'N/A';
                $httpStatusCode = method_exists($e, 'getHttpStatusCode') ? $e->getHttpStatusCode() : 'N/A';
                if (method_exists($e, 'getResponseBody') && $e->getResponseBody()) {
                    $responseBody = $e->getResponseBody();
                } elseif ($e->getPrevious() && method_exists($e->getPrevious(), 'hasResponse') && $e->getPrevious()->hasResponse()) {
                    $guzzleResponse = $e->getPrevious()->getResponse();
                    if ($guzzleResponse) { $responseBody = (string) $guzzleResponse->getBody(); }
                }
                $vendor_bill_sync_status['status'] = 'Error (QBO API)';
                $vendor_bill_sync_status['message'] = "Msg: " . $e->getMessage() . " | HTTP: " . $httpStatusCode . " | Resp: " . $responseBody;
            } catch (Exception $e) { // General PHP errors for this vendor's bill
             $vendor_bill_sync_status['status'] = 'Error (PHP)';
             $vendor_bill_sync_status['message'] = $e->getMessage();
            }
            $output_data['vendors_sync_status'][] = $vendor_bill_sync_status; // Add this vendor's bill sync status to the overall output
        } // End foreach $vendors_temp_data

    } catch (IdsException $e) { // QBO API errors in the main supplier/bill block
        $errMsg = "QBO API Error (Supplier/Bill Main Block): " . $e->getMessage();
        $output_data['debug_notes'][] = $errMsg; $output_data['qbo_operations_log'][] = $errMsg;
    } catch (Exception $e) { // General PHP errors in the main supplier/bill block
        $errMsg = "PHP Error (Supplier/Bill Main Block): " . $e->getMessage();
        $output_data['debug_notes'][] = $errMsg; $output_data['qbo_operations_log'][] = $errMsg;
    }
    $output_data['qbo_operations_log'][] = "--- Finished Supplier and Bill Processing ---";
    // =========================================================================
    // --- End: Supplier and Bill Processing ---
    // =========================================================================

    return $output_data;
}
?>