<?php
// compare_cost_logic.php
// A test script to compare supplier cost calculation between the Dashboard UI and the QBO Sync Processor.
// Version 2: Corrects the array key access bug in the QBO Sync logic path.

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// --- Essential Includes ---
require_once __DIR__ . '/../dbconn.php';

// =========================================================================
// --- SCRIPT SETUP AND INPUT VALIDATION ---
// =========================================================================

if (!isset($argv[1]) || empty(trim($argv[1]))) {
    echo "Usage: php " . basename(__FILE__) . " <quote_no>\n";
    die();
}

$quote_no_to_test = trim($argv[1]);

global $conn;
if (!$conn) {
    die("CRITICAL: Database connection failed. Check dbconn.php.\n");
}

$sql_get_qid = "SELECT quoteid FROM vtiger_quotes WHERE quote_no = '" . mysqli_real_escape_string($conn, $quote_no_to_test) . "' LIMIT 1;";
$result_qid = mysqli_query($conn, $sql_get_qid);
if (!$result_qid || mysqli_num_rows($result_qid) == 0) {
    die("ERROR: Could not find a quote with Quote Number: {$quote_no_to_test}\n");
}
$quoteid = mysqli_fetch_assoc($result_qid)['quoteid'];

echo "--- Starting Comparison for Quote No: {$quote_no_to_test} (Quote ID: {$quoteid}) ---\n\n";

// =========================================================================
// --- COMMON DATA FETCHING ---
// =========================================================================

$sql_pax_base = "SELECT adults, children, infants FROM vtiger_quotes WHERE quoteid = '$quoteid' LIMIT 1;";
$result_pax_base = mysqli_query($conn, $sql_pax_base);
$pax_data = mysqli_fetch_assoc($result_pax_base);
$adults_no = (int)($pax_data['adults'] ?? 0);
$children_no = (int)($pax_data['children'] ?? 0);
$infants_no = (int)($pax_data['infants'] ?? 0);

$invoice_initial_query = "SELECT MAX(created_at) AS created_at FROM vtiger_invoice WHERE quoteid = '$quoteid' AND type = 'initial'";
$invoice_initial_result = mysqli_query($conn, $invoice_initial_query);
$created_at_query = !empty(mysqli_fetch_assoc($invoice_initial_result)['created_at']) ? " AND created_at < '" . mysqli_fetch_assoc(mysqli_query($conn, $invoice_initial_query))['created_at'] . "' " : "";

$sql_pax_no = "SELECT meta_key, meta_value FROM vtiger_itinerary WHERE quoteid = '$quoteid' AND meta_key IN ('single_rooms', 'double_rooms', 'triple_rooms', 'child_without_bed') AND (meta_key, created_at) IN (SELECT meta_key, MAX(created_at) FROM vtiger_itinerary WHERE quoteid = '$quoteid' {$created_at_query} AND meta_key IN ('single_rooms', 'double_rooms', 'triple_rooms', 'child_without_bed') GROUP BY meta_key);";
$result_pax_no = mysqli_query($conn, $sql_pax_no);
$single_rooms = $double_rooms = $triple_rooms = $child_without_bed = 0;
while($row = mysqli_fetch_assoc($result_pax_no)) {
    if($row['meta_key'] == 'single_rooms') $single_rooms = (int)($row['meta_value'] ?? 0);
    if($row['meta_key'] == 'double_rooms') $double_rooms = (int)($row['meta_value'] ?? 0);
    if($row['meta_key'] == 'triple_rooms') $triple_rooms = (int)($row['meta_value'] ?? 0);
    if($row['meta_key'] == 'child_without_bed') $child_without_bed = (int)($row['meta_value'] ?? 0);
}

echo "Detected Pax/Room Config: Adults({$adults_no}), Children({$children_no}), Infants({$infants_no}), Single({$single_rooms}), Double({$double_rooms}), Triple({$triple_rooms}), ChildNoBed({$child_without_bed})\n";
echo "------------------------------------------------------------------------\n";

// =========================================================================
// --- Logic Path 1: DASHBOARD CALCULATION (from div12) ---
// =========================================================================
echo "Executing Logic Path 1: Dashboard Calculation...\n";

// Simulate the intended logic of the original query by allowing any subquoteid >= 1
$sql_dashboard = "SELECT DISTINCT vv.vendorname, vi.cf_928, vps.*, vq.*, vi.checkin, vi.checkout
                  FROM (SELECT * FROM vtiger_quotes WHERE quoteid='$quoteid') vq
                  LEFT JOIN vtiger_inventoryproductrel vi ON vq.quoteid = vi.id
                  LEFT JOIN (SELECT vendorid, vendorname FROM vtiger_vendor_custom UNION ALL SELECT vendorid, vendorName AS vendorname FROM tdu_vendors) vv ON vi.vendorid=vv.vendorid
                  LEFT JOIN vtiger_products_saleprice vps ON vq.quoteid=vps.quoteid AND vi.sequence_no=vps.sequence_no AND vps.subquoteid >= 1
                  WHERE vv.vendorname IS NOT NULL
                  ORDER BY vv.vendorname ASC, vi.sequence_no ASC;";

$result_dashboard = mysqli_query($conn, $sql_dashboard);
$dashboard_totals = [];

while ($row = mysqli_fetch_assoc($result_dashboard)) {
    $total_sale_price_each = 0;
    if($row['cf_928'] == 'Hotel') {
        $nights = (new DateTime($row['checkin']))->diff(new DateTime($row['checkout']))->days;
        $total_sale_price_each = ($single_rooms * (float)$row['sale_price_single'] + $double_rooms * (float)$row['sale_price_double'] + $triple_rooms * (float)$row['sale_price_triple'] + $child_without_bed * (float)$row['sale_price_child_no_bed']) * $nights;
    } else if ($row['cf_928'] == 'Transfers' || $row['cf_928'] == 'Guide') {
        $total_sale_price_each = (float) $row['sale_price'];
    } else {
        $total_sale_price_each = ($adults_no * (float)$row['sale_price'] + $children_no * (float)$row['sale_price_child'] + $infants_no * (float)$row['sale_price_infant']);
    }
    $dashboard_totals[$row['vendorname']] = ($dashboard_totals[$row['vendorname']] ?? 0) + $total_sale_price_each;
}
echo "Finished. Found " . count($dashboard_totals) . " vendors.\n\n";

// =========================================================================
// --- Logic Path 2: QBO SYNC CALCULATION (from qbo_sync_processor.php) ---
// =========================================================================
echo "Executing Logic Path 2: QBO Sync Calculation...\n";

// This is the original, flawed query from the sync script.
$sql_qbo_sync = "SELECT DISTINCT vv.vendorname, vi.cf_928, vps.*, vq.*, vi.checkin, vi.checkout
                 FROM (SELECT * FROM vtiger_quotes WHERE quoteid='$quoteid') vq
                 LEFT JOIN vtiger_inventoryproductrel vi ON vq.quoteid = vi.id
                 LEFT JOIN (SELECT vendorid, vendorname FROM vtiger_vendor_custom UNION ALL SELECT vendorid, vendorName AS vendorname FROM tdu_vendors) vv ON vi.vendorid=vv.vendorid
                 LEFT JOIN vtiger_products_saleprice vps ON vq.quoteid = vps.quoteid AND vi.sequence_no = vps.sequence_no AND vps.subquoteid <= 1
                 WHERE vv.vendorname IS NOT NULL
                 ORDER BY vv.vendorname ASC, vi.sequence_no ASC;";

$result_qbo_sync = mysqli_query($conn, $sql_qbo_sync);
$qbo_sync_totals = [];

while ($row = mysqli_fetch_assoc($result_qbo_sync)) {
    $total_sale_price_each = 0;
    if($row['cf_928'] == 'Hotel') {
        $nights = (new DateTime($row['checkin']))->diff(new DateTime($row['checkout']))->days;
        $total_sale_price_each = ($single_rooms * (float)$row['sale_price_single'] + $double_rooms * (float)$row['sale_price_double'] + $triple_rooms * (float)$row['sale_price_triple'] + $child_without_bed * (float)$row['sale_price_child_no_bed']) * $nights;
    } 
    // =========================================================================
    // --- THE FIX IS HERE ---
    // The QBO Sync script uses `sale_price_adult` for Transfers/Guides, but the query aliases it as `sale_price`.
    // We must use the correct key from the fetched row data.
    else if ($row['cf_928'] == 'Transfers' || $row['cf_928'] == 'Guide') {
        $total_sale_price_each = (float) $row['sale_price']; // CORRECTED KEY
    } else {
        $total_sale_price_each = ($adults_no * (float)$row['sale_price'] + $children_no * (float)$row['sale_price_child'] + $infants_no * (float)$row['sale_price_infant']); // CORRECTED KEY
    }
    // =========================================================================
    $qbo_sync_totals[$row['vendorname']] = ($qbo_sync_totals[$row['vendorname']] ?? 0) + $total_sale_price_each;
}
echo "Finished. Found " . count($qbo_sync_totals) . " vendors.\n\n";


// =========================================================================
// --- FINAL COMPARISON ---
// =========================================================================
echo "==================== FINAL COMPARISON RESULTS ====================\n";
$all_vendors = array_unique(array_merge(array_keys($dashboard_totals), array_keys($qbo_sync_totals)));
sort($all_vendors);

$mismatch_found = false;

$red = "\033[0;31m"; $green = "\033[0;32m"; $nc = "\033[0m";

printf("%-45s | %-20s | %-20s | %-10s\n", "Vendor Name", "Dashboard Cost", "QBO Sync Cost", "Mismatch?");
echo str_repeat("-", 105) . "\n";

foreach ($all_vendors as $vendor_name) {
    $dashboard_cost = $dashboard_totals[$vendor_name] ?? 0;
    $qbo_sync_cost = $qbo_sync_totals[$vendor_name] ?? 0;
    $is_mismatch = (abs($dashboard_cost - $qbo_sync_cost) > 0.01);

    if ($is_mismatch) {
        $mismatch_found = true;
        printf("%-45s | %-20.2f | %-20.2f | {$red}%-10s{$nc}\n", $vendor_name, $dashboard_cost, $qbo_sync_cost, "YES");
    } else {
        printf("%-45s | %-20.2f | %-20.2f | {$green}%-10s{$nc}\n", $vendor_name, $dashboard_cost, $qbo_sync_cost, "No");
    }
}
echo "==================================================================\n\n";

if ($mismatch_found) {
    echo "{$red}BUG CONFIRMED: One or more vendors have a cost mismatch between the Dashboard and the QBO Sync logic.{$nc}\n";
} else {
    echo "{$green}No mismatches found for this quote. The logic paths produce identical results.{$nc}\n";
}

echo "\nTest Finished.\n";
mysqli_close($conn);

?>