<?php
// prototype_phase1_debug.php
// Goal: Isolate and debug the CDC timestamp format issue.

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// --- Use the bootstrap file for a clean, consistent connection ---
// We will modify the bootstrap logic slightly to enable logging.
require_once __DIR__ . '/../qbo_bootstrap.php';
require_once __DIR__ . '/../config/qbo_config.php';

use QuickBooksOnline\API\DataService\DataService;

// =========================================================================
// --- PHASE 1 DEBUG LOGIC ---
// =========================================================================

echo "--- Starting Prototype Phase 1 (DEBUG MODE) ---\n\n";

global $qboBaseConfig, $qboUtil;

// --- Step 1: Enable SDK Logging for Deep Analysis ---
// To see the raw request, we need to re-initialize the DataService with logging enabled.

// Create a directory for the logs if it doesn't exist
$logDirectory = __DIR__ . '/qbo_sdk_logs';
if (!is_dir($logDirectory)) {
    mkdir($logDirectory, 0755, true);
}

// Get existing tokens from the bootstrap's functions
$currentTokens = loadQBOTokens(); 
if (!$currentTokens) {
    die("CRITICAL: Could not load tokens. Please run the bootstrap process first.\n");
}

$qboConfigForLibrary = $qboBaseConfig;
$qboConfigForLibrary['accessTokenKey'] = $currentTokens['access_token'];
$qboConfigForLibrary['refreshTokenKey'] = $currentTokens['refresh_token'];
$qboConfigForLibrary['logLocation'] = $logDirectory; // Tell the SDK where to write logs
$qboConfigForLibrary['log_enabled'] = true;

try {
    // Re-configure DataService with logging ON
    $dataService = DataService::Configure($qboConfigForLibrary);
    echo "SUCCESS: DataService re-initialized with logging enabled.\n";
    echo "Log files will be written to: " . $logDirectory . "\n\n";
} catch (Exception $e) {
    die("CRITICAL: Failed to re-initialize DataService: " . $e->getMessage() . "\n");
}


// --- Step 2: Test Different Timestamp Formats ---
$entitiesToQuery = ['BillPayment'];

// --- TEST CASE 1: Explicit UTC 'Z' Suffix ---
// This is the most common and least ambiguous format for APIs.
echo "--- Test Case 1: Using UTC 'Z' format ---\n";
try {
    // Get the time for 1 day ago and format it in UTC.
    $utc_tz = new DateTimeZone("UTC");
    $utc_dt = new DateTime("-1 day", $utc_tz);
    $timestamp_utc = $utc_dt->format('Y-m-d\TH:i:s\Z');
    
    echo "Attempting fetch since: " . $timestamp_utc . "\n";
    
    $changedEntities = $dataService->CDC($entitiesToQuery, $timestamp_utc);
    
    echo "SUCCESS: Test Case 1 was successful!\n";
    echo "The correct format appears to be the UTC 'Z' format.\n";
    echo "Raw Response:\n";
    print_r($changedEntities);

} catch (Exception $e) {
    echo "FAILED: Test Case 1 failed. Error: " . $e->getMessage() . "\n";
}

echo "\n------------------------------------------------\n\n";


// --- TEST CASE 2: Explicit Timezone Offset '+' Suffix ---
// This directly tests the '+' character that was missing before.
echo "--- Test Case 2: Using manual '+' offset format ---\n";
try {
    // Get the time for 1 day ago and format it with a specific offset.
    $offset_tz = new DateTimeZone("+10:00");
    $offset_dt = new DateTime("-1 day", $offset_tz);
    $timestamp_offset = $offset_dt->format('Y-m-d\TH:i:sP'); // Use P for "+10:00" format
    
    echo "Attempting fetch since: " . $timestamp_offset . "\n";

    $changedEntities = $dataService->CDC($entitiesToQuery, $timestamp_offset);
    
    echo "SUCCESS: Test Case 2 was successful!\n";
    echo "The explicit offset format works.\n";
    echo "Raw Response:\n";
    print_r($changedEntities);

} catch (Exception $e) {
    echo "FAILED: Test Case 2 failed. Error: " . $e->getMessage() . "\n";
}

echo "\n--- Prototype Phase 1 (DEBUG MODE) Finished ---\n";
echo "Please check the 'qbo_sdk_logs' directory for detailed request/response logs.\n";

?>