<?php
// prototype_phase4_final.php
// FINAL REVISION based on read/write constraints of vtiger_supplier_payment
// Goal: To correctly "upsert" a payment, handling the VARCHAR paid_amount by calculating in PHP.

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// --- Essential Includes ---
require_once __DIR__ . '/../dbconn.php';

// =========================================================================
// --- PHASE 4 (FINAL): PROTOTYPE LOGIC ---
// =========================================================================

echo "--- Starting Prototype Phase 4 (Final Logic): Test DB Upsert ---\n\n";

global $conn;
if (!$conn) {
    die("CRITICAL: Database connection failed. Check dbconn.php.\n");
}
echo "SUCCESS: Database connection established.\n";

// --- Hardcoded Data Packet (Simulating a NEW payment from QBO) ---
// To re-run this test, change the qbo_bill_payment_id to a new unique number (e.g., '1337_test2')
$paymentData = [
    'qbo_bill_payment_id' => '1336_test_run_2', // This MUST be unique for each test run
    'qbo_bill_id'         => '1335',
    'payment_amount'      => 70.00, // Let's test with a partial payment of 50
    'payment_date'        => '2025-06-20',
    'vendor_name'         => 'Yarra Tours Melbourne',
    'quote_no'            => 'TDU31121'
];

echo "\nStep 1: Preparing to process the following data packet:\n";
print_r($paymentData);

mysqli_begin_transaction($conn);

try {
    // --- Step 2: Get internal Vtiger IDs ---
    echo "\nStep 2: Fetching internal Vtiger IDs...\n";
    $sql_get_quote_id = "SELECT quoteid FROM vtiger_quotes WHERE quote_no = ? LIMIT 1";
    $stmt_quote = mysqli_prepare($conn, $sql_get_quote_id);
    mysqli_stmt_bind_param($stmt_quote, "s", $paymentData['quote_no']);
    mysqli_stmt_execute($stmt_quote);
    $vtiger_quote_id = mysqli_fetch_assoc(mysqli_stmt_get_result($stmt_quote))['quoteid'] ?? null;
    mysqli_stmt_close($stmt_quote);

    $sql_get_vendor_id = "(SELECT vendorid FROM tdu_vendors WHERE vendorName = ?) UNION (SELECT vendorid FROM vtiger_vendor WHERE vendorname = ?) LIMIT 1";
    $stmt_vendor = mysqli_prepare($conn, $sql_get_vendor_id);
    mysqli_stmt_bind_param($stmt_vendor, "ss", $paymentData['vendor_name'], $paymentData['vendor_name']);
    mysqli_stmt_execute($stmt_vendor);
    $vtiger_vendor_id = mysqli_fetch_assoc(mysqli_stmt_get_result($stmt_vendor))['vendorid'] ?? null;
    mysqli_stmt_close($stmt_vendor);

    if (!$vtiger_quote_id || !$vtiger_vendor_id) {
        throw new Exception("Could not find internal ID for Quote '{$paymentData['quote_no']}' or Vendor '{$paymentData['vendor_name']}'.");
    }
    echo "SUCCESS: Found Quote ID: {$vtiger_quote_id}, Vendor ID: {$vtiger_vendor_id}\n";

    // --- Step 3: Check Audit Log & Log the new payment ---
    echo "\nStep 3: Checking audit log and inserting new payment record...\n";
    $sql_insert_log = "INSERT IGNORE INTO tdu_qbo_payment_log (qbo_bill_id, qbo_bill_payment_id, vtiger_quote_id, vtiger_vendor_id, payment_date, payment_amount) VALUES (?, ?, ?, ?, ?, ?)";
    $stmt_log = mysqli_prepare($conn, $sql_insert_log);
    mysqli_stmt_bind_param($stmt_log, "ssiisd", $paymentData['qbo_bill_id'], $paymentData['qbo_bill_payment_id'], $vtiger_quote_id, $vtiger_vendor_id, $paymentData['payment_date'], $paymentData['payment_amount']);
    mysqli_stmt_execute($stmt_log);
    
    if (mysqli_stmt_affected_rows($stmt_log) === 0) {
        // This is our safety net. If the payment is already logged, we stop here.
        throw new Exception("This payment (QBO ID: {$paymentData['qbo_bill_payment_id']}) has already been processed and logged. Halting to prevent duplicate update.");
    }
    mysqli_stmt_close($stmt_log);
    echo "SUCCESS: New payment logged in 'tdu_qbo_payment_log'.\n";

    // --- Step 4: Perform the "Upsert" logic ---
    echo "\nStep 4: Performing 'Upsert' on 'vtiger_supplier_payment'...\n";
    
    // Check if a row already exists
    $sql_check = "SELECT paid_amount FROM vtiger_supplier_payment WHERE quoteid = ? AND vendorid = ? LIMIT 1";
    $stmt_check = mysqli_prepare($conn, $sql_check);
    mysqli_stmt_bind_param($stmt_check, "ii", $vtiger_quote_id, $vtiger_vendor_id);
    mysqli_stmt_execute($stmt_check);
    $result_check = mysqli_stmt_get_result($stmt_check);
    $existing_row = mysqli_fetch_assoc($result_check);
    mysqli_stmt_close($stmt_check);
    
    if ($existing_row) {
        // UPDATE PATH
        echo "INFO: Existing row found. Performing UPDATE.\n";
        
        // **Calculate the new total in PHP**
        $current_paid_amount = (float) $existing_row['paid_amount'];
        $new_total_paid = $current_paid_amount + $paymentData['payment_amount'];
        
        echo "Current paid amount: {$current_paid_amount}. New payment: {$paymentData['payment_amount']}. New Total: {$new_total_paid}\n";

        $sql_update = "UPDATE vtiger_supplier_payment SET paid_amount = ? WHERE quoteid = ? AND vendorid = ?";
        $stmt_update = mysqli_prepare($conn, $sql_update);
        mysqli_stmt_bind_param($stmt_update, "sii", $new_total_paid, $vtiger_quote_id, $vtiger_vendor_id);
        mysqli_stmt_execute($stmt_update);
        mysqli_stmt_close($stmt_update);

    } else {
        // INSERT PATH
        echo "INFO: No existing row found. Performing INSERT.\n";
        
        $sql_insert = "INSERT INTO vtiger_supplier_payment (quoteid, vendorid, paid_amount, total_amount) VALUES (?, ?, ?, '0')";
        $stmt_insert = mysqli_prepare($conn, $sql_insert);
        mysqli_stmt_bind_param($stmt_insert, "iid", $vtiger_quote_id, $vtiger_vendor_id, $paymentData['payment_amount']);
        mysqli_stmt_execute($stmt_insert);
        mysqli_stmt_close($stmt_insert);
    }
    echo "SUCCESS: 'vtiger_supplier_payment' table updated.\n";

    mysqli_commit($conn);
    echo "\nDatabase transaction committed successfully.\n";

} catch (Exception $e) {
    mysqli_rollback($conn);
    die("\nERROR: An exception occurred. Transaction rolled back. Message: " . $e->getMessage() . "\n");
}

mysqli_close($conn);

echo "\n--- Prototype Phase 4 (Final Logic) Finished ---\n";
?>