<?php
// cronjob_quickbook_test2.php
// This script fetches quote data from a local database, interacts with QuickBooks Online (QBO)
// to manage OAuth tokens, dynamically get/create a QBO Class based on the quote number,
// find a specific "GST Free" TaxCode, get/create a "Sales Category" Product/Service,
// get/create a default Payment Term, and then creates an invoice in QBO.
// Invoice lines are assigned the determined Class and TaxCode,
// and the Sales Category item is added as a line summarizing the main package.
// The script includes logging for monitoring and debugging.

// Enable error reporting for development/testing
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// I. INCLUDES, CONFIGURATION, AND HELPER FUNCTIONS
// -----------------------------------------------------------------------------
header('Content-Type: application/json'); // Set JSON response type for this script's output

// Adjust paths as necessary for your project structure
require_once __DIR__ . '/../vendor/autoload.php'; // QuickBooks SDK Autoloader
require_once __DIR__ . '/dbconn.php';           // Your database connection script
require_once __DIR__ . '/../quote_pricing.php';    // Your custom quote pricing logic
require_once __DIR__ . '/../qbo_functions.php'; // Your QBOUtilityLibrary

// --- Logging and Token Management ---
$logFile = __DIR__ . '/qbo_invoice_cron.log'; // Path to the log file
$tokenStorageFile = __DIR__ . '/../tokens/qbo_token.json'; // Path to store OAuth tokens. Secure this file and its directory!

/**
 * Logs a message to the specified log file with a timestamp.
 * @param string $message The message to log.
 */
function logMessage($message) {
    global $logFile;
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($logFile, "[$timestamp] " . $message . "\n", FILE_APPEND);
}

/**
 * Loads OAuth tokens (access and refresh) from the JSON storage file.
 * @return array|null The tokens array or null if the file doesn't exist or an error occurs.
 */
function loadTokens() {
    global $tokenStorageFile;
    if (file_exists($tokenStorageFile)) {
        $json = file_get_contents($tokenStorageFile);
        if ($json === false) {
            logMessage("ERROR: Could not read token storage file: " . $tokenStorageFile);
            return null;
        }
        logMessage("Tokens successfully loaded from storage: " . $tokenStorageFile);
        return json_decode($json, true); // Returns null on json_decode error
    } else {
        logMessage("Token storage file not found at: " . $tokenStorageFile);
    }
    return null;
}

/**
 * Saves OAuth tokens (access and refresh) to the JSON storage file.
 * Creates the directory if it doesn't exist.
 * @param string $accessToken The new access token.
 * @param string $refreshToken The new refresh token.
 */
function saveTokens($accessToken, $refreshToken) {
    global $tokenStorageFile;
    $tokens = [
        'access_token' => $accessToken,
        'refresh_token' => $refreshToken,
        'last_updated' => date('Y-m-d H:i:s')
    ];
    $tokenDir = dirname($tokenStorageFile);
    if (!is_dir($tokenDir)) { // Check if token directory exists
        if (!mkdir($tokenDir, 0755, true) && !is_dir($tokenDir)) { // Check again after mkdir
            logMessage("FATAL: Could not create token storage directory: " . $tokenDir);
            exit;
        }
        logMessage("Token storage directory created: " . $tokenDir);
    }
    if (file_put_contents($tokenStorageFile, json_encode($tokens, JSON_PRETTY_PRINT))) {
        logMessage("Tokens successfully saved to storage: " . $tokenStorageFile);
    } else {
        logMessage("FATAL: Could not write tokens to storage file: " . $tokenStorageFile . ". Check permissions.");
        exit;
    }
}

logMessage("--------------------------------------------------");
logMessage("Cron job started for QBO invoice creation.");

$storedTokens = loadTokens();
if (!$storedTokens || !isset($storedTokens['access_token']) || !isset($storedTokens['refresh_token'])) {
    logMessage("FATAL: Could not load valid tokens from storage. Ensure '{$tokenStorageFile}' contains valid tokens or re-authorize the application.");
    exit;
}
$accessToken = $storedTokens['access_token'];
$refreshToken = $storedTokens['refresh_token'];

// --- QBO API Configuration ---
// !!! YOU MUST REPLACE ClientID, ClientSecret, QBORealmID, and baseUrl WITH YOUR ACTUAL QBO APP SETTINGS !!!
$qboConfig = [
    'auth_mode'       => 'oauth2',
    'ClientID'        => "AB3WEwQtfG1Ws43pqxY7Y2Ok3s7QfN97VniZxSssaQhl9PA4JE",
    'ClientSecret'    => "GRjmj4WxpMo5ZkMOjfnEQ5LmzXkAna1xb9YpwEWL",
    'accessTokenKey'  => $accessToken,
    'refreshTokenKey' => $refreshToken,
    'QBORealmID'      => "9341454710400258",
    'baseUrl'         => "Development",
    // 'logLocation'     => __DIR__ . "/qbo_sdk_logs"
];

// --- Item, Tax, Sales Category, and Term Configuration ---
// !!! UPDATE THESE WITH IDs/NAMES RELEVANT TO YOUR QBO COMPANY !!!
define('QBO_DEFAULT_ITEM_ID', '1');     // Fallback QBO Item ID for adjustment lines if no specific mapping found.
define('TARGET_GST_FREE_TAX_CODE_NAME', 'GST free'); // EXAMPLE: 'FRE', 'NON', 'GST Free Sales'. CHECK YOUR QBO.
define('FALLBACK_NON_TAXABLE_TAX_CODE_ID', 'NON'); // EXAMPLE: QBO often has 'NON'. VERIFY.
define('SALES_CATEGORY_ITEM_TYPE', 'Service');
define('DEFAULT_INCOME_ACCOUNT_ID_FOR_SALES_CATEGORY', '84'); // EXAMPLE ID. REPLACE WITH A VALID INCOME ACCOUNT ID FROM YOUR QBO.
define('DEFAULT_QBO_HOME_CURRENCY', 'AUD');
define('QBO_DEFAULT_TERM_NAME_FOR_INVOICE', 'Due on receipt'); // Term for Invoices

$qboItemMapping = [
    "Package Cost Per Adult on Single Occupancy" => "1",
    "Package Cost Per Adult on Double Occupancy" => "2",
    "Package Cost Per Adult on Triple Occupancy" => "3",
    "Package Cost Per Adult No Hotel" => "4",
    "Package Cost Per Child with Bed" => "5",
    "Package Cost Per Child without Bed" => "6",
    "Package Cost Per Child No Hotel" => "7",
    "Package Cost Per Infant" => "8",
];

// II. FETCH QUOTE DATA FROM YOUR DATABASE
// -----------------------------------------------------------------------------
$ajaxResponseData = [
    'quote_id_processed' => null, 'quote_no_reference' => null, 'accountName' => null, 'customer_email' => null,
    'qbo_class_id_used' => null, 'invoice_transaction_date_used' => null, 'tax_code_id_used_for_lines' => null,
    'qbo_term_id_used_for_invoice' => null, // Added for logging
    'sales_category_item_name_determined' => null, 'sales_category_item_id_used' => null,
    'total_package_amount_calculated' => null,
    'error' => null, 'qbo_result' => null, 'warnings' => []
];

$quoteid = isset($_REQUEST['quoteid']) ? mysqli_real_escape_string($conn, $_REQUEST['quoteid']) : '2349421';
$ajaxResponseData['quote_id_processed'] = $quoteid;
logMessage("Processing Quote ID: " . $quoteid);

if (empty($quoteid)) {
    $errorMessage = "Quote ID is required for processing.";
    logMessage("ERROR: " . $errorMessage); $ajaxResponseData['error'] = $errorMessage;
    echo json_encode($ajaxResponseData, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES); exit();
}

// --- Initialize variables ---
$accountName = ''; $address_line1 = ''; $address_city = ''; $address_postal_code = ''; $quote_country = '';
$subject = ''; $currency = 'AUD'; $adults_no = 0; $children_no = 0; $infants_no = 0; $payment_deadline = '';
$quote_no_ref = ''; $invoice_transaction_date = date('Y-m-d'); $qboInvoiceLineItems = [];
$single_rooms = 0; $double_rooms = 0; $triple_rooms = 0; $child_no_bed = 0; $child_with_bed = 0;
$pricing_single_room = 0; $pricing_double_room = 0; $pricing_triple_room = 0;
$pricing_child_with_bed = 0; $pricing_child_no_bed = 0; $pricing_infant = 0;

// --- SQL to fetch quote, customer, and custom date details ---
$sql_quote_info = "SELECT vq.quoteid, vq.subject, vq.adults, vq.children, vq.infants, vq.quote_no, va.organization_name AS accountname, va.address AS full_address, va.country AS bill_country_from_org, va.email AS account_email, vqcf.cf_1182 AS payment_due_date_custom_field, vqcf.cf_1162 AS invoice_date_custom_field, vq.country AS quote_country FROM vtiger_quotes vq LEFT JOIN vtiger_quotescf vqcf ON vq.quoteid = vqcf.quoteid LEFT JOIN tdu_organisation va ON vq.accountid = va.organizationid WHERE vq.quoteid = '$quoteid' LIMIT 1";
$result_quote_info = mysqli_query($conn, $sql_quote_info);
if (!$result_quote_info) {
    $errorMessage = "DB Error (Quote Info): " . mysqli_error($conn); logMessage("ERROR: ".$errorMessage." | SQL: ".$sql_quote_info);
    $ajaxResponseData['error'] = $errorMessage; echo json_encode($ajaxResponseData, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES); exit();
}
if (mysqli_num_rows($result_quote_info) > 0) {
    $row_quote = mysqli_fetch_assoc($result_quote_info);
    $quote_no_ref = trim($row_quote['quote_no'] ?? ''); $ajaxResponseData['quote_no_reference'] = $quote_no_ref;
    $accountName = isset($row_quote['accountname']) ? str_replace('&','and',$row_quote['accountname']) : 'N/A'; $ajaxResponseData['accountName'] = $accountName;
    $subject = isset($row_quote['subject']) ? str_replace('&','and',$row_quote['subject']) : ''; $ajaxResponseData['customer_email'] = $row_quote['account_email'] ?? null;
    $address_line1 = isset($row_quote['full_address']) ? str_replace('&','and',$row_quote['full_address']) : '';
    $quote_country = $row_quote['quote_country'] ??  '';
    $address_country = $row_quote['bill_country_from_org'] ??  '';
    // $address_city = $row_quote['bill_city'] ?? ''; // Populate if you add bill_city to SQL and your tdu_organisation table
    // $address_postal_code = $row_quote['bill_postal_code'] ?? ''; // Populate if you add bill_postal_code to SQL
    if (strcasecmp($quote_country, 'New Zealand') == 0) $currency = 'NZD'; else $currency = 'AUD'; $ajaxResponseData['currency'] = $currency;
    $adults_no = (int)($row_quote['adults']??0); $children_no = (int)($row_quote['children']??0); $infants_no = (int)($row_quote['infants']??0);
    $no_pax = $adults_no + $children_no; if ($no_pax == 0 && $infants_no == 0) { $errorMessage = "No passengers for Quote ID: {$quoteid}."; logMessage("ERROR: ".$errorMessage); $ajaxResponseData['error']=$errorMessage; echo json_encode($ajaxResponseData, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES); exit(); }
    if (isset($row_quote['payment_due_date_custom_field']) && $row_quote['payment_due_date_custom_field']!='0000-00-00' && !empty($row_quote['payment_due_date_custom_field'])) { try { $payment_deadline = (new DateTime($row_quote['payment_due_date_custom_field']))->format('Y-m-d'); logMessage("Due date from cf_1182 for QID {$quoteid}: {$payment_deadline}"); } catch (Exception $e) { logMessage("Warn: Invalid due date '{$row_quote['payment_due_date_custom_field']}' for QID {$quoteid}. Due date empty."); $payment_deadline = ''; $ajaxResponseData['warnings'][]="Invalid payment due date from DB.";}}
    if (isset($row_quote['invoice_date_custom_field']) && !empty($row_quote['invoice_date_custom_field']) && $row_quote['invoice_date_custom_field'] != '0000-00-00') { try { $invoice_transaction_date = (new DateTime($row_quote['invoice_date_custom_field']))->format('Y-m-d'); logMessage("Using custom invoice date from cf_1162 for QID {$quoteid}: {$invoice_transaction_date}"); } catch (Exception $e) { logMessage("Warn: Invalid custom invoice date '{$row_quote['invoice_date_custom_field']}' for QID {$quoteid}. Using default: {$invoice_transaction_date}."); $ajaxResponseData['warnings'][]="Invalid custom invoice date from DB. Defaulting.";}} else { logMessage("Custom invoice date (cf_1162) not set/zero for QID {$quoteid}. Using default: {$invoice_transaction_date}."); }
    $ajaxResponseData['invoice_transaction_date_used'] = $invoice_transaction_date;
} else { $errorMessage = "Quote details not found for ID: " . $quoteid; logMessage("ERROR: ".$errorMessage); $ajaxResponseData['error']=$errorMessage; echo json_encode($ajaxResponseData, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES); exit(); }
if (empty($quote_no_ref)) { $errorMessage = "Quote Number (from vq.quote_no, for Class name) missing for QID: {$quoteid}. Cannot proceed."; logMessage("FATAL: ".$errorMessage); $ajaxResponseData['error']=$errorMessage; echo json_encode($ajaxResponseData, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES); exit(); }

$qboUtilLibrary = new QBOUtilityLibrary($qboConfig); // Instantiate library once for subsequent QBO operations

// --- Determine the "Sales Category Product/Service" Name ---
$salesCategoryItemName = ''; $isGroupQuote = (substr(strtoupper($quote_no_ref), -1) === 'G');
if (strcasecmp($quote_country, 'Australia') == 0) { $salesCategoryItemName = $isGroupQuote ? 'Sales - AU Groups' : 'Sales'; }
elseif (strcasecmp($quote_country, 'New Zealand') == 0) { $salesCategoryItemName = $isGroupQuote ? 'Sales - NZ Groups' : 'Sales - Newzealand'; }
else { $warningMessage = "Unsupported country ('{$quote_country}') for Sales Category determination. Quote ID {$quoteid}. Category item will not be added/used."; logMessage("WARNING: ".$warningMessage); $ajaxResponseData['warnings'][]=$warningMessage; }
$ajaxResponseData['sales_category_item_name_determined'] = $salesCategoryItemName;
logMessage("Determined Sales Category Item Name: '{$salesCategoryItemName}' (Country: {$quote_country}, IsGroup: ".($isGroupQuote?'Yes':'No').") for QID: {$quoteid}");

// --- Get or Create the "Sales Category Product/Service" Item in QBO ---
$salesCategoryItemId = null;
if (!empty($salesCategoryItemName)) {
    if (DEFAULT_INCOME_ACCOUNT_ID_FOR_SALES_CATEGORY === 'YOUR_INCOME_ACCOUNT_ID' || empty(DEFAULT_INCOME_ACCOUNT_ID_FOR_SALES_CATEGORY)) {
        $errorMessage = "CRITICAL CONFIG ERROR: DEFAULT_INCOME_ACCOUNT_ID_FOR_SALES_CATEGORY is not correctly set in the script. Cannot create/use Sales Category item '{$salesCategoryItemName}'. Please update the define statement.";
        logMessage($errorMessage); $ajaxResponseData['warnings'][] = $errorMessage; $salesCategoryItemName = ''; // Clear name to prevent trying to add it as a line later
    } else {
        try {
            logMessage("Attempting to get/create Sales Category Item in QBO: '{$salesCategoryItemName}' for QID: {$quoteid}");
            $defaultItemDataForSalesCategory = [
                'Type' => SALES_CATEGORY_ITEM_TYPE,
                'IncomeAccountRef' => ['value' => DEFAULT_INCOME_ACCOUNT_ID_FOR_SALES_CATEGORY],
                'TrackQtyOnHand' => false
            ];
            $salesCategoryItemObject = $qboUtilLibrary->getOrCreateItemByName($salesCategoryItemName, $defaultItemDataForSalesCategory);
            if ($salesCategoryItemObject && !empty($salesCategoryItemObject->Id)) {
                $salesCategoryItemId = $salesCategoryItemObject->Id;
                $ajaxResponseData['sales_category_item_id_used'] = $salesCategoryItemId;
                logMessage("Using Sales Category Item '{$salesCategoryItemName}' with ID: {$salesCategoryItemId} for QID: {$quoteid}");
            } else {
                $warningMessage = "Failed to get or create Sales Category Item '{$salesCategoryItemName}' in QBO. No valid Item ID returned. This item will not be added to the invoice for QID: {$quoteid}.";
                logMessage("WARNING: " . $warningMessage); $ajaxResponseData['warnings'][] = $warningMessage;
            }
        } catch (IdsException $e) {
            $errorMessage = "QBO API Error during get/create Sales Category Item '{$salesCategoryItemName}' (QID: {$quoteid}): " . $e->getMessage() . " | Response: " . $e->getResponseBody();
            logMessage("ERROR: " . $errorMessage); $ajaxResponseData['warnings'][] = "API error processing Sales Category Item '{$salesCategoryItemName}'. It may not be added.";
        } catch (\Exception $e) {
            $errorMessage = "PHP Error during get/create Sales Category Item '{$salesCategoryItemName}' (QID: {$quoteid}): " . $e->getMessage();
            logMessage("ERROR: " . $errorMessage); $ajaxResponseData['warnings'][] = "PHP error processing Sales Category Item '{$salesCategoryItemName}'. It may not be added.";
        }
    }
}

// --- Fetch Room Configuration and Pricing ---
logMessage("Fetching room configuration and pricing for Quote ID: " . $quoteid); $mode = 'group';
$invoice_initial_query = "SELECT MAX(created_at) AS created_at FROM vtiger_invoice WHERE quoteid = '$quoteid' AND type = 'final'";
$invoice_initial_result = mysqli_query($conn, $invoice_initial_query); $created_at_query = "";
if ($invoice_initial_result && $invoice_initial = mysqli_fetch_assoc($invoice_initial_result)) { $created_at_query = !empty($invoice_initial['created_at']) ? " AND created_at < '" . mysqli_real_escape_string($conn, $invoice_initial['created_at']) . "' " : ""; }
$sql_rooms = "SELECT meta_key, meta_value FROM vtiger_itinerary WHERE quoteid = '$quoteid' AND meta_key IN ('single_rooms', 'double_rooms', 'triple_rooms', 'child_without_bed') AND (meta_key, created_at) IN (SELECT meta_key, MAX(created_at) FROM vtiger_itinerary WHERE quoteid = '$quoteid' {$created_at_query} AND meta_key IN ('single_rooms', 'double_rooms', 'triple_rooms', 'child_without_bed') GROUP BY meta_key)";
$result_rooms = mysqli_query($conn, $sql_rooms);
if ($result_rooms) { while ($row_room = mysqli_fetch_assoc($result_rooms)) { if ($row_room['meta_key'] == 'single_rooms') $single_rooms = (int)$row_room['meta_value']; elseif ($row_room['meta_key'] == 'double_rooms') $double_rooms = (int)$row_room['meta_value']; elseif ($row_room['meta_key'] == 'triple_rooms') $triple_rooms = (int)$row_room['meta_value']; elseif ($row_room['meta_key'] == 'child_without_bed') $child_no_bed = (int)$row_room['meta_value']; } logMessage("Room counts for QID {$quoteid}: Sgl={$single_rooms}, Dbl={$double_rooms}, Tpl={$triple_rooms}, CNB={$child_no_bed}"); } else { logMessage("WARN: DB query for rooms failed (QID {$quoteid}). Err: ".mysqli_error($conn)); $ajaxResponseData['warnings'][]="DB room query failed.";}
$child_with_bed = $children_no - $child_no_bed; if ($child_with_bed < 0) { logMessage("Warn: ChildWithBed negative for QID {$quoteid}. Set to 0."); $child_with_bed = 0; } logMessage("Child counts for QID {$quoteid}: Tot={$children_no}, CWB={$child_with_bed}, CNB={$child_no_bed}");
$quote_pax_for_pricing = -1; $no_pax_calc_for_tier = $adults_no + $children_no;
if ($mode == 'group' && $no_pax_calc_for_tier > 0) { $mode_sql_condition = '>1'; $sql_group_pricing_tier = "SELECT MIN(vps.pax_min) AS pax_min, MAX(vps.pax_max) AS pax_max FROM vtiger_products_saleprice vps WHERE vps.quoteid='$quoteid' AND vps.subquoteid $mode_sql_condition AND vps.cf_928='Transfers' GROUP BY vps.subquoteid ORDER BY pax_min ASC"; $result_group_pricing_tier = mysqli_query($conn, $sql_group_pricing_tier); if ($result_group_pricing_tier) { if (mysqli_num_rows($result_group_pricing_tier) > 0) { while ($row_tier = mysqli_fetch_assoc($result_group_pricing_tier)) { if ($row_tier['pax_min'] <= ($no_pax_calc_for_tier) && ($no_pax_calc_for_tier) <= $row_tier['pax_max']) { $quote_pax_for_pricing = $row_tier['pax_min']; logMessage("Matched pricing tier for QID {$quoteid}: PaxMin={$quote_pax_for_pricing}"); break; }}} if ($quote_pax_for_pricing === -1) logMessage("Warn: No specific pricing tier matched for Group (QID {$quoteid}). SQL: ".$sql_group_pricing_tier); } else { logMessage("Warn: DB query for group pricing failed (QID {$quoteid}). Err: ".mysqli_error($conn)); $ajaxResponseData['warnings'][]="DB group pricing query failed.";} } elseif ($mode != 'group') logMessage("Not 'group' mode for QID {$quoteid}. Using default pricing tier."); elseif ($no_pax_calc_for_tier == 0) logMessage("No paying pax for QID {$quoteid}. Using default pricing tier for group logic.");
$pricing = getQuotePricing($conn, $quoteid, $quote_pax_for_pricing);
if (!is_array($pricing) || empty($pricing)) { $warningMessage = "Pricing info not found/invalid from getQuotePricing() for QID:{$quoteid}, PaxTier:{$quote_pax_for_pricing}. Using zeros."; logMessage("Warn: ".$warningMessage); $ajaxResponseData['warnings'][]=$warningMessage; $pricing = ['single_room'=>0,'double_room'=>0,'triple_room'=>0,'child_with_bed'=>0,'child_no_bed'=>0,'infant'=>0];}
$pricing_single_room = $pricing['single_room']??0; $pricing_double_room = $pricing['double_room']??0; $pricing_triple_room = $pricing['triple_room']??0; $pricing_child_with_bed = $pricing['child_with_bed']??0; $pricing_child_no_bed = $pricing['child_no_bed']??0; $pricing_infant = $pricing['infant']??0;
logMessage("Pricing (QID {$quoteid}): SglP={$pricing_single_room}, DblP={$pricing_double_room}, TplP={$pricing_triple_room}, CWBP={$pricing_child_with_bed}, CNBP={$pricing_child_no_bed}, InfP={$pricing_infant}");

// --- Get or Create QBO Class ---
$qboClassIdForInvoiceLines = null;
try { logMessage("Get/Create QBO Class for: '" . $quote_no_ref . "' for QID: {$quoteid}"); $classObject = $qboUtilLibrary->getOrCreateClassByName($quote_no_ref); if ($classObject && !empty($classObject->Id)) { $qboClassIdForInvoiceLines = $classObject->Id; $ajaxResponseData['qbo_class_id_used'] = $qboClassIdForInvoiceLines; logMessage("Using QBO Class ID: {$qboClassIdForInvoiceLines} for '{$quote_no_ref}' (QID: {$quoteid})"); } else { logMessage("Warn: Failed to get/create Class '{$quote_no_ref}' for QID: {$quoteid}."); $ajaxResponseData['warnings'][]="Class '{$quote_no_ref}' not processed.";}}
catch (Exception $e) { $errMsg = "ERR (Class '{$quote_no_ref}' QID: {$quoteid}): ".$e->getMessage().($e instanceof IdsException ? " | Resp: ".$e->getResponseBody() : ""); logMessage($errMsg); $ajaxResponseData['warnings'][]=$errMsg;}

// --- Determine TaxCode ID to use ---
$taxCodeIdForLines = FALLBACK_NON_TAXABLE_TAX_CODE_ID;
try { logMessage("Find TaxCode: '" . TARGET_GST_FREE_TAX_CODE_NAME . "' for QID: {$quoteid}"); $foundTaxCode = $qboUtilLibrary->findTaxCodeByName(TARGET_GST_FREE_TAX_CODE_NAME); if ($foundTaxCode && !empty($foundTaxCode->Id)) { $taxCodeIdForLines = $foundTaxCode->Id; logMessage("Found TaxCode '" . TARGET_GST_FREE_TAX_CODE_NAME . "' ID: " . $taxCodeIdForLines . " (QID: {$quoteid})"); } else { logMessage("Warn: TaxCode '" . TARGET_GST_FREE_TAX_CODE_NAME . "' not found for QID: {$quoteid}. Using fallback: ".FALLBACK_NON_TAXABLE_TAX_CODE_ID); $ajaxResponseData['warnings'][]="Target TaxCode not found, used fallback.";}}
catch (Exception $e) { $errMsg = "ERR (TaxCode '".TARGET_GST_FREE_TAX_CODE_NAME."' QID: {$quoteid}): ".$e->getMessage().($e instanceof IdsException ? " | Resp: ".$e->getResponseBody() : ""); logMessage($errMsg); $ajaxResponseData['warnings'][]=$errMsg;}
$ajaxResponseData['tax_code_id_used_for_lines'] = $taxCodeIdForLines;

// --- Get or Create QBO Term "Due on receipt" for Invoices ---
$qboTermRefIdForInvoice = null;
try {
    $termNameForInvoice = QBO_DEFAULT_TERM_NAME_FOR_INVOICE;
    $defaultTermDataForInvoice = ['Name' => $termNameForInvoice, 'Type' => 'STANDARD', 'DueDays' => 0];
    logMessage("Attempting to get/create QBO Term for Invoices: '{$termNameForInvoice}' for QID: {$quoteid}");
    $termObject = $qboUtilLibrary->getOrCreateTermByName($termNameForInvoice, $defaultTermDataForInvoice);
    if ($termObject && !empty($termObject->Id)) {
        $qboTermRefIdForInvoice = $termObject->Id;
        $ajaxResponseData['qbo_term_id_used_for_invoice'] = $qboTermRefIdForInvoice;
        logMessage("QBO Term '{$termObject->Name}' (ID: {$qboTermRefIdForInvoice}) processed for invoice headers for QID: {$quoteid}.");
    } else {
        $warningMsg = "Failed to get/create QBO Term '{$termNameForInvoice}' for invoices for QID: {$quoteid}.";
        logMessage("WARN: " . $warningMsg);
        $ajaxResponseData['warnings'][] = $warningMsg;
    }
} catch (IdsException $e) {
    $errMsg = "QBO API Error during get/create Term '{$termNameForInvoice}' for QID {$quoteid}: " . $e->getMessage() . " | Resp: " . $e->getResponseBody();
    logMessage("ERROR: " . $errMsg);
    $ajaxResponseData['warnings'][] = "API error processing Term '{$termNameForInvoice}'. It may not be applied to the invoice.";
} catch (\Exception $e) {
    $errMsg = "PHP Error during get/create Term '{$termNameForInvoice}' for QID {$quoteid}: " . $e->getMessage();
    logMessage("ERROR: " . $errMsg);
    $ajaxResponseData['warnings'][] = "PHP error processing Term '{$termNameForInvoice}'. It may not be applied to the invoice.";
}
// --- End Get or Create QBO Term ---

// --- Helper function to build QBO Line Items ---
function addQBOLineItem($lineDescription, $unitPrice, $quantity, &$qboLinesArray, $classIdForLine, $classNameForRef, $taxCodeId, $qboSpecificItemId) {
    if ($quantity <= 0 && $unitPrice > 0) {
        logMessage("Skipping line '{$lineDescription}' due to zero/negative quantity with a positive price. For reductions, use negative price and positive quantity.");
        return;
    }
    if (empty($qboSpecificItemId)) {
        logMessage("CRITICAL WARNING: qboSpecificItemId empty for line '{$lineDescription}'. Line will likely fail or use unintended QBO default item.");
    }
    $lineAmount = round((float)$unitPrice * (int)$quantity, 2);
    $salesItemLineDetail = ["ItemRef"=>["value"=>$qboSpecificItemId], "TaxCodeRef"=>["value"=>$taxCodeId], "Qty"=>(int)$quantity, "UnitPrice"=>(float)$unitPrice];
    if (!empty($classIdForLine)) $salesItemLineDetail["ClassRef"] = ["value"=>$classIdForLine, "name"=>$classNameForRef];
    $qboLinesArray[] = ["Amount"=>$lineAmount, "DetailType"=>"SalesItemLineDetail", "SalesItemLineDetail"=>$salesItemLineDetail, "Description"=>$lineDescription];
}
// --- Temporary $content array to build line details (as per your backup file) ---
$contentForInvoiceLines = [];
logMessage("Building contentForInvoiceLines array for QID: " . $quoteid);

// Add Package Components to $contentForInvoiceLines
if ($single_rooms > 0) { // Only add if quantity is positive
    $contentForInvoiceLines[] = ['Product'=>'Package Cost Per Adult on Single Occupancy', 'Pricing'=>$pricing_single_room, 'Number'=>$single_rooms, 'IsReduction'=>false];
}
if ($double_rooms > 0) {
    $contentForInvoiceLines[] = ['Product'=>'Package Cost Per Adult on Double Occupancy', 'Pricing'=>$pricing_double_room, 'Number'=>($double_rooms * 2), 'IsReduction'=>false];
}
if ($triple_rooms > 0) {
    $adults_in_triple = $triple_rooms * 3 - $child_with_bed;
    if ($adults_in_triple < 0) $adults_in_triple = 0; // Ensure not negative
    if ($adults_in_triple > 0) {
        $contentForInvoiceLines[] = ['Product'=>'Package Cost Per Adult on Triple Occupancy', 'Pricing'=>$pricing_triple_room, 'Number'=>$adults_in_triple, 'IsReduction'=>false];
    }
}
if ($single_rooms == 0 && $double_rooms == 0 && $triple_rooms == 0) { // No hotel options
    if($adults_no > 0) {
        $contentForInvoiceLines[] = ['Product'=>'Package Cost Per Adult No Hotel', 'Pricing'=>$pricing_single_room, 'Number'=>$adults_no, 'IsReduction'=>false];
    }
    if($children_no > 0) {
        $contentForInvoiceLines[] = ['Product'=>'Package Cost Per Child No Hotel', 'Pricing'=>$pricing_child_no_bed, 'Number'=>$children_no, 'IsReduction'=>false];
    }
} else { // Hotel options for children
    if ($child_with_bed > 0) {
        $contentForInvoiceLines[] = ['Product'=>'Package Cost Per Child with Bed', 'Pricing'=>$pricing_child_with_bed, 'Number'=>$child_with_bed, 'IsReduction'=>false];
    }
    if ($child_no_bed > 0) {
        $contentForInvoiceLines[] = ['Product'=>'Package Cost Per Child without Bed', 'Pricing'=>$pricing_child_no_bed, 'Number'=>$child_no_bed, 'IsReduction'=>false];
    }
}
if($infants_no > 0) {
    // Add infant line even if price is $0, as long as quantity is present
    $contentForInvoiceLines[] = ['Product'=>'Package Cost Per Infant', 'Pricing'=>$pricing_infant, 'Number'=>$infants_no, 'IsReduction'=>false];
}

// Processing Adjustments from vtiger_invoice_add_products
logMessage("Processing adjustments for QID: " . $quoteid . " to add to contentForInvoiceLines.");
$sql_adjustments = "SELECT remark, product_field, price_field, quantity FROM vtiger_invoice_add_products WHERE quoteid='$quoteid' ORDER BY created_at DESC";
$result_adjustments = mysqli_query($conn, $sql_adjustments);
if ($result_adjustments && mysqli_num_rows($result_adjustments) > 0) {
    while ($row_adjustment = mysqli_fetch_assoc($result_adjustments)) {
        $remark_cleaned = str_replace('&', 'and', $row_adjustment['remark'] ?? 'Adjustment');
        $product_field_adj = str_replace('&', 'and', $row_adjustment['product_field'] ?? '');
        $price_field_adj = (float)($row_adjustment['price_field'] ?? 0);
        $quantity_adj = (int)($row_adjustment['quantity'] ?? 1);

        $adjustmentLineDescription = $remark_cleaned;
        if (!empty($product_field_adj) && $product_field_adj !== $remark_cleaned) {
            $adjustmentLineDescription .= " - " . $product_field_adj;
        }
        // A negative price_field_adj indicates a reduction-type adjustment
        $contentForInvoiceLines[] = ['Product' => $adjustmentLineDescription, 'Pricing' => $price_field_adj, 'Number' => $quantity_adj, 'IsReduction'=> ($price_field_adj < 0) ];
    }
} elseif (!$result_adjustments) {
    logMessage("WARNING: Database query for adjustments failed for QID {$quoteid}. Error: " . mysqli_error($conn));
    $ajaxResponseData['warnings'][] = "Could not query for invoice adjustments.";
}


// NEW: Processing Reductions from vtiger_invoice_refunds (as per backup logic)
logMessage("Processing refunds for QID: " . $quoteid . " to add to contentForInvoiceLines.");
$sql_refunds = "SELECT product, refund_amount, refund_reason FROM vtiger_invoice_refunds WHERE quoteid='$quoteid' ORDER BY created_at DESC";
$result_refunds = mysqli_query($conn, $sql_refunds);
if ($result_refunds && mysqli_num_rows($result_refunds) > 0) {
    while ($row_refund = mysqli_fetch_assoc($result_refunds)) {
        $product_refund = str_replace('&', 'and', $row_refund['product'] ?? 'Refund');
        $refund_amount_val = (float)($row_refund['refund_amount'] ?? 0);
        $refund_reason_val = $row_refund['refund_reason'] ?? '';

        $reductionDescription = "Reduction: " . $product_refund;
        if (!empty($refund_reason_val)) {
            $reductionDescription .= " (" . $refund_reason_val . ")";
        }
        // Reductions are inherently negative. Store price as negative, ensure quantity is positive.
        $contentForInvoiceLines[] = ['Product' => $reductionDescription, 'Pricing' => -$refund_amount_val, 'Number' => 1, 'IsReduction'=>true];
    }
} elseif (!$result_refunds) {
    logMessage("WARNING: Database query for refunds failed for QID {$quoteid}. Error: " . mysqli_error($conn));
    $ajaxResponseData['warnings'][] = "Could not query for invoice refunds.";
}


// --- Construct QBO Invoice Line Items by iterating through $contentForInvoiceLines ---
$qboInvoiceLineItems = []; // Reset before populating
logMessage("Constructing QBO invoice lines from content array for QID: {$quoteid}. Using Sales Category Item ID: {$salesCategoryItemId}, TaxCode ID: {$taxCodeIdForLines}");

if (!empty($salesCategoryItemId)) { // CRITICAL: Only proceed if we have a valid Sales Category Item ID
    foreach ($contentForInvoiceLines as $contentLine) {
        $productDescription = $contentLine['Product'];
        $unitPrice = (float)$contentLine['Pricing'];
        $number = (int)$contentLine['Number'];

        if ($unitPrice != 0 && $number == 0) {
            $number = 1; // If there's a price, default quantity to 1 if it was 0.
            logMessage("Adjusted quantity to 1 for line '{$productDescription}' because unit price was non-zero with zero quantity.");
        }
        if ($unitPrice < 0 && $number < 0) { // If price is negative (reduction), ensure quantity is positive.
            $number = abs($number);
            logMessage("Adjusted quantity to positive for reduction line '{$productDescription}'.");
        }

        // Skip adding line if description is empty AND it's a zero-value, zero-quantity line (truly empty)
        if ($number == 0 && $unitPrice == 0 && empty(trim($productDescription)) ) {
            logMessage("Skipping truly empty line from content array.");
            continue;
        }

        addQBOLineItem(
            $productDescription,
            $unitPrice,
            $number,
            $qboInvoiceLineItems,
            $qboClassIdForInvoiceLines,
            $quote_no_ref,
            $taxCodeIdForLines,
            $salesCategoryItemId // Product/Service for ALL lines is the Sales Category Item
        );
        logMessage("Added QBO line: Desc='{$productDescription}', Qty: {$number}, Rate: {$unitPrice}, using SalesCatID: {$salesCategoryItemId} for QID: {$quoteid}");
    }
} else {
    $errorMessage = "CRITICAL: Sales Category Item ID is missing for Quote ID {$quoteid}. Cannot add invoice lines as they depend on this item for their Product/Service. Check previous logs for Sales Category item creation issues.";
    logMessage($errorMessage);
    $ajaxResponseData['error']=$errorMessage;
    $ajaxResponseData['warnings'][]=$errorMessage;
    // If this is critical, you might want to exit or prevent further processing:
    // echo json_encode($ajaxResponseData, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES); exit();
}
// --- End of Construct QBO Invoice Line Items ---

// This adjustment processing block seems redundant if the contentForInvoiceLines already includes them.
// However, keeping it for now if there's a specific reason it was separate and using $salesCategoryItemId.
// If $contentForInvoiceLines handles all lines, this block and the separate refund block might be simplified.
logMessage("Processing adjustments for QID: " . $quoteid); // This log might be for the original adjustment block if it's still needed.
// $sql_adjustments... and its processing loop already ran and added to $contentForInvoiceLines
// If those lines were *not* using $salesCategoryItemId before and now *should*, the logic in addQBOLineItem
// and the loop over $contentForInvoiceLines already covers this.

if (empty($qboInvoiceLineItems) && !empty($ajaxResponseData['error'])) { logMessage("Exiting due to previous critical error (likely missing Sales Category Item): " . $ajaxResponseData['error'] . " for QID: {$quoteid}"); }
elseif (empty($qboInvoiceLineItems)) { $errorMessage = "No lines (main category or adjustments) generated for invoice from QID: {$quoteid}."; logMessage("ERR: ".$errorMessage); $ajaxResponseData['error']=$errorMessage; echo json_encode($ajaxResponseData, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES); exit(); }
logMessage("Total invoice lines: ".count($qboInvoiceLineItems)." for QID: ".$quoteid);

// III. PREPARE DATA FOR QBO UTILITY LIBRARY (CUSTOMER & INVOICE METADATA)
logMessage("Preparing customer & invoice metadata for QID: ".$quoteid);
$customerDataForQBO = ['DisplayName'=>$accountName, 'CompanyName'=>$accountName];
if ($currency === 'NZD') {
    $customerDataForQBO['CurrencyRef'] = ['value' => 'NZD'];
    logMessage("Setting new customer {$accountName} to NZD currency during creation.");
}
if (!empty($ajaxResponseData['customer_email'])) $customerDataForQBO['PrimaryEmailAddr'] = ["Address"=>$ajaxResponseData['customer_email']];
$billAddr = []; if (!empty($address_line1)) $billAddr['Line1'] = $address_line1; if (!empty($address_city)) $billAddr['City'] = $address_city; if (!empty($address_postal_code)) $billAddr['PostalCode'] = $address_postal_code; if (!empty($quote_country)) $billAddr['Country'] = $address_country;
if (!empty($billAddr)) $customerDataForQBO['BillAddr'] = $billAddr;
$customerConfigForQBO = ['search_field'=>'DisplayName', 'search_value'=>$accountName, 'create_data'=>$customerDataForQBO];

$newInvoiceDocNumber = 'C'.date('YmdHis');
$invoiceMetaDataForQBO = ["DocNumber"=>$newInvoiceDocNumber, "TxnDate"=>$invoice_transaction_date];
if (!empty($payment_deadline)) $invoiceMetaDataForQBO["DueDate"] = $payment_deadline;
if (!empty($ajaxResponseData['customer_email'])) $invoiceMetaDataForQBO["BillEmail"] = ["Address"=>$ajaxResponseData['customer_email']];

// Add SalesTermRef if a term was successfully processed
if (!empty($qboTermRefIdForInvoice)) {
    $invoiceMetaDataForQBO["SalesTermRef"] = ["value" => $qboTermRefIdForInvoice];
    logMessage("Added SalesTermRef ID {$qboTermRefIdForInvoice} to invoice metadata for QID: {$quoteid}.");
}

if ($currency !== DEFAULT_QBO_HOME_CURRENCY) {
    $invoiceMetaDataForQBO["CurrencyRef"] = ["value" => $currency];
    logMessage("Invoice currency set to: {$currency} for customer '{$accountName}'. TxnDate for rate lookup: {$invoice_transaction_date}.");

    $fetchedExchangeRate = null;
    $exchangeRateFetchError = null;

    try {
        $exchangeRateObject = $qboUtilLibrary->getMostRecentExchangeRate($currency);

        if ($exchangeRateObject && property_exists($exchangeRateObject, 'Rate')) {
            $rateValue = (float) $exchangeRateObject->Rate;
            if ($rateValue > 0) {
                $fetchedExchangeRate = $rateValue;
                logMessage("QBO API returned exchange rate for INVOICE: {$fetchedExchangeRate} for {$currency} to " . DEFAULT_QBO_HOME_CURRENCY . " as of " . ($exchangeRateObject->AsOfDate ?? $invoice_transaction_date) . " (LastUpdated: " . ($exchangeRateObject->MetaData->LastUpdatedTime ?? 'N/A') . ")");
                $ajaxResponseData['qbo_exchange_rate_applied_to_invoice'] = $fetchedExchangeRate;
            } else {
                $exchangeRateFetchError = "QBO API returned a non-positive rate ({$rateValue}) for INVOICE currency {$currency}. Effective Date: " . ($exchangeRateObject->AsOfDate ?? 'N/A');
            }
        } else {
            $exchangeRateFetchError = "QBO did not return a valid exchange rate object or 'Rate' property via API for INVOICE currency {$currency}.";
        }
    } catch (IdsException $e) {
        $exchangeRateFetchError = "QBO API Error querying for exchange rate for INVOICE currency {$currency} (QID: {$quoteid}): " . $e->getMessage() . " | Resp: " . $e->getResponseBody();
    } catch (\Exception $e) {
        $exchangeRateFetchError = "PHP Error querying for exchange rate for INVOICE currency {$currency} (QID: {$quoteid}): " . $e->getMessage();
    }

    if ($fetchedExchangeRate !== null) {
        $invoiceMetaDataForQBO["ExchangeRate"] = $fetchedExchangeRate;
    } else {
        if ($exchangeRateFetchError) {
            logMessage("WARNING: " . $exchangeRateFetchError);
            $ajaxResponseData['warnings'][] = $exchangeRateFetchError;
        }
        logMessage("WARNING: No valid exchange rate obtained/set via API for INVOICE currency {$currency}. Letting QBO attempt to apply its own automated rate (if any). Invoice for QID: {$quoteid}.");
        $ajaxResponseData['warnings'][] = "No valid exchange rate obtained for Invoice currency {$currency}. QBO will attempt to use its own rate.";
    }
} else {
    if (!empty($currency)) {
         $invoiceMetaDataForQBO["CurrencyRef"] = ["value" => $currency];
         logMessage("Invoice currency is home currency: {$currency}. Explicitly setting CurrencyRef for QID: {$quoteid}.");
    }
}
// $invoiceMetaDataForQBO["CustomerMemo"] = ["value" => "Ref Quote: " . $quote_no_ref . " / Subject: " . $subject];

// IV. CALL QBO UTILITY LIBRARY TO CREATE INVOICE
logMessage("Attempting QBO Invoice workflow for QID: ".$quoteid);
try {
    $ajaxResponseData['debug_customer_config_sent'] = $customerConfigForQBO; $ajaxResponseData['debug_invoice_metadata_sent'] = $invoiceMetaDataForQBO;
    // $ajaxResponseData['debug_invoice_lines_sent'] = $qboInvoiceLineItems;
    $qboResult = $qboUtilLibrary->processInvoiceCreationWorkflow($customerConfigForQBO, $qboInvoiceLineItems, $invoiceMetaDataForQBO, null);
    $invoiceLinkBase = ($qboConfig['baseUrl'] === "Development") ? "https://app.sandbox.qbo.intuit.com/app/invoice?txnId=" : "https://app.qbo.intuit.com/app/invoice?txnId=";
    $ajaxResponseData['qbo_result'] = ['status'=>'SUCCESS', 'customer_id'=>$qboResult['customer']->Id??'N/A', 'customer_name'=>$qboResult['customer']->DisplayName??'N/A', 'invoice_id'=>$qboResult['invoice']->Id??'N/A', 'invoice_doc_number'=>$qboResult['invoice']->DocNumber??'N/A', 'invoice_total_amt'=>$qboResult['invoice']->TotalAmt??'N/A', 'qbo_invoice_link'=>$invoiceLinkBase.($qboResult['invoice']->Id??'')];
    logMessage("SUCCESS: QBO Invoice created. QID: {$quoteid}. InvID: ".($qboResult['invoice']->Id??'N/A').", DocNum: ".($qboResult['invoice']->DocNumber??'N/A'));
} catch (IdsException $e) {
    $errorMessage = "QBO API Err (Invoice Creation - QID: {$quoteid}): ".$e->getMessage()." | HTTP: ".$e->getHttpStatusCode()." | QBO ErrCode: ".$e->getOAuthHelperError()." | Resp: ".$e->getResponseBody();
    logMessage("ERROR: ".$errorMessage); $ajaxResponseData['error']="QBO API Err (Invoice): ".$e->getMessage(); $ajaxResponseData['qbo_result']=['status'=>'ERROR_QBO_API_INVOICE', 'message'=>$e->getMessage(), 'details'=>$e->getResponseBody(), 'http_status'=>$e->getHttpStatusCode()];
    if ($e->getHttpStatusCode()==401 || stripos($e->getResponseBody()??'','AuthenticationFailed')!==false || stripos($e->getResponseBody()??'','bearer_token_expired')!==false) { logMessage("CRITICAL: OAuth token likely expired/invalid. QID: {$quoteid}. Err: ".$e->getMessage()); }
} catch (\Exception $e) {
    $errorMessage = "PHP Err (Invoice Creation - QID: {$quoteid}): ".$e->getMessage()." in ".$e->getFile()." on line ".$e->getLine();
    logMessage("ERROR: ".$errorMessage); $ajaxResponseData['error']="PHP Err (Invoice): ".$e->getMessage(); $ajaxResponseData['qbo_result']=['status'=>'ERROR_PHP_INVOICE', 'message'=>$e->getMessage(), 'file'=>$e->getFile(), 'line'=>$e->getLine()];
}

// V. OUTPUT FINAL JSON RESPONSE
logMessage("Cron job finished for Quote ID: " . $quoteid . ". Outputting response.");
echo json_encode($ajaxResponseData, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
?>