<?php

function getFromDatabase($conn, $column) {
    $sql = "SELECT DISTINCT $column FROM vtiger_vendor";
    $result = $conn->query($sql);
    $data = [];
    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $data[] = $row[$column];
        }
    }
    return $data;
}

function getProductsBySupplier($conn) {
    $sql = "SELECT DISTINCT vp.productid, vp.productname, vv.vendorname
            FROM vtiger_products vp
            LEFT JOIN vtiger_vendor vv ON vp.vendor_id = vv.vendorid";
    $result = $conn->query($sql);
    $supplierDictionary = array();
    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $supplierName = $row['vendorname'];
            $name = $row['productname'];
    
            if (!isset($supplierDictionary[$supplierName])) {
                $supplierDictionary[$supplierName] = array();
            }
            $supplierDictionary[$supplierName][] = $name;
        }
    }
    return $supplierDictionary;
}

function getProductIDByProduct($conn) {
    $sql = "SELECT DISTINCT productid, productname FROM vtiger_products
            UNION
            SELECT DISTINCT productid, productname FROM vtiger_products_custom;";
    $result = $conn->query($sql);
    $supplierDictionary = array();
    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $supplierName = $row['productname'];
            $name = $row['productid'];
    
            if (!isset($supplierDictionary[$supplierName])) {
                $supplierDictionary[$supplierName] = $name;
            }
            // $supplierDictionary[$supplierName][] = $name;
        }
    }
    return $supplierDictionary;
}

function getProductByProductID($conn) {
    $sql = "SELECT DISTINCT productid, productName AS productname FROM tdu_products
            UNION
            SELECT DISTINCT productid, productname FROM vtiger_products_custom;";
    $result = $conn->query($sql);
    $supplierDictionary = array();
    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $supplierName = $row['productid'];
            $name = $row['productname'];
    
            if (!isset($supplierDictionary[$supplierName])) {
                $supplierDictionary[$supplierName] = $name;
            }
            // $supplierDictionary[$supplierName][] = $name;
        }
    }
    return $supplierDictionary;
}

function getProductPriceByProduct($conn) {
    $sql = "SELECT DISTINCT productname, unit_price FROM vtiger_products;";
    $result = $conn->query($sql);
    $supplierDictionary = array();
    
    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $supplierName = $row['productname'];
            if (isset($row['unit_price']) && is_float($row['unit_price'])) {
                $price = number_format($row['unit_price'], 2);
            } else {
                $price = 0;
            }
    
            if (!isset($supplierDictionary[$supplierName])) {
                $supplierDictionary[$supplierName] = $price;
            }
        }
    }
    return $supplierDictionary;
}

function getProductChildPriceByProduct($conn) {
    $sql = "SELECT DISTINCT vp.productname, cf.cf_1168 AS unit_price FROM vtiger_products vp
            LEFT JOIN vtiger_productcf cf ON vp.productid=cf.productid;";
    $result = $conn->query($sql);
    $supplierDictionary = array();
    
    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $supplierName = $row['productname'];
            $price = number_format((float) $row['unit_price'] ?? 0, 2);
    
            if (!isset($supplierDictionary[$supplierName])) {
                $supplierDictionary[$supplierName] = $price;
            }
        }
    }
    return $supplierDictionary;
}

function getCFByProductID($conn) {
    $sql = "SELECT DISTINCT productid, cf_928 FROM vtiger_productcf;";
    $result = $conn->query($sql);
    $productDictionary = array();
    
    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $cfValue = $row['cf_928'];
            $productID = $row['productid'];

            if (!isset($productDictionary[$cfValue])) {
                $productDictionary[$productID] = $cfValue;
            }
            // $productDictionary[$cfValue][] = $productID;
        }
    }
    
    return $productDictionary;
}

function getCFByProductID2($conn) {
    $sql = "SELECT DISTINCT productid, category FROM tdu_products;";
    $result = $conn->query($sql);
    $productDictionary = array();
    
    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $cfValue = $row['category'];
            $productID = $row['productid'];

            if (!isset($productDictionary[$cfValue])) {
                $productDictionary[$productID] = $cfValue;
            }
            // $productDictionary[$cfValue][] = $productID;
        }
    }
    
    return $productDictionary;
}

function getTagByTagID($conn) {
    $sql = "SELECT DISTINCT id, tag, raw_tag FROM vtiger_freetags WHERE visibility = 'public';";
    $result = $conn->query($sql);
    $productDictionary = array();
    
    if ($result->num_rows > 0) {
        $blockedUsers = getBlockedUsers($conn);
        while ($row = $result->fetch_assoc()) {
            if ($row['raw_tag'] == 'person' && in_array($row['tag'], $blockedUsers)) {
                continue;
            }
            $cfValue = $row['tag'];
            $productID = $row['id'];

            // Otherwise, store the tag and ID in the dictionary
            if (!isset($productDictionary[$cfValue])) {
                $productDictionary[$productID] = $cfValue;
            }
        }
    }
    
    return $productDictionary;
}

function getTagTagType($conn) {
    $sql = "SELECT DISTINCT id, tag, raw_tag FROM vtiger_freetags ORDER BY raw_tag ASC, tag ASC;";
    $result = $conn->query($sql);
    $productDictionary = array();
    
    if ($result->num_rows > 0) {
        $blockedUsers = getBlockedUsers($conn);
        while ($row = $result->fetch_assoc()) {
            if ($row['raw_tag'] == 'person' && in_array($row['tag'], $blockedUsers)) {
                continue;
            }
            $cfValue = $row['tag'];
            $productID = $row['id'];
            $rawTag = $row['raw_tag'];

            // Otherwise, store both tag and raw_tag in the dictionary
            if (!isset($productDictionary[$productID])) {
                $productDictionary[$productID] = array(
                    'tag' => $cfValue,
                    'raw_tag' => $rawTag
                );
            }
        }
    }
    
    return $productDictionary;
}

function getBlockedUsers($conn) {
    $sql = "SELECT DISTINCT CONCAT(first_name, ' ', last_name) AS full_name 
            FROM vtiger_users 
            WHERE title = 'block';";
    
    $result = $conn->query($sql);
    $blockedUsers = array();

    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $blockedUsers[] = $row['full_name'];
        }
    }
    
    return $blockedUsers;
}

function getTagCategory($conn) {
    $sql = "SELECT DISTINCT id, tag FROM vtiger_freetags WHERE raw_tag = 'person';";
    $result = $conn->query($sql);
    $productDictionary = array();
    
    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $cfValue = $row['tag'];
            $productID = $row['id'];

            // Otherwise, store the tag and ID in the dictionary
            if (!isset($productDictionary[$cfValue])) {
                $productDictionary[$productID] = $cfValue;
            }
        }
    }
    
    return $productDictionary;
}

function fetchOrganizationNames($conn) {
    $sql = "SELECT DISTINCT(accountname) FROM vtiger_account";
    $result = mysqli_query($conn, $sql);

    $organization_names = array();
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $organization_names[] = $row['accountname'];
        }
    }
    return $organization_names;
}

function fetchInternalSales($conn) {
    $sql = "SELECT DISTINCT(name) FROM tdu_agents WHERE category = 'internal'";
    $result = mysqli_query($conn, $sql);

    $organization_names = array();
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $organization_names[] = $row['name'];
        }
    }
    return $organization_names;
}

function fetchExternalSales($conn) {
    $sql = "SELECT DISTINCT(name) FROM tdu_agents WHERE category = 'external'";
    $result = mysqli_query($conn, $sql);

    $organization_names = array();
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $organization_names[] = $row['name'];
        }
    }
    return $organization_names;
}

function fetchAssignedToNames($conn) {
    $sql = "SELECT DISTINCT CONCAT(first_name, ' ', last_name) AS full_name FROM vtiger_users";
    $result = mysqli_query($conn, $sql);

    $assigned_to_names = array();
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $assigned_to_names[] = $row['full_name'];
        }
    }
    $sql = "SELECT DISTINCT groupname AS full_name FROM vtiger_groups";
    $result = mysqli_query($conn, $sql);
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $assigned_to_names[] = '(Group) '.$row['full_name'];
        }
    }
    
    return $assigned_to_names;
}

function fetchAssignedToNamesOnly($conn) {
    $sql = "SELECT DISTINCT CONCAT(first_name, ' ', last_name) AS full_name FROM vtiger_users WHERE status='Active'";
    $result = mysqli_query($conn, $sql);

    $assigned_to_names = array();
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $assigned_to_names[] = $row['full_name'];
        }
    }    
    return $assigned_to_names;
}

function fetchAssignedToNamesOnlySales($conn) {
    $sql = "SELECT DISTINCT CONCAT(first_name, ' ', last_name) AS full_name FROM vtiger_users WHERE status='Active' AND title IN ('sales','external','qa')";
    $result = mysqli_query($conn, $sql);

    $assigned_to_names = array();
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $assigned_to_names[] = $row['full_name'];
        }
    }    
    return $assigned_to_names;
}

function fetchAssignedToNamesOnlyOps($conn) {
    $sql = "SELECT DISTINCT CONCAT(first_name, ' ', last_name) AS full_name FROM vtiger_users WHERE status='Active' AND title IN ('group','fit','qa')";
    $result = mysqli_query($conn, $sql);

    $assigned_to_names = array();
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $assigned_to_names[] = $row['full_name'];
        }
    }    
    return $assigned_to_names;
}

function fetchAssignedToRegionOnly($conn) {
    $assigned_to_names = array();
    $sql = "SELECT DISTINCT groupname AS full_name FROM vtiger_groups";
    $result = mysqli_query($conn, $sql);
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $assigned_to_names[] = '(Group) '.$row['full_name'];
        }
    }
    
    return $assigned_to_names;
}

function fetchAssignedToFITRegionOnly($conn) {
    $assigned_to_names = array();
    $sql = "SELECT DISTINCT groupname AS full_name FROM vtiger_groups";
    $result = mysqli_query($conn, $sql);
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $assigned_to_names[] = $row['full_name'];
        }
    }
    
    return $assigned_to_names;
}

function fetchContactNames($conn) {
    $sql = "SELECT DISTINCT CONCAT(firstname, ' ', lastname) AS full_name FROM vtiger_contactdetails";
    $result = mysqli_query($conn, $sql);

    $contact_names = array();
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $contact_names[] = $row['full_name'];
        }
    }
    return $contact_names;
}

function fetchQuoteStages($conn) {
    $sql = "SELECT DISTINCT quotestage FROM vtiger_quotestage WHERE quotestage != '';";
    $result = mysqli_query($conn, $sql);

    $quote_stages = array();
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $quote_stages[] = $row['quotestage'];
        }
    }
    return $quote_stages;
}

function fetchGroupNames($conn) {
    $sql = "SELECT DISTINCT category FROM tdu_task_category WHERE category != '';";
    $result = mysqli_query($conn, $sql);

    $group_names = array();
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $group_names[] = $row['category'];
        }
    }
    return $group_names;
}

function fetchUserTitlesAndNames($conn) {
    $sql = "SELECT title, CONCAT(first_name, ' ', last_name) AS name 
            FROM vtiger_users 
            WHERE title != 'block' 
            ORDER BY title ASC, name ASC;";
    
    $result = mysqli_query($conn, $sql);

    $users = array();
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $users[] = array(
                'title' => $row['title'],
                'name' => $row['name']
            );
        }
    }
    return $users;
}

function fetchAllowedQuoteStages($conn, $role, $current_stage) {
    // Get the allowed quote stages for the given role
    $sql = "SELECT quote_stages FROM tdu_user_roles WHERE user_role='$role';";
    $result = mysqli_query($conn, $sql);
    if ($result && mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);
        $quote_stages = $row['quote_stages'];
    } else {
        return array();
    }

    // Fetch the auto_id(s) of the current stage
    $sql = "SELECT stage_no FROM vtiger_quotestage WHERE quotestage = '$current_stage';";
    $result = mysqli_query($conn, $sql);
    if ($result && mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);
        $current_auto_id = $row['stage_no'];
    } else {
        return array(); // Return empty if current stage is not found
    }

    // Fetch allowed quote stages
    if (!empty($quote_stages)) {
        $sql = "SELECT auto_id, quotestage FROM vtiger_quotestage 
                WHERE auto_id IN ($quote_stages) AND quotestage != '';";
        $result = mysqli_query($conn, $sql);
    }

    $quote_stages = array();
    $quote_stages[] = [
        'quotestage' => $current_stage,
        'enabled' => true
    ];
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $stage_name = $row['quotestage'];
            if($stage_name == $current_stage) {
                continue;
            }
            $stage_auto_id = $row['auto_id'];

            // Check if the current_auto_id exists in the allowed list (comma-separated values)
            $enabled_auto_ids = explode(',', $current_auto_id);
            $is_enabled = in_array($stage_auto_id, $enabled_auto_ids);

            if($role == 'admin') {
                $is_enabled = true;
            }
            
            $quote_stages[] = [
                'quotestage' => $stage_name,
                'enabled' => $is_enabled
            ];
        }
    }

    return $quote_stages;
}

function shouldDisableAccepted($conn, $quoteid) {
    $invoice_initial_query = "SELECT MAX(created_at) AS created_at FROM vtiger_invoice WHERE quoteid = '$quoteid' AND type = 'initial'";
    $invoice_initial_result = mysqli_query($conn, $invoice_initial_query);
    $invoice_initial = mysqli_fetch_assoc($invoice_initial_result);
    $created_at_query = !empty($invoice_initial['created_at']) ? $invoice_initial['created_at'] : null;
    if($created_at_query) {
        $created_at_query = " AND created_at < '$created_at_query' ";
    }
    else { 
        $created_at_query = "";
    }

    // query the room info
    $sql = "SELECT meta_key, meta_value 
            FROM vtiger_itinerary 
            WHERE quoteid = '$quoteid' 
            AND meta_key IN ('single_rooms', 'double_rooms', 'triple_rooms', 'child_without_bed')
            AND (meta_key, created_at) IN (
                SELECT meta_key, MAX(created_at)
                FROM vtiger_itinerary
                WHERE quoteid = '$quoteid' $created_at_query
                AND meta_key IN ('single_rooms', 'double_rooms', 'triple_rooms', 'child_without_bed')
                GROUP BY meta_key);";
    $result = mysqli_query($conn, $sql);

    $single_rooms = 0;
    $double_rooms = 0;
    $triple_rooms = 0;
    $child_no_bed = 0;
    while ($row = mysqli_fetch_assoc($result)) {
        switch ($row['meta_key']) {
            case 'single_rooms':
                $single_rooms = (int) $row['meta_value'];
                break;
            case 'double_rooms':
                $double_rooms = (int) $row['meta_value'];
                break;
            case 'triple_rooms':
                $triple_rooms = (int) $row['meta_value'];
                break;
            case 'child_without_bed':
                $child_no_bed = (int) $row['meta_value'];
                break;
        }
    }
    $total_rooming = $single_rooms + $double_rooms + $triple_rooms;

    $rooming_count_check = false;
    $rooming_fillin_check = true;

    // query the pax list
    $single_room_hotel = 0;
    $double_room_hotel = 0;
    $triple_room_hotel = 0;
    $self_booking_hotel = 0;

    $sql_rooming3 = "SELECT MAX(room_type) AS room_type FROM vtiger_pax_in_hotel WHERE quoteid = '$quoteid' GROUP BY room_seq_no;";
    $result_rooming3 = mysqli_query($conn, $sql_rooming3);
    while ($row = mysqli_fetch_assoc($result_rooming3)) {
        $room_type = strtoupper(trim($row['room_type']));

        switch ($room_type) {
            case 'SINGLE':
                $single_room_hotel++;
                break;
            case 'DOUBLE':
            case 'TWIN':
                $double_room_hotel++;
                break;
            case 'TRIPLE':
                $triple_room_hotel++;
                break;
            case 'SELF BOOKING':
                $self_booking_hotel++;
                break;
        }
    }

    if ($single_rooms == $single_room_hotel && $double_rooms == $double_room_hotel && $triple_rooms == $triple_room_hotel) {
        $rooming_count_check = true;
    }
    if ($total_rooming === $self_booking_hotel) {
        $rooming_count_check = true;
    }

    $sql_rooming2 = "SELECT * FROM vtiger_pax_in_hotel WHERE quoteid = '$quoteid';";
    $result_rooming2 = mysqli_query($conn, $sql_rooming2);
    
    if ($result_rooming2 && mysqli_num_rows($result_rooming2) > 0) {
        while ($row_rooming2 = mysqli_fetch_assoc($result_rooming2)) {
            // if ($row_rooming2['room_seq_no'] >= $total_rooming) {
            //     $rooming_count_check = true;
            // }
            if ($row_rooming2['surname'] == '' || $row_rooming2['given_name'] == '' || $row_rooming2['phone'] == '') {
                $rooming_fillin_check = false;
                break;
            }
        }
    } else {
        return true;
    }

    return !$rooming_count_check || !$rooming_fillin_check;
}

function isPaymentReceivedToAccepted($conn, $quoteid) {
    $sql_payment_due_check =   "SELECT 
                                    vq.quoteid, 
                                    SUM(CAST(vph.total_amount AS DECIMAL(10,2))) AS amount_due, 
                                    SUM(CAST(vph.trams_received_amount AS DECIMAL(10,2))) AS amount_received
                                FROM vtiger_quotes vq 
                                LEFT JOIN vtiger_quotescf vqcf ON vq.quoteid = vqcf.quoteid
                                LEFT JOIN vtiger_payment_history vph ON vq.quoteid = vph.quoteid
                                WHERE vq.quoteid = '$quoteid' AND (vq.quotestage = 'Created' OR vq.quotestage = 'Requote') AND vqcf.cf_1162 <= DATE_ADD(CURDATE(), INTERVAL 15 DAY) AND (vph.source != 'final' OR vph.source IS NULL)
                                GROUP BY vq.quoteid
                                HAVING amount_received < amount_due - 30 OR amount_due <= 0 OR amount_due IS NULL OR amount_received IS NULL;";
    $result_payment_due_check = mysqli_query($conn, $sql_payment_due_check);
    if ($result_payment_due_check && mysqli_num_rows($result_payment_due_check) > 0) {
        return true;
    } else {
        return false;
    }
}

function fetchActiveQuoteStages($conn) {
    $sql = "SELECT DISTINCT vq.quotestage FROM vtiger_quotestage vq
            WHERE vq.quotestage NOT LIKE '%deliver%' 
                AND vq.quotestage NOT LIKE '%reject%'
                AND vq.quotestage NOT LIKE '%create%'
                AND vq.quotestage NOT LIKE '%requote%'
                AND vq.quotestage NOT LIKE '%complete%'
                AND vq.quotestage NOT LIKE '%Ground%'
                AND vq.quotestage NOT LIKE '%Reconciliation%'
                AND vq.quotestage NOT LIKE '%Final QA%';";
    $result = mysqli_query($conn, $sql);

    $quote_stages = array();
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $quote_stages[] = $row['quotestage'];
        }
    }
    return $quote_stages;
}

function fetchNotRejectQuoteStages($conn) {
    $sql = "SELECT DISTINCT vq.quotestage FROM vtiger_quotestage vq
            WHERE vq.quotestage NOT LIKE '%reject%';";
    $result = mysqli_query($conn, $sql);

    $quote_stages = array();
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $quote_stages[] = $row['quotestage'];
        }
    }
    return $quote_stages;
}

function fetchGroupQuoteStages($conn) {
    $sql = "SELECT DISTINCT vq.quotestage FROM vtiger_quotestage vq
            WHERE vq.quotestage != 'Handover to Ops'
                AND vq.quotestage != 'Create Project'
                AND vq.quotestage NOT LIKE '%Partial Payment Due%'
                AND vq.quotestage NOT LIKE '%TA Verification%';";
    $result = mysqli_query($conn, $sql);

    $quote_stages = array();
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            if($row['quotestage'] == 'Final QA (Rakesh/Kunjan)') {
                $quote_stages[] = 'Final QA';
            }
            else if($row['quotestage'] == 'Accounts - Reconciliation (Rakesh/Kunjan)') {
                $quote_stages[] = 'Accounts - Reconciliation';
            }
            else {
                $quote_stages[] = $row['quotestage'];
            }
        }
    }
    return $quote_stages;
}

function fetchAcceptedQuoteStages($conn) {
    $sql = "SELECT DISTINCT vq.quotestage FROM vtiger_quotestage vq
            WHERE vq.quotestage LIKE '%Accepted%';";
    $result = mysqli_query($conn, $sql);

    $quote_stages = array();
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $quote_stages[] = $row['quotestage'];
        }
    }
    return $quote_stages;
}

function getAccountContacts($conn) {
    $sql = "SELECT DISTINCT va.accountname, CONCAT(vcd.firstname, ' ', vcd.lastname) AS contactname 
            FROM vtiger_contactdetails vcd 
            LEFT JOIN vtiger_account va ON vcd.accountid = va.accountid
            WHERE va.accountname NOT LIKE '%TDU%'";

    $result = mysqli_query($conn, $sql);
    
    if (!$result) {
        die("Query failed: " . mysqli_error($conn));
    }

    $accountContacts = array();

    // Fetch the result set and populate the dictionary
    while ($row = mysqli_fetch_assoc($result)) {
        $accountname = $row['accountname'];
        $contactname = $row['contactname'];

        // If the accountname already exists in the array, append the contactname
        if (isset($accountContacts[$accountname])) {
            $accountContacts[$accountname][] = $contactname;
        } else {
            // Otherwise, create a new array for this accountname
            $accountContacts[$accountname] = array($contactname);
        }
    }

    return $accountContacts;
}

function getAccountContacts2($conn) {
    $sql = "SELECT DISTINCT va.organization_name AS accountname, vcd.name AS contactname 
            FROM tdu_contacts vcd 
            LEFT JOIN tdu_organisation va ON vcd.organizationid = va.organizationid
            WHERE va.active = 1 AND va.organization_name NOT LIKE '%TDU%'";

    $result = mysqli_query($conn, $sql);
    
    if (!$result) {
        die("Query failed: " . mysqli_error($conn));
    }

    $accountContacts = array();

    // Fetch the result set and populate the dictionary
    while ($row = mysqli_fetch_assoc($result)) {
        $accountname = $row['accountname'];
        $contactname = $row['contactname'];

        // If the accountname already exists in the array, append the contactname
        if (isset($accountContacts[$accountname])) {
            $accountContacts[$accountname][] = $contactname;
        } else {
            // Otherwise, create a new array for this accountname
            $accountContacts[$accountname] = array($contactname);
        }
    }

    return $accountContacts;
}

function citiesAU() {
    return [
        "Sydney", "Melbourne", "Brisbane", "Perth", "Adelaide",
        "Gold Coast", "Canberra", "Newcastle", "Wollongong",
        "Central Coast", "Sunshine Coast", "Cairns", "Darwin",
        "Hobart", "Geelong", "Townsville", "Mackay", "Ballarat",
        "Bendigo", "Launceston", "Alice Springs", "Rockhampton",
        "Coffs Harbour", "Tamworth", "Wagga Wagga", "Hervey Bay",
        "Mount Gambier", "Gladstone", "Bundaberg", "Mildura",
        "Kalgoorlie", "Coffs Harbour", "Maryborough", "Port Macquarie",
        "Nowra", "Lismore", "Goulburn", "Shepparton", "Bairnsdale",
        "Katherine", "Ayr", "Mackay", "Innisfail", "Gympie",
        "Burnie", "Launceston", "Devonport", "Gisborne", "Bendigo",
        "Horsham", "Port Lincoln", "Armidale", "Alice Springs", 
        "Broome", "Coffs Harbour", "Orange", "Tamworth", "Batemans Bay"
    ];
}

function citiesNZ() {
    return [
        "Auckland", "Wellington", "Christchurch", "Hamilton", "Tauranga",
        "Napier", "Dunedin", "Palmerston North", "Rotorua", "New Plymouth",
        "Whangarei", "Nelson", "Invercargill", "Gisborne", "Ashburton",
        "Timaru", "Masterton", "Richmond", "Pukekohe", "Taupo",
        "Blenheim", "Mount Maunganui", "Tokoroa", "Hastings", "Oamaru",
        "Upper Hutt", "Lower Hutt", "Porirua", "East Tamaki", "Waiuku",
        "Kawerau", "Te Awamutu", "Opotiki", "Warkworth", "Dargaville",
        "Feilding", "Kaitaia", "Papakura", "Thames", "Whakatane",
        "Matamata", "Clutha", "Motueka", "Hutt City", "Kaikoura",
        "Otorohanga", "Tawa", "Greymouth", "Wanganui", "Balclutha",
        "Ashburton", "Pahiatua", "Hawera", "Lumsden", "Hokitika",
        "Akaroa", "Wairoa", "Westport", "Raglan", "Hastings",
        "Carterton", "Feilding", "Kawerau", "Blenheim", "Morrinsville"
    ];
}

function getQuoteDictionary($conn) {
    $sql = "
        SELECT q1.quote_no, q1.quoteid
        FROM vtiger_quotes q1
        WHERE q1.quoteid = (
            SELECT MIN(sub.quoteid) 
            FROM vtiger_quotes sub 
            WHERE sub.quote_no = q1.quote_no
        );
    ";

    $result = $conn->query($sql);
    $quoteDictionary = [];

    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $quoteDictionary[$row['quote_no']] = $row['quoteid'];
        }
    }

    return $quoteDictionary;
}

function downloadType() {
    $downloadTypes = [
        'adjust' => 'New Adjustment Made',
        'quote' => 'Download Quote',
        'so' => 'Download Sales Order',
        'initial' => 'Generate Initial Invoice',
        'reset-initial' => 'Reset Initial Invoice',
        'download-initial' => 'Download Initial Invoice',
        'final' => 'Generate Final Invoice',
        'download-final' => 'Download Final Invoice'
    ];

    return $downloadTypes;
}

function getEmailsByRole($role) {
    // Define the role-to-emails mapping
    $roleToEmails = [
        'admin' => [
            'sales@turtledownunder.com.au',
            'groupsales@turtledownunder.com.au',
            'operations@turtledownunder.com.au',
            'groups@turtledownunder.com.au',
            'accounts@turtledownunder.com.au',
            'contracts@turtledownunder.com.au',
            'groups@turtledownunder.co.nz',
            'operations@turtledownunder.co.nz',
            'accounts@turtledownunder.co.nz',
            'contracts@turtledownunder.co.nz',
            'asia@turtledownunder.com.au',
        ],
        'sales' => [
            'sales@turtledownunder.com.au',
            'groupsales@turtledownunder.com.au',
            'asia@turtledownunder.com.au',
        ],
        'group' => [
            'sales@turtledownunder.com.au',
            'operations@turtledownunder.com.au',
            'operations@turtledownunder.co.nz',
            'asia@turtledownunder.com.au',
            'groupsales@turtledownunder.com.au',
            'groups@turtledownunder.com.au',
            'groups@turtledownunder.co.nz',
        ],
        'account' => [
            'operations@turtledownunder.com.au',
            'groups@turtledownunder.com.au',
            'accounts@turtledownunder.com.au',
            'contracts@turtledownunder.com.au',
            'groups@turtledownunder.co.nz',
            'operations@turtledownunder.co.nz',
            'accounts@turtledownunder.co.nz',
            'contracts@turtledownunder.co.nz',
        ],
        'qa' => [
            'sales@turtledownunder.com.au',
            'groupsales@turtledownunder.com.au',
            'operations@turtledownunder.com.au',
            'groups@turtledownunder.com.au',
            'groups@turtledownunder.co.nz',
            'operations@turtledownunder.co.nz',
            'asia@turtledownunder.com.au',
        ],
    ];

    // Return emails for the specified role, or an empty array if the role doesn't exist
    return $roleToEmails[$role] ?? [];
}

function getEmailId($mail) {
    $email_index = array(
        "sales@turtledownunder.com.au" => 1,
        "groupsales@turtledownunder.com.au" => 2,
        "operations@turtledownunder.com.au" => 3,
        "groups@turtledownunder.com.au" => 4,
        "accounts@turtledownunder.com.au" => 5,
        "contracts@turtledownunder.com.au" => 6,
        "groups@turtledownunder.co.nz" => 7,
        "operations@turtledownunder.co.nz" => 8,
        "accounts@turtledownunder.co.nz" => 9,
        "contracts@turtledownunder.co.nz" => 10,
        "asia@turtledownunder.com.au" => 11
    );

    if (array_key_exists($mail, $email_index)) {
        return $email_index[$mail];
    } else {
        return 0;
    }
}

function fetchOrganizationNames80($conn) {
    $sql = "SELECT DISTINCT organization_name AS accountname FROM tdu_organisation UNION SELECT DISTINCT accountname FROM vtiger_account_custom";
    $result = mysqli_query($conn, $sql);

    $organization_names = array();
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $organization_names[] = $row['accountname'];
        }
    }
    return $organization_names;
}

function fetchTDUNames80($conn) {
    $sql = "SELECT DISTINCT quote_no FROM vtiger_quotes";
    $result = mysqli_query($conn, $sql);

    $organization_names = array();
    if ($result) {
        while ($row = mysqli_fetch_assoc($result)) {
            $organization_names[] = $row['quote_no'];
        }
    }
    return $organization_names;
}

function getTaskCategories($conn) {
    $sql = "SELECT DISTINCT category FROM tdu_task_category;";
    
    $result = $conn->query($sql);
    $taskCategories = array();

    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $taskCategories[] = $row['category'];
        }
    }
    
    return $taskCategories;
}

function getSqlTableToDeleteQuoteProducts($quoteid, $sequence_no) {
    return [
        "DELETE FROM tdu_flight_details WHERE quoteid='$quoteid' AND sequence_no='$sequence_no'",
        "DELETE FROM tdu_product_alt_vendors_in_quote WHERE quoteid='$quoteid' AND sequence_no='$sequence_no'",
        "DELETE FROM vtiger_coach_info WHERE quoteid='$quoteid' AND sequence_no='$sequence_no'",
        "DELETE FROM vtiger_display_order WHERE id='$quoteid' AND sequence_no='$sequence_no'",
        "DELETE FROM vtiger_inventoryproductrel WHERE id='$quoteid' AND sequence_no='$sequence_no'",
        "DELETE FROM vtiger_itinerary WHERE quoteid='$quoteid' AND sequence_no='$sequence_no'",
        "DELETE FROM vtiger_itinerary_new_event WHERE quoteid='$quoteid' AND sequence_no='$sequence_no'",
        "DELETE FROM vtiger_products_saleprice WHERE quoteid='$quoteid' AND sequence_no='$sequence_no'",
        "DELETE FROM vtiger_product_payment WHERE quoteid='$quoteid' AND sequence_no='$sequence_no'",
        "DELETE FROM vtiger_transfer_index WHERE quoteid='$quoteid' AND sequence_no='$sequence_no'"
    ];
}

function recordDeleteConfirmedProductHistory($conn, $quoteid, $sequence_no) {
    $details = '';

    $product_sql = "SELECT vi.day, vi.comment, vi.cf_928, vp.productname
                    FROM vtiger_inventoryproductrel vi
                    LEFT JOIN (
                        SELECT productid, productname, unit_price, 'true' AS customised, vendor_id, '' AS description FROM vtiger_products_custom
                        UNION ALL
                        SELECT productid, productName AS productname, unitPrice AS unit_price, '' AS customised, vendorid AS vendor_id, description FROM tdu_products
                    ) vp ON vi.productid = vp.productid
                    WHERE vi.id='$quoteid' AND vi.sequence_no='$sequence_no';";
    $product_result = mysqli_query($conn, $product_sql);
    if ($product_result && mysqli_num_rows($product_result) > 0) {
        if ($product_row = mysqli_fetch_assoc($product_result)) {
            $day = $product_row['day'];
            $comment = $product_row['comment'];
            $cf_928 = $product_row['cf_928'];
            $productname = $product_row['productname'];
            $details .= "Day $day: $productname ($cf_928)\n$comment\n\n";
        }
    }

    $price_sql = "SELECT * FROM vtiger_products_saleprice WHERE quoteid='$quoteid' AND sequence_no='$sequence_no';";
    $price_result = mysqli_query($conn, $price_sql);
    if ($price_result && mysqli_num_rows($price_result) > 0) {
        while ($price_row = mysqli_fetch_assoc($price_result)) {
            if ($cf_928 == 'Hotel') {
                $sale_price_single = $price_row['sale_price_single'];
                $sale_price_double = $price_row['sale_price_double'];
                $sale_price_triple = $price_row['sale_price_triple'];
                $details.= "Hotel Price (Single/Double/Triple): $$sale_price_single / $$sale_price_double / $$sale_price_triple\n";
            } else if ($cf_928 == 'Transfers') {
                $sale_price = $price_row['sale_price'];
                if ($price_row['subquoteid'] == 0) {
                    if (mysqli_num_rows($price_result) > 1) {
                        continue;
                    }
                    $details.= "Transfer Price: $$sale_price\n";
                } else {
                    $pax_min = $price_row['pax_min'];
                    $pax_max = $price_row['pax_max'];
                    $details.= "Transfer Price (Pax $pax_min - $pax_max): $$sale_price\n";
                }
            } else {
                $sale_price = $price_row['sale_price'];
                $sale_price_child = $price_row['sale_price_child'];
                $sale_price_infant = $price_row['sale_price_infant'];
                $details.= "Product Price (Single/Child/Infant): $$sale_price / $$sale_price_child / $$sale_price_infant\n";
            }
        }
    }

    $itinerary_sql = "SELECT t.quoteid, t.sequence_no, t.meta_key, t.meta_value
                    FROM vtiger_itinerary t
                    JOIN (
                        SELECT quoteid, sequence_no, meta_key, MAX(created_at) AS max_created_at
                        FROM vtiger_itinerary
                        WHERE quoteid = '$quoteid' AND sequence_no = '$sequence_no'
                        GROUP BY quoteid, sequence_no, meta_key
                    ) latest 
                    ON t.quoteid = latest.quoteid 
                    AND t.sequence_no = latest.sequence_no
                    AND t.meta_key = latest.meta_key
                    AND t.created_at = latest.max_created_at;";
    $itinerary_result = mysqli_query($conn, $itinerary_sql);
    if ($itinerary_result && mysqli_num_rows($itinerary_result) > 0) {
        while ($itinerary_row = mysqli_fetch_assoc($itinerary_result)) {
            $meta_key = ucwords(str_replace('_', ' ', $itinerary_row['meta_key']));
            $meta_value = $itinerary_row['meta_value'] ?? '';
            if (trim($meta_value) == '' || trim($meta_value) == '0') {
                continue;
            }
            $details .= "$meta_key: $meta_value\n";
        }
    }

    // SQL query
    $details = mysqli_real_escape_string($conn, $details);
    $sql = "INSERT INTO vtiger_products_delete_history (quoteid, sequence_no, details) VALUES ('$quoteid', '$sequence_no', '$details');";
    mysqli_query($conn, $sql);
}

function fetchAllowedQuoteTabs($conn, $role) {
    // Get the allowed quote stages for the given role
    $sql = "SELECT quote_tabs FROM tdu_user_roles WHERE user_role='$role';";
    $result = mysqli_query($conn, $sql);
    if ($result && mysqli_num_rows($result) > 0) {
        $row = mysqli_fetch_assoc($result);
        $quote_tabs = $row['quote_tabs'];
    } else {
        return array();
    }

    return $quote_tabs;
}

function createTaskWithMessage($conn, $quotestage, $quote_no, $priority = 'Normal', $deadline = null, $task_category = 'quotestage', $productname_global = null) {
    // Sanitize inputs
    $quotestage = mysqli_real_escape_string($conn, $quotestage);
    $quote_no = mysqli_real_escape_string($conn, $quote_no);
    $priority = mysqli_real_escape_string($conn, $priority);
    $deadline_sql = $deadline ? "'" . mysqli_real_escape_string($conn, $deadline) . "'" : "NULL";

    // Fetch rule by quote stage
    $rule_sql = "SELECT assigned_to_region AS assigned_category, assigned_to_user, message 
                FROM tdu_tasks_management_define_rule 
                WHERE quotestage = '$quotestage' AND category = '$task_category' AND active = 1;";
    $rule_result = mysqli_query($conn, $rule_sql);
    if (!$rule_result || mysqli_num_rows($rule_result) == 0) {
        return [];
    }
    $rule = mysqli_fetch_assoc($rule_result);
    $subject = $rule['message'];
    $category = mysqli_real_escape_string($conn, $rule['assigned_category']);
    $assigned_to_user_rule = $rule['assigned_to_user'] ?? '';

    if($task_category == 'quotestage' || ($task_category == 'custom' && $quotestage == 'Quote Payment Due')) {
        // Fetch assignment info
        $info_sql = "SELECT vqinfo.assigned_to_region, 
                            vqinfo.assigned_to_sales_agent, 
                            vqinfo.assigned_to_external_sales_agent, 
                            vqinfo.assigned_to_ops, 
                            vqinfo.assigned_to_qa 
                    FROM vtiger_quotes vq
                    LEFT JOIN vtiger_quotes_info vqinfo ON vq.quoteid = vqinfo.quoteid AND vqinfo.quotetype = 'group'
                    WHERE quote_no = '$quote_no'";
        $info_result = mysqli_query($conn, $info_sql);
        if (!$info_result || mysqli_num_rows($info_result) == 0) {
            return [];
        }
        $info = mysqli_fetch_assoc($info_result);
        $assigned_region = mysqli_real_escape_string($conn, $info['assigned_to_region']);

        $inserted_ids = [];
        if (strtolower($category) === 'sales') {
            // One task for each sales-related agent
            $sales_users = array_unique([
                $info['assigned_to_sales_agent'] ?? '', $info['assigned_to_external_sales_agent'] ?? ''
            ]);

            foreach ($sales_users as $user) {
                $assigned_user = mysqli_real_escape_string($conn, $user);
                $insert_sql = "INSERT INTO tdu_tasks_management 
                                (subject, category, quote_no, priority, deadline, assigned_to_region, assigned_to_user, created_by) 
                                VALUES 
                                ('$subject', '$category', '$quote_no', '$priority', $deadline_sql, '$assigned_region', '$assigned_user', 'SYSTEM');";
                if (!mysqli_query($conn, $insert_sql)) {
                    die("Task Insert Error (Sales): " . mysqli_error($conn));
                }
                $inserted_ids[] = mysqli_insert_id($conn);
            }
        } else {
            $assigned_user = '';
            if (!empty($assigned_to_user_rule)) {
                $assigned_user = $assigned_to_user_rule;
            } else if (strtolower($category) === 'operation') {
                $assigned_user = $info['assigned_to_ops'] ?? '';
            } elseif (strtolower($category) === 'qa') {
                $assigned_user = $info['assigned_to_qa'] ?? '';
            }

            $assigned_user = mysqli_real_escape_string($conn, $assigned_user);
            $insert_sql = "INSERT INTO tdu_tasks_management 
                            (subject, category, quote_no, priority, deadline, assigned_to_region, assigned_to_user, created_by) 
                            VALUES 
                            ('$subject', '$category', '$quote_no', '$priority', $deadline_sql, '$assigned_region', '$assigned_user', 'SYSTEM');";
            if (!mysqli_query($conn, $insert_sql)) {
                die("Task Insert Error (Non-Sales): " . mysqli_error($conn));
            }
            $inserted_ids[] = mysqli_insert_id($conn);
        }

        return $inserted_ids;
    } else if ($task_category == 'custom') {
        if ($quotestage == 'Customers On Ground') {
            $sql_quote_products = " SELECT vp.productname 
                                    FROM vtiger_inventoryproductrel vi
                                    LEFT JOIN vtiger_quotes vq ON vi.id=vq.quoteid
                                    LEFT JOIN (
                                        SELECT productid, productname FROM vtiger_products_custom
                                        UNION ALL
                                        SELECT productid, productName AS productname FROM tdu_products
                                    ) vp ON vi.productid = vp.productid
                                    WHERE vq.quote_no = '$quote_no'";
            $result_quote_products = mysqli_query($conn, $sql_quote_products);
            while ($row = mysqli_fetch_assoc($result_quote_products)) {
                $productname = mysqli_real_escape_string($conn, $row['productname']);

                // Insert into tasks management
                $insert_sql5 = "INSERT INTO tdu_tasks_management (subject, quote_no, assigned_to_user, category, deadline, created_by) 
                                VALUES ('Client Service: $productname', '$quote_no', '$assigned_to_user_rule', '$category', CURDATE(), 'SYSTEM')";
                mysqli_query($conn, $insert_sql5) or die(mysqli_error($conn));

                $task_id = mysqli_insert_id($conn);

                // Insert the default message
                $message = "$quote_no Customers On Ground, Please action";
                $insert_msg_sql = "INSERT INTO tdu_tasks_management_msg (task_id, message, created_by) 
                                    VALUES ('$task_id', '$message', 'SYSTEM')";
                mysqli_query($conn, $insert_msg_sql) or die(mysqli_error($conn));
            }
        } else if ($quotestage == 'Vendor Payment Due') {
            $productname = $productname_global;
            // Check if quote_no already exists in tdu_tasks_management
            $check_sql = "SELECT 1 FROM tdu_tasks_management WHERE subject = 'Payment Due soon: $productname' AND quote_no = '$quote_no' AND category = '$category' LIMIT 1";
            $check_result = mysqli_query($conn, $check_sql);

            if (!$check_result) {
                die("Error checking tasks: " . mysqli_error($conn));
            }

            if (mysqli_num_rows($check_result) == 0) {
                // Insert new task if quote_no does not exist
                $insert_sql = "INSERT INTO tdu_tasks_management (subject, quote_no, assigned_to_user, category, deadline, created_by)  
                            VALUES ('Payment Due soon: $productname', '$quote_no', '$assigned_to_user_rule', '$category', '$deadline', 'SYSTEM')";
                if (!mysqli_query($conn, $insert_sql)) {
                    die("Error inserting task: " . mysqli_error($conn));
                }
            }
        }
    }
    
    return [];
}

?>
