<?php
// qbo_ajax_fetch_chunked_report.php
// ADVANCED & BULLETPROOF DIAGNOSTIC VERSION:
// Merges aggressive logging and error handling with a switchable "Force Fetch" mode
// to bypass early cutoffs for deep debugging of API pagination issues.

// --- Step 1: Set Up Aggressive Error Handling & Logging ---
ini_set('display_errors', 0); // Do not display errors to the user, we will control the output.
ini_set('log_errors', 1);     // Ensure errors are logged.
// Make sure error_log is configured in your php.ini to a writable file.

// This function will be called on script shutdown, especially after a fatal error.
register_shutdown_function('handleFatalError');
function handleFatalError() {
    $error = error_get_last();
    // Check if it was a fatal error (E_ERROR, E_PARSE, etc.)
    if ($error !== null && in_array($error['type'], [E_ERROR, E_CORE_ERROR, E_COMPILE_ERROR, E_USER_ERROR, E_PARSE])) {
        // We have a fatal error. Log it and try to send a clean JSON response.
        $errorMessage = "FATAL ERROR: {$error['message']} in {$error['file']} on line {$error['line']}";
        error_log($errorMessage);

        // If headers haven't been sent yet, send a JSON error.
        if (!headers_sent()) {
            header('Content-Type: application/json');
            http_response_code(500); // Internal Server Error
            echo json_encode([
                'success' => false,
                'error' => 'A fatal server error occurred. Please check the PHP error log.',
                'debug' => ['fatal_error' => $errorMessage]
            ]);
        }
    }
}

// This function will catch any uncaught exceptions.
set_exception_handler('handleUncaughtException');
function handleUncaughtException($exception) {
    $errorMessage = "UNCAUGHT EXCEPTION: {$exception->getMessage()} in {$exception->getFile()} on line {$exception->getLine()}";
    error_log($errorMessage);

    if (!headers_sent()) {
        header('Content-Type: application/json');
        http_response_code(500); // Internal Server Error
        echo json_encode([
            'success' => false,
            'error' => 'An uncaught exception occurred. Please check the PHP error log.',
            'debug' => ['exception' => $errorMessage]
        ]);
    }
    // Prevent the default PHP fatal error.
    exit();
}

// --- Main Script Execution with Timing and Memory Measurement ---
$startTime = microtime(true);
$positionForLog = $_GET['start_position'] ?? 1;
error_log("--- [CHUNK START] --- Position: " . $positionForLog);

header('Content-Type: application/json');
require_once __DIR__ . '/qbo_bootstrap.php';

if (!$qboUtil) {
    $errorMessage = "Bootstrap failed. qboUtil object not available.";
    error_log($errorMessage);
    echo json_encode(['success' => false, 'error' => $errorMessage]);
    exit;
}

// --- Step 2: Get Parameters & Set Up Diagnostic Controls ---
$startDate = $_GET['start_date'] ?? null;
$endDate = $_GET['end_date'] ?? null;
$selectedProductServices = $_GET['product_service'] ?? [];
$startPosition = (int)($_GET['start_position'] ?? 1);
$maxResultsPerPage = 700; // Using your more aggressive chunk size.
$maxRetries = 2; // Number of retries for QBO API calls

// --- ========================================================== ---
// --- NEW ADVANCED DEBUGGING CONTROLS                            ---
// --- ========================================================== ---

/**
 * @const FORCE_MINIMUM_PAGES_MODE
 * Set to `true` to force the script to fetch at least MINIMUM_PAGES_TO_FETCH pages,
 * ignoring any pages that return fewer than $maxResultsPerPage.
 * Set to `false` to use the standard, robust logic (stop when 0 records are returned).
 */
define('FORCE_MINIMUM_PAGES_MODE', true); // <-- SWITCH HERE (true/false)

/**
 * @const MINIMUM_PAGES_TO_FETCH
 * The number of pages to force when the above mode is active.
 * For example, with $maxResultsPerPage=700, setting this to 4 will attempt to fetch from
 * positions 1, 701, 1401, and 2101.
 */
define('MINIMUM_PAGES_TO_FETCH', 5);

// --- ========================================================== ---

if (!$startDate || !$endDate) {
    $errorMessage = "Missing required date parameters.";
    error_log($errorMessage);
    echo json_encode(['success' => false, 'error' => $errorMessage]);
    exit;
}

// Give ourselves a generous timeout for each chunk. 120 seconds should be plenty.
set_time_limit(120);

$dataService = $qboUtil->getDataService();

// --- Step 3: Fetch ONE BATCH of Invoices (with robust error handling) ---
$invoiceQuery = "SELECT * FROM Invoice WHERE TxnDate >= '{$startDate}' AND TxnDate <= '{$endDate}' STARTPOSITION {$startPosition} MAXRESULTS {$maxResultsPerPage}";
$invoicesBatch = null;
$qboError = null;

error_log("[CHUNK INFO] Executing Invoice Query: " . $invoiceQuery);

for ($attempt = 1; $attempt <= $maxRetries; $attempt++) {
    try {
        $invoicesBatch = $dataService->Query($invoiceQuery);
        $lastError = $dataService->getLastError();
        if ($lastError) {
            throw new Exception("QBO API Error After Query: " . $lastError->getResponseBody(), $lastError->getHttpStatusCode());
        }
        $qboError = null;
        break;
    } catch (\Exception $e) {
        $qboError = $e;
        $httpCode = $e->getCode();
        $message = $e->getMessage();
        error_log("[CHUNK WARNING] QBO API Call Attempt #{$attempt} failed. Code: {$httpCode}. Message: {$message}");

        if ($httpCode == 429 || strpos($message, 'ThrottleException') !== false) {
            if ($attempt < $maxRetries) {
                error_log("[CHUNK INFO] Throttling detected. Waiting 30 seconds before retry...");
                sleep(30);
                continue;
            } else {
                error_log("[CHUNK ERROR] Max retries reached for throttling. Aborting.");
                break;
            }
        }
        break;
    }
}

if ($qboError !== null) {
    $finalErrorMessage = "Failed to fetch invoices from QBO after {$maxRetries} attempts. Last error: " . $qboError->getMessage();
    error_log($finalErrorMessage);
    echo json_encode(['success' => false, 'error' => $finalErrorMessage]);
    exit;
}

// Defensive fix for null SDK response.
if ($invoicesBatch === null) {
    $invoicesBatch = [];
}
error_log("[CHUNK INFO] Successfully fetched " . count($invoicesBatch) . " invoices from QBO.");


// --- Step 4: Batch Fetch Related Data (Customers, Items, Classes) ---
// This part is only executed if we have invoices to process.
$customerMap = [];
$itemMap = [];
$classMap = [];
if (!empty($invoicesBatch)) {
    function queryInClause($dataService, $entity, $ids, $field = 'Id')
    {
        $unique_ids = array_unique(array_filter($ids));
        if (empty($unique_ids)) return [];

        $results = [];
        // Max 1000 characters for the IN clause values. We chunk it to be safe.
        $idChunks = array_chunk($unique_ids, 50); // Chunking by 50 IDs is very safe.
        foreach($idChunks as $chunk) {
            $idString = "('" . implode("','", $chunk) . "')";
            $query = "SELECT * FROM {$entity} WHERE {$field} IN {$idString}";
            
            error_log("[CHUNK INFO] Fetching related data for {$entity} with query: {$query}");
            $entities = $dataService->Query($query);

            if ($dataService->getLastError()) {
                error_log("[CHUNK WARNING] QBO API Error while fetching {$entity}: " . $dataService->getLastError()->getResponseBody());
                // Continue to the next chunk instead of failing the whole operation.
                continue;
            }

            if ($entities) {
                foreach ($entities as $entityObject) {
                    $results[$entityObject->Id] = $entityObject;
                }
            }
        }

        error_log("[CHUNK INFO] Fetched " . count($results) . " unique {$entity} records.");
        return $results;
    }

    $customerIds = array_column($invoicesBatch, 'CustomerRef');
    $itemIds = [];
    $classIds = [];
    foreach ($invoicesBatch as $txn) {
        if (empty($txn->Line)) continue;
        foreach ($txn->Line as $line) {
            if (isset($line->DetailType) && $line->DetailType === 'SalesItemLineDetail') {
                if (!empty($line->SalesItemLineDetail->ItemRef)) $itemIds[] = $line->SalesItemLineDetail->ItemRef;
                if (!empty($line->SalesItemLineDetail->ClassRef)) $classIds[] = $line->SalesItemLineDetail->ClassRef;
            }
        }
    }

    $customerMap = queryInClause($dataService, 'Customer', $customerIds);
    $itemMap = queryInClause($dataService, 'Item', $itemIds);
    $classMap = queryInClause($dataService, 'Class', $classIds);
}

// --- Step 5: Process and Structure the Final Data ---
$reportData = [];
$lowercaseSelectedServices = array_map('strtolower', $selectedProductServices);

foreach ($invoicesBatch as $txn) {
    $customerId = $txn->CustomerRef;
    if (!isset($customerMap[$customerId])) continue;

    foreach ($txn->Line as $line) {
        if (isset($line->DetailType) && $line->DetailType === 'SalesItemLineDetail' && !empty($line->SalesItemLineDetail->ItemRef)) {
            $itemId = $line->SalesItemLineDetail->ItemRef;
            $itemName = isset($itemMap[$itemId]) ? $itemMap[$itemId]->Name : 'Unknown Item';

            if (!empty($lowercaseSelectedServices) && !in_array(strtolower($itemName), $lowercaseSelectedServices)) {
                continue;
            }

            if (!isset($reportData[$customerId])) {
                $reportData[$customerId] = [
                    'customerInfo' => ['id' => $customerId, 'name' => $customerMap[$customerId]->DisplayName],
                    'total' => 0.00,
                    'quantity' => 0,
                    'details' => []
                ];
            }

            $classId = $line->SalesItemLineDetail->ClassRef;
            $className = isset($classMap[$classId]) ? $classMap[$classId]->Name : '';

            $reportData[$customerId]['details'][] = [
                'date'          => $txn->TxnDate,
                'type'          => 'Invoice',
                'num'           => $txn->DocNumber ?: 'ID: ' . $txn->Id,
                'class'         => $className,
                'product_service' => $itemName,
                'qty'           => (int)($line->SalesItemLineDetail->Qty ?? 0),
                'amount'        => (float)($line->Amount ?? 0),
                'sales_price'   => (float)($line->SalesItemLineDetail->UnitPrice ?? 0)
            ];
            $reportData[$customerId]['total'] += (float)($line->Amount ?? 0);
            $reportData[$customerId]['quantity'] += (int)($line->SalesItemLineDetail->Qty ?? 0);
        }
    }
}


// --- Step 6: The INTELLIGENT Completion Logic ---
$nextPosition = $startPosition + count($invoicesBatch);

// Calculate the current page number for our logic.
$currentPageNumber = floor(($startPosition - 1) / $maxResultsPerPage) + 1;
$completionReason = '';
$isComplete = false;

if (FORCE_MINIMUM_PAGES_MODE === true && $currentPageNumber < MINIMUM_PAGES_TO_FETCH) {
    // --- FORCE MODE IS ACTIVE ---
    // We are in "force mode" and haven't reached our target page count yet.
    // The process is NEVER considered complete in this state, even if we received 0 records.
    // We set nextPosition to keep the loop going on the frontend.
    $isComplete = false;
    if (count($invoicesBatch) === 0) {
        // Manually jump to the start of the next page to prevent an infinite loop on a "hole".
        $nextPosition = ($currentPageNumber * $maxResultsPerPage) + 1;
    }
    $completionReason = "Forced Fetch Mode: Continuing to next page (currently on page {$currentPageNumber} of " . MINIMUM_PAGES_TO_FETCH . ")";

} else {
    // --- STANDARD MODE or Force Mode is finished ---
    // We use the most robust logic: the process is only complete when QBO returns exactly zero records.
    // This is more reliable than checking `count < maxResults`.
    $isComplete = count($invoicesBatch) === 0;
    if ($isComplete) {
       $completionReason = "Standard Mode: Received 0 records, signaling the end of the data set.";
    } else if (FORCE_MINIMUM_PAGES_MODE === true) {
       $completionReason = "Forced Fetch Mode: Minimum page count reached. Now continuing until 0 records are returned.";
    } else {
       $completionReason = "Standard Mode: More data may be available.";
    }
}

// --- Step 7: Return the Final Response with Rich Debug Info ---
$execTime = round(microtime(true) - $startTime, 3);
$peakMemoryMb = round(memory_get_peak_usage(true) / 1024 / 1024, 2);

error_log("[CHUNK COMPLETE] Position {$startPosition} finished. Reason: '{$completionReason}'. Time: {$execTime}s. Memory: {$peakMemoryMb}MB.");

echo json_encode([
    'success'      => true,
    'data'         => array_values($reportData), // Re-index the array for clean JSON
    'nextPosition' => $nextPosition,
    'isComplete'   => $isComplete,
    'debug'        => [
        'chunk_start_position' => $startPosition,
        'requested_chunk_size' => $maxResultsPerPage,
        'returned_chunk_size'  => count($invoicesBatch),
        'execution_time_sec'   => $execTime,
        'peak_memory_mb'       => $peakMemoryMb,
        'logic_mode'           => FORCE_MINIMUM_PAGES_MODE ? 'FORCE_FETCH' : 'STANDARD',
        'current_page_number'  => $currentPageNumber,
        'completion_reason'    => $completionReason
    ]
]);