<?php
// qbo_ajax_fetch_full_report.php
// CORRECTED VERSION: Implements robust pagination directly in the query string.

header('Content-Type: application/json');
require_once __DIR__ . '/qbo_bootstrap.php';

if (!$qboUtil) {
    echo json_encode(['success' => false, 'error' => 'Bootstrap failed']);
    exit;
}

// --- Get Parameters from the AJAX Request ---
$startDate = $_GET['start_date'] ?? null;
$endDate = $_GET['end_date'] ?? null;
$selectedProductServices = $_GET['product_service'] ?? [];

if (!$startDate || !$endDate) {
    echo json_encode(['success' => false, 'error' => 'Missing required date parameters']);
    exit;
}

try {
    $dataService = $qboUtil->getDataService();
    
// --- Step 1: Fetch All Invoices in Date Range (with MEMORY CHECK and other defenses) ---
    $allInvoices = [];
    $startPosition = 1;
    $maxResultsPerPage = 1000;
    $maxRetries = 3;
    $memoryWarning = null; // This will hold our warning message if needed.

    // --- MEMORY CHECK (part 1: Setup) ---
    function convertToBytes(string $from): ?int {
        $from = strtolower(trim($from));
        $val = (int) $from;
        switch (substr($from, -1)) {
            case 'g': $val *= 1024; // Fall-through
            case 'm': $val *= 1024; // Fall-through
            case 'k': $val *= 1024;
        }
        return $val > 0 ? $val : null;
    }

    $memoryLimitString = ini_get('memory_limit');
    $memoryLimitBytes = convertToBytes($memoryLimitString);
    // We'll use 90% as our safety buffer.
    $safetyThreshold = $memoryLimitBytes ? $memoryLimitBytes * 0.90 : null; 
    // ----------------------------------------

    // --- TIMEOUT DEFENSE SETUP ---
    $time_limit = ini_get('max_execution_time');
    if ($time_limit < 30 && $time_limit != 0) { $time_limit = 30; } 
    else if ($time_limit == 0) { $time_limit = 300; }
    // -----------------------------

    do {
        set_time_limit($time_limit); // Timeout Defense

        $invoicesBatch = null;
        $retryCount = 0;

        // --- Rate Limit Defense Loop ---
        while ($retryCount < $maxRetries) {
            try {
                $invoiceQuery = "SELECT * FROM Invoice WHERE TxnDate >= '{$startDate}' AND TxnDate <= '{$endDate}' STARTPOSITION {$startPosition} MAXRESULTS {$maxResultsPerPage}";
                $invoicesBatch = $dataService->Query($invoiceQuery);
                break;
            } catch (\QuickBooksOnline\API\Exception\IdsException $e) {
                if (strpos($e->getMessage(), 'ThrottleException') !== false || strpos($e->getMessage(), '429') !== false) {
                    $retryCount++;
                    if ($retryCount >= $maxRetries) { throw $e; }
                    sleep(60);
                } else {
                    throw $e;
                }
            }
        }
        // --- End Rate Limit Defense ---

        if ($dataService->getLastError()) {
            throw new Exception("QBO API Error after retries: " . $dataService->getLastError()->getResponseBody());
        }

        if (!empty($invoicesBatch)) {
            $allInvoices = array_merge($allInvoices, $invoicesBatch);
            $startPosition += count($invoicesBatch);
        }

        // --- MEMORY CHECK (part 2: Execution) ---
        if ($safetyThreshold) {
            $currentMemoryUsage = memory_get_usage(true);
            if ($currentMemoryUsage >= $safetyThreshold) {
                // We're approaching the limit! Prepare a warning and stop fetching more data.
                $memoryWarning = "Memory limit approaching. The report was stopped to prevent a crash. Fetched " . count($allInvoices) . " records before stopping. ";
                // Break the main do...while loop safely.
                break; 
            }
        }
        // ----------------------------------------

    } while (!empty($invoicesBatch) && count($invoicesBatch) === $maxResultsPerPage);
    
    if (empty($allInvoices)) {
        echo json_encode(['success' => true, 'data' => []]);
        exit;
    }

    // --- Step 2: Collect all Unique IDs (No changes needed here) ---
    $customerIds = [];
    $itemIds = [];
    $classIds = [];
    foreach ($allInvoices as $txn) {
        if (!empty($txn->CustomerRef)) {
            $customerIds[] = $txn->CustomerRef;
        }
        if (empty($txn->Line)) continue;
        foreach ($txn->Line as $line) {
            if (isset($line->DetailType) && $line->DetailType === 'SalesItemLineDetail') {
                if (!empty($line->SalesItemLineDetail->ItemRef)) $itemIds[] = $line->SalesItemLineDetail->ItemRef;
                if (!empty($line->SalesItemLineDetail->ClassRef)) $classIds[] = $line->SalesItemLineDetail->ClassRef;
            }
        }
    }
    
    // --- Step 3: Batch Fetch Names for all collected IDs (No changes needed here) ---
    function queryInClause($dataService, $entity, $ids, $field = 'Id') {
        $unique_ids = array_unique(array_filter($ids));
        if (empty($unique_ids)) return [];
        
        $results = [];
        $chunks = array_chunk($unique_ids, 1000);
        
        foreach ($chunks as $chunk) {
            $idString = "('" . implode("','", $chunk) . "')";
            $query = "SELECT * FROM {$entity} WHERE {$field} IN {$idString}";
            $entities = $dataService->Query($query);
            if ($entities) {
                foreach ($entities as $entity) {
                    $results[$entity->Id] = $entity;
                }
            }
        }
        return $results;
    }

    $customerMap = queryInClause($dataService, 'Customer', $customerIds);
    $itemMap = queryInClause($dataService, 'Item', $itemIds);
    $classMap = queryInClause($dataService, 'Class', $classIds);

    // --- Step 4: Process and Structure the Final Data (No changes needed here) ---
    $reportData = [];
    $lowercaseSelectedServices = array_map('strtolower', $selectedProductServices);

    foreach ($allInvoices as $txn) {
        $customerId = $txn->CustomerRef;
        if (!isset($customerMap[$customerId])) continue;

        foreach ($txn->Line as $line) {
            if (isset($line->DetailType) && $line->DetailType === 'SalesItemLineDetail' && !empty($line->SalesItemLineDetail->ItemRef)) {
                $itemId = $line->SalesItemLineDetail->ItemRef;
                $itemName = isset($itemMap[$itemId]) ? $itemMap[$itemId]->Name : 'Unknown Item';

                if (!empty($lowercaseSelectedServices) && !in_array(strtolower($itemName), $lowercaseSelectedServices)) {
                    continue;
                }

                if (!isset($reportData[$customerId])) {
                    $reportData[$customerId] = [
                        'customerInfo' => ['id' => $customerId, 'name' => $customerMap[$customerId]->DisplayName],
                        'total' => 0.00,
                        'quantity' => 0,
                        'details' => []
                    ];
                }

                $classId = $line->SalesItemLineDetail->ClassRef;
                $className = isset($classMap[$classId]) ? $classMap[$classId]->Name : '';
                
                $lineAmount = (float)($line->Amount ?? 0);
                $lineQty = (int)($line->SalesItemLineDetail->Qty ?? 0);
                $salesPrice = (float)($line->SalesItemLineDetail->UnitPrice ?? 0);

                $reportData[$customerId]['details'][] = [
                    'date'          => $txn->TxnDate,
                    'type'          => 'Invoice',
                    'num'           => $txn->DocNumber ?: 'ID: ' . $txn->Id,
                    'class'         => $className,
                    'product_service' => $itemName,
                    'qty'           => $lineQty,
                    'amount'        => $lineAmount,
                    'sales_price'   => $salesPrice
                ];
                $reportData[$customerId]['total'] += $lineAmount;
                $reportData[$customerId]['quantity'] += $lineQty;
            }
        }
    }
    
    echo json_encode(['success' => true, 'data' => $reportData]);

} catch (Exception $e) {
    error_log("AJAX Full Report Error: " . $e->getMessage());
    echo json_encode(['success' => false, 'error' => 'An API error occurred while generating the report.']);
    exit;
}