<?php
// qbo_ajax_pnl_graph_data.php
// FINAL, COMPLETE VERSION: Handles all graph types including Yearly Average.

header('Content-Type: application/json');

use QuickBooksOnline\API\DataService\DataService;
use QuickBooksOnline\API\ReportService\ReportService;

require_once __DIR__ . '/../vendor/autoload.php';
require_once __DIR__ . '/qbo_functions.php';
require_once __DIR__ . '/config/qbo_config.php';
global $qboBaseConfig;

$startDate = $_GET['start_date'] ?? null;
$endDate = $_GET['end_date'] ?? null;
$columns = $_GET['columns'] ?? 'Months';

if (!$startDate || !$endDate) {
    echo json_encode(['success' => false, 'error' => 'Missing date parameters.']);
    exit;
}

// --- QBO Connection & Token Refresh ---
$dataService = null;
$tokenStorageFile = __DIR__ . '/tokens/qbo_token.json';
$currentTokens = file_exists($tokenStorageFile) ? json_decode(file_get_contents($tokenStorageFile), true) : null;
if ($currentTokens && !empty($currentTokens['access_token'])) {
    $qboBaseConfig['accessTokenKey'] = $currentTokens['access_token'];
    try { $dataService = DataService::Configure($qboBaseConfig); } catch (Exception $e) { /* handled below */ }
}

if (!$dataService) {
    echo json_encode(['success' => false, 'error' => 'QBO connection failed.']);
    exit;
}

/**
 * HELPER FUNCTION: Makes one simple P&L API call and parses all key summary rows.
 */
function getPnlTotalsForPeriod(DataService $dataSvc, $start, $end) {
    $result = ['income' => 0, 'cogs' => 0, 'expenses' => 0, 'net' => 0];
    try {
        $reportService = new ReportService($dataSvc->getServiceContext());
        $reportService->setStartDate($start);
        $reportService->setEndDate($end);
        $report = $reportService->executeReport("ProfitAndLoss");

        if ($report && isset($report->Rows, $report->Rows->Row)) {
            foreach ($report->Rows->Row as $row) {
                if (isset($row->Summary)) {
                    $rowLabel = $row->Summary->ColData[0]->value ?? null;
                    $rowValue = (float)($row->Summary->ColData[1]->value ?? 0);
                    
                    switch ($rowLabel) {
                        case 'Total Income': $result['income'] = $rowValue; break;
                        case 'Total Cost of Sales': $result['cogs'] = $rowValue; break;
                        case 'Total Expenses': $result['expenses'] = $rowValue; break;
                        case 'Net earnings': $result['net'] = $rowValue; break;
                    }
                }
            }
        }
    } catch (Exception $e) {
        error_log("QBO API Error in getPnlTotalsForPeriod for {$start}-{$end}: " . $e->getMessage());
    }
    return $result;
}


// --- MAIN LOGIC ---
$labels = []; $incomeData = []; $cogsData = []; $expensesData = []; $netData = [];

try {
    if ($columns === 'Months' || $columns === 'Years') {
        $loopStart = ($columns === 'Months') ? 1 : (int)date('Y', strtotime($startDate));
        $loopEnd = ($columns === 'Months') ? 12 : (int)date('Y', strtotime($endDate));
        $yearForMonths = date('Y', strtotime($startDate));

        for ($i = $loopStart; $i <= $loopEnd; $i++) {
            if ($columns === 'Months') {
                $labels = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];
                $monthStart = sprintf('%s-%02d-01', $yearForMonths, $i);
                $monthEnd = date('Y-m-t', strtotime($monthStart));
                $periodTotals = getPnlTotalsForPeriod($dataService, $monthStart, $monthEnd);
            } else { // Years
                $labels[] = (string)$i;
                $yearStart = "$i-01-01";
                $yearEnd = "$i-12-31";
                $periodTotals = getPnlTotalsForPeriod($dataService, $yearStart, $yearEnd);
            }
            $incomeData[] = $periodTotals['income'];
            $cogsData[] = abs($periodTotals['cogs']);
            $expensesData[] = abs($periodTotals['expenses']);
            $netData[] = $periodTotals['net'];
        }

    } elseif ($columns === 'Total_for_avg') { // NEW: Specific case for the yearly average graph
        $startYear = (int)date('Y', strtotime($startDate));
        $endYear = (int)date('Y', strtotime($endDate));
        for ($year = $startYear; $year <= $endYear; $year++) {
            $labels[] = (string)$year;
            $yearStart = "$year-01-01";
            $yearEnd = "$year-12-31";
            $periodTotals = getPnlTotalsForPeriod($dataService, $yearStart, $yearEnd);
            // Calculate the monthly average for each component
            $incomeData[] = $periodTotals['income'] / 12;
            $cogsData[] = abs($periodTotals['cogs']) / 12;
            $expensesData[] = abs($periodTotals['expenses']) / 12;
            $netData[] = $periodTotals['net'] / 12;
        }
    }
    
    echo json_encode([
        'success' => true,
        'data' => [ 'labels' => $labels, 'income' => $incomeData, 'cogs' => $cogsData, 'expenses' => $expensesData, 'net' => $netData ]
    ]);

} catch (Exception $e) {
    echo json_encode(['success' => false, 'error' => 'A critical error occurred: ' . $e->getMessage()]);
    exit;
}