<?php
// qbo_bootstrap.php
// This file handles all QuickBooks Online connection setup and authentication.
// Include this file in any script that needs to interact with the QBO API.

// --- Essential Includes (Using Corrected Paths) ---
require_once __DIR__ . '/qbo_functions.php';
require_once __DIR__ . '/../vendor/autoload.php';

use QuickBooksOnline\API\Core\OAuth\OAuth2\OAuth2LoginHelper;

// --- Configuration & Helper Functions ---
$qboBaseConfigFile = __DIR__ . '/config/qbo_config.php';
$qboTokenStorageFile = __DIR__ . '/tokens/qbo_token.json';
$qboBaseConfig = null;

function loadQBOTokens(): ?array {
    global $qboTokenStorageFile;
    if (!file_exists($qboTokenStorageFile)) return null;
    $json = file_get_contents($qboTokenStorageFile);
    return json_decode($json, true) ?: null;
}

function saveQBOTokens(string $accessToken, string $refreshToken): bool {
    global $qboTokenStorageFile;
    $tokens = ['access_token' => $accessToken, 'refresh_token' => $refreshToken, 'last_updated' => date('Y-m-d H:i:s')];
    return (bool)file_put_contents($qboTokenStorageFile, json_encode($tokens, JSON_PRETTY_PRINT));
}


// --- Main Bootstrap Logic ---

// This variable will be available to any script that includes this file.
$qboUtil = null;

// Use a try/catch block to handle any potential connection errors gracefully.
try {
    if (file_exists($qboBaseConfigFile)) {
        require $qboBaseConfigFile;
    } else {
        // Throw an exception if the critical config file is missing.
        throw new Exception("CRITICAL: QBO base config file not found at " . $qboBaseConfigFile);
    }

    if ($qboBaseConfig) {
        $currentTokens = loadQBOTokens();
        $qboConfigForLibrary = $qboBaseConfig;

        // Attempt to refresh the token if we have a refresh token.
        if ($currentTokens && !empty($currentTokens['refresh_token'])) {
            $oauth2LoginHelper = new OAuth2LoginHelper($qboBaseConfig['ClientID'], $qboBaseConfig['ClientSecret']);
            $refreshedAccessTokenObj = $oauth2LoginHelper->refreshAccessTokenWithRefreshToken($currentTokens['refresh_token']);
            if (saveQBOTokens($refreshedAccessTokenObj->getAccessToken(), $refreshedAccessTokenObj->getRefreshToken())) {
                $currentTokens['access_token'] = $refreshedAccessTokenObj->getAccessToken();
            }
        }

        // Proceed to create the utility object if we have a valid access token.
        if ($currentTokens && !empty($currentTokens['access_token'])) {
            $qboConfigForLibrary['accessTokenKey'] = $currentTokens['access_token'];
            $qboConfigForLibrary['refreshTokenKey'] = $currentTokens['refresh_token'] ?? null;
            
            // This instantiates the library and makes the $qboUtil object ready for use.
            $qboUtil = new QBOUtilityLibrary($qboConfigForLibrary);
        } else {
            // If we still don't have a token, we cannot proceed.
            throw new Exception("CRITICAL: No valid QBO access token available. Please re-authenticate.");
        }
    }
} catch (Exception $e) {
    // In a real application, you might want to die() here or redirect to an error page.
    error_log("Error in qbo_bootstrap.php: " . $e->getMessage());
    // For now, we can let the consuming script handle the fact that $qboUtil is null.
}

?>