<?php
// qbo_functions.php
// QuickBooks Online Utility Library
// This library provides a set of functions to interact with the QuickBooks Online API
// for common tasks like managing customers, vendors, classes, items (products/services),
// invoices, and finding tax codes.

// Ensure the QuickBooks SDK autoloader is included (typically via composer's vendor/autoload.php)
require_once __DIR__ . '/../vendor/autoload.php'; // Adjust path if this file is not in the same directory as 'vendor'

use QuickBooksOnline\API\DataService\DataService;
use QuickBooksOnline\API\Facades\Customer;
use QuickBooksOnline\API\Facades\Vendor;
use QuickBooksOnline\API\Facades\Invoice;
use QuickBooksOnline\API\Facades\Bill;
use QuickBooksOnline\API\Facades\QuickBookClass;
use QuickBooksOnline\API\Facades\Item;
use QuickBooksOnline\API\Facades\Term;
use QuickBooksOnline\API\QueryFilter\QueryMessage;
use QuickBooksOnline\API\Exception\IdsException;
use QuickBooksOnline\API\ReportService\ReportService;
// use QuickBooksOnline\API\Core\Http\Serialization\XmlObjectSerializer; // Uncomment for XML request/response debugging if needed

class QBOUtilityLibrary
{
    private $dataService; // Holds the configured DataService instance for API communication
    private $dbConn = null;
    // private $debugLogFile = __DIR__ . '/merlin_debug_library.log'; // This was unused
    
    /**
     * Injects the database connection and ensures the sync history table exists.
     *
     * @param mysqli $dbConn A live mysqli database connection.
     */
    public function setDatabaseConnection(mysqli $dbConn)
    {
        $this->dbConn = $dbConn;
        $this->ensureSyncHistoryTableExists(); // Call the table creation check.
    }
    
    /**
     * Checks if the sync history table exists and creates it if it does not.
     * This makes the logging feature self-sufficient.
     */
    private function ensureSyncHistoryTableExists()
    {
        if (!$this->dbConn) {
            return; // Do nothing if there's no DB connection.
        }
    
        $tableName = 'tdu_qbo_sync_history';
        $checkTableQuery = "SHOW TABLES LIKE '{$tableName}'";
        $result = $this->dbConn->query($checkTableQuery);
    
        if ($result && $result->num_rows == 0) {
            error_log("QBO Lib: Table '{$tableName}' not found. Attempting to create it.");
            
            // Renamed 'vtiger_quote_no' to the more generic 'source_identifier'
            $createTableSQL = "
            CREATE TABLE `{$tableName}` (
              `id` INT(11) NOT NULL AUTO_INCREMENT,
              `qbo_entity_id` VARCHAR(50) NOT NULL,
              `qbo_entity_type` VARCHAR(50) NOT NULL COMMENT 'e.g., Customer, Vendor, Invoice, Bill',
              `qbo_entity_number` VARCHAR(50) NULL COMMENT 'The human-readable invoice or bill number',
              `organisation_name` VARCHAR(255) NULL COMMENT 'The DisplayName of the Customer or Vendor',
              `source_identifier` VARCHAR(100) NULL COMMENT 'The related source system identifier (e.g., quote no, order id)',
              `sync_datetime` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
              PRIMARY KEY (`id`),
              UNIQUE KEY `qbo_entity_id_type` (`qbo_entity_id`, `qbo_entity_type`),
              KEY `organisation_name` (`organisation_name`),
              KEY `source_identifier` (`source_identifier`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";
    
            if (!$this->dbConn->query($createTableSQL)) {
                // If creation fails, log a critical error.
                error_log("QBO Lib FATAL ERROR: Could not create table '{$tableName}': " . $this->dbConn->error);
            } else {
                error_log("QBO Lib: Table '{$tableName}' created successfully.");
            }
        }
    }

    /**
     * QBOUtilityLibrary constructor.
     * Initializes the DataService for communication with the QBO API.
     * @param array $qboConfig Configuration array for DataService.
     *        Requires: ClientID, ClientSecret, accessTokenKey, refreshTokenKey, QBORealmID, baseUrl.
     */
    public function __construct(array $qboConfig)
    {
        // Validate essential configuration parameters
        if (empty($qboConfig['ClientID']) || empty($qboConfig['ClientSecret']) ||
            empty($qboConfig['accessTokenKey']) || empty($qboConfig['refreshTokenKey']) ||
            empty($qboConfig['QBORealmID']) || empty($qboConfig['baseUrl'])) {
            throw new \InvalidArgumentException("QBO configuration is incomplete. All core OAuth2 parameters, RealmID, and baseUrl are required.");
        }

        // Configure the DataService object using settings from $qboConfig
        $this->dataService = DataService::Configure([
            'auth_mode'       => $qboConfig['auth_mode'] ?? 'oauth2',
            'ClientID'        => $qboConfig['ClientID'],
            'ClientSecret'    => $qboConfig['ClientSecret'],
            'accessTokenKey'  => $qboConfig['accessTokenKey'],
            'refreshTokenKey' => $qboConfig['refreshTokenKey'],
            'QBORealmID'      => $qboConfig['QBORealmID'],
            'baseUrl'         => $qboConfig['baseUrl'] // Should be "Development" or "Production"
        ]);

        // Optional: Set log location for the SDK's internal logging activities
        if (isset($qboConfig['logLocation'])) {
            $this->dataService->setLogLocation($qboConfig['logLocation']);
        }
        // Configure the SDK to throw exceptions on API errors, which simplifies error handling in consuming code.
        $this->dataService->throwExceptionOnError(true);
    }

    /**
     * Queries a QuickBooks Online entity by a specific field and value.
     * This is a generic helper to find various types of QBO entities.
     * @param string $entityName The name of the QBO entity (e.g., "Customer", "Vendor", "Class", "TaxCode", "Item").
     * @param string $field The field to query by (e.g., "DisplayName" for Customer, "Name" for Class/Item/TaxCode).
     * @param mixed $value The value to search for.
     * @param string $operator The comparison operator (e.g., "=", "LIKE"). Defaults to "=".
     * @param string $selectColumns Columns to select. Defaults to "*".
     * @return \QuickBooksOnline\API\Data\IPPIdType|null The first matching entity object, or null if not found.
     * @throws IdsException If the QBO API call fails.
     * @throws \InvalidArgumentException If input parameters are invalid.
     */
    public function queryEntityByField(string $entityName, string $field, $value, string $operator = '=', string $selectColumns = "*")
    {
        if (!is_string($value) && !is_numeric($value) && !is_bool($value)) {
            throw new \InvalidArgumentException("Value for querying must be a string, number, or boolean.");
        }
        if (empty(trim($field))) {
            throw new \InvalidArgumentException("Field name for querying cannot be empty.");
        }

        // This flag is now unused as the specific echo statements have been removed. Kept for context or if needed later.
        // $isMerlinVendorQuery = ($entityName === 'Vendor' && is_string($value) && strcasecmp($value, "Merlin Entertainment") == 0);

        $queryMessage = new QueryMessage();
        $queryMessage->sql = "SELECT";
        if ($selectColumns !== "*" && !empty($selectColumns)) {
            $queryMessage->selectClause = $selectColumns;
        }
        $queryMessage->entity = $entityName;

        $formattedValue = $value;
        if (is_string($value)) {
            $escapedValue = str_replace("'", "\\'", $value);
            $formattedValue = "'" . $escapedValue . "'";
        } elseif (is_bool($value)) {
            $formattedValue = $value ? 'true' : 'false';
        }

        $conditionString = "{$field} {$operator} {$formattedValue}";
        $queryMessage->whereClause = [$conditionString];
        $queryMessage->maxresults = "1";

        $queryString = $queryMessage->getString();
        
        error_log("QBO Utility Lib Query: Entity='{$entityName}', Field='{$field}', Value (formatted)='{$formattedValue}', Operator='{$operator}', Query='{$queryString}'");

        if (!$queryString) {
            error_log("QBO Utility Lib: Failed to construct query string for entity: {$entityName} with condition: {$conditionString}. Original Value: " . print_r($value, true));
            throw new \RuntimeException("Failed to construct query string for entity: {$entityName} with condition: {$conditionString}.");
        }
        
        try {
            $entities = $this->dataService->Query($queryString);
            
            if (!empty($entities) && count($entities) > 0) {
                return reset($entities); 
            }
        } catch (IdsException $e) {
            error_log("QBO Utility Lib IdsException for Query: {$queryString}. Error: " . $e->getMessage());
            throw $e;
        } catch (Exception $e) {
            error_log("QBO Utility Lib Generic Exception for Query: {$queryString}. Error: " . $e->getMessage());
            throw $e;
        }
        return null; // Entity not found
    }

    /**
     * Creates a new Class entity in QuickBooks Online.
     * @param array $classData Data for the new class. Must include 'Name'.
     * @return \QuickBooksOnline\API\Data\IPPQuickBookClass The created Class object.
     */
    public function createClass(array $classData)
    {
        if (empty($classData['Name'])) {
            throw new \InvalidArgumentException("Class data must include 'Name' for creation.");
        }
        $classObj = QuickBookClass::create($classData);
        return $this->dataService->Add($classObj);
    }

    /**
     * Finds a Class by its Name, or creates it if not found.
     * @param string $className The Name of the Class to find or create.
     * @return \QuickBooksOnline\API\Data\IPPQuickBookClass The found or newly created Class object.
     */
    public function getOrCreateClassByName(string $className)
    {
        if (empty(trim($className))) {
            throw new \InvalidArgumentException("Class Name cannot be empty for getOrCreateClassByName.");
        }
        $class = $this->queryEntityByField('Class', 'Name', $className);

        if ($class) {
            return $class;
        } else {
            error_log("QBO Utility Lib: Creating new QBO Class with Name: '" . $className . "'");
            $newClassData = ['Name' => $className];
            return $this->createClass($newClassData);
        }
    }

    /**
     * Creates a new Item (Product/Service) in QuickBooks Online.
     * @param array $itemData Data for the new item. Must include 'Name' and 'Type'.
     * @return \QuickBooksOnline\API\Data\IPPItem The created Item object.
     */
    public function createItem(array $itemData)
    {
        if (empty($itemData['Name'])) {
            throw new \InvalidArgumentException("Item data must include 'Name' for creation.");
        }
        if (empty($itemData['Type'])) {
            throw new \InvalidArgumentException("Item data must include 'Type' (e.g., 'Service', 'NonInventory').");
        }
        if (($itemData['Type'] === 'Service' || $itemData['Type'] === 'NonInventory') && empty($itemData['IncomeAccountRef']['value'])) {
            // This validation might need adjustment based on $costOfSalesAccountId usage in getOrCreateItemByName
            // However, if $costOfSalesAccountId is used, IncomeAccountRef will be populated there.
            // This check is more for direct createItem calls or when COGS account isn't overriding.
            // Consider if ExpenseAccountRef is also mandatory for NonInventory if COGS logic isn't used.
            error_log("Warning: Item data for '{$itemData['Type']}' items ('{$itemData['Name']}') usually requires 'IncomeAccountRef'. Ensure it's set or handled by COGS logic.");
            // throw new \InvalidArgumentException("Item data for '{$itemData['Type']}' items must include 'IncomeAccountRef' with a valid account ID if not using COGS override.");
        }
        $itemObj = Item::create($itemData);
        return $this->dataService->Add($itemObj);
    }

    /**
     * Finds a TaxCode in QuickBooks Online by its Name.
     * @param string $taxCodeName The exact Name of the TaxCode to find.
     * @return \QuickBooksOnline\API\Data\IPPTaxCode|null The found TaxCode object, or null if not found.
     */
    public function findTaxCodeByName(string $taxCodeName)
    {
        if (empty(trim($taxCodeName))) {
            throw new \InvalidArgumentException("TaxCode Name cannot be empty for findTaxCodeByName.");
        }
        error_log("QBO Utility Lib: Querying for TaxCode with Name: '" . $taxCodeName . "'");
        $taxCode = $this->queryEntityByField('TaxCode', 'Name', $taxCodeName);

        if ($taxCode) {
            error_log("QBO Utility Lib: Found TaxCode '{$taxCodeName}' with ID: {$taxCode->Id}");
            return $taxCode;
        } else {
            error_log("QBO Utility Lib: TaxCode with Name '{$taxCodeName}' not found in QBO company.");
            return null;
        }
    }

    /**
     * Creates a new customer in QuickBooks Online.
     * @param array $customerData Data for the new customer. Must include 'DisplayName'.
     * @return \QuickBooksOnline\API\Data\IPPCustomer The created customer object.
     */
    public function createCustomer(array $customerData, ?string $sourceIdentifier = null)
    {
        if (empty($customerData['DisplayName'])) {
            throw new \InvalidArgumentException("Customer data must include at least 'DisplayName'.");
        }
        $customerObj = Customer::create($customerData);
        $createdCustomer = $this->dataService->Add($customerObj); // 1. Create the customer
        $this->recordSyncHistory($createdCustomer->Id, 'Customer', null, $customerData['DisplayName'], $sourceIdentifier);
        
        return $createdCustomer;
    }

    /**
     * Finds a customer by a specific field, or creates them if not found.
     * @param array $customerCreateData Data to use if the customer needs to be created.
     * @param string $searchField The field to search by.
     * @param mixed $searchValue The value to search for.
     * @return \QuickBooksOnline\API\Data\IPPCustomer The found or newly created customer object.
     */
    public function getOrCreateCustomer(array $customerCreateData, string $searchField, $searchValue, ?string $sourceIdentifier = null)
    {
        if (empty($searchField) || ($searchValue === null || $searchValue === '')) {
             throw new \InvalidArgumentException("Search field and value cannot be empty for getOrCreateCustomer.");
        }
        $customer = $this->queryEntityByField('Customer', $searchField, $searchValue);
        if ($customer) {
            return $customer;
        } else {
            if (empty($customerCreateData['DisplayName'])) {
                if ($searchField === 'DisplayName' && !empty($searchValue) && is_string($searchValue)) {
                    $customerCreateData['DisplayName'] = $searchValue;
                } else {
                    throw new \InvalidArgumentException("Customer creation data must include 'DisplayName' or it must be determinable from search parameters.");
                }
            }
            if (empty($customerCreateData[$searchField]) && $searchField !== 'Id' && $searchField !== 'DisplayName' && is_string($searchValue)) {
                 // This logic to add search field/value to create data might need careful review based on specific use cases
                 // For simple DisplayName search, it's fine. For complex fields like 'PrimaryEmailAddr.Address', direct assignment might not work as expected.
                 // $customerCreateData[$searchField] = $searchValue;
            }
            return $this->createCustomer($customerCreateData, $sourceIdentifier);
        }
    }

    /**
     * Creates an invoice in QuickBooks Online.
     * @param string $customerId The ID of the customer for this invoice.
     * @param array $lineItems An array of line items for the invoice.
     * @param array $invoiceMetaData Optional additional data for the invoice header.
     * @return \QuickBooksOnline\API\Data\IPPInvoice The created invoice object.
     */
    public function createInvoice(string $customerId, array $lineItems, array $invoiceMetaData = [], ?string $sourceIdentifier = null, ?string $orgNameToLog = null)
    {
        if (empty($customerId)) {
            throw new \InvalidArgumentException("Customer ID cannot be empty for creating an invoice.");
        }
        if (empty($lineItems)) {
            throw new \InvalidArgumentException("Line items cannot be empty for creating an invoice.");
        }
        foreach ($lineItems as $idx => $item) {
            if (empty($item['DetailType'])) {
                throw new \InvalidArgumentException("Invoice Line item at index {$idx} is missing 'DetailType'.");
            }
            if ($item['DetailType'] === "SalesItemLineDetail") {
                if (empty($item['SalesItemLineDetail']['ItemRef']['value'])) {
                     throw new \InvalidArgumentException("SalesItemLineDetail at index {$idx} must have an ItemRef value (Product/Service ID).");
                }
                if (!isset($item['Amount']) && (!isset($item['SalesItemLineDetail']['Qty']) || !isset($item['SalesItemLineDetail']['UnitPrice']))) {
                     throw new \InvalidArgumentException("SalesItemLineDetail at index {$idx} must have an Amount or both Qty and UnitPrice for amount calculation.");
                }
            }
        }

        $invoiceData = array_merge($invoiceMetaData, [
            "Line" => $lineItems,
            "CustomerRef" => [ "value" => $customerId ]
        ]);
        
        error_log("QBOUtilityLibrary: Preparing to create invoice. Customer ID: {$customerId}. Line count: " . count($lineItems));
        // error_log("QBOUtilityLibrary: Invoice Data for creation: " . json_encode($invoiceData)); // Can be verbose
        
        $invoiceObj = Invoice::create($invoiceData);
        $createdInvoice = $this->dataService->Add($invoiceObj);
    
        // Use the new parameter name in the call
        $this->recordSyncHistory($createdInvoice->Id, 'Invoice', $createdInvoice->DocNumber, $orgNameToLog, $sourceIdentifier);
    
        return $createdInvoice;
    }
    
    /**
     * Creates a new vendor in QuickBooks Online.
     * @param array $vendorData Data for the new vendor. Must include 'DisplayName'.
     * @return \QuickBooksOnline\API\Data\IPPVendor The created vendor object.
     */
    public function createVendor(array $vendorData, ?string $sourceIdentifier = null)
    {
        if (empty($vendorData['DisplayName'])) {
            throw new \InvalidArgumentException("Vendor data must include at least 'DisplayName'.");
        }
        $vendorObj = Vendor::create($vendorData);
        error_log("QBO Utility Lib: Attempting to create new Vendor with DisplayName: '" . $vendorData['DisplayName'] . "'");
        $createdVendor = $this->dataService->Add($vendorObj); // 1. Create the vendor
        $this->recordSyncHistory($createdVendor->Id, 'Vendor', null, $vendorData['DisplayName'], $sourceIdentifier); // 2. Then record it
        
        return $createdVendor;
    }
    
    /**
     * Finds a vendor by a specific field, or creates them if not found.
     * @param array $vendorCreateData Data to use if the vendor needs to be created.
     * @param string $searchField The field to search by.
     * @param mixed $searchValue The value to search for.
     * @return \QuickBooksOnline\API\Data\IPPVendor The found or newly created vendor object.
     */
    public function getOrCreateVendor(array $vendorCreateData, string $searchField, $searchValue, ?string $sourceIdentifier = null)
    {
        if (empty($searchField) || ($searchValue === null || $searchValue === '')) {
             throw new \InvalidArgumentException("Search field and value cannot be empty for getOrCreateVendor.");
        }

        // This flag is now unused as the specific echo statements have been removed. Kept for context or if needed later.
        // $isMerlin = (is_string($searchValue) && strcasecmp($searchValue, "Merlin Entertainment") == 0);
        
        error_log("QBO Utility Lib: getOrCreateVendor. SearchField: '{$searchField}', SearchValue: '{$searchValue}'");

        $vendor = $this->queryEntityByField('Vendor', $searchField, $searchValue);

        if ($vendor) {
            error_log("QBO Utility Lib: Found Vendor '{$searchValue}' with ID: " . ($vendor->Id ?? 'N/A_ID'));
            return $vendor;
        } else {
            error_log("QBO Utility Lib: Vendor '{$searchValue}' not found by {$searchField}. Proceeding to create.");
            if (empty($vendorCreateData['DisplayName'])) {
                if ($searchField === 'DisplayName' && !empty($searchValue) && is_string($searchValue)) {
                    $vendorCreateData['DisplayName'] = $searchValue;
                } else {
                    throw new \InvalidArgumentException("Vendor creation data must include 'DisplayName' or it must be determinable from search parameters.");
                }
            }
            // Similar to getOrCreateCustomer, this part for non-DisplayName fields might need more specific handling
            // if (empty($vendorCreateData[$searchField]) && $searchField !== 'Id' && $searchField !== 'DisplayName' && is_string($searchValue)) {
            //      $vendorCreateData[$searchField] = $searchValue;
            // }
            return $this->createVendor($vendorCreateData, $sourceIdentifier);
        }
    }

    /**
     * Orchestrates getting/creating customer and creating an invoice.
     * @param array $customerConfig Config for customer.
     * @param array $invoiceLineItems Array of line items for the invoice.
     * @param array $invoiceMetaData Optional data for the invoice header.
     * @param array|null $vendorConfig Optional. Config for vendor (not used in this specific workflow but kept for signature consistency if extended).
     * @return array Results: ['customer' => IPPCustomer, 'invoice' => IPPInvoice, 'vendor' => IPPVendor|null]
     */
    public function processInvoiceCreationWorkflow(
        array $customerConfig,
        array $invoiceLineItems,
        array $invoiceMetaData = [],
        ?array $vendorConfig = null,
        ?string $sourceIdentifier = null,
        ?string $orgNameToLog = null
    ) {
        if (empty($customerConfig['search_field']) || !isset($customerConfig['search_value']) || empty($customerConfig['create_data'])) {
            throw new \InvalidArgumentException("Customer configuration is incomplete. Required: 'search_field', 'search_value', 'create_data'.");
        }

        $customer = $this->getOrCreateCustomer(
            $customerConfig['create_data'], $customerConfig['search_field'], $customerConfig['search_value'], $sourceIdentifier
        );

        $createdOrFoundVendor = null; // Vendor part is optional for this workflow
        if ($vendorConfig !== null) {
            // This block would be used if vendor processing was part of this specific workflow
            if (empty($vendorConfig['search_field']) || !isset($vendorConfig['search_value']) || empty($vendorConfig['create_data'])) {
                throw new \InvalidArgumentException("Vendor configuration is incomplete if provided. Required: 'search_field', 'search_value', 'create_data'.");
            }
            $createdOrFoundVendor = $this->getOrCreateVendor(
                $vendorConfig['create_data'], $vendorConfig['search_field'], $vendorConfig['search_value'], $sourceIdentifier
            );
        }

        if (!$customer || empty($customer->Id)) {
             throw new \RuntimeException("Failed to retrieve or create a customer with a valid ID. Cannot proceed with invoice creation.");
        }

        $invoice = $this->createInvoice($customer->Id, $invoiceLineItems, $invoiceMetaData, $sourceIdentifier, $orgNameToLog);

        return [
            'customer' => $customer,
            'vendor'   => $createdOrFoundVendor,
            'invoice'  => $invoice
        ];
    }

    /**
     * Helper function to directly access the configured DataService object.
     * @return DataService The configured DataService instance.
     */
    public function getDataService()
    {
        return $this->dataService;
    }
    
    /**
     * Creates a new Term in QuickBooks Online.
     * @param array $termData Example: ['Name' => 'Due on receipt', 'Type' => 'STANDARD', 'DueDays' => 0]
     * @return \QuickBooksOnline\API\Data\IPPTerm The created Term object.
     */
    public function createTerm(array $termData)
    {
        if (empty($termData['Name'])) {
            throw new \InvalidArgumentException("Term data must include 'Name' for creation.");
        }
        if (empty($termData['Type'])) {
            $termData['Type'] = 'STANDARD'; 
            error_log("QBO Utility Lib: Term data missing 'Type', defaulting to 'STANDARD' for Term: " . $termData['Name']);
        }
        if ($termData['Type'] === 'STANDARD' && !isset($termData['DueDays'])) {
            // For STANDARD type, DueDays is usually expected, even if 0 for "Due on receipt"
            error_log("QBO Utility Lib: Term data for 'STANDARD' type ('" . $termData['Name'] . "') is missing 'DueDays'. QBO might default or error.");
            // throw new \InvalidArgumentException("Term data for 'STANDARD' type must include 'DueDays'.");
        }
    
        $termObj = Term::create($termData);
        error_log("QBO Utility Lib: Attempting to create new Term with data: " . json_encode($termData));
        return $this->dataService->Add($termObj);
    }
    
    /**
     * Finds a Term by its Name, or creates it if not found.
     * @param string $termName The Name of the Term.
     * @param array $defaultTermDataForCreation Default data if creating.
     * @return \QuickBooksOnline\API\Data\IPPTerm The found or newly created Term object.
     */
    public function getOrCreateTermByName(string $termName, array $defaultTermDataForCreation)
    {
        if (empty(trim($termName))) {
            throw new \InvalidArgumentException("Term Name cannot be empty for getOrCreateTermByName.");
        }
        $term = $this->queryEntityByField('Term', 'Name', $termName);
    
        if ($term) {
            error_log("QBO Utility Lib: Found Term '{$termName}' with ID: {$term->Id}");
            return $term;
        } else {
            error_log("QBO Utility Lib: Term '{$termName}' not found. Attempting to create.");
            $newTermData = $defaultTermDataForCreation;
            $newTermData['Name'] = $termName; 
            if (empty($newTermData['Type'])) {
                $newTermData['Type'] = 'STANDARD'; // Default if not provided
            }
            if ($newTermData['Type'] === 'STANDARD' && !isset($newTermData['DueDays'])) {
                 // Ensure DueDays for standard terms
                 throw new \InvalidArgumentException("Default Term data for 'STANDARD' type must include 'DueDays' for Term: {$termName}");
            }
            return $this->createTerm($newTermData);
        }
    }
    
    /**
     * Creates a Bill in QuickBooks Online.
     * @param string $vendorId The ID of the Vendor for this bill.
     * @param array $lineItems An array of line items for the bill.
     * @param array $billMetaData Optional additional data for the bill header.
     * @return \QuickBooksOnline\API\Data\IPPBill The created Bill object.
     */
    public function createBill(string $vendorId, array $lineItems, array $billMetaData = [], ?string $sourceIdentifier = null, ?string $orgNameToLog = null)
    {
        if (empty($vendorId)) {
            throw new \InvalidArgumentException("Vendor ID cannot be empty for creating a bill.");
        }
        if (empty($lineItems)) {
            throw new \InvalidArgumentException("Line items cannot be empty for creating a bill.");
        }

        foreach ($lineItems as $idx => $item) {
            if (empty($item['DetailType'])) {
                throw new \InvalidArgumentException("Bill Line item at index {$idx} is missing 'DetailType'.");
            }
            if (!isset($item['Amount'])) {
                throw new \InvalidArgumentException("Bill Line item at index {$idx} is missing 'Amount'.");
            }
            switch ($item['DetailType']) {
                case "ItemBasedExpenseLineDetail":
                    if (empty($item['ItemBasedExpenseLineDetail']['ItemRef']['value'])) {
                        throw new \InvalidArgumentException("ItemBasedExpenseLineDetail at index {$idx} must have an ItemRef value.");
                    }
                    break;
                case "AccountBasedExpenseLineDetail":
                    if (empty($item['AccountBasedExpenseLineDetail']['AccountRef']['value'])) {
                        throw new \InvalidArgumentException("AccountBasedExpenseLineDetail at index {$idx} must have an AccountRef value.");
                    }
                    break;
                default:
                    throw new \InvalidArgumentException("Bill Line item at index {$idx} has an unsupported 'DetailType': {$item['DetailType']}.");
            }
        }

        $billData = array_merge($billMetaData, [
            "Line" => $lineItems,
            "VendorRef" => [ "value" => $vendorId ]
        ]);
        
        error_log("QBOUtilityLibrary: Preparing to create bill for Vendor ID: {$vendorId}. Line count: " . count($lineItems));
        // error_log("QBOUtilityLibrary: Bill Data for creation: " . json_encode($billData)); // Can be verbose

        $billObj = Bill::create($billData);
        $createdBill = $this->dataService->Add($billObj);

        // Use the new parameter name in the call
        $this->recordSyncHistory($createdBill->Id, 'Bill', $createdBill->DocNumber, $orgNameToLog, $sourceIdentifier);
        
        return $createdBill;
    }
    
    /**
     * Gets the exchange rate for a specific currency against the home currency for a given date.
     * @param string $sourceCurrencyCode The three-letter ISO currency code.
     * @param string|null $asOfDate The date for the exchange rate in 'YYYY-MM-DD' format.
     * @return \QuickBooksOnline\API\Data\IPPExchangeRate|null The ExchangeRate object or null.
     */
    public function getExchangeRateForCurrency(string $sourceCurrencyCode, ?string $asOfDate = null)
    {
        if (empty(trim($sourceCurrencyCode))) {
            throw new \InvalidArgumentException("Source currency code cannot be empty.");
        }
        $targetDate = $asOfDate ?? date('Y-m-d');
        // Querying by AsOfDate is the most direct way if QBO has a rate for that specific date.
        $queryString = "SELECT * FROM ExchangeRate WHERE SourceCurrencyCode = '{$sourceCurrencyCode}' AND AsOfDate = '{$targetDate}'";
        
        error_log("QBO Utility Lib Query: Fetching ExchangeRate for {$sourceCurrencyCode} as of {$targetDate}. Query: {$queryString}");

        try {
            $exchangeRates = $this->dataService->Query($queryString);
            if (!empty($exchangeRates) && count($exchangeRates) > 0) {
                $theRate = reset($exchangeRates);
                error_log("QBO Utility Lib: Found ExchangeRate. Rate: " . ($theRate->Rate ?? 'N/A') . " for {$sourceCurrencyCode} as of {$targetDate}");
                return $theRate;
            } else {
                error_log("QBO Utility Lib: No ExchangeRate found for {$sourceCurrencyCode} as of {$targetDate}.");
            }
        } catch (IdsException $e) {
            error_log("QBO Utility Lib IdsException for ExchangeRate Query: {$queryString}. Error: " . $e->getMessage());
            throw $e; 
        }
        return null;
    }
    
    /**
     * Gets the most recent exchange rate recorded for a specific currency.
     * @param string $sourceCurrencyCode The three-letter ISO currency code.
     * @return \QuickBooksOnline\API\Data\IPPExchangeRate|null The ExchangeRate object or null.
     */
    public function getMostRecentExchangeRate(string $sourceCurrencyCode)
    {
        if (empty(trim($sourceCurrencyCode))) {
            throw new \InvalidArgumentException("Source currency code cannot be empty.");
        }
        // Orders by LastUpdatedTime to get the most recent entry for the currency.
        $queryString = "SELECT * FROM ExchangeRate WHERE SourceCurrencyCode = '{$sourceCurrencyCode}' ORDERBY MetaData.LastUpdatedTime DESC MAXRESULTS 1";
        
        error_log("QBO Utility Lib Query: Fetching most recent ExchangeRate for {$sourceCurrencyCode}. Query: {$queryString}");

        try {
            $exchangeRates = $this->dataService->Query($queryString);
            if (!empty($exchangeRates) && count($exchangeRates) > 0) {
                $theRate = reset($exchangeRates);
                error_log("QBO Utility Lib: Found most recent ExchangeRate. Rate: " . ($theRate->Rate ?? 'N/A') . " as of " . ($theRate->AsOfDate ?? 'N/A') . " (Last Updated: " . ($theRate->MetaData->LastUpdatedTime ?? 'N/A') . ")");
                return $theRate;
            } else {
                error_log("QBO Utility Lib: No most recent ExchangeRate found for {$sourceCurrencyCode}.");
            }
        } catch (IdsException $e) {
            error_log("QBO Utility Lib IdsException for most recent ExchangeRate Query: {$queryString}. Error: " . $e->getMessage());
            throw $e; 
        }
        return null;
    }
    
    /**
     * Finds an Account in QuickBooks Online by its Name (tries FullyQualifiedName as fallback).
     * @param string $accountName The Name of the Account to find.
     * @return \QuickBooksOnline\API\Data\IPPAccount|null The found Account object, or null.
     */
    public function findAccountByName(string $accountName)
    {
        if (empty(trim($accountName))) {
            throw new \InvalidArgumentException("Account Name cannot be empty for findAccountByName.");
        }
        error_log("QBO Utility Lib: Querying for Account with Name: '" . $accountName . "'");
        $account = $this->queryEntityByField('Account', 'Name', $accountName);

        if ($account) {
            error_log("QBO Utility Lib: Found Account '{$accountName}' by Name with ID: {$account->Id}, Type: {$account->AccountType}");
            return $account;
        } else {
            error_log("QBO Utility Lib: Account with Name '{$accountName}' not found. Trying FullyQualifiedName.");
            $account = $this->queryEntityByField('Account', 'FullyQualifiedName', $accountName);
            if ($account) {
                error_log("QBO Utility Lib: Found Account '{$accountName}' (via FQN) with ID: {$account->Id}, Type: {$account->AccountType}");
                return $account;
            } else {
                error_log("QBO Utility Lib: Account with Name or FullyQualifiedName '{$accountName}' not found in QBO company.");
                return null;
            }
        }
    }

    /**
     * Finds an Item (Product/Service) by its Name, or creates it if not found.
     * @param string $itemName The Name of the Item to find or create.
     * @param array $defaultItemDataForCreation Default data for creation. 'Name' will be overridden.
     * @param string|null $costOfSalesAccountId Optional COGS account ID.
     * @return \QuickBooksOnline\API\Data\IPPItem The found or newly created Item object.
     */
    public function getOrCreateItemByName(string $itemName, array $defaultItemDataForCreation, ?string $costOfSalesAccountId = null)
    {
        if (empty(trim($itemName))) {
            throw new \InvalidArgumentException("Item Name cannot be empty for getOrCreateItemByName.");
        }

        $item = $this->queryEntityByField('Item', 'Name', $itemName);

        if ($item) {
            error_log("QBO Utility Lib: Found Item '{$itemName}' with ID: {$item->Id}, Type: {$item->Type}");
            return $item;
        } else {
            error_log("QBO Utility Lib: Item '{$itemName}' not found. Attempting to create.");

            $newItemData = $defaultItemDataForCreation;
            $newItemData['Name'] = $itemName; // Set/override name

            if ($costOfSalesAccountId !== null) {
                $newItemData['Type'] = 'NonInventory'; // Prefer NonInventory for COGS items
                $newItemData['IncomeAccountRef'] = ['value' => $costOfSalesAccountId];
                $newItemData['ExpenseAccountRef'] = ['value' => $costOfSalesAccountId];
                // PurchaseCost might be in $defaultItemDataForCreation or could be set to 0 if not provided
                $newItemData['PurchaseCost'] = $newItemData['PurchaseCost'] ?? 0;
                error_log("QBO Utility Lib: Setting new item '{$itemName}' to Type: NonInventory, Income/Expense Account ID: {$costOfSalesAccountId}");
            } else {
                // Ensure required fields if not using COGS override
                if (empty($newItemData['Type'])) {
                    throw new \InvalidArgumentException("Default item data for creation must include 'Type' if CostOfSalesAccountID is not specified.");
                }
                if (($newItemData['Type'] === 'Service' || $newItemData['Type'] === 'NonInventory') && empty($newItemData['IncomeAccountRef']['value'])) {
                    throw new \InvalidArgumentException("Default item data for '{$newItemData['Type']}' items ('{$itemName}') must include 'IncomeAccountRef' if CostOfSalesAccountID is not specified.");
                }
                 if ($newItemData['Type'] === 'NonInventory' && empty($newItemData['ExpenseAccountRef']['value'])) {
                    throw new \InvalidArgumentException("Default item data for 'NonInventory' items ('{$itemName}') must include 'ExpenseAccountRef' if CostOfSalesAccountID is not specified.");
                }
            }
            // Ensure TrackQtyOnHand is set, defaults to false if not present, which is typical for Service/NonInventory
            $newItemData['TrackQtyOnHand'] = $newItemData['TrackQtyOnHand'] ?? false;

            error_log("QBO Utility Lib: Creating new QBO Item '{$itemName}' with data: " . json_encode($newItemData));
            return $this->createItem($newItemData);
        }
    }
    
    /**
     * Finds Bills by Supplier name and Class name.
     * Note: Class filtering is done client-side as QBO API doesn't support direct querying of Bills by ClassRef on lines.
     *
     * @param string $supplierName The DisplayName of the Supplier (Vendor).
     * @param string $className The Name of the Class.
     * @return array An array of \QuickBooksOnline\API\Data\IPPBill objects that match the criteria.
     * @throws \InvalidArgumentException If supplierName or className are empty.
     * @throws IdsException If a QBO API call fails.
     * @throws \Exception For other general exceptions during processing.
     */
    public function findBillsBySupplierAndClassName(string $supplierName, string $className): array
    {
        if (empty(trim($supplierName))) {
            throw new \InvalidArgumentException("Supplier Name cannot be empty for findBillsBySupplierAndClassName.");
        }
        if (empty(trim($className))) {
            throw new \InvalidArgumentException("Class Name cannot be empty for findBillsBySupplierAndClassName.");
        }

        // 1. Get Vendor ID using the existing queryEntityByField method
        $vendor = $this->queryEntityByField('Vendor', 'DisplayName', $supplierName);
        if (!$vendor || empty($vendor->Id)) {
            error_log("QBO Utility Lib: Vendor '{$supplierName}' not found. Cannot query bills.");
            return []; // Or throw an exception if preferred
        }
        $vendorId = $vendor->Id;
        error_log("QBO Utility Lib: Found Vendor '{$supplierName}' with ID: {$vendorId}");

        // 2. Get Class ID using the existing queryEntityByField method
        $classEntity = $this->queryEntityByField('Class', 'Name', $className);
        if (!$classEntity || empty($classEntity->Id)) {
            error_log("QBO Utility Lib: Class '{$className}' not found. Cannot filter bills by this class.");
            return []; // Or throw an exception if preferred
        }
        $classId = $classEntity->Id;
        error_log("QBO Utility Lib: Found Class '{$className}' with ID: {$classId}");

        // 3. Query Bills by VendorRef, handling pagination
        $allVendorBills = [];
        $startPosition = 1;
        $maxResultsPerPage = 500; // QBO API can return up to 1000, using a slightly smaller batch size.

        error_log("QBO Utility Lib: Querying bills for Vendor ID: {$vendorId} (Supplier: '{$supplierName}')");

        while (true) {
            $query = "SELECT * FROM Bill WHERE VendorRef = '{$vendorId}' STARTPOSITION {$startPosition} MAXRESULTS {$maxResultsPerPage}";
            
            try {
                $billsBatch = $this->dataService->Query($query);
            } catch (IdsException $e) {
                error_log("QBO Utility Lib: IdsException while querying bills for Vendor ID {$vendorId}. Query: {$query}. Error: " . $e->getMessage());
                throw $e; // Re-throw the IdsException
            } catch (Exception $e) {
                error_log("QBO Utility Lib: Generic Exception while querying bills for Vendor ID {$vendorId}. Query: {$query}. Error: " . $e->getMessage());
                throw $e; // Re-throw other exceptions
            }

            if (empty($billsBatch)) {
                break; // No more bills found in this batch (or ever)
            }
            
            $allVendorBills = array_merge($allVendorBills, $billsBatch);
            
            if (count($billsBatch) < $maxResultsPerPage) {
                break; // This was the last page of results
            }
            
            $startPosition += count($billsBatch); // Prepare for the next page
        }

        if (empty($allVendorBills)) {
            error_log("QBO Utility Lib: No bills found for Vendor '{$supplierName}' (ID: {$vendorId}).");
            return [];
        }
        error_log("QBO Utility Lib: Retrieved " . count($allVendorBills) . " total bills for Vendor '{$supplierName}'. Now filtering by Class '{$className}' (ID: {$classId}).");

        // 4. Filter Bills by Class ID (client-side)
        error_log("QBO Filter Start: Supplier '{$supplierName}', ClassName '{$className}', TargetClassID '{$classId}'. Processing " . count($allVendorBills) . " bills for VendorID '{$vendorId}'.");

        $matchingBills = [];
        foreach ($allVendorBills as $bill_index => $bill) {
            $actualLines = [];
            if (!empty($bill->Line)) {
                if (is_array($bill->Line)) {
                    $actualLines = $bill->Line;
                } elseif (is_object($bill->Line) && $bill->Line instanceof \QuickBooksOnline\API\Data\IPPLine) {
                    $actualLines = [$bill->Line];
                } else {
                    error_log("QBO Filter: Bill #{$bill_index} (ID {$bill->Id}) has unusual Line property. Type: " . gettype($bill->Line));
                    continue;
                }
            }
    
            if (empty($actualLines)) {
                // error_log("QBO Filter: Bill #{$bill_index} (ID {$bill->Id}) has no processable lines.");
                continue;
            }
    
            $classFoundInThisBill = false;
            // error_log("QBO Filter: Bill #{$bill_index} (ID {$bill->Id}), DocNo: " . ($bill->DocNumber ?? 'N/A') . ". TargetClassID '{$classId}'. Lines: " . count($actualLines));
    
            foreach ($actualLines as $lineIdx => $line) {
                if (!is_object($line) || !property_exists($line, 'DetailType')) {
                    // error_log("QBO Filter: Bill #{$bill_index} (ID {$bill->Id}), Line #{$lineIdx} invalid structure.");
                    continue;
                }
    
                $lineClassRefValue = null;
                // error_log("QBO Filter: Bill #{$bill_index} (ID {$bill->Id}), Line #{$lineIdx}, DetailType: {$line->DetailType}");
    
                switch ($line->DetailType) {
                    case 'ItemBasedExpenseLineDetail':
                        if (isset($line->ItemBasedExpenseLineDetail->ClassRef)) {
                            $lineClassRefValue = (string) $line->ItemBasedExpenseLineDetail->ClassRef;
                            // Minimal log here to reduce noise, more detail in comparison
                        }
                        break;
                    case 'AccountBasedExpenseLineDetail':
                        if (isset($line->AccountBasedExpenseLineDetail->ClassRef)) {
                            $lineClassRefValue = (string) $line->AccountBasedExpenseLineDetail->ClassRef;
                        }
                        break;
                }
    
                if ($lineClassRefValue !== null) {
                    // THIS IS THE MOST IMPORTANT LOG NOW
                    error_log("QBO Filter: Bill #{$bill_index} (ID {$bill->Id}), Line #{$lineIdx}, LineClassRefValue: '{$lineClassRefValue}', TargetClassID: '{$classId}'");
                    if (trim($lineClassRefValue) === trim((string)$classId)) {
                        $classFoundInThisBill = true;
                        error_log("QBO Filter: MATCH FOUND! Bill #{$bill_index} (ID {$bill->Id}), Line #{$lineIdx}. Added to results.");
                        break; // Found matching class in this bill's lines
                    }
                }
            } // End foreach $actualLines
    
            if ($classFoundInThisBill) {
                $matchingBills[] = $bill;
            }
        } // End foreach $allVendorBills
    
        error_log("QBO Filter End: Found " . count($matchingBills) . " bills for Supplier '{$supplierName}' AND ClassName '{$className}'.");
        return $matchingBills;
    }
    
    /**
     * Finds a Vendor in QuickBooks Online by matching the email domain in their PrimaryEmailAddr.
     * Searches by fetching all vendors and filtering them client-side, as the API does not support
     * direct querying by email address.
     * @param string $emailDomain The email domain to search for (e.g., "example.com").
     * @return \QuickBooksOnline\API\Data\IPPVendor|null The first matching Vendor object, or null if not found.
     * @throws IdsException If a QBO API call to fetch vendors fails.
     * @throws \InvalidArgumentException If the email domain is empty.
     */
    public function findVendorByEmailDomain(string $emailDomain): ?\QuickBooksOnline\API\Data\IPPVendor
    {
        if (empty(trim($emailDomain))) {
            throw new \InvalidArgumentException("Email domain cannot be empty for findVendorByEmailDomain.");
        }
        $trimmedDomain = trim(strtolower($emailDomain));
        error_log("QBO Utility Lib: Searching for Vendor by Email Domain (Client-Side Filter): '{$trimmedDomain}'");

        // 1. Fetch ALL active vendors from QBO, handling pagination
        $allVendors = [];
        $startPosition = 1;
        while (true) {
            try {
                $vendorsBatch = $this->dataService->Query("SELECT * FROM Vendor WHERE Active = true STARTPOSITION {$startPosition} MAXRESULTS 1000");
            } catch (IdsException $e) {
                error_log("QBO Utility Lib: IdsException while fetching all vendors (Batch starting at {$startPosition}). Error: " . $e->getMessage());
                throw $e; // Re-throw the exception as we cannot proceed
            }

            if (empty($vendorsBatch)) {
                break; // No more vendors to fetch
            }
            $allVendors = array_merge($allVendors, $vendorsBatch);
            $startPosition += count($vendorsBatch);
        }

        error_log("QBO Utility Lib: Fetched " . count($allVendors) . " active vendors. Now filtering for domain '{$trimmedDomain}'.");

        // 2. Loop through the fetched vendors and check their email address
        foreach ($allVendors as $vendor) {
            if (isset($vendor->PrimaryEmailAddr->Address)) {
                // Use case-insensitive comparison to find the domain within the email string
                if (stripos($vendor->PrimaryEmailAddr->Address, $trimmedDomain) !== false) {
                    error_log("QBO Utility Lib: Found match! Vendor: '{$vendor->DisplayName}', Email: '{$vendor->PrimaryEmailAddr->Address}'");
                    return $vendor; // Return the first vendor that matches
                }
            }
        }

        error_log("QBO Utility Lib: No vendor found with domain '{$trimmedDomain}' after checking all " . count($allVendors) . " active vendors.");
        return null; // No match found
    }
    
    /**
     * Updates an existing Bill in QuickBooks Online.
     * The Bill object provided must contain a valid Id and SyncToken.
     *
     * @param \QuickBooksOnline\API\Data\IPPBill $billObject The complete Bill object with modified values.
     * @return \QuickBooksOnline\API\Data\IPPBill The updated Bill object returned from QBO (with a new SyncToken).
     * @throws \InvalidArgumentException If the Bill object is missing Id or SyncToken.
     * @throws IdsException If the QBO API call fails.
     */
    public function updateBill(\QuickBooksOnline\API\Data\IPPBill $billObject)
    {
        // THE FIX: Change the validation to correctly handle a SyncToken of "0".
        if (empty($billObject->Id) || !isset($billObject->SyncToken) || $billObject->SyncToken === '') {
            throw new \InvalidArgumentException("To update a Bill, the object must include its Id and SyncToken.");
        }
    
        error_log("QBO Utility Lib: Attempting to update Bill ID: {$billObject->Id} with SyncToken: {$billObject->SyncToken}");
    
        // The DataService->Update() method handles the API call
        $updatedBill = $this->dataService->Update($billObject);
        
        error_log("QBO Utility Lib: Successfully updated Bill ID: {$billObject->Id}. New SyncToken: {$updatedBill->SyncToken}");
    
        return $updatedBill;
    }

    /**
     * Gets the FULL transaction history (Invoices AND Payments) using the TransactionList report.
     * This function's logic is based on the proven structure of the TransactionList debug log.
     *
     * @param string $customerId The ID of the Customer.
     * @return array A list of all transactions.
     */
    public function getFullTransactionHistory(string $customerId): array
    {
        $transactions = [];
        if (empty(trim($customerId))) return $transactions;
    
        error_log("QBO Lib: Running TransactionList for FULL HISTORY. Customer ID: {$customerId}");
        try {
            $serviceContext = $this->dataService->getServiceContext();
            $reportService = new ReportService($serviceContext);
            $reportService->setCustomer($customerId);
            $reportService->setStartDate('2000-01-01'); // Wide date range to get all history
            $reportService->setEndDate(date('Y-m-d'));
            
            // This is the crucial part: we are NOT filtering by paid/unpaid status.
            $report = $reportService->executeReport('TransactionList');
    
            if (!$report || empty($report->Rows->Row)) return $transactions;
    
            $header = [];
            foreach ($report->Columns->Column as $column) {
                // This normalization now correctly handles the '.' from 'No.'
                $header[] = str_replace([' ', '/', '-', '.'], '_', strtolower($column->ColTitle));
            }
    
            foreach ($report->Rows->Row as $row) {
                if (isset($row->type) && $row->type === 'Data') {
                    $transaction = [];
                    foreach ($row->ColData as $colIndex => $col) {
                        $key = $header[$colIndex] ?? 'column_' . $colIndex;
                        $transaction[$key] = $col->value;
                    }
                    $transactions[] = $transaction;
                }
            }
        } catch (Exception $e) {
            error_log("QBO Lib: Could not get transaction list. Error: " . $e->getMessage());
        }
        
        // Sort transactions by date for correct display.
        usort($transactions, function($a, $b) {
            return strtotime($a['date']) <=> strtotime($b['date']);
        });
        return $transactions;
    }
    
    /**
     * Gets detailed financial data (Summary numbers, Open Balances, Due Dates, and Class/Quote # for each transaction).
     * This uses a combination of the CustomerBalanceDetail report and direct Invoice queries for richness.
     *
     * @param string $customerId The ID of the Customer.
     * @return array An array containing 'summary' and 'invoice_details'.
     */
    public function getFinancialDetailData(string $customerId): array
    {
        $result = [
            'summary' => ['total_balance' => 0.0, 'overdue_balance' => 0.0],
            'invoice_details' => [] // We will store open balance, due date, AND quote_no, keyed by Invoice No.
        ];
        if (empty(trim($customerId))) return $result;
    
        error_log("QBO Lib: Running CustomerBalanceDetail for FINANCIAL DETAILS. Customer ID: {$customerId}");
        try {
            // --- Part 1: Get Open Balances and Due Dates from the Report ---
            $serviceContext = $this->dataService->getServiceContext();
            $reportService = new ReportService($serviceContext);
            $reportService->setCustomer($customerId);
            $report = $reportService->executeReport('CustomerBalanceDetail');
    
            if ($report && !empty($report->Rows->Row)) {
                $header = [];
                foreach ($report->Columns->Column as $column) {
                    $header[] = str_replace([' ', '/', '-', '.'], '_', strtolower($column->ColTitle));
                }
                
                $overdueTotal = 0.0;
        
                foreach ($report->Rows->Row as $sectionRow) {
                    if (isset($sectionRow->Rows) && isset($sectionRow->Rows->Row)) {
                        foreach ($sectionRow->Rows->Row as $dataRow) {
                            if (isset($dataRow->type) && $dataRow->type === 'Data') {
                                $transaction = [];
                                foreach ($dataRow->ColData as $colIndex => $col) {
                                    $key = $header[$colIndex] ?? 'column_' . $colIndex;
                                    $transaction[$key] = $col->value;
                                }
                                
                                $invoiceNum = $transaction['no_'] ?? null;
                                $openBalance = (float)($transaction['open_balance'] ?? 0);
                                
                                if ($invoiceNum && !empty(trim($invoiceNum))) {
                                    $result['invoice_details'][$invoiceNum] = [
                                        'open_balance' => $openBalance,
                                        'due_date'     => $transaction['due_date'] ?? null,
                                        'quote_no'     => 'N/A' // Default value
                                    ];
                                }
        
                                if ($openBalance > 0 && !empty($transaction['due_date'])) {
                                    if (new DateTime($transaction['due_date']) < new DateTime('today')) {
                                        $overdueTotal += $openBalance;
                                    }
                                }
                            }
                        }
                    }
                }
                 $result['summary']['overdue_balance'] = $overdueTotal;
            }

            // --- Part 2: Get ClassRef/Quote No from direct Invoice queries ---
            $all_customer_invoices = $this->dataService->Query("SELECT Id, DocNumber, Line FROM Invoice WHERE CustomerRef = '{$customerId}' MAXRESULTS 1000");
            
            if ($all_customer_invoices) {
                $class_ids_to_find = [];
                $invoice_id_to_class_id = [];

                foreach ($all_customer_invoices as $invoice) {
                    if (!empty($invoice->Line)) {
                        $lines = is_array($invoice->Line) ? $invoice->Line : [$invoice->Line];
                        foreach($lines as $line) {
                            if (isset($line->SalesItemLineDetail->ClassRef)) {
                                $class_id = $line->SalesItemLineDetail->ClassRef;
                                $class_ids_to_find[] = $class_id;
                                // Map the Invoice's DocNumber to the Class ID
                                if ($invoice->DocNumber) {
                                    $invoice_id_to_class_id[$invoice->DocNumber] = $class_id;
                                }
                                break; // Found first class, move to next invoice
                            }
                        }
                    }
                }

                // Translate Class IDs to Class Names (Quote Numbers)
                $class_id_to_name_map = [];
                $unique_class_ids = array_unique($class_ids_to_find);
                if (!empty($unique_class_ids)) {
                    $id_list_string = "'" . implode("','", $unique_class_ids) . "'";
                    $class_query = "SELECT Id, Name FROM Class WHERE Id IN ({$id_list_string})";
                    $qbo_classes = $this->dataService->Query($class_query);
                    foreach($qbo_classes as $class) {
                        $class_id_to_name_map[$class->Id] = $class->Name;
                    }
                }

                // --- Part 3: Merge the Class/Quote data into our result array ---
                foreach ($invoice_id_to_class_id as $docNum => $classId) {
                    if (isset($result['invoice_details'][$docNum])) {
                        $result['invoice_details'][$docNum]['quote_no'] = $class_id_to_name_map[$classId] ?? 'N/A';
                    }
                }
            }
            
            // Get the final total balance from the customer object itself
            $customerObject = $this->queryEntityByField('Customer', 'Id', $customerId);
            $result['summary']['total_balance'] = $customerObject ? (float)$customerObject->Balance : 0.0;
    
        } catch (Exception $e) {
            error_log("QBO Lib: Could not get financial detail data. Error: " . $e->getMessage());
        }
        return $result;
    }
    
    /**
     * Gets detailed financial data for a VENDOR (Summary, Open Balances, and Due Dates).
     * This uses the VendorBalanceDetail report.
     *
     * @param string $vendorId The ID of the Vendor.
     * @return array An array containing 'summary' and 'bill_details'.
     */
    public function getVendorFinancialDetailData(string $vendorId): array
    {
        $result = [
            'summary' => ['total_balance' => 0.0, 'overdue_balance' => 0.0],
            'bill_details' => [] // We will store open balance and due date, keyed by Bill No.
        ];
        if (empty(trim($vendorId))) return $result;
    
        error_log("QBO Lib: Running VendorBalanceDetail for FINANCIAL DETAILS. Vendor ID: {$vendorId}");
        try {
            $serviceContext = $this->dataService->getServiceContext();
            $reportService = new ReportService($serviceContext);
            $reportService->setVendor($vendorId); // Use setVendor for vendors
            $report = $reportService->executeReport('VendorBalanceDetail'); // The vendor equivalent report
    
            if (!$report || empty($report->Rows->Row)) return $result;
            
            $header = [];
            foreach ($report->Columns->Column as $column) {
                $header[] = str_replace([' ', '/', '-', '.'], '_', strtolower($column->ColTitle));
            }
            
            $overdueTotal = 0.0;
    
            foreach ($report->Rows->Row as $sectionRow) {
                if (isset($sectionRow->Rows) && isset($sectionRow->Rows->Row)) {
                    foreach ($sectionRow->Rows->Row as $dataRow) {
                        if (isset($dataRow->type) && $dataRow->type === 'Data') {
                            $transaction = [];
                            foreach ($dataRow->ColData as $colIndex => $col) {
                                $key = $header[$colIndex] ?? 'column_' . $colIndex;
                                $transaction[$key] = $col->value;
                            }
                            
                            $billNum = $transaction['no_'] ?? null;
                            $openBalance = (float)($transaction['open_balance'] ?? 0);
                            
                            if ($billNum && !empty(trim($billNum))) {
                                $result['bill_details'][$billNum] = [
                                    'open_balance' => $openBalance,
                                    'due_date'     => $transaction['due_date'] ?? null
                                ];
                            }
    
                            if ($openBalance > 0 && !empty($transaction['due_date'])) {
                                if (new DateTime($transaction['due_date']) < new DateTime('today')) {
                                    $overdueTotal += $openBalance;
                                }
                            }
                        }
                    }
                }
            }
            
            $vendorObject = $this->queryEntityByField('Vendor', 'Id', $vendorId);
            $result['summary']['total_balance'] = $vendorObject ? (float)$vendorObject->Balance : 0.0;
            $result['summary']['overdue_balance'] = $overdueTotal;
    
        } catch (Exception $e) {
            error_log("QBO Lib: Could not get VENDOR financial detail data. Error: " . $e->getMessage());
        }
        return $result;
    }
    
    /**
     * Records a newly created QBO entity into the sync history table.
     *
     * @param string $entityId The ID of the created QBO entity.
     * @param string $entityType The type of entity (e.g., "Customer", "Invoice").
     * @param string|null $entityNumber The human-readable number (DocNumber).
     * @param string|null $orgName The DisplayName of the related customer or vendor.
     * @param string|null $sourceId The source system identifier (e.g., quote no).
     */
    private function recordSyncHistory(string $entityId, string $entityType, ?string $entityNumber, ?string $orgName = null, ?string $sourceId = null)
    {
        if (!$this->dbConn) {
            error_log("QBO Lib Recording Error: Database connection is not available.");
            return;
        }
    
        // Updated SQL to include the new 'qbo_entity_number' column
        $sql = "INSERT INTO tdu_qbo_sync_history (qbo_entity_id, qbo_entity_type, qbo_entity_number, organisation_name, source_identifier) 
                VALUES (?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE qbo_entity_number = VALUES(qbo_entity_number), sync_datetime = CURRENT_TIMESTAMP";
                
        try {
            $stmt = $this->dbConn->prepare($sql);
            if ($stmt === false) {
                throw new Exception("Failed to prepare statement: " . $this->dbConn->error);
            }
            
            $stmt->bind_param("sssss", $entityId, $entityType, $entityNumber, $orgName, $sourceId);
            $stmt->execute();
            $stmt->close();
            
            error_log("QBO Lib: Successfully recorded sync history for {$entityType} ID {$entityId} (No: {$entityNumber}).");
    
        } catch (Exception $e) {
            error_log("QBO Lib Recording Error: " . $e->getMessage());
        }
    }
    
    /**
     * Specifically updates an existing sync history record with the correct org name and source ID.
     * This is useful after a workflow has completed.
     *
     * @param string $entityId The ID of the QBO entity.
     * @param string|null $entityNumber The human-readable number (DocNumber).
     * @param string|null $orgName The DisplayName of the related customer or vendor.
     * @param string|null $sourceId The source system identifier.
     */
    public function updateInvoiceSyncHistory(string $entityId, ?string $entityNumber, ?string $orgName, ?string $sourceId)
    {
        if (!$this->dbConn) return;
    
        // Use UPDATE instead of INSERT...ON DUPLICATE KEY UPDATE for clarity
        $sql = "UPDATE tdu_qbo_sync_history 
                SET 
                    qbo_entity_number = ?,
                    organisation_name = ?,
                    source_identifier = ?
                WHERE 
                    qbo_entity_id = ? AND qbo_entity_type = 'Invoice'";
                    
        try {
            $stmt = $this->dbConn->prepare($sql);
            if ($stmt === false) throw new Exception("Failed to prepare statement: " . $this->dbConn->error);
            
            $stmt->bind_param("ssss", $entityNumber, $orgName, $sourceId, $entityId);
            $stmt->execute();
            $stmt->close();
            
            error_log("QBO Lib: Updated sync history for Invoice ID {$entityId}.");
    
        } catch (Exception $e) {
            error_log("QBO Lib Recording Error during update: " . $e->getMessage());
        }
    }
    
    /**
     * Finds a QBO entity (e.g., Customer, Vendor) using a fuzzy name matching algorithm.
     * This is ideal for matching names from an external system that may have minor variations.
     *
     * @param string $entityType The type of entity to search for (e.g., 'Customer', 'Vendor').
     * @param string $targetName The name you are trying to find.
     * @param int $similarityThreshold The percentage similarity required to consider a name a match (default: 80).
     * @param bool $activeOnly Whether to search only active entities (default: true).
     * @return \QuickBooksOnline\API\Data\IPPIdType|null The best matching entity object, or null if no confident match is found.
     */
    public function findEntityByFuzzyName(string $entityType, string $targetName, int $similarityThreshold = 80, bool $activeOnly = true)
    {
        if (empty(trim($targetName))) {
            throw new \InvalidArgumentException("Target name for fuzzy search cannot be empty.");
        }

        error_log("QBO Lib: Starting fuzzy search for {$entityType} with name '{$targetName}'. Threshold: {$similarityThreshold}%");

        // 1. Fetch all entities from QBO to compare against
        $allEntities = [];
        $startPosition = 1;
        $activeClause = $activeOnly ? " WHERE Active = true" : "";

        while (true) {
            try {
                // We only need Id and DisplayName for matching, which is more efficient
                $query = "SELECT Id, DisplayName FROM {$entityType}{$activeClause} STARTPOSITION {$startPosition} MAXRESULTS 1000";
                $entitiesBatch = $this->dataService->Query($query);
            } catch (IdsException $e) {
                error_log("QBO Lib Fuzzy Search Error: Could not fetch {$entityType} list. " . $e->getMessage());
                return null; // Cannot proceed if we can't get the list
            }

            if (empty($entitiesBatch)) {
                break; // No more entities to fetch
            }
            $allEntities = array_merge($allEntities, $entitiesBatch);
            $startPosition += count($entitiesBatch);
        }

        if (empty($allEntities)) {
            error_log("QBO Lib: No entities of type '{$entityType}' found in QBO to perform fuzzy match.");
            return null;
        }
        
        error_log("QBO Lib: Fetched " . count($allEntities) . " total entities. Now performing fuzzy comparison.");

        // 2. Find the best match from the list
        $bestMatchName = $this->findBestMatchInList($targetName, $allEntities, $similarityThreshold);

        // 3. If a good match was found, fetch the full object and return it
        if ($bestMatchName) {
            error_log("QBO Lib: Confident match found: '{$bestMatchName}'. Fetching full entity object.");
            // Use the precise query to get the full, fresh object
            return $this->queryEntityByField($entityType, 'DisplayName', $bestMatchName);
        }

        error_log("QBO Lib: No match found for '{$targetName}' above the {$similarityThreshold}% threshold.");
        return null; // No confident match found
    }

    /**
     * (Private Helper) Generates variations of a name for more robust matching.
     * @param string $name The name to generate variations for.
     * @return array An array of unique name variations.
     */
    private function generateSearchVariations(string $name): array {
        // Common business suffixes and noise words to remove
        $noise = [' pty ltd', ' pty', ' ltd', ' inc', ' llc', ' group', ' hotel', '& co', ' limited'];
        $searchTerms = [trim($name)];
        
        // Handle "trading as" (t/a)
        if (preg_match('/t\s?\/?\s?a\s(.+)/i', $name, $matches)) {
            array_unshift($searchTerms, trim($matches[1]));
        }
        
        // Add a version with noise words removed
        $cleanName = str_ireplace($noise, '', $name);
        if (trim($cleanName) !== trim($name)) {
            $searchTerms[] = trim($cleanName);
        }
        
        // Add a version with parenthetical text removed
        $noParenthesesName = trim(preg_replace('/\s*\(.*?\)\s*/', ' ', $name));
        if (trim($noParenthesesName) !== trim($name)) {
            $searchTerms[] = trim($noParenthesesName);
        }
        
        return array_unique($searchTerms);
    }

    /**
     * (Private Helper) Compares a target name against a list of QBO entities to find the best match.
     *
     * @param string $targetName The name we are looking for.
     * @param array $entityList An array of QBO entity objects (must have a DisplayName property).
     * @param int $threshold The similarity percentage required to be considered a match.
     * @return string|null The DisplayName of the best match, or null.
     */
    private function findBestMatchInList(string $targetName, array $entityList, int $threshold): ?string {
        $bestMatch = null;
        $highestPercentage = 0.0;
        
        $targetVariations = $this->generateSearchVariations($targetName);

        foreach ($entityList as $entity) {
            if (empty($entity->DisplayName)) continue;

            $dbEntityVariations = $this->generateSearchVariations($entity->DisplayName);

            foreach ($targetVariations as $targetTerm) {
                foreach ($dbEntityVariations as $dbTerm) {
                    similar_text(strtolower($targetTerm), strtolower($dbTerm), $percent);
                    // If we find a near-perfect match, return immediately
                    if ($percent > 99) {
                        return $entity->DisplayName;
                    }
                    if ($percent > $highestPercentage) {
                        $highestPercentage = $percent;
                        $bestMatch = $entity->DisplayName;
                    }
                }
            }
        }
        
        // Only return a match if it met our confidence threshold
        return ($highestPercentage > $threshold) ? $bestMatch : null;
    }
    
    /**
     * Gets the FULL transaction history for a specific Vendor using the TransactionList report.
     *
     * @param string $vendorId The ID of the Vendor.
     * @return array A list of all transactions for the vendor.
     */
    public function getVendorTransactionHistory(string $vendorId): array
    {
        $transactions = [];
        if (empty(trim($vendorId))) return $transactions;
    
        error_log("QBO Lib: Running TransactionList for VENDOR. Vendor ID: {$vendorId}");
        try {
            $serviceContext = $this->dataService->getServiceContext();
            $reportService = new ReportService($serviceContext);
            $reportService->setVendor($vendorId); // This is the key change: setVendor()
            $reportService->setStartDate('2000-01-01'); // Wide date range for all history
            $reportService->setEndDate(date('Y-m-d'));
            
            $report = $reportService->executeReport('TransactionList');
    
            if (!$report || empty($report->Rows->Row)) return $transactions;
    
            $header = [];
            foreach ($report->Columns->Column as $column) {
                $header[] = str_replace([' ', '/', '-', '.'], '_', strtolower($column->ColTitle));
            }
    
            foreach ($report->Rows->Row as $row) {
                if (isset($row->type) && $row->type === 'Data') {
                    $transaction = [];
                    foreach ($row->ColData as $colIndex => $col) {
                        $key = $header[$colIndex] ?? 'column_' . $colIndex;
                        $transaction[$key] = $col->value;
                    }
                    $transactions[] = $transaction;
                }
            }
        } catch (Exception $e) {
            error_log("QBO Lib: Could not get VENDOR transaction list. Error: " . $e->getMessage());
        }
        
        // Sort transactions by date for correct display.
        usort($transactions, function($a, $b) {
            return strtotime($a['date']) <=> strtotime($b['date']);
        });
        return $transactions;
    }
    
    /**
     * (NEW FOR REPORTS) Generates search-friendly variations of a name.
     * @param string $name The name to generate variations for.
     * @return array An array of unique name variations.
     */
    private function _generateReportSearchVariations(string $name): array {
        $noise = [' pty ltd', ' pty', ' ltd', ' inc', ' llc', ' group', ' hotel', '& co', ' limited'];
        $searchTerms = [trim($name)];
        
        if (preg_match('/t\s?\/?\s?a\s(.+)/i', $name, $matches)) {
            array_unshift($searchTerms, trim($matches[1]));
        }
        
        $cleanName = str_ireplace($noise, '', $name);
        if (trim($cleanName) !== trim($name)) {
            $searchTerms[] = trim($cleanName);
        }
        
        $noParenthesesName = trim(preg_replace('/\s*\(.*?\)\s*/', ' ', $name));
        if (trim($noParenthesesName) !== trim($name)) {
            $searchTerms[] = trim($noParenthesesName);
        }
        
        return array_unique($searchTerms);
    }

    /**
     * (NEW FOR REPORTS) Builds a searchable map from a list of QBO entities.
     * @param array $entities An array of QBO entity objects (e.g., from a DataService->Query call).
     * @return array A map where keys are original DisplayNames and values are arrays of search variations.
     */
    public function buildReportSearchMap(array $entities): array {
        $searchMap = [];
        foreach ($entities as $entity) {
            if (!empty($entity->DisplayName)) {
                $searchMap[$entity->DisplayName] = $this->_generateReportSearchVariations($entity->DisplayName);
            }
        }
        return $searchMap;
    }

    /**
     * (NEW FOR REPORTS) Finds the best matching QBO entity name for a given target name.
     *
     * @param string $targetName The name to search for (e.g., from Vtiger).
     * @param array $qboSearchMap The pre-built search map from buildReportSearchMap().
     * @param int $threshold The similarity percentage required to be considered a match.
     * @return string|null The DisplayName of the best match, or null.
     */
    public function findBestEntityMatch(string $targetName, array $qboSearchMap, int $threshold = 80): ?string {
        $bestMatch = null;
        $highestPercentage = 0.0;
        
        $targetVariations = $this->_generateReportSearchVariations($targetName);

        foreach ($qboSearchMap as $originalQboName => $qboSearchTerms) {
            foreach ($targetVariations as $targetTerm) {
                foreach ($qboSearchTerms as $qboTerm) {
                    similar_text(strtolower($targetTerm), strtolower($qboTerm), $percent);
                    if ($percent > 99) {
                        return $originalQboName; // Return immediately on a near-perfect match
                    }
                    if ($percent > $highestPercentage) {
                        $highestPercentage = $percent;
                        $bestMatch = $originalQboName;
                    }
                }
            }
        }
        
        return ($highestPercentage >= $threshold) ? $bestMatch : null;
    }
    
    
} // End of QBOUtilityLibrary class