<?php
// qbo_income_by_customer_report.php
// FINAL CORRECTED VERSION - Panel has all filters, columns, and dynamic counting.

use QuickBooksOnline\API\Core\OAuth\OAuth2\OAuth2LoginHelper;
use QuickBooksOnline\API\DataService\DataService;
use QuickBooksOnline\API\ReportService\ReportService;

require_once __DIR__ . '/../vendor/autoload.php';
require_once __DIR__ . '/qbo_functions.php';
require_once __DIR__ . '/config/qbo_config.php';
global $qboBaseConfig;

if (!function_exists('getQBOTransactionLink')) { function getQBOTransactionLink($txnId, $txnType, $baseUrl) { if (empty($txnId)) return '#'; $isSandbox = (strcasecmp($baseUrl, 'Development') == 0); $qboDomain = $isSandbox ? "https://app.sandbox.qbo.intuit.com/app/" : "https://app.qbo.intuit.com/app/"; $page = strtolower(str_replace(' ', '', $txnType)); if (in_array($page, ['invoice', 'salesreceipt', 'creditmemo', 'payment'])) { return $qboDomain . $page . "?txnId=" . htmlspecialchars($txnId); } return '#'; } }

// --- Report Filters & Variables ---
$finYearStart = (date('m') < 7) ? date('Y-07-01', strtotime('-1 year')) : date('Y-07-01');
$startDate = isset($_GET['start_date']) && !empty($_GET['start_date']) ? $_GET['start_date'] : $finYearStart;
$endDate = isset($_GET['end_date']) && !empty($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-d');
$accountingMethod = isset($_GET['accounting_method']) ? $_GET['accounting_method'] : 'Accrual';

$dataService = null;
$reportData = [];
$grandTotalIncome = 0.00;

// --- QBO Connection & Token Refresh ---
$tokenStorageFile = __DIR__ . '/tokens/qbo_token.json';
$currentTokens = file_exists($tokenStorageFile) ? json_decode(file_get_contents($tokenStorageFile), true) : null;
if ($currentTokens && !empty($currentTokens['refresh_token'])) {
    try {
        $oauth2LoginHelper = new OAuth2LoginHelper($qboBaseConfig['ClientID'], $qboBaseConfig['ClientSecret']);
        $refreshedAccessTokenObj = $oauth2LoginHelper->refreshAccessTokenWithRefreshToken($currentTokens['refresh_token']);
        $newAccessToken = $refreshedAccessTokenObj->getAccessToken();
        $newRefreshToken = $refreshedAccessTokenObj->getRefreshToken();
        if ($newAccessToken && $newRefreshToken) {
            file_put_contents($tokenStorageFile, json_encode(['access_token' => $newAccessToken, 'refresh_token' => $newRefreshToken], JSON_PRETTY_PRINT));
            $currentTokens['access_token'] = $newAccessToken;
        }
    } catch (Exception $e) { /* Suppress error */ }
}

if ($currentTokens && !empty($currentTokens['access_token'])) {
    $qboBaseConfig['accessTokenKey'] = $currentTokens['access_token'];
    try { $dataService = DataService::Configure($qboBaseConfig); } catch (Exception $e) { $dataService = null; }
}

// --- DATA FETCHING FROM QBO ---
if ($dataService) {
    try {
        $customerTotals = []; 
        $periodStartDate = new DateTime($startDate);
        $periodEndDate = new DateTime($endDate);
        $currentStartDate = clone $periodStartDate;

        while ($currentStartDate <= $periodEndDate) {
            $currentEndDate = clone $currentStartDate;
            $currentEndDate->modify('last day of this month');
            if ($currentEndDate > $periodEndDate) { $currentEndDate = $periodEndDate; }

            $reportService = new ReportService($dataService->getServiceContext());
            $reportService->setStartDate($currentStartDate->format('Y-m-d'));
            $reportService->setEndDate($currentEndDate->format('Y-m-d'));
            $reportService->setAccountingMethod($accountingMethod);
            
            $report = $reportService->executeReport("CustomerIncome");

            if ($report && isset($report->Rows->Row)) {
                $rows = is_array($report->Rows->Row) ? $report->Rows->Row : [$report->Rows->Row];
                foreach ($rows as $row) {
                    $customerId = null; $customerName = null; $income = 0;
                    if (isset($row->ColData) && !isset($row->Summary)) {
                        $income = (float)($row->ColData[1]->value ?? 0);
                        if ($income != 0) { $customerId = $row->ColData[0]->id; $customerName = $row->ColData[0]->value; }
                    } elseif (isset($row->Header) && isset($row->Summary)) {
                        $income = (float)($row->Summary->ColData[1]->value ?? 0);
                        if ($income != 0) { $customerId = $row->Header->ColData[0]->id; $customerName = $row->Header->ColData[0]->value; }
                    }
                    if ($customerId) {
                        if (!isset($customerTotals[$customerId])) {
                            $customerTotals[$customerId] = ['id' => $customerId, 'name' => $customerName, 'income' => 0];
                        }
                        $customerTotals[$customerId]['income'] += $income;
                    }
                }
            }
            $currentStartDate->modify('first day of next month');
        }

        $reportData = array_values($customerTotals);
        $reportData = array_filter($reportData, function($customer) { return $customer['income'] != 0; });
        usort($reportData, function($a, $b) { return $b['income'] <=> $a['income']; });
        $grandTotalIncome = array_sum(array_column($reportData, 'income'));

    } catch (Exception $e) { 
        error_log("Failed to fetch or process multi-part CustomerIncome report from QBO: " . $e->getMessage());
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Income by Customer Summary</title>
<style>
    body { font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif; }
    .report-header { margin-bottom: 20px; }
    .filter-form { display: flex; flex-wrap: wrap; gap: 15px; align-items: center; padding: 15px; background-color: #f7f7f7; border: 1px solid #ddd; border-radius: 5px; margin-bottom: 20px; }
    .filter-form div { display: flex; flex-direction: column; }
    .filter-form label { font-weight: bold; font-size: 12px; margin-bottom: 4px; color: #333; }
    .filter-form input[type="date"], .filter-form select { padding: 8px; border: 1px solid #ccc; border-radius: 4px; font-size: 14px; }
    .filter-form button { padding: 10px 15px; background-color: #007bff; color: white; border: none; border-radius: 4px; cursor: pointer; font-size: 14px; align-self: flex-end; }
    .sortable { cursor: pointer; position: relative; padding-right: 20px !important; }
    .sortable::after, .sortable::before { content: ""; position: absolute; right: 5px; border: 4px solid transparent; opacity: 0.5; }
    .sortable::before { top: 40%; border-bottom-color: #666; }
    .sortable::after { bottom: 40%; border-top-color: #666; }
    .sortable.asc::before { opacity: 1; }
    .sortable.desc::after { opacity: 1; }
    .filter-form .compare-btn { background-color: #28a745; }
    .comparison-container { display: flex; gap: 20px; width: 100%; transition: all 0.5s ease-in-out; }
    .report-panel { width: 100%; transition: width 0.5s ease-in-out; flex-shrink: 0; }
    .db-quotes-panel { width: 0; overflow: hidden; transition: width 0.5s ease-in-out; flex-shrink: 0; }
    .comparison-container.comparison-active .report-panel { width: 50%; }
    .comparison-container.comparison-active .db-quotes-panel { width: 50%; }
    .db-quotes-panel .panel-content { max-height: 80vh; overflow-y: auto; border: 1px solid #ddd; border-radius: 5px; background-color: #fff; }
    .db-quotes-panel h3 { padding: 10px 12px; margin: 0; background-color: #f7f7f7; border-bottom: 1px solid #ddd; font-size: 1.1em; color: #333; position: sticky; top: 0; z-index: 2; }
    .centered-loader { text-align: center; padding: 50px; font-size: 1.1em; color: #555; }
    .loading-spinner { display: inline-block; width: 20px; height: 20px; border: 3px solid #ccc; border-top-color: #007bff; border-radius: 50%; animation: spin 1s linear infinite; vertical-align: middle; margin-right: 10px; }
    @keyframes spin { to { transform: rotate(360deg); } }
    .db-quotes-panel table { width: 100%; }
    .db-quotes-panel th.currency, .db-quotes-panel td.currency { text-align: right; }
    .db-panel-filters { padding: 10px; background-color: #f7f7f7; border-bottom: 1px solid #ddd; display: flex; flex-wrap: wrap; gap: 15px; align-items: center; position: sticky; top: 43px; z-index: 2; }
    .db-panel-filters label { font-weight: bold; font-size: 12px; }
    .db-panel-filters select { padding: 5px; font-size: 12px; border-radius: 4px; border: 1px solid #ccc; }
    .multiselect-dropdown { position: relative; display: inline-block; }
    .multiselect-dropdown-btn { width: 150px; text-align: left; background-color: white; padding: 5px; font-size: 12px; border: 1px solid #ccc; border-radius: 4px; cursor: pointer; }
    .multiselect-dropdown-content { display: none; position: absolute; background-color: #f9f9f9; min-width: 150px; box-shadow: 0px 8px 16px 0px rgba(0,0,0,0.2); z-index: 10; border-radius: 4px; padding: 5px; max-height: 200px; overflow-y: auto; }
    .multiselect-dropdown-content label { display: block; padding: 5px 10px; cursor: pointer; font-weight: normal; font-size: 12px; }
    .multiselect-dropdown-content label:hover { background-color: #f1f1f1; }
</style>
</head>
<body>

<div class="report-header">
    <h2>Income by Customer Summary</h2>
    <p>For period: <?php echo htmlspecialchars(date("F j, Y", strtotime($startDate))) . " - " . htmlspecialchars(date("F j, Y", strtotime($endDate))); ?> </p>
</div>

<form method="get" class="filter-form">
    <input type="hidden" name="report" value="income_by_customer">
    <div>
        <label>Report period:</label>
        <select id="report_period_selector">
            <option value="This_month_to_date">This month to date</option><option value="This_Month">This Month</option>
            <option value="This_Fiscal_Year" selected>This Financial Year</option><option value="Last_Month">Last Month</option>
            <option value="Last_Fiscal_Year">Last Financial Year</option><option value="custom">Custom</option>
        </select>
    </div>
    <div><label>From:</label><input type="date" name="start_date" id="start_date" value="<?php echo htmlspecialchars($startDate); ?>"></div>
    <div><label>To:</label><input type="date" name="end_date" id="end_date" value="<?php echo htmlspecialchars($endDate); ?>"></div>
    <div>
        <label>Accounting Method:</label>
        <select name="accounting_method">
            <option value="Accrual" <?php if($accountingMethod == 'Accrual') echo 'selected'; ?>>Accrual</option>
            <option value="Cash" <?php if($accountingMethod == 'Cash') echo 'selected'; ?>>Cash</option>
        </select>
    </div>
    <button type="submit">Run Report</button>
    <button type="button" id="compare-db-btn" class="compare-btn">Compare with DB Quotes</button>
</form>

<div id="comparison-container" class="comparison-container">
    <section id="income-summary-report" class="report-panel">
        <table>
            <thead>
                <tr>
                    <th>Customer</th>
                    <th class="currency sortable desc" id="income-header">Income</th>
                </tr>
            </thead>
            <tbody id="report-body">
                <?php if (!empty($reportData)): ?>
                    <?php foreach ($reportData as $customerData): ?>
                        <tr class="customer-row" data-income="<?php echo $customerData['income']; ?>">
                            <td><strong>
                                <a href="?report=sales_by_customer_detail&customer_id=<?php echo $customerData['id']; ?>&start_date=<?php echo $startDate; ?>&end_date=<?php echo $endDate; ?>&accounting_method=<?php echo $accountingMethod; ?>">
                                    <?php echo htmlspecialchars($customerData['name']); ?>
                                </a>
                            </strong></td>
                            <td class="currency income-cell"><?php echo number_format($customerData['income'], 2); ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php else: ?>
                    <tr><td colspan="2" class="no-data-message">No income data found for the selected period, or could not connect to QBO.</td></tr>
                <?php endif; ?>
            </tbody>
            <tfoot>
                <tr class="total-row">
                    <td><strong>TOTAL</strong></td>
                    <td class="currency" id="total-income"><strong><?php echo number_format($grandTotalIncome, 2); ?></strong></td>
                </tr>
            </tfoot>
        </table>
    </section>

    <section id="db-quotes-panel" class="db-quotes-panel">
        <div class="panel-content">
            <h3>DB Quotes (by Travel Date)</h3>
            <div class="db-panel-filters">
                <div>
                    <label>Country:</label>
                    <select id="db-country-filter"></select>
                </div>
                <div>
                    <label>Region:</label>
                    <div class="multiselect-dropdown" id="db-region-filter">
                        <button type="button" class="multiselect-dropdown-btn" id="db-region-filter-btn">All Regions</button>
                        <div class="multiselect-dropdown-content" id="db-region-filter-content"></div>
                    </div>
                </div>
                <div>
                    <label>Quote Stage:</label>
                    <div class="multiselect-dropdown" id="db-stage-filter">
                        <button type="button" class="multiselect-dropdown-btn" id="db-stage-filter-btn">All Stages</button>
                        <div class="multiselect-dropdown-content" id="db-stage-filter-content"></div>
                    </div>
                </div>
            </div>
            <div id="db-quotes-table-container"></div>
        </div>
    </section>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const incomeHeader = document.getElementById('income-header');
    const incomeTbody = document.getElementById('report-body');
    let incomeSortDirection = 'desc'; 

    const compareBtn = document.getElementById('compare-db-btn');
    const comparisonContainer = document.getElementById('comparison-container');
    const dbQuotesTableContainer = document.getElementById('db-quotes-table-container');
    const startDate = '<?php echo $startDate; ?>';
    const endDate = '<?php echo $endDate; ?>';
    let dbQuotesLoaded = false;
    let allDbQuotes = [];

    const countryFilter = document.getElementById('db-country-filter');
    const regionFilterBtn = document.getElementById('db-region-filter-btn');
    const regionFilterContent = document.getElementById('db-region-filter-content');
    const stageFilterBtn = document.getElementById('db-stage-filter-btn');
    const stageFilterContent = document.getElementById('db-stage-filter-content');
    let dbTableSort = { column: 'quoteCount', dir: 'desc', type: 'number' };

    if (incomeHeader) {
        incomeHeader.addEventListener('click', () => {
            incomeSortDirection = (incomeSortDirection === 'desc' ? 'asc' : 'desc');
            incomeHeader.classList.toggle('desc', incomeSortDirection === 'desc');
            incomeHeader.classList.toggle('asc', incomeSortDirection === 'asc');
            const rows = Array.from(incomeTbody.querySelectorAll('tr.customer-row'));
            rows.sort((rowA, rowB) => {
                const incomeA = parseFloat(rowA.dataset.income || -Infinity);
                const incomeB = parseFloat(rowB.dataset.income || -Infinity);
                return incomeSortDirection === 'desc' ? incomeB - incomeA : incomeA - incomeB;
            });
            rows.forEach(row => incomeTbody.appendChild(row));
        });
    }
    
    document.getElementById('report_period_selector').addEventListener('change', function() {
        const selectedPeriod = this.value; const today = new Date();
        let startDate, endDate;
        const formatDate = (d) => { const pad = (num) => num.toString().padStart(2, '0'); return `${d.getFullYear()}-${pad(d.getMonth() + 1)}-${pad(d.getDate())}`; }
        const finYearStartMonth = 6;
        switch (selectedPeriod) {
            case 'This_month_to_date': startDate = new Date(today.getFullYear(), today.getMonth(), 1); endDate = today; break;
            case 'This_Month': startDate = new Date(today.getFullYear(), today.getMonth(), 1); endDate = new Date(today.getFullYear(), today.getMonth() + 1, 0); break;
            case 'Last_Month': startDate = new Date(today.getFullYear(), today.getMonth() - 1, 1); endDate = new Date(today.getFullYear(), today.getMonth(), 0); break;
            case 'This_Fiscal_Year':
                let finYear = today.getFullYear();
                if (today.getMonth() < finYearStartMonth) finYear--;
                startDate = new Date(finYear, finYearStartMonth, 1);
                endDate = new Date(finYear + 1, finYearStartMonth, 0);
                break;
            case 'Last_Fiscal_Year':
                let lastFinYear = today.getFullYear();
                if (today.getMonth() < finYearStartMonth) lastFinYear--;
                startDate = new Date(lastFinYear - 1, finYearStartMonth, 1);
                endDate = new Date(lastFinYear, finYearStartMonth, 0);
                break;
            case 'custom': default: return;
        }
        if(startDate && endDate){
            document.getElementById('start_date').value = formatDate(startDate);
            document.getElementById('end_date').value = formatDate(endDate);
        }
    });

    compareBtn.addEventListener('click', async function() {
        comparisonContainer.classList.toggle('comparison-active');
        if (comparisonContainer.classList.contains('comparison-active') && !dbQuotesLoaded) {
            dbQuotesTableContainer.innerHTML = `<div class="centered-loader"><span class="loading-spinner"></span><p>Loading DB Quotes...</p></div>`;
            const url = `ajax_fetch_quote.php?start_date=${startDate}&end_date=${endDate}`;
            try {
                const response = await fetch(url);
                const result = await response.json();
                if (result.success) {
                    allDbQuotes = result.data;
                    populateDbFilters(allDbQuotes);
                    applyDbFiltersAndAggregate();
                    dbQuotesLoaded = true;
                } else {
                    dbQuotesTableContainer.innerHTML = `<p style="color:red;padding:20px;">Error: ${result.error || 'Failed to load data.'}</p>`;
                }
            } catch (e) {
                dbQuotesTableContainer.innerHTML = `<p style="color:red;padding:20px;">Error: Network request failed.</p>`;
            }
        }
    });

    function populateDbFilters(data) {
        const countries = new Set();
        const regions = new Set();
        const stages = new Set();
        data.forEach(row => {
            if(row.country && row.country !== 'N/A') countries.add(row.country);
            if(row.region && row.region !== 'N/A') regions.add(row.region);
            if(row.quotestage) stages.add(row.quotestage);
        });

        countryFilter.innerHTML = '<option value="all">All Countries</option>';
        Array.from(countries).sort().forEach(c => countryFilter.innerHTML += `<option value="${escapeHtml(c)}">${escapeHtml(c)}</option>`);

        regionFilterContent.innerHTML = '<label><input type="checkbox" value="all" checked> (Select All)</label>';
        Array.from(regions).sort().forEach(r => regionFilterContent.innerHTML += `<label><input type="checkbox" value="${escapeHtml(r)}" checked> ${escapeHtml(r)}</label>`);
        
        stageFilterContent.innerHTML = '<label><input type="checkbox" value="all" checked> (Select All)</label>';
        Array.from(stages).sort().forEach(s => stageFilterContent.innerHTML += `<label><input type="checkbox" value="${escapeHtml(s)}" checked> ${escapeHtml(s)}</label>`);
    }

    function applyDbFiltersAndAggregate() {
        const selectedCountry = countryFilter.value;
        const selectedRegions = Array.from(regionFilterContent.querySelectorAll('input:checked')).map(cb => cb.value);
        const selectedStages = Array.from(stageFilterContent.querySelectorAll('input:checked')).map(cb => cb.value);

        let filteredQuotes = [...allDbQuotes];

        if (selectedCountry !== 'all') {
            filteredQuotes = filteredQuotes.filter(row => row.country === selectedCountry);
        }
        if (!selectedRegions.includes('all')) {
            filteredQuotes = filteredQuotes.filter(row => selectedRegions.includes(row.region));
        }
        if (!selectedStages.includes('all')) {
            filteredQuotes = filteredQuotes.filter(quote => selectedStages.includes(quote.quotestage));
        }

        const customerSummary = {};
        filteredQuotes.forEach(quote => {
            const key = `${quote.organization_name}|${quote.region}|${quote.country}`;
            if (!customerSummary[key]) {
                customerSummary[key] = { 
                    organization_name: quote.organization_name,
                    region: quote.region,
                    country: quote.country,
                    quoteCount: 0, 
                    paxCount: 0 
                };
            }
            customerSummary[key].quoteCount++;
            customerSummary[key].paxCount += parseInt(quote.pax, 10);
        });

        let summaryArray = Object.values(customerSummary);

        const col = dbTableSort.column;
        const dir = dbTableSort.dir;
        const type = dbTableSort.type;
        summaryArray.sort((a, b) => {
            let valA = a[col], valB = b[col];
            if (type === 'number') { valA = parseFloat(valA); valB = parseFloat(valB); } 
            else { valA = (valA || '').toLowerCase(); valB = (valB || '').toLowerCase(); }
            if (valA < valB) return dir === 'asc' ? -1 : 1;
            if (valA > valB) return dir === 'asc' ? 1 : -1;
            return 0;
        });

        renderDbSummaryTable(summaryArray);
    }
    
    function renderDbSummaryTable(data) {
        if (data.length === 0) {
            dbQuotesTableContainer.innerHTML = `<p style="text-align:center;padding:20px;">No data matches the current filters.</p>`;
            return;
        }
        let totalPaxSum = 0;
        let totalQuotesSum = 0;
        let tableHtml = `
            <table>
                <thead>
                    <tr>
                        <th class="sortable" data-col="organization_name" data-type="string">Customer</th>
                        <th class="sortable" data-col="region" data-type="string">Region</th>
                        <th class="sortable" data-col="country" data-type="string">Country</th>
                        <th class="sortable currency" data-col="quoteCount" data-type="number">Quote Count</th>
                        <th class="sortable currency" data-col="paxCount" data-type="number">Total PAX</th>
                    </tr>
                </thead>
                <tbody>`;
        data.forEach(row => {
            totalQuotesSum += row.quoteCount;
            totalPaxSum += row.paxCount;
            tableHtml += `
                <tr>
                    <td>${escapeHtml(row.organization_name)}</td>
                    <td>${escapeHtml(row.region)}</td>
                    <td>${escapeHtml(row.country)}</td>
                    <td class="currency">${row.quoteCount}</td>
                    <td class="currency">${row.paxCount}</td>
                </tr>`;
        });
        tableHtml += `
                </tbody>
                <tfoot>
                    <tr class="total-row">
                        <td colspan="3"><strong>TOTAL (Filtered)</strong></td>
                        <td class="currency"><strong>${totalQuotesSum}</strong></td>
                        <td class="currency"><strong>${totalPaxSum}</strong></td>
                    </tr>
                </tfoot>
            </table>`;
        dbQuotesTableContainer.innerHTML = tableHtml;

        const currentSortHeader = dbQuotesTableContainer.querySelector(`th[data-col="${dbTableSort.column}"]`);
        if (currentSortHeader) {
            currentSortHeader.classList.add(dbTableSort.dir);
        }
    }

    countryFilter.addEventListener('change', applyDbFiltersAndAggregate);
    
    function setupMultiSelect(filterContent, filterBtn) {
        filterContent.addEventListener('change', function(e) {
            const allCheckbox = filterContent.querySelector('input[value="all"]');
            const btnText = filterBtn.id.includes('region') ? 'Regions' : 'Stages';

            if (e.target.value === 'all') {
                filterContent.querySelectorAll('input').forEach(cb => cb.checked = e.target.checked);
            } else {
                allCheckbox.checked = !filterContent.querySelector('input:not([value="all"]):not(:checked)');
            }
            
            const checkedCount = filterContent.querySelectorAll('input:not([value="all"]):checked').length;
            if (allCheckbox.checked) {
                filterBtn.textContent = `All ${btnText}`;
            } else if (checkedCount === 0) {
                filterBtn.textContent = `No ${btnText} Selected`;
            } else if (checkedCount === 1) {
                filterBtn.textContent = filterContent.querySelector('input:not([value="all"]):checked').value;
            } else {
                filterBtn.textContent = `${checkedCount} ${btnText} Selected`;
            }
            applyDbFiltersAndAggregate();
        });

        filterBtn.addEventListener('click', function(e) {
            e.stopPropagation();
            document.querySelectorAll('.multiselect-dropdown-content').forEach(el => {
                if(el !== filterContent) el.style.display = 'none';
            });
            filterContent.style.display = filterContent.style.display === 'block' ? 'none' : 'block';
        });
    }

    setupMultiSelect(regionFilterContent, regionFilterBtn);
    setupMultiSelect(stageFilterContent, stageFilterBtn);

    dbQuotesTableContainer.addEventListener('click', function(e) {
        const header = e.target.closest('th.sortable');
        if (!header) return;

        const column = header.dataset.col;
        const type = header.dataset.type;

        if (dbTableSort.column === column) {
            dbTableSort.dir = dbTableSort.dir === 'desc' ? 'asc' : 'desc';
        } else {
            dbTableSort.column = column;
            dbTableSort.dir = 'asc';
            dbTableSort.type = type;
        }
        applyDbFiltersAndAggregate();
    });
    
    document.addEventListener('click', function(e) {
        if (!e.target.closest('.multiselect-dropdown')) {
            document.querySelectorAll('.multiselect-dropdown-content').forEach(el => el.style.display = 'none');
        }
    });

    function escapeHtml(str) {
        if (!str) return '';
        const p = document.createElement("p");
        p.textContent = str;
        return p.innerHTML;
    }
});
</script>
</body>
</html>