<?php
// qbo_pl_report.php
// FINAL, CORRECTED VERSION: Fixes rendering errors by defensively checking for ColData property before access.

use QuickBooksOnline\API\Core\OAuth\OAuth2\OAuth2LoginHelper;
use QuickBooksOnline\API\DataService\DataService;
use QuickBooksOnline\API\ReportService\ReportService;

require_once __DIR__ . '/../vendor/autoload.php';
require_once __DIR__ . '/qbo_functions.php';
require_once __DIR__ . '/config/qbo_config.php';
global $qboBaseConfig;

if (!function_exists('saveTokens')) { function saveTokens($accessToken, $refreshToken) { $tokenStorageFile = __DIR__ . '/tokens/qbo_token.json'; file_put_contents($tokenStorageFile, json_encode(['access_token' => $accessToken, 'refresh_token' => $refreshToken, 'last_updated' => date('Y-m-d H:i:s')], JSON_PRETTY_PRINT)); return true; } }

// --- Report Filters & Variables ---
$finYearStart = (date('m') < 7) ? date('Y-07-01', strtotime('-1 year')) : date('Y-07-01');
$startDate = isset($_GET['start_date']) && !empty($_GET['start_date']) ? $_GET['start_date'] : $finYearStart;
$endDate = isset($_GET['end_date']) && !empty($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-d');
$accountingMethod = isset($_GET['accounting_method']) ? $_GET['accounting_method'] : 'Accrual';
$customerIdFilter = isset($_GET['customer']) ? $_GET['customer'] : 'ALL';

$dataService = null; $report = null; $allCustomers = []; $reportPeriodForDisplay = '';

// --- QBO Connection & Token Refresh ---
$tokenStorageFile = __DIR__ . '/tokens/qbo_token.json';
$currentTokens = file_exists($tokenStorageFile) ? json_decode(file_get_contents($tokenStorageFile), true) : null;
if ($currentTokens && !empty($currentTokens['refresh_token'])) {
    try {
        $oauth2LoginHelper = new OAuth2LoginHelper($qboBaseConfig['ClientID'], $qboBaseConfig['ClientSecret']);
        $refreshedAccessTokenObj = $oauth2LoginHelper->refreshAccessTokenWithRefreshToken($currentTokens['refresh_token']);
        $newAccessToken = $refreshedAccessTokenObj->getAccessToken();
        $newRefreshToken = $refreshedAccessTokenObj->getRefreshToken();
        if ($newAccessToken && $newRefreshToken) {
            saveTokens($newAccessToken, $newRefreshToken);
            $currentTokens['access_token'] = $newAccessToken;
        }
    } catch (Exception $e) { /* Suppress error */ }
}
if ($currentTokens && !empty($currentTokens['access_token'])) {
    $qboBaseConfig['accessTokenKey'] = $currentTokens['access_token'];
    try { $dataService = DataService::Configure($qboBaseConfig); } catch (Exception $e) { $dataService = null; }
} else { exit("QBO Authentication failed. Please reconnect."); }

// --- Data Fetching ---
if ($dataService) {
    try {
        $allCustomers = $dataService->Query("SELECT Id, DisplayName FROM Customer WHERE Active = true ORDERBY DisplayName ASC");
        $reportService = new ReportService($dataService->getServiceContext());
        $reportService->setStartDate($startDate);
        $reportService->setEndDate($endDate);
        $reportService->setAccountingMethod($accountingMethod);
        if ($customerIdFilter !== 'ALL') { $reportService->setCustomer($customerIdFilter); }
        $report = $reportService->executeReport("ProfitAndLoss");
        if($report && isset($report->Header)) {
            $reportPeriodForDisplay = htmlspecialchars(date("F j, Y", strtotime($report->Header->StartPeriod))) . " - " . htmlspecialchars(date("F j, Y", strtotime($report->Header->EndPeriod)));
        }
    } catch (Exception $e) { /* Suppress error */ }
}
?>

<div class="report-header">
    <h2>Profit and Loss</h2>
    <p>For period: <?php echo $reportPeriodForDisplay; ?></p>
    <?php if ($customerIdFilter !== 'ALL' && $report && isset($report->Header->Customer)) echo "<h3>For Customer: " . htmlspecialchars($report->Header->Customer) . "</h3>"; ?>
</div>

<form method="get" class="filter-form">
    <input type="hidden" name="report" value="pl_report">
    <div>
        <label>Report period:</label>
        <select id="report_period_selector">
            <option value="This_month_to_date">This month to date</option>
            <option value="This_Month">This Month</option>
            <option value="This_Fiscal_Year" selected>This Financial Year</option>
            <option value="Last_Month">Last Month</option>
            <option value="Last_Fiscal_Year">Last Financial Year</option>
            <option value="custom">Custom</option>
        </select>
    </div>
    <div><label>From:</label><input type="date" name="start_date" id="start_date" value="<?php echo htmlspecialchars($startDate); ?>"></div>
    <div><label>To:</label><input type="date" name="end_date" id="end_date" value="<?php echo htmlspecialchars($endDate); ?>"></div>
    <div>
        <label>Accounting Method:</label>
        <select name="accounting_method">
            <option value="Accrual" <?php if($accountingMethod == 'Accrual') echo 'selected'; ?>>Accrual</option>
            <option value="Cash" <?php if($accountingMethod == 'Cash') echo 'selected'; ?>>Cash</option>
        </select>
    </div>
     <div>
        <label>Customer:</label>
        <select name="customer">
            <option value="ALL">All Customers</option>
            <?php if(!empty($allCustomers)): ?>
                <?php foreach($allCustomers as $customer): ?>
                    <option value="<?php echo $customer->Id; ?>" <?php if($customerIdFilter == $customer->Id) echo 'selected'; ?>>
                        <?php echo htmlspecialchars($customer->DisplayName); ?>
                    </option>
                <?php endforeach; ?>
            <?php endif; ?>
        </select>
    </div>
    <button type="submit">Run Report</button>
</form>

<table>
    <thead><tr><th></th><th class="currency">Total</th></tr></thead>
    <tbody>
        <?php
        if ($report && isset($report->Rows->Row)) {
             foreach($report->Rows->Row as $row) {
                // This part handles the main section headers like "Income", "Expenses", etc. It is safe.
                if(isset($row->Header->ColData[0]->value)) echo "<tr class='pl-header-row'><td colspan='2'><strong>" . htmlspecialchars($row->Header->ColData[0]->value) . "</strong></td></tr>";

                // This part handles the individual data lines within a section.
                if(isset($row->Rows->Row)) {
                    $dataRows = is_array($row->Rows->Row) ? $row->Rows->Row : [$row->Rows->Row];
                    foreach($dataRows as $dataRow) {
                        // --- THIS IS THE FIX ---
                        // We check if the 'ColData' property exists before trying to use it.
                        // This safely skips over structural or empty rows that don't have data columns.
                        if (isset($dataRow->ColData)) {
                            $label = $dataRow->ColData[0]->value ?? ''; // Get label safely
                            $value = (float)($dataRow->ColData[1]->value ?? 0); // Get value safely
                            echo "<tr><td class='pl-indent-1'>" . htmlspecialchars($label) . "</td><td class='currency'>" . number_format($value, 2) . "</td></tr>";
                        }
                    }
                }
                
                // This part handles the summary line for a section, e.g., "Total Income". It is safe.
                if(isset($row->Summary->ColData[0]->value)) echo "<tr class='pl-total-row'><td><strong>" . htmlspecialchars($row->Summary->ColData[0]->value) . "</strong></td><td class='currency'><strong>" . number_format((float)($row->Summary->ColData[1]->value ?? 0), 2) . "</strong></td></tr>";
             }
        } else {
            echo "<tr><td colspan='2' class='no-data-message'>Could not load report data. Please check filters.</td></tr>";
        }
        ?>
    </tbody>
</table>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const periodSelect = document.getElementById('report_period_selector');
    const startDateInput = document.getElementById('start_date');
    const endDateInput = document.getElementById('end_date');

    periodSelect.addEventListener('change', function() {
        const selectedPeriod = this.value;
        if (selectedPeriod === 'custom') {
            return;
        }

        const today = new Date();
        let startDate, endDate;
        
        const formatDate = (d) => { const pad = (num) => num.toString().padStart(2, '0'); return `${d.getFullYear()}-${pad(d.getMonth() + 1)}-${pad(d.getDate())}`; }
        const finYearStartMonth = 6; // July

        switch (selectedPeriod) {
            case 'This_month_to_date':
                startDate = new Date(today.getFullYear(), today.getMonth(), 1);
                endDate = today;
                break;
            case 'This_Month':
                startDate = new Date(today.getFullYear(), today.getMonth(), 1);
                endDate = new Date(today.getFullYear(), today.getMonth() + 1, 0);
                break;
            case 'This_Fiscal_Year':
                let finYear = today.getFullYear();
                if (today.getMonth() < finYearStartMonth) { finYear--; }
                startDate = new Date(finYear, finYearStartMonth, 1);
                endDate = new Date(finYear + 1, finYearStartMonth, 0);
                break;
            case 'Last_Month':
                startDate = new Date(today.getFullYear(), today.getMonth() - 1, 1);
                endDate = new Date(today.getFullYear(), today.getMonth(), 0);
                break;
            case 'Last_Fiscal_Year':
                let lastFinYear = today.getFullYear();
                if (today.getMonth() < finYearStartMonth) { lastFinYear--; }
                startDate = new Date(lastFinYear - 1, finYearStartMonth, 1);
                endDate = new Date(lastFinYear, finYearStartMonth, 0);
                break;
            default:
                return;
        }
        
        startDateInput.value = formatDate(startDate);
        endDateInput.value = formatDate(endDate);
        
        document.querySelector('form.filter-form').submit();
    });
});
</script>