<?php
// qbo_reconciliation.php
// This script is now the User Interface. It handles uploads, triggers the background
// processor, and displays the results from the database.

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// --- Essential Includes ---
require_once __DIR__ . '/qbo_functions.php';
require_once __DIR__ . '/../vendor/autoload.php';
require_once __DIR__ . '/dbconn.php';
require_once __DIR__ . '/qbo_reconciliation_helpers.php'; // For history table creation
global $conn;

/**
 * NEW: Creates the reconciliation results table if it doesn't already exist.
 * This prevents a fatal error on the first run.
 * @param mysqli $dbConn The database connection.
 */
function create_reconciliation_table_if_not_exists(mysqli $dbConn) {
    $tableName = 'tdu_qbo_reconciliation';
    $checkTable = mysqli_query($dbConn, "SHOW TABLES LIKE '{$tableName}'");

    if ($checkTable && mysqli_num_rows($checkTable) == 0) {
        $sql = "CREATE TABLE `{$tableName}` (
            `id` INT(11) NOT NULL AUTO_INCREMENT,
            `file_name` VARCHAR(255) NOT NULL,
            `ai_raw_json` LONGTEXT NULL,
            `vtiger_quote_amounts_json` LONGTEXT NULL,
            `qbo_bills_json` LONGTEXT NULL,
            `processing_status` VARCHAR(50) NOT NULL DEFAULT 'processing',
            `error_message` TEXT NULL,
            `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            `processed_at` TIMESTAMP NULL ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            UNIQUE KEY `file_name` (`file_name`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";

        if (!mysqli_query($dbConn, $sql)) {
            // If creation fails, die gracefully instead of causing more errors.
            die("FATAL ERROR: Could not create required database table '{$tableName}'. Please check database permissions. Error: " . mysqli_error($dbConn));
        }
    }
}

$upload_message = '';
$upload_message_type = '';

// --- Handle File Upload & Trigger Background Job ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['statementFile'])) {
    $uploadDirectory = __DIR__ . '/statements/';
    if (!is_dir($uploadDirectory)) {
        if (!mkdir($uploadDirectory, 0777, true) && !is_dir($uploadDirectory)) {
            die('Failed to create upload directory.');
        }
    }
    $uploadedFile = $_FILES['statementFile'];
    if ($uploadedFile['error'] === UPLOAD_ERR_OK) {
        $fileTmpPath = $uploadedFile['tmp_name'];
        $safeBaseName = preg_replace("/[^a-zA-Z0-9._-]/", "_", basename($uploadedFile['name']));
        $destinationPath = $uploadDirectory . $safeBaseName;
        if (move_uploaded_file($fileTmpPath, $destinationPath)) {
            // Before triggering the job, ensure the tables exist
            create_reconciliation_table_if_not_exists($conn);
            create_history_table_if_not_exists($conn); // Ensure history table exists too
            $command = "php " . __DIR__ . "/cronjob/cronjob_qbo_reconciliation.php " . escapeshellarg($safeBaseName) . " > /dev/null 2>&1 &";
            exec($command);
            header("Location: " . $_SERVER['PHP_SELF'] . "?upload_status=success&file=" . urlencode($safeBaseName));
            exit();
        } else {
            header("Location: " . $_SERVER['PHP_SELF'] . "?upload_status=error&msg=" . urlencode('Failed to write file.'));
            exit();
        }
    } else {
        header("Location: " . $_SERVER['PHP_SELF'] . "?upload_status=error&msg=" . urlencode('Upload error code: ' . $uploadedFile['error']));
        exit();
    }
}

// --- Process GET parameters for status messages ---
if (isset($_GET['upload_status'])) {
    if ($_GET['upload_status'] === 'success') {
        $upload_message = 'File "' . htmlspecialchars($_GET['file']) . '" uploaded. It is now being processed. The list will update automatically when ready.';
        $upload_message_type = 'success';
    } else {
        $upload_message = 'Upload failed: ' . htmlspecialchars($_GET['msg']);
        $upload_message_type = 'error';
    }
}

// ===================================================================
// DATA FETCHING AND PREPARATION FOR DISPLAY
// ===================================================================

create_reconciliation_table_if_not_exists($conn);
create_history_table_if_not_exists($conn); // Check for history table on page load

$display_table_data = [];
$processed_files_list = [];
$file_being_viewed = $_GET['view_file'] ?? null;
$reconciliation_data = null; 

// 1. Get list of all processed files for the navigation menu
$result_files = mysqli_query($conn, "SELECT file_name, processing_status, created_at FROM tdu_qbo_reconciliation ORDER BY created_at DESC");
if ($result_files) {
    while ($row = mysqli_fetch_assoc($result_files)) {
        $processed_files_list[] = $row;
    }
    if (!$file_being_viewed && !empty($processed_files_list)) {
        $file_being_viewed = $processed_files_list[0]['file_name'];
    }
}

// 2. If a file is selected for viewing, fetch its data from the DB
if ($file_being_viewed) {
    $stmt = mysqli_prepare($conn, "SELECT * FROM tdu_qbo_reconciliation WHERE file_name = ?");
    mysqli_stmt_bind_param($stmt, "s", $file_being_viewed);
    mysqli_stmt_execute($stmt);
    $result_data = mysqli_stmt_get_result($stmt);

    if ($result_data && mysqli_num_rows($result_data) > 0) {
        $reconciliation_data = mysqli_fetch_assoc($result_data);

        if ($reconciliation_data['processing_status'] === 'completed') {
            $ai_data = json_decode($reconciliation_data['ai_raw_json'], true);
            $vtiger_quote_amounts = json_decode($reconciliation_data['vtiger_quote_amounts_json'], true);
            $qbo_bills_data = json_decode($reconciliation_data['qbo_bills_json'], false);

            if (is_array($ai_data) && is_array($vtiger_quote_amounts) && is_array($qbo_bills_data)) {
                foreach ($ai_data as $index => $item) {
                    $current_supplier_quote_cost = $vtiger_quote_amounts[$index] ?? 'Data N/A';
                    
                    $aiInvoiceAmount = is_numeric($item['invoice_amount'] ?? null) ? floatval($item['invoice_amount']) : 'N/A';
                    $statementMatch = ''; $difference = '';
                    $quoteAmountForMatch = is_numeric($current_supplier_quote_cost) ? floatval($current_supplier_quote_cost) : null;
                    if ($quoteAmountForMatch !== null && is_numeric($aiInvoiceAmount)) {
                        if (abs($quoteAmountForMatch - $aiInvoiceAmount) < 0.001) {
                            $statementMatch = 'YES'; $difference = 0.0;
                        } else {
                            $statementMatch = 'NO'; $difference = $aiInvoiceAmount - $quoteAmountForMatch;
                        }
                    }

                    $display_table_data[] = [
                        'item_index' => $index,
                        'supplier_name' => $item['supplier_name'] ?? 'N/A', 'tdu_class_number' => $item['tdu_class_number'] ?? 'N/A',
                        'passenger_name' => $item['passenger_name'] ?? 'N/A', 'description' => $item['description'] ?? 'N/A',
                        'confirmation_number' => $item['confirmation_number'] ?? 'N/A', 'invoice_number' => $item['invoice_number'] ?? 'N/A',
                        'service_date' => $item['service_date'] ?? 'N/A',
                        'quote_amount' => $current_supplier_quote_cost,
                        'invoice_amount' => $aiInvoiceAmount,
                        'statement_match' => $statementMatch,
                        'difference' => $difference,
                    ];
                }
            } else {
                 $upload_message = "Could not display results for ".htmlspecialchars($file_being_viewed).". The stored data may be corrupt.";
                 $upload_message_type = 'error';
            }
        } elseif ($reconciliation_data['processing_status'] === 'error') {
            $upload_message = "Error processing ".htmlspecialchars($file_being_viewed).": " . htmlspecialchars($reconciliation_data['error_message']);
            $upload_message_type = 'error';
        } else {
            $upload_message = "File ".htmlspecialchars($file_being_viewed)." is still processing. This page will auto-refresh.";
            $upload_message_type = 'success';
            header("Refresh: 20;");
        }
    }
}

// --- Calculate Totals ---
$total_invoice_amount = 0; $total_qbo_amount_val = 0; $qbo_total_contains_non_numeric = false;
$total_supplier_quote_cost_val = 0; $supplier_quote_total_contains_non_numeric = false;
$total_difference_val = 0;

if (isset($qbo_bills_data) && is_array($qbo_bills_data)) {
    foreach($display_table_data as $row) {
        $item_index = $row['item_index'];
        $qbo_match = $qbo_bills_data[$item_index] ?? null;
        $qbo_amount = null;
        if (is_object($qbo_match) && ($qbo_match->status ?? '') === 'matched') {
            $qbo_amount = $qbo_match->matched_line_amount;
        }

        if (is_numeric($row['invoice_amount'])) { $total_invoice_amount += floatval($row['invoice_amount']); }
        if (is_numeric($qbo_amount)) { $total_qbo_amount_val += floatval($qbo_amount); } elseif(!empty($qbo_amount)) { $qbo_total_contains_non_numeric = true; }
        if (is_numeric($row['quote_amount'])) { $total_supplier_quote_cost_val += floatval($row['quote_amount']); } elseif (!empty($row['quote_amount'])) { $supplier_quote_total_contains_non_numeric = true; }
        if (is_numeric($row['difference'])) { $total_difference_val += floatval($row['difference']); }
    }
}

$total_supplier_quote_cost_display = $supplier_quote_total_contains_non_numeric ? 'N/A' : number_format($total_supplier_quote_cost_val, 2);
$total_qbo_amount_display = $qbo_total_contains_non_numeric ? 'N/A' : number_format($total_qbo_amount_val, 2);
$total_difference_display = number_format($total_difference_val, 2);

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Statement Data Processor - TDU</title>
    <style>
        /* Styles unchanged */
        .tdu-statement-h2 { color: #005ea2; border-bottom: 2px solid #005ea2; padding-bottom: 8px; margin-top: 0; }
        .tdu-statement-table { border-collapse: collapse; width: 100%; margin-top: 15px; box-shadow: 0 0 10px rgba(0,0,0,0.07); font-size: 0.85em; }
        .tdu-statement-table th, .tdu-statement-table td { border: 1px solid #dee2e6; padding: 8px 10px; text-align: left; vertical-align: top; word-break: break-word; }
        .tdu-statement-table th { background-color: #005ea2; color: white; font-weight: 600; }
        .tdu-statement-table tr:nth-child(even) { background-color: #f8f9fa; }
        .tdu-statement-table tfoot tr { font-weight: bold; background-color: #e9ecef !important; }
        .tdu-message { padding: 12px 18px; margin-bottom: 20px; border-radius: 4px; border: 1px solid transparent; }
        .tdu-message.tdu-success { background-color: #d4edda; color: #155724; border-color: #c3e6cb; }
        .tdu-message.tdu-error { background-color: #f8d7da; color: #721c24; border-color: #f5c6cb; }
        input[type="submit"] { background-color: #28a745; color: white; padding: 10px 18px; border: none; border-radius: 4px; cursor: pointer; font-size: 1em; }
        .diff-positive { background-color: #ffcccc; color: #a60000; font-weight: bold; }
        .diff-negative { background-color: #d4edda; color: #155724; font-weight: bold; }
        .qbo-status-message { font-style: italic; color: #6c757d; font-size: 0.9em; }
        .sidenav { height: 100%; width: 0; position: fixed; z-index: 2; top: 0; left: 0; background-color: #1a2530; overflow-x: hidden; transition: 0.5s; padding-top: 60px; }
        .sidenav a { padding: 8px 8px 8px 32px; text-decoration: none; font-size: 18px; color: #8197a6; display: block; transition: 0.3s; word-break: break-all; }
        .sidenav a:hover { color: #f1f1f1; }
        .sidenav a.active { background-color: #005ea2; color: white; font-weight: bold; }
        .sidenav .closebtn { position: absolute; top: 0; right: 25px; font-size: 36px; margin-left: 50px; }
        .sidenav h2 { color: white; padding: 0 32px; font-size: 1.5em; border-bottom: 1px solid #4f5f6c; padding-bottom: 10px; }
        .sidenav .status { font-size: 0.7em; color: #8197a6; display: block; margin-top: 2px; text-transform: capitalize; }
        #main { transition: margin-left .5s; padding: 20px; max-width: 1400px; margin-left: auto; margin-right: auto; }
        .open-nav-btn { font-size: 30px; cursor: pointer; position: fixed; top: 15px; left: 20px; z-index: 1; background-color: #005ea2; color: white; border: none; border-radius: 5px; padding: 0px 10px; line-height: 1.2; }
    </style>
</head>
<body>

<div id="mySidenav" class="sidenav">
  <a href="javascript:void(0)" class="closebtn" onclick="closeNav()">×</a>
  <h2>Statement Files</h2>
  <?php if (empty($processed_files_list)): ?>
      <a href="#">No files processed yet.</a>
  <?php else: ?>
      <?php foreach ($processed_files_list as $file_item): ?>
          <a href="?view_file=<?php echo urlencode($file_item['file_name']); ?>" class="<?php echo ($file_item['file_name'] == $file_being_viewed ? 'active' : ''); ?>" title="<?php echo htmlspecialchars($file_item['file_name']); ?>">
              <?php echo htmlspecialchars($file_item['file_name']); ?>
              <span class="status"><?php echo htmlspecialchars($file_item['processing_status']); ?> - <?php echo date('d-M-y', strtotime($file_item['created_at'])); ?></span>
          </a>
      <?php endforeach; ?>
  <?php endif; ?>
</div>

<span class="open-nav-btn" onclick="openNav()">☰</span>

<div id="main">
    <h1>Statement Data Processor</h1>
    
    <details open>
        <summary style="font-weight: bold; cursor: pointer; color: #005ea2; font-size: 1.2em;">Upload New Statement</summary>
        <div style="padding: 15px; border: 1px solid #ccc; border-radius: 5px; margin-top: 10px;">
            <?php if (!empty($upload_message)): ?>
                <div class="tdu-message <?php echo ($upload_message_type === 'success' ? 'tdu-success' : 'tdu-error'); ?>"><?php echo $upload_message; ?></div>
            <?php endif; ?>
            <form action="" method="post" enctype="multipart/form-data">
                <input type="file" name="statementFile" required>
                <input type="submit" value="Upload & Process" name="submit">
            </form>
        </div>
    </details>

    <h2 class="tdu-statement-h2" style="margin-top: 30px;">Reconciliation for: <?php echo htmlspecialchars($file_being_viewed ?? 'No file selected'); ?></h2>
    
    <?php if(!empty($display_table_data) && isset($qbo_bills_data)): ?>
        <div style="overflow-x: auto;">
            <table class="tdu-statement-table" id="tdu-main-statement-table">
                 <thead>
                    <tr>
                        <th>SUPPLIER NAME</th> <th>TDU / CLASS #</th> <th>PASSENGER NAME</th> <th>DESCRIPTION</th>
                        <th>CONF #</th> <th>INV #</th> <th>SERVICE DATE</th> <th>QUOTE AMT</th>
                        <th>QBO AMT</th> <th>INV AMT</th> <th>MATCH</th> <th>DIFFERENCE</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($display_table_data as $row): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($row['supplier_name']); ?></td>
                            <td><?php echo htmlspecialchars($row['tdu_class_number']); ?></td>
                            <td><?php echo htmlspecialchars($row['passenger_name']); ?></td>
                            <td><?php echo htmlspecialchars($row['description']); ?></td>
                            <td><?php echo htmlspecialchars($row['confirmation_number']); ?></td>
                            <td><?php echo htmlspecialchars($row['invoice_number']); ?></td>
                            <td><?php echo htmlspecialchars($row['service_date']); ?></td>
                            <td><?php echo is_numeric($row['quote_amount']) ? number_format(floatval($row['quote_amount']), 2) : "<span class='qbo-status-message'>" . htmlspecialchars($row['quote_amount']) . "</span>"; ?></td>
                            <td>
                                <?php
                                $item_index = $row['item_index'];
                                $qbo_match_data = $qbo_bills_data[$item_index] ?? null;
                            
                                if (is_object($qbo_match_data) && ($qbo_match_data->status ?? '') === 'matched') {
                                    $qbo_bill_object = $qbo_match_data->bill_object ?? null;
                                    if (isset($qbo_bill_object) && !empty($qbo_bill_object->Id)) {
                                        $qbo_url = "https://app.qbo.intuit.com/app/bill?txnId=" . $qbo_bill_object->Id;
                                        $qbo_amount_display = is_numeric($qbo_match_data->matched_line_amount) ? number_format(floatval($qbo_match_data->matched_line_amount), 2) : "<span class='qbo-status-message'>" . htmlspecialchars($qbo_match_data->matched_line_amount) . "</span>";
                                        echo "<a href='{$qbo_url}' target='_blank' title='View Bill in QBO'>{$qbo_amount_display}</a>";
                                        
                                        if (!empty($qbo_match_data->matched_line_id)) {
                                            echo '<button class="edit-qbo-bill-btn" 
                                                    data-rec-id="'.htmlspecialchars($reconciliation_data['id']).'"
                                                    data-item-index="'.htmlspecialchars($item_index).'"
                                                    data-bill-id="'.htmlspecialchars($qbo_bill_object->Id).'" 
                                                    data-sync-token="'.htmlspecialchars($qbo_bill_object->SyncToken).'" 
                                                    data-line-id="'.htmlspecialchars($qbo_match_data->matched_line_id).'" 
                                                    data-current-amount="'.htmlspecialchars(is_numeric($qbo_match_data->matched_line_amount) ? $qbo_match_data->matched_line_amount : '').'" 
                                                    data-doc-number="'.htmlspecialchars($qbo_bill_object->DocNumber ?? '').'" 
                                                    title="Edit this Bill Amount">✎</button>';
                                        }
                                    }
                                } elseif (is_object($qbo_match_data) && ($qbo_match_data->status ?? '') === 'needs_review') {
                                    $potential_matches = $qbo_match_data->potential_matches ?? [];
                                    if (!empty($potential_matches)) {
                                        echo '<div class="manual-review-container">';
                                        echo '<select style="max-width:150px;" class="manual-match-select" data-rec-id="'.htmlspecialchars($reconciliation_data['id']).'" data-item-index="'.htmlspecialchars($item_index).'">';
                                        echo '<option style="max-width:150px;" value="">Select a Bill to Match...</option>';
                                        foreach ($potential_matches as $match) {
                                            $option_text = 'Bill ' . htmlspecialchars($match->doc_number) . ' - Line ' . htmlspecialchars($match->line_id) . ' (' . number_format($match->amount, 2) . ') - ' . htmlspecialchars(substr($match->description, 0, 30)).'...';
                                            echo '<option value="'.htmlspecialchars($match->bill_id).'|'.htmlspecialchars($match->line_id).'">'.$option_text.'</option>';
                                        }
                                        echo '</select>';
                                        echo '<button class="manual-match-confirm-btn" style="margin-left: 5px;" disabled>Match</button>';
                                        echo '</div>';
                                    } else {
                                        echo "<span class='qbo-status-message'>Needs Review (No Options)</span>";
                                    }
                                } else {
                                    $status_text = 'N/A';
                                    if (is_object($qbo_match_data) && !empty($qbo_match_data->status)) {
                                        $status_text = $qbo_match_data->status;
                                    } elseif (is_object($qbo_match_data) && !empty($qbo_match_data->message)) {
                                        $status_text = $qbo_match_data->message;
                                    }
                                    echo "<span class='qbo-status-message'>" . htmlspecialchars($status_text) . "</span>";
                                }
                                ?>
                            </td>
                            <td><?php echo is_numeric($row['invoice_amount']) ? number_format($row['invoice_amount'], 2) : "<span class='qbo-status-message'>".htmlspecialchars($row['invoice_amount'])."</span>"; ?></td>
                            <td><?php echo htmlspecialchars($row['statement_match']); ?></td>
                            <td class="<?php echo is_numeric($row['difference']) ? ($row['difference'] > 0 ? 'diff-positive' : ($row['difference'] < 0 ? 'diff-negative' : '')) : ''; ?>">
                                <?php echo is_numeric($row['difference']) ? htmlspecialchars(number_format($row['difference'], 2)) : ''; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
                <tfoot>
                    <tr>
                        <td colspan="7" style="text-align:right;">TOTALS:</td>
                        <td><?php echo $total_supplier_quote_cost_display; ?></td>
                        <td><?php echo $total_qbo_amount_display; ?></td>
                        <td><?php echo number_format($total_invoice_amount, 2); ?></td>
                        <td></td>
                        <td><?php echo $total_difference_display; ?></td>
                    </tr>
                </tfoot>
            </table>
        </div>
    <?php else: ?>
        <p style="padding: 20px; background-color: #e9ecef; border-radius: 5px;">
            <?php if($file_being_viewed) { echo "No results to display for this file, or it may still be processing."; } else { echo "Please select a file from the menu (☰) or upload a new one to begin."; } ?>
        </p>
    <?php endif; ?>
</div>

<div id="edit-bill-modal" style="display:none; position:fixed; z-index:1000; left:0; top:0; width:100%; height:100%; background-color:rgba(0,0,0,0.5);">
    <div style="background-color:#fefefe; margin:10% auto; padding:20px; border-radius:8px; max-width:500px; box-shadow: 0 4px 8px 0 rgba(0,0,0,0.2);">
        <h2 style="color:#005ea2; border-bottom:1px solid #ccc; padding-bottom:10px;">Edit QBO Bill <span id="modal_bill_id_display" style="font-size:0.6em; color:#6c757d; font-weight:normal;"></span></h2>
        <form id="edit-bill-form">
            <input type="hidden" id="modal_reconciliation_id" name="reconciliation_id">
            <input type="hidden" id="modal_item_index" name="item_index">
            <input type="hidden" id="modal_bill_id" name="bill_id">
            <input type="hidden" id="modal_sync_token" name="sync_token">
            <input type="hidden" id="modal_line_id" name="line_id">
            
            <!-- MODIFIED: This is the hidden input for the filename -->
            <input type="hidden" id="modal_file_name" name="file_name">

            <div style="margin-bottom:15px;"><label for="modal_doc_number" style="display:block; margin-bottom:5px; font-weight:bold;">Invoice / Doc Number:</label><input type="text" id="modal_doc_number" name="doc_number" style="width:95%; padding:8px;"></div>
            <div style="margin-bottom:20px;"><label for="modal_bill_amount" style="display:block; margin-bottom:5px; font-weight:bold;">Bill Amount:</label><input type="number" step="0.01" id="modal_bill_amount" name="bill_amount" required style="width:95%; padding:8px;"></div>
            <p id="modal-error-msg" style="color:#721c24; background-color:#f8d7da; padding:10px; border-radius:4px; display:none;"></p>
            <div style="text-align:right;"><button type="button" class="modal-cancel-btn" style="padding:10px 15px; border-radius:4px; cursor:pointer; margin-right:10px;">Cancel</button><button type="submit" class="modal-save-btn" style="background-color:#28a745; color:white; padding:10px 15px; border-radius:4px; cursor:pointer;">Save Changes</button></div>
        </form>
    </div>
</div>

<script>
function openNav() { document.getElementById("mySidenav").style.width = "300px"; }
function closeNav() { document.getElementById("mySidenav").style.width = "0"; }
</script>

<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script>
$(document).ready(function() {
    // MODIFIED: Pass the PHP filename variable to JavaScript
    const currentFileName = "<?php echo htmlspecialchars($file_being_viewed ?? ''); ?>";

    let activeEditButton = null;
    const modal = $('#edit-bill-modal');

    $('#main').on('click', '.edit-qbo-bill-btn', function() {
        activeEditButton = $(this);
        
        $('#modal_reconciliation_id').val(activeEditButton.data('rec-id'));
        $('#modal_item_index').val(activeEditButton.data('item-index'));
        $('#modal_bill_id').val(activeEditButton.data('bill-id'));
        $('#modal_sync_token').val(activeEditButton.data('sync-token'));
        $('#modal_line_id').val(activeEditButton.data('line-id'));
        $('#modal_bill_amount').val(activeEditButton.data('current-amount'));
        $('#modal_doc_number').val(activeEditButton.data('doc-number'));
        $('#modal_bill_id_display').text('(ID: ' + activeEditButton.data('bill-id') + ')');
        
        // MODIFIED: Populate the hidden filename input
        $('#modal_file_name').val(currentFileName);
        
        $('#modal-error-msg').hide().text('');
        modal.show();
    });

    $('#edit-bill-form').on('submit', function(e) {
        e.preventDefault();
        const form = $(this);
        const saveButton = form.find('.modal-save-btn');
        const originalButtonText = saveButton.text();
        saveButton.prop('disabled', true).text('Saving...');
        $('#modal-error-msg').hide().text('');

        $.ajax({
            type: 'POST',
            url: 'ajax_qbo_bill_update.php',
            data: form.serialize(),
            dataType: 'json',
            success: function(response) {
                if (response.success) {
                    modal.hide();
                    alert('Bill updated successfully! The page will now reload to reflect the changes.');
                    location.reload(); 
                } else {
                    $('#modal-error-msg').text(response.error).show();
                }
            },
            error: function(jqXHR) {
                const errorMsg = jqXHR.responseJSON ? jqXHR.responseJSON.error : 'An unexpected server error occurred.';
                $('#modal-error-msg').text('Error: ' + errorMsg).show();
            },
            complete: function() {
                saveButton.prop('disabled', false).text(originalButtonText);
            }
        });
    });

    $('.modal-cancel-btn').on('click', function() { modal.hide(); });
    $(window).on('click', function(event) { if ($(event.target).is(modal)) { modal.hide(); } });
    
    $('#main').on('change', '.manual-match-select', function() {
        const select = $(this);
        const confirmBtn = select.siblings('.manual-match-confirm-btn');
        if (select.val() !== "") {
            confirmBtn.prop('disabled', false);
        } else {
            confirmBtn.prop('disabled', true);
        }
    });

    $('#main').on('click', '.manual-match-confirm-btn', function() {
        const button = $(this);
        const select = button.siblings('.manual-match-select');
        const selectedValue = select.val();
        
        if (selectedValue === "") {
            alert('Please select a bill from the dropdown first.');
            return;
        }

        const [billId, lineId] = selectedValue.split('|');
        const recId = select.data('rec-id');
        const itemIndex = select.data('item-index');

        button.prop('disabled', true).text('Matching...');

        $.ajax({
            type: 'POST',
            url: 'ajax_qbo_manual_match.php',
            data: {
                reconciliation_id: recId,
                item_index: itemIndex,
                bill_id: billId,
                line_id: lineId
            },
            dataType: 'json',
            success: function(response) {
                if (response.success) {
                    alert('Match successful! The page will now reload.');
                    location.reload();
                } else {
                    alert('Error: ' + response.error);
                    button.prop('disabled', false).text('Match');
                }
            },
            error: function() {
                alert('An unexpected server error occurred during the match process.');
                button.prop('disabled', false).text('Match');
            }
        });
    });
});
</script>
</body>
</html>