<?php
// qbo_reconciliation_helpers.php
// Contains helper functions for the reconciliation history feature.

/**
 * Creates the 'tdu_qbo_reconciliation_history' table if it doesn't already exist.
 * This should be called before any attempt to log data.
 *
 * @param mysqli $dbConn The database connection.
 */
function create_history_table_if_not_exists(mysqli $dbConn) {
    $tableName = 'tdu_qbo_reconciliation_history';
    $checkTable = mysqli_query($dbConn, "SHOW TABLES LIKE '{$tableName}'");

    if ($checkTable && mysqli_num_rows($checkTable) == 0) {
        $sql = "CREATE TABLE `{$tableName}` (
            `history_id` INT(11) NOT NULL AUTO_INCREMENT,
            `reconciliation_id` INT(11) NOT NULL,
            `item_index` INT(11) NOT NULL,
            `user_id` INT(11) NULL COMMENT 'The ID of the user who made the change',
            `user_name` VARCHAR(255) NULL COMMENT 'The name of the user for quick reference',
            `action_type` VARCHAR(50) NOT NULL COMMENT 'e.g., ''Bill Update'', ''Manual Match''',
            `change_details_before` JSON NULL,
            `change_details_after` JSON NULL,
            `change_timestamp` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (`history_id`),
            INDEX `idx_reconciliation_id` (`reconciliation_id`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";

        if (!mysqli_query($dbConn, $sql)) {
            // Log this error but don't kill the parent script
            error_log("FATAL ERROR: Could not create required database table '{$tableName}'. Error: " . mysqli_error($dbConn));
        }
    }
}

/**
 * Logs a change made during the reconciliation process to the history table.
 *
 * @param mysqli $conn              The database connection.
 * @param int    $recId             The ID of the main reconciliation record.
 * @param int    $itemIndex         The index of the line item that was changed.
 * @param string $actionType        The type of action performed (e.g., 'Bill Update').
 * @param array  $detailsBefore     An associative array of the data before the change.
 * @param array  $detailsAfter      An associative array of the data after the change.
 * @param int    $userId            (Optional) The ID of the logged-in user.
 * @param string $userName          (Optional) The name of the logged-in user.
 */
function log_reconciliation_change(mysqli $conn, int $recId, int $itemIndex, string $actionType, array $detailsBefore, array $detailsAfter, ?int $userId = null, ?string $userName = null) {
    // Ensure the history table exists before we try to insert into it.
    create_history_table_if_not_exists($conn);

    $sql = "INSERT INTO tdu_qbo_reconciliation_history 
                (reconciliation_id, item_index, user_id, user_name, action_type, change_details_before, change_details_after) 
            VALUES (?, ?, ?, ?, ?, ?, ?)";

    $stmt = mysqli_prepare($conn, $sql);

    // NOTE: For now, user_id and user_name are placeholders.
    // Replace with actual session data when user authentication is integrated.
    $placeholderUserId = $userId ?? 0;
    $placeholderUserName = $userName ?? 'System/Unknown';

    $jsonBefore = json_encode($detailsBefore);
    $jsonAfter = json_encode($detailsAfter);

    mysqli_stmt_bind_param($stmt, "iiissss", 
        $recId, 
        $itemIndex,
        $placeholderUserId,
        $placeholderUserName,
        $actionType,
        $jsonBefore,
        $jsonAfter
    );

    if (!mysqli_stmt_execute($stmt)) {
        error_log("History Log Error: Failed to log reconciliation change for rec_id {$recId}. Error: " . mysqli_stmt_error($stmt));
    }

    mysqli_stmt_close($stmt);
}