<?php
// qbo_sales_by_customer_detail_report.php
// FINAL CORRECTED VERSION - Implements robust Event Delegation for all UI handlers.

// Only default dates are set here. The rest is handled by JavaScript.
$finYearStart = (date('m') < 7) ? date('Y-07-01', strtotime('-1 year')) : date('Y-07-01');
$startDate = isset($_GET['start_date']) && !empty($_GET['start_date']) ? $_GET['start_date'] : $finYearStart;
$endDate = isset($_GET['end_date']) && !empty($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-d');
$defaultProductServices = ['Sales', 'Sales Group AU', 'Sales- Newzealand', 'Sales Group NZ'];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Sales by Customer Detail</title>
<style>
    body { font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif; }
    .report-header { margin-bottom: 20px; }
    .global-controls { display: flex; justify-content: space-between; align-items: center; padding: 15px; background-color: #e9ecef; border-radius: 5px; margin-bottom: 20px; }
    .global-controls button { padding: 10px 15px; border: none; border-radius: 4px; cursor: pointer; font-size: 14px; }
    .settings-btn { background-color: #6c757d; color: white; }
    #add-report-btn { background-color: #28a745; color: white; }
    
    .report-instance { border: 2px solid #dee2e6; border-radius: 8px; padding: 15px; margin-bottom: 20px; background-color: #fff; }
    .report-instance h3 { margin-top: 0; border-bottom: 1px solid #ccc; padding-bottom: 10px; }
    .instance-controls { display: flex; flex-wrap: wrap; gap: 15px; align-items: center; margin-bottom: 15px; }
    .instance-controls .filter-group { display: flex; flex-direction: column; }
    .instance-controls label { font-weight: bold; font-size: 12px; margin-bottom: 4px; color: #333; }
    .instance-controls input, .instance-controls select { padding: 8px; border: 1px solid #ccc; border-radius: 4px; font-size: 14px; }
    .instance-controls .run-report-btn { background-color: #007bff; color: white; padding: 8px 15px; margin-top: 18px; border:none; border-radius:4px; }

    table { width: 100%; border-collapse: collapse; font-size: 14px; }
    th, td { border: 1px solid #ddd; padding: 10px; text-align: left; }
    th { background-color: #f2f2f2; font-weight: bold; }
    .customer-header { background-color: #e9f5ff; font-weight: bold; cursor: pointer; }
    .detail-row { background-color: #fff; display: none; }
    .total-row { background-color: #f2f2f2; font-weight: bold; }
    .currency { text-align: right; }
    
    .multiselect-dropdown { position: relative; display: inline-block; }
    .multiselect-dropdown-btn { width: 200px; text-align: left; background-color: white; padding: 5px; font-size: 12px; border: 1px solid #ccc; border-radius: 4px; }
    .multiselect-dropdown-content { display: none; position: absolute; background-color: #f9f9f9; min-width: 400px; box-shadow: 0px 8px 16px 0px rgba(0,0,0,0.2); z-index: 10; border-radius: 4px; padding: 5px; max-height: 250px; overflow-y: auto; }
    .multiselect-dropdown-content label { display: block; padding: 8px 10px; cursor: pointer; font-weight: normal; font-size: 12px; }
    
    .toggle-arrow { display: inline-block; width: 1em; transition: transform 0.2s; }
    .customer-header.expanded .toggle-arrow { transform: rotate(90deg); }
    .loading-spinner { display: inline-block; width: 16px; height: 16px; border: 2px solid #ccc; border-top-color: #007bff; border-radius: 50%; animation: spin 1s linear infinite; }
    @keyframes spin { to { transform: rotate(360deg); } }
    .sortable { cursor: pointer; position: relative; padding-right: 20px !important; }
    .sortable::after, .sortable::before { content: ""; position: absolute; right: 5px; border: 4px solid transparent; opacity: 0.5; }
    .sortable::before { top: 40%; border-bottom-color: #666; }
    .sortable::after { bottom: 40%; border-top-color: #666; }
    .sortable.asc::before, .sortable.desc::after { opacity: 1; }
    .panel-content { border: 1px solid #ddd; border-radius: 5px; background: #fff; }
    .centered-loader { text-align: center; padding: 50px; }
    .panel-filters { padding: 10px; background-color: #f0f0f0; border-bottom: 1px solid #ddd; display: flex; flex-wrap: wrap; gap: 10px; align-items: center; }
    .panel-filters .filter-group { display: flex; flex-direction: column; }
    .panel-filters label { font-weight: bold; font-size: 11px; margin-bottom: 3px; }
    .panel-filters input, .panel-filters .multiselect-dropdown-btn { padding: 5px; font-size: 12px; border-radius: 4px; border: 1px solid #ccc; width: 140px; }
    .modal-backdrop { position: fixed; top: 0; left: 0; width: 100%; height: 100%; background-color: rgba(0,0,0,0.5); z-index: 9990; display: none; }
    .modal { position: fixed; top: 50%; left: 50%; transform: translate(-50%, -50%); background: white; border-radius: 8px; box-shadow: 0 5px 15px rgba(0,0,0,0.3); z-index: 9991; width: 90%; max-width: 500px; display: none; }
    .modal-header { padding: 15px; border-bottom: 1px solid #dee2e6; display: flex; justify-content: space-between; align-items: center; }
    .modal-header h3 { margin: 0; }
    .modal-header .close-btn { background: none; border: none; font-size: 1.5rem; cursor: pointer; }
    .modal-body { padding: 20px; }
    .modal-body .form-group { margin-bottom: 15px; }
    .modal-body label { display: block; margin-bottom: 5px; font-weight: bold; }
    .modal-body input { width: 100%; padding: 8px; border: 1px solid #ccc; border-radius: 4px; box-sizing: border-box; }
    .modal-footer { padding: 15px; border-top: 1px solid #dee2e6; text-align: right; }
    .modal-footer button { padding: 10px 20px; border-radius: 4px; border: none; cursor: pointer; }
    .modal-footer .btn-secondary { background-color: #6c757d; color: white; }
    .modal-footer .btn-primary { background-color: #007bff; color: white; }
    .cat-high { background-color: #d4edda !important; color: #155724; }
    .cat-mid { background-color: #fff3cd !important; color: #856404; }
    .cat-low { background-color: #f8d7da !important; color: #721c24; }
    .db-customer-header { cursor: pointer; background-color: #e9f5ff; font-weight: bold; }
    .db-customer-header.expanded { background-color: #d1eaff; }
    .toggle-arrow-db { display: inline-block; width: 1em; transition: transform 0.2s; }
    .db-details-row > td { padding: 0 10px 10px 40px !important; background-color: #fdfdfd; }
    .db-details-row .nested-table { width: 100%; box-shadow: none; border-radius: 0; margin-top: 5px; }
    .db-details-row .nested-table th { background-color: #f1f1f1; font-weight: normal; font-size: 12px; }
    .db-details-row .nested-table td { font-size: 12px; }
    .tab-nav { border-bottom: 2px solid #ddd; margin-bottom: 20px; }
    .tab-link { background-color: #f1f1f1; border: 1px solid #ccc; border-bottom: none; cursor: pointer; padding: 10px 15px; font-size: 16px; margin-bottom: -2px; border-radius: 5px 5px 0 0; }
    .tab-link.active { background-color: #fff; border-bottom: 2px solid #fff; color: #007bff; font-weight: bold; }
    .tab-content { display: none; }
    .tab-content.active { display: block; }
     #report-instances-container {
        display: flex; /* Aligns children (the reports) in a row */
        flex-direction: row; /* Explicitly sets the direction to horizontal */
        overflow-x: auto; /* Adds a horizontal scrollbar if the reports overflow */
        gap: 20px; /* Adds space between each report instance */
        padding-bottom: 15px; /* Prevents the scrollbar from overlapping content */
    }

    .report-instance {
        /*
         * This is the key change. We set a larger minimum width to prevent
         * the table from being squished. Adjust this value as needed.
         * 800px is a good starting point.
        */
        min-width: 800px;
    }
    
    .report-instance {
    position: relative; /* This is needed for the button's absolute positioning */
    }
    
    .delete-instance-btn {
        position: absolute;
        top: 10px;
        right: 10px;
        background-color: #f8d7da;
        color: #721c24;
        border: 1px solid #f5c6cb;
        border-radius: 50%;
        width: 28px;
        height: 28px;
        font-size: 14px;
        font-weight: bold;
        line-height: 26px;
        text-align: center;
        cursor: pointer;
        z-index: 10; /* Ensures it's above other elements */
        transition: background-color 0.2s;
    }
    
    .delete-instance-btn:hover {
        background-color: #f1b0b7;
    }
</style>
</head>
<body>

<div class="report-header">
    <h2>Sales by Customer Detail</h2>
</div>

<div class="global-controls">
    <button type="button" id="add-report-btn">➕ Add Comparison Report</button>
    <button type="button" id="settings-btn" class="settings-btn" title="Classification Settings">⚙️</button>
</div>

<div id="report-instances-container">
    <!-- Report instances will be injected here by JavaScript -->
</div>

<template id="report-template">
    <div class="report-instance">
        <button type="button" class="delete-instance-btn" title="Remove Report">✖</button>
        <div class="instance-controls">
            <div class="filter-group">
                <label>Date Period:</label>
                <select class="date-period-select"></select>
            </div>
            <div class="filter-group">
                <label>From:</label><input type="date" class="start-date-input">
            </div>
            <div class="filter-group">
                <label>To:</label><input type="date" class="end-date-input">
            </div>
            <button type="button" class="run-report-btn">Run Report</button>
        </div>

        <div class="tab-nav">
            <button class="tab-link active" data-tab="qbo-report-tab">QBO Invoice Detail</button>
            <button class="tab-link" data-tab="db-report-tab">DB Quotes Comparison</button>
        </div>

        <div class="qbo-report-tab tab-content active">
            <section class="report-panel qbo-report-panel">
                <div class="panel-filters">
                    <div class="filter-group">
                        <label>Value:</label>
                        <div class="multiselect-dropdown qbo-value-filter">
                            <button type="button" class="multiselect-dropdown-btn">All Values</button>
                            <div class="multiselect-dropdown-content"></div>
                        </div>
                    </div>
                    <div class="filter-group">
                        <label>Product/Service:</label>
                        <div class="multiselect-dropdown product-service-filter">
                            <button type="button" class="multiselect-dropdown-btn">Select Services</button>
                            <div class="multiselect-dropdown-content"></div>
                        </div>
                    </div>
                    <div class="filter-group"><label>Filter by Customer:</label><input type="text" class="qbo-customer-filter" placeholder="Type to filter..."></div>
                    <div class="filter-group"><label>Filter by Quote No.:</label><input type="text" class="qbo-quote-filter" placeholder="Type to filter..."></div>
                </div>
                <table>
                    <thead class="qbo-report-head">
                        <tr>
                            <th class="sortable" data-col="value" data-type="string">Value</th>
                            <th>Date</th><th>Transaction Type</th><th>No.</th><th>Quote No.</th><th>Product/Service</th>
                            <th class="sortable currency" data-col="qty" data-type="number">No. of Invoices</th> <!-- This is the original column, UNCHANGED -->
                            <th class="sortable currency" data-col="lineCount" data-type="number">Invoice Lines</th> <!-- THIS IS THE NEW COLUMN -->
                            <th class="sortable currency" data-col="price" data-type="number">Total Amount</th>
                        </tr>
                    </thead>
                    <tbody class="report-body"></tbody>
                    <tfoot class="report-footer">
                        <tr class="total-row">
                            <td colspan="6"><strong>GRAND TOTAL</strong></td>
                            <td class="currency" id="grand-total-qty-${reportInstanceCounter}">0</td> <!-- Original footer cell, UNCHANGED -->
                            <td class="currency" id="grand-total-line-count-${reportInstanceCounter}">0</td> <!-- New footer cell -->
                            <td class="currency" id="grand-total-amt-${reportInstanceCounter}">0.00</td>
                        </tr>
                    </tfoot>
                </table>
                <div class="no-data-message" style="display: none; text-align: center; padding: 40px;"><p>No data for the selected criteria.</p></div>
            </section>
        </div>

        <div class="db-report-tab tab-content">
            <section class="db-quotes-panel" style="width: 100%; overflow: visible;">
                <div class="panel-content">
                    <div class="panel-filters">
                         <div class="filter-group"><label>Internal Agent:</label><input type="text" class="db-internal-agent-filter" placeholder="Filter by name..."></div>
                         <div class="filter-group"><label>External Agent:</label><input type="text" class="db-external-agent-filter" placeholder="Filter by name..."></div>
                         <div class="filter-group"><label>Quote No.:</label><input type="text" class="db-quote-no-filter" placeholder="Filter by number..."></div>
                         <div class="filter-group"><label>Customer:</label><input type="text" class="db-customer-filter" placeholder="Filter by name..."></div>
                         <div class="filter-group"><label>Region:</label><div class="multiselect-dropdown db-region-filter"><button type="button" class="multiselect-dropdown-btn">All Regions</button><div class="multiselect-dropdown-content"></div></div></div>
                         <div class="filter-group"><label>Country:</label><div class="multiselect-dropdown db-country-filter"><button type="button" class="multiselect-dropdown-btn">All Countries</button><div class="multiselect-dropdown-content"></div></div></div>
                         <div class="filter-group"><label>Quote Stage:</label><div class="multiselect-dropdown db-stage-filter"><button type="button" class="multiselect-dropdown-btn">All Stages</button><div class="multiselect-dropdown-content"></div></div></div>
                    </div>
                    <div class="db-quotes-table-container"></div>
                </div>
            </section>
        </div>
    </div>
</template>

<div class="modal-backdrop" id="settings-modal-backdrop"></div>
<div class="modal" id="settings-modal">
    <div class="modal-header"><h3>Account Classification Rules</h3><button class="close-btn" id="close-settings-modal-btn">✖</button></div>
    <div class="modal-body">
        <h4>Part A: Billed Accounts</h4>
        <p style="font-size:12px; margin-top:-10px; color:#666;">Classification is determined by total invoice value.</p>
        <div class="form-group"><label for="rule-high-value">High-Value ( > )</label><input type="number" id="rule-high-value"></div>
        <div class="form-group"><label for="rule-mid-value">Mid-Value ( > )</label><input type="number" id="rule-mid-value"></div>
        <div class="form-group"><label for="rule-low-value">Low-Value ( >= )</label><input type="number" id="rule-low-value"></div><hr>
        <h4>Part B: Non-Billed Accounts (Prospects)</h4>
        <p style="font-size:12px; margin-top:-10px; color:#666;">Classification is based on the volume of quotes provided.</p>
        <div class="form-group"><label for="rule-high-engagement">High-Engagement ( >= )</label><input type="number" id="rule-high-engagement"></div>
        <div class="form-group"><label for="rule-mid-engagement">Mid-Engagement ( >= )</label><input type="number" id="rule-mid-engagement"></div>
        <div class="form-group"><label for="rule-low-engagement">Low-Engagement ( >= )</label><input type="number" id="rule-low-engagement"></div>
    </div>
    <div class="modal-footer"><button type="button" class="btn-secondary" id="cancel-settings-btn">Cancel</button><button type="button" class="btn-primary" id="save-settings-btn">Save Settings</button></div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    
    // --- Global State & Configuration ---
    const instancesContainer = document.getElementById('report-instances-container');
    const reportTemplate = document.getElementById('report-template');
    const addReportBtn = document.getElementById('add-report-btn');
    const defaultProductServices = <?php echo json_encode($defaultProductServices); ?>;
    let reportInstanceCounter = 0;

    // --- Core Function: Create a Report Instance ---
    function createReportInstance(defaultStartDate, defaultEndDate) {
        reportInstanceCounter++;
        const instanceId = `report-instance-${reportInstanceCounter}`;
        const instanceClone = reportTemplate.content.cloneNode(true);
        const instanceDiv = instanceClone.querySelector('.report-instance');
        instanceDiv.id = instanceId;
        instancesContainer.appendChild(instanceClone);
        
        const datePeriods = [
            'Custom dates', 'All Dates', 'Today', 'This week', 'This week to date', 
            'This month', 'This month to date', 'This quarter', 'This quarter to date',
            'This year', 'This year to date', 'This year to last month', 'This financial year', 'Yesterday', 
            'Last week', 'Last month', 'Last quarter', 'Last year', 'Last financial year',
            'Last 7 days', 'Last 30 days', 'Last 90 days', 'Last 12 months',
            'Since 30 days ago', 'Since 60 days ago', 'Since 90 days ago', 'Since 365 days ago',
            'Next week', 'Next 4 weeks', 'Next month', 'Next quarter', 'Next year'
        ];
        
        const scoped = (selector) => document.querySelector(`#${instanceId} ${selector}`);
        const scopedAll = (selector) => document.querySelectorAll(`#${instanceId} ${selector}`);

        const startDateInput = scoped('.start-date-input');
        const endDateInput = scoped('.end-date-input');
        const runReportBtn = scoped('.run-report-btn');
        const tbody = scoped('.report-body');
        const tfoot = scoped('.report-footer');
        const noDataMessage = scoped('.no-data-message');
        const qboReportHead = scoped('.qbo-report-head');
        
        let qboTableSort = { column: 'price', dir: 'desc' };
        let dbTableSort = { column: 'engagementCategory', dir: 'asc', type: 'string' };
        const valueCategories = ['High-Value', 'Mid-Value', 'Low-Value', 'N/A'];
        let selectedValueCategories = [...valueCategories]; // Initially, all are selected
        let selectedProductServices = [...defaultProductServices];
        let dbQuotesLoaded = false;
        let allDbQuotes = [];
        let currentAggregatedDbData = [];
        
        // --- NEW: This will store the full dataset from the database ---
        let allQboData = {};
        
        startDateInput.value = defaultStartDate;
        endDateInput.value = defaultEndDate;
        
        const deleteBtn = scoped('.delete-instance-btn');
        deleteBtn.addEventListener('click', () => {
            instanceDiv.remove(); // Removes the entire report instance from the page
        });
        
        const formatDate = (d) => d.toISOString().split('T')[0];
        function calculateDateRange(period) {
            const today = new Date();
            today.setHours(0, 0, 0, 0); // Normalize to the beginning of the day

            let startDate = new Date(today);
            let endDate = new Date(today);

            const year = today.getFullYear();
            const month = today.getMonth(); // 0-11
            const dayOfMonth = today.getDate();
            const dayOfWeek = today.getDay(); // 0=Sunday, 6=Saturday

            // Assumption: Fiscal/Financial year starts July 1st.
            const fiscalYearStartMonth = 6; // July
            
            switch (period) {
                case 'All Dates':
                    startDate = new Date(year - 20, 0, 1); // Go back 20 years
                    endDate = new Date(year + 5, 11, 31); // Go forward 5 years
                    break;
                case 'Today':
                    // Default is already today
                    break;
                case 'This week':
                    startDate.setDate(dayOfMonth - dayOfWeek);
                    endDate.setDate(dayOfMonth - dayOfWeek + 6);
                    break;
                case 'This week to date':
                     startDate.setDate(dayOfMonth - dayOfWeek);
                    // endDate is today
                    break;
                case 'This month':
                    startDate = new Date(year, month, 1);
                    endDate = new Date(year, month + 1, 0);
                    break;
                case 'This month to date':
                    startDate = new Date(year, month, 1);
                    // endDate is today
                    break;
                case 'This quarter':
                    const quarterStartMonth = Math.floor(month / 3) * 3;
                    startDate = new Date(year, quarterStartMonth, 1);
                    endDate = new Date(year, quarterStartMonth + 3, 0);
                    break;
                case 'This quarter to date':
                     const qStartMonth = Math.floor(month / 3) * 3;
                    startDate = new Date(year, qStartMonth, 1);
                    // endDate is today
                    break;
                case 'This year':
                    startDate = new Date(year, 0, 1);
                    endDate = new Date(year, 11, 31);
                    break;
                case 'This year to date':
                    startDate = new Date(year, 0, 1);
                    // endDate is today
                    break;
                case 'This year to last month':
                    startDate = new Date(year, 0, 1);
                    endDate = new Date(year, month, 0);
                    break;
                case 'This financial year':
                    const finYear = (month >= fiscalYearStartMonth) ? year : year - 1;
                    startDate = new Date(finYear, fiscalYearStartMonth, 1);
                    endDate = new Date(finYear + 1, fiscalYearStartMonth, 0);
                    break;
                case 'Yesterday':
                    startDate.setDate(dayOfMonth - 1);
                    endDate.setDate(dayOfMonth - 1);
                    break;
                case 'Last week':
                    startDate.setDate(dayOfMonth - dayOfWeek - 7);
                    endDate.setDate(dayOfMonth - dayOfWeek - 1);
                    break;
                case 'Last month':
                    startDate = new Date(year, month - 1, 1);
                    endDate = new Date(year, month, 0);
                    break;
                case 'Last quarter':
                    const lastQStartMonth = Math.floor(month / 3) * 3 - 3;
                    startDate = new Date(year, lastQStartMonth, 1);
                    endDate = new Date(year, lastQStartMonth + 3, 0);
                    break;
                case 'Last year':
                    startDate = new Date(year - 1, 0, 1);
                    endDate = new Date(year - 1, 11, 31);
                    break;
                case 'Last financial year':
                    const lastFinYear = (month >= fiscalYearStartMonth) ? year - 1 : year - 2;
                    startDate = new Date(lastFinYear, fiscalYearStartMonth, 1);
                    endDate = new Date(lastFinYear + 1, fiscalYearStartMonth, 0);
                    break;
                case 'Last 7 days':
                    startDate.setDate(dayOfMonth - 6);
                    // endDate is today
                    break;
                case 'Last 30 days':
                    startDate.setDate(dayOfMonth - 29);
                    // endDate is today
                    break;
                case 'Last 90 days':
                    startDate.setDate(dayOfMonth - 89);
                    // endDate is today
                    break;
                case 'Last 12 months':
                     startDate.setMonth(month - 11);
                     startDate.setDate(1);
                     endDate = new Date(year, month + 1, 0);
                    break;
                case 'Since 30 days ago':
                    startDate.setDate(dayOfMonth - 30);
                    // endDate is today
                    break;
                case 'Since 60 days ago':
                     startDate.setDate(dayOfMonth - 60);
                    // endDate is today
                    break;
                case 'Since 90 days ago':
                     startDate.setDate(dayOfMonth - 90);
                    // endDate is today
                    break;
                case 'Since 365 days ago':
                     startDate.setDate(dayOfMonth - 365);
                    // endDate is today
                    break;
                case 'Next week':
                    startDate.setDate(dayOfMonth - dayOfWeek + 7);
                    endDate.setDate(dayOfMonth - dayOfWeek + 13);
                    break;
                case 'Next 4 weeks':
                     startDate.setDate(dayOfMonth - dayOfWeek + 7);
                     endDate.setDate(dayOfMonth - dayOfWeek + 34);
                    break;
                case 'Next month':
                    startDate = new Date(year, month + 1, 1);
                    endDate = new Date(year, month + 2, 0);
                    break;
                case 'Next quarter':
                    const nextQStartMonth = Math.floor(month / 3) * 3 + 3;
                    startDate = new Date(year, nextQStartMonth, 1);
                    endDate = new Date(year, nextQStartMonth + 3, 0);
                    break;
                case 'Next year':
                    startDate = new Date(year + 1, 0, 1);
                    endDate = new Date(year + 1, 11, 31);
                    break;
            }
            return { startDate, endDate };
        }
        
        // --- QBO Panel Logic (HEAVILY MODIFIED) ---
        async function fetchAllQBOData() {
            tbody.innerHTML = '';
            tfoot.innerHTML = '';
            noDataMessage.style.display = 'block';
            noDataMessage.innerHTML = `<span class="loading-spinner"></span> Fetching report data from database...`;
            
            // The URL now points to the new DB script.
            const url = `ajax_fetch_report_from_db.php?start_date=${startDateInput.value}&end_date=${endDateInput.value}`;

            try {
                const response = await fetch(url);
                if (!response.ok) throw new Error(`Server responded with status: ${response.status}`);
                const result = await response.json();
                if (!result.success) throw new Error(result.error || 'The backend returned an error.');

                allQboData = result.data; // Store the complete dataset

                // Trigger the first render
                filterAndRenderQboTable();

            } catch (e) {
                noDataMessage.innerHTML = `An error occurred while fetching data: ${e.message}. Please try again.`;
                console.error("Fetch Error:", e);
            }
        }
        
        function initializeValueFilter() {
            const valueFilterContainer = scoped('.qbo-value-filter .multiselect-dropdown-content');
            let vfHtml = '';
            valueCategories.forEach(category => {
                vfHtml += `<label><input type="checkbox" value="${category}" checked> ${category}</label>`;
            });
            valueFilterContainer.innerHTML = vfHtml;

            scoped('.qbo-value-filter .multiselect-dropdown-btn').addEventListener('click', (e) => {
                e.stopPropagation();
                valueFilterContainer.style.display = valueFilterContainer.style.display === 'block' ? 'none' : 'block';
            });

            valueFilterContainer.addEventListener('change', () => {
                selectedValueCategories = Array.from(valueFilterContainer.querySelectorAll('input:checked')).map(cb => cb.value);
                // When the filter changes, we just need to re-render the table with the existing data
                filterAndRenderQboTable();
            });
        }
        
        function filterAndRenderQboTable() {
            tbody.innerHTML = '';
            tfoot.innerHTML = `<tr class="total-row"><td colspan="6"><strong>GRAND TOTAL</strong></td><td class="currency" id="grand-total-qty-${reportInstanceCounter}">0</td><td class="currency" id="grand-total-line-count-${reportInstanceCounter}">0</td><td class="currency" id="grand-total-amt-${reportInstanceCounter}">0.00</td></tr>`;
            noDataMessage.style.display = 'none';

            const customerFilter = scoped('.qbo-customer-filter').value.toLowerCase();
            const quoteFilter = scoped('.qbo-quote-filter').value.toLowerCase();
            const lowercaseSelectedServices = selectedProductServices.map(s => s.toLowerCase());

            for (const customerId in allQboData) {
                const customerData = allQboData[customerId];
                
                const filteredDetails = customerData.details.filter(txn => {
                    const serviceMatch = lowercaseSelectedServices.includes(txn.product_service.toLowerCase());
                    const quoteMatch = !quoteFilter || (txn.class && txn.class.toLowerCase().includes(quoteFilter));
                    return serviceMatch && quoteMatch;
                });

                if (filteredDetails.length === 0) continue;

                // --- Original logic for totals, UNCHANGED ---
                const newTotals = filteredDetails.reduce((acc, txn) => {
                    acc.total += txn.sales_price;
                    acc.quantity += txn.qty; // Note: 'qty' in the original data is always 1 per line
                    return acc;
                }, { total: 0, quantity: 0 });
                
                // --- NEW logic to get the pure line count ---
                const lineCount = filteredDetails.length;

                const customerCategory = classifyQboValue(newTotals.total, classificationRules);
                if (!selectedValueCategories.includes(customerCategory)) continue;
                if (customerFilter && !customerData.customerInfo.name.toLowerCase().includes(customerFilter)) continue;
                
                // Pass the new lineCount to the rendering function
                renderCustomerData(customerId, customerData.customerInfo, filteredDetails, newTotals.quantity, lineCount, newTotals.total);
            }
            
            if (tbody.children.length === 0) {
                noDataMessage.style.display = 'block';
                noDataMessage.innerHTML = 'No data found for the selected criteria.';
                tfoot.innerHTML = '';
            } else {
                runClassification();
                updateGrandTotals();
                sortQBOReport();
            }
        }

        function renderCustomerData(customerId, customer, detailData, quantity, lineCount, total) {
            if (!customer) return;

            let detailRowsHtml = '';
            // Add a cell for the new column
            detailData.forEach(txn => {
                detailRowsHtml += `<tr class="detail-row" data-customer-id="${customerId}">
                    <td></td><td>${txn.date}</td><td>${txn.type}</td><td>${txn.num}</td><td>${escapeHtml(txn.class)}</td>
                    <td>${escapeHtml(txn.product_service)}</td>
                    <td class="currency">${txn.qty}</td>
                    <td class="currency">1</td>
                    <td class="currency">${txn.sales_price.toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}</td></tr>`;
            });
            
            const customerRowHtml = `
                <tr class="customer-header" data-customer-id="${customerId}" data-customer-name="${escapeHtml(customer.name)}" data-billed-total="${total}" data-quantity-total="${quantity}" data-line-count="${lineCount}" data-details-loaded="true">
                    <td class="value-cell" data-value=""></td><td colspan="5"><strong><span class="toggle-arrow">▶</span> ${escapeHtml(customer.name)}</strong></td>
                    <td class="currency total-cell-qty">${quantity}</td>
                    <td class="currency total-cell-line-count">${lineCount}</td>
                    <td class="currency total-cell-amt">${total.toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}</td></tr>`;
            
            tbody.insertAdjacentHTML('beforeend', customerRowHtml + detailRowsHtml);
        }
        
        function updateGrandTotals() {
            let grandTotalQty = 0;
            let grandTotalLineCount = 0; // Add a variable for the new total
            let grandTotalAmt = 0.0;
            scopedAll('.customer-header:not([style*="display: none"])').forEach(header => {
                 grandTotalQty += parseInt(header.dataset.quantityTotal, 10) || 0;
                 grandTotalLineCount += parseInt(header.dataset.lineCount, 10) || 0; // Sum the new value
                 grandTotalAmt += parseFloat(header.dataset.billedTotal.replace(/,/g, '')) || 0;
            });
            const qtyCell = scoped(`#grand-total-qty-${reportInstanceCounter}`);
            const lineCountCell = scoped(`#grand-total-line-count-${reportInstanceCounter}`); // Select the new cell
            const amtCell = scoped(`#grand-total-amt-${reportInstanceCounter}`);
            if(qtyCell) qtyCell.textContent = grandTotalQty.toLocaleString();
            if(lineCountCell) lineCountCell.textContent = grandTotalLineCount.toLocaleString(); // Update the new cell
            if(amtCell) amtCell.textContent = grandTotalAmt.toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2});
        }
        
        tbody.addEventListener('click', function(e) {
            const header = e.target.closest('.customer-header');
            if (!header || header.dataset.detailsLoaded !== "true") return;
            header.classList.toggle('expanded');
            const isExpanded = header.classList.contains('expanded');
            header.querySelector('.toggle-arrow').style.transform = isExpanded ? 'rotate(90deg)' : 'rotate(0deg)';
            scopedAll(`.detail-row[data-customer-id='${header.dataset.customerId}']`).forEach(row => {
                row.style.display = isExpanded ? 'table-row' : 'none';
            });
        });
        
        // --- MODIFIED: Event listeners now call the new filter/render function ---
        function initializeQBOFilters() {
            const productServiceContainer = scoped('.product-service-filter .multiselect-dropdown-content');
            let psHtml = '';
            defaultProductServices.forEach(service => {
                const isChecked = selectedProductServices.includes(service);
                psHtml += `<label><input type="checkbox" value="${escapeHtml(service)}" ${isChecked ? 'checked' : ''}> ${escapeHtml(service)}</label>`;
            });
            productServiceContainer.innerHTML = psHtml;
            
            scoped('.product-service-filter .multiselect-dropdown-btn').addEventListener('click', (e) => {
                e.stopPropagation();
                productServiceContainer.style.display = productServiceContainer.style.display === 'block' ? 'none' : 'block';
            });
            
            // This now filters in-memory instead of re-fetching
            productServiceContainer.addEventListener('change', () => {
                 selectedProductServices = Array.from(productServiceContainer.querySelectorAll('input:checked')).map(cb => cb.value);
                 filterAndRenderQboTable();
            });

            scoped('.qbo-customer-filter').addEventListener('input', filterAndRenderQboTable);
            scoped('.qbo-quote-filter').addEventListener('input', filterAndRenderQboTable);
        }
        
        function sortQBOReport() {
            if (!qboTableSort.column) return;
            const headers = Array.from(tbody.querySelectorAll('.customer-header'));
            const dir = qboTableSort.dir === 'asc' ? 1 : -1;
            const categoryOrder = { "High-Value": 1, "Mid-Value": 2, "Low-Value": 3, "No Activity": 4, "N/A": 5 };
            
            headers.sort((a, b) => {
                let valA, valB;
                switch(qboTableSort.column) {
                    case 'value': valA = categoryOrder[a.querySelector('.value-cell').dataset.value] || 99; valB = categoryOrder[b.querySelector('.value-cell').dataset.value] || 99; break;
                    case 'qty': valA = parseFloat(a.dataset.quantityTotal) || 0; valB = parseFloat(b.dataset.quantityTotal) || 0; break;
                    case 'lineCount': valA = parseFloat(a.dataset.lineCount) || 0; valB = parseFloat(b.dataset.lineCount) || 0; break; // Make the new column sortable
                    case 'price': valA = parseFloat(a.dataset.billedTotal) || 0; valB = parseFloat(b.dataset.billedTotal) || 0; break;
                    default: return 0;
                }
                if (valA < valB) return -1 * dir;
                if (valA > valB) return 1 * dir;
                const secondaryA = parseFloat(a.dataset.billedTotal) || 0;
                const secondaryB = parseFloat(b.dataset.billedTotal) || 0;
                return secondaryB - secondaryA;
            });

            const fragment = document.createDocumentFragment();
            headers.forEach(header => {
                fragment.appendChild(header);
                scopedAll(`.detail-row[data-customer-id='${header.dataset.customerId}']`).forEach(row => fragment.appendChild(row));
                const isExpanded = header.classList.contains('expanded');
                 scopedAll(`.detail-row[data-customer-id='${header.dataset.customerId}']`).forEach(row => {
                    row.style.display = isExpanded ? 'table-row' : 'none';
                });
            });
            tbody.appendChild(fragment);
        }

        qboReportHead.addEventListener('click', function(e) {
            const header = e.target.closest('th.sortable'); if (!header) return;
            const column = header.dataset.col;
            if (qboTableSort.column === column) qboTableSort.dir = qboTableSort.dir === 'asc' ? 'desc' : 'asc';
            else { qboTableSort.column = column; qboTableSort.dir = (column === 'value') ? 'asc' : 'desc'; }
            qboReportHead.querySelectorAll('th.sortable').forEach(th => th.classList.remove('asc', 'desc'));
            header.classList.add(qboTableSort.dir);
            sortQBOReport();
        });

        // --- DB Comparison Panel Logic (UNCHANGED) ---
        async function fetchAndRenderDbData() {
            const dbQuotesTableContainer = scoped('.db-quotes-table-container');
            dbQuotesTableContainer.innerHTML = `<div class="centered-loader"><span class="loading-spinner"></span> Loading DB Quotes...</div>`;
            
            const url = `ajax_fetch_quote_details_for_comparison.php?start_date=${startDateInput.value}&end_date=${endDateInput.value}`;

            try {
                const response = await fetch(url);
                const result = await response.json();
                if (result.success) {
                    allDbQuotes = result.data;
                    populateDbFilters(allDbQuotes);
                    setupDbMultiSelectCheckboxes(scoped('.db-region-filter .multiselect-dropdown-content'));
                    setupDbMultiSelectCheckboxes(scoped('.db-country-filter .multiselect-dropdown-content'));
                    setupDbMultiSelectCheckboxes(scoped('.db-stage-filter .multiselect-dropdown-content'));
                    applyDbFiltersAndRender(); 
                    dbQuotesLoaded = true;
                } else {
                    dbQuotesTableContainer.innerHTML = `<p style="color:red;padding:20px;">Error: ${result.error || 'Failed to load data.'}</p>`;
                }
            } catch (e) {
            // Log the actual error object to the console for detailed debugging
            console.error("An error occurred after fetching DB data:", e);
            
            // Display a more accurate error message to the user
            dbQuotesTableContainer.innerHTML = `<p style="color:red;padding:20px;"><strong>Error processing data:</strong> ${e.message}.<br><small>Check the developer console for more details.</small></p>`;
        }
        }

        function populateDbFilters(data) {
            const regions = new Set(), countries = new Set(), stages = new Set();
            data.forEach(row => {
                if(row.region && row.region !== 'N/A') regions.add(row.region);
                if(row.country && row.country !== 'N/A') countries.add(row.country);
                if(row.quotestage && row.quotestage !== 'N/A') stages.add(row.quotestage);
            });
            
            const dbRegionFilterContent = scoped('.db-region-filter .multiselect-dropdown-content');
            const dbCountryFilterContent = scoped('.db-country-filter .multiselect-dropdown-content');
            const dbStageFilterContent = scoped('.db-stage-filter .multiselect-dropdown-content');

            dbRegionFilterContent.innerHTML = '<label><input type="checkbox" value="all"> (Select All)</label>';
            Array.from(regions).sort().forEach(r => dbRegionFilterContent.innerHTML += `<label><input type="checkbox" value="${escapeHtml(r)}" checked> ${escapeHtml(r)}</label>`);
            
            dbCountryFilterContent.innerHTML = '<label><input type="checkbox" value="all"> (Select All)</label>';
            Array.from(countries).sort().forEach(c => dbCountryFilterContent.innerHTML += `<label><input type="checkbox" value="${escapeHtml(c)}" checked> ${escapeHtml(c)}</label>`);
            
            dbStageFilterContent.innerHTML = '<label><input type="checkbox" value="all"> (Select All)</label>';
            Array.from(stages).sort().forEach(s => dbStageFilterContent.innerHTML += `<label><input type="checkbox" value="${escapeHtml(s)}" checked> ${escapeHtml(s)}</label>`);
        }

        function applyDbFiltersAndRender() {
            const internalAgentVal = scoped('.db-internal-agent-filter').value.toLowerCase().trim();
            const externalAgentVal = scoped('.db-external-agent-filter').value.toLowerCase().trim();
            const dbQuoteNoVal = scoped('.db-quote-no-filter').value.toLowerCase().trim();
            const dbCustomerVal = scoped('.db-customer-filter').value.toLowerCase().trim();
            
            const selectedRegions = Array.from(scoped('.db-region-filter .multiselect-dropdown-content').querySelectorAll('input:checked')).map(cb => cb.value);
            const selectedCountries = Array.from(scoped('.db-country-filter .multiselect-dropdown-content').querySelectorAll('input:checked')).map(cb => cb.value);
            const selectedStages = Array.from(scoped('.db-stage-filter .multiselect-dropdown-content').querySelectorAll('input:checked')).map(cb => cb.value);

            let filteredData = [...allDbQuotes];
            if (internalAgentVal) filteredData = filteredData.filter(row => (row.internal_agent || '').toLowerCase().includes(internalAgentVal));
            if (externalAgentVal) filteredData = filteredData.filter(row => (row.external_agent || '').toLowerCase().includes(externalAgentVal));
            if (dbQuoteNoVal) filteredData = filteredData.filter(row => (row.quote_no || '').toLowerCase().includes(dbQuoteNoVal));
            if (dbCustomerVal) filteredData = filteredData.filter(row => (row.organization_name || '').toLowerCase().includes(dbCustomerVal));
            if (!selectedRegions.includes('all')) filteredData = filteredData.filter(row => selectedRegions.includes(row.region));
            if (!selectedCountries.includes('all')) filteredData = filteredData.filter(row => selectedCountries.includes(row.country));
            if (!selectedStages.includes('all')) filteredData = filteredData.filter(row => selectedStages.includes(row.quotestage));

            const customerSummary = {};
            filteredData.forEach(quote => {
                const customerName = quote.organization_name;
                if (!customerSummary[customerName]) {
                    customerSummary[customerName] = { organization_name: customerName, quoteCount: 0, paxCount: 0, quotes: [], nestedSort: { column: 'travel_date', dir: 'desc', type: 'string' }};
                }
                customerSummary[customerName].quoteCount++;
                customerSummary[customerName].paxCount += parseInt(quote.pax, 10) || 0;
                customerSummary[customerName].quotes.push(quote);
            });
            
            currentAggregatedDbData = Object.values(customerSummary);
            currentAggregatedDbData.forEach(cust => { cust.engagementCategory = classifyDbEngagement(cust.quoteCount, classificationRules); });

            const col = dbTableSort.column, dir = dbTableSort.dir, type = dbTableSort.type;
            const engagementOrder = { "High-Engagement": 1, "Mid-Engagement": 2, "Low-Engagement": 3, "N/A": 4 };
            currentAggregatedDbData.sort((a, b) => {
                let valA, valB;
                if (col === 'engagementCategory') { valA = engagementOrder[a.engagementCategory] || 99; valB = engagementOrder[b.engagementCategory] || 99; }
                else { valA = a[col]; valB = b[col]; if (type === 'number') { valA = parseFloat(valA || 0); valB = parseFloat(valB || 0); } else { valA = (valA || '').toLowerCase(); valB = (valB || '').toLowerCase(); } }
                if (valA < valB) return dir === 'asc' ? -1 : 1; if (valA > valB) return dir === 'asc' ? 1 : -1; return 0;
            });
            renderDbDrilldownTable(currentAggregatedDbData);
        }
        
        function renderDbDrilldownTable(data) {
            const dbQuotesTableContainer = scoped('.db-quotes-table-container');
            if (data.length === 0) { dbQuotesTableContainer.innerHTML = `<p style="text-align:center;padding:20px;">No data matches the current filters.</p>`; return; }
            let totalPaxSum = 0, totalQuotesSum = 0;
            let tableHtml = `<table><thead><tr>
                        <th class="sortable" data-col="organization_name" data-type="string">Customer</th>
                        <th class="sortable" data-col="engagementCategory" data-type="string">Engagement</th>
                        <th class="sortable currency" data-col="quoteCount" data-type="number">Number of Quotes</th>
                        <th class="sortable currency" data-col="paxCount" data-type="number">Total PAX</th>
                    </tr></thead><tbody>`;
            data.forEach(customer => {
                totalQuotesSum += customer.quoteCount; totalPaxSum += customer.paxCount;
                let engagementClass = '';
                if (customer.engagementCategory.includes('High')) engagementClass = 'cat-high';
                else if (customer.engagementCategory.includes('Mid')) engagementClass = 'cat-mid';
                else if (customer.engagementCategory.includes('Low')) engagementClass = 'cat-low';
                tableHtml += `<tr class="db-customer-header" data-customer="${escapeHtml(customer.organization_name)}">
                        <td><span class="toggle-arrow-db">▶</span> ${escapeHtml(customer.organization_name)}</td>
                        <td class="${engagementClass}">${customer.engagementCategory}</td>
                        <td class="currency">${customer.quoteCount}</td><td class="currency">${customer.paxCount}</td></tr>
                    <tr class="db-details-row" data-customer-for="${escapeHtml(customer.organization_name)}" style="display: none;"><td colspan="4">
                        ${renderNestedTable(customer)}
                    </td></tr>`;
            });
            tableHtml += `</tbody><tfoot><tr class="total-row"><td colspan="2"><strong>TOTAL (Filtered)</strong></td>
                        <td class="currency"><strong>${totalQuotesSum}</strong></td>
                        <td class="currency"><strong>${totalPaxSum}</strong></td></tr></tfoot></table>`;
            dbQuotesTableContainer.innerHTML = tableHtml;
            scopedAll('.db-quotes-table-container thead > tr > th.sortable').forEach(th => th.classList.remove('asc', 'desc'));
            const currentSortHeader = scoped(`.db-quotes-table-container thead > tr > th[data-col="${dbTableSort.column}"]`);
            if (currentSortHeader) currentSortHeader.classList.add(dbTableSort.dir);
        }

        function renderNestedTable(customer) {
            let nestedHtml = `<table class="nested-table"><thead><tr>
                    <th class="sortable" data-col="quote_no" data-type="string">Quote No.</th>
                    <th class="sortable" data-col="internal_agent" data-type="string">Internal Agent</th>
                    <th class="sortable" data-col="external_agent" data-type="string">External Agent</th>
                    <th class="sortable" data-col="travel_date" data-type="string">Travel Date</th>
                    <th class="sortable" data-col="region" data-type="string">Region</th>
                    <th class="sortable" data-col="country" data-type="string">Country</th>
                    <th class="sortable" data-col="quotestage" data-type="string">Stage</th>
                    <th class="sortable currency" data-col="pax" data-type="number">PAX</th>
                </tr></thead><tbody>`;
            customer.quotes.forEach(quote => {
                nestedHtml += `<tr>
                    <td>${escapeHtml(quote.quote_no)}</td><td>${escapeHtml(quote.internal_agent)}</td>
                    <td>${escapeHtml(quote.external_agent)}</td><td>${escapeHtml(quote.travel_date)}</td>
                    <td>${escapeHtml(quote.region)}</td><td>${escapeHtml(quote.country)}</td>
                    <td>${escapeHtml(quote.quotestage)}</td><td class="currency">${parseInt(quote.pax, 10) || 0}</td></tr>`;
            });
            nestedHtml += `</tbody></table>`;
            return nestedHtml;
        }
    
            function setupDbMultiSelectCheckboxes(filterContent) {
                 const allCheckbox = filterContent.querySelector('input[value="all"]');
                 const checkboxes = filterContent.querySelectorAll('input:not([value="all"])');
                 const btn = filterContent.previousElementSibling;
                const updateButtonText = () => {
                    let btnText = 'Items';
                    if (btn.closest('.db-country-filter')) btnText = 'Countries';
                    else if (btn.closest('.db-stage-filter')) btnText = 'Stages';
                    else if (btn.closest('.db-region-filter')) btnText = 'Regions';
                    
                    const checkedCount = filterContent.querySelectorAll('input:not([value="all"]):checked').length;
                    if (allCheckbox.checked) btn.textContent = `All ${btnText}`;
                    else if (checkedCount === 0) btn.textContent = `No ${btnText} Selected`;
                    else if (checkedCount === 1) btn.textContent = filterContent.querySelector('input:not([value="all"]):checked').value;
                    else btn.textContent = `${checkedCount} ${btnText} Selected`;
                };
                filterContent.addEventListener('change', function(e) {
                    if (e.target.matches('input[type="checkbox"]')) {
                        if (e.target.value === 'all') {
                            checkboxes.forEach(cb => cb.checked = e.target.checked);
                        } else {
                            allCheckbox.checked = ![...checkboxes].some(cb => !cb.checked);
                        }
                        updateButtonText();
                    }
                });
                updateButtonText();
            }    
        function setupAllDbUiHandlers() {
            const dbPanel = scoped('.db-quotes-panel');
            dbPanel.addEventListener('input', function(e) {
                if (e.target.matches('.db-internal-agent-filter, .db-external-agent-filter, .db-quote-no-filter, .db-customer-filter, .multiselect-dropdown-content input[type="checkbox"]')) {
                    applyDbFiltersAndRender();
                }
            });
            dbPanel.addEventListener('click', function(e) {
                const mainHeader = e.target.closest('.db-quotes-table-container > table > thead > tr > th.sortable');
                const nestedHeader = e.target.closest('.nested-table th.sortable');
                const customerHeader = e.target.closest('.db-customer-header');
                const multiSelectButton = e.target.closest('.multiselect-dropdown-btn');

                if (mainHeader) {
                    const column = mainHeader.dataset.col, type = mainHeader.dataset.type;
                    if (dbTableSort.column === column) dbTableSort.dir = dbTableSort.dir === 'desc' ? 'asc' : 'desc';
                    else { dbTableSort.column = column; dbTableSort.dir = (type === 'number' || column === 'paxCount') ? 'desc' : 'asc'; dbTableSort.type = type; }
                    applyDbFiltersAndRender(); return;
                }
                if (nestedHeader) {
                    e.stopPropagation();
                    const customerName = nestedHeader.closest('.db-details-row').dataset.customerFor;
                    const customerData = currentAggregatedDbData.find(c => c.organization_name === customerName);
                    if (!customerData) return;
                    
                    const column = nestedHeader.dataset.col, type = nestedHeader.dataset.type;
                    if (customerData.nestedSort.column === column) {
                        customerData.nestedSort.dir = customerData.nestedSort.dir === 'desc' ? 'asc' : 'desc';
                    } else {
                        customerData.nestedSort = { column, type, dir: (type === 'number') ? 'desc' : 'asc' };
                    }
                    const dir = customerData.nestedSort.dir === 'asc' ? 1 : -1;
                    customerData.quotes.sort((a,b) => {
                        let valA = a[column] || (type === 'number' ? 0 : '');
                        let valB = b[column] || (type === 'number' ? 0 : '');
                        if (type === 'number') { valA = parseFloat(valA); valB = parseFloat(valB); }
                        if (valA < valB) return -1 * dir; if (valA > valB) return 1 * dir; return 0;
                    });
                    const detailsRow = nestedHeader.closest('.db-details-row');
                    detailsRow.querySelector('td').innerHTML = renderNestedTable(customerData);
                    detailsRow.querySelectorAll('.nested-table th.sortable').forEach(th => th.classList.remove('asc', 'desc'));
                    detailsRow.querySelector(`.nested-table th[data-col="${column}"]`).classList.add(customerData.nestedSort.dir); return;
                }
                if (customerHeader) {e
                    customerHeader.classList.toggle('expanded');
                    const customerName = customerHeader.dataset.customer;
                    const detailsRow = scoped(`.db-details-row[data-customer-for="${customerName}"]`);
                    if (detailsRow) {
                        const arrow = customerHeader.querySelector('.toggle-arrow-db');
                        const isExpanded = detailsRow.style.display === 'table-row';
                        detailsRow.style.display = isExpanded ? 'none' : 'table-row';
                        if (arrow) arrow.style.transform = isExpanded ? 'rotate(0deg)' : 'rotate(90deg)';
                    } return;
                }
                if (multiSelectButton) {
                    e.stopPropagation();
                    const dropdownContent = multiSelectButton.nextElementSibling;
                    scopedAll('.multiselect-dropdown-content').forEach(el => {
                        if (el !== dropdownContent) el.style.display = 'none';
                    });
                    dropdownContent.style.display = dropdownContent.style.display === 'block' ? 'none' : 'block'; return;
                }
            });
        }
        
        scopedAll('.tab-link').forEach(link => {
            link.addEventListener('click', async () => {
                const tabId = link.dataset.tab;
                scopedAll('.tab-link').forEach(l => l.classList.remove('active'));
                scopedAll('.tab-content').forEach(c => c.classList.remove('active'));
                link.classList.add('active');
                scoped(`.${tabId}`).classList.add('active');
                if (tabId === 'db-report-tab' && !dbQuotesLoaded) fetchAndRenderDbData();
            });
        });

        runReportBtn.addEventListener('click', () => {
            dbQuotesLoaded = false;
            // The main data fetch for the QBO tab is now separate from the DB tab
            fetchAllQBOData();
            if (scoped('.db-report-tab').classList.contains('active')) {
                fetchAndRenderDbData();
            } else {
                // Clear the DB tab content if it's not active to ensure it reloads on next click
                scoped('.db-quotes-table-container').innerHTML = '';
            }
        });

        if(reportInstanceCounter === 1) fetchAllQBOData();
        initializeQBOFilters();
        initializeValueFilter();
        setupAllDbUiHandlers();
        
        const periodSelect = scoped('.date-period-select');
        
        // Populate the dropdown with options
        datePeriods.forEach(period => {
            const option = document.createElement('option');
            option.value = period;
            option.textContent = period;
            periodSelect.appendChild(option);
        });

        // Add the event listener to handle changes
        periodSelect.addEventListener('change', (e) => {
            const selectedPeriod = e.target.value;
            if (selectedPeriod === 'Custom dates') {
                return; // Do nothing if they select custom
            }

            const dates = calculateDateRange(selectedPeriod);
            
            // Update the date input fields
            startDateInput.value = formatDate(dates.startDate);
            endDateInput.value = formatDate(dates.endDate);
            
            // Optional but recommended: Automatically run the report after changing the dates
            runReportBtn.click();
        });

        // Set the default selection to 'Custom dates' to match the initial state
        periodSelect.value = 'Custom dates';

    
    }

    // --- Global Controls (UNCHANGED) ---
    const settingsBtn = document.getElementById('settings-btn'), settingsModal = document.getElementById('settings-modal'), settingsBackdrop = document.getElementById('settings-modal-backdrop');
    const closeSettingsBtn = document.getElementById('close-settings-modal-btn'), cancelSettingsBtn = document.getElementById('cancel-settings-btn'), saveSettingsBtn = document.getElementById('save-settings-btn');
    const ruleInputs = {
        highValue: document.getElementById('rule-high-value'), midValue: document.getElementById('rule-mid-value'), lowValue: document.getElementById('rule-low-value'),
        highEngage: document.getElementById('rule-high-engagement'), midEngage: document.getElementById('rule-mid-engagement'), lowEngage: document.getElementById('rule-low-engagement'),
    };
    let classificationRules = {};

    function openSettingsModal() { settingsModal.style.display = 'block'; settingsBackdrop.style.display = 'block'; }
    function closeSettingsModal() { settingsModal.style.display = 'none'; settingsBackdrop.style.display = 'none'; }
    function saveClassificationRules() {
        const newRules = {
            highValue: parseFloat(ruleInputs.highValue.value) || 0, midValue: parseFloat(ruleInputs.midValue.value) || 0, lowValue: parseFloat(ruleInputs.lowValue.value) || 0,
            highEngage: parseInt(ruleInputs.highEngage.value, 10) || 0, midEngage: parseInt(ruleInputs.midEngage.value, 10) || 0, lowEngage: parseInt(ruleInputs.lowEngage.value, 10) || 0,
        };
        localStorage.setItem('classificationRules', JSON.stringify(newRules)); 
        classificationRules = newRules;
        closeSettingsModal(); 
        document.querySelectorAll('.report-instance').forEach(instance => instance.querySelector('.run-report-btn').click());
    }
    function loadClassificationRules() {
        const savedRules = JSON.parse(localStorage.getItem('classificationRules'));
        const defaultRules = { highValue: 20000, midValue: 3000, lowValue: 1, highEngage: 20, midEngage: 3, lowEngage: 1 };
        classificationRules = { ...defaultRules, ...savedRules };
        ruleInputs.highValue.value = classificationRules.highValue; 
        ruleInputs.midValue.value = classificationRules.midValue; 
        ruleInputs.lowValue.value = classificationRules.lowValue;
        ruleInputs.highEngage.value = classificationRules.highEngage; 
        ruleInputs.midEngage.value = classificationRules.midEngage; 
        ruleInputs.lowEngage.value = classificationRules.lowEngage;
    }
    settingsBtn.addEventListener('click', openSettingsModal); closeSettingsBtn.addEventListener('click', closeSettingsModal); cancelSettingsBtn.addEventListener('click', closeSettingsModal); saveSettingsBtn.addEventListener('click', saveClassificationRules); settingsBackdrop.addEventListener('click', closeSettingsModal);

    function classifyQboValue(billedTotal, rules) {
        if (billedTotal > rules.highValue) return "High-Value";
        if (billedTotal > rules.midValue) return "Mid-Value";
        if (billedTotal >= rules.lowValue) return "Low-Value";
        return "N/A";
    }
    function classifyDbEngagement(quoteCount, rules) {
        if (quoteCount >= rules.highEngage) return "High-Engagement";
        if (quoteCount >= rules.midEngage) return "Mid-Engagement";
        if (quoteCount >= rules.lowEngage) return "Low-Engagement";
        return "N/A";
    }

    function runClassification() {
        document.querySelectorAll('.customer-header').forEach(header => {
            const billedTotal = parseFloat(header.dataset.billedTotal);
            const category = classifyQboValue(billedTotal, classificationRules);
            const valueCell = header.querySelector('.value-cell');
            valueCell.textContent = category; valueCell.dataset.value = category;
            valueCell.className = 'value-cell';
            if (category.includes('High')) valueCell.classList.add('cat-high');
            else if (category.includes('Mid')) valueCell.classList.add('cat-mid');
            else if (category.includes('Low')) valueCell.classList.add('cat-low');
        });
    }
    
    // --- Initial Page Load (UNCHANGED) ---
    loadClassificationRules();
    createReportInstance("<?php echo $startDate; ?>", "<?php echo $endDate; ?>");
    
    addReportBtn.addEventListener('click', () => {
        const today = new Date();
        const lastMonth = new Date(today.getFullYear(), today.getMonth() - 1, 1);
        const lastMonthEnd = new Date(today.getFullYear(), today.getMonth(), 0);
        const formatDate = (d) => d.toISOString().split('T')[0];
        createReportInstance(formatDate(lastMonth), formatDate(lastMonthEnd));
    });

    function escapeHtml(str) { if (!str) return ''; const p = document.createElement("p"); p.textContent = str; return p.innerHTML; }
    document.addEventListener('click', function(e) { if (!e.target.closest('.multiselect-dropdown')) { document.querySelectorAll('.multiselect-dropdown-content').forEach(dd => dd.style.display = 'none'); }});
    
});
</script>
</body>
</html>